{$I Options}
UNIT CheckIO;

{
  Copyright (C) 1991 Julian Byrne. All rights reserved.

  Title:	Check Input/Output
  File name:	CHECKIO.PAS
  Version:	1.00
  Usage:	USES CheckIO;
  Description:	Contains I/O statements similar to Turbo Pascal's but which
		give more meaningful error messages if they fail. There are
		also a few extra routines intended for a network environment.
  Dependencies: See USES statement
  Author:	Julian Byrne
  Address:	Electrical and Computer Systems Engineering Department
		Monash University, Wellington Road, Clayton, Victoria, 3168
		Australia
  Internet:	julian.byrne@monash.edu.au.
  Other nets:	Quaterman & Hoskins "Notable Computer Networks"
		CACM Oct'86 pp932-971
  History:	90/ 5/ 1 Initial version
  Notes:
}

INTERFACE { CheckIO }

  USES
    CRT,
    DOS,
    Error;

  CONST
    AccessRead	       = $00; { File access mode bits }
    AccessWrite        = $01;
    AccessBoth	       = $02;
    ShareCompatibility = $00;
    ShareDenyBoth      = $10;
    ShareDenyWrite     = $20;
    ShareDenyRead      = $30;
    ShareDenyNone      = $40;
    FileInherit        = $80;

    Shareable	       = $80; { File attribute bits }

    PageSize	       = 512;  { EXE file size is specified in these units }

  TYPE
    STRING2	  = STRING[  2]; { Turbo doesn't allow STRING[] in some places}
    STRING4	  = STRING[  4];
    STRING8	  = STRING[  8];
    STRING9	  = STRING[  9];
    STRING16	  = STRING[ 16];
    STRING48	  = STRING[ 48];
    STRING128	  = STRING[128];
    DriveType	  = 'A'..'a';   { Disk drives - 32 possible values + Undefined }
{
  The EXE file header is described in the Microsoft Programmer's Reference
  Chapter 5. Note opposite segment/offset order for SS/SP and IP/CS.
}
    EXEHeader	  = RECORD
		      Magic,
		      LastPageSize,
		      PageFileSize,
		      NumReloc,
		      HeadSize,
		      MinSize,
		      MaxSize,
		      StartSS,
		      StartSP,
		      CheckSum,
		      StartIP,
		      StartCS,
		      RelTab,
		      OverLayNumber : WORD;
		    END;

  FUNCTION  StrName(VAR x) : STRING;
  FUNCTION  FileName(VAR f) : PathStr;
  FUNCTION  Str0(x : BYTE   ) : STRING2;
  FUNCTION  StrBYTE(x, len : BYTE) : STRING8;
  FUNCTION  StrWORD(x : WORD; len : BYTE) : STRING8;
  FUNCTION  StrLONGINT(x : LONGINT; len : BYTE) : STRING16;
  FUNCTION  Hex1(x : BYTE   ) : CHAR;
  FUNCTION  Hex2(x : BYTE   ) : STRING2;
  FUNCTION  Hex4(x : WORD   ) : STRING4;
  FUNCTION  Hex8(x : LONGINT) : STRING8;
  FUNCTION  Hex9(x : POINTER) : STRING9;
  PROCEDURE ValHex(s : STRING16; VAR v : LONGINT; VAR code : INTEGER);
  PROCEDURE KeyDiscard;
  FUNCTION  GetBP : WORD; INLINE($89/$E8); { MOV AX, BP ; Get frame pointer }
  PROCEDURE StackDump;
  PROCEDURE FatalError(Status : INTEGER; Op : STRING48; Arg : STRING);
  PROCEDURE SetSwitchChar(x : CHAR);
  FUNCTION  GetSwitchChar : CHAR;
  FUNCTION  CanonicalPath(x : STRING128) : STRING128;
  PROCEDURE SelectDisk(Drive : DriveType);
  FUNCTION  GetCurrentDisk : DriveType;
  PROCEDURE ChDirCheck(Path : PathStr);
  FUNCTION  CreateNewFile    (VAR f : FILE; RecSizeArg : WORD) : BOOLEAN;
  PROCEDURE ReWriteCheck     (VAR f : FILE; RecSizeArg : WORD);
  PROCEDURE AssignCheck      (VAR f : FILE; Path : PathStr);
  PROCEDURE ResetCheck	     (VAR f : FILE; RecSize : WORD);
  PROCEDURE CloseCheck	     (VAR f : FILE);
  PROCEDURE BlockReadCheck   (VAR f : FILE; VAR Buf; BufSize : WORD);
  PROCEDURE BlockWriteCheck  (VAR f : FILE; VAR Buf; BufSize : WORD);
  PROCEDURE RenameCheck      (VAR f : FILE; Path : PathStr);
  PROCEDURE SeekCheck	     (VAR f : FILE; p : LONGINT);
  FUNCTION  FilePosCheck     (VAR f : FILE) : LONGINT;
  FUNCTION  FileSizeCheck    (VAR f : FILE) : LONGINT;
  FUNCTION  EOFCheck	     (VAR f : FILE) : BOOLEAN;
  PROCEDURE ResetCheckShare  (VAR f : FILE; RecSize : WORD);
  PROCEDURE ReWriteCheckShare(VAR f : FILE; RecSize : WORD);
  PROCEDURE EraseIfPresent   (VAR f : FILE);
  FUNCTION  EncodeSTRING     (x : STRING) : STRING;
  FUNCTION  DecodeSTRING     (x : STRING) : STRING;
  FUNCTION  HeaderFileSize(   VAR Header : EXEHeader) : LONGINT;
  PROCEDURE DumpHeader	     (VAR f : FILE; VAR Header : EXEHeader);

  VAR
    FileAttr	: WORD;
    StackOnExit : BOOLEAN;

IMPLEMENTATION { CheckIO }

CONST
  PrefixParaSize = $10; { 256 bytes in paragraphs }

{
  Convert a 0 terminated string to a Turbo length string
}
FUNCTION StrName(VAR x) : STRING;

  VAR
    b	   : ARRAY[0..255] OF CHAR ABSOLUTE x;
    I	   : BYTE;
    Result : STRING;

  BEGIN { StrName }
    I := 0;
    WHILE b[I] <> CHR(0) DO
      INC(I);
    IF I <> 0 THEN
      Move(x, Result[1], I);
    Result[0] := CHR(I);
    StrName   := Result;
  END { StrName };
{
  Given a FILE variable of any type return the name of the file it is accessing.
  Results are not meaningful if f is redirected INPUT or OUTPUT.
}
FUNCTION FileName(VAR f) : PathStr;

  BEGIN { FileName }
    FileName := StrName(FileRec(f).Name);
  END { FileName };


FUNCTION Str0(x : BYTE) : STRING2;

  BEGIN { Str0 }
    Str0 := CHR(ORD('0')+(x DIV 10))+CHR(ORD('0')+(x MOD 10));
  END { Str0 };


FUNCTION StrBYTE(x, len : BYTE) : STRING8;

  VAR
    Result : STRING8;

  BEGIN { StrBYTE }
    Str(x:len, Result);
    StrBYTE := Result;
  END { StrBYTE };


FUNCTION StrWORD(x : WORD; len : BYTE) : STRING8;

  VAR
    Result : STRING8;

  BEGIN { StrWORD }
    Str(x:len, Result);
    StrWORD := Result;
  END { StrWORD };


FUNCTION StrLONGINT(x : LONGINT; len : BYTE) : STRING16;

  VAR
    Result : STRING16;

  BEGIN { StrLONGINT }
    Str(x:len, Result);
    StrLONGINT := Result;
  END { StrLONGINT };


FUNCTION Hex1(x : BYTE) : CHAR;

  BEGIN { Hex1 }
    x := x AND $F;
    IF x < 10 THEN
      Hex1 := CHR(ORD('0')+x)
    ELSE
      Hex1 := CHR(ORD('A')+x-10);
  END { Hex1 };


FUNCTION Hex2(x : BYTE) : STRING2;

  BEGIN { Hex2 }
    Hex2 := Hex1(x SHR 4)+Hex1(x AND $F);
  END { Hex2 };


FUNCTION Hex4(x : WORD) : STRING4;

  BEGIN { Hex4 }
    Hex4 := Hex2(x SHR 8)+Hex2(x AND $FF);
  END { Hex4 };


FUNCTION Hex8(x : LONGINT) : STRING8;

  BEGIN { Hex8 }
    Hex8 := Hex4((x SHR 16) AND $FFFF)+Hex4(x AND $FFFF);
  END { Hex8 };


FUNCTION Hex9(x : POINTER) : STRING9;

  BEGIN { Hex9 }
    Hex9 := Hex4(SEG(x^))+':'+Hex4(OFS(x^));
  END { Hex 9 };


PROCEDURE ValHex(s : STRING16; VAR v : LONGINT; VAR code : INTEGER);

  VAR
    c	 : CHAR;
    Done : BOOLEAN;

  BEGIN { ValHex }
    s	 := s + '?';
    v	 := 0;
    code := 0;
    Done := FALSE;
    REPEAT
      INC(code);
      c := s[code];
      IF (c >= '0') AND (c <= '9') THEN
	v := (v*$10) + ORD(c)-ORD('0')
      ELSE
      IF (c >= 'A') AND (c <= 'F') THEN
	v := (v*$10) + ORD(c)-ORD('A')+10
      ELSE
      IF (c >= 'a') AND (c <= 'f') THEN
	v := (v*$10) + ORD(c)-ORD('a')+10
      ELSE
	Done := TRUE;
    UNTIL Done;
    IF (code >= Length(s)) AND (Length(s) > 1) THEN
      code := 0;
  END { ValHex };


PROCEDURE KeyDiscard;

  VAR
    Dummy : CHAR;

  BEGIN { KeyDiscard }
    Dummy := ReadKey;
    IF Dummy = CHR(0) THEN
      Dummy := ReadKey;
  END { KeyDiscard };
{
  Display a stack trace. This follows the dynamic link up the stack and
  lists all call addresses found.  Determining whether a call is near or far
  is not a well defined operation - if it's not sure a '?' is displayed.
  If it's a near call then 'Next' is displayed, implying that the segment
  part of the address is the same as the address below it in the trace.
  The addresses displayed can be used with TPC/F to determine where in the
  source the routines were called.
}
PROCEDURE StackDump;

  CONST
    CallFar	   = $9A;
    PushCSCallNear = $E80E;

  VAR
    StackSize,
    Frame,
    FrameNext,
    CallOfs,
    CallSeg : WORD;
    NearCall,
    NotSure : BOOLEAN;

  BEGIN { StackDump }
    WriteLn('Stack Dump');
    WriteLn(' BP   Seg Ofs');
    StackSize := (OvrHeapOrg-SSEG) SHL 4;
    FrameNext := GetBP;
    REPEAT
      Frame	:= FrameNext;
      FrameNext := MemW[SSEG:Frame  ];
      CallOfs	:= MemW[SSEG:Frame+2];
      CallSeg	:= MemW[SSEG:Frame+4];
      NearCall	:= (CallSeg < (PrefixSeg+PrefixParaSize)) OR
		   (CallSeg >= SSEG) OR
		   (CallOfs < 5) OR
		   ((Frame+6) > StackSize);
      NotSure	:= (NOT NearCall) AND
		   (Mem [CallSeg:CallOfs-5] <> CallFar) AND
		   (MemW[CallSeg:CallOfs-4] <> PushCSCallNear);
      Write(Hex4(Frame), ' ');
      IF NearCall THEN
	Write('Next')
      ELSE
	Write(Hex4(CallSeg-PrefixSeg-PrefixParaSize));
      Write(':', Hex4(CallOfs-2));
      IF NotSure THEN
	Write('?');
      WriteLn;
    UNTIL (FrameNext <= Frame) OR ((FrameNext+4) > StackSize);
    IF NearCall THEN
      WriteLn('     0000');
  END { StackDump };

{
  FatalError must be FAR called by a FAR called routine to work properly.
  The error address displayed is the caller of the caller of FatalError.
}
{$IFOPT F-}
{$DEFINE FMINUS}
{$F+}
{$ENDIF}
PROCEDURE FatalError(Status : INTEGER; Op : STRING48; Arg : STRING);

  VAR
    pW	 : ^WORD;
    pP	 : ^POINTER;
    Dir  : DirStr;
    Name : NameStr;
    Ext  : ExtStr;

  BEGIN { FatalError }
    FN := pSTRING(@OutName);
    Close(OUTPUT);
    AssignCRT(OUTPUT);
    ReWrite(OUTPUT);
    FSplit(ParamStr(0), Dir, Name, Ext);
    Write(Name, ': Fatal');
    IF Status < 0 THEN
      Write(' DOS')
    ELSE
    IF Status > 0 THEN
      Write(' network');
    WriteLn(' error.');
    IF Status <> 0 THEN
      BEGIN
	Write('Status:      ');
	IF Status > 0 THEN
	  IF Status >= 128 THEN
	    WriteLn(NetErrorMsg(Status))
	  ELSE
	    WriteLn(IOErrorMsg(Status))
	ELSE
	  WriteLn(IOErrorMsg(-Status));
      END;
    IF Op <> '' THEN
      WriteLn('Operation:   ', Op);
    IF Arg <> '' THEN
      WriteLn('Arguments:   ', Arg);
    pW := PTR(SSEG, GetBP); { Pointer to NOS routines's BP }
    pP := PTR(SSEG, pW^+2); { Pointer to caller of NOS routine's address }
    pP := pP^;		    { NOS routine address now in pP }
    pP := PTR(SEG(pP^)-PrefixSeg-PrefixParaSize, OFS(pP^)-2);
			    { Relative to program start, can use TPC /F }
    WriteLn('Address:     ', Hex9(pP));
{$IFDEF DEBUG}
    StackDump;
{$ENDIF}
    FN := NIL;
    Halt(0);
  END { FatalError };
{
  Undocumented MSDOS 2+ function to set the command line switch character.
  Usually "/" but might sometimes be "-" or something odd in a non-English
  speaking country.
}
PROCEDURE SetSwitchChar(x : CHAR);

  VAR
    R : Registers;

  BEGIN { SetSwitchChar }
    WITH R DO
      BEGIN
	AX := $3701;
	DL := ORD(x);
	MSDOS(R);
      END;
  END { SetSwitchChar };
{
  Undocumented MSDOS 2+ function to get the command line switch character.
}
FUNCTION GetSwitchChar : CHAR;

  VAR
    R : Registers;

  BEGIN { GetSwitchChar }
    WITH R DO
      BEGIN
	AX := $3700;
	MSDOS(R);
	IF (Flags AND FCarry) <> 0 THEN
	  DL := ORD('/'); { Default }
	GetSwitchChar := CHR(DL);
      END;
  END { GetSwitchChar };
{
  Undocumented MSDOS 3+ function to convert a file path to canonical form.
}
FUNCTION CanonicalPath(x : STRING128) : STRING128;

  VAR
    R	   : Registers;
    Result : STRING128;

  BEGIN { CanonicalPath }
    WITH R DO
      BEGIN
	x  := x + CHR(0);
	AH := $60;
	DS := SEG(x[1]);
	SI := OFS(x[1]);
	ES := SEG(Result[1]);
	DI := OFS(Result[1]);
	MSDOS(R);
	IF (Flags AND FCarry) <> 0 THEN
	  FatalError(-AX, 'Canonical Path', x);
	CanonicalPath := StrName(Result[1]);
      END;
  END { CanonicalPath };


PROCEDURE SelectDisk(Drive : DriveType);

  VAR
    R : Registers;

  BEGIN { SelectDisk }
    WITH R DO
      BEGIN
	AH := $0E;
	DL := ORD(Drive)-ORD('A');
	MSDOS(R); { No defined error return }
      END;
  END { SelectDisk };


FUNCTION GetCurrentDisk : DriveType;

  VAR
    R : Registers;

  BEGIN { GetCurrentDisk }
    WITH R DO
      BEGIN
	AH := $19;
	MSDOS(R); { No defined error return }
	GetCurrentDisk := CHR(AL+ORD('A'));
      END;
  END { GetCurrentDisk };


PROCEDURE ChDirCheck(Path : PathStr);

  VAR
    Result : WORD;

  BEGIN { ChDirCheck }
    ChDir(Path);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'Change Directory', Path);
  END { ChDirCheck };
{
  Create a new file, failing if it already exists.

  This is necessary in a network where multiple stations may all try to see if
  a file exists and create it at the same time. Using ReWrite there is a race
  condition if two stations test and decide a file doesn't exist
  simultaneously and then both try to create simultaneously.

  Returns TRUE if file has just been created, FALSE if it already exists and
  terminates the program for any other error. Arguments are the same as
  ReWrite.
}
FUNCTION CreateNewFile(VAR f : FILE; RecSizeArg : WORD) : BOOLEAN;

  VAR
    R : Registers;

  BEGIN { CreateNewFile }
    WITH R, FileRec(f) DO
      BEGIN
	IF Mode <> fmClosed THEN
	  FatalError(-102, 'Create New File', StrName(Name));
	AH   := $5B; { Create New File }
	AL   := FileMode; { Not documented but just in case }
	CX   := FileAttr;
	DS   := SEG(Name);
	DX   := OFS(Name);
	MSDOS(R);
	IF (Flags AND FCarry) <> 0 THEN
	  IF AX = 80 THEN { File already exists }
	    CreateNewFile := FALSE
	  ELSE
	    FatalError(-AX, 'Create New File', StrName(Name))
	ELSE
	  BEGIN
	    Handle	  := AX;
	    Mode	  := fmInOut;
	    RecSize	  := RecSizeArg;
	    FillChar(Private, SIZEOF(Private), CHR(0));
	    CreateNewFile := TRUE;
	  END;
      END;
  END { CreateNewFile };
{
  Version of ReWrite which uses the FileMode and FileAttr variables.
  FileMode is not documented to work, but just in case.
}
PROCEDURE ReWriteCheck(VAR f : FILE; RecSizeArg : WORD);

  VAR
    R : Registers;

  BEGIN { ReWriteCheck }
    WITH R, FileRec(f) DO
      BEGIN
	IF (Mode = fmOutput) OR (Mode = fmInput) OR (Mode = fmInOut) THEN
	  CloseCheck(f);
	IF Mode <> fmClosed THEN
	  FatalError(-102, 'Rewrite', StrName(Name));
	IF Name[0] = CHR(0) THEN { Opening standard output - '' }
	  AX := 1
	ELSE
	  BEGIN
	    AH := $3C; { Create Handle }
	    AL := FileMode; { Not documented but just in case }
	    CX := FileAttr;
	    DS := SEG(Name);
	    DX := OFS(Name);
	    MSDOS(R);
	    IF (Flags AND FCarry) <> 0 THEN
	      FatalError(-AX, 'Rewrite', StrName(Name));
	  END;
	Handle	:= AX;
	Mode	:= fmInOut;
	RecSize := RecSizeArg;
	FillChar(Private, SIZEOF(Private), CHR(0));
      END;
  END { ReWriteCheck };
{
  All the standard Turbo routines for untyped files.
  The only difference is that if they fail they give a more meaningful
  error message which includes the file name.
}
{$IFOPT I+}
{$DEFINE IPLUS}
{$I-}
{$ENDIF}
PROCEDURE AssignCheck(VAR f : FILE; Path : PathStr);

  VAR
    Result : WORD;

  BEGIN { AssignCheck }
    Assign(f, Path);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'Assign', Path);
  END { AssignCheck };


PROCEDURE ResetCheck(VAR f : FILE; RecSize : WORD);

  VAR
    Result : WORD;

  BEGIN { ResetCheck }
    Reset(f, RecSize);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'Reset', FileName(f));
  END { ResetCheck };


PROCEDURE CloseCheck(VAR f : FILE);

  VAR
    Result : WORD;

  BEGIN { CloseCheck }
    Close(f);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'Close', FileName(f));
  END { CloseCheck };


PROCEDURE BlockReadCheck(VAR f : FILE; VAR Buf; BufSize : WORD);

  VAR
    Result : WORD;

  BEGIN { BlockReadCheck }
    BlockRead(f, Buf, BufSize);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'Block Read', FileName(f));
  END { BlockReadCheck };


PROCEDURE BlockWriteCheck(VAR f : FILE; VAR Buf; BufSize : WORD);

  VAR
    Result : WORD;

  BEGIN { BlockWriteCheck }
    BlockWrite(f, Buf, BufSize);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'Block Write', FileName(f));
  END { BlockWriteCheck };


PROCEDURE RenameCheck(VAR f : FILE; Path : PathStr);

  VAR
    Result : WORD;

  BEGIN { RenameCheck }
    Rename(f, Path);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'Rename', FileName(f)+' to '+Path);
  END { RenameCheck };


PROCEDURE SeekCheck(VAR f : FILE; p : LONGINT);

  VAR
    Result : WORD;

  BEGIN { SeekCheck }
    Seek(f, p);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'Seek', FileName(f)+' to '+StrLONGINT(p,0));
  END { SeekCheck };


FUNCTION FilePosCheck(VAR f : FILE) : LONGINT;

  VAR
    Result : WORD;

  BEGIN { FilePosCheck }
    FilePosCheck := FilePos(f);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'File Pos', FileName(f));
  END { FilePosCheck };


FUNCTION FileSizeCheck(VAR f : FILE) : LONGINT;

  VAR
    Result : WORD;

  BEGIN { FileSizeCheck }
    FileSizeCheck := FileSize(f);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'File Size', FileName(f));
  END { FileSizeCheck };


FUNCTION EOFCheck(VAR f : FILE) : BOOLEAN;

  VAR
    Result : WORD;

  BEGIN { EOFCheck }
    EOFCheck := EOF(f);
    Result := IOResult;
    IF Result <> 0 THEN
      FatalError(-Result, 'EOF', FileName(f));
  END { EOFCheck };
{
  Routine to open a file for read in a multi-tasking environment where
  there may be multiple processes trying to read the file and one trying
  to write to the file.
}
PROCEDURE ResetCheckShare(VAR f : FILE; RecSize : WORD);

  VAR
    FMsave : BYTE;

  BEGIN { ResetCheckShare }
    FMsave   := FileMode;
    FileMode := ShareDenyNone+AccessRead;
    ResetCheck(f, RecSize);
    FileMode := FMsave;
  END { ResetCheckShare };
{
  Routine to open a file for write in a multi-tasking environment where
  there may be multiple processes trying to read the file. The file share
  attribute is enabled. This will fail on a local disk which doesn't support
  the Shareable attribute.
}
PROCEDURE ReWriteCheckShare(VAR f : FILE; RecSize : WORD);

  VAR
    FAsave : WORD;
    FMsave : BYTE;

  BEGIN { ReWriteCheckShare }
    FAsave   := FileAttr;
    FMsave   := FileMode;
    FileAttr := Shareable;
    FileMode := ShareDenyWrite+AccessWrite;
    ReWriteCheck(f, RecSize);
    FileAttr := FAsave;
    FileMode := FMsave;
  END { ReWriteCheckShare };


PROCEDURE EraseIfPresent(VAR f : FILE);

  VAR
    Dummy : WORD;

  BEGIN { EraseIfPresent }
    Erase(f); { Ignore errors }
    Dummy := IOResult; { Reset IO system if error }
  END { EraseIfPresent };
{$IFDEF IPLUS}
{$UNDEF IPLUS}
{$I+}
{$ENDIF}
{$IFDEF FMINUS}
{$UNDEF FMINUS}
{$F-}
{$ENDIF}

{
  Encode a string so that it is not easy to read from the application
  binary using a debugger.
}
  FUNCTION EncodeSTRING(x : STRING) : STRING;

    VAR
      I,
      c : BYTE;

    BEGIN { EncodeSTRING }
      c := $5A;
      FOR I := 1 TO Length(x) DO
	BEGIN
	  c    := ORD(UpCase(x[I])) XOR c;
	  x[I] := CHR(c);
	END;
      EncodeSTRING := x;
    END { EncodeSTRING };
{
  Inverse to EncodeSTRING. Used when unlocking an application.
}
  FUNCTION DecodeSTRING(x : STRING) : STRING;

    VAR
      I,
      c,
      d : BYTE;

    BEGIN { DecodeSTRING }
      c := $5A;
      FOR I := 1 TO Length(x) DO
	BEGIN
	  d    := ORD(x[I]);
	  x[I] := CHR(ORD(x[I]) XOR c);
	  c    := d;
	END;
      DecodeSTRING := x;
    END { DecodeSTRING };
{
  Calculate EXE file size from header values. If this is less than the
  true file size then the file probably contains overlays.
}
  FUNCTION HeaderFileSize(VAR Header : EXEHeader) : LONGINT;

    BEGIN { HeaderFileSize }
      WITH Header DO
	HeaderFileSize := (LONGINT(PageFileSize)-1)*LONGINT(PageSize)
	  +LONGINT(LastPageSize);
    END { HeaderFileSize };
{
  Display the contents of an EXE file's header. Used for debugging.
}
  PROCEDURE DumpHeader(VAR f : FILE; VAR Header : EXEHeader);

    VAR
      RelEnt	   : POINTER;
      CalcFileSize : LONGINT;
      I 	   : WORD;

    BEGIN { DumpHeader }
      WITH Header DO
	BEGIN
	  Write  ('Magic number:                 ', Hex4(Magic));
	  IF Magic <> $5A4D THEN
	    Write(' Bad!');
	  WriteLn;
	    Write  ('Last page size (bytes):       ', LastPageSize);
	 IF (LastPageSize < 0) OR (LastPageSize > PageSize) THEN
	  Write(' Bad!');
	  WriteLn;
	  Write('File size (512 byte pages):   ', PageFileSize);
	  CalcFileSize := HeaderFileSize(Header);
	  IF CalcFileSize <> FileSize(f) THEN
	    Write(' File size: ', CalcFileSize, '(Calc) <> ',
	    FileSize(f), '(Actual)');
	  WriteLn;
	  WriteLn('Number of relocation entries: ', NumReloc);
	  WriteLn('Header size (paragraphs):     ', HeadSize);
	  WriteLn('Minimum paragraphs required:  ', MinSize);
	  WriteLn('Maximum paragraphs required:  ', MaxSize);
	  WriteLn('Initial SS:                   ', StartSS);
	  WriteLn('Initial SP:                   ', StartSP);
	  WriteLn('Check sum:                    ', CheckSum);
	  WriteLn('Initial IP:                   ', StartIP);
	  WriteLn('Initial CS:                   ', StartCS);
	  WriteLn('Offset to relocation table:   ', RelTab);
	  WriteLn('Overlay number:               ', OverlayNumber);
	  WriteLn;
	  WriteLn('     Entry Offset Segment');
	  SeekCheck(f, RelTab);
	  FOR I := 1 TO NumReloc DO
	    BEGIN
	      BlockRead(f, RelEnt, SIZEOF(RelEnt));
	      WriteLn(I:9, OFS(RelEnt^):7, SEG(RelEnt^):8);
	    END;
	END;
    END { DumpHeader };

  VAR
    ExitSave  : POINTER;

{$F+}
  PROCEDURE ExitHandler;

    BEGIN { ExitHandler }
      ExitProc := ExitSave;
      IF StackOnExit AND (ExitCode <> 0) THEN
	StackDump;
    END { ExitHandler };
{$F-}

BEGIN { CheckIO }
  FileAttr    := 0;
  StackOnExit := {$IFDEF DEBUG} TRUE {$ELSE} FALSE {$ENDIF};
  ExitSave    := ExitProc;
  ExitProc    := @ExitHandler;
END { CheckIO }.

