PROGRAM Activity(INPUT, OUTPUT);
{$I Options} 
{$F+}
{$M 8192,8192,16384}

{
  Copyright (C) 1991 Julian Byrne. All rights reserved.

  Title:        Activity Display.
  File name:    ACTIVITY.PAS
  Version:      1.00
  Usage:        activity
  Description:  Displays in graphical form Novell Netware (tm) system activity.
  Dependencies: See USES statement
  Author:       Julian Byrne
  Address:      Electrical and Computer Systems Engineering Department
                Monash University, Wellington Road, Clayton, Victoria, 3168
                Australia
  Internet:     julian.byrne@monash.edu.au.
  Other nets:   Quaterman & Hoskins "Notable Computer Networks"
                CACM Oct'86 pp932-971
  History:      90/ 5/ 1 Initial version
  Notes:
}

USES
  CRT,
  DOS,
  BGIDriv, { In Turbo directory - Was DRIVERS.PAS in previous Turbo version }
  BGIFont, { "                  - Was FONTS.PAS }
  ASCII,
  CheckIO,
  Graph,
  Error,
  NOS;

{$I Site}

CONST
  Eps              = 1e-6;      { For real number comparisons          }
  ConMax           = 100;       { Maximum number of connections        }
  MinYTop          = 4.0;       { Minimum Y axis maximum value         }
  TicksPerSec      = 18.2;      { System heartbeat interrupt (SysTime) }
  SecsPerTick      = 1/TicksPerSec;
  InvLn10          = 0.4342944819032518;
  k2to32           = 4294967296.0;

TYPE
{
  If this is compiled on a machine with an '87 then it will use it,
  otherwise the most efficient floating point subroutines are used.
}
  Scalar           = {$IFOPT N+} SINGLE {$ELSE} REAL {$ENDIF} ;
  SelDataType      = (dRead, dWrite, dReadWrite, dRequestPackets);
  SelPlotType      = (pCurrent, pMoving, pPeak, pTotal);
  SelDataLabelType = ARRAY[SelDataType] OF STRING[22];
  SelPlotLabelType = ARRAY[SelPlotType] OF STRING[10];
  ConnectType      = STRING[ConMax];
  ConInfoType      = ARRAY[1..ConMax] OF GCIReplyType;
  ConStatsType     = ARRAY[1..ConMax] OF GCUSReplyType;
  ArrayWORD        = ARRAY[1..ConMax] OF WORD;
  ArrayScalar      = ARRAY[1..ConMax] OF Scalar;
  ControlStateType = RECORD
                       SelData     : SelDataType;
                       SelPlot     : SelPlotType;
                       DelTime     : LONGINT;
                       Base        : ArrayScalar;
                       MyCon       : BYTE;
                       LogScale,
                       BaseValid,
                       ReBase,
                       ReScale,
                       Done        : BOOLEAN;
                     END;
  DataStateType    = RECORD
                       Control     : ControlStateType;
                       DataConnect : ConnectType;
                       ConInfo     : ConInfoType;
                       ConStats    : ConStatsType;
                       Abs,
                       Del,
                       Res         : ArrayScalar;
                       Valid       : BOOLEAN;
                     END;
  PlotStateType    = RECORD
                       Data        : DataStateType;
                       ColorTable  : ArrayWORD;
                       PlotConnect : ConnectType;
                       White,
                       xSize, ySize, zSize,
                       SizeBar,
                       SizeTitle,
                       SizeXLabel1,
                       SizeXLabel2, 
                       SizeYLabel1,
                       SizeYLabel2 : WORD;
                       Title,
                       YLabel      : STRING80;
                       ya, yb,
                       yScaleStep,
                       yScaleLabel,
                       YTop        : Scalar;
                       BarSize     : ArrayWORD;
                       Valid       : BOOLEAN;
                     END;

CONST
  SelDataLabel : SelDataLabelType =
    ('bytes read','bytes written','bytes read and written',
     'request packets sent');
  SelPlotLabel : SelPlotLabelType =
    ('Current', 'Moving av.', 'Peak', 'Total');

VAR
  SysTime       : LONGINT ABSOLUTE $0040:$006C; { Increments at 18.2/Sec }
  AbsTime       : LONGINT;
  ExitSave      : POINTER;
  InTextMode    : BOOLEAN;
  BroadcastMode : BYTE;
  Dummy         : INTEGER;
  PlotLast,
  Plot          : PlotStateType;
  Dir           : DirStr;
  Name          : NameStr;
  Ext           : ExtStr;


FUNCTION LongInt2Scalar(VAR x : LongInt2) : Scalar;

  BEGIN { LongInt2Scalar }
    LongInt2Scalar := (((WORD((x[0] SHL 8) OR x[1]))*65536.0+
                         WORD((x[2] SHL 8) OR x[3]))*65536.0+
                         WORD((x[4] SHL 8) OR x[5]));
  END { LongInt2Scalar };


PROCEDURE ZeroArrayScalar(N : WORD; VAR x : ArrayScalar);

  BEGIN { ZeroArrayScalar }
    FillChar(x, N*SIZEOF(Scalar), CHR(0));
  END { ZeroArrayScalar };


FUNCTION MaxArrayScalar(N : WORD; VAR x : ArrayScalar) : Scalar;

  VAR
    I      : WORD;
    Result : Scalar;

  BEGIN { MaxArrayScalar }
    Result := 0.0;
    FOR I := 1 TO N DO
      IF Result < x[I] THEN
        Result := x[I];
    MaxArrayScalar := Result;
  END { MaxArrayScalar };


PROCEDURE Max2ArrayScalar(N : WORD; VAR x, y, z : ArrayScalar);

  VAR
    I : WORD;

  BEGIN { Max2ArrayScalar }
    FOR I := 1 TO N DO
      IF x[I] > y[I] THEN
        z[I] := x[I]
      ELSE
        z[I] := y[I];
  END { Max2ArrayScalar };


FUNCTION Power2Bound(x : Scalar) : Scalar;

  VAR
    Result : Scalar;

  BEGIN { Power2Bound }
    Result := 1.0;
    WHILE Result < x DO
      Result := 2.0 * Result;
    Power2Bound := Result;
  END { Power2Bound };


FUNCTION Power10LowBound(x : Scalar) : Scalar;

  VAR
    Result : Scalar;

  BEGIN { Power10LowBound }
    Result := 1.0;
    WHILE Result < x DO
      Result := 10.0 * Result;
    Power10LowBound := 0.1 * Result ;
  END { Power10LowBound };


PROCEDURE SubArrayScalar(N : WORD; VAR x, y, z : ArrayScalar);

  VAR
    I : WORD;

  BEGIN { SubArrayScalar }
    FOR I := 1 TO N DO
      z[I] := x[I] - y[I];
  END { SubArrayScalar };


PROCEDURE InitScales(VAR Plot : PlotStateType);

  VAR
    I : WORD;

  BEGIN { InitScales }
    WITH Plot DO
      BEGIN
        Valid := TRUE;
        xSize := GetMaxX+1;
        ySize := GetMaxY+1;
        zSize := GetMaxColor+1;
        White := zSize-1;
        IF zSize <= 2 THEN
          FOR I := 1 TO ConMax DO
            ColorTable[I] := White
        ELSE
          FOR I := 1 TO ConMax DO
            ColorTable[I] := (I MOD (zSize-2))+1;
        SetTextStyle(SmallFont, HorizDir, 4);
        SetUserCharSize(2, 1, 2, 1);
        SetTextJustify(CenterText, TopText);
        SizeTitle := TextHeight('Pp');
        SizeTitle := SizeTitle + (SizeTitle SHR 1);

        SetTextStyle(SmallFont, VertDir, 4); 
        SetUserCharSize(2, 1, 2, 1); 
        SetTextJustify(CenterText, TopText); 
        SizeYLabel1 := TextWidth('Pp'); 
        SizeYLabel1 := SizeYLabel1 + (SizeYLabel1 SHR 1);

        SetTextStyle(SmallFont, HorizDir, 1);
        SetUserCharSize(1, 1, 1, 1);
        SetTextJustify(RightText, CenterText);
        SizeYLabel2 := SizeYLabel1 + TextWidth('8888')+4;
      END;
  END { InitScales };


{$IFOPT F+}
{$DEFINE FPLUS}
{$ELSE}
{$F+}
{$ENDIF}
  PROCEDURE GraphEnd;

    BEGIN { GraphEnd }
      ExitProc   := ExitSave;
      ClearDevice;
      Delay(40);
      CloseGraph;
      BroadcastMode := SetBroadcastMode(BroadcastMode); { Restore broadcasts }
      InTextMode    := TRUE;
    END { GraphEnd };
{$IFDEF FPLUS}
{$UNDEF FPLUS}
{$ELSE}
{$F-}
{$ENDIF}

PROCEDURE CheckGraph;

  VAR
    Result : INTEGER;

  BEGIN { CheckGraph }
    Result := GraphResult;
    IF Result <> 0 THEN
      BEGIN
        IF NOT InTextMode THEN
          GraphEnd;
        Write('CheckGraph: ');
        Halt(WORD(Result));
      END;
  END { CheckGraph };


PROCEDURE GraphBegin(VAR PlotLast, Plot : PlotStateType);

  VAR
    I,
    GraphDriver,
    GraphMode    : INTEGER;

  BEGIN { GraphBegin }
    ExitSave      := ExitProc;
    ExitProc      := @GraphEnd;
    Graphdriver   := Detect;
    GraphMode     := 0;
    BroadcastMode := SetBroadcastMode(3); { Turn off broadcasts }
    ClrScr;
    Delay(40);
    InitGraph(GraphDriver, GraphMode, Dir);
    CheckGraph;
    InTextMode     := FALSE;
    PlotLast.Valid := FALSE;
    InitScales(Plot);
    SetBkColor(BLACK);
    SetColor(Plot.White);
    ClearDevice;
    Rectangle(0, 0, Plot.xSize-1, Plot.ySize-1);
    CheckGraph;
    AbsTime := SysTime;
  END { GraphBegin };


FUNCTION GetMyConnectionNum : BYTE;

  VAR
    p : pConIDTable;

  BEGIN { GetMyConnectionNum }
    p                  := GetConnectionIDTable;
    GetMyConnectionNum := p^[GetDefaultConnectionID].ConnectionNum;
  END { GetMyConnectionNum };


PROCEDURE GetConnectionStats(    MyCon       : BYTE;
                             VAR Connections : ConnectType;
                             VAR ConInfo     : ConInfoType;
                             VAR ConStats    : ConStatsType);

  VAR
    I, J      : WORD;
    GFSIReply : GFSIReplyType;

  BEGIN { GetConnectionStats }
    Connections := '';
    GetFileServerInformation(GFSIReply);
    I := 1;
    J := Swap(GFSIReply.ConnectPeak);
    FOR I := 1 TO J DO
      BEGIN
        IF (I <> MyCon) THEN
          BEGIN
            GetConnectionInformation(I, ConInfo[I]);
            IF (ConInfo[I].ObjID <> 0) THEN
              BEGIN
                Connections := Connections + CHR(I);
                GetConnectionUsageStatistics(I, ConStats[I]);
              END;
          END
        ELSE
          ConInfo[I].ObjID := 0;
      END;
    FOR I := J+1 TO ConMax DO
      ConInfo[I].ObjID := 0;
  END { GetConnectionStats };


PROCEDURE DumpConnectionStats(VAR Connections : ConnectType;
                              VAR ConInfo     : ConInfoType;
                              VAR ConStats    : ConStatsType);

  VAR
    I, J : WORD;

  BEGIN { DumpConStats }
    FOR I := 1 TO Length(Connections) DO
      BEGIN
        J := ORD(Connections[I]);
        WITH ConInfo[J], ConStats[J] DO
          BEGIN
            WriteLn(J:2,
              ' ', Swap(ObjType):3,
              ' ', StrName48(ObjName):10,
              ' ', StrTime(LoginTime),
              ' ', Swap4(ElapsedTime):10,
              ' ', StrLongInt2(BytesRead,8),
              ' ', StrLongInt2(BytesWritten,8),
              ' ', Swap4(RequestPackets):6);
          END;
      END;
    WriteLn;
  END { DumpConStats };


PROCEDURE InitControl(VAR Control : ControlStateType);

  BEGIN { InitControl }
    WITH Control DO
      BEGIN
        SelData   := dReadWrite;
        SelPlot   := pTotal;
        DelTime   := 73; { 4 seconds }
        MyCon     := GetMyConnectionNum;
        LogScale  := FALSE;
        BaseValid := FALSE;
        ReBase    := FALSE;
        ReScale   := FALSE;
        Done      := FALSE;
      END;
  END { InitControl };


PROCEDURE InitData(VAR Data : DataStateType);

  BEGIN { InitData }
    WITH Data DO
      BEGIN
        InitControl(Control);
        Valid := TRUE;
      END;
  END { InitData };


PROCEDURE DeriveData(VAR DataLast,
                         Data    : DataStateType);

  VAR
    I, J : WORD;
    T    : BOOLEAN;
    DT   : LONGINT;

  BEGIN { DeriveData }
    WITH Data, Control DO
      BEGIN
        IF DataLast.Valid AND (
          (SelData <> DataLast.Control.SelData) OR
          (SelPlot <> DataLast.Control.SelPlot)) THEN
          DataLast.Valid := FALSE;
        ZeroArrayScalar(ConMax, Abs);
        CASE SelData OF
          dRead      :
            FOR I := 1 TO Length(DataConnect) DO
              BEGIN
                J      := ORD(DataConnect[I]);
                Abs[J] := Longint2Scalar(ConStats[J].BytesRead   );
              END;
          dWrite     :
            FOR I := 1 TO Length(DataConnect) DO
              BEGIN
                J      := ORD(DataConnect[I]);
                Abs[J] := Longint2Scalar(ConStats[J].BytesWritten);
              END;
          dReadWrite :
            FOR I := 1 TO Length(DataConnect) DO
              BEGIN
                J      := ORD(DataConnect[I]);
                Abs[J] := LongInt2Scalar(ConStats[J].BytesRead   )+
                          LongInt2Scalar(ConStats[J].BytesWritten);
              END; 
          dRequestPackets :
            FOR I := 1 TO Length(DataConnect) DO
              BEGIN
                J      := ORD(DataConnect[I]);
                Abs[J] := Swap4(ConStats[J].RequestPackets);
                IF Abs[J] < 0.0 THEN
                  Abs[J] := Abs[J] + k2to32;
              END;
        END;
        CASE SelPlot OF
          pCurrent,
          pMoving,
          pPeak :
            BEGIN
              ZeroArrayScalar(ConMax, Del);
              IF DataLast.Valid THEN
                FOR I := 1 TO Length(DataConnect) DO
                  BEGIN
                    J := ORD(DataConnect[I]);
                    IF DataLast.ConInfo[J].ObjID <> 0 THEN
                      BEGIN
                        DT := Swap4(ConStats[J].ElapsedTime)-
                              Swap4(DataLast.ConStats[J].ElapsedTime);
                        IF DT <> 0 THEN
                          BEGIN
                            Del[J] := (Abs[J] - DataLast.Abs[J])*TicksPerSec/DT;
                            IF Del[J] < 0.0 THEN { Connection reused }
                              Del[J] := 0.0;
                          END
                        ELSE
                          Del[J] := 0.0;
                      END
                    ELSE
                      BEGIN
                        DT := Swap4(ConStats[J].ElapsedTime);
                        IF DT <> 0 THEN
                          BEGIN
                            Del[J] := Abs[J]*TicksPerSec/DT;
                            IF Del[J] < 0.0 THEN { Time wrap around }
                              Del[J] := 0.0;
                          END
                        ELSE
                          Del[J] := 0.0;
                      END;
                  END;
            END;
          pTotal :
            Del := Abs;
        END;
        IF ReBase THEN
          BEGIN
            ReBase := FALSE;
            CASE SelPlot OF
              pCurrent,
              pMoving  : ;
              pPeak    : BEGIN
                           BaseValid := TRUE;
                           ZeroArrayScalar(ConMax, Base);
                         END;
              pTotal   : BEGIN
                           BaseValid := TRUE;
                           Base      := Abs;
                         END;
            END;
          END;
        CASE SelPlot OF
          pCurrent : ;
          pMoving  : IF DataLast.Valid THEN
                       FOR I := 1 TO Length(DataConnect) DO
                         BEGIN
                           J := ORD(DataConnect[I]);
                           IF DataLast.ConInfo[J].ObjID <> 0 THEN
                             Del[J] := 0.5*Del[J]+0.5*DataLast.Del[J];
                         END;
          pPeak    : IF BaseValid THEN
                       FOR I := 1 TO Length(DataConnect) DO
                         BEGIN
                           J := ORD(DataConnect[I]);
                           IF Base[J] < Del[J] THEN
                             Base[J] := Del[J]
                           ELSE
                             Del[J] := Base[J];
                         END;
          pTotal   : IF BaseValid THEN
                       FOR I := 1 TO Length(DataConnect) DO
                         BEGIN
                           J := ORD(DataConnect[I]);
                           IF Base[J] > (Del[J]+Eps) THEN { Connection reused }
                             Base[J] := 0.0;
                           Del[J] := Del[J] - Base[J];
                         END;
        END;
        IF LogScale THEN
          BEGIN
            ZeroArrayScalar(ConMax, Res);
            FOR I := 1 TO Length(DataConnect) DO
              BEGIN
                J := ORD(DataConnect[I]);
                IF Del[J] >= 1.0 THEN
                  Res[J] := Ln(Del[J])*InvLn10;
              END;
          END
        ELSE
          Res := Del;
      END;
  END { DeriveData };


PROCEDURE GraphScales(VAR PlotLast, Plot : PlotStateType);

  VAR
    I, J : WORD;
    S    : STRING[20];

  BEGIN { GraphScales }
    IF InTextMode THEN
      GraphBegin(PlotLast, Plot);
    WITH Plot DO
      BEGIN
        Str(Data.Control.DelTime*SecsPerTick:3:1, Title);
        Title  := SelPlotLabel[Data.Control.SelPlot]+' '+
          SelDataLabel[Data.Control.SelData]+', '+
          Title+' s/sample';

        PlotConnect := Data.DataConnect;

        SetTextStyle(SmallFont, VertDir, 4);
        SetUserCharSize(1, 1, 1, 1);
        SetTextJustify(CenterText, TopText);
        SizeXLabel2 := 0;
        FOR I := 1 TO Length(Data.DataConnect) DO
          WITH Data.ConInfo[ORD(Data.DataConnect[I])] DO
            BEGIN
              SizeXLabel1 := TextWidth(StrName48(ObjName));
              IF SizeXLabel1 > SizeXLabel2 THEN
                SizeXLabel2 := SizeXLabel1;
            END;
        SizeXLabel1 := TextWidth(' 88');
        SizeXLabel2 := SizeXLabel2 + SizeXLabel1 + 5 + SizeTitle;

        SizeBar := ((xSize-SizeYLabel2-2) DIV Length(PlotConnect));
        I       := (xSize-2) DIV 10;
        IF (SizeBar > I) THEN
          SizeBar := I;

        YTop := Power2Bound(MaxArrayScalar(ConMax, Data.Res));
        IF YTop < MinYTop THEN
          YTop := MinYTop;
        IF (NOT Data.Control.ReScale) AND PlotLast.Valid AND
          (PlotLast.YTop > (YTop+Eps)) THEN
          YTop := PlotLast.YTop;
        Data.Control.ReScale := FALSE;

        ya          := -(ySize-2-SizeXLabel2) / YTop;
        yb          := ySize-1-SizeXLabel2;
        yScaleLabel := 1.0;
        WHILE YTop > (4e3 * yScaleLabel) DO
          yScaleLabel := 1e3 * yScaleLabel;
        yScaleStep := Power10LowBound(YTop);
        IF yScaleStep*3 > YTop THEN
          yScaleStep := yScaleStep * 0.1;

        IF Data.Control.SelData = dRequestPackets THEN
          YLabel := 'Packets'
        ELSE
          YLabel := 'Bytes';
        IF Data.Control.SelPlot <> pTotal THEN
          YLabel := YLabel + '/s';
        IF Data.Control.LogScale THEN
          YLabel := YLabel + ' (x10^?)'
        ELSE
        IF ABS(yScaleLabel-1.0) > Eps THEN
          BEGIN
            Str(ln(yScaleLabel)*Invln10:1:0, S);
            YLabel := YLabel+' (x10^'+Copy(S,1,255)+')';
          END;

        FOR I := 1 TO Length(PlotConnect) DO
          BEGIN
            J          := ORD(PlotConnect[I]);
            BarSize[I] := ROUND(Data.Res[J]*ya+yb);
          END;
      END;
  END { GraphScales };


PROCEDURE GraphData(VAR PlotLast, Plot : PlotStateType);

  VAR
    S      : STRING12;
    I, J, 
    IX, IY : WORD;
    X, Y   : Scalar;
    Con    : BYTE;

  BEGIN { GraphData }
    WITH Plot DO
      BEGIN
        IF (NOT PlotLast.Valid) OR
          (PlotLast.SizeTitle <> SizeTitle) OR
          (PlotLast.Title <> Title) THEN { Title }
          BEGIN
            SetTextStyle(SmallFont, HorizDir, 4);
            SetUserCharSize(2, 1, 2, 1);
            SetTextJustify(CenterText, TopText);
            SetFillStyle(EmptyFill, BLACK);
            Bar(1, ySize-SizeTitle+1, xSize-2, ySize-2);
            SetColor(White);
            Line(1, ySize-SizeTitle, xSize-2, ySize-SizeTitle);
            OutTextXY(xSize DIV 2, ySize-SizeTitle+1, Title);
          END;
        IF (NOT PlotLast.Valid) OR
          (PlotLast.SizeTitle <> SizeTitle) OR
          (PLotLast.SizeBar <> SizeBar) OR
          (PlotLast.SizeXLabel1 <> SizeXLabel1) OR
          (PlotLast.SizeXLabel2 <> SizeXLabel2) OR
          (PlotLast.SizeYLabel2 <> SizeYLabel2) OR
          (PlotLast.PlotConnect <> PlotConnect) THEN { X axis labels }
          BEGIN
            Con := GetMyConnectionNum;
            IX  := SizeYLabel2 + (SizeBar DIV 2);
            IY  := ySize-1-SizeXLabel2;
            SetFillStyle(EmptyFill, BLACK);
            Bar(SizeYLabel2, IY+1, xSize-2, ySize-SizeTitle-1);
            SetTextStyle(SmallFont, VertDir, 4);
            SetUserCharSize(1, 1, 1, 1);
            SetTextJustify(CenterText, TopText);
            SetColor(White);
            FOR I := 1 TO Length(PlotConnect) DO
              BEGIN
                J := ORD(PlotConnect[I]);
                WITH Data.ConInfo[J] DO
                  BEGIN
                    S := StrWORD(J,0);
                    IF J = Con THEN
                      S := '*' + S;
                    OutTextXY(IX, IY+2, S);
                    OutTextXY(IX, IY+2+SizeXLabel1, StrName48(ObjName));
                    IX := IX + SizeBar;
                  END;
              END;
          END;
        IF (NOT PlotLast.Valid) OR
          (PlotLast.SizeTitle <> SizeTitle) OR
          (PlotLast.SizeXLabel2 <> SizeXLabel2) OR
          (PlotLast.SizeYLabel1 <> SizeYLabel1) OR
          (PlotLast.YLabel <> YLabel) THEN { Y axis title }
          BEGIN
            SetFillStyle(EmptyFill, BLACK);
            Bar(1, 1, SizeYLabel1-1, ySize-SizeTitle-1);
            SetTextStyle(SmallFont, VertDir, 4);
            SetUserCharSize(2, 1, 2, 1);
            SetTextJustify(LeftText, CenterText);
            OutTextXY(4, (ySize-1-SizeXLabel2) DIV 2, YLabel);
          END;
        IF (NOT PlotLast.Valid) OR
          (PlotLast.SizeTitle <> SizeTitle) OR
          (PlotLast.SizeXLabel2 <> SizeXLabel2) OR
          (PlotLast.SizeYLabel1 <> SizeYLabel1) OR
          (PlotLast.SizeYLabel2 <> SizeYLabel2) OR
          (ABS(PlotLast.YTop-YTop) > Eps) THEN { Y axis labels }
          BEGIN
            SetFillStyle(EmptyFill, BLACK);
            Bar(SizeYLabel1, 1, SizeYLabel2-2, ySize-SizeTitle-1);
            SetColor(White);
            Line(SizeYLabel2-1, 1, SizeYLabel2-1, ySize-SizeXLabel2-1);
            SetTextStyle(SmallFont, HorizDir, 4);
            SetUserCharSize(1, 1, 1, 1);
            SetTextJustify(RightText, CenterText);
            SetColor(White);
            Y  := 0.0;
            REPEAT
              Str(Y / yScaleLabel:3:0, S);
              IY := ROUND(ya*Y+yb);
              Line(SizeYLabel2-3, IY, SizeYLabel2-2, IY);
              OutTextXY(SizeYLabel2-4, IY, S);
              Y := Y + yScaleStep;
            UNTIL Y >= YTop;
          END;
        IF (NOT PlotLast.Valid) OR
          (PlotLast.SizeBar <> SizeBar) OR
          (PlotLast.SizeXLabel2 <> SizeXLabel2) OR
          (PlotLast.SizeYLabel2 <> SizeYLabel2) OR
          (Length(PlotLast.PlotConnect) <> Length(PlotConnect)) THEN { Data }
          BEGIN
            IX  := SizeYLabel2;
            SetFillStyle(EmptyFill, BLACK);
            Bar(SizeYLabel2, 1, xSize-2, ySize-1-SizeXLabel2);
            FOR I := 1 TO Length(PlotConnect) DO
              BEGIN
                SetFillStyle(SolidFill, ColorTable[I]);
                Bar(IX, BarSize[I], IX+SizeBar-1, ySize-1-SizeXLabel2);
                IX := IX + SizeBar;
              END;
          END
        ELSE
          BEGIN
            IX  := SizeYLabel2;
            FOR I := 1 TO Length(PlotConnect) DO
              BEGIN
                IF BarSize[I] < PlotLast.BarSize[I] THEN
                  BEGIN
                    SetFillStyle(SolidFill, ColorTable[I]);
                    Bar(IX, BarSize[I], IX+SizeBar-1, PlotLast.BarSize[I]-1);
                  END
                ELSE
                IF BarSize[I] > PlotLast.BarSize[I] THEN
                  BEGIN
                    SetFillStyle(EmptyFill, BLACK);
                    Bar(IX, PlotLast.BarSize[I], IX+SizeBar-1, BarSize[I]-1);
                  END;
                IX := IX + SizeBar;
              END
          END
      END;
  END { GraphData };


PROCEDURE Usage;

  CONST
    Indent1 = 23;
    Head1   = 'Novell Netware (tm) Activity Monitor';
    Head2   = SiteName;
    Head3   = 'Copyright 1st June 1991 Julian Byrne. All rights reserved.';
    Head4   = 'Press space bar to continue';

  BEGIN { Usage }
    IF NOT InTextMode THEN
      GraphEnd;
    WriteLn;
    WriteLn('':(80-Length(Head1)) DIV 2, Head1);
    WriteLn('':(80-Length(Head2)) DIV 2, Head2);
    WriteLn('':(80-Length(Head3)) DIV 2, Head3);
    WriteLn;
    WriteLn('':Indent1, '?   This help screen');
    WriteLn('':Indent1, 'Q   Quit');
    WriteLn('':Indent1, 'S   Scale reset');
    WriteLn('':Indent1, 'I   Include/exclude this connection');
    WriteLn('':Indent1, 'Z   Zero/unzero baseline');
    WriteLn('':Indent1, 'L   Log/linear scale');
    WriteLn('':Indent1, '-   Sample time: Decrease');
    WriteLn('':Indent1, '+                Increase');
    WriteLn('':Indent1, 'R   Statistics:  Read byte');
    WriteLn('':Indent1, 'W                Write byte');
    WriteLn('':Indent1, 'B                Both read and write byte');
    WriteLn('':Indent1, 'N                Network request packet');
    WriteLn('':Indent1, 'C   Display:     Current');
    WriteLn('':Indent1, 'M                Moving average');
    WriteLn('':Indent1, 'P                Peak');
    WriteLn('':Indent1, 'T                Total');
    WriteLn;
    WriteLn('':(80-Length(Head4)) DIV 2, Head4);
    Write('':40);
    REPEAT
    UNTIL KeyPressed;
    WriteLn;
    Write('':40);
  END { Usage };

 
PROCEDURE ProcessKey(VAR Control : ControlStateType);

  VAR
    I     : LONGINT;
    c, c2 : CHAR;

  BEGIN { ProcessKey }
    WITH Control DO
      BEGIN
        c := UpCase(ReadKey);
        IF c = NUL THEN
          c2 := ReadKey;
        CASE c OF
          NUL : IF c2 = NUL THEN
                  Done := TRUE
                ELSE
                  BEGIN
                    Write(BEL);
                    Usage;
                  END;
          ' ' : ;
          '?',
          'H' : Usage;
          ETX,
          ESC,
          'Q' : Done := TRUE; 
          'S' : ReScale := TRUE;
          'I' : BEGIN
                  IF MyCon = 0 THEN
                    MyCon := GetMyConnectionNum
                  ELSE
                    MyCon := 0;
                END;
          'Z' : BEGIN
                  ReBase    := (NOT BaseValid) OR (SelPlot <> pTotal);
                  BaseValid := FALSE;
                  ReScale   := TRUE;
                END;
          'L' : BEGIN
                  LogScale := NOT LogScale;
                  ReScale  := TRUE;
                END;
          '-' : BEGIN
                  I := ROUND(ROUND(DelTime*SecsPerTick)*TicksPerSec*0.5);
                  IF I >= 9 THEN
                    DelTime := I
                  ELSE
                    Write(BEL);
                END;
          '+' : DelTime :=
                  ROUND(ROUND(DelTime*(SecsPerTick*2.0))*TicksPerSec);
          'R' : IF SelData <> dRead THEN
                  BEGIN
                    SelData   := dRead;
                    ReBase    := SelPlot = pPeak;
                    BaseValid := FALSE;
                    ReScale   := TRUE;
                    AbsTime   := SysTime;
                  END
                ELSE
                  Write(BEL); 
          'W' : IF SelData <> dWrite THEN
                  BEGIN
                    SelData   := dWrite;
                    ReBase    := SelPlot = pPeak;
                    BaseValid := FALSE;
                    ReScale   := TRUE;
                    AbsTime   := SysTime;
                  END
                ELSE
                  Write(BEL);
          'B' : IF SelData <> dReadWrite THEN
                  BEGIN
                    SelData   := dReadWrite;
                    ReBase    := SelPlot = pPeak;
                    BaseValid := FALSE;
                    ReScale   := TRUE;
                    AbsTime   := SysTime;
                  END
                ELSE
                  Write(BEL);
          'N' : IF SelData <> dRequestPackets THEN
                  BEGIN
                    SelData   := dRequestPackets;
                    ReBase    := SelPlot = pPeak;
                    BaseValid := FALSE;
                    ReScale   := TRUE;
                    AbsTime   := SysTime;
                  END
                ELSE
                  Write(BEL);
          'C' : IF SelPlot <> pCurrent THEN
                  BEGIN
                    SelPlot := pCurrent;
                    ReScale := TRUE;
                    AbsTime := SysTime;
                  END
                ELSE
                  Write(BEL);
          'M' : IF SelPlot <> pMoving THEN
                  BEGIN
                    SelPlot := pMoving;
                    ReScale := TRUE;
                    AbsTime := SysTime;
                  END
                ELSE
                  Write(BEL);
          'P' : IF SelPlot <> pPeak THEN
                  BEGIN
                    SelPlot := pPeak;
                    ReScale := TRUE;
                    ReBase  := TRUE;
                    AbsTime := SysTime;
                  END
                ELSE
                  Write(BEL);
          'T' : IF SelPlot <> pTotal THEN
                  BEGIN
                    SelPlot := pTotal;
                    ReScale := TRUE;
                    AbsTime := SysTime;
                  END
                ELSE
                  Write(BEL);
        ELSE
          BEGIN
            Write(BEL);
            Usage;
          END;
        END { CASE c };
      END { WITH Control };
  END { ProcessKey };


BEGIN { Activity }
  FSplit(ParamStr(0), Dir, Name, Ext);
  IF ParamCount <> 0 THEN
    BEGIN
      WriteLn('Usage: ', Name);
      WriteLn('Enter ? once running for help.');
      Halt(0);
    END;
  IF NOT NovellAPI THEN
    BEGIN
      WriteLn(Name, ': Novell Netware (tm) required.');
      Halt(0);
    END;
  IF NOT CheckConsolePrivileges THEN
    BEGIN
      WriteLn(Name, ': Console privileges required.');
      Halt(0);
    END;
  PErrorMsg           := GraphErrorMsg;
  Dummy               := RegisterBGIdriver(@HercDriverProc);
  Dummy               := RegisterBGIdriver(@CGADriverProc);
  Dummy               := RegisterBGIDriver(@EGAVGADriverProc);
  Dummy               := RegisterBGIfont(@SmallFontProc);
  CheckGraph;
  AbsTime             := SysTime;
  ExitSave            := NIL;
  InTextMode          := TRUE;
  PlotLast.Valid      := FALSE;
  Plot.Valid          := FALSE;
  PlotLast.Data.Valid := FALSE;
  Plot.Data.Valid     := FALSE;
  InitData(Plot.Data);
  REPEAT
    REPEAT
      WHILE KeyPressed DO
        ProcessKey(Plot.Data.Control);
    UNTIL (SysTime >= AbsTime) OR Plot.Data.Control.Done;
    IF NOT Plot.Data.Control.Done THEN
      BEGIN
        WITH Plot.Data, Control DO
          GetConnectionStats(MyCon, DataConnect, ConInfo, ConStats);
        DeriveData(PlotLast.Data, Plot.Data);
        GraphScales(PlotLast, Plot);
        GraphData(PlotLast, Plot);
        AbsTime  := AbsTime + Plot.Data.Control.DelTime;
        IF SysTime > AbsTime THEN
          AbsTime := SysTime;
        PlotLast := Plot;
      END
  UNTIL Plot.Data.Control.Done;
END { Activity }.

