unit utils;

interface
(*
Unit: Utils
Description: Performs various utility operations, and provides a number
             of utility services.
Notes: 
References: References to the documentation is to the file index.doc
            and section numbers are given as relative to this document.
Revision History;
         18/12/1994 - Post pointer create check and polite shutdown
         29/03/1995 - Extensive commenting.
         30/03/1995 - Documented (utils.doc)
         27/04/1995 - Removed TEMP env. var. bug - Added user info for
                      file crashes
         05/06/1995 - Removed writes to stdoutput, instead to
                      crt directly. Much quicker, and no crash if CTRL-C
                      pressed.
         05/06/1995 - Fixed crash if output duplicate file locked.
         22/05/1996 - Placed under the GNU General Public License.
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)

{$O+}

uses dtypes;

{procedure check_avail_mem(size:word);}
procedure check_assign(ptr:pointer;
                       desc:string);
procedure fatal_error(errorno:word;
                      errorstr:string);
procedure nonfatal_error(errorno:word;
                         errorstr:string);
function get_token(text:string;
                   seperator:char):string;
function cut_token(var text:string;
                       seperator:char):string;
function cut_to_right_bracket(var s:string;
                                  bdepth:shortint):string;
function find_tokens(var s:string):string;
function get_first_expression(var text:string;
                              var boolval:byte):string;
procedure strip_leading_spaces(var s:string);
procedure strip_tail_spaces(var s:string);
procedure strip_leading_seperators(var s:string);
function pad_zeros(    c:word):string;
procedure pad(var str:string;
                  pad_size:word);
function generate_random_string(    n:word):string;
function get_datetime:string;
procedure report(    reportstring:string);
procedure report_shutdown;
procedure report_flush;
function upstring(    s:string):string;
procedure setup_stdout;
procedure write_Stdout(    s:string);
procedure writeln_stdout(    s:string);
procedure writeln_stdout_nl;
procedure writeln_stdoutcpy(    s:string);
procedure write_stdoutcpy(    s:string);
procedure stdout_close;
function contains(    str:string;
                      ch:char):boolean;
procedure process_param(var debug:boolean);
function brackets_ok(    s:string;
                     var nopen,nclose:byte):boolean;
function exist(filen:string):boolean;
procedure do_trace(    s:string);
procedure io_on;
procedure io_set;
procedure set_cli_error(   eval:integer);
function get_cli_error(    reset:boolean):integer;

const
     days:array[0..6] of string[9]=('Sunday','Monday','Tuesday','Wednesday',
                                    'Thursday','Friday','Saturday');
     open_b='(';
     close_b=')';

var
   lowest_mem:longint;

implementation

uses dbase,dos,crt,inform,panic;

var
   error_file:text;
   output_file:text;
   stdout,stdoutcpy:text;
   stdoutok,stdoutcpyok:boolean;
   output_error:boolean;
   linecount:word;
   xcount:word;

procedure write_check_dir;
const
     message=newline+'Check that the DIR parameter has been set as necessary.'+
                    newline+
             'eg. with LEAP in c:\rdbms\leap start LEAP with:'+newline+
             'C:\RDBMS\LEAP\LEAP dir=c:\rdbms\leap';
begin
     if stdoutok then
        writeln_stdout(message)
     else
         writeln(message);
end;

procedure fatal_error(errorno:word;
                      errorstr:string);
(* Reports some form of fatal error which halts further execution
   of the program *)

(* Ref. Section x.1.1 *)

var
   count:word;
   data,s:string;
begin
     system.str(errorno,s);

     {$I-}
     reset(error_file);
     {$I+}

     (* Check we opened the error file ok! *)
     if (IOResult<>0) then      {Open error}
     begin
          writeln_stdout('Fatal Error #2: Error opening file. Aborting');
          writeln_stdout('File: '+DIR_MASK+ERROR_DIR+ERROR_FILENAME);
          writeln_stdout('*** Previous error occured #'+s+' - '+errorstr+' *** ');
          write_check_dir;

          (* Report the abnormal termination *)
          report('Abnormal termination - unable to open error file.');

          (* Halt with an error *)
          halt(1);
     end
        else

        (* No error occured *)
        begin
             count:=1;

             (* Read a line of text *)
             readln(error_file,data);

             strip_leading_spaces(data);

             (* Whilst the line counter is less than the error number *)
             while (count<errorno) and (not eof(error_file)) do
             begin
                  if data[1]<>'#' then
                     inc(count);

                  (* Read another line *)
                  readln(error_file,data);
                  strip_leading_spaces(data);
             end;

             (* We reached the end of the file *)
             if eof(error_file) then
             begin
                writeln_stdout('Fatal Error #3: Unknown Error Occured. Aborting');
                report('Fatal Error #3: Unknown Error Occured. Aborting')
             end
             else

             (* We ran past where we expected the error details to be *)
             if count>errorno then
             begin
                  writeln_stdout('Fatal Error Occured. Error Not known. Aborting (OEC#'+s+')');
                  report('Fatal Error Occured. Error Not known. Aborting (OEC#'+s+')');
             end
             else
             (* We found it *)
             begin
                  (* Cut out the line number *)
                  delete(data,1,4);

                  (* Print and report the error *)
                  writeln_stdout('Fatal Error: #'+s+': '+data);
                  report('Fatal Error: #'+s+': '+data);

                  (* If there was additional information *)
                  if (errorstr<>'') then
                  begin
                       writeln_stdout(errorstr);
                       report(errorstr);
                  end;

                  if errorno<=REF_ERROR_CHECK_DIR then
                     write_check_dir;
             end;

             (* Close the error file *)
             close(error_file);

             if not output_error then
                close(output_file);

             (* Halt with an unsuccessful error code *)
             halt(1);
        end;
end;



procedure nonfatal_error(errorno:word;
                         errorstr:string);
(* Reports some form of nonfatal error which reports an error to the
   screen, but does not halt execution of the program (unless
   some fatal error occurs *during* the error reporting process!) *)

(* Ref. Section x.1.2 *)

var
   count:word;
   data:string;
   s:string;
begin
     system.str(errorno,s);
     {$I-}
     reset(error_file);
     {$I+}
     (* Check we opened the error file ok! *)
     if (IOResult<>0) then      {Open error}
     begin
          (* Report a fatal error on file opening *)
          writeln_stdout('Fatal Error #2: Error opening file. Aborting');
          writeln_stdout('File: '+DIR_MASK+ERROR_DIR+ERROR_FILENAME);
          write_check_dir;
          report('Fatal Error #2: Error opening file. Aborting');
          report('File: '+DIR_MASK+ERROR_DIR+ERROR_FILENAME+' - errorproc('+s+').');
          halt(1);
     end
        else
        begin
             count:=1;

             (* Load the appropriate line from the error file *)
             readln(error_file,data);

             strip_leading_spaces(data);
             while (count<errorno) and (not eof(error_file)) do
             begin
                  if data[1]<>'#' then
                     inc(count);

                  readln(error_file,data);
                  strip_leading_spaces(data);
             end;

             (* We reached the end of the file *)
             if eof(error_file) then
             begin
                  writeln_Stdout('Error #3: Unknown Error Occured. (UEC#'+s+')');
                  writeln_Stdout(errorstr);
                  report('Error #3: Unknown Error Occured. (UEC#'+s+')');
                  report(errorstr);
             end
             else
             (* We ran past where we expected the error details to be *)
             if count>errorno then
             begin
                writeln_stdout('Undefined Error Occured. Error Not known. Aborting (OEC#'+s+')');
                report('Undefined Error Occured. Error Not known. Aborting (OEC#'+s+')')
             end
             else
             (* We found it *)
             begin
                  delete(data,1,4);
                  writeln_stdout('Error: #'+s+': '+data);
                  report('Error: #'+s+': '+data);
                  if (errorstr<>'') then
                  begin
                       writeln_stdout(errorstr);
                       report(errorstr);
                  end;

                  if errorno<=REF_ERROR_CHECK_DIR then
                     write_check_dir;
             end;

             (* Close the error file *)
             close(error_file);

        end;
end;




procedure strip_leading_spaces(var s:string);
(* Cut the leading spaces out of a specified string *)

(* Ref. Section x.1.3 *)

var
   x:byte;
begin
     x:=1;
     (* Find the end of the spaces *)
     while (s[x]=' ') and (x<=length(s)) do
           inc(x);

     (* Delete the characters up to the first non-space *)
     delete(s,1,x-1);
end;



procedure strip_tail_spaces(var s:string);
var
   st,x:byte;
begin
     x:=length(s);

     while (s[x]=' ') and (x<=length(s)) do
           dec(x);

     (* Delete the characters up to the last non-space *)

     delete(s,x+1,length(s));
end;



procedure strip_leading_seperators(var s:string);
(* Cut the leading seperators from a specified string *)

(* Ref. Section x.1.4 *)

var
   x:byte;
begin
     x:=1;
     (* Locate the first non seperator *)
     while (s[x] in seperators) do
           inc(x);

     (* Delete the characters up to the first non-seperator *)
     delete(s,1,x-1);
end;





function get_token(text:string;
                   seperator:char):string;
(* Get a token from the string, and return it *)

(* Ref. Section x.1.5 *)

var
   cnt,scnt:word;
begin
     cnt:=1;

     (* If seperator is a space, search for defined seperators *)
     if seperator=' ' then
     begin
          (* Find first NON-Seperator *)
          while (text[cnt] in seperators) and (cnt<=length(text)) do
                inc(cnt);

          (* If we've overshot the string *)
          if (cnt>length(text)) then
          begin
               (* Return nothing *)
               get_token:='';
               text:='';
               exit;
          end;
          scnt:=cnt;

          (* Find first seperator *)
          while (not(text[cnt] in seperators)) do
                inc(cnt);

          (* Return the data *)
          get_token:=copy(text,scnt,cnt-scnt);
     end
     else
         begin
              (* Locate the first non-seperator (as defined in
                 parameter) *)
              while (text[cnt]<>seperator) and (cnt<=length(text)) do
                    inc(cnt);

              (* Reached end of line *)
              if (cnt>length(text)) then
              begin
                   (* Return nothing *)
                   get_token:='';
                   text:='';
                   exit;
              end;

              (* Return the data *)
              get_token:=copy(text,1,cnt-1);
         end;

end;



function cut_token(var text:string;
                       seperator:char):string;
(* Cut out a token from the string, and return it *)

(* Ref. Section x.1.6 *)

var
   cnt,scnt:word;
begin
     cnt:=1;

     if (seperator=' ') then
     begin
          (* Find first NON-Seperator *)
          while (text[cnt] in seperators) and (cnt<=length(text)) do
                inc(cnt);

          (* Overshot the end of the string *)
          if (cnt>length(text)) then
          begin
               cut_token:='';
               text:='';
               exit;
          end;
          scnt:=cnt;

          (* Find first seperator *)
          while (not(text[cnt] in seperators)) do
                inc(cnt);

          (* Copy the text *)
          cut_token:=copy(text,scnt,cnt-scnt);

          (* Delete the text from the specifed string *)
          delete(text,scnt,cnt-scnt);

          end
     else
         begin
              if seperator=#0 then seperator:=' ';
              (* Patch to enable spaces to be used as seperators *)

              (* Locate the first non-seperator *)
              while (text[cnt]<>seperator) and (cnt<=length(text)) do
                    inc(cnt);

              (* If we overshot *)
              if (cnt>length(text)) then
              begin
                   (* Return nothing *)
                   cut_token:='';
                   text:='';
                   exit;
              end;

              (* Cut out and return the data *)
              cut_token:=copy(text,1,cnt-1);
              delete(text,1,cnt);
         end;
end;




function find_tokens(var s:string):string;
(* Cut out all of the strings within the specified string, and return
   a string with each sub-string seperated by a space *)

(* Ref. Section x.1.7 *)

var
   rs:string;
begin
     rs:='';

     (* Whilst there is information left in the parameter string *)
     while length(s)<>0 do
           (* Concatenate the return from cut_token with itself & a space *)
           rs:=concat(rs,cut_token(s,' '),' ');

     (* Return the result *)
     find_tokens:=rs;
end;



function pad_zeros(    c:word):string;
(* Pad out numbers less than 10 with a leading zero *)

(* Ref. Section x.1.8 *)

var
   str:string;
begin
     (* Convert the word to a string *)
     system.Str(c,str);

     (* If its less than 10 *)
     if c<10 then
        (* Add a zero *)
        str:=concat('0',str);

     (* Return the string *)
     pad_zeros:=str;
end;



procedure pad(var str:string;
                  pad_size:word);
(* Pad a string with spaces to take str to the specified size *)

(* Ref. Section x.1.9 *)

var
   l,x:word;
begin
     (* Get the length of the string *)
     l:=length(str);

     (* Whilst we haven't reached the specified size of string *)
     for x:=l to pad_size do
         (* Add a space *)
         str:=concat(str,' ');

end;



function generate_random_string(    n:word):string;
(* Generate a random string of n characters *)

(* Ref. Section x.1.10 *)

var
   s:string;
   c:word;
begin
     (* Reset the string (preceed with an L for filenames! Keeps
        them together in filemanager/Windows Explorer *)
     s:='L';

     (* Whilst we haven't reached the desired number *)
     for c:=1 to n-1 do
         (* Add a random ASCII character in range A..Z to the string *)
         s:=concat(s,chr(random(26)+65));

     (* Return the string *)
     generate_random_string:=s;
end;




function HeapFunc(Size:Word):Integer; far;
(* Reports an error with a pointer creatre request, and ensures that
   a nil pointer is returned (Return result 1), as opposed to
   a runtime error (Returns 0), or 100% successful (returns 2).
   Checks for nil after create ensure that graceful shutdown occurs *)

(* Ref. Section x.1.11 *)

var
   ssize,smax,smem:string;
begin
     (* Check the size of the creation *)
     if size>maxavail then
     begin
          (* Convert data to string *)
          system.str(size,ssize);
          system.str(memavail,smem);
          system.str(maxavail,smax);

          (* Report a nonfatal error, although this is likely to cause
             all manner of problems if not handled properly *)
          nonfatal_error(ERROR_INSUFFICENT_MEM,'ATALLOC: '+ssize+' MAXAVAIL: '+smax+' MEMAVIL: '+smem);
          HeapFunc:=1;
     end;
end;


function get_datetime:string;
var
   year,month,day,dayOfWeek:word;
   hour,minute,second,sec100:word;
begin
     (* Get the date and time *)
     getdate(year,month,day,dayOfWeek);
     gettime(hour,minute,second,sec100);

     (* Write the date & time to the output file *)
     if USTIME then
        get_datetime:=DAYS[dayOfWeek]+', '+pad_zeros(month)+'/'+pad_zeros(day)+'/'+pad_zeros(year)+
                                         ' - '+pad_zeros(hour)+':'+pad_zeros(minute)+':'+pad_zeros(second)+'.'
                                         +pad_zeros(sec100)
     else
     get_datetime:=DAYS[dayOfWeek]+', '+pad_zeros(day)+'/'+pad_zeros(month)+'/'+pad_zeros(year)+
                                         ' - '+pad_zeros(hour)+':'+pad_zeros(minute)+':'+pad_zeros(second)+'.'
                                         +pad_zeros(sec100);


end;


procedure report(    reportstring:string);
(* Write a date/time stamp to the report file, and output a string
   of associated text *)

(* Ref. Section x.1.12 *)

{var
   year,month,day,dayOfWeek:word;
   hour,minute,second,sec100:word;}

begin
     (* If an output_error hasn't occured *)
     if not output_error then
     begin
          (* Get the date and time *)
{          getdate(year,month,day,dayOfWeek);
          gettime(hour,minute,second,sec100);}


          writeln(output_file,get_datetime+' : '+reportstring);
          (* Write the date & time to the output file *)
{          writeln(output_file,DAYS[dayOfWeek],', ',pad_zeros(day),'/',pad_zeros(month),'/',pad_zeros(year),
                                             ' - ',pad_zeros(hour),':',pad_zeros(minute),':',pad_zeros(second),' : ',
                                             reportstring);}
     end;
end;



procedure report_shutdown;
(* Close the report file - this must be called before termination,
   otherwise data is lost in the file *)

(* Ref. Section x.1.13 *)

begin
     report('Reporting terminated.');
     close(output_file);
end;





procedure report_flush;
(* Erase the report file - and recreate it *)

(* Ref. Section x.1.14 *)

begin
     report_shutdown;
     erase(output_file);
     rewrite(output_file);
     writeln_stdout('Report file flushed and restarted.');
     report('Report file flushed. Restarted');
end;


procedure check_assign(ptr:pointer;
                       desc:string);
(* If the pointer passed is a nil pointer, then cause a fatal error
   to occur. This procedure is *always* called after a call to new,
   thus ensuring that all pointers are checked. Redirection of errors
   ensures that insufficent memory creates a nil ptr *)

(* Ref. Section x.1.15 *)

begin
{    if TRACE then begin
        write('>');
        write(memavail);
        write('<');
     end;}
     if memavail<lowest_mem then
        lowest_mem:=memavail;
     if not assigned(ptr) then
        fatal_Error(ERROR_INSUFFICENT_MEM,desc);

end;




function upstring(    s:string):string;
(* Convert a string to upper case *)

(* Ref. Section x.1.16 *)

var
   x,l:byte;
begin
     (* Get the length of the string *)
     l:=length(s);

     (* Whilst we haven't reached the end of the string *)
     for x:=1 to l do
         (* Convert each character to uppercase *)
         s[x]:=upcase(s[x]);

     (* Return the string *)
     upstring:=s;
end;





procedure setup_stdout;
(* Initialise the stdout file for writing to stdout *)

(* Ref. Section x.1.17 *)

var
   ioe:integer;

begin
     (* Clear the screen *)
     clrscr;

     (* Assign the stdout file to '' - which is standard output *)
     assigncrt(stdout);
{     assign(stdout,'');}



     (* 'Create' the files, to allow writing *)

     {$I-}
     rewrite(stdout);
     {$I+}
     ioe:=IOResult;

     if ioe<>0 then
     begin
          stdoutok:=FALSE;
          fatal_Error(ERROR_FILE_OPENING,'Standard Output');
     end
     else
         stdoutok:=TRUE;


     {CPYFILE is TRUE if a file should be created and written to}
     if cpyfile then
     begin

          (* Assign the stdoutcpy file to a specified file *)
          assign(stdoutcpy,tempdir+stdoutcpy_file);

          {$I-}
          rewrite(stdoutcpy);
          {$I+}
          ioe:=IOResult;

          if ioe<>0 then
          begin
               stdoutcpyok:=FALSE;
               fatal_Error(ERROR_FILE_OPENING,'Standard Output Duplicate file: '+tempdir+stdoutcpy_file+
                                                        #13+#10+'Check TEMP environment variable is set properly');
          end
          else
              stdoutcpyok:=TRUE;

          {So the file is closed at termination}
          cpyfile_Created:=TRUE;
     end
     else
         stdoutcpyok:=FALSE;
end;




procedure stdout_close;
(* Close the stdout file *)

(* Ref. Section x.1.18 *)

begin
     close(stdout);
     stdoutok:=FALSE;

     if cpyfile_created then
     begin
          close(stdoutcpy);
          stdoutcpyok:=FALSE;
     end;
end;




procedure write_stdoutcpy(    s:string);

(* Ref. Section x.2.1 *)

begin
     if cpyfile and ioon then
     begin
         (* Write, without a newline character, the specified data *)
         if stdoutcpyok then
            write(stdoutcpy,s);
     end;
end;




procedure write_Stdout(    s:string);
(* Write a string to the stdout file *)

(* Ref. Section x.1.19 *)

begin
     if ioon then
     begin
         (* Write, without a newline character, the specified data *)
         if stdoutok then
            write(stdout,s)
         else
             write(s);

         if cpyfile then
            write_stdoutcpy(s);
     end
end;




procedure writeln_stdoutcpy(    s:string);

(* Ref. Section x.2.2 *)

begin
     (* Write, with a newline character, the specified data *)
     if cpyfile and stdoutcpyok and ioon then
        writeln(stdoutcpy,s);
end;




procedure writeln_stdout(    s:string);
(* Write a string to the stdout file, followed by a carriage return *)

(* Ref. Section x.1.20 *)

begin
     if ioon then
     begin
         (* Write, with a newline character, the specified data *)
         if stdoutok then
            writeln(stdout,s)
         else
             writeln(s);

         if cpyfile then
            writeln_stdoutcpy(s);
     end
end;




procedure writeln_stdout_nl;
(* Write a new line to the stdout file *)

(* Ref. Section x.1.21 *)

begin
     if ioon then
     begin
         (* Write a newline character to the specified files *)
         if stdoutcpyok then
            writeln(stdout)
         else
             writeln;

         if cpyfile then
            writeln_stdoutcpy('');
     end
end;




function in_array(    ss:string;
                      ar:array of string):boolean;
(* Search an array of strings for a given string, and return
   true or false *)

(* Ref. Section x.1.22 *)

var
   c:word;
begin
     c:=0;

     (* Move through the array *)
     while (c<=High(ar)) and (ar[c]<>ss) do
           inc(c);

     (* Return whether the string is in the array *)
     in_array:=(c<=High(ar));
end;





function get_first_expression(var text:string;
                              var boolval:byte):string;
(* Takes an expression from the string specified, where a string
   is seperated by boolean expressions in the bool array *)

(* Ref. Section x.1.23 *)

const
     NO_BOOL=3;
     bool:array[1..NO_BOOL] of string=('AND','OR','');
var
   expr,token:string;
   c:word;



   function cut_first_bool(var str:string):string;
   (* Locates the first boolean condition in the specified string, and
      cuts it out, and returns the boolean condition *)
   var
      position,c:byte;
      t:string;
   begin
        c:=1;

        (* Locate the position of the first boolean condition *)
        position:=pos(bool[c],str);

        (* Whilst we haven't found a boolean condition, and there are
           others to search for *)
        while (c<NO_BOOL) and (position=0) do
        begin
             inc(c);

             (* Locate the position of the next boolean condition *)
             position:=pos(bool[c],str);
        end;

        (* If we haven't found any, return nothing *)
        if position=0 then
           cut_first_bool:=''
        else
            begin
                 (* Copy the string and return it *)
                 cut_first_bool:=copy(str,position,length(bool[c]));

                 (* Remove the boolean condition from the parameter string *)
                 delete(str,position,length(bool[c]));
            end;
   end;

begin
     (* Get the first expression contained in second level brackets *)
     expr:=cut_to_right_bracket(text,2);

     (* Get the boolean condition *)
     token:=cut_first_bool(text);

     (* Set the return result *)
     if token='AND' then boolval:=1
     else
     if token='OR' then boolval:=2
     else
     boolval:=0;

     (* Return the expression string *)
     get_first_expression:=expr;
end;




function contains(    str:string;
                      ch:char):boolean;
(* Search a specified string for a specified character, and return a boolean
   value indicating the success of the search *)

(* Ref. Section x.1.24 *)

var
   count:byte;
begin
     count:=1;

     (* Whilst we haven't reached the end of the string, and haven't found
        the specified character *)
     while (count<=length(str))
       and (ch<>str[count]) do
       (* Move down one character *)
       inc(count);

     (* If we hadn't reached the end of the string, return true, false
        otherwise *)
     contains:=(count<=length(str));
end;





function brackets_ok(    s:string;
                     var nopen,nclose:byte):boolean;
(* Check the number of open & close paranthesis in a specified string *)

(* Ref. Section x.1.25 *)

var
   c:byte;
begin
     (* Reset counters *)
     c:=1;
     nopen:=0;
     nclose:=0;

     (* Whilst we haven't reached the end of the string *)
     while (c<=length(s)) do
     begin
          (* If the current character is an open paranthesis, increment
             the open counter *)
          if s[c]=open_b then
             inc(nopen);

          (* If the current character is a close paranthesis, increment
             the close counter *)
          if s[c]=close_b then
             inc(nclose);

          (* Increment the position counter *)
          inc(c);
     end;

     (* Return TRUE if the open and close paranthesis counters match *)
     brackets_ok:=(nopen=nclose);
end;







function cut_to_right_bracket(var s:string;
                                  bdepth:shortint):string;
(* Cut the data contained within the outer brackets *)

(* Ref. Section x.1.26 *)

var
   bexist:boolean; {Brackets exist}
   st,fin:byte;
   c,no:byte;
   depth:shortint;
   bracket:array[1..MAX_NO_BRACKETS] of record
                                              pos:byte;
                                              depth:byte;
   end;
   rstr:string;


begin
     (* Reset data *)
     for c:=1 to MAX_NO_BRACKETS do
     begin
          bracket[c].pos:=0;
          bracket[c].depth:=0;
     end;
     no:=0;
     depth:=1;
     c:=1;
     bracket[1].pos:=1;
     bracket[1].depth:=1;
     bracket[2].pos:=length(s)+2; {+2 for brackets added if none exist}
     bracket[2].depth:=1;
     bexist:=false;

     (* Whilst we haven't reached the end of the string *)
     while (c<=length(s)) do
     begin
          (* Open brackets... *)
          if s[c]=open_b then
          begin
               (* Mark the position *)
               inc(no);
               bracket[no].pos:=c;
               bracket[no].depth:=depth;

               (* Increment the bracket 'depth' counter *)
               inc(depth);

               (* Set the brackets-exist flag *)
               bexist:=true;
          end;

          (* Close brackets... *)
          if s[c]=close_b then
          begin
               inc(no);

               (* Decrement the bracket 'depth' counter *)
               dec(depth);

               (* Mark the position *)
               bracket[no].pos:=c;
               bracket[no].depth:=depth;
          end;

          (* Increment the position *)
          inc(c);
     end;

     (* If the specified depth is greater than the maximum depth of
        the brackets in the given string, set the parameter to the
        maximum 'depth' possible *)

     if bdepth>depth then
        bdepth:=depth;

     (* If no brackets actually exist... *)
     if not bexist then
        (* Add a couple of brackets *)
        s:=concat('(',s,')');

     rstr:='';

     (* Locate the first open bracket of the specified depth *)
     c:=1;
     while (bracket[c].depth<bdepth) do
           inc(c);
     st:=c;

     inc(c);

     (* Locate the corresponding close bracket of the specified depth *)
     while (bracket[c].depth>bdepth) do
           inc(c);
     fin:=c;

     (* Copy out the data contained within the brackets *)
     rstr:=copy(s,bracket[st].pos+1,bracket[fin].pos-bracket[st].pos-1);

     (* Remove the data contained within the brackets *)
     system.delete(s,bracket[st].pos,bracket[fin].pos-bracket[st].pos+1);

     (* Return the data copied out *)
     cut_to_right_bracket:=rstr;
end;




procedure process_param(var debug:boolean);
(* Process the parameter string passed at the command line *)

(* Ref. Section x.1.27 *)

var
   c:byte;
   op,s:string;
begin
     c:=1;

     while (c<=paramcount) do
     begin
          (* Convert all of the parameters to uppercase *)
          s:=upstring(paramstr(c));

          if s='INFO' then
             warranty_conditions
          else
          if (s='/?') or (s='?') then
             leap_options
          else
          begin

               (* Cut the first string before an equals *)
               op:=cut_token(s,'=');

               (* If its 'DEBUG' *)
               if op='DEBUG' then
                  (* Turn on debug information *)
                  debug:=(s='Y')
               else
               (* If its 'DIR' *)
               if op='DIR' then
                  (* Set the directory mask *)
                  DIR_MASK:=s
               else
               if op='INFIX' then
                  infix:=(s='Y')
               else
               if op='TIMING' then
                  TIMING:=(s='Y')
               else
               if op='USTIME' then
                  USTIME:=(s='Y')
               else
               if op='COPY' then
                  CPYFILE:=(s='Y')
               else
               if op='NAG' then
                  NAG:=(s='Y')
               else
               if (op='COLOUR') or (op='COLOR') then
                  COLOUR:=(s='Y')
               else
               if (op='TRACE') then
                  TRACE:=(s='Y')
               else
{               if (op='DESC') then
                  DESC:=(s='Y')
               else}
               if (op='IOON') then
                  IOON_SETTING:=(s='Y')
               else
               if (op='CACHE') then
                  CACHEENABLED:=(s='Y')
               else
               if (op='ITERATIVE') then
                  ITERATIVE:=(s='Y')
               else
               if (op='PARSE') then
                  DISPPARSE:=(s='Y')
          end;

          (* Go to the next parameter *)
          inc(c);
     end;

end;


function exist(filen:string):boolean;
(* Check the existance of the specified file *)
var
   DirInfo:SearchRec;
begin
     FindFirst(filen, anyfile, DirInfo);

     (* Return the equating of DosError and 0 - TRUE if no error, ie.
        file exists *)
     exist:=(DosError=0);
end;

procedure do_trace(    s:string);
begin
     if TRACE then
     begin
          if debug then
             report(s);
          writeln_stdout(s);
     end;
end;


procedure set_cli_error(   eval:integer);
begin
     cli_error:=eval;
end;

function get_cli_error(    reset:boolean):integer;
begin
     get_cli_error:=cli_Error;
     if reset then
        cli_error:=0;
end;


procedure io_on;
begin
     ioon:=true;
end;

procedure io_set;
begin
     ioon:=(ioon_setting);
end;

var
   dirinfo:searchrec;
   c:byte;

begin
     stdoutcpyok:=FALSE;
     stdoutok:=FALSE;


     title;

     (* Process parameters *)
     process_param(debug);
                          (* Sets the heap error function
                             to a user defined function for
                             graceful shutdown *)
     HeapError:=@HeapFunc;

     (* Initializes the built-in random number
        generator with a random value (obtained from
        the system clock). *)

     randomize;

     (* Locate the error file *)
     findfirst(DIR_MASK+ERROR_DIR+ERROR_FILENAME,anyfile,dirinfo);

     (* If an error occured when opening the error file *)
     if doserror<>0 then
     begin
          (* Abort *)
          writeln('Fatal Error #1: File not found. Aborting.');
          writeln('File: ',DIR_MASK+ERROR_DIR+ERROR_FILENAME);
          write_check_dir;
          halt(1);
     end;

     (* Otherwise, assign the file to the error file descriptor *)
     assign(error_file,DIR_MASK+ERROR_DIR+ERROR_FILENAME);

     (* Find the report file *)
     findfirst(DIR_MASK+REPORT_DIR+REPORT_FILENAME,anyfile,dirinfo);

     assign(output_file,DIR_MASK+REPORT_DIR+REPORT_FILENAME);
     (* If an error occured when opening the output file *)
     if doserror<>0 then
     begin
          (* Create a new output file *)
          rewrite(output_file);
          report('File Created.')
     end
     else
         begin
              (* Turn off the error flag *)
              output_error:=TRUE;

              (* Set it for appending *)
              {$I-}
              append(output_file);
              {$I+}

              (* If an error occured *)
              if IOResult<>0 then
              begin
                   (* Raise an error *)

                   setup_stdout;

                   fatal_error(2,'reports.txt');
              end;

              (* Write a new line to the output file *)
              writeln(output_file);

              (* Write out a line of hashes *)
              for c:=1 to 79 do
                  write(output_file,'#');

              writeln(output_file);
              writeln(output_file,'# Report File Started');
              for c:=1 to 79 do
                  write(output_file,'#');
              writeln(output_file);
              report('File appending started.');

              output_error:=false;
         end;
end.
