unit tuples;

(* Unit: Tuples
   Created: 13/11/1994
   Notes: 
   References: References to the documentation is to the file tuples.doc
               and section numbers are given as relative to this document.
   Description: Contains routines for the maintenance of the tuples
                held within relations, including reading, writing,
                deleting, updating etc.
   Revision History:
         18/12/1994 - Post pointer create check and polite shutdown
         12/03/1995 - User create tuple
         27/03/1995 - Documented (tuples.doc)
         01/06/1995 - Fixed case bug in user_tuple_add
         10/12/1995 - Implemented support for escaped data files, whilst
                      also supporting space delimited data files.
         22/06/1996 - Placed under the GNU General Public License.
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)


{ Thoughts...}

{
 There is something weird about the 38th node in a tuple structure,
 it seems to be being set to a ptr value thats been disposed previously,
 when building a hash table...

}

{
Cache - What if its a large relation, and it cannot locate the node
in the cache - so it reads it from disk, but the read ptr has not been
incremented in the input file - it'll read the first record...

Apart from that it looks good!

}

interface

{$O+}

uses dtypes;

const
     TE_EOF=6; (* End of File *)
     TE_DUPLICATE=1; (* Duplicate tuple attempted to be written *)

type
    tuple_Datum=^tuple_datum_def;
    tuple_datum_def=record
                          relationPtr:relation;
                          fieldPtr:field;
                          data:string;
    end;

    tuple_attributes=^tuple_attribute_def;
    tuple_attribute_def=record
                              tdatum:array[1..MAX_NO_ATTRIBUTES] of tuple_datum;
    end;

    stack_node=^stack_node_def;
    stack_node_def=record
                         data:tuple_datum;
                         next:stack_node;
    end;

    stack=^stack_def;
    stack_def=record
                    head_node:stack_node;
                    no_items:word;
    end;

var
   TupleError:word;
   ttt:tuple_attributes;


function tuple_prepare_fields(var rel:relation):tuple_attributes;
function tuple_prep:tuple_Attributes;
function create_tuple_datum(    rel:relation;
                                fld:field;
                                data:string):tuple_datum;
procedure datum_dispose(var datum:tuple_datum);
function create_stack:stack;
procedure push_stack(var st:stack;
                         data:tuple_datum);
procedure flush_stack(var st:stack);
function attach_datums(var ta:tuple_attributes;
                       var tdatums:stack):boolean;
function check_tuple(var ta:tuple_Attributes):boolean;
procedure write_tuple(var tuple:tuple_Attributes;
                      var write_pos:loc_type);
procedure tuple_close(var ta:tuple_attributes;
                       var tdatums:text);
function readfirst_tuple(var rel:relation;
                         var tdatums:text;
                         var nofields:word):tuple_Attributes;
procedure tuple_print(    ta:tuple_attributes;
                          print_hdr:boolean);
procedure tuple_dispose(var ta:tuple_attributes;
                            all:boolean;
                            relations:boolean);
procedure readnext_tuple(var ta:tuple_attributes;
                         var tdatums:text;
                             nofields:word);
function tuple_findfield(    ta:tuple_attributes;
                             fld:string):word;
procedure break_Down(var data:string;
                     var ta:tuple_Attributes;
                     var nofields:word);
function readfirst_tuple_b(var rel:relation;
                          var tdatums:file;
                          var nofields:word;
                          var spos,epos:word):tuple_Attributes;
procedure readnext_tuple_b(var ta:tuple_attributes;
                           var tdatums:file;
                               nofields:word;
                          var spos,epos:word);
function tuple_to_string(    t:tuple_attributes):string;
function tuple_to_string_orderbyrel(    t:tuple_attributes;
                                        rel:relation):string;
function tuple_user_add(    rel:relation):relation;
function tuple_flex_add(    rel:relation;
                            data:string;
                            seperator:char):relation;
procedure tuple_count_inc(var tcount:integer);
procedure tuple_count_reset(var tcount:integer);
procedure tuple_count_reset_all;

implementation

uses utils,fields,strings,index,hashing,caching,rtional;


procedure tuple_dispose(var ta:tuple_attributes;
                            all:boolean;
                            relations:boolean);
(* Dispose of a tuple structure *)
(* Ref. x.4.1 *)
var
   count,cnt2:word;
   rel:relation;
begin
     count:=1;
     (* While there are items to be disposed *)
     while ( (assigned(ta^.tdatum[count])) and (count<=MAX_NO_ATTRIBUTES)) do
     begin
          (* Are we to dispose of all structures? *)
          if all then
          begin
               if assigned(ta^.tdatum[count]^.fieldPtr) then
               begin
                  dispose(ta^.tdatum[count]^.fieldPtr);
                  ta^.tdatum[count]^.fieldptr:=nil;
               end;{
               else
                   nonfatal_error(ERROR_DISPOSE_UNALLOCMEM,'(tuple_dispose fieldptr)');}

               if ((relations) and (assigned(ta^.tdatum[count]^.relationPtr))) then
               begin
                     rel:=ta^.tdatum[count]^.relationPtr;
                     for cnt2:=count+1 to MAX_NO_ATTRIBUTES do
                         if ta^.tdatum[cnt2]^.relationPtr=rel then
                            ta^.tdatum[cnt2]^.relationPtr:=nil;
                     if assigned(rel) then
                     begin
                        dispose(rel);
                        rel:=nil;
                     end
                     else
                         nonfatal_error(ERROR_DISPOSE_UNALLOCMEM,'(tuple_dispose rel)');
               end;
          end;
          if assigned(ta^.tdatum[count]) then
          begin
               dispose(ta^.tdatum[count]);
               ta^.tdatum[count]:=nil;
          end
          else
              nonfatal_error(ERROR_DISPOSE_UNALLOCMEM,'(tuple_dispose ta^.tdatum[count])');

          inc(count);
     end;
     if assigned(ta) then
        dispose(ta)
     else
         nonfatal_error(ERROR_DISPOSE_UNALLOCMEM,'(tuple_dispose ta)');
     ta:=nil;
end;




function tuple_prep:tuple_Attributes;
(* Prepare a tuple for use *)
(* Ref. x.4.2 *)

var
   ta:tuple_Attributes;
   td:tuple_datum;
   count:word;
begin
     new(ta);
     check_assign(ta,'tuple_prep');

     (* Reset all items within the node *)
     for count:=1 to MAX_NO_ATTRIBUTES do
     begin
          (* Allocate memory *)
          new(td);
          check_assign(td,'tuple_prep');

          (* Reset items *)
          td^.fieldPtr:=nil;
          td^.relationPtr:=nil;
          td^.data:='';
          ta^.tdatum[count]:=td;
     end;

     (* Return the information *)
     tuple_prep:=ta;

end;



function tuple_prepare_fields(var rel:relation):tuple_attributes;
(* Prepare a tuple by taking all of the fields from a relation
   and placing reference information within the tuple *)
(* Ref. x.3.1 *)
var
   fields:tuple_attributes;
   node:field;
   x,count:word;

begin
     (* Create a tuple *)
     fields:=tuple_prep;

     (* Get the first field *)
     node:=field_findfirst(rel);
     count:=0;

     (* Whilst the field is valid *)
     while (assigned(node)) do
     begin
          inc(count);

          (* Populate the tuple with the information *)
          fields^.tdatum[count]^.fieldptr:=node;
          fields^.tdatum[count]^.relationPtr:=rel;

          (* Locate the next field *)
          node:=field_findnext(rel,node,TRUE,FALSE);
     end;

     (* Clear the remainding information *)
     inc(count);
     while (count<MAX_NO_ATTRIBUTES) do
     begin
          fields^.tdatum[count]^.fieldptr:=nil;
          fields^.tdatum[count]^.relationPtr:=nil;
          inc(count);
     end;

     (* Return the details *)
     tuple_prepare_fields:=fields;
end;




function create_tuple_datum(    rel:relation;
                                fld:field;
                                data:string):tuple_datum;
(* Create and populate a node for a tuple *)
var
   tdatum:tuple_datum;
begin
     new(tdatum);
     check_assign(tdatum,'create_tuple_datum');

     tdatum^.relationPtr:=rel;
     tdatum^.fieldPtr:=fld;
     tdatum^.data:=data;

     create_tuple_datum:=tdatum;
end;




procedure datum_dispose(var datum:tuple_datum);
(* Dispose of a  datum *)
begin
     if assigned(datum) then
     begin
          dispose(datum);
          datum:=nil
     end
        else
            nonfatal_error(ERROR_DISPOSE_UNALLOCMEM,'(datum_dispose)');
end;



function create_stack:stack;
(* Create a stack *)
var
   stk:stack;
begin
{     check_avail_mem(sizeof(stk));}
     new(stk);
     check_Assign(stk,'tuples.create_stack');
     stk^.head_node:=nil;
     stk^.no_items:=0;
     create_stack:=stk;
end;

procedure push_stack(var st:stack;
                         data:tuple_datum);
(* Push a datum onto the stack *)
var
   st_node:stack_node;
begin
{     check_avail_mem(sizeof(st_node));}
     new(st_node);
     check_assign(st_node,'tuples.push_stack');
     st_node^.data:=data;
     st_node^.next:=st^.head_node;
     st^.head_node:=st_node;
     inc(st^.no_items);
end;

function pop_stack(var st:stack):tuple_Datum;
(* Pop a datum off of the stack *)
var
   node:stack_node;
begin
     pop_stack:=st^.head_node^.data;

     node:=st^.head_node;
     st^.head_node:=node^.next;

     if assigned(node) then
        dispose(node)
     else
         nonfatal_error(ERROR_DISPOSE_UNALLOCMEM,'(pop_stack)');

     dec(st^.no_items);
end;



function stack_size(    st:stack):word;
(* Return the number of nodes in a stack *)
begin
     stack_size:=st^.no_items;
end;




function stack_empty(    st:stack):boolean;
(* Return TRUE if the stack is empty *)
begin
     stack_empty:=(stack_size(st)=0);
end;



procedure stack_dispose(var st:stack);
(* Dispose of the memory used by the stack *)
begin
     if assigned(st) then
        dispose(st)
     else
         nonfatal_error(ERROR_DISPOSE_UNALLOCMEM,'(stack_dispose)');
end;




procedure flush_stack(var st:stack);
(* Flush all of the memory used by the entire stack *)
begin
     while not(stack_empty(st)) do
           pop_stack(st);       {TP 7 allows functions to be called
                                 as procedures}
     stack_dispose(st);
end;




function attach_datums(var ta:tuple_attributes;
                       var tdatums:stack):boolean;
(* Attach the datums held within the stack to the tuple specified *)
var
   td:tuple_datum;
   index:word;

begin
     TupleError:=0;

     (* Whilst the stack is not empty *)

     while not(stack_empty(tdatums)) do
     begin
          (* Take a datum off of the stack *)
          td:=pop_stack(tdatums);
          index:=1;

          (* Locate the field that the datum refers to *)
          while ( (ta^.tdatum[index]^.fieldptr^.name<>td^.fieldPtr^.name)
              and (assigned(ta^.tdatum[index]^.fieldptr)) ) do
              begin
                   inc(index);
                   writeln(td^.fieldPtr^.name,' == ',td^.data);
              end;

          (* Drop the datum onto the tuple *)
          if assigned( (ta^.tdatum[index]^.fieldptr) ) then
             ta^.tdatum[index]:=td
          else
              (* Unless, of course, the field doesn't exist! *)
          begin
               nonfatal_error(5,'Field Name: '+td^.fieldPtr^.name);
               attach_datums:=false;
               TupleError:=5;
               flush_stack(tDatums);
               exit;
          end;
     end;

     (* Dispose of the stack *)
     stack_dispose(tDatums);
     attach_datums:=true;
end;



function check_tuple(var ta:tuple_Attributes):boolean;
(* Check that a tuple is ok *)
var
   index:word;
   error:boolean;
begin
     index:=1;
     error:=false;
     while ( (assigned(ta^.tdatum[index]^.fieldptr) ) and (not(error)) ) do
     begin
          writeln(ta^.tdatum[index]^.data);
          inc(index);
          error:=(ta^.tdatum[index]^.fieldptr^.primary) and not( assigned(ta^.tdatum[index]) );
                  {Primary Key TRUE=Yes                Data Present = TRUE=Yes
                              FALSE=No                               FALSE=No }
     end;

     check_tuple:=not error;
end;


{ Old version of write_tuple, which writes with spaces padded.
procedure write_tuple(var tuple:tuple_Attributes;
                      var write_pos:loc_type);
(* Write a tuple to disk, returning the position to which it was stored
   through write_pos *)
(* Ref x.3.2 *)
var
   rel:relation;
   fl_byte:file of byte;
   fl_text:text;
   fl:file;
   x,nofields:word;
   s,s2:string;
   success:boolean;
begin
     TupleError:=0;
     (* Get the relation to which the tuple belongs *)
     rel:=tuple^.tdatum[1]^.relationptr;
     nofields:=rel^.nofields;

     s:=tuple_to_string_orderbyrel(tuple,rel);
     hashing_retrieve(rel^.hash_table,s,s2,success);

     if not success then
     begin
         (* Open the file *)
         assign(fl_byte,rel^.filepath+rel^.filename);

         (* Determine the position at which the write will occur *)
         reset(fl_byte);
         write_pos:=filesize(fl_byte);      (* This gets the pos. of the *)
         close(fl_byte);                    (* Datum in the file written *)
                                            (* to                        *)

         (* Insert the tuple into the index structures appropriately *)
         insert_tuple_idx(tuple,rel,write_pos,idx_master);

         (* Open the file *)
         assign(fl_text,rel^.filepath+rel^.filename);
         append(fl_text);

         (* Write the information to the file *)
         for x:=1 to nofields do
         begin
              if length( tuple^.tdatum[x]^.data ) <= MAX_SIZE_DATUM_STRING then
                 pad(string(tuple^.tdatum[x]^.data),MAX_SIZE_DATUM_STRING)
              else
                  delete( tuple^.tdatum[x]^.data, tuple^.tdatum[x]^.fieldptr^.field_size,
                          length(tuple^.tdatum[x]^.data)-(tuple^.tdatum[x]^.fieldptr^.field_size));
              write(fl_text,tuple^.tdatum[x]^.data);
         end;
         (* Write an end-of-line series of markers *)
         writeln(fl_text);

         (* Insert the tuple into the appropriate hash table *)
         hashing_insert(rel^.hash_table,s,REQ_CALC);

         (* Close the file *)
         close(fl_text);
     end
     else
         begin
              (* Report error with duplicate entry into relation *)
              TupleError:=TE_DUPLICATE;
              do_trace('TRACE> Attempt to write duplicate tuple');
         end;
end;}

procedure write_tuple(var tuple:tuple_Attributes;
                      var write_pos:loc_type);
(* Revision 2 - Writes to disk with a delimeter rather than padded
   spaces *)
(* Write a tuple to disk, returning the position to which it was stored
   through write_pos *)
(* Ref x.3.2 *)
var
   rel:relation;
   fl_byte:file of byte;
   fl_text:text;
   fl:file;
   x,nofields:word;
   scratch,s,s2:string;
   success:boolean;
   posi,cp:byte;

   procedure escape_delimiters(var s:string);
   var
      p:byte;
   begin
        p:=1;
        while (p<length(s)) do
        begin
             if s[p]=DATA_DELIMITER then
             begin
                  insert(DATA_DELIMITER,s,p);
                  writeln(s);
                  (* The size has incrased one, so we must move on
                     otherwise the "if" will always find an escape
                     character! *)
                  inc(p);
             end;
             inc(p);
        end;
   end;

begin
     TupleError:=0;
     (* Get the relation to which the tuple belongs *)
     rel:=tuple^.tdatum[1]^.relationptr;
     nofields:=rel^.nofields;

     s:=tuple_to_string_orderbyrel(tuple,rel);
     hashing_retrieve(rel^.hash_table,s,s2,success);

     if not success then
     begin
         (* Open the file *)
         assign(fl_byte,rel^.filepath+rel^.filename);

         (* Determine the position at which the write will occur *)
         reset(fl_byte);
         write_pos:=filesize(fl_byte);      (* This gets the pos. of the *)
         close(fl_byte);                    (* Datum in the file written *)
                                            (* to                        *)

         (* Insert the tuple into the index structures appropriately *)
         insert_tuple_idx(tuple,rel,write_pos,idx_master);

         (* Open the file *)
         assign(fl_text,rel^.filepath+rel^.filename);
         append(fl_text);

         (* Write the information to the file *)
         for x:=1 to nofields do
         begin
              scratch:=tuple^.tdatum[x]^.data;
              (* Locate the position of the first delimeter *)

              if scratch<>'' then
                 escape_delimiters(scratch)
              else
                  scratch:=NULL_INDICATOR;
{              posi:=pos(scratch,DATA_DELIMITER);
              if posi<>0 then
              begin
                   (* Insert ANOTHER delimeter into the string, to
                      "escape" the escape character *)
                   insert(DATA_DELIMITER,scratch,posi);
              end;}

{              if length( tuple^.tdatum[x]^.data ) <= MAX_SIZE_DATUM_STRING then
                 pad(string(tuple^.tdatum[x]^.data),MAX_SIZE_DATUM_STRING)
              else
                  delete( tuple^.tdatum[x]^.data, tuple^.tdatum[x]^.fieldptr^.field_size,
                          length(tuple^.tdatum[x]^.data)-(tuple^.tdatum[x]^.fieldptr^.field_size));
              write(fl_text,tuple^.tdatum[x]^.data);}

              (* Write the data, plus the actual data delimiter *)
              write(fl_text,scratch+DATA_DELIMITER);
         end;
         (* Write an end-of-line series of markers *)
         writeln(fl_text);
         tuple_count_inc(outcount);

         (* Insert the tuple into the appropriate hash table *)
         hashing_insert(rel^.hash_table,s,REQ_CALC);

         (* Close the file *)
         close(fl_text);
     end
     else
         begin
              (* Report error with duplicate entry into relation *)
              TupleError:=TE_DUPLICATE;
              do_trace('TRACE> Attempt to write duplicate tuple');
         end;
end;

{ Old version with fixed size fields... Replaced by version with
  escaped data
procedure break_Down(var data:string;
                     var ta:tuple_Attributes;
                     var nofields:word);
(* Break a string down into a tuple, given that we know the field
   names and so forth, from ta *)
(* Ref. x.4.3 *)
var
   x,c:word;
begin
     x:=0;

     (* While there is still information *)
     while length(data)>0 do
     begin
          inc(x);

          (* Copy the information out *)
          ta^.tdatum[x]^.data:=copy(data,1,MAX_SIZE_DATUM_STRING+1);
          c:=length(ta^.tdatum[x]^.data);

          (* Locate the start of the padding *)
          while ta^.tdatum[x]^.data[c] in seperators do
                dec(c);
          (* Delete the padding from the read data *)
          delete(ta^.tdatum[x]^.data,c+1,length(ta^.tdatum[x]^.data));

          (* Delete the datum from the source string *)
          delete(data,1,MAX_SIZE_DATUM_STRING+1);
     end;

     (* Return the number of attributes processed *)
     nofields:=x;

     (* Blank out any remaining attribute data *)
     for c:=x+1 to MAX_NO_ATTRIBUTES do
         ta^.tdatum[c]^.data:='';
end;}

procedure unescape_string(var s:string);
var p:byte;
begin
     (* We can't search on two delimiters and replace each occurance
        of them with one delimiter, because then if there are two
        delimiters adjacent to each other, they will be removed... *)
     p:=pos(DATA_DELIMITER+DATA_DELIMITER,s);
     if (p<>0) then
     begin
          while (p<length(s)) do
          begin
               if (s[p]=DATA_DELIMITER) and (s[p+1]=DATA_DELIMITER) then
               begin
                   (* Delete one of the delimeters from the string *)
                   delete(s,p,1);

                   (* Move the pointer on one position *)
                   (* So that it doesn't cover the single delimiter with
                      the next delimiter by mistake *)
{                   inc(p);}
               end;
               (* Move the pointer on one position *)
               inc(p);
          end;
     end;
end;

procedure break_Down(var data:string;
                     var ta:tuple_Attributes;
                     var nofields:word);
(* Revision 2 - Breaks down tuples with escaped characters as the
   field delimiter - Intelligently determines whether it is an
   'old' type of data file, or a 'new' type of file, and processes
   it accordingly *)
(* Break a string down into a tuple, given that we know the field
   names and so forth, from ta *)
(* Ref. x.4.3 *)
var
   x,c:word;
   s:string;
begin

     if data[length(data)]=DATA_DELIMITER then
     begin
          c:=1;
          x:=0;
          while c<=length(data) do
          begin
               (* Have we found an escaped escape character? *)
               if( (data[c]=DATA_DELIMITER) and (c=length(data)) )
               or
               ( (data[c]=DATA_DELIMITER) and (data[c+1]<>DATA_DELIMITER)) then
               begin
                         (* We're onto the next data item *)
                         inc(x);

                         (* Copy the data into the tuple *)
                         s:=copy(data,1,c-1);

                         unescape_string(s);

                         (* Remove the data from the source data *)
                         delete(data,1,c);


                         if s=NULL_INDICATOR then
                            s:='';

                         ta^.tdatum[x]^.data:=s;


                         (* Reset the pointer to the start of the string *)
                         c:=1;
               end
               else
               if (data[c]=DATA_DELIMITER) and (data[c+1]=DATA_DELIMITER) then
                  inc(c);

               (* Iterate through the string *)
               inc(c);
          end;
     end
     else
     begin
          (* This is an 'old' style of file, with spaces as the
             delimeters at a specific size for each field *)
         x:=0;

         (* While there is still information *)
         while length(data)>0 do
         begin
              inc(x);

              (* Copy the information out *)
              ta^.tdatum[x]^.data:=copy(data,1,MAX_SIZE_DATUM_STRING+1);
              c:=length(ta^.tdatum[x]^.data);

              (* Locate the start of the padding *)
              while ta^.tdatum[x]^.data[c] in seperators do
                    dec(c);
              (* Delete the padding from the read data *)
              delete(ta^.tdatum[x]^.data,c+1,length(ta^.tdatum[x]^.data));

              (* Delete the datum from the source string *)
              delete(data,1,MAX_SIZE_DATUM_STRING+1);
         end;
     end;

     (* Return the number of attributes processed *)
     nofields:=x;

     (* Blank out any remaining attribute data *)
     for c:=x+1 to MAX_NO_ATTRIBUTES do
         ta^.tdatum[c]^.data:='';

end;


procedure read_updatecache(var f:text;
                               c:cache;
                           var s:string;
                           var p:word);
begin
     readln(f,s);
     cache_insert(c,s,p,eof(f));
     inc(p);
end;


function readfirst_tuple(var rel:relation;
                         var tdatums:text;
                         var nofields:word):tuple_Attributes;

(* Reads a tuple from the disk, but prepares the tuple first *)

(* Limited to 255 characters at the moment because of string data type *)

(* Ref. x.3.4 *)

var
   data:string;
   fld:field;
   ta:tuple_attributes;
   td:tuple_datum;
   count:word;
   eof:boolean;
begin
     (* Open the file *)
     assign(tdatums,rel^.filepath+rel^.filename);
     reset(tdatums);

     rel^.current_pos:=1;

     (* Read the first tuple *)
     data:=cache_read(rel^.rcache,rel^.current_pos);

     (* What if the data returned is empty - Ans->Check current_pos *)

     if (data='') and (rel^.current_pos=1) then
        read_updatecache(tdatums,rel^.rcache,data,rel^.current_pos);
{        readln(tdatums,data); {This may have to become a BlockRead at some point}

     (* Check case sensitivity HERE so that the data (in cache or not)
        is converted once off the disk. This will also enable the cache
        to be used even if the sensitivity is reversed, which otherwise
        would ruin the cache *)

     if not casesensitivity then
        data:=upstring(data);

     if (data<>'') then
     begin
          (* Create the tuple *)
          ta:=tuple_prep;

          (* Populate all of the field information *)
          populate_fields(rel,ta,nofields);

          break_down(data,ta,nofields)

     end
     else
          ta:=nil;

     (* Return the tuple *)
     readfirst_tuple:=ta;

end;


procedure tuple_close(var ta:tuple_attributes;
                       var tdatums:text);
begin
     (* Dispose of the tuple and report an 'error' *)
     tuple_dispose(ta,false,false);
{     tuple_dispose(ta,true,false);}
     close(tdatums);
     ta:=nil;
     TupleError:=ERROR_EOF; {End of File}
end;


procedure readnext_tuple(var ta:tuple_attributes;
                         var tdatums:text;
                             nofields:word);
(* Reads the next tuple from the specified file into ta *)

(* Ref. x.3.4 *)

var
   data:string;
   rel:relation;
begin
     rel:=ta^.tdatum[1]^.relationptr;

     TupleError:=0;
     (* If we haven't reached the end of the file
        - Check both the physical file & cache *)
     if (not eof(tdatums)) and (not cache_ateof(rel^.rcache,rel^.current_pos)) then
     begin
          ttt:=ta;
          data:=cache_read(rel^.rcache,rel^.current_pos);

          if data='' then
             read_updatecache(tdatums,rel^.rcache,data,rel^.current_pos);

          {readln(tdatums,data);}
          break_Down(data,ta,nofields);
     end
     else       (* If we have reached the end of the file *)
         begin
              tuple_close(ta,tdatums);
         end;
end;



procedure tuple_print(    ta:tuple_attributes;
                          print_hdr:boolean);
(* Print the specified tuple *)

(* Ref. x.3.5 *)
var
   count:word;
   c:word;
begin
     if print_hdr then
        fields_print(ta);

     count:=1;
     while (assigned(ta^.tdatum[count]^.fieldptr)) and (count<MAX_NO_ATTRIBUTES) do
     begin
          write_stdout(ta^.tdatum[count]^.data);

          (* To make it readable, tabulate it *)
          if ta^.tdatum[count+1]^.data<>'' then
             for c:=length(ta^.tdatum[count]^.data) to MAX_SIZE_DATUM_STRING do
                 write_stdout(' ');
          inc(count);
     end;
     writeln_stdout_nl;
end;



function tuple_findfield(    ta:tuple_attributes;
                             fld:string):word;
(* Locates a given field and returns an index ptr to the attribute *)

(* Ref. x.3.6 *)

var
   c:word;
begin
     c:=1;
     do_trace('tff: searching for: '+fld);
     do_trace('tff: upstring');

     fld:=upstring(fld);

     do_trace('tff: upstring done');

     (* Whilst the field is assigned, and we are within its bounds,
        and the field is not the field we are searching for *)
     while (assigned(ta))
       and (c<=MAX_NO_ATTRIBUTES)
       and (assigned(ta^.tdatum[c]))
       and (assigned(ta^.tdatum[c]^.fieldptr))
       and (ta^.tdatum[c]^.fieldptr^.name<>fld) do
       begin
            do_trace('tff: Its not: '+ta^.tdatum[c]^.fieldptr^.name);
           (* Increment the counter *)
           inc(c);
       end;

     do_trace('tff: done search');
     (* If the node is assigned, and we are within bounds, we must have
        found the node, so return the counter *)
     if  (assigned(ta)) and (c<=MAX_NO_ATTRIBUTES)
     and (assigned(ta^.tdatum[c])) and (assigned(ta^.tdatum[c]^.fieldptr)) then
        tuple_findfield:=c
     (* Otherwise, return an error value of 0 *)
     else
         begin
              do_trace('tff: Not found');
              tuple_findfield:=0;
         end
end;




function readfirst_tuple_b(var rel:relation;
                          var tdatums:file;
                          var nofields:word;
                          var spos,epos:word):tuple_Attributes;
(* Reads a tuple into memory, and returns a ptr to the tuple structure,
   but uses a direct access method, to locate the position in the
   file for indexing purposes *)

(* Ref. x.3.7 *)

var
   data:array[1..MAX_FILE_DATA_SIZE] of char;
   s:string;
   noret,x,y:word;
   fld:field;
   ta:tuple_attributes;
   td:tuple_datum;
   count:word;
begin
     (* Prepare the tuple *)
     ta:=tuple_prep;

     (* Populate the tuple with the fields from the specified relation*)
     populate_fields(rel,ta,nofields);


     (* Open the file *)
     assign(tdatums,rel^.filepath+rel^.filename);
     reset(tdatums,1);

     (* Read in the data *)
     blockread(tdatums,data,MAX_FILE_DATA_SIZE,noret);
     tuple_count_inc(incount);
     x:=2;
     while (data[x-1]<>#13) and (data[x]<>#10) do
           inc(x);

     s:='';
     for y:=1 to x-2 do
         s:=concat(s,data[y]);

     epos:=spos+x;

     (* Break down the string into the tuple *)
     break_down(s,ta,nofields);

     (* Return the information *)
     readfirst_tuple_b:=ta;
end;



procedure readnext_tuple_b(var ta:tuple_attributes;
                           var tdatums:file;
                               nofields:word;
                           var spos,epos:word);
(* Reads in the next tuple from a given position *)

(* Ref. x.3.8 *)
var
   data:array[1..MAX_FILE_DATA_SIZE] of char;
   s:string;
   x,y,noret:word;
begin
     TupleError:=0;

     (* Unless we've reached the end of the file *)
     if filesize(tdatums)>spos+2 then (* Check this condition *)
     begin
          (* Locate the start of the tuple *)
          seek(tdatums,spos);

          (* Read in a block of information *)
          blockread(tdatums,data,MAX_FILE_DATA_SIZE,noret);
          x:=2;
          while (data[x-1]<>#13) and (data[x]<>#10) do
                inc(x);

          s:='';
          for y:=1 to x-2 do
              s:=concat(s,data[y]);

          epos:=spos+x;

          (* Break down the string into the specified tuple *)
          break_Down(s,ta,nofields);

          tuple_count_inc(incount);
     end
     else
         begin
              (* Otherwise, we've reached the end of the file and the
                 last tuple, so report an 'error' *)
              tuple_dispose(ta,true,false);
              ta:=nil;
              close(tdatums);
              TupleError:=6; {End of File}
         end;
end;


procedure tuple_count_inc(var tcount:integer);
begin
     inc(tcount);
end;

procedure tuple_count_reset(var tcount:integer);
begin
     tcount:=0;
end;

procedure tuple_count_reset_all;
begin
     tuple_count_reset(incount);
     tuple_count_reset(outcount);
end;

function tuple_to_string(    t:tuple_attributes):string;

(* Copies the data held within the specified tuple, and adds it to
   a string, which is subsequently returned *)

(* Ref. x.3.9 *)

var
   c:word;
   s:string;
begin
     (* Initialise values *)
     c:=1;
     s:='';

     (* Whilst there is data to process *)
     while (t^.tdatum[c]^.data<>'') do
     begin

          (* Concatenate the data to the string *)
          s:=concat(s,t^.tdatum[c]^.data);
          inc(c);
     end;

     (* Return the string *)
     tuple_to_string:=s;
end;



function tuple_to_string_orderbyrel(    t:tuple_attributes;
                                        rel:relation):string;

(* Converts a tuple to a string, in the ordering of the specified
   relation's fields (uses the field names).

   A pre-requisite of this function is that the relation of which
   the tuple is from, and the relation specified are union-
   compatible *)
(* Ref. x.3.10 *)

var
   c:word;
   s:string;
   fld:field;
begin

     (* Find the first field *)
{     fld:=field_findfirst(rel);}

     s:='';

     c:=1;
     (* Whilst the field is valid and assigned *)
     while (c<=MAX_NO_ATTRIBUTES) and (assigned(t^.tdatum[c])) do
     begin
          (* Add the data to the string *)
          {s:=concat(s,t^.tdatum[tuple_findfield(t,field_name(fld))]^.data);}
          s:=concat(s,t^.tdatum[c]^.data);

          inc(c);
          (* Find the next field *)
{          fld:=field_findnext(rel,fld,FALSE,TRUE);}
     end;

     (* Return the string *)
     tuple_to_string_orderbyrel:=s;
end;

function tuple_user_add(    rel:relation):relation;
var
   ta:tuple_Attributes;
   c:byte;
   writepos:word;
   redo:string;
begin
     (* Check that the relation exists *)
     if not assigned(rel) then
        tuple_user_add:=nil
     else
     begin
          (* Create a blank tuple for the relation *)
          ta:=tuple_prepare_fields(rel);

          (* Whilst the tuple is valid *)
          while assigned(ta) do
          begin
               (* Loop through the fields *)
               for c:=1 to (rel^.nofields) do
               begin
                    (* Prompt for input for each field *)
                    write_stdout('Enter data for field '+ta^.tdatum[c]^.fieldptr^.name+': ');

                    (* Read the data into the tuple *)
                    readln(ta^.tdatum[c]^.data);
                    ta^.tdatum[c]^.data:=upstring(ta^.tdatum[c]^.data);
                    writeln_stdoutcpy(ta^.tdatum[c]^.data);
               end;

               (* Write the tuple to disk *)
               write_tuple(ta,writepos);

               (* Check the user wishes to repeat *)
               write_stdout('Again? (Y/N): ');
               readln(redo);
               writeln_stdoutcpy(redo);

               (* If not, then dispose of the memory used by the tuple *)
               if upstring(redo)='N' then
                  tuple_dispose(ta,true,false);
          end;

          (* Return the relation to indicate success *)
          tuple_user_add:=rel;
     end;
end;


function tuple_flex_add(    rel:relation;
                            data:string;
                            seperator:char):relation;
(* First pass at a more flexible tuple add operation. This one assumes
   that the data in the data string is sufficent to add to the relation.
   No fields can be specified. Seperator is the seperator in the data
   seperating datum for each field *)

var
   ta:tuple_Attributes;
   c:byte;
   writepos:word;
   redo,datum:string;
begin
     (* Check that the relation exists *)
     if not assigned(rel) then
        tuple_flex_add:=nil
     else
     begin
          (* Create a blank tuple for the relation *)
          ta:=tuple_prepare_fields(rel);

          (* Loop through the fields *)
          for c:=1 to (rel^.nofields) do
          begin
               (* Extract the data from the data string *)
               datum:=cut_token(data,seperator);

               (* Add the data to the internal representation of the
                  tuple *)
               ta^.tdatum[c]^.data:=datum;
          end;

          (* Write the tuple to disk *)
          write_tuple(ta,writepos);

          tuple_dispose(ta,true,false);

          (* Return the relation to indicate success *)
          tuple_flex_add:=rel;
     end;
end;


begin
     (* Default 'error' *)
     TupleError:=0;
end.


{

Unlimited data size
===================

Making the size of attribute data size unlimited will require quite
a bit of revision to the structure of LEAP's tuple module. Bearing
in mind the issue of how to make the planned port to 'C' as easy as
possible, this will have some added complications.

The best approach will be to use a linked list of structures internally,
and then when data is required, fetch the data in from the structure
as needed. But how to deal with the issue of "if" conditions.

This will probably require some form of comparator. "ifls" (if[largestring])
or something, which takes the operator as a parameter.

loading a tuple will require reading the line in (this will be big
as well), and taking chunks at a time.

You know, as I read and think this through, I think reviewing how 'C'
in Unix deals with strings, and taking it from their. If Unix/C does not
have a maximum data size for strings (I don't think it does), then
not a problem. Its simply a matter of dealing with strings appropriately.


}