unit rt_stack;

(*
Unit: rt_stack
Description: Temporary relation stack operations
References: References to the documentation is to the file rtstack.doc
            and section numbers are given as relative to this document.
Notes: 
Revision Notes:
         14/03/1995 - Initial development.
         27/03/1995 - Extensive commenting.
         29/03/1995 - Documented (rtstack.doc)
	 22/05/1996 - Placed under the GNU General Public License.
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)

interface

uses dtypes;

(* Define the relation stack types *)
type
    rtion_stack=^rtion_stack_def;
    rtion_stack_node=^rtion_stack_node_def;

    (* Header node, with count information *)
    rtion_stack_def=record
                    head_node:rtion_stack_node;
                    no_items:word;
    end;

    (* Stack node, with 'next' information *)
    rtion_stack_node_def=record
                         data:relation;
                         next:rtion_stack_node;
    end;

function rtion_create_stack:rtion_stack;
procedure rtion_push_stack(    st:rtion_stack;
                               data:relation);
procedure rtion_dispose_all(    db:database;
                            var st:rtion_stack);

implementation

uses utils,relations;




function rtion_create_stack:rtion_stack;
(* Create a stack *)

(* Ref. Section x.1.1 *)

var
   stk:rtion_stack;
begin
     (* Create the node *)
     new(stk);
     check_Assign(stk,'rt_stack.rtion_create_stack');

     (* Reset the data *)
     stk^.head_node:=nil;
     stk^.no_items:=0;
     rtion_create_stack:=stk;
end;




procedure rtion_push_stack(    st:rtion_stack;
                               data:relation);
(* Push a datum onto the stack *)

(* Ref. Section x.1.2 *)

var
   st_node:rtion_stack_node;
begin
     (* Create a node *)
     new(st_node);
     check_assign(st_node,'rt_stack.rtion_push_stack');

     (* Populate the fields within the node according to the parameters *)
     st_node^.data:=data;
     st_node^.next:=st^.head_node;
     st^.head_node:=st_node;
     inc(st^.no_items);
end;




function rtion_pop_stack(var st:rtion_stack):relation;
(* Pop a datum off of the stack *)

(* Ref. Section x.1.3 *)

var
   node:rtion_stack_node;
begin
     (* Get the top node *)
     rtion_pop_stack:=st^.head_node^.data;

     node:=st^.head_node;
     st^.head_node:=node^.next;

     (* If its an ok node *)
     if assigned(node) then
     begin
          (* Dispose of its memory *)
          dispose(node);
          node:=nil;
     end
     else
         (* Report an error *)
         nonfatal_error(ERROR_INTERNAL_STACK,'(rtion_pop_stack)');

     (* Update book-keeping information *)
     dec(st^.no_items);
end;





function rtion_stack_size(    st:rtion_stack):word;
(* Return the number of nodes in a stack *)

(* Ref. Section x.2.1 *)

begin
     rtion_stack_size:=st^.no_items;
end;



function rtion_stack_empty(    st:rtion_stack):boolean;
(* Return TRUE if the stack is empty *)

(* Ref. Section x.2.2 *)

begin
     rtion_stack_empty:=(rtion_stack_size(st)=0);
end;



procedure rtion_stack_dispose(var st:rtion_stack);
(* Dispose of the memory used by the stack *)

(* Ref. Section x.2.3 *)

begin
     (* If the node is valid *)
     if assigned(st) then
        (* Dispose of the memory *)
        dispose(st)
     else
         (* Report an error *)
         nonfatal_error(ERROR_INTERNAL_STACK,'(rtion_stack_dispose)');
     st:=nil;
end;




procedure rtion_flush_stack(var st:rtion_stack);
(* Flush all of the memory used by the entire stack *)

(* Ref. Section x.2.4 *)

begin
     (* Whilst there is a node to pop *)
     while not(rtion_stack_empty(st)) do
           (* Pop the node off of the stack *)
           rtion_pop_stack(st);       {TP 7 allows functions to be called
                                        as procedures}

     (* Dispose of the header structure *)
     rtion_stack_dispose(st);
end;




procedure rtion_dispose_all(    db:database;
                            var st:rtion_stack);
(* Processes the stack, disposing of temporary relations completely *)

(* Ref. Section x.1.4 *)

var
   rel:relation;
begin
     (* Whilst the stack is not empty *)
     while not (rtion_stack_empty(st)) do
     begin
          (* Pop a node off of the stack *)
          rel:=rtion_pop_stack(st);

          (* If the relation is temporary *)
          if relation_temporary(rel) then
             (* Remove it from the relation structure and delete files *)
             relation_remove(db,rel);
     end;

     (* Finish off the stack *)
     rtion_flush_stack(st);
end;



end.

