unit rtional;

(*
Unit: rtional
Initial Date: 17/12/1994
Description: Relational operators
References: References to the documentation is to the file rtional.doc
            and section numbers are given as relative to this document.
Notes: 
Revision History:
         17/12/1994 - Initial Development.
         18/12/1994 - Post pointer create check and polite shutdown.
         15/01/1995 - 1st Projection operation implemented and tested.
         16/01/1995 - Hashing functionality.
         18/01/1995 - Union Implementend.
         29/01/1995 - Cartesian Product.
         29/01/1995 - Hashing Intersect/Difference.
         30/01/1995 - Selection (One condition).
         01/02/1995 - Selection (Multiple conditions).
         14/02/1995 - Nested Loop Join (Multiple Conditions).
         12/03/1995 - Debugged crash if relation doesn't exist.
         13/03/1995 - S/M Join automatically creates indexes on join fields if
                      none exist.
         16/03/1995 - Minor modification to select for generality.
         29/03/1995 - Extensive commenting.
         29/03/1995 - Documented (rtional.doc)
         09/04/1995 - Fixed minor bug in rl_union.
         01/06/1995 - Fixed bug with number as specified condition string.
         07/06/1995 - Fixed bug with spaces in a condition string.
         10/02/1996 - Removed duplicate inc(nofields) in relation create
                      ops, done in field_create.
         19/02/1996 - Protected mode operation cleaner.
         22/05/1996 - Placed under the GNU General Public License.
To Do:
???Combine hash_intersect/hash_difference?
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)

interface

{$O+}

uses dtypes;

function rl_join(    db:database;
                     r1,r2:relation;
                     qual:string;
                     destname:string):relation;
function rl_one_pass_join(    db:database;
                              r1,r2:relation;
                              qual:string;
                              name:string):relation;
function rl_project(    db:database;
                        rel:relation;
                        field_list:string;
                        destname:string):relation;

function rl_union(    db:database;
                      rel1,rel2:relation;
                      destname:string):relation;

function rl_hash_intersect(    db:database;
                               rel1,rel2:relation;
                               destname:string):relation;
function rl_hash_difference(    db:database;
                                rel1,rel2:relation;
                                destname:string):relation;
function rl_product(    db:database;
                        rel1,rel2:relation;
                        destname:string):relation;
function rl_select(    db:database;
                       rel:relation;
                       qual:string;
                       destname:string):relation;

function rl_display(   rel:relation):relation;
function rl_duplicate(    db:database;
                          rel:relation;
                          destname:string):relation;
procedure rl_idxprint(    db:database;
                          idx:index_struct_ptr);
function rl_is_union_compatible(    rel1,rel2:relation):boolean;
function Build_Hash_Table(    rel:relation):HashTable;
function rl_getitem(   rel:relation):string;




implementation



uses dcompare,index,relations,utils,fields,tuples,hashing;


function get_first(var text:string):string;
(* Locate a string from the parameter and return it. *)

(* Ref. Section x.2.1 *)

var
   cnt,scnt:word;
begin
     scnt:=1;
     { Fixed bug with >=0 and <=9 which meant numbers were missed }
     while (text[scnt] in seperators) or not ( ((upcase(text[scnt])>='A') and (upcase(text[scnt])<='Z'))
                                               or ( ((text[scnt]>='0') and (text[scnt]<='9') ) ) ) do
           inc(scnt);
     cnt:=scnt;
     while ( ( ( (upcase(text[cnt])>='A') and (upcase(text[cnt])<='Z') )
              or ( (text[cnt]='_') or (text[cnt]='-') or (text[cnt]=quote) or (text[cnt]=dbl_quote) )
              or ( (text[cnt]>='0') and (text[cnt]<='9') )
              or ( text[cnt]='.' ) or ( text[cnt]=' ') ) )
              and (cnt<=length(text)) do
                  { Added cnt<=length(text) brought with number bug }
           inc(cnt);

     (* Copy the text to the return result *)
     get_first:=copy(text,scnt,cnt-scnt);

     (* Delete the data from the parameter *)
     delete(text,scnt,cnt-scnt);
end;


function create_duplicate_rtn_relation(    db:database;
                                           rel:relation;
                                           destname:string):relation;
(* Create a relation with the same attribute names as specified relation *)

(* Ref. Section x.2.2 *)

var
   rtrel:relation;
   fld:field;
   count:byte;
   s:string;
begin
     (* Check the new relation name *)
     if destname='' then
        (* Create a random name *)
        rtrel:=relation_create(db,generate_random_String(REL_NAME_SIZE),true)
     else
         (* Create it with specified name *)
         rtrel:=relation_create(db,destname,true);

     (* If an error occured during creation *)
     if (not assigned(rtrel)) or (not assigned(rel)) then
        create_duplicate_rtn_relation:=nil
     else
     begin
          (* Populate values *)
          rtrel^.nofields:=0;

          (* Get the first field in the source relation *)
          fld:=field_findfirst(rel);
          count:=0;

          (* While the field is valid *)
          while assigned(fld) do
          begin
               (* Create a field in the new relation with the correct
                         name & types *)
               field_Create(rtrel,field_name(fld),field_type(fld));
               inc(count);

               (* If exceeded the max. number of attributes *)
               if count>MAX_NO_ATTRIBUTES then
               begin
                    (* Report an error *)
                    system.str(MAX_NO_ATTRIBUTES,s);
                    nonfatal_error(ERROR_EXCEEDED_FIELD_LIMIT,'Current limit set to: '+s);

                    (* Dispose of the field *)
                    field_dispose(fld);
               end
               else
                   (* Otherwise, locate the next field *)

                   fld:=field_findnext(rel,fld,FALSE,TRUE);
          end;

          (* Close the new relation *)
          relation_close(rtrel);

          (* Return the new relation *)
          create_duplicate_rtn_relation:=rtrel;
     end;
end;



function create_rtn_relation(    db:database;
                                 r1,r2:relation;
                                 name:string):relation;
(* Create a new relation that contains the attributes from two source
   relations *)

(* Ref. Section x.2.3 *)

var
   rtrel:relation;
   f1,f2:field;
   count,c2:byte;
   s:string;
   fnames:array[1..MAX_NO_ATTRIBUTES] of string[25];

   function fname_used(name:string):boolean;
   var
      x:byte;
   begin
        fname_used:=true;

        x:=1;

        while ( (x<=rtrel^.nofields) and (name<>fnames[x]) ) do
              inc(x);

        if x<=rtrel^.nofields then
           fname_used:=true
        else
            fname_used:=false;
   end;

begin
     (* Create and Insert a new relation *)
     if name='' then
        (* If the name is empty, create a random name *)
        rtrel:=relation_create(db,generate_random_string(REL_NAME_SIZE-2),true)
     else
         (* Create the relation with the specified name *)
         rtrel:=relation_create(db,name,true);

     (* If an error occured, or either source relation is empty *)
     if (not assigned(rtrel)) or ( (not assigned(r1)) or (not assigned(r2)))
     then
         (* Return a nil ptr *)
          rtrel:=nil
     else
     begin
          rtrel^.nofields:=0;

          count:=0;
          c2:=1;

          (* Find the first attribute *)
          f1:=field_findfirst(r1);

          (* While the current attribute is valid *)
          while assigned(f1) do
          begin
               (* Already being increased by field_create *)
               {inc(rtrel^.nofields);}

               fnames[c2]:=field_name(f1);
               inc(c2);

               (* Create the attribute with the appropriate name & type *)
               field_create(rtrel,field_name(f1),field_type(f1));
               inc(count);

               (* If no. attributes exceeds maximum number allowable *)
               if count>MAX_NO_ATTRIBUTES then
               begin
                    (* Report an error *)
                    system.str(MAX_NO_ATTRIBUTES,s);
                    nonfatal_error(ERROR_EXCEEDED_FIELD_LIMIT,'Current limit set to: '+s);

                    (* Dispose of the attribute *)
                    field_dispose(f1);
               end
               else
                   (* Otherwise, load the next field *)
                   f1:=field_findnext(r1,f1,FALSE,TRUE);
          end;


          (* Load the first attribute from the second relation *)
          f2:=field_findfirst(r2);

          (* Whilst the current attribute is valid *)
          while (assigned(f2)) do
          begin
               {inc(rtrel^.nofields);}

               if fname_used(field_name(f2)) then
                  insert(relation_name(r2)+'.',f2^.name,1);

               fnames[c2]:=field_name(f2);
               inc(c2);

               (* Create the attribute with the appropriate name & type *)
               field_create(rtrel,field_name(f2),field_type(f2));
               inc(count);

               (* If no. attributes exceeds maximum number allowable *)
               if count>MAX_NO_ATTRIBUTES then
               begin
                    (* Report an error *)
                    system.str(MAX_NO_ATTRIBUTES,s);
                    nonfatal_error(ERROR_EXCEEDED_FIELD_LIMIT,'Current limit set to: '+s);

                    (* Dispose of the attribute *)
                    field_dispose(f2);
               end
               else
                   (* Otherwise, load the next field *)
                   f2:=field_findnext(r2,f2,FALSE,TRUE);
          end;

          (* Close the new relation *)
          relation_close(rtrel);

     end;


     (* Return the appropriate value (nil or ptr) *)
     create_rtn_relation:=rtrel;

end;






function project_create_rtn_relation(    db:database;
                                         rel:relation;
                                         flds,destname:string):relation;
(* Create a relation with the specified fields *)

(* Ref. Section x.2.4 *)

var
   rtrel:relation;
   fld:field;
   field:string;
begin
     (* If the specified name is empty *)
     if destname='' then
        (* Generate a random name for the relation *)
        rtrel:=relation_create(db,generate_random_string(REL_NAME_SIZE-2),true)
     else
         (* Otherwise, create a relation with the specified name *)
         rtrel:=relation_create(db,destname,true);

     (* If an error occured, return nil *)
     if rtrel=nil then
        project_create_rtn_relation:=nil
     else
     begin
          rtrel^.nofields:=0;

          (* Remove the first field name *)
          field:=cut_token(flds,' ');

          fld:=field_find(rel,field);

          (* Whilst the current field name is valid *)
          while field<>'' do
          begin
               (* Create the field with a string data type *)
               field_Create(rtrel,field_name(fld),field_type(fld));

               (* Remove the next field name *)
               field:=cut_token(flds,' ');

               fld:=field_find(rel,field);
          end;

          (* Close the current relation *)
          relation_close(rtrel);

          (* Return the resulting relation *)
          project_create_rtn_relation:=rtrel;
     end
end;



function tuple_concat(    t1,t2:tuple_attributes;
                          r:relation):tuple_attributes;

(* Concatenate two specified tuples, and place in relation r. The two
   tuples must be from the component parts of a joined relation *)

(* Ref. Section x.2.5 *)

var
   rt:tuple_attributes;
   nf1,nf2,c:word;
begin
     (* Create the return tuple with structure from specified relation *)
     rt:=tuple_prepare_fields(r);

     c:=1;
     (* Whilst the attribute from the first tuple is valid *)
     while assigned(t1^.tdatum[c]^.fieldptr) do
     begin
          (* Populate the new tuple with data *)
          rt^.tdatum[c]^.fieldPtr^:=t1^.tdatum[c]^.fieldPtr^;
          rt^.tdatum[c]^.relationPtr^:=r^;
          rt^.tdatum[c]^.data:=t1^.tdatum[c]^.data;
          inc(c);
     end;
     nf1:=c-1;

     (* Whilst the attribute from the second tuple is valid *)
     while assigned(t2^.tdatum[c-nf1]^.fieldptr) do
     begin
          (* Populate the new tuple with data *)
          rt^.tdatum[c]^.fieldPtr^:=t2^.tdatum[c-nf1]^.fieldPtr^;
          if tuple_findfield(rt,t2^.tdatum[c-nf1]^.fieldPtr^.name)<>0 then
             insert(relation_name(r)+'.',rt^.tdatum[c]^.fieldPtr^.name,1);
          rt^.tdatum[c]^.relationPtr^:=r^;
          rt^.tdatum[c]^.data:=t2^.tdatum[c-nf1]^.data;
          inc(c);
     end;

     (* Return the newly created tuple *)
     tuple_concat:=rt;

end;






function evaluate(    expression:string;
                      tuple1,tuple2:tuple_Attributes):boolean;
(* Evaluate the given expression to TRUE or FALSE *)

(* Ref. Section x.2.7 *)


   function get_operator(var text:string):operators;
   (* Search the given text for any operators, and return
      the type of operator using the ops_x_ref array. Delete
      the preceeding text and the operator *)
   var
      cnt:word;
      opstr:string;
   begin
        cnt:=1;

        (* Find first seperator *)
        while ( ( (text[cnt]>='<') and (text[cnt]<='>') ) ) do
              inc(cnt);

        (* Copy the operator *)
        opstr:=copy(text,1,cnt-1);

        (* Delete operator *)
        delete(text,1,cnt-1);

        (* Start searching the x_ref array *)
        cnt:=1;
        while (ops_x_ref[cnt].s<>opstr) do
              inc(cnt);

        (* Return the operator *)
        get_operator:=ops_x_ref[cnt].op;
   end;





   function check_fld(    text:string;
                          ct:tuple_Attributes;
                      var dtype:data_types):string;
   (* Take a string, and expand it if its a field definition,
      or do nothing if its a string *)
   var
      rtnval:string;
      idx:word;
      x:byte;
   begin
        rtnval:='';

        dtype:=undef;

        x:=length(text);
        (* If the last character is a quotation mark [']
           (First char stripped away by get_first) *)
        if ( (text[x]=quote) or (text[x]=dbl_quote)) then
           (* Its data, so cut out the data *)
           rtnval:=copy(text,1,x-1)
        else
            begin
                 do_trace('check_fld: Find field');
                 (* Otherwise, locate the field *)
                 idx:=tuple_findfield(ct,text);

                 do_trace('check_fld: done');


                 if idx=0 then
                 begin
                      do_trace('check_fld: No field found');
                      rtnval:='';
                      dtype:=undef;
                 end
                 else
                 begin
                      (* And return the data associated with this field in the
                         current tuple *)
                      rtnval:=ct^.tdatum[idx]^.data;

                      dtype:=ct^.tdatum[idx]^.fieldptr^.data_type;
                 end
            end;

        (* Return the appropriate data *)
        check_fld:=rtnval;
   end;



var
   ft,vt:data_types;
   fs,vs:string;
   result:boolean;
   fld:field;
   flds,fldsrel:string;
   op:operators;
   vari,varirel:string;
   tuple:tuple_attributes;
   fldtuple,varituple:tuple_attributes;
begin
     do_trace('evaluate: entered');
     result:=false;

     do_trace('evaluate: preparation...');
     strip_leading_spaces(expression);
     strip_tail_spaces(expression);
     do_trace('evaluate: prep done.');

     (* Break down the expression and Find the field *)

     (* Get the first item *)
     flds:=get_First(expression);
     strip_tail_spaces(flds);

     do_trace('evaluate: determine field definition');
     if contains(flds,'.') then
     begin
          do_trace('evaluate: dot delimited');
          fldsrel:=cut_token(flds,'.');
          if assigned(tuple1^.tdatum[1]^.relationptr) and (relation_name(tuple1^.tdatum[1]^.relationptr)=fldsrel) then
             fldtuple:=tuple1
          else
              fldtuple:=tuple2;
     end
     else
     begin
          do_trace('evaluate: rel name');
          fldsrel:=relation_name(tuple1^.tdatum[1]^.relationptr);
          fldtuple:=tuple1;
     end;

     do_trace('evaluate: done');
     (* Get the operator, <;<=;=;>=;> *)

     do_trace('evaluate: get operator');
     op:=get_operator(expression);
     do_trace('evaluate: done');

     (* Get the second item *)
     vari:=get_first(expression);
     strip_tail_spaces(vari);

     do_trace('Determine 2nd field');
     if contains(vari,'.') then
     begin
          varirel:=cut_token(vari,'.');
          if relation_name(tuple1^.tdatum[1]^.relationptr)=varirel then
             varituple:=tuple1
          else
              varituple:=tuple2;
     end
     else
         begin
              varirel:=relation_name(tuple1^.tdatum[1]^.relationptr);
             varituple:=tuple1
{             varituple:=tuple1;}
         end;
     do_trace('evaluate: done');

     fs:=check_fld(flds,fldtuple,ft);
     vs:=check_fld(vari,varituple,vt);

     do_trace('evaluate: do operation...');
     case op of
          LESS:result:=dcompare.compare(fs,vs,ft,vt,'<');
          LESS_THAN_OR_EQUAL:result:=dcompare.compare(fs,vs,ft,vt,'L');
          EQUAL:result:=dcompare.compare(fs,vs,ft,vt,'=');
          GREATER_THAN_OR_EQUAL:result:=dcompare.compare(fs,vs,ft,vt,'G');
          GREATER:result:=dcompare.compare(fs,vs,ft,vt,'>');
          NOT_EQUAL:result:=dcompare.compare(fs,vs,ft,vt,'N');
     end;
     do_trace('evaluate: Done');
{     (* Evaluate the expression according to the type of condition *)
     case op of
          LESS:result:=(check_fld(flds,fldtuple,ft)<check_fld(vari,varituple,vt));
          LESS_THAN_OR_EQUAL:result:=(check_fld(flds,fldtuple)<=check_fld(vari,varituple));
          EQUAL:result:=(check_fld(flds,fldtuple)=check_fld(vari,varituple));
          GREATER_THAN_OR_EQUAL:result:=(check_fld(flds,fldtuple)>=check_fld(vari,varituple));
          GREATER:result:=(check_fld(flds,fldtuple)>check_fld(vari,varituple));
          NOT_EQUAL:result:=(check_fld(flds,fldtuple)<>check_fld(vari,varituple))
     end;

     {
     case op of
          LESS:result:=compare(check_fld(flds,fldtuple),check_fld(vari,varituple),'<');
          LESS_THAN_OR_EQUAL:result:=compare(check_fld(flds,fldtuple),check_fld(vari,varituple),'L');
          EQUAL:result:=(check_fld(flds,fldtuple)=check_fld(vari,varituple));
          GREATER_THAN_OR_EQUAL:result:=(check_fld(flds,fldtuple)>=check_fld(vari,varituple));
          GREATER:result:=(check_fld(flds,fldtuple)>check_fld(vari,varituple));
          NOT_EQUAL:result:=(check_fld(flds,fldtuple)<>check_fld(vari,varituple))
     end;
     }
     (* Return the result *)
     evaluate:=result;
end;





function rl_join(    db:database;
                     r1,r2:relation;
                     qual:string;
                     destname:string):relation;
(* Perform an algebraic join on the two specified relations, where the
   join condition is specified in the string 'qual', and the destination
   relation for the result is contained in the string 'destname' *)

(* Ref. Section x.1.1 *)

var
   rtrel:relation;
   concat_tuple,ta1,ta2:tuple_Attributes;
   isp1,isp2:index_struct_ptr;
   ss1,ss2:np_Stack;
   nofields1,nofields2,k1,k2,nf1,nf2,sp1,ep1,sp2,ep2,fk1,fk2,cctwp:word;
   n1,n2:nodeptr;
   fl1,fl2:file;
   t1,t2:text;
   conds:condition_struct;
   noconditions,count:word;
   boolval:byte;
   s:string;
   output:boolean;
begin
     (* Create the return relation with the specified name *)
     rtrel:=create_rtn_relation(db,r1,r2,destname);

     (* If an error occured, return nil and exit *)
     if not assigned(rtrel) then
     begin
          rl_join:=nil;
          exit;
     end
     else
     begin

          (* Reset condition structure *)
          for count:=MAX_NO_CONDITIONS downto 1 do
          begin
               conds[count].str:='';
               conds[count].bool:=0;
               conds[count].output:=true;
          end;

          (* Remove the first expression from the qualification *)
          s:=get_first_expression(qual,boolval);

          (* Whilst the expression is not empty *)
          while (s<>'') do
          begin
               (* Populate the condition structure *)
               conds[count].str:=s;
               conds[count].bool:=boolval;

               inc(count);

               (* Remove the next expression from the qualification *)
               s:=get_first_expression(qual,boolval);
          end;

          (* Record the number of conditions *)
          noconditions:=count-1;

          (* Get the first tuple from the first relation *)
          ta1:=readfirst_tuple(r1,t1,nofields1);

          (* Whilst the current tuple from the first relation is valid *)
          while assigned(ta1) do
          begin
               (* Get the first tuple from the second relation *)
               ta2:=readfirst_tuple(r2,t2,nofields2);

               (* Whilst the current tuple from the second relation is valid *)
               while assigned(ta2) do
               begin

                    (* Evaluate each of the conditions with the current tuples
                       from each of the relations, and store the current
                       result *)
                    for count:=1 to noconditions do
                         conds[count].output:=evaluate(conds[count].str,ta1,ta2);

                   (* Evaluate all of the results from each of the conditions
                      according to the specified boolean operators, AND,
                      OR etc. *)

                   output:=conds[1].output;
                   count:=2;
                   while (count<=noconditions) and (noconditions>1) do
                   begin
                        case conds[count-1].bool of
                             1:output:=output and conds[count].output;
                             2:output:=output or conds[count].output;
                        end;
                        inc(count);
                   end;


                   (* If the condition evaluates to TRUE *)
                   if output then
                   begin
                        (* Concatenate the two tuples *)
                        concat_tuple:=tuple_concat(ta1,ta2,rtrel);

                        (* Write the tuple to disk *)
                        write_tuple(concat_tuple, cctwp);

                        (* Dispose of the tuple *)
                        tuple_dispose(concat_tuple,true,false);
                   end;

                   (* Get the next tuple from the second relation *)
                   readnext_tuple(ta2,t2,nofields2);
               end;

               (* Get the next tuple from the first relation *)
               readnext_tuple(ta1,t1,nofields1);
          end;

          (* Return the resulting relation *)
          rl_join:=rtrel;
     end;
end;





function eval(var curR:string;
              var crr:tuple_attributes;
              var conds:condition_struct;
              var n:word;
                  ta1,ta2:tuple_Attributes;
                  fk1,fk2,noconditions:word
                  ):boolean;

(* Evaluate the *)
var
   output:boolean;
   count:word;
begin
     curR:=ta1^.tdatum[fk1]^.data;
     crr:=ta1;

     for count:=1 to noconditions do
         conds[count].output:=evaluate(conds[count].str,ta1,ta2);

     count:=2;

     while (count<=noconditions) and (noconditions>1) do
     begin
          case conds[count-1].bool of
               1:output:=output and conds[count].output;
               2:output:=output or conds[count].output;
          end;
          inc(count);
     end;
     eval:=output;
end;



function rl_one_pass_join(    db:database;
                              r1,r2:relation;
                              qual:string;
                              name:string):relation;
(* First run at one pass operator... This has some problems that
   need resolving, namely the use of crr and curR - Getting the next
   key within the concat section is causing crr to be updated...
*)

var
   rtrel:relation;
   crr,concat_tuple,ta1,ta2:tuple_Attributes;
   curR,f1,f2,s:string;
   isp1,isp2:index_struct_ptr;
   ss1,ss2:np_Stack;
   k1,k2,nf1,nf2,sp1,ep1,sp2,ep2,fk1,fk2,cctwp:word;
   n1,n2:nodeptr;
   fl1,fl2:file;
   Xa:array[1..30] of tuple_attributes;
   i,n,noconditions:word;
   conds:condition_Struct;
   boolval:byte;
   output:boolean;
   count:word;
   rel1,fld1,rel2,fld2,qual1:string;
   pka,ska:keyarray;
   idx:index_struct_ptr;
begin
{     check_avail_mem(0);}
     rtrel:=create_rtn_relation(db,r1,r2,name);

     if not assigned(rtrel) then
     begin
          rl_one_pass_join:=nil;
          exit;
     end;

     for count:=MAX_NO_CONDITIONS downto 1 do
     begin
          conds[count].str:='';
          conds[count].bool:=0;
          conds[count].output:=true;
     end;

     qual1:=qual;
     s:=get_first_expression(qual,boolval);

     while (s<>'') do
     begin
          conds[count].str:=s;
          conds[count].bool:=boolval;

          inc(count);
          s:=get_first_expression(qual,boolval);
     end;

     noconditions:=count-1;

     f1:=get_first(qual1);
     if contains(f1,'.') then
     begin
          rel1:=cut_token(f1,'.');
          fld1:=cut_token(f1,' ');
          f1:=fld1;
          fk1:=tuple_findfield(ta1,f1);
     end;
     f2:=get_first(qual1);
     if contains(f2,'.') then
     begin
          rel2:=cut_token(f2,'.');
          fld2:=cut_token(f2,' ');
          f2:=fld2;
          fk2:=tuple_findfield(ta2,f2);
     end;

     isp1:=get_index(idx_master,r1,fld1);
     if not(assigned(isp1)) then
     begin
          pka[1]:=fld1;
          pka[2]:='';
          clear_keyarray(ska);

          idx:=add_index_hdr(idx_master,rel1,pka,ska,nil);
{         display_idx_struct(idx_master);}
          build_index(idx,db,rel1);
          isp1:=get_index(idx_master,r1,fld1);
     end;

     isp2:=get_index(idx_master,relation_find(db,rel2),fld2);
     if not(assigned(isp2)) then
     begin
          pka[1]:=fld2;
          pka[2]:='';
          clear_keyarray(ska);

          idx:=add_index_hdr(idx_master,rel2,pka,ska,nil);
{         display_idx_struct(idx_master);}
          build_index(idx,db,rel2);
          isp2:=get_index(idx_master,relation_find(db,rel2),fld2);
     end;
     ss1:=np_create_stack;

     k1:=0;
     n1:=btree_findfirst(isp1^.index_data.index_struct,k1,ss1);
     ta1:=loadfirst_tuple(isp1^.fileptr,n1^.key[k1].diskAddr,r1);
     fk1:=tuple_findfield(ta1,f1);

     K1:=0;
     ss2:=np_create_stack;
     n2:=btree_findfirst(isp2^.index_data.index_struct,k2,ss2);
     ta2:=loadfirst_tuple(isp2^.fileptr,n2^.key[k2].diskAddr,r2);
     fk2:=tuple_findfield(ta2,f2);

     while assigned(n1) and assigned(n2) do
     begin
          while assigned(n1)
             and assigned(n2)
             and (ta1^.tdatum[fk1]^.data<>ta2^.tdatum[fk2]^.data)
             do
          begin
               if (ta1^.tdatum[fk1]^.data<ta2^.tdatum[fk2]^.data) then
               begin
                    n1:=btree_findnext(n1,k1,ss1);
                    if assigned(n1) then
                       ta1:=loadnext_tuple(isp1^.fileptr,ta1,n1^.key[k1].diskaddr)
               end
               else
               begin
                    n2:=btree_findnext(n2,k2,ss2);
                    if assigned(n2) then
                       ta2:=loadnext_tuple(isp2^.fileptr,ta2,n2^.key[k2].diskaddr)
               end
          end;

               if assigned(n1) or assigned(n2) then
               begin
                    n:=0;
{                    curR:=ta1^.tdatum[fk1]^.data;
                    crr:=ta1;

                    for count:=1 to noconditions do
                        conds[count].output:=evaluate(conds[count].str,ta1,ta2);

                    count:=2;

                    while (count<=noconditions) and (noconditions>1) do
                    begin
                         case conds[count-1].bool of
                              1:output:=output and conds[count].output;
                              2:output:=output or conds[count].output;
                         end;

                         inc(count);
                    end;

                    {We've got to replace these while loops with evaluations
                     of the expressions - Perhaps put the above into a
                     function that returns boolean value?
function eval(var curR:string;
              var crr:tuple_attributes;
              var conds:condition_struct;
                  ta1,ta2:tuple_Attributes;
                  fk1,fk2,noconditions:word
                  ):boolean;

{                    while (ta2^.tdatum[fk2]^.data=ta1^.tdatum[fk1]^.data)}
                    while assigned(n2)
                      and (eval(curR,crr,conds,n,ta1,ta2,fk1,fk2,noconditions))
                       do
                    begin
                         inc(n);
                         Xa[n]:=ta2;
                         n2:=btree_findnext(n2,k2,ss2);
                         ta2:=loadfirst_tuple(isp2^.fileptr,n2^.key[k2].diskAddr,r2);
                    end;

{                    while (eval(curR,crr,conds,n,ta1,ta2,fk1,fk2,noconditions))}
                    while assigned(n1)
                    and   (ta1^.tdatum[fk2]^.data=curR)
                       do begin
                               for i:=1 to n do
                               begin
                                    concat_tuple:=tuple_concat(crr,Xa[i],rtrel);
                                    write_tuple(concat_tuple, cctwp);
                                    insert_tuple_idx(concat_tuple,rtrel,cctwp,idx_master);
                                    tuple_dispose(concat_tuple,true,false);
                                end;
                                tuple_dispose(ta1,true,false);
                                n1:=btree_findnext(n1,k1,ss1);
                                if assigned(n1) then
                                   ta1:=loadfirst_tuple(isp1^.fileptr,n1^.key[k1].diskaddr,r1)
                          end;
                          for i:=1 to n do
                              tuple_dispose(Xa[i],true,false);

               end;
     end;

     np_flush_stack(ss1);
     np_flush_stack(ss2);

     rl_one_pass_join:=rtrel;
end;





function rl_project(    db:database;
                        rel:relation;
                        field_list:string;
                        destname:string):relation;

(* Perform the algebraic operation PROJECT on the specified relation
   and return the resulting relation *)

(* Ref. Section x.1.2 *)


var
   rtrel:relation;
   fld_list,token,s,s2:string;
   fld:field;
   ta:tuple_attributes;
   t:text;
   writepos,c,refpos:word;
   ref:array[1..MAX_NO_ATTRIBUTES] of word;
   nt:tuple_Attributes;
   ht:Hashtable;
   success:boolean;

begin
     (* Clear the array *)
     for c:=1 to MAX_NO_ATTRIBUTES do
         ref[c]:=0;

     (* Create the return relation *)
     rtrel:=project_create_rtn_relation(db,rel,field_list,destname);

     (* If an error occured *)
     if (not assigned(rtrel)) or (not assigned(rel)) then
        (* Return nil *)
        rl_project:=nil
     else
     begin
          (* Otherwise... *)

          (* Create a hash table *)
{          hashing_create(ht);}

          (* Get the first field *)
          fld_list:=field_list;

          (* Get the first field from the field list *)
          token:=cut_token(field_list,' ');

          (* Locate the field in the relation *)
          fld:=field_find(rel,token);

          (* Load the first tuple *)
          ta:=readfirst_tuple(rel,t,rel^.nofields);

          refpos:=1;
          (* Whilst the token/field is valid *)
          while (assigned(ta)) and (token<>'') do
          begin
               token:=upstring(token);
               c:=1;

               (* Locate where it is in the tuple loaded earlier *)
               while (assigned(ta^.tdatum[c]^.fieldPtr) and (token<>ta^.tdatum[c]^.fieldPtr^.name)) do
                     inc(c);

               (* If we haven't overshot *)
               if assigned(ta^.tdatum[c]^.fieldPtr) then
               begin

                    (* Store the position *)
                    ref[refpos]:=c;
                    inc(refpos);
               end
               else
                   nonfatal_error(ERROR_CANTFIND_FLD,'Project: '+token);

               (* Get the next field item *)
               token:=cut_token(field_list,' ');
          end;

          (* Create a new tuple with the fields from the resulting
             relation *)
          nt:=tuple_prepare_fields(rtrel);

          (* Whilst we have a valid tuple *)
          while assigned(ta) do
          begin

               refpos:=1;
               (* Move through our array of fields *)
               while (ref[refpos]<>0) do
               begin
                    (* Populate the data into the new tuple *)
                    nt^.tdatum[refpos]^.data:=ta^.tdatum[ref[refpos]]^.data;
                    inc(refpos);
               end;

{               (* Convert the specified tuple to a string *)
               s:=tuple_to_string(nt);

               (* Try and locate the string version of the tuple in the
                  hash table *)
               hashing_retrieve(ht,s,s2,success);}

{               (* If the tuple was not located in the hash table *)
               if not success then
               begin
                    (* Insert the string version of the tuple into the
                       hash table *)
                    hashing_insert(ht,s,REQ_CALC);}

                    (* Write out the tuple to disk *)
                    write_tuple(nt,writepos);
{               end
               else
                   (* It was found, so report that a 'clash' occured, so
                      we know its working *)
                   report('[Project] Attempted to add duplicate tuple to hash table! ['+s+']');}

               (* Load the next tuple *)
               readnext_tuple(ta,t,rel^.nofields);
          end;

          (* Dispose of our tuple *)
          tuple_dispose(nt,true,false);

{          (* Dispose of the hashing structure *)
          hashing_terminate(ht);}

          (* Return our projection *)
          rl_project:=rtrel;
     end;
end;





function rl_display(   rel:relation):relation;
(* Display a given relation on the screen *)

(* Ref. Section x.1.3 *)

var
   ct:tuple_attributes;
   t:text;
begin

     (* If the specified relation is valid *)
     if assigned(rel) then
     begin
          (* Print the relation name *)
          writeln_stdout('Relation: '+relation_name(rel));

          (* Load the first tuple *)
          ct:=readfirst_tuple(rel,t,rel^.nofields);

          (* Print the fields *)
          flds_print_rel(rel);

          (* While we have a valid tuple *)
          while assigned(ct) do
          begin
               (* Print out the tuple, without field information *)
               tuple_print(ct,false);

               (* Read the next tuple *)
               readnext_tuple(ct,t,rel^.nofields);
          end;

     end
        else
            (* Report that a non-existant relation was almost displayed *)
            nonfatal_Error(ERROR_DISPLAY_NONEX_REL,'');
end;



function rl_getitem(   rel:relation):string;
(* Get the first tuple in the relation, and return as a string *)

var
   ct:tuple_attributes;
   t:text;
begin

     (* If the specified relation is valid *)
     if assigned(rel) then
     begin
          (* Load the first tuple *)
          ct:=readfirst_tuple(rel,t,rel^.nofields);

          (* Return the item *)
          if assigned(ct) then
          begin
               rl_getitem:=tuple_to_string(ct);
               (* Close tuple early *)
               tuple_close(ct,t)
          end
          else
              (* Or return nothing *)
              rl_getitem:='';


     end
        else
            (* Report that a non-existant relation was almost displayed *)
            nonfatal_Error(ERROR_DISPLAY_NONEX_REL,'');
end;



procedure rl_idxprint(    db:database;
                          idx:index_struct_ptr);
(* Print the specified index structure in order *)

(* Ref. Section 1.4 *)

var
   np:nodeptr;
   ct:tuple_attributes;
   k:word;
   ss:np_stack;
   t:file;
   rel:relation;
   spos,epos:word;
begin
     spos:=0;

     (* Create the search stack *)
     ss:=np_create_stack;
     k:=0;

     (* Get the relation the index relates to *)
     rel:=relation_find(db,idx^.index_data.relation);

     (* Get the first node from the index structure *)
     np:=btree_findfirst(idx^.index_data.index_struct,k,ss);

     (* Load the appropriate tuple from disk *)
     ct:=loadfirst_tuple(idx^.fileptr,np^.key[k].diskAddr,relation_find(db,idx^.index_data.relation));



     (* Whilst we have a valid index node *)
     while assigned(np) do
     begin
          (* Display the current tuple *)
          tuple_print(ct,false);

          (* Get the next node from the index structure *)
          np:=btree_findnext(np,k,ss);

          (* Load the appropriate tuple from the disk *)
          loadnext_tuple(idx^.fileptr,ct,np^.key[k].diskaddr);
     end;

     (* Flush out the search stack *)
     np_flush_stack(ss);
end;






function rl_union(    db:database;
                      rel1,rel2:relation;
                      destname:string):relation;

(* Performs the algebraic operation UNION on the specified relations *)

(* Ref. Section x.1.5 *)
var
   c,writepos,count,nofields:word;
   rtrel:relation;
   htable:hashtable;
   ct,nt:tuple_Attributes;
   t:text;
   s,s2:string;
   success:boolean;

begin

     (* Check that the source relations are union compatible *)
     if rl_is_union_compatible(rel1,rel2) then
     begin

          (* Create a relation to return *)
          rtrel:=create_duplicate_rtn_relation(db,rel2,destname);

          (* If no relation was created *)
          if not assigned(rtrel) then
          begin

               (* Return nothing and exit *)
               rl_union:=nil;
               exit;
          end;

          (* Initialise the hashing table *)
{          hashing_create(htable);}

          (* Create the 'output' tuple *)
          nt:=tuple_prepare_fields(rtrel);

          (* Process the first relation, and populate the hashing table *)
          (* Read the first tuple *)
          ct:=readfirst_tuple(rel1,t,nofields);

          (* Whilst valid tuple *)
          while assigned(ct) do
          begin
               (* Convert the tuple to a string for hashing *)
{               s:=tuple_to_string(ct);}

{               (* Insert the tuple into the hashing table *)
               hashing_insert(htable,s,REQ_CALC);}

               c:=1;
               (* Process the first relation *)
               while (c<=nofields) do
               begin
                    nt^.tdatum[c]^.data:=ct^.tdatum[c]^.data;
                    inc(c);
               end;
               write_tuple(nt,writepos);

               (* Load the next tuple *)
               readnext_tuple(ct,t,nofields);
          end;

          (* Process the second relation, and write any tuples not in the
             hash table to the disk - Must ensure that this relation's
             fields are in the same order as the first, so the hashing
             key is the same in both cases *)

          (* Load the first tuple *)
          ct:=readfirst_tuple(rel2,t,nofields);

          (* Whilst valid tuple *)
          while assigned(ct) do
          begin
               (* Convert the tuple to a string, with the fields
                  ordered by the first relation *)
{               s:=tuple_to_string_orderbyrel(ct,rel1);}

               (* Use the string to check if the node has been
                  put into the hashing table *)
{               hashing_retrieve(htable,s,s2,success);}

               (* If it hasn't - write a copy of the node to disk *)
{               if not success then
               begin
}
                    (* Build the data in the output node *)
                    count:=1;
                    while assigned(ct^.tdatum[count]^.fieldptr) do
                    begin
                         nt^.tdatum[count]^.data:=ct^.tdatum[count]^.data;
                         inc(count);
                    end;

                    (* Write the tuple to disk *)
                    write_tuple(nt,writepos);
{               end;}

               (* Load the next tuple from the second relation *)
               readnext_tuple(ct,t,nofields);
          end;

          (* Dispose of the 'output' relation *)
          tuple_dispose(nt,true,false);

          (* Dispose of the hashing table *)
{          hashing_terminate(htable);}
     end
        else
            begin
                 (* If the relations are not union compatible, return
                    nothing, and report an error *)
                 rtrel:=nil;
                 nonfatal_error(ERROR_UNION_COMPATIBILITY,'Union of: '+relation_name(rel1)+' and '+relation_name(rel2));
            end;

     (* Return the contents of rtrel, which will be a relation if
        successful, or nil if not *)
     rl_union:=rtrel;
end;



function rl_intersection(    db:database;
                             rel1,rel2:relation;
                             destname:string):relation;

(* Unfinished intersection algorithm using a similair approach to the
   one pass algorithm used in the rl_one_pass_join operator, but development
   stopped after the idea of the use of hashing was successfully implemented *)

var
   rtrel:relation;
   isp1,isp2:index_Struct_ptr;
   ss1,ss2:np_Stack;
   refpos,k1,k2:word;
   n1,n2:nodeptr;
   ta1,ta2,nt:tuple_Attributes;
   key1,key2:word;


begin
     rtrel:=create_duplicate_rtn_relation(db,rel1,destname);

     if not assigned(rtrel) then
     begin
          rl_intersection:=nil;
          exit;
     end;

     (* Get the index structure for the first field in each relation,
        which is the default primary key - Must ensure that this is
        always created... *)
     isp1:=get_index(idx_master,rel1,field_name(field_findfirst(rel1)));
     isp2:=get_index(idx_master,rel2,field_name(field_findfirst(rel2)));

     (* Create our search stack for traversing the indices *)
     ss1:=np_create_Stack;
     ss2:=np_create_Stack;
     k1:=0;
     k2:=0;

     n1:=btree_findfirst(isp1^.index_data.index_struct,k1,ss1);
     n2:=btree_findfirst(isp2^.index_Data.index_Struct,k2,ss2);
     ta1:=loadfirst_tuple(isp1^.fileptr,n1^.key[k1].diskAddr,rel1);
     ta2:=loadfirst_tuple(isp2^.fileptr,n2^.key[k2].diskAddr,rel2);

     key1:=tuple_findfield(ta1,field_name(field_findfirst(rel1)));
     key2:=tuple_findfield(ta1,field_name(field_findfirst(rel2)));

     nt:=tuple_prepare_fields(rtrel);

     while assigned(n1) and assigned(n2) do
     begin
          while assigned(n1) and assigned(n2)
            and (ta1^.tdatum[key1]^.data<>ta1^.tdatum[key2]^.data)
             do
          begin
               if (ta1^.tdatum[key1]^.data<ta2^.tdatum[key2]^.data) then
               begin
                    n1:=btree_findnext(n1,k1,ss1);
                    if assigned(n1) then
                       ta1:=loadnext_tuple(isp1^.fileptr,ta1,n1^.key[k1].diskaddr)
               end
               else
               begin
                    n2:=btree_findnext(n2,k2,ss2);
                    if assigned(n2) then
                       ta2:=loadnext_tuple(isp2^.fileptr,ta2,n2^.key[k2].diskaddr)
               end
          end;

          if assigned(n1) or assigned(n2) then
          begin
               refpos:=1;
               while assigned(ta1^.tdatum[refpos]^.fieldptr) do
               begin
                    nt^.tdatum[refpos]^.data:=ta1^.tdatum[refpos]^.data;
                    inc(refpos);
               end;

               n1:=btree_findnext(n1,k1,ss1);
               if assigned(n1) then
                  ta1:=loadnext_tuple(isp1^.fileptr,ta1,n1^.key[k1].diskaddr);
               n2:=btree_findnext(n2,k2,ss2);
               if assigned(n2) then
                  ta2:=loadnext_tuple(isp2^.fileptr,ta2,n2^.key[k2].diskaddr)

          end;
     end;
end;





function rl_hash_intersect(    db:database;
                               rel1,rel2:relation;
                               destname:string):relation;
(* Intersection through the use of hashing.
   This algorithm uses a two pass approach - the first pass is of the
   first relation, whose tuples are inserted into the hash table. The
   second pass is of the second relation - and its tuples are checked
   against the hash table - if clashes occur, then the tuple exists
   in both relations, and may safely be added to the output relation *)

(* Ref. Section x.1.6 *)

var
   writepos,count,nofields:word;
   rtrel:relation;
   htable:hashtable;
   ct,nt:tuple_Attributes;
   t:text;
   s,s2:string;
   success:boolean;

begin

     (* Check that the source relations are union compatible *)
     if rl_is_union_compatible(rel1,rel2) then
     begin

          (* Create a relation to return *)
          rtrel:=create_duplicate_rtn_relation(db,rel2,destname);

          (* If no relation was created *)
          if not assigned(rtrel) then
          begin

               (* Return nothing and exit *)
               rl_hash_intersect:=nil;
               exit;
          end;

          (* Initialise the hashing table *)
          hashing_create(htable);

          (* Create the 'output' tuple *)
          nt:=tuple_prepare_fields(rtrel);

          (* Process the first relation, and populate the hashing table *)
          (* Read the first tuple *)
          ct:=readfirst_tuple(rel1,t,nofields);

          (* Whilst valid tuple *)
          while assigned(ct) do
          begin
               (* Convert the tuple to a string for hashing *)
               s:=tuple_to_string(ct);

               (* Insert the tuple into the hashing table *)
               hashing_insert(htable,s,REQ_CALC);

               (* Load the next tuple *)
               readnext_tuple(ct,t,nofields);
          end;

          (* Process the second relation, and write any duplicates to
             the disk - Must ensure that this relation's fields are in
             the same order as the first, so the hashing key is the
             same in both cases *)

          (* Load the first tuple *)
          ct:=readfirst_tuple(rel2,t,nofields);

          (* Whilst valid tuple *)
          while assigned(ct) do
          begin
               (* Convert the tuple to a string, with the fields
                  ordered by the first relation *)
               s:=tuple_to_string_orderbyrel(ct,rel1);

               (* Use the string to check if the node has been
                  put into the hashing table *)
               hashing_retrieve(htable,s,s2,success);

               (* If it has write a copy of the node to disk *)
               if success then
               begin

                    (* Build the data in the output node *)
                    count:=1;
                    while assigned(ct^.tdatum[count]^.fieldptr) do
                    begin
                         nt^.tdatum[count]^.data:=ct^.tdatum[count]^.data;
                         inc(count);
                    end;

                    (* Write the tuple to disk *)
                    write_tuple(nt,writepos);
               end;

               (* Load the next tuple from the second relation *)
               readnext_tuple(ct,t,nofields);
          end;

          (* Dispose of the 'output' relation *)
          tuple_dispose(nt,true,false);

          (* Dispose of the hashing table *)
          hashing_terminate(htable);
     end
        else
            begin
                 (* If the relations are not union compatible, return
                    nothing, and report an error *)
                 rtrel:=nil;
                 nonfatal_error(ERROR_UNION_COMPATIBILITY,'Intersection of: '+relation_name(rel1)+' and '+relation_name(rel2));
            end;

     (* Return the contents of rtrel, which will be a relation if
        successful, or nil if not *)
     rl_hash_intersect:=rtrel;
end;





function rl_hash_difference(    db:database;
                                rel1,rel2:relation;
                                destname:string):relation;
(* Difference through the use of hashing.
   This algorithm is similair to hash_intersect - The first pass puts
   all of the second relations tuples into the hash table. The second pass
   is of the first relation - and its tuples are checked against the hash
   table - if clashes occur - then the tuple is in both, if clashes do not
   occur, then the tuple may be written to the output relation.
*)

(* Ref. Section x.1.7 *)

var
   writepos,count,nofields:word;
   rtrel:relation;
   htable:hashtable;
   ct,nt:tuple_Attributes;
   t:text;
   s,s2:string;
   success:boolean;

begin

     (* Check the relations for union compatibility *)
     if rl_is_union_compatible(rel1,rel2) then
     begin

          (* If union compatibe, create a relation *)
          rtrel:=create_duplicate_rtn_relation(db,rel2,destname);

          (* If no relation could be created *)
          if not assigned(rtrel) then
          begin
               (* Return nothing and exit *)
               rl_hash_difference:=nil;
               exit;
          end;

          (* Create the hashing table *)
          hashing_create(htable);

          (* Create the 'output' tuple *)
          nt:=tuple_prepare_fields(rtrel);

          (* Process the second relation, and populate the hashing table *)
          ct:=readfirst_tuple(rel2,t,nofields);

          while assigned(ct) do
          begin
               (* Convert the tuple to a string *)
               s:=tuple_to_string(ct);

               (* Insert the string into the hashing table *)
               hashing_insert(htable,s,REQ_CALC);

               (* Load the next tuple *)
               readnext_tuple(ct,t,nofields);
          end;

          (* Process the first relation, and write any uniques to
             the disk - Must ensure that this relation's fields are in
             the same order as the first, so the hashing key is the
             same in both cases *)

          (* Load the first tuple from the first relation *)
          ct:=readfirst_tuple(rel1,t,nofields);

          (* Whilst valid tuple *)
          while assigned(ct) do
          begin
               (* Convert the tuple to a string, ordered by the first
                  relation *)
               s:=tuple_to_string_orderbyrel(ct,rel2);

               (* Check the existance of this tuple in the hashing table *)
               hashing_retrieve(htable,s,s2,success);

               (* If it DOESN'T exist in the table *)
               if not success then
               begin

                    (* Build the 'output' tuple's data *)
                    count:=1;
                    while assigned(ct^.tdatum[count]^.fieldptr) do
                    begin
                         nt^.tdatum[count]^.data:=ct^.tdatum[count]^.data;
                         inc(count);
                    end;

                    (* Write the output tuple *)
                    write_tuple(nt,writepos);
               end;

               (* Load the next tuple *)
               readnext_tuple(ct,t,nofields);
          end;

          (* Dispose of the 'output' tuple *)
          tuple_dispose(nt,true,false);

          (* Dispose of the hashing table *)
          hashing_terminate(htable);
     end
        else
            begin
                 (* If the relations are not union compatible, then
                    return nothing and report an error *)
                 rtrel:=nil;
                 nonfatal_error(ERROR_UNION_COMPATIBILITY,'Difference of: '+relation_name(rel1)+' and '+relation_name(rel2));
            end;

     (* Return the appropriate result *)
     rl_hash_difference:=rtrel;
end;





function rl_is_union_compatible(    rel1,rel2:relation):boolean;
(* Checks that the two specified relations are union compatible
   and returns TRUE if this is the case, FALSE in all other cases *)

(* Ref. Section x.2.6 *)


(* We require a type here for the passing as a parameter *)
type
    string_array=array[1..MAX_NO_ATTRIBUTES] of string[MAX_SIZE_DATUM_STRING];

var
   result:boolean;
   r1,r2:string_Array;
   nofields,idx1,idx2:word;

             (* Internal procedure for populating the array *)
             procedure populate(var r:string_Array;
                                    rel:relation);
             (* Populates the specified array with the fields from the
                specified relation *)
             var
                fld:field;
                count:word;
             begin
                  count:=0;

                  (* Load the first field *)
                  fld:=field_findfirst(rel);

                  (* Whilst valid field *)
                  while assigned(fld) do
                  begin
                       inc(count);

                       (* Drop the field name in the array *)
                       r[count]:=field_name(fld);

                       (* Get the next field name *)
                       fld:=field_findnext(rel,fld,FALSE,TRUE);
                  end;
             end;

begin
     (* Reset the return value *)
     result:=false;

     (* First check the number of fields *)
     if ( (assigned(rel1) and assigned(rel2))
     and  (rel1^.nofields=rel2^.nofields)) then
     begin
          (* Make a copy of one of the relations nofields value as its
             used so often, it'll be a small efficency *)
          nofields:=rel1^.nofields;

          (* Get the fields from the two relations *)
          populate(r1,rel1);
          populate(r2,rel2);

          idx1:=1;

          (* Whilst nothing has happened to abort, and the index of the
             first relation is within bounds *)
          while (not result) and (idx1 <= nofields ) do
          begin

               (* Scan the second relation for a matching field *)
               idx2:=1;
               while (r1[idx1]<>r2[idx2]) and (idx2<=nofields) do
                     inc(idx2);

               (* If we went over the # fields, then we've found a field
                  that doesn't exist *)
               if idx2>nofields then
                  (* Cause an error through negation later *)
                  result:=true
               else
                   (* Move onto the next field *)
                   inc(idx1);
          end;

          (* Negate the result, because for union compatibility post-
             requisite of loop will be result=FALSE *)
          result:=not result;

     end;

     (* Return the contents of result *)
     rl_is_union_compatible:=result;
end;





function rl_product(    db:database;
                        rel1,rel2:relation;
                        destname:string):relation;
(* Produce the cartesian product of the two specified relations *)

(* Ref. Section x.1.8 *)

var
   rtrel:relation;
   ct1,ct2,nt:tuple_attributes;
   t1,t2:text;
   writepos,count,nofields1,nofields2:word;

begin
     (* Create a relation *)
     rtrel:=create_rtn_relation(db,rel1,rel2,destname);

     (* If no relation was created *)
     if not assigned(rtrel) then
     begin
          (* Return nothing and exit *)
          rl_product:=nil;
          exit;
     end;

     (* Create the 'output' tuple *)
     nt:=tuple_prepare_fields(rtrel);

     (* Load the first tuple from the first relation *)
     ct1:=readfirst_tuple(rel1,t1,nofields1);

     (* Whilst valid tuple *)
     while assigned(ct1) do
     begin

          (* Load the first tuple from the second relation *)
          ct2:=readfirst_tuple(rel2,t2,nofields2);

          (* Whilst valid tuple *)
          while assigned(ct2) do
          begin

               (* Put the contents of the first tuple into the first
                  part of the 'output' tuple *)
               count:=1;
               while assigned(ct1^.tdatum[count]^.fieldptr) do
               begin
                    nt^.tdatum[count]^.data:=ct1^.tdatum[count]^.data;
                    inc(count);
               end;

               (* Without resetting the counter (needed for 'output' tuple),
                  put the contents of the second tuple into the 'output' tuple *)
               while assigned(ct2^.tdatum[count-nofields1]^.fieldptr) do
               begin
                    nt^.tdatum[count]^.data:=ct2^.tdatum[count-nofields1]^.data;
                    inc(count);
               end;

               (* Write the 'output' tuple to disk *)
               write_tuple(nt,writepos);

               (* Read the next tuple from the second relation *)
               readnext_tuple(ct2,t2,nofields2);
          end;

          (* Read the next tuple from the first relation *)
          readnext_tuple(ct1,t1,nofields1);
     end;

     (* Dispose of the 'output' tuple *)
     tuple_dispose(nt,true,false);

     (* Return the result *)
     rl_product:=rtrel;
end;





function rl_duplicate(    db:database;
                          rel:relation;
                          destname:string):relation;
var
   rtrel:relation;
   ct,nt:tuple_attributes;
   t:text;
   writepos,nofields:word;
   count:byte;

begin
     rtrel:=create_duplicate_rtn_relation(db,rel,destname);

     if assigned(rtrel) then
     begin
          rl_duplicate:=rtrel;

          (* Read the first tuple from the relation *)
          ct:=readfirst_tuple(rel,t,nofields);

          (* Prepare the 'output' tuple *)
          nt:=tuple_prepare_fields(rtrel);

          while assigned(ct) do
          begin
               count:=1;

               (* While the current tuple node is valid *)
               while assigned(ct^.tdatum[count]^.fieldptr) do
               begin
                    (* Populate the new tuple's data *)
                    nt^.tdatum[count]^.data:=ct^.tdatum[count]^.data;
                    inc(count);
               end;

               (* Write the data to the 'output' tuple *)
               write_tuple(nt,writepos);

               readnext_tuple(ct,t,nofields);
          end;

          tuple_dispose(nt,true,false);
     end
     else
         rl_duplicate:=nil;
end;



function rl_select(    db:database;
                       rel:relation;
                       qual:string;
                       destname:string):relation;

(* Selects the given tuples (according to the qualification specified)
   from the given relation, and store the result in a relation created
   as destname, or if this is empty, a new relation *)

(* Ref. Section x.1.9 *)


var
   rtrel:relation;
   nt,ct:tuple_attributes;
   t:text;
   writepos,noconditions,count,nofields:word;
   output:boolean;
   s,op,condition:string;
   conds:array[1..MAX_NO_CONDITIONS] of record
                                     str:string;
                                     bool:byte;
                                     output:boolean;
   end;
   boolval,boolval1,boolval2:byte;

begin
     do_trace('Select.');
     (* Make a copy of the condition part *)
     condition:=qual;

     (* Create the relation to return *)
     rtrel:=create_duplicate_rtn_relation(db,rel,destname);

     (* If no relation was created *)
     if (not assigned(rtrel)) or (not assigned(rel)) then
     begin

          (* Return nothing and exit *)
          rl_select:=nil;
          exit;
     end;

     do_trace('Select: starting...');

     (* Read the first tuple from the relation *)
     ct:=readfirst_tuple(rel,t,nofields);

     (* Prepare the 'output' tuple *)
     nt:=tuple_prepare_fields(rtrel);

     do_trace('select: Resetting values.');

     (* Reset the array appropriately *)
     for count:=MAX_NO_CONDITIONS downto 1 do
     begin
          conds[count].str:='';
          conds[count].bool:=0;
     end;

     do_trace('select: get expressions.');

     (* Load the first expression passed *)
     s:=get_first_expression(condition,boolval);

     (* While the expression is containing something *)
     while (s<>'') do
     begin

          (* Store it in the array *)
          conds[count].str:=s;
          conds[count].bool:=boolval;

          (* Load the next expression *)
          inc(count);
          s:=get_first_expression(condition,boolval);
     end;

     do_trace('Done getting expressions');

     (* Store a number of conditions that were read *)
     noconditions:=count-1;


     (* Whilst the tuple read is valid *)
     while assigned(ct) do
     begin
          do_trace('select: tuple is valid');

          do_trace('select: evaluate expression');
          (* Evaluate each of the conditions, and store the result *)
          for count:=1 to noconditions do
              conds[count].output:=evaluate(conds[count].str,ct,nil);

          do_trace('select: done evaluation');

          (* Get the first result *)
          output:=conds[1].output;

          (* Starting at the second result *)
          count:=2;

          (* If we've got more than one condition *)
          while (count<=noconditions) and (noconditions>1) do
          begin
                  (* Incorporate the result of the previous
                     conditions into the current result accordingly -
                     each AND or OR is stored with the *previous*
                     condition result, hence count-1 *)
                  case conds[count-1].bool of
                       1 : output:=output and conds[count].output;
                       2 : output:=output or conds[count].output;
                  end;

                  (* Move onto the next condition *)
                  inc(count);
          end;

          (* If we're to output the result *)
          if output then
          begin
               do_trace('select: Output tuple');
               (* Build the 'output' tuple's data *)
               count:=1;

               (* While the current tuple node is valid *)
               while assigned(ct^.tdatum[count]^.fieldptr) do
               begin
                    (* Populate the new tuple's data *)
                    nt^.tdatum[count]^.data:=ct^.tdatum[count]^.data;
                    inc(count);
               end;

               do_trace('select: write tuple');
               (* Write the data to the 'output' tuple *)
               write_tuple(nt,writepos);
               do_trace('select: done write tuple');
          end;

          do_trace('select: reading next tuple...');
          (* Load the next tuple *)
          readnext_tuple(ct,t,nofields);
          do_trace('select: read next tuple.');
     end;

     do_trace('select: Tidying up...');
     (* Dispose of the 'output' tuple *)
     tuple_dispose(nt,true,false);

     (* Return the relation *)
     rl_select:=rtrel;
end;





function Build_Hash_Table(    rel:relation):HashTable;
(* Build a hash table, and return a pointer to it. *)

(* This operation is not really appropriate here, but it
   is easier to place it here than to mess around with
   other units at this stage. It must be moved sometime *)
var
   htable:HashTable;
   ct:tuple_attributes;
   s:string;
   t:text;
   nofields:word;
begin
          (* Create the hashing table *)
          hashing_create(htable);

          (* Process the second relation, and populate the hashing table *)
          ct:=readfirst_tuple(rel,t,nofields);

          while assigned(ct) do
          begin
               (* Convert the tuple to a string *)
               s:=tuple_to_string(ct);

               (* Insert the string into the hashing table *)
               hashing_insert(htable,s,REQ_CALC);

               (* Load the next tuple *)
               readnext_tuple(ct,t,nofields);
          end;

          Build_Hash_Table:=htable;
end;



end.
