unit relations;

interface

(*
Unit: Relations
Description: Manages the relational reference structure, and
             traversal of it. Provides an interface to the
             relation data type
Notes: 
References: References to the documentation is to the file relation.doc
            and section numbers are given as relative to this document.
Revision History:
         18/12/1994 - Post pointer create check and polite shutdown
         12/03/1995 - User create relation.
         26/03/1995 - Extensive commenting.
         27/03/1995 - Documentation (relation.doc)
         27/05/1995 - Fixed load relations bug. (B0.9)
         23/06/1995 - Introduced check for wildcards in relation names.
         22/05/1996 - Placed under the GNU General Public License.

To-do:
      Duplicity of relation_read and relation_create ?
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)
{$O+}

uses dtypes;

function relation_create(    db:database;
                             relation_name:string;
                             temp:boolean):relation;
procedure relation_display(db:database);
procedure relation_close(    rel:relation);
procedure relation_dispose(    db:database;
                           var rel:relation);
procedure relation_clean(var db:database);
procedure relations_dispose_temp(    db:database);
procedure delete_relation(    db:database;
                              rel:relation);
procedure relation_remove(    db:database;
                          var rel:relation);
procedure relations_dispose_all(    db:database);
function relation_findfirst(db:database):relation;
function relation_findnext(rel:relation):relation;
function relation_find(    db:database;
                  name:string):relation;
procedure open_relations(    db:database);
function relation_name(rel:relation):string;
function relation_temporary(rel:relation):boolean;
procedure relation_change(    db:database;
                              name:string);
function relation_user_create(    db:database;
                                  name,fields:string):relation;
procedure relation_open_fileptr(    rel:relation);
procedure relation_reset_relfile(    rel:relation);
procedure relation_close_relfile(    rel:relation);
procedure relation_Reset_fldfile(    rel:relation);
procedure relation_close_fldfile(    rel:relation);
procedure relation_info(    db:database);
function relation_rename(    db:database;
                             rel1,rel2:string;
                             destname:string):relation;


implementation

uses crt,utils,dos,fields,hashing,rtional,dbase,caching,datadictionary;



procedure relation_insert(    db:database;
                              newRel:relation);
(* Insert a relation into the relation structure *)

(* Ref. Section x.3.1 *)
var
   currentRel,previousRel:relation;

begin
     (* If we don't have a relation in the structure *)
          if db^.firstRelation=nil then
             db^.firstRelation:=newRel
          else
          begin
               currentRel:=db^.firstRelation;
               previousRel:=nil;
               (* Locate the position at which to insert the relation *)
               while (assigned(currentrel)) and (currentRel^.name<newRel^.name) do
               begin
                    previousRel:=currentRel;
                    currentRel:=currentRel^.storage.next;
               end;

               (* We're to insert it at the start of the structure *)
               if (previousRel=nil) then
               begin
                    db^.firstRelation^.storage.previous:=newRel;
                    newRel^.storage.next:=db^.firstRelation;
                    db^.firstRelation:=newRel;
               end
               else
                   (* We're to insert it at the end of the structure *)
                   if (currentRel=nil) then
                   begin
                        previousRel^.storage.next:=newRel;
                        newRel^.storage.previous:=previousRel;
                        db^.lastRelation:=newRel;
                   end
                   else
                       (* We're to insert it *normally!* *)
                   begin
                        previousRel^.storage.next:=newRel;
                        currentRel^.storage.previous:=newRel;
                        newRel^.storage.previous:=previousRel;
                        newRel^.storage.next:=currentRel;
                   end;
          end;
end;



procedure delete_relation(    db:database;
                              rel:relation);
(* Delete the files associated with the given relation *)

(* Ref. Section x.3.2 *)
var
   error:integer;
   s:string;
   rlfile:file;
   dirinfo:searchrec;
begin
     (* Find the first file of the specified relation *)
     findfirst(get_basedir(db)+RELATION_DIR+'\'+relation_name(rel)+'.*',anyfile,dirinfo);
     while doserror=0 do
     begin
          (* Assign a file descriptor to the file *)
          assign(rlfile,get_basedir(db)+RELATION_DIR+'\'+dirinfo.name);

          (* Erase the file *)
          {$I-}
          erase(rlfile);
          {$I+}

          (* If an error occured *)
          error:=IOResult;
          if error<>0 then
          begin
               (* Cause a fatal error with report *)
               system.str(error,s);
               fatal_Error(ERROR_ERASE_FILE,'File: '+dirinfo.name+' Error #'+s);
          end
             else
                 (* Report the successful erasing of the file *)
                 report('Erased file '+dirinfo.name);
          (* Locate the next file of the relation *)
          findnext(dirinfo);
     end;
end;



procedure relation_dispose(    db:database;
                           var rel:relation);
(* Dispose of a relation *)

(* Ref. Section x.2.2 *)
begin
     (* If the relation exists *)
     if assigned(rel) then
     begin
          (* If the relation is temporary *)
          if rel^.temporary then
             if ((get_dbname(db)=MASTER_DB_NAME) and (dtypes.shutdown))
              or (get_dbname(db)<>MASTER_DB_NAME) then
             (* Delete the relation files from disk *)
             delete_relation(db,rel);


{          if db^.firstrelation=rel then
             db^.firstrelation:=rel^.storage.next
          else
              rel^.storage.previous^.storage.next:=rel^.storage.next;
}
          cache_destroy(rel^.rcache);

          (* Dispose of the relation hash table *)
          Hashing_Terminate(rel^.hash_table);

          (* Dispose of the memory associated with the relation *)
          dispose(rel);
          rel:=nil;
     end
        else
            (* Attempted to delete a non-existant relation *)
            nonfatal_error(ERROR_DELETE_NONEX_REL,'(relation_dispose)');
end;


procedure relation_remove(    db:database;
                          var rel:relation);
(* Remove a relation from the master relation structure *)

(* Ref. Section x.2.3 *)

begin
     (* If the relation does not exist *)
     if not assigned(rel) then
        (* Report an error *)
        nonfatal_error(ERROR_DELETE_NONEX_REL,'(relation_remove)')
     else
     begin
          (* If the relation is the first relation *)
          if (rel=db^.firstrelation) then
          begin
               (* Set the header to the next relation *)
               db^.firstrelation:=rel^.storage.next;
          end
          else
              (* The relation is in the body somewhere *)
          begin
               (* Remove the relation from the structure *)
               rel^.storage.previous^.storage.next:=rel^.storage.next;
               rel^.storage.next^.storage.previous:=rel^.storage.previous;
          end;

          (* Remove hashing table from memory *)
          if assigned(rel^.hash_table) then
             hashing_terminate(rel^.hash_table);

          cache_destroy(rel^.rcache);

          (* If the relation is temporary *)
          if rel^.temporary then
             (* Delete the files *)
             delete_relation(db,rel);

          (* Dispose of the memory utilised *)
          dispose(rel);

          rel:=nil;
     end;
end;



procedure relations_dispose_all(    db:database);
(* Dispose of all relations in a relation structure *)

(* Ref. Section x.2.4 *)

var
   prel,crel:relation;
begin
     writeln_stdout('--- Updating Hash Tables ---');
     (* Locate the first relation *)
     crel:=relation_findfirst(db);
     (* Whilst the relation is assigned *)
     while (assigned(crel)) do
     begin
          write_stdout(relation_name(crel)+' ');

          if not crel^.temporary then
          begin
              Hashing_Save(crel^.hash_table,get_basedir(db)+RELATION_DIR+'\'+relation_name(crel)+HASH_EXT);
          end
          else
              write_stdout('(T) ');

          (* Get the next relation *)
          crel:=relation_findnext(crel);
     end;
     writeln_stdout_nl;

     writeln_stdout('--- Disposing relations ---');

     (* Locate the first relation *)
     crel:=relation_findfirst(db);

     (* Whilst the relation is assigned *)
     while (assigned(crel)) do
     begin
          (* Make a copy *)
          prel:=crel;

          (* Get the next relation *)
          crel:=relation_findnext(crel);

          (* Dispose of the relation's memory *)
          relation_dispose(db,prel);
     end;

     (* Set the header to nil *)
     db^.firstRelation:=nil;
end;


procedure relations_dispose_temp(    db:database);
(* Dispose of all Temporary relations in a relation structure *)

var
   prel,crel:relation;
begin
     writeln_stdout('--- Disposing temporary relations ---');

     (* Locate the first relation *)
     crel:=relation_findfirst(db);

     (* Whilst the relation is assigned *)
     while (assigned(crel)) do
     begin
          (* Make a copy *)
          prel:=crel;

          (* Get the next relation *)
          crel:=relation_findnext(crel);


          if relation_temporary(prel) then
          begin
               writeln_stdout('--- '+relation_name(prel));
               (* Dispose of the relation's memory *)
               relation_dispose(db,prel);
          end;
     end;

end;



procedure create_tempfile(    db:database;
                              relation_name:string);
(* Make the file indicating that a relation is temporary *)

(* Ref. Section x.3.3 *)
var
   tempfile:text;
begin
     (* Create the file *)
     assign(tempfile,get_basedir(db)+RELATION_DIR+'\'+relation_name+RELATION_TEMPORARY_IND);
     {$I-}
     rewrite(tempfile);
     {$I+}

     (* If an error occured, then report a fatal error *)
     if IOResult<>0 then
        fatal_error(ERROR_FILE_OPENING,'Temporary Indicator File: '+relation_name+RELATION_TEMPORARY_IND);

     (* Otherwise, add some information to the file to indicate what it is *)
     writeln(tempfile,RELATION_TEMPORARY_TXT+#13+#10+'Relation: '+relation_name);

     (* Close the file *)
     close(tempfile);
end;




function relation_create(    db:database;
                             relation_name:string;
                             temp:boolean):relation;
(* Create a relation *)

(* Ref. Section x.2.1 *)

var
   tempfile:text;
   rel:relation;
   rf:relation;
   s:string;
begin
     relation_name:=upstring(relation_name);
     s:=relation_name;

     if (pos('*',s)<>0) or (pos('?',s)<>0) then
     begin
          nonfatal_error(ERROR_WILDCARD_IN_NAME,'Relation '+relation_name);
          rel:=nil
     end
     else
     begin
         (* Check that the relation doesn't already exist *)
         rf:=relation_find(db,relation_name);

         (* If the relation exists *)
         if assigned(rf) then
         begin
              (* Report the duplicate *)
              nonfatal_error(ERROR_DUPLICATE_ITEM,'relation: '+relation_name);

              (* Return nil, indicating failure to calling procedure *)
              rel:=nil;
         end
         else
         begin
              (* Create a new relation *)
              new(rel);
              check_Assign(rel,'relation.relation_Create');

              (* With the new relation *)
              with rel^ do
              begin
                   (* Set its name *)
                   name:=relation_name;

                   (* Set the path *)
                   filepath:=get_basedir(db)+RELATION_DIR;
                   filename:='\'+relation_name+RELATION_EXT;

                   (* Reset the number of fields *)
                   nofields:=0;

                   (* Reset the current position *)
                   current_pos:=0;

                   rcache:=cache_create;

                   (* Create the relation data file *)
                   assign(fileptr,filepath+filename);
                   {$I-}
                   rewrite(fileptr);
                   {$I+}

                   (* If an error occured, report a fatal error *)
                   if IOResult<>0 then
                      fatal_error(ERROR_FILE_OPENING,'File: '+filepath+filename);

                   fieldname:='\'+relation_name+FIELD_EXT;

                   (* Create the attribute descriptor file *)
                   assign(fieldptr,filepath+fieldname);
                   {$I-}
                   rewrite(fieldptr);
                   {$I+}

                   (* If an error occured, report a fatal error *)
                   if IOResult<>0 then
                      fatal_error(ERROR_FILE_OPENING,'File: '+fieldname);


                   (* If the relation is to be temporary, create the
                      appropriate structure(s) *)
                   temporary:=temp;
                   if temporary then
                      create_tempfile(db,relation_name);

                   (* Reset the structural information *)
                   storage.next:=nil;
                   storage.previous:=nil;
              end;

              (* Insert the relation into the master structure *)
              relation_insert(db,rel);

              (* We need to create the hash table *)
              hashing_create(rel^.hash_table);
         end;
     end;

     (* Return the new (or nil) relation *)
     relation_create:=rel;
end;



procedure relation_print(    rel:relation);
(* Print out a relations information *)

(* Ref. Section x.3.4 *)

var
   x:byte;
begin
     (* If the relation is valid *)
     if assigned(rel) then
     begin
          (* Display the relations name *)
          write_stdout(relation_name(rel));

          (* Print tabulating spaces *)
          for x:=length(relation_name(rel)) to 15 do
              write_stdout(' ');

          (* If the relation is temporary print a tick*)
          if relation_temporary(rel) then
             writeln_stdout(#251)
             (* Otherwise print a cross *)
          else
              writeln_stdout('X');
     end;
end;




function relation_findfirst(db:database):relation;
(* Find the first relation in the database and return ptr *)

(* Ref. Section x.2.5 *)
begin
     relation_findfirst:=db^.firstRelation;
end;



function relation_findnext(rel:relation):relation;
(* Find the next relation in the database from given rel. and return ptr *)

(* Ref. Section x.2.6 *)

begin
     relation_findnext:=rel^.storage.next;
end;




procedure relation_display(db:database);
(* Display all of the relations in the database *)

(* Ref. Section x.2.7 *)

var
   currentRel:relation;
begin
     writeln_stdout('Name        Temporary?');

     (* Get the first relation *)
     currentRel:=relation_findfirst(db);
     while (currentRel<>nil) do
     begin

          (* Print the current relation *)
          relation_print(currentRel);

          (* Find the next relation *)
          currentRel:=relation_findnext(currentRel);
     end;
end;



function relation_name(rel:relation):string;
(* Returns the name of a relation *)

(* Ref. Section x.2.8 *)

begin
     relation_name:=rel^.name;
end;




function relation_temporary(rel:relation):boolean;
(* Returns the temporary status of a relation *)

(* Ref. Section x.2.9 *)

begin
     relation_temporary:=(rel^.temporary);
end;



function relation_find(    db:database;
                  name:string):relation;
(* Locate a relation, and return ptr to it given its name *)

(* Ref. Section x.2.10 *)

var
   currentRel:relation;
begin
     name:=upstring(name);
     strip_tail_spaces(name);
     strip_leading_spaces(name);

     (* Locate first relation *)
     currentRel:=relation_findfirst(db);

     (* Get the next relation until found or end of list *)
     while assigned(currentrel) and (relation_name(currentRel)<>name) do
           currentRel:=relation_findnext(currentRel);

     (* Nothing found... *)
     if (not assigned(currentRel)) {or (relation_name(currentRel)>name) }then
     begin
          relation_find:=nil;
          searchError:=SEARCH_UNSUCCESSFUL;
     end
        (* something found...*)
        else
            relation_find:=currentRel;
end;


procedure relation_close(    rel:relation);
(* Close a specified relation *)

(* Ref. Section x.2.11 *)

begin
     (* If the relation exists *)
     if assigned(rel) then
     begin
          (* Close the file descriptors *)
          close(rel^.fieldptr);
          close(rel^.fileptr);
     end;
end;



procedure relation_open_fileptr(    rel:relation);
(* Open the file descriptors within the relation *)

(* Ref. Section x.2.12 *)

begin
     assign(rel^.fileptr,rel^.filepath+rel^.filename);
     assign(rel^.fieldptr,rel^.filepath+rel^.fieldname);
end;



procedure relation_reset_relfile(    rel:relation);
(* Reset for reading, the file descriptor *)

(* Ref. Section x.2.13 *)

var
   ior:integer;
   s:string;
begin
    {$I-}
    reset(rel^.fileptr);
    {$I+}

    (* If an error occured, then report a fatal error *)
    ior:=IOResult;
    system.str(ior,s);
    if ior<>0 then
       fatal_error(ERROR_FILE_OPENING,'IOE: '+s+' File: '+rel^.fieldname);
end;




procedure relation_close_relfile(    rel:relation);
(* Close the relation file descriptor *)

(* Ref. Section x.2.14 *)

begin
     close(rel^.fileptr);
end;




procedure relation_Reset_fldfile(    rel:relation);
(* Reset for reading, the attribute descriptor *)

(* Ref. Section x.2.15 *)

var
   ior:integer;
   s:String;
begin
    {$I-}
    reset(rel^.fieldptr);
    {$I+}

    (* If an error occured, report a fatal error *)
    ior:=IOResult;
    system.str(ior,s);
    if ior<>0 then
       nonfatal_error(ERROR_FILE_OPENING,'IOE: '+s+' File: '+rel^.fieldname);
end;

procedure relation_close_fldfile(    rel:relation);
(* Close the attribute file descriptor *)

(* Ref. Section x.2.16 *)

begin
     close(rel^.fieldptr);
end;


function relation_read(    rfilepath,rfilename:string):relation;
(* Read a specified relation file into memory, return a ptr to the
   node *)

(* Ref. Section x.3.5 *)

var
   rel:relation;
   relname:string;
   dirinfo:searchrec;
   fld:field;
begin
     (* Create a relation pointer *)
     new(rel);
     check_assign(rel,'relation.relation_Read');

     relname:=get_token(rfilename,'.');
     with rel^ do
     begin
          (* Populate the fields *)
          name:=relname;
          filepath:=rfilepath;
          filename:='\'+rfilename;

          fieldname:='\'+relname+FIELD_EXT;

          (* Get the temporary file *)
          findfirst(rfilepath+'\'+relname+RELATION_TEMPORARY_IND,anyfile,dirinfo);

          (* If no file, not temporary, & vice versa, hence ... *)
          temporary:=(DosError=0);

          nofields:=0; {This needs to be calculated as well}

          (* Open the file *)
          relation_open_fileptr(rel);

          (* Load the first attribute *)
          fld:=field_findfirst(rel);

          (* Reset the current position *)
          current_pos:=0;

          rcache:=cache_create;


          (* Whilst the field is valid *)
          while assigned(fld) do
          begin
               inc(nofields);

               (* Load the next attribute *)
               fld:=field_findnext(rel,fld,FALSE,TRUE);
          end;

          storage.next:=nil;
          storage.previous:=nil;
     end;

     (* Open the files *)
{     relation_open_fileptr(rel);}

     (* If the hash file exists *)
     if exist(rfilepath+'\'+relname+HASH_EXT) then
     begin
        if TRACE then
           write_stdout('Loading hash table for relation '+relname+'...');

        (* Load the file from disk *)
        Hashing_Load(rel^.hash_table,rfilepath+'\'+relname+HASH_EXT);

        if TRACE then
           writeln_stdout(' Done.');
     end
     else
     begin
          write_stdout('No hash table exists for relation '+relname+' - Creating...');
         (* Generate the file *)
         rel^.hash_table:=Build_Hash_Table(rel);

         (* Save the newly generated hash table *)
         Hashing_Save(rel^.hash_table,rfilepath+'\'+relname+HASH_EXT);
         writeln_stdout(' Done.');
     end;

     (* Return the relation *)
     relation_read:=rel;
end;



procedure open_relations(    db:database);
(* Open up all of the relations and insert the nodes into the
   relation structure *)

(* Ref. Section x.2.17 *)

var
   relinfo,fldinfo,tinfo:searchrec;
   ioer,relerror,flderror,terror,ior:integer;
   rel:relation;
   relname,fname:string;
   ismasterdb:boolean;
   ch:char;
   x:byte;
   f,tf:file;
begin
     ismasterdb:=(get_dbname(db)=MASTER_DB_NAME);
     (* Inform user what is happening... *)
     writeln_stdout('--- Retrieving Relations ---');

     (* Locate the first relation file *)
     findfirst(get_basedir(db)+RELATION_DIR+RELATION_MASK,anyfile,relinfo);
     relerror:=doserror;
     findfirst(get_basedir(db)+RELATION_DIR+FIELD_MASK,anyfile,fldinfo);
     flderror:=doserror;

     (* While matching files exist *)
     while (relerror=0) do
     begin
          if flderror=0 then
          begin
               fname:=relinfo.name;
               relname:=get_token(fname,'.');


               if ismasterdb then
               begin
                    x:=1;
                    while (LEAP_SYSTABLES[x]<>relname) and (x<=NOSYSTABLES) do
                          inc(x);


                    if x>NOSYSTABLES then
                    begin
                         writeln_stdout(relname+' is not needed');
                         findfirst(get_basedir(db)+RELATION_DIR+'\'+relname+'.*',anyfile,tinfo);
                         terror:=doserror;
                         while (terror=0) do
                         begin
                              assign(tf,get_basedir(db)+RELATION_DIR+'\'+tinfo.name);

                              {I-}
                              erase(tf);
                              {$I+}
                              ior:=IOResult;

                              if ior<>0 then
                                 nonfatal_error(ERROR_ERASE_FILE,get_basedir(db)+RELATION_DIR+'\'+tinfo.name);

                              findnext(tinfo);
                              terror:=doserror;
                         end
                    end
                    else
                    begin
                        writeln_stdout(relname+' is a LEAP system table!');
                        rel:=relation_read(get_basedir(db)+RELATION_DIR,fname);
                        relation_insert(db,rel);
                    end;
               end
               else
               begin
                   (* Read the relation from the disk *)
                   rel:=relation_read(get_basedir(db)+RELATION_DIR,fname);

                   (* Insert the relation into the structure *)
                   relation_insert(db,rel);
               end;
          end
          else
              begin
                   nonfatal_error(ERROR_EXPECTED_FILE_NOT_FOUND,get_token(relinfo.name,'.')+FIELD_EXT);
                   write_stdout('Remove existing files associated with error? (Y/N) : ');
                   ch:=upcase(readkey);
                   writeln_stdout(ch);
                   if ch='Y' then
                   begin
                        writeln_stdout('Erasing '+get_basedir(db)+RELATION_DIR+'\'+get_token(relinfo.name,'.')+RELATION_EXT);
                        assign(f,get_basedir(db)+RELATION_DIR+'\'+get_token(relinfo.name,'.')+RELATION_EXT);
                        {$I-}
                        erase(f);
                        {$I+}
                        if IOResult<>0 then
                           nonfatal_error(ERROR_ERASE_FILE,get_token(relinfo.name,'.')+RELATION_EXT);
                   end;
              end;

          (* Get the next matching file *)
          findnext(relinfo);
          relerror:=doserror;
          findnext(fldinfo);
          flderror:=doserror;

     end;

     writeln_stdout_nl;
end;


procedure relation_info(    db:database);
var
   tf:text;
   ior:integer;
   s:string;
   x:byte;
begin
     writeln_stdout_nl;

     for x:=1 to 79 do write_stdout('-');

     assign(tf,get_basedir(db)+DB_DESCRIPTOR_FILE);

     {$I-}
     reset(tf);
     {$I+}

     ior:=IOResult;
     if (ior=0) then
     begin
          while not eof(tf) do
          begin
               readln(tf,s);
               writeln_stdout(s);
          end;
          close(tf);
     end
     else
         nonfatal_error(ERROR_FILE_NOT_FOUND,'Database description: '+get_basedir(db)+DB_DESCRIPTOR_FILE);

     for x:=1 to 79 do write_stdout('-');
end;

procedure relation_change(    db:database;
                              name:string);
(* Change the permanent status of a specified relation *)

(* Ref. Section x.2.18 *)

var
   dirinfo:searchrec;
   f:file;
   rel:relation;
begin
     (* Determine the name of the relation *)
     rel:=relation_find(db,name);

     (* If the relation exists *)
     if assigned(rel) then
     begin
          (* Find the temporary indicator file *)
          findfirst(get_basedir(db)+RELATION_DIR+'\'+name+RELATION_TEMPORARY_IND,anyfile,dirinfo);

          (* If no error, ie. the file exists *)
          if doserror=0 then
          begin
               (* Assign a file descriptor to it and erase the file *)
               assign(f,get_basedir(db)+RELATION_DIR+'\'+dirinfo.name);
               erase(f);
          end
          else
              (* An error implies that the file does not exist, so create
                 it *)
              create_tempfile(db,name);

          (* Update the temporary status of the relation *)
          rel^.temporary:=not rel^.temporary;
     end;
end;



function relation_user_create(    db:database;
                                  name,fields:string):relation;
(* Create a relation according to users specification *)

(* Ref. Section x.2.19 *)

var
   rel:relation;
   fld:field;
   s,tp:string;
   dt:data_types;
   count:byte;
begin
     (* Create the relation, with specified name *)
     rel:=relation_create(db,name,false);

     (* If a relation was created *)
     if assigned(rel) then
     begin
          count:=0;

          (* Get the first attribute *)
          s:=cut_token(fields,' ');

          (* Whilst the attribute is valid, and we're within bounds *)
          while (s<>'') and (count<MAX_NO_ATTRIBUTES) do
          begin
               tp:=upstring(cut_token(fields,' '));

               if (tp='INTEGER') or (tp='NUMBER') or (tp='REAL') then
                  dt:=int
               else
               if (tp='STRING') or (tp='STR') or (tp='CHAR') then
                   dt:=string_t
               else
                   dt:=undef;

               (* Create a field *)
               if dt=undef then
               begin
                    nonfatal_error(ERROR_NO_TYPE,'Defaulting to STRING for attribute '+s);
                    field_create(rel,s,string_t);
                    (* Add the 'data type' back on, it's probably a new attribute *)
                    fields:=concat(tp,' ',fields);
               end
               else
                   field_create(rel,s,dt);

               (* Increment our counter *)
               inc(count);

               (* Get the next attribute *)
               s:=cut_token(fields,' ');
          end;

          (* If the count is at the maximum attribute *)
          if (count=MAX_NO_ATTRIBUTES) and (s<>'') then
          begin
               system.str(MAX_NO_ATTRIBUTES,s);

               (* Report an error *)
               nonfatal_error(ERROR_EXCEEDED_FIELD_LIMIT,'Limit is set to: '+s);
          end;

          (* Close the relation *)
          relation_close(rel);

          (* Return the relation *)
          relation_user_create:=rel;
     end
     else
         (* No relation could be created, so return nil *)
         relation_user_create:=nil;
end;


function relation_rename(    db:database;
                             rel1,rel2:string;
                             destname:string):relation;

var
   r1,r2:relation;
   tf,outf:text;
   sr:searchrec;
   done:boolean;
   x:byte;
   ior:integer;
   s,sa,name,dtype:string;
   p:pathstr; d:dirstr; n:namestr; e:extstr;
begin
     relation_rename:=nil;

     if pos('.',rel1)=0 then
     begin

          r1:=relation_find(db,rel1);

          if assigned(r1) then
          begin
               (* Find the temporary indicator file *)
               findfirst(get_basedir(db)+RELATION_DIR+'\'+rel1+'.*',anyfile,sr);


               (* If no error, ie. the file exists *)
               while (doserror=0) do
               begin
                    (* Assign a file descriptor to it and erase the file *)
                    assign(tf,get_basedir(db)+RELATION_DIR+'\'+sr.name);

                    fsplit(get_basedir(db)+RELATION_DIR+'\'+sr.name,d,n,e);

                    rename(tf,get_basedir(db)+RELATION_DIR+'\'+rel2+e);

                    findnext(sr);
               end;
               r1^.name:=upstring(rel2);
               r1^.filename:='\'+r1^.name+RELATION_EXT;
               r1^.fieldname:='\'+r1^.name+FIELD_EXT;
               relation_open_fileptr(r1);

               relation_rename:=r1;
          end
          else
              nonfatal_error(ERROR_CANNOT_FIND_REL,rel1);
     end
     else
         begin
              x:=pos('.',rel1);
              s:=copy(rel1,1,x-1);
              sa:=copy(rel1,x+1,length(rel1));
              sa:=upstring(sa);

              rel2:=upstring(rel2);

              r1:=relation_find(db,s);

              if assigned(r1) then
              begin
                   assign(tf,get_basedir(db)+RELATION_DIR+r1^.fieldname);

                   {$I-}
                   reset(tf);
                   {$I+}

                   ior:=IOResult;
                   if (ior<>0) then
                      nonfatal_error(ERROR_FILE_OPENING,get_basedir(db)+RELATION_DIR+r1^.fieldname)
                   else
                       begin
                            assign(outf,get_basedir(db)+RELATION_DIR+'\'+r1^.name+FIELD_TMP);
                            {$I-}
                            rewrite(outf);
                            {$I+}
                            ior:=IOResult;
                            if ior<>0 then
                               nonfatal_error(ERROR_FILE_OPENING,get_basedir(db)+'\'+RELATION_DIR+r1^.name+FIELD_TMP)
                            else
                            begin
                                done:=false;
                                while not eof(tf) do
                                begin
                                     readln(tf,name);
                                     readln(tf,dtype);

                                     if upstring(name)=sa then
                                     begin
                                          writeln(outf,rel2);
                                          done:=true;
                                     end
                                     else
                                         writeln(outf,name);

                                     writeln(outf,dtype);
                                end;

                                close(outf);
                                close(tf);

                                if not done then
                                begin
                                     nonfatal_error(ERROR_CANTFIND_FLD,sa);
                                     erase(outf);
                                end
                                else
                                    begin

                                         erase(tf);
                                         rename(outf,get_basedir(db)+RELATION_DIR+'\'+r1^.name+FIELD_EXT);
                                         assign(r1^.fieldptr,get_basedir(db)+RELATION_DIR+'\'+r1^.name+FIELD_EXT);
                                         relation_rename:=r1;
                                    end
                            end
                       end
              end
              else
                  nonfatal_error(ERROR_CANNOT_FIND_REL,rel1);

         end;
end;


procedure relation_clean(var db:database);
var
   rel:relation;
   x:byte;
   s:string;
begin
     if get_dbname(db)=MASTER_DB_NAME then
     begin
         rel:=relation_findfirst(db);

         while assigned(rel) do
         begin
              x:=1;
              s:=relation_name(rel);

              while (LEAP_SYSTABLES[x]<>s) and (x<=NOSYSTABLES) do
                    inc(x);


              if x>NOSYSTABLES then
              begin
                   if not relation_temporary(rel) then
                   begin
                        relation_change(db,s);
                        writeln_stdout('Flagging '+s+' as temporary for deletion.');
                   end
                   else
                       writeln_stdout('Leaving '+s+' as temporary.');
              end
              else
                  writeln_stdout(s+' is a LEAP system table!');

              rel:=relation_findnext(rel);
         end;

         relations_dispose_all(db);
     end
     else
         nonfatal_error(ERROR_MASTER_ONLY,'relation_clean on '+get_dbname(db));
end;

end.