unit p_stack;

(*
Unit: p_stack
Description: Stack operations for Parse Tree traversal
References:
Notes: 
Revision Notes:
         11/02/1995 - Initial development.
	 22/05/1996 - Placed under GNU General Public License.
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)
interface

uses dtypes;

(* Define the parse_tree stack types *)
type
    pt_stack=^pt_stack_def;
    pt_stack_node=^pt_stack_node_def;

    (* Header node, with count information *)
    pt_stack_def=record
                    head_node:pt_stack_node;
                    no_items:word;
    end;

    (* Stack node, with 'next' information *)
    pt_stack_node_def=record
                         data:parse_tree;
                         next:pt_stack_node;
    end;

function pt_create_stack:pt_stack;
procedure pt_push_stack(    st:pt_stack;
                               data:parse_tree);
procedure pt_flush_stack(var st:pt_stack);
function pt_pop_stack(var st:pt_stack):parse_tree;
function pt_stack_empty(    st:pt_stack):boolean;

implementation

uses utils;

function pt_create_stack:pt_stack;
(* Create a stack *)

(* Ref. Section x.1.1 *)

var
   stk:pt_stack;
begin
     (* Create the node *)
     new(stk);
     check_Assign(stk,'rt_stack.pt_create_stack');

     (* Reset the data *)
     stk^.head_node:=nil;
     stk^.no_items:=0;
     pt_create_stack:=stk;
end;




procedure pt_push_stack(    st:pt_stack;
                            data:parse_tree);
(* Push a datum onto the stack *)

(* Ref. Section x.1.2 *)

var
   st_node:pt_stack_node;
begin
     (* Create a node *)
     new(st_node);
     check_assign(st_node,'rt_stack.pt_push_stack');

     (* Populate the fields within the node according to the parameters *)
     do_trace('Populating stack...');
     st_node^.data:=data;
     do_trace('Beep');
     st_node^.next:=st^.head_node;
     do_trace('Beep');
     st^.head_node:=st_node;
     do_trace('Beep');
     inc(st^.no_items);
     do_trace('Done.');
end;




function pt_pop_stack(var st:pt_stack):parse_tree;
(* Pop a datum off of the stack *)

(* Ref. Section x.1.3 *)

var
   node:pt_stack_node;
begin
     (* Get the top node *)
     pt_pop_stack:=st^.head_node^.data;

     node:=st^.head_node;
     st^.head_node:=node^.next;

     (* If its an ok node *)
     if assigned(node) then
     begin
          (* Dispose of its memory *)
          dispose(node);
          node:=nil;
     end
     else
         (* Report an error *)
         nonfatal_error(ERROR_INTERNAL_STACK,'(pt_pop_stack)');

     (* Update book-keeping information *)
     dec(st^.no_items);
end;





function pt_stack_size(    st:pt_stack):word;
(* Return the number of nodes in a stack *)

(* Ref. Section x.2.1 *)

begin
     pt_stack_size:=st^.no_items;
end;



function pt_stack_empty(    st:pt_stack):boolean;
(* Return TRUE if the stack is empty *)

(* Ref. Section x.2.2 *)

begin
     pt_stack_empty:=(pt_stack_size(st)=0);
end;



procedure pt_stack_dispose(var st:pt_stack);
(* Dispose of the memory used by the stack *)

(* Ref. Section x.2.3 *)

begin
     (* If the node is valid *)
     if assigned(st) then
        (* Dispose of the memory *)
        dispose(st)
     else
         (* Report an error *)
         nonfatal_error(ERROR_INTERNAL_STACK,'(pt_stack_dispose)');
     st:=nil;
end;




procedure pt_flush_stack(var st:pt_stack);
(* Flush all of the memory used by the entire stack *)

(* Ref. Section x.2.4 *)

begin
     (* Whilst there is a node to pop *)
     while not(pt_stack_empty(st)) do
           (* Pop the node off of the stack *)
           pt_pop_stack(st);       {TP 7 allows functions to be called
                                        as procedures}

     (* Dispose of the header structure *)
     pt_stack_dispose(st);
end;





end.

