unit parser;

(* Unit: Pareser
   Created: 11/02/1996
   Notes: 
   References:
   Description: This unit will house the parser operations that will
                process the queries. The only return value should be
                a relation that results from a query. Internal processing
                should be hidden.
   Revision History:
            11/02/1996 - First started. Moved old parser operations
            11/02/1996 - Started a parser that builds a parse tree, then
                         processes.
	    22/06/1996 - Placed under the GNU General Public License.
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)

interface

uses dtypes,rt_stack;


function infix_get_relation(    db:database;
                            var s:string;
                                destname:string;
                            var rtstack:rtion_stack;
                                depth:integer):relation;
function new_get_relation(    db:database;
                          var s:string;
                              destname:string;
                          var rtstack:rtion_stack;
                              depth:integer):relation;


function process_query(    db:database;
                           query,dest:string):relation;


implementation

uses utils,rtional,relations,tuples,p_stack;

function is_operator(    s:string):shortint;
(* Find the specified string in the command cross-reference, and return
   the index number *)

(* Ref. Section x.1.10 *)

var
   c:shortint;
begin
     c:=1;

     (* Whilst we haven't reached the end of the array, and it doesn't
        match *)
     while (c<=NOCMDS) and (commands[c].text<>s) do
           inc(c);

     (* If we exceeded the array bounds *)
     if c>NOCMDS then
        c:=0;

     (* Return the result *)
     is_operator:=c;
end;





function new_get_relation(    db:database;
                          var s:string;
                              destname:string;
                          var rtstack:rtion_stack;
                              depth:integer):relation;
(* Executes a command by recursively processing an expression until
   a relation is found, and recursing out resolving higher level expressions *)

(* Ref. Section x.1.10 *)

var
   t3,t2,tr:relation;
   c:byte;
   params,done:string;
   tok:string;
   operator:string;
   remainder:string;
   command:string;
   sub:string;
   op:string;
   rls:string;
   temp:string;
begin
     (* Reset values *)
     tr:=nil;
     params:=s;

     system.str(depth,rls);
     {$IFDEF DEBUG}
     writeln_stdout('RECURSE LEVEL: '+rls);
     {$ENDIF}

     (* Check the depth of the recursion, to prevent an error if possible *)
     if depth>=MAX_RECURSION_DEPTH then
     begin
          nonfatal_error(ERROR_RECURSION_DEPTH,'Check brackets are correct. (See help page #2)');
          new_get_relation:=nil;
          set_cli_Error(ERROR_CLI_RECURSION);
     end
     else
     begin
          (* Get the first command *)
          command:=get_token(s,' ');

          (* If debugging is on *)
          if debug then
             report('DEBUG: Processing: '+s);
     {$IFDEF DEBUG}
             writeln_stdout('DEBUG: Processing: '+s);
     {$ENDIF}

          (* Cut away any leading spaces *)
          strip_leading_spaces(s);
          c:=1;
          done:='UNKNOWN';

          (* Locate the command *)
          while (c<=NOCMDS) and (commands[c].text<>command) do inc(c);

          (* If we found a command *)
          if c<=NOCMDS then
          begin
               (* Get the remainder of the command *)
               remainder:=cut_to_right_bracket(s,1);
               temp:=remainder;


               (* Depends which command *)
               case commands[c].command of
                    project:begin
                                 (* Get the relation/command *)
                                 sub:=cut_to_right_bracket(remainder,1);

                                 if sub<>temp then
                                 begin
                                      if timing then
                                         writeln_stdout(get_datetime+' Executing '+command);
                                      (* Resolve the relation/command to get a
                                         ptr to a relation *)
                                      t2:=new_get_relation(db,sub,'',rtstack,depth+1);

                                      (* Perform the operation *)
                                      tr:=rl_project(db,t2,cut_to_right_bracket(remainder,1),destname);

                                      (* Return the relation ptr *)
                                      new_get_relation:=tr;

                                      (* Push the relation onto the temporary stack *)
                                      rtion_push_stack(rtstack,t2);

                                 end
                                    else
                                        begin
                                             nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+sub);
                                             new_get_relation:=nil;
                                        end;
                                 (* Record what was done *)
                                 done:='PROJECT';
                            end;
                    select:begin
                                (* Get the relation/command *)
                                SUB:=cut_to_right_bracket(remainder,1);

                                if sub<>temp then
                                begin
                                      if timing then
                                         writeln_stdout(get_datetime+' Executing '+command);

                                     (* Resolve the relation/command to get a ptr to
                                        a relation *)
                                     t2:=new_get_relation(db,sub,'',rtstack,depth+1);


                                     (* Perform the operation *)
                                     tr:=rl_select(db,t2,cut_to_right_bracket(remainder,1),destname);

                                     (* Return the relation ptr *)
                                     new_get_relation:=tr;

                                     (* Push the relation onto the temporary stack *)
                                     rtion_push_stack(rtstack,t2);

                                end
                                   else
                                       begin
                                            nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+sub);
                                            new_get_relation:=nil;
                                       end;
                                (* Record what was done *)
                                done:='SELECT';
                          end;
                    join:begin
                              (* Get the first relation/command *)
                              sub:=cut_to_right_bracket(remainder,1);

                              if sub<>temp then
                              begin
                                      if timing then
                                         writeln_stdout(get_datetime+' Executing '+command);
                                   (* Resolve the first relation/command to get a ptr
                                      to a relation *)
                                   t2:=new_get_relation(db,sub,'',rtstack,depth+1);

                                   (* Get the second relation/command *)
                                   sub:=cut_to_right_bracket(remainder,1);

                                   (* Resolve the second relation/command to get a ptr
                                      to a relation *)
                                   t3:=new_get_relation(db,sub,'',rtstack,depth+1);



                                   (* Perform the operation *)
                                   tr:=rl_join(db,t2,t3,cut_to_right_bracket(remainder,1),destname);

                                   (* Return the relation ptr *)
                                   new_get_relation:=tr;

                                   (* Push the relations onto the temporary stack *)
                                   rtion_push_stack(rtstack,t2);
                                   rtion_push_stack(rtstack,t3);

                              end
                                 else
                                     begin
                                          nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+sub);
                                          new_get_relation:=nil;
                                     end;
                              (* Record what was done *)
                              done:='JOIN';
                         end;
                    sm_join:begin
                                 (* Get the first relation/command *)
                                 sub:=cut_to_right_bracket(remainder,1);

                                 if sub<>temp then
                                 begin
                                      if timing then
                                         writeln_stdout(get_datetime+' Executing '+command);
                                      (* Resolve the relation/command to get a
                                         ptr to a relation *)
                                      t2:=new_get_relation(db,sub,'',rtstack,depth+1);

                                      (* Get the second relation/command *)
                                      sub:=cut_to_right_bracket(remainder,1);

                                      (* Resolve the second relation/command to get a
                                         ptr to a relation *)
                                      t3:=new_get_relation(db,sub,'',rtstack,depth+1);

                                      (* Perform the operation *)
                                      tr:=rl_one_pass_join(db,t2,t3,cut_to_right_bracket(remainder,1),destname);

                                      (* Return the relation ptr *)
                                      new_get_relation:=tr;

                                      (* Push the relations onto the temporary stack *)
                                      rtion_push_stack(rtstack,t2);
                                      rtion_push_stack(rtstack,t3);

                                 end
                                    else
                                        begin
                                             nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+sub);
                                             new_get_relation:=nil;
                                        end;
                                 (* Record what was done *)
                                 done:='JOIN';
                    end;
                    union:begin
                               (* Get the first relation/command *)
                               sub:=cut_to_right_bracket(remainder,1);

                               if sub<>temp then
                               begin
                                      if timing then
                                         writeln_stdout(get_datetime+' Executing '+command);
                                    (* Resolve the relation/command to get a ptr to
                                       a relation *)
                                    t2:=new_get_relation(db,sub,'',rtstack,depth+1);

                                    (* Get the second relation/command *)
                                    sub:=cut_to_right_bracket(remainder,1);

                                    (* Resolve the second relation/command to get a
                                       ptr to a relation *)
                                    t3:=new_get_relation(db,sub,'',rtstack,depth+1);

                                    (* Perform the operation *)
                                    tr:=rl_union(db,t2,t3,destname);

                                    (* Return the relation ptr *)
                                    new_get_relation:=tr;

                                    (* Push the relations onto the temporary stack *)
                                    rtion_push_stack(rtstack,t2);
                                    rtion_push_stack(rtstack,t3);

                               end
                                  else
                                      begin
                                           nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+sub);
                                           new_get_relation:=nil;
                                      end;
                               (* Record what was done *)
                               done:='UNION';
                          end;
                    intersect:begin
                                   (* Get the first relation/command *)
                                   sub:=cut_to_right_bracket(remainder,1);

                                   if sub<>temp then
                                   begin
                                      if timing then
                                         writeln_stdout(get_datetime+' Executing '+command);
                                        (* Resolve the relation/command to get a ptr
                                           to a relation *)
                                        t2:=new_get_relation(db,sub,'',rtstack,depth+1);

                                        (* Get the second relation/command *)
                                        sub:=cut_to_right_bracket(remainder,1);

                                        (* Resolve the second relation/command to get
                                           a ptr to a relation *)
                                        t3:=new_get_relation(db,sub,'',rtstack,depth+1);

                                        (* Perform the operation *)
                                        tr:=rl_hash_intersect(db,t2,t3,destname);

                                        (* Return the relation ptr *)
                                        new_get_relation:=tr;

                                        (* Push the relations onto the temporary stack *)
                                        rtion_push_stack(rtstack,t2);
                                        rtion_push_stack(rtstack,t3);

                                   end
                                      else
                                          begin
                                               nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+sub);
                                               new_get_relation:=nil;
                                          end;
                                   (* Record what was done *)
                                   done:='INTERSECT';
                              end;
                    difference:begin
                                    (* Get the first relation/command *)
                                    sub:=cut_to_right_bracket(remainder,1);

                                    if sub<>temp then
                                    begin
                                      if timing then
                                         writeln_stdout(get_datetime+' Executing '+command);
                                         (* Resolve the relation/command to get a ptr
                                            to a relation *)
                                         t2:=new_get_relation(db,sub,'',rtstack,depth+1);

                                         (* Get the second relation/command *)
                                         sub:=cut_to_right_bracket(remainder,1);

                                         (* Resolve the second relation/command to get
                                            a ptr to a relation *)
                                         t3:=new_get_relation(db,sub,'',rtstack,depth+1);

                                         (* Perform the operation *)
                                         tr:=rl_hash_difference(db,t2,t3,destname);

                                         (* Return the relation ptr *)
                                         new_get_relation:=tr;

                                         (* Push the relations onto the temporary stack *)
                                         rtion_push_stack(rtstack,t2);
                                         rtion_push_stack(rtstack,t3);

                                    end
                                       else
                                           begin
                                                nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+sub);
                                                new_get_relation:=nil;
                                           end;
                                    (* Record what was done *)
                                    done:='DIFFERENCE';
                               end;
                    display:begin
                                 (* Get the first relation/command *)
                                 sub:=cut_to_right_bracket(remainder,1);

                                 if sub<>temp then
                                 begin
                                           if timing then
                                              writeln_stdout(get_datetime+' Executing '+command);
                                      (* Resolve the relation/command to get a ptr
                                         to a relation *)
                                      t2:=new_get_relation(db,sub,'',rtstack,depth+1);

                                      if assigned(t2) then
                                           rl_display(t2);

                                      tr:=t2;

                                      new_get_relation:=tr;

                                 end
                                    else
                                        begin
                                             nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+sub);
                                             new_get_relation:=nil;
                                        end;
                                 (* Record what was done *)
                                 done:='DISPLAY';
                            end;
                    product:begin
                                 (* Get the first relation/command *)
                                 sub:=cut_to_right_bracket(remainder,1);

                                 if sub<>temp then
                                 begin
                                      if timing then
                                         writeln_stdout(get_datetime+' Executing '+command);
                                      (* Resolve the relation/command to get a ptr to
                                         a new relation *)
                                      t2:=new_get_relation(db,sub,'',rtstack,depth+1);

                                      (* Get the second relation/command *)
                                      sub:=cut_to_right_bracket(remainder,1);

                                      (* Resolve the relation/command to get a ptr to
                                         a new relation *)
                                      t3:=new_get_relation(db,sub,'',rtstack,depth+1);

                                      (* Perform the operation *)
                                      tr:=rl_product(db,t2,t3,destname);

                                      (* Return the relation ptr *)
                                      new_get_relation:=tr;

                                      (* Push the relations onto the temporary stack *)
                                      rtion_push_stack(rtstack,t2);
                                      rtion_push_stack(rtstack,t3);

                                 end
                                    else
                                        begin
                                             nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+sub);
                                             new_get_relation:=nil;
                                        end;
                                 (* Record what was done *)
                                 done:='PRODUCT';
                            end;
               end;
          end
          else
              (* It wasn't a command, it's probably a relation name *)
              begin
                   if timing then
                      writeln_stdout(get_datetime+' Executing '+command);
                   (* Locate the relation *)
                   tr:=relation_find(db,command);

                   (* Return the relation ptr *)
                   new_get_relation:=tr;

                   (* Record what was done *)
                   done:='FIND RELATION';
              end;

          (* If nothing was done, then report an error *)
          if not assigned(tr) and not (get_cli_error(false)=ERROR_CLI_RECURSION) then
             nonfatal_error(ERROR_NO_RELATION_CREATED,'Whilst executing: '+done+' in '+params);

          (* Report progress *)
          if debug then
               report('DEBUG: Processed: '+params);
     {$IFDEF DEBUG}
               writeln_stdout('DEBUG: Processed: '+params);
               writeln_stdout('Returned: '+relation_name(tr));
     {$ENDIF}

     end;

     if timing then
        writeln_stdout(get_datetime+' Executed '+command);

end;


function infix_get_relation(    db:database;
                            var s:string;
                                destname:string;
                            var rtstack:rtion_stack;
                                depth:integer):relation;
(* Executes a command by recursively processing an expression until
   a relation is found, and recursing out resolving higher level expressions *)

(* Ref. Section x.1.10 *)

var
   c:integer;
   tr,tr2:relation;
   did,s1,s2,s3,s4,s5,t,ic,oc:string;
begin
{     if depth=0 then
     begin
          tr2:=process_query(db,s,destname,destname);
          do_trace('Iterative parser returns: '+relation_name(tr2));
     end;}

     tuple_count_reset_all;
     did:='';
     tr:=nil;
     s3:=s;

     s1:=cut_to_right_bracket(s3,1);
     s2:=cut_to_right_bracket(s3,1);
     s4:=get_token(s3,' ');
     s5:=cut_to_right_bracket(s3,1);
     {get_token needed to remove additional spaces}

     {iff s2='' -> relation}
     {iff s3='' -> binary operation, s1=relation_1 s2=relation_2}
     {iff s3 contains WHERE -> join}
     {iff s4='select' s2=condition, s1=relation}

     if debug then
        report('DEBUG: Processing: '+s);

     if s2='' then
     begin
          if timing then
          begin
               did:='Retrieved '+s1;
               writeln_stdout(get_datetime+' Retrieving '+s1);
          end;
          tr:=relation_find(db,s1)
     end
     else
     if s3='' then
     begin
          c:=1;

          s4:=upstring(s4);

          (* Locate the command *)
          while (c<=NOCMDS) and (commands[c].text<>s4) do inc(c);

          if c<=NOCMDS then
          begin
               if timing then
               begin
                    did:='Executed '+s4;
                    writeln_stdout(get_datetime+' Executing '+s4);
               end;
          tuple_count_reset_all;
          case commands[c].command of
               select:tr:=rl_select(db,infix_get_relation(db,s1,'',rtstack,depth+1),
                                       s2,
                                       destname);

               product:tr:=rl_product(db,
                                         infix_get_relation(db,s1,'',rtstack,depth+1),
                                         infix_get_relation(db,s2,'',rtstack,depth+1),
                                         destname);
               intersect:tr:=rl_hash_intersect(db,
                                                  infix_get_relation(db,s1,'',rtstack,depth+1),
                                                  infix_get_relation(db,s2,'',rtstack,depth+1),
                                                  destname);
               difference:tr:=rl_hash_difference(db,
                                                    infix_get_relation(db,s1,'',rtstack,depth+1),
                                                    infix_get_relation(db,s2,'',rtstack,depth+1),
                                                    destname);
               union:tr:=rl_union(db,
                                     infix_get_relation(db,s1,'',rtstack,depth+1),
                                     infix_get_relation(db,s2,'',rtstack,depth+1),
                                     destname);
               project:tr:=rl_project(db,
                                         infix_get_relation(db,s1,'',rtstack,depth+1),
                                         s2,
                                         destname);
               display:tr:=rl_display(infix_get_relation(db,s1,'',rtstack,depth+1));
               lrename:tr:=relation_rename(db,s1,s2,destname);
          end
          end
          else
          if get_token(s2,' ')='DISPLAY' then
          begin
               if timing then
               begin
                    did:='Displayed '+s1;
                    writeln_stdout(get_datetime+' Displaying '+s1);
               end;

               tr:=infix_get_relation(db,s1,'',rtstack,depth+1);
               tuple_count_reset_all;
               rl_display(tr);
          end
          else
          if get_token(s2,' ')='DUPLICATE' then
          begin
               if timing then
               begin
                    did:='Duplicate '+s1;
                    writeln_stdout(get_datetime+' Duplicating '+s1);
               end;

               tr:=rl_duplicate(db,infix_get_relation(db,s1,'',rtstack,depth+1),destname);

          end
          else
              nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+s);
     end
     else
     if get_token(s3,' ')='JOIN' then
     begin
        if timing then
        begin
             did:='Executed JOIN';
             writeln_stdout(get_datetime+' Executing JOIN');
        end;
        tr:=rl_join(db,infix_get_relation(db,s1,'',rtstack,depth+1),
                       infix_get_relation(db,s2,'',rtstack,depth+1),
                       s5,
                       destname);
     end
     else
         nonfatal_error(ERROR_EVALUATING_EXPR,'Whilst evaluating '+s);



     if debug then
        report('DEBUG: Processed: '+s);

     if timing then
     begin
          t:=get_datetime;
          str(incount,ic);
          str(outcount,oc);
          writeln_stdout(t+' '+did);
          writeln_stdout(t+' Read '+ic+' tuples.');
          writeln_stdout(t+' Wrote '+oc+' tuples.');
     end;
     infix_get_relation:=tr;

end; {infix_get_relation}




const
     node_root=0;
     node_left=1;
     node_right=2;

function new_parse_node(    parent:parse_tree;
                            handed:byte):parse_tree;
var
   ptree:parse_tree;
begin
     new(ptree);
     check_assign(ptree,'parser.new_parse_node');

     ptree^.parent:=parent;
     with ptree^ do
     begin
          left:=nil;
          ldone:=false;
          right:=nil;
          rdone:=false;
          expression:='';
          result:=nil;
          lresult:=nil;
          rresult:=nil;
          hand:=handed;
          target:='';
     end;

     new_parse_node:=ptree;
end;

procedure display_ptree(ptree:parse_tree;
                        depth:byte);
var
   s:string;
begin
     writeln_stdout_nl;
     case ptree^.hand of
          node_left:writeln_stdout('LEFT NODE');
          node_right:writeln_stdout('RIGHT NODE');
          node_root:writeln_stdout('ROOT NODE');
     end;
     str(depth,s);
     writeln_stdout('Depth: '+s);
     writeln_stdout('Expression: '+ptree^.expression);

     if assigned(ptree^.result) then
          writeln_stdout('result: '+relation_name(ptree^.result))
     else
         writeln_stdout('result: <NOT KNOWN>');

     if assigned(ptree^.lresult) then
        writeln_stdout('lresult: '+relation_name(ptree^.lresult))
     else
         writeln_stdout('lresult: <NOT KNOWN>');

     if assigned(ptree^.rresult) then
        writeln_stdout('rresult: '+relation_name(ptree^.rresult))
     else
         writeln_stdout('rresult: <NOT KNOWN>');

     if assigned(ptree^.left) then
     begin
          display_ptree(ptree^.left,depth+1);
          str(depth,s);
          writeln_stdout('Returned to Depth: '+s+' Expression: '+ptree^.expression);
     end;
     if assigned(ptree^.right) then
     begin
          display_ptree(ptree^.right,depth+1);
          str(depth,s);
          writeln_stdout('Returned to Depth: '+s+' Expression: '+ptree^.expression);
     end;
end;

function process_parse_tree(    db:database;
                                parsetree:parse_tree):relation;
(*
  This function should take a parse tree, and apply the operations
  as appropriate, in the correct order, IN AN ITERATIVE MANNER.
*)

var
   ptree,result_tree:parse_tree;
   ptstack:pt_stack;
   t,ic,oc:string;
   abort:boolean;
begin
     do_trace('Processing parse tree...');
     abort:=false;
     ptree:=parsetree;
     result_tree:=nil;

     ptstack:=pt_create_stack;

     while not assigned(result_tree) and not abort do
     begin
          (* Locate the left most node *)
          while assigned(ptree^.left) do
          begin
               pt_push_stack(ptstack,ptree);
               ptree:=ptree^.left;
               do_trace('Done assignment');
          end;
{


          if assigned(ptree^.left) then
             ptree:=ptree^.left
          else
          if assigned(ptree^.right) then
             ptree:=ptree^.right
          else
          begin
               (* Make sure that this is not the root node (which
                  contains the result relation!) *)
               if assigned(ptree^.parent) then
               begin
                    (* Make sure we have a copy of the parent *)
                    ntree:=ptree^.parent;

                    (* Pass the result up the tree *)
                    case ptree^.hand of
                         node_left:ntree^.lresult:=ptree^.result;
                         node_right:ntree^.rresult:=ptree^.result;
                         node_root:nonfatal_error(ERROR_UNDEFINED,'Oooer');
                    end;

                    ntree^.left:=nil;
                    ntree^.right:=nil;
                    dispose(ptree);
                    ptree:=ntree;
               end
               else
                   result_tree:=ptree;
          end;}
          tuple_count_reset_all;
          if not assigned(result_tree) then
             case ptree^.operation of
                  LRELATION:begin
                                 ptree^.result:=relation_find(db,ptree^.s1);
                                 if not assigned(ptree^.result) then
                                 begin
                                      abort:=true;
                                      nonfatal_error(ERROR_CANNOT_FIND_REL,ptree^.s1)
                                 end;
                            end;
                  SELECT:ptree^.result:=rl_select(db,ptree^.lresult,ptree^.s2,ptree^.target);
                  PRODUCT:ptree^.result:=rl_product(db,ptree^.lresult,ptree^.rresult,ptree^.target);
                  INTERSECT:ptree^.result:=rl_hash_intersect(db,ptree^.lresult,ptree^.rresult,ptree^.target);
                  DIFFERENCE:ptree^.result:=rl_hash_difference(db,ptree^.lresult,ptree^.rresult,ptree^.target);
                  UNION:ptree^.result:=rl_union(db,ptree^.lresult,ptree^.rresult,ptree^.target);
                  PROJECT:ptree^.result:=rl_project(db,ptree^.lresult,ptree^.s2,ptree^.target);
                  DISPLAY:ptree^.result:=rl_display(ptree^.lresult);
                  JOIN:ptree^.result:=rl_join(db,ptree^.lresult,ptree^.rresult,ptree^.s5,ptree^.target);
                  lrename:ptree^.result:=relation_rename(db,ptree^.s1,ptree^.s2,ptree^.target);
                  duplicate:ptree^.result:=rl_duplicate(db,ptree^.lresult,ptree^.target);
                  else
                      nonfatal_error(ERROR_UNIMPLEMENTED,'Can''t determine operation');
             end;

             if timing then
             begin
                  t:=get_datetime;
                  str(incount,ic);
                  str(outcount,oc);
                  writeln_stdout(t+' '+ptree^.expression);
                  writeln_stdout(t+' Read '+ic+' tuples.');
                  writeln_stdout(t+' Wrote '+oc+' tuples.');
             end;

          (* Pass the results up the tree, so they're preserved *)
          if assigned(ptree^.parent) then
          begin
               (* Make sure we have a copy of the parent *)
               {ntree:=ptree^.parent;} (* Its on the stack *)

               (* Pass the result up the tree *)
               case ptree^.hand of
                    node_left:begin
                                   ptree^.parent^.lresult:=ptree^.result;
                                   ptree^.parent^.left:=nil;
                              end;
                    node_right:begin
                                    ptree^.parent^.rresult:=ptree^.result;
                                    ptree^.parent^.right:=nil;
                               end;
                    node_root:nonfatal_error(ERROR_UNDEFINED,'Oooer');
               end;


               (* Dispose of the node *)
               dispose(ptree);

               {ptree:=ntree;}
               if not pt_stack_empty(ptstack) then
                  ptree:=pt_pop_stack(ptstack)
               else
                   nonfatal_error(ERROR_UNDEFINED,'parser.pop: Node not available when expected');
          end
          else
              result_tree:=ptree;

          (* Get the parent node off of the stack. *)

          if assigned(ptree^.right) then
          begin
               (* Push the node back onto the stack (again!) *)
               pt_push_stack(ptstack,ptree);

               (* Go to the right hand node! *)
               ptree:=ptree^.right;
          end;

     end;

     do_trace('Finished processing...');

     process_parse_tree:=result_tree^.result;

     do_trace('Tidying up...');

     if assigned(result_tree) then
        dispose(result_tree);

     pt_flush_stack(ptstack);

     do_trace('Completed parse tree execution.');
end;


function process_query(    db:database;
                           query,dest:string):relation;
(*
  This function takes a string, and should build a parse tree that
  is then processed accordingly.

  IT SHOULD BE ITERATIVE, RATHER THAN RECURSIVE.

  The reason for this is that when running as a Windows program, their
  is insufficent stack space for the program, and it terminates with
  error 202.

  It will also start the process towards a query optimiser that can
  be incorporated as a step which processes the parse tree before it
  is processed.
*)

var
   reftree,ptree:parse_tree;
   c:byte;
   rtrel:relation;
   dest_done:boolean;
begin
     do_trace('Using Iterative parser');
     do_trace('Building Parse Tree');
     dest_done:=false;

     (* Create the root node *)
     ptree:=new_parse_node(nil,node_root);

     (* Keep a record of the root node, because we'll loose track of it
        when we move around *)
     reftree:=ptree;

     (* Populate the node with the bumf *)
     ptree^.expression:=query;

     while (assigned(ptree)) do
     begin
          (* Get the expression in the current parse tree node *)
         ptree^.s3:=ptree^.expression;


          (* Break it down into the appropriate chunks *)
         ptree^.s1:=cut_to_right_bracket(ptree^.s3,1);
         ptree^.s2:=cut_to_right_bracket(ptree^.s3,1);
         ptree^.s4:=get_token(ptree^.s3,' ');
         ptree^.s5:=cut_to_right_bracket(ptree^.s3,1);

         if not dest_done then
         begin
              ptree^.target:=dest;
              dest_done:=true;
         end;

         (* If its just a relation on its own *)
         if ptree^.s2='' then
         begin
              ptree^.operation:=LRELATION;
              do_trace('Parser: (Leaf node) Relation: '+ptree^.expression);
              (* We don't want to do anything, this will be evaluated
                 as a relation because left and right are nil *)
              {ptree^.left:=new_parse_node(ptree);}
              {s1 is a relation}
              {ptree^.left^.expression:=s1;}
         end
         else
         if get_token(ptree^.s2,' ')='DUPLICATE' then
         begin
              ptree^.left:=new_parse_node(ptree,node_left);
              ptree^.left^.expression:=ptree^.s1;

              ptree^.rdone:=true;

              ptree^.operation:=duplicate;
         end
         else
         if ptree^.s3='' then
         begin
              c:=1;

              ptree^.s4:=upstring(ptree^.s4);

              (* Locate the command *)
              while (c<=NOCMDS) and (commands[c].text<>ptree^.s4) do inc(c);

              if c<=NOCMDS then
              begin
                   ptree^.operation:=commands[c].command;

              do_trace('Parser: (Branch node) Query: '+ptree^.expression);
              case commands[c].command of
                   select,project:begin
                               ptree^.left:=new_parse_node(ptree,node_left);

                               (* Mark the right (empty) tree as "done" *)
                               ptree^.rdone:=true;

                               ptree^.left^.expression:=ptree^.s1;
                          end;
                   product,union,intersect,difference:begin
                                ptree^.left:=new_parse_node(ptree,node_left);
                                ptree^.right:=new_parse_node(ptree,node_right);

                                ptree^.left^.expression:=ptree^.s1;
                                ptree^.right^.expression:=ptree^.s2;
                           end;
                   lrename:begin
                                ptree^.ldone:=true;
                                ptree^.rdone:=true;
                           end;
                   else
                        nonfatal_error(ERROR_UNIMPLEMENTED,commands[c].text);
              end;
              end;
         end
         else
         if get_token(ptree^.s3,' ')='JOIN' then
         begin
              ptree^.left:=new_parse_node(ptree,node_left);
              ptree^.right:=new_parse_node(ptree,node_right);

              ptree^.left^.expression:=ptree^.s1;
              ptree^.right^.expression:=ptree^.s2;

              ptree^.operation:=join;
         end;


         (* Determine where to go next *)
         if (assigned(ptree^.left) and not ptree^.ldone) then
         begin
              ptree^.ldone:=true;
              ptree:=ptree^.left
         end
         else
         if (assigned(ptree^.right) and not ptree^.rdone) then
         begin
              ptree^.rdone:=true;
              ptree:=ptree^.right
         end
         else
         if assigned(ptree^.parent) then
         begin
              ptree:=ptree^.parent;
              while assigned(ptree^.parent) and (ptree^.ldone) and (ptree^.rdone) do
                    ptree:=ptree^.parent;
              if assigned(ptree^.parent) then
                 if assigned(ptree^.right) and not ptree^.rdone then
                 begin
                      ptree^.rdone:=true;
                      ptree:=ptree^.right;
                 end;
         end;
{         else}


         (* This is the root node... *)
         if not assigned(ptree^.parent) then
            if (ptree^.ldone) and (ptree^.rdone) then
               ptree:=nil
            else
            begin
                 if assigned(ptree^.left) and not ptree^.ldone then
                 begin
                      { This shouldn't ever need to be executed... }
                      ptree^.rdone:=true;
                      ptree:=ptree^.left;
                 end
                 else
                 if assigned(ptree^.right) and not ptree^.rdone then
                 begin
                      ptree^.rdone:=true;
                      ptree:=ptree^.right;
                 end
                 else
                 begin
                      nonfatal_error(ERROR_UNDEFINED,'parser.determine.next.node: Something strange has happened!');
                      ptree:=nil;
                 end
            end;

    end;
    do_trace('Parse Tree build completed.');

    if dispparse then
       display_ptree(reftree,0);

    rtrel:=process_parse_tree(db,reftree);

    process_query:=rtrel;
end;

end.