program leap;

(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)
(*
Program: LEAP
Description: Command line interface to relational algebraic operators
             Description of developments to functionality of units
             is included in the relevant modules.
References: References to the documentation is to the file cli.doc
            and section numbers are given as relative to this document.
Revision Notes (Revision Number Assigned):
         18/01/1995 - Initial program constructed and extended. (A0.1)
         19/01/1995 - Further refinement.
         18/01/1995 to 28/01/1995 - Extension of basic functionality  (A0.3)
         29/01/1995 - Further relational operations added
                    (Intersect, Difference and Product).
         01/02/1995 - Select operations added. (A0.4)
         14/02/1995 - Implementation of conditions on join. (A0.5)
         15/02/1995 - Wider scale Incorporation of index utilisation. (A0.6)
         11/03/1995 - Screen mode options.
         12/03/1995 - User creation of tables and tuples added. (A0.7a)
         14/03/1995 - Nested commands implemented! (Phew!) (A0.7b)
         15/03/1995 - All Memory leaks resolved.
         15/03/1995 - Stack for temporary relations implemented.
         15/03/1995 - Stability when brackets incorrect.
         16/03/1995 - Received job offer from AMS... *Phew!*
         17/03/1995 - General Tidying. (A0.7c)
         24/03/1995 - Minor bug fix for the user setting of directory.
         29/03/1995 - Extensive commenting.
         30/03/1995 - Documented (cli.doc)
         02/05/1995 - Renamed to leap.pas (B0.8)
         21/05/1995 - Help page bug fixed.
         26/05/1995 - Expression bug fixed (provisionally). (B0.9)
         26/05/1995 - Removed return to prev. mode as crash set to 80x50 lines
         26/05/1995 - Removed CTRL-C bug.
         07/06/1995 - Removed tailing space bug from assignment ops.
         07/06/1995 - Added 'nag' screens, and secret disabler.
         07/06/1995 - Removed relational_op flag from expression xref.
         07/06/1995 - Hit 10,000 lines (10,001 with this line!)
                    - Beta 0.9 released.
         23/06/1995 - Wildcards not valid as relation names in create.
         09/12/1995 - Database/Relation creation operations added.
         09/12/1995 - Hashing tables stored/retrieved between runs.
         10/12/1995 - Delimited data files supported as default. (Space
                      delimited files are still supported)
         17/01/1996 - Couple of small changes. Copyright updated.
         18/01/1996 - Fixed Memory leaks introduced with multiple databases,
                      cache & hash tables. Added database information,
                      and switches to disable info.
         22/01/1996 - Infix expressions are now the default expression format.
         22/01/1996 - Rename operator implemented.
         06/02/1996 - create database operator implemented.
         07/02/1996 - Duplicate attributes handled consistently.
                      Implemented duplicate operator.
         09/02/1996 - Numeric Data types supported!
         10/02/1996 - DB open script automatically executed if exists.
         10/02/1996 - Script Comments/Text Output modified/enhanced.
         11/02/1996 - Iterative parser implemented!!!
         19/02/1996 - Resolved some Protected mode compiled GPF's
	 22/05/1996 - Placed under GNU General Public License (0.10.1)

To-do:
      Prompter for building of nested queries.
*)

(* Enable hashing tables to be maintained throughout
   LEAP execution *)

{$DEFINE DOS}

{$IFDEF DOS}
uses dtypes,utils,rtional,index,dbase,relations,
     crt,dos,fields,tuples,rt_stack,inform,panic,
     parser,datadictionary;
{$ENDIF}

{$IFDEF WINDOWS}
uses dtypes,utils,rtional,index,dbase,relations,
     wincrt,dos,fields,tuples,rt_stack,inform,panic,
     parser,datadictionary;
{$ENDIF}





var
   input_file:text;
   input_file_file:boolean;
   screen_mode:Integer;
   terminate:(none,shutdown,halt);
   loadcount:word;
   prompt_string:string;




procedure do_prompt;
(* Display a prompt to standard output *)

(* Ref. Section x.1.1 *)

begin
     if colour then
        textcolor(yellow);
     write_stdout(prompt_string);
     if colour then
        textcolor(lightgray);
end;







procedure report_on_off(    bool:boolean);
begin
     if bool then
        writeln_stdout('on.')
     else
         writeln_stdout('off.')
end;


procedure report_timeformat;
var
   t:string;
begin
   write_stdout('Time format is ');

   if UStime then
      t:='US'
   else
       t:='European';

   writeln_stdout(t);

   report('Time format status:'+t);
end;


procedure report_stat(    s:string;
                          b:boolean);
var
   t:string[3];
begin
     write_stdout(s+' is ');

     report_on_off(b);

     if b then
        t:='ON'
     else
         t:='OFF';

     report(s+': '+t);
end;

procedure report_db;
begin
     writeln_stdout('Current database is '+get_dbname(current_db));
end;

procedure report_status;
begin
     writeln_stdout('LEAP Status');
     writeln_stdout('===========');
     writeln_stdout_nl;
     report_stat('Debug Information',debug);
     report_stat('Infix operators',infix);
     report_stat('Operation Timing',timing);
     report_timeformat;
     report_stat('File copy',cpyfile);
     report_stat('Colour',colour);
     report_stat('Additional IO',ioon_setting);
     report_stat('Case sensitivity',casesensitivity);
     report_stat('Internal Cache',cacheenabled);
     report_stat('Display Parse Tree',dispparse);
     report_stat('Iterative Parser',iterative);
     writeln_stdout_nl;
     report_db;
     write_stdout('Data directory is set to ');
     writeln_stdout(get_basedir(current_db)+DATA_DIR);
     writeln_stdout(what_version);
{     writeln_stdout(DIR_MASK);}
end;


procedure assign_inputfile(    filename:string);
(* Assign the source file to a specified filename *)
(* filename of '' is stdin *)

(* Ref. Section x.1.3 *)

var
   er:integer;
   s:string;
   currentd:string;
begin
     stdinput:=false;
     (* If the filename is not '' then add a source file extension to it *)
     if get_token(filename,' ')<>'' then
        filename:=concat(filename,SOURCE_FILE_EXT)
     else
         stdinput:=true;

     (* Get the current directory *)
     getdir(0,currentd);

     {$I-}
     (* Change directory to the script directory *)
     chdir(get_basedir(current_db)+SCRIPT_DIR);
{     chdir(DIR_MASK+DATA_DIR+SCRIPT_DIR);}
     if IOResult<>0 then
        fatal_error(ERROR_CANNOT_CHDIR,get_basedir(current_db)+SCRIPT_DIR+
                                       ' - Directory structure correct?');

     {$I+}

     (* Remove any leading spaces in the filename *)
     strip_leading_spaces(filename);

     {$I-}
     (* Ensure that the input file is closed. If an error
        occurs ignore it. *)
     close(input_file);
     {$I+}
     er:=IOResult;

     (* Open the new input file *)
     assign(input_file,filename);

     {$I-}
     reset(input_file);
     {$I+}


     (* If an error occurs *)
     er:=IOResult;
     if (er<>0) then
     begin
          (* Default the input to stdin, by calling itself with '' as file *)
          system.str(er,s);
          if filename='' then
             filename:='<stdin>';
          if filename<>SCRIPT_OPEN+SOURCE_FILE_EXT then
             nonfatal_error(ERROR_FILE_OPENING,'File: '+filename+' - DE#'+s+'.');
          assign_inputfile('');
     end
     else
         (* If there is a file, then we're inputing from a file, so set
            flags accordingly *)
         if (filename<>'') then
            input_file_file:=true
         else
         begin
              input_file_file:=false;
              stdinput:=true;
         end;

     (* Go back to the old directory *)
     chdir(currentd);
end;





function get_command_string:string;
(* Get a command from the input file, and return it *)

(* Ref. Section x.1.4 *)

var
   s:string;
begin
     repeat
           s:='';

           (* If we're not at the end of the input file *)
           if not stdinput then
              if not eof(input_file) then
              begin

                   (* Read a line of data from it *)
                   readln(input_file,s)
              end
              else
              begin
                   (* Otherwise, reset input to stdin *)
                   nonfatal_error(ERROR_EOF,'Reassigning input to <stdin>.');
                   assign_inputfile('');
              end
           else
           begin
               (* Display the prompt *)
               do_prompt;
               readln(s);
           end;

           strip_leading_spaces(s);

           if (pos(FILE_COMMENT,s)<>1) and (pos(FILE_PRINT,s)<>1) then
           begin
                (* If we're reading from a file *)
                if input_file_file then
                (* Display the command with a couple of extra prompts, so
                   sourcing is apparant *)
                   writeln_stdout('>>'+s)
                else
                    (* Otherwise, just write a copy to the stdoutcpy file, for
                    the record *)
                    writeln_stdoutcpy(s)
           end
           else
           (* If the first character is a print char (>) then
              print the string (after deleting the first character! *)
           if (pos(FILE_PRINT,s)=1) then
           begin
                system.delete(s,1,1);
                if colour then
                   textcolor(lightblue);
                writeln_stdout(s);
                if colour then
                   textcolor(lightgray);
                s:=FILE_COMMENT;
           end;

     until s<>'';
     (* Repeat the above until we have a command of some sort *)

     (* Convert it to upper case *)
     if not casesensitivity then
        s:=upstring(s);

{     strip_leading_spaces(s);}
     if s[1]='#' then
        s:='#';

     (* Return the string *)
     get_command_string:=s;
end;





procedure do_help_page(    page:string);
(* Display the specified help page number, or if 0, all pages *)

(* Ref. Section x.1.5 *)

var
   helpfile:text;
   err:integer;
   tmp2,tmp,s:string;
   ch:char;
   ccnt,cols,cp,c:byte;
   pg:shortint;
   pindex,finish,output,located,name_search:boolean;
begin
     pindex:=false;
     finish:=false;
     (* If no page number specified, do all pages *)
     name_search:=false;

     cols:=1;

     output:=false;
     if page='' then
     begin
          page:='0';
          located:=true
     end
     else
         located:=false;

     (* Convert the page string to a number *)
     system.val(page,pg,err);
     if err<>0 then
     begin
{          (* An error occured with the conversion, report a nonfatal error *)
          cp:=0;
          nonfatal_error(ERROR_UNKNOWN_HELPPAGE,'Page: '+page);}
          {We're searching for a name...}
          pg:=-1;
          name_search:=TRUE;
          strip_leading_spaces(page);
          if pos('INDEX',page)>0 then
          begin
               pindex:=true;
               writeln_stdout('Help subjects within the help file: ');
               writeln_Stdout_nl;
          end;
     end;


     begin
          (* Open the helpfile *)
          cp:=1;
          assign(helpfile,get_basedir(master_db)+HELP_DIR+HELP_PAGE_FILE);
{          assign(helpfile,DIR_MASK+HELP_DIR+HELP_PAGE_FILE);}
          {$I-}
          reset(helpfile);
          {$I+}

          (* If an error occured, then report an error *)
          err:=IOResult;
          if err<>0 then
             nonfatal_error(ERROR_FILE_OPENING,'Help File: '+get_basedir(master_db)+HELP_DIR+HELP_PAGE_FILE)
{             nonfatal_error(ERROR_FILE_OPENING,'Help File: '+DIR_MASK+HELP_DIR+HELP_PAGE_FILE)}
          else
              begin
                   (* Otherwise, read a line from the helpfile *)
                   readln(helpfile,s);

                   (* Whilst we're not at the end of the file *)
                   while not(eof(helpfile)) and not(finish) do
                   begin
                        (* If a page seperator is found *)
                        if s='+' then
                        begin
                             (* If we've just found the start of the page
                                to display, set a flag *)
                             output:=false;

                             if pg=cp then
                                located:=true;

                             (* And we're to display them all *)
                             if pg=0 then
                             begin
                                  (* Display a prompt *)
                                  write('Press any key to display the next page...');

                                  (* Wait for input *)
                                  ch:=readkey;

                                  (* Clear the current line with the prompt on *)
                                  write(#13);
                                  clreol;

                                  writeln_stdout_nl;
                             end;

                             (* Increment the current page number *)
                             inc(cp);

                        end else
                            (* If we're to display a specific page
                               and we've found it, or we're to display
                               all pages *)
                                if ((pg<>0) and (cp=pg)) or
                                   (pg=0) or (name_search) then
                                   begin
                                        tmp2:=s;
                                        tmp:=cut_token(tmp2,' ');
                                        strip_leading_spaces(tmp2);

                                        if pindex then
                                        begin
                                             if (tmp='#') and pindex then
                                             begin
                                                  if cols<3 then
                                                  begin
                                                       write_stdout(tmp2);
                                                       for ccnt:=length(tmp2) to 25 do
                                                           write_stdout(' ');
                                                       inc(cols);
                                                  end
                                                  else
                                                  begin
                                                       cols:=1;
                                                       writeln_stdout(tmp2);
                                                  end
                                             end;

                                             located:=true;
                                        end
                                        else
                                        if ((tmp='#') and (page=tmp2))
                                        or (((pg<>0) and (cp=pg)) or (pg=0)) then
                                        begin
                                             output:=true;
                                             located:=true;
                                        end;

                                        if (tmp='#') then
                                        begin
                                             if colour then
                                                textcolor(lightred);
                                             s:=newline+page+newline;
                                        end;

                                        (* Write out the current line to stdout *)
                                        if output then
                                           writeln_stdout(s);

                                        if (tmp='#') and colour then
                                           textcolor(lightgray);

                                   end;

                        (* Get the next line from the helpfile *)
                        readln(helpfile,s);
                        if (pg>=0) and (s='=') then
                           finish:=true;
                   end;

                   (* Close the helpfile *)
                   close(helpfile);
              end;

          (* If we didn't find the specified help page, report an error *)
          if not located then
             nonfatal_Error(ERROR_UNKNOWN_HELPPAGE,'Page: '+page);
     end;

     (* Straighten out the lines *)
     if pindex then
        writeln_stdout_nl;

     (* Display a line, and the title banner *)
     for c:=1 to 79 do
         write_stdout('=');
     writeln_stdout_nl;
     title;
end;




procedure do_list(    s:string);
var
   t:text;
   err:integer;
   line:string;
   c:word;
begin
     assign(t,get_basedir(current_db)+SCRIPT_DIR+'\'+s+SOURCE_FILE_EXT);
{     assign(t,DIR_MASK+DATA_DIR+SCRIPT_DIR+'\'+s+SOURCE_FILE_EXT);}

     {$I-}
     reset(t);
     {$I+}

     err:=IOResult;
     if err<>0 then
        nonfatal_error(ERROR_FILE_OPENING,'Source File: '+get_basedir(current_db)+SCRIPT_DIR+'\'+s+SOURCE_FILE_EXT)
     else
         begin
              writeln_stdout('Listing '+get_basedir(current_db)+SCRIPT_DIR+'\'+s+SOURCE_FILE_EXT);
              write_stdout(  '--------');
              for c:=1 to length(get_basedir(current_db)+SCRIPT_DIR+'\'+s+SOURCE_FILE_EXT) do
                  write_stdout('-');
              writeln_stdout_nl;
              readln(t,line);
              writeln_stdout(line);
              while not eof(t) do
              begin
                   readln(t,line);
                   writeln_stdout(line);
              end;
              close(t); {Close the file}
         end;
end;



function get_command(    s:string):command_type;
(* Convert a command string to its internal representation *)

(* Ref. Section x.1.6 *)

var
   x:byte;
begin
     s:=upstring(s);

     x:=1;

     (* Whilst we have commands left, and we haven't found a command *)
     while (x<=NOCMDS) and (commands[x].text<>s) do
           inc(x);

     (* If we found the command *)
     if commands[x].text=s then
        (* Return the appropriate command type *)
        get_command:=commands[x].command
     else
         (* Otherwise, we didn't find it *)
         get_command:=unknown;
end;




procedure list_srcs;
(* Display all files that are in the source file directory *)

(* Ref. Section x.1.7 *)

var
   dirinfo:searchrec;
begin

     (* Locate the first file *)
     findfirst(get_basedir(current_db)+SCRIPT_DIR+SOURCE_FILE_MASK,AnyFile,dirinfo);

     (* Whilst we have still got files to process *)
     while (doserror=0) do
     begin
          (* Display the first part of the filename, without the extension *)
          writeln_stdout(get_token(dirinfo.name,'.'));

          (* Locate the next file *)
          findnext(dirinfo);
     end;
end;





function check_command(var s:string;
                       var name:string;
                       var command:string):command_check;
(* Check the command and determine if it should be executed as an
   assignment, or 'raw' *)

(* Ref. Section x.1.8 *)

var
   rcc:command_check;
   str,sbu:string;
begin
     (* Initialise data *)
     sbu:=s;
     rcc:=standalone;

     (* Cut out the first token before an equals *)
     str:=cut_token(s,'=');

     (* Clear any spaces after the equals sign... *)
     strip_tail_spaces(str);

     (* If we have something *)
     if (str<>'') and (length(str)<=REL_NAME_SIZE) then
                      (* P A T C H    P A T C H ! *)
     begin
          (* Its an equals *)
          rcc:=assignmt;

          (* Copy the name of the relation to assign to *)
          name:=str;

          (* Copy the remainder (the command) *)
          command:=s;
     end
     else
     begin
          (* Its just a 'raw' execution command *)
          s:=sbu;
          name:='A relation';
          command:='';
     end;

     (* Return the appropriate type etc. *)
     check_command:=rcc;
end;




procedure check_rel(    rel:relation;
                       name:string);
(* Check the specified relation *)

(* Ref. Section x.1.9 *)

begin
     (* If its valid *)
     if assigned(rel) then
     begin
          (* Report the fact *)
          writeln_stdout(relation_name(rel)+' Returned.');
          report('Relation '+relation_name(rel)+' Returned.');
     end
     else
     begin
          (* Otherwise, report an error *)
          writeln_stdout(name+' could not be accessed/created.');
          report(name+' could not be accessed/created.');
     end
end;








procedure do_cli;
(* Main command line interface routine *)

(* Ref. Section x.1.12 *)

var
{   db:database;}
   loadcount:byte;
   t,fld,m,c,s,name,command:string;
   comm:command_type;
   queryrel,trel:relation;
   cc:command_check;
   pka,ska:keyarray;
   idx:index_struct_ptr;
   create_op,mst,st,exec,rl:string;
   relop:boolean;
   old_screen_mode:integer;
   f:file;
   rtstack:rtion_stack;
   nopen,nclose:byte;
   b_ok:boolean;
   ch:char;
begin
     (* Open the master database *)
     master_db:=open_db(MASTER_DB_NAME);
     open_relations(master_db);
{     rd_indexes(master_db);}

{     relation_display(master_db);
 }

     (* Should be an option to open on another database at
        some later stage. For now this will have to do *)

     (* Open the master database as the starting database *)
     current_db:=open_db(STARTUP_DB);
     open_relations(current_db);
{     rd_indexes(current_db);}

  {   relation_display(current_db);
   }

     (* During startup the active database is the master database *)
     {active_db:=master_db;}

     (* Create the main database *)
{     db:=create_db('BASE');}

     (* Create the main temporary stack *)
     rtstack:=rtion_create_stack;

     (* Reset some information *)
     terminate:=none;
     loadcount:=0;

     prompt_string:='>';

     (* Source the startup script file *)
     writeln_Stdout('Startup sequence initiated.');
     assign_inputfile(get_basedir(master_db)+SCRIPT_DIR+SCRIPT_STARTUP);

{     assign_inputfile(DIR_MASK+DATA_DIR+SCRIPT_DIR+SCRIPT_STARTUP);}

     (* Whilst not ready to terminate *)
     while terminate<>halt do
     begin
          (* Don't add a new line if its a script thats being executed
             - this leads to loads of spaces. *)
          if stdinput then
             writeln_stdout_nl;

          (* Get a command *)
          s:=get_command_string;

          (* Check the brackets on the expression *)
          b_ok:=brackets_ok(s,nopen,nclose);

          (* If a problem *)
          if not b_ok then
          begin
               (* Report details *)
               write_stdout('Mismatch! No. Open Brackets=');
               system.str(nopen,t);
               write_stdout(t+' No. Close Brackets=');
               system.str(nclose,t);
               writeln_stdout(t);
          end;

          (* Check the command and get the internal command type *)
          cc:=check_command(s,name,command);
          exec:=get_token(s,' ');
          relop:=false;
          mst:=s; {Needed at end of standalone case}
          case cc of
               (* Standalone, 'raw' commands *)
               standalone:
                   case get_command(cut_token(s,' ')) of
                        (* Exit - If its the first time then start up
                           the termination sequence by first sourcing
                           the shutdown script *)
                        lexit:terminate:=halt;
                        {if terminate=none then
                             begin
                                  writeln_stdout('Shutdown sequence started.');
                                  report('Initiating shutdown sequence.');
                                  terminate:=shutdown;
                                  assign_inputfile(get_basedir(active_db)+SCRIPT_DIR+SCRIPT_SHUTDOWN);
                             end
                             else
                                 (* Its the second time, so get ready to
                                    terminate proper *)
                                 terminate:=halt;               }

                             (* Display the specified help page *)
                        help:do_help_page(s);

                              (* Load the data on disk *)
                        load: begin
                                   if loadcount=0 then
                                   begin
                                        (* Load relations/indexes *)
                                        open_relations(current_db);
                                        rd_indexes(current_db);
                                        inc(loadcount);
                                   end
                                   else
                                       (* Attempted to load relations twice *)
                                       nonfatal_error(ERROR_ATTEMPT_RELOAD,'Use DISPOSE first.');
                              end;
                                  (* Display the relations in the database *)
                        display_rel:relation_display(current_db);

                                  (* Display the indexes in the database *)
                        display_index:display_idx_struct(idx_master);

                                  (* Display the specified relation *)
                        print:rl_display(relation_find(current_db,cut_token(s,' ')));

                                  (* Print attributes in the specified relation *)
                        describe:flds_print(relation_find(current_db,cut_token(s,' ')));

                                  (* List the source files *)
                        listsrcs:list_srcs;

                                  (* Flush the report file *)
                        flush:report_flush;

                                  (* Return the screen to the mode it was
                                     at on startup *)
                        normal:begin
                                    textmode(co80);
                                    title;
                                    window(1,3,lo(Windmax),hi(Windmax));
                               end;

                                  (* Set the screen to EGA/VGA 43/50 line mode *)
                        lhigh:begin
                                  textmode(co80+font8x8);
                                  title;
                                  window(1,3,lo(Windmax),hi(Windmax));
                             end;

                                  (* Display the amount of memory available *)
                        memavailable:begin
                                      system.str(memavail,m);
                                      writeln_stdout('Memory Available = '+m);
                                    end;


                                  (* Dispose of the index and relation structures *)
                        ldispose:begin
                                     idx_destroy(current_db);

                                     (* Dispose of the temporary relation stack *)
                                     rtion_dispose_all(current_db,rtstack);
                                     relations_dispose_all(current_db);
                                     loadcount:=0;
                                end;

                                  (* Delete a specified relation *)
                        ldelete:begin
                                    trel:=relation_find(current_db,cut_token(s,' '));
                                    relation_remove(current_db,trel);
                               end;

                                  (* Build an index on a specified relation *)
                        specindex:begin
                                       rl:=cut_token(s,' ');
                                       pka[1]:=cut_token(s,' ');
                                       pka[2]:=cut_token(s,' ');
                                       clear_keyarray(ska);

                                       idx:=add_index_hdr(idx_master,rl,pka,ska,nil);
                                       build_index(idx,current_db,rl);
                                  end;

                                  (* Print the specified relation using the
                                     specified index *)
                        idxprint:begin
                                      rl:=cut_token(s,' ');
                                      fld:=cut_token(s,' ');
                                      idx:=get_index(idx_master,relation_find(current_db,rl),fld);
                                      rl_idxprint(current_db,idx);
                                 end;

                                  (* Write the indexes to disk *)
                        idxstore:begin
                                      st_indexes(idx_master,current_db);
                                 end;

                                  (* Create an object from user input *)
                        create:begin
                                    (* Cut the data from the string to determine
                                       what object to create. Branch on result *)
                                    create_op:=cut_token(s,' ');

                                    (* Creating a new database! *)
                                    if (create_op='DATABASE') then
                                    begin
                                         (* Cut the next token, ie. the relation name *)
                                         create_op:=cut_token(s,' ');

                                         (* Check that there is infact some data there!*)
                                         if length(create_op)=0 then
                                            (* Report an error *)
                                            nonfatal_error(ERROR_NONAME,'A database name is necessary in order to create '+
                                                                           'a database!')
                                         else
                                         begin
                                              (* Create the database...*)
                                              writeln_stdout('Creating a new database: '+create_op);
                                              create_new_db(create_op,s)
                                         end
                                    end
                                    (* Creating a new relation *)
                                    else if (create_op='RELATION') then
                                    begin
                                         (* Get the relation name *)
                                         create_op:=cut_token(s,' ');

                                         (* Check the name is valid *)
                                         if length(create_op)=0 then
                                            (* Report an error *)
                                            nonfatal_error(ERROR_NONAME,'A relation name is necessary in order to create '+
                                                                           'the relation!')
                                         else
                                         begin
                                              (* Start the creation process by stripping the
                                                 leading spaces to the attribute list *)
                                              strip_leading_spaces(s);

                                              (* If there are no attributes specified in the list *)
                                              if length(s)=0 then
                                                 (* Report an error *)
                                                 nonfatal_error(ERROR_INSUFFICENT_ATTRIBUTES,'At least one attribute is '+
                                                    'necessary in a relation.')
                                              else
                                              begin
                                                   (* Set the relation created flag *)
                                                   relop:=true;

                                                   (* Create the new relation *)
                                                   trel:=relation_user_create(current_db,create_op,s)
                                              end;
                                         end
                                    end
                               end;

                                  (* Add a tuple to a specified relation *)
                        add:tuple_user_add(relation_find(current_db,cut_token(s,' ')));

                                  (* Source a specified file *)
                        srcfile:assign_inputfile(s);

                                  (* Reverse the temporary status of a specified
                                     relation *)
                        changestatus:relation_change(current_db,cut_token(s,' '));

                                  (* Remove all temporary relations in the stack *)
                        rmvtmp:begin
                                    rtion_dispose_all(current_db,rtstack);
                                    rtstack:=rtion_create_stack;
                               end;

                                  (* Build a query through user response *)
                        prompt:begin
                                    prompt_string:=s;
                                    strip_leading_spaces(prompt_string);
                                    strip_tail_spaces(prompt_string);
                                    prompt_String:=prompt_String+'>';
                               end;

                                  (* Clear the screen *)
                        clear:begin
                                   clrscr;
                                   title;
                              end;

                        war_con_info:warranty_conditions;
                        status:report_status;
                        cdebug:begin
                                    if get_token(s,' ')='' then
                                    begin
                                         report_stat('Debug setting',debug);
                                         write_stdout('Reverse setting? (Y/N) : ');
                                         ch:=upcase(readkey);
                                         writeln_stdout(ch);
                                         if ch='Y' then
                                              debug:=not(debug)
                                    end
                                    else
                                        debug:=(upstring(get_token(s,' '))='Y');

                                    report_stat('Debug setting',debug);
                               end;
                        cinfix:begin
                                    if get_token(s,' ')='' then
                                    begin
                                         report_stat('Infix operators',infix);
                                         write_stdout('Reverse setting? (Y/N) : ');
                                         ch:=upcase(readkey);
                                         writeln_stdout(ch);
                                         if ch='Y' then
                                              infix:=not(infix);
                                    end
                                    else
                                        infix:=(upstring(get_token(s,' '))='Y');

                                    report_stat('Infix operators',infix);
                               end;
                        timeformat:begin
                                    if get_token(s,' ')='' then
                                    begin
                                         report_timeformat;
                                         write_stdout('Reverse setting? (Y/N) : ');
                                         ch:=upcase(readkey);
                                         writeln_stdout(ch);
                                         if ch='Y' then
                                              ustime:=not(ustime);
                                    end
                                    else
                                        ustime:=(upstring(get_token(s,' '))='Y');

                                    report_timeformat;
                               end;
                         time:begin
                                    if get_token(s,' ')='' then
                                    begin
                                         report_stat('Operation Timing',timing);
                                         write_stdout('Reverse setting? (Y/N) : ');
                                         ch:=upcase(readkey);
                                         writeln_stdout(ch);
                                         if ch='Y' then
                                              timing:=not(timing);
                                    end
                                    else
                                        timing:=(upstring(get_token(s,' '))='Y');

                                    writeln_stdout('Accuracy is affected in multi-tasking environments.');
                                    report_stat('Operation Timing',timing);
                              end;
                         casesense:begin
                                        if get_token(s,' ')='' then
                                        begin
                                             report_stat('Case Sensitivity',casesensitivity);
                                             write_stdout('Reverse setting? (Y/N) : ');
                                             ch:=upcase(readkey);
                                             writeln_stdout(ch);
                                             if ch='Y' then
                                                casesensitivity:=not(casesensitivity);
                                    end
                                    else
                                        casesensitivity:=(upstring(get_token(s,' '))='Y');
                                    report_stat('Case Sensitivity',casesensitivity);
                                   end;
                         lcache:begin
                                     writeln_stdout('Be warned, cache is buggy with relations with a cardinality');
                                     writeln_stdout('greater than 20. This will be fixed in a later release.');
                                     writeln_stdout_nl;
                                    if get_token(s,' ')='' then
                                    begin
                                         report_stat('Internal Cache',cacheenabled);
                                         write_stdout('Reverse setting? (Y/N) : ');
                                         ch:=upcase(readkey);
                                         writeln_stdout(ch);
                                         if ch='Y' then
                                            cacheenabled:=not(cacheenabled)
                                    end
                                    else
                                        cacheenabled:=(upstring(get_token(s,' '))='Y');
                                    report_stat('Internal Cache',cacheenabled);
                               end;
                         literate:begin
                                    if get_token(s,' ')='' then
                                    begin
                                         report_stat('Iterative Parser',iterative);
                                         write_stdout('Reverse setting? (Y/N) : ');
                                         ch:=upcase(readkey);
                                         writeln_stdout(ch);
                                         if ch='Y' then
                                            iterative:=not(iterative)
                                    end
                                    else
                                        iterative:=(upstring(get_token(s,' '))='Y');
                                    report_stat('Iterative Parser',iterative);
                               end;
                         dptree:begin
                                    if get_token(s,' ')='' then
                                    begin
                                         report_stat('Display Parse Tree',dispparse);
                                         write_stdout('Reverse setting? (Y/N) : ');
                                         ch:=upcase(readkey);
                                         writeln_stdout(ch);
                                         if ch='Y' then
                                            dispparse:=not(dispparse)
                                    end
                                    else
                                        dispparse:=(upstring(get_token(s,' '))='Y');
                                    report_stat('Display Parse Tree',dispparse);
                               end;
                         ioon_s:begin
                                    if get_token(s,' ')='' then
                                    begin
                                         report_stat('Additional IO',ioon_setting);
                                         write_stdout('Reverse setting? (Y/N) : ');
                                         ch:=upcase(readkey);
                                         writeln_stdout(ch);
                                         if ch='Y' then
                                              ioon_setting:=not(ioon_setting);
                                    end
                                    else
                                        ioon_setting:=(upstring(get_token(s,' '))='Y');

                                    report_stat('Additional IO',ioon_setting);
                              end;
                         dopanic:begin
                                      panic_handler;
                                 end;
                         break:begin
                                    checkbreak:=true;
                               end;
                        version:begin
                                     writeln_stdout(what_Version);
                                end;
                        list:begin
                                  do_list(get_token(s,' '));
                             end;
                        comment:;
                        reports:begin
                                    report('User Report:'+s);
                                    writeln('Reported:'+s);
                                end;
                        what:begin
                                  writeln_stdout('You''re using:');
                                  writeln_stdout_nl;
                                  writeln_stdout_nl;
                                  leap_start;
                                  writeln_stdout_nl;
                                  writeln_Stdout_nl;
                                  writeln_Stdout_nl;
                                  title;
                             end;
                        use:begin
                                 strip_leading_spaces(s);
                                 s:=upstring(s);
                                 if s=get_dbname(current_db) then
                                    nonfatal_error(ERROR_ALREADY_OPEN,'Database: '+s)
                                 else
                                 if s='' then
                                 begin
                                      writeln_stdout('Available databases:');
                                      rl_display(rl_project(master_db,relation_find(master_db,LEAP_DATABASES),'NAME',''));
                                 end
                                 else
                                 begin
                                     dd(true);
                                     trel:=rl_select(master_db,relation_find(master_db,LEAP_DATABASES),
                                               '(NAME="'+s+'")','');
                                     queryrel:=rl_project(master_db,trel,'NAME','');
                                     dd(false);
                                     if TRACE then
                                     begin
                                          check_rel(queryrel,name);
                                          rl_display(relation_find(master_db,relation_name(queryrel)));
                                     end;
                                     s:=rl_getitem(queryrel);
                                     {relations_dispose_temp(master_db);}
                                     if s<>'' then
                                     begin
                                          writeln_stdout('Closing database ['+get_dbname(current_db)+']...');
                                          io_set;

                                          (* Must validate the database first *)
                                          idx_destroy(current_db);

                                          relations_dispose_all(current_db);

                                          destroy_db(current_db);

                                          writeln_stdout('Done.');
                                          io_on;
                                          writeln_stdout('Opening database ['+s+']');
                                          io_set;
                                          current_db:=open_db(s);
                                          open_relations(current_db);
                                          rd_indexes(current_db);

                                          {active_db:=current_db;}

                                          io_on;

                                          relation_info(current_db);

                                          assign_inputfile(SCRIPT_OPEN);

                                     end
                                     else
                                     begin
                                          nonfatal_error(ERROR_UNKNOWN_DATABASE,'');
                                     end
                                 end
                            end;
                        dir:writeln_stdout('This isn''t MS-DOS you know!');
                        unknown:writeln_stdout('Unknown command. Type HELP for help pages.');
                   else
                        relop:=true;
                        st:=concat(exec,s);
                        if iterative then
                           trel:=process_query(current_db,mst,'')
                        else
                            if infix then
                               trel:=infix_get_relation(current_db,mst,'',rtstack,0)
                            else
                                trel:=new_get_relation(current_db,st,'',rtstack,0);
                   end;
               assignmt:
                      begin
                          relop:=true;
                          st:=concat(exec,s);
                          if iterative then
                             trel:=process_query(current_db,s,name)
                          else
                              if infix then
                                 trel:=infix_get_relation(current_db,s,name,rtstack,0)
                              else
                                  trel:=new_get_relation(current_db,s,name,rtstack,0);
                      end;
               else
                   nonfatal_error(ERROR_CANNOT_PROCESS,'');
               end;

               if relop then
               begin
                    check_rel(trel,name);
                    relop:=false;
               end;

               if (memavail<=PANIC_THRESHOLD) or (get_dbname(current_db)=MASTER_DB_NAME) then
               begin
                    io_set;
                    panic_handler;
                    io_on;
               end;

               get_cli_error(true);
     end;

     writeln_stdout('Closing database ['+get_dbname(current_db)+']');

     io_set;

{     (* Dispose of the database *)
     rtion_dispose_all(current_db,rtstack);}

     idx_destroy(current_db);

     relations_dispose_all(current_db);

     destroy_db(current_db);

     (* Set the shutdown flag, so master is cleaned up properly *)
     dtypes.shutdown:=true;

     idx_destroy(master_db);

     relations_dispose_all(master_db);

     destroy_db(master_db);

     io_on;
end;

var
   smem,emem:longint;
   s:string;
   c:byte;
   ch:char;

begin
     (* Record the amount of memory available at startup *)
     lowest_mem:=memavail;
     smem:=memavail;

     set_cli_Error(0);

     (* Disable ctrl-c *)
     checkbreak:=false;

     (* Store the screen mode at startup *)
     screen_mode:=lastmode;

     (* Configure standard output *)
     setup_stdout;

     (* Display the title banner *)
     title;

     window(1,3,lo(Windmax),hi(Windmax));

     if nag then
     begin
          leap_start;

          writeln_stdout_nl;
          writeln_stdout_nl;
          writeln_stdout_nl;
     end;


     if nag then
        delay(2500);

     (* Report the start of a session *)
     report('LEAP Execution Started.');
     report(REVISION_TEXT+REVISION_NUMBER+' - '+REVISION_DATE);


     (* Report startup to the user *)
     writeln_stdout_nl;
     writeln_stdout('Starting.');

     (* Report the amount of free memory *)
     system.str(smem,s);
     if debug then
        writeln_stdout('Memory available at start: '+s);
     writeln_stdout_nl;
     writeln_stdout('This is LEAP - '+what_Version);
     writeln_stdout_nl;

     writeln_stdout('LEAP B0.10 offers:');
     writeln_stdout_nl;
     if colour then
        textcolor(lightred);
     writeln_stdout('+ Support for Relational Structures.');
     writeln_stdout('+ Support for the Relational Algebra.');
     writeln_stdout('+ Nested expressions.');
     writeln_stdout('+ Parse tree review.');
     writeln_stdout('+ Query timing.');
     writeln_stdout('+ Data dictionary.');
     writeln_stdout('+ Extensive documentation.');
     writeln_stdout('+ Examples from popular database texts.');
     writeln_stdout('+ Online help.');
     writeln_stdout('+ Full source code under the GNU General Public License.');
     writeln_stdout_nl;
     if colour then
        textcolor(lightgray);
     if nag then
     begin
          write_stdout('Press any key to continue.');
          ch:=readkey;
          writeln_stdout_nl;
     end;

     if INFO then
        warranty_conditions;

     (* Do the command line interface, and everything associated with it! *)
     do_cli;

     (* Shutdown the reporting *)
     report_shutdown;

     (* Return the text mode to 80x25 *)
     TextMode(co80);

     (* Get the amount of free memory *)
     emem:=memavail;

     if DEBUG then
     begin
          (* Display how much *)
          system.str(emem,s);
          writeln_stdout('Memory available at termination: '+s);

          (* Calculate how much was 'lost' whilst work progressed *)
          system.str(smem-emem,s);
          writeln_stdout(' Memory Leakage: '+s);
          system.str(lowest_mem,s);
          writeln_stdout(' Lowest memory = '+s);
     end;

     (* Display the title banner *)
     title;

     (* Display contact/download information *)
     writeln_stdout_nl;
     writeln_stdout('Please send all comments, bugs, suggestions etc. to:');
     writeln_stdout('richard_leyton@mail.amsinc.com and/or e0190404@brookes.ac.uk');
     writeln_stdout_nl;
     writeln_stdout('Latest version available via anonymous ftp from:');
     writeln_stdout_nl;
     writeln_stdout('ftp.demon.co.uk            /pub/compsci/databases/leap');
     writeln_stdout('micros.hensa.ac.uk         /micros/ibmpc/dos/k/k174');
     writeln_stdout('sunsite.doc.ic.ac.uk       /computing/systems/ibmpc/simtel/msdos');
     writeln_stdout('oak.oakland.edu            /SimTel/msdos/math');
     writeln_stdout('(See the file location.txt for a list of all locations known');
     writeln_stdout(' containing LEAP)');
     writeln_stdout_nl;
     writeln_stdout('Also WWW at following URL:');
     writeln_stdout('  http://www.brookes.ac.uk/~e0190404/leap.html');
     writeln_stdout('  http://www.brookes.ac.uk/cms/students/richard_leyton/leap.html');
     writeln_stdout_nl;
     writeln_stdout('For warranty and conditions, start LEAP with INFO parameter, or type INFO');
     writeln_stdout('at the LEAP prompt. Alternatively, see the accompanying documentation for');
     writeln_stdout('more details. Use of the program implies you have done and accepted this.');
     writeln_stdout_nl;
     if nag then
        writeln_stdout('Press <ENTER> to continue');

     if nag then
     begin
          repeat
                ch:=readkey;
          until ch=#13;

          writeln_stdout_nl;
          writeln_stdout_nl;

          leap_start;

          delay(500);

     end;

     (* Close standard output *)
     stdout_close;

     (* Finish *)
end.

