unit index;

(*
Unit: Index
Description: Indexing facilities for relations
References: References to the documentation is to the file index.doc
            and section numbers are given as relative to this document.
Notes: B-Tree code based on examples from
       'Data Structures and Program Design', by Robert. L. Kruse,
       1984 - Published by Prentice-Hall, ISBN 0131962531
Revision Notes:
         07/12/1994 - Initial Development
         15/12/1994 - Index Storage & retrieval
         18/12/1994 - Post pointer create check and polite shutdown
         16/01/1995 - Export of comparekeyequal so comparisons can be made of keytype
         16/01/1995 - Export of nodesearch to enable location of given key to be found
         27/03/1995 - Extensive Commenting.
         28/03/1995 - Documented (index.doc)
	 22/06/1996 - Placed under GNU General Public License.

To-Do:
      There is duplicity with grab_tuple, loadfirst_tuple and loadnext_tuple
            and the funcionality provided by the tuples unit for binary reads
            of tuples.
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)

interface

{$O+}

uses fields,dtypes,tuples;

function create_idx_ref_node:index_struct_ptr;
function create_idx_struct:index_Struct_hdr_ptr;
function add_index_hdr(    ll:index_struct_hdr_ptr;
                           relation:string;
                           primary,secondary:keyarray;
                           index_struct:nodeptr):index_struct_ptr;
procedure build_keyarray(     a:array of string;
                         var  ka:keyarray);
procedure clear_keyarray(var k:keyarray);
procedure display_idx(i:index_Struct_ptr;
                      x:word);
procedure display_idx_struct(ll:index_struct_hdr_ptr);
procedure dispose_idx_struct(var idx:index_Struct_hdr_ptr);
procedure insert_tuple_idx(    ta:tuple_attributes;
                               rel:relation;
                               diskaddr:loc_type;
                               idx:index_struct_hdr_ptr);
function grab_tuple(var f:file;
                        pos:loc_type):string;
function loadfirst_tuple(var f:file;
                             pos:loc_type;
                             rel:relation):tuple_attributes;
function loadnext_tuple(var f:file;
                        var ta:tuple_attributes;
                            pos:loc_type):tuple_attributes;
procedure nodesearch(    target:keytype;
                         root:nodeptr;
                     var found:boolean;
                     var targetnode:nodeptr;
                     var targetpos:position);
function btree_findfirst(    idx:nodeptr;
                         var item:word;
                         var searchstack:np_stack):nodeptr;
function btree_findnext(    idx:nodeptr;
                        var item:word;
                        var searchstack:np_stack):nodeptr;
procedure build_index(    cidx:index_struct_ptr;
                          db:database;
                          rel:string);
procedure rebuild_indexes(    idx:index_struct_hdr_ptr;
                              db:database);
function idx_findnext(    idx_node:index_struct_ptr):index_struct_ptr;
function idx_findfirst(    idx:index_struct_hdr_ptr;
                           relation:string):index_struct_ptr;
function np_create_stack:np_stack;
procedure np_flush_stack(var st:np_stack);
procedure st_indexes(var idx:index_Struct_hdr_ptr;
                     var db:database);
procedure rd_indexes(    db:database);
procedure btree_destroy(var idx:nodeptr);
function get_index(    idx:index_struct_hdr_ptr;
                       rel:relation;
                       pk:string):index_struct_ptr;
function comparekeyequal(    first,second:keytype):boolean;
procedure idx_Destroy(    db:database);

implementation

uses dos,utils,dbase,relations;       (* Units needed internally *)




function create_idx_ref_node:index_struct_ptr;

(* Creates a node for the index reference structure *)

(* Ref. Section x.1.1 *)

var
   x:index_struct_ptr;
   cnt:word;
begin
     new(x);
     check_Assign(x,'index.create_idx_ref_node');

     (* Initialise the variables within the node *)
     with x^ do
     begin
          next:=nil;
          previous:=nil;
          with index_data do
          begin
               relation:='';
               for cnt:=1 to MAXKEYS do
               begin
                    primary[cnt]:='';
                    secondary[cnt]:='';
               end;
               index_struct:=nil;
          end;
     end;
     (* Return the node *)
     create_idx_ref_node:=x;
end;




function create_idx_struct:index_Struct_hdr_ptr;

(* Create the main header node of the index reference structure *)

(* Ref. Section x.1.2 *)

var
   x:index_Struct_hdr_ptr;
begin
     (* Create the node *)
     new(x);
     check_assign(x,'index.creade_idx_struct');

     (* Initialise information in node *)
     with x^ do
     begin
          head:=nil;
          no_nodes:=0;
     end;
     (* Return the structure *)
     create_idx_struct:=x;
end;

procedure build_keyarray(     a:array of string;
                         var  ka:keyarray);
(* Builds values in an array of type keyarray from an open array of strings *)

(* Ref. Section x.1.3 *)

var
   c:1..MAXKEYS;
begin
     (* Process entire source array - HIGH returns max. element in array *)
     for c:=1 to HIGH(ka) do
         ka[c]:=a[c];
end;


procedure build_node(    isp:index_struct_ptr;
                         relation:string;
                         primary,secondary:keyarray;
                         index_struct:nodeptr);
(* Populate the index reference node with reference information *)

(* Ref. Section x.2.1 *)

begin
     isp^.index_data.relation:=relation;
     isp^.index_data.primary:=primary;
     isp^.index_data.secondary:=secondary;
     isp^.index_data.index_struct:=index_struct;
     assign(isp^.fileptr,DIR_MASK+DATA_DIR+RELATION_DIR+'\'+relation+RELATION_EXT);
end;


function compare_pos(current_node,insert_node:index_struct_ptr):boolean;
(* Compare two index reference nodes (True if current_node < insert_node *)

(* This will only suffice until there are more primary keys in use,
   priority should be given to primary keys, *then* secondary keys,
   which isn't happening at the moment *)

(* Ref. Section x.2.2 *)

var
   x:word;
   b:boolean;
begin
     compare_pos:=false;
     if (current_node^.index_data.relation<>insert_node^.index_data.relation)
     then compare_pos:=true
     else
     begin
          x:=0;
          b:=false;
          (* Check the keys *)
          while (x<=MAXKEYS) and not b do
          begin
               inc(x);
               if (current_node^.index_data.primary[x]<insert_node^.index_data.primary[x])
               then b:=true
               else
               if (current_node^.index_data.secondary[x]<insert_node^.index_data.secondary[x])
               then b:=true;
          end;
          compare_pos:=b;
     end;
end;




function add_index_hdr(    ll:index_struct_hdr_ptr;
                           relation:string;
                           primary,secondary:keyarray;
                           index_struct:nodeptr):index_struct_ptr;
(*
Add an index structure to the index reference structure in the correct
location, ensuring an ordered linked list results
*)

(* Ref. Section x.1.4 *)

var
   idx_node:index_struct_ptr;
   current,previous:index_Struct_ptr;
begin
     idx_node:=create_idx_ref_node;

     build_node(idx_node,relation,primary,secondary,index_struct);


     (* Nothing in the list at the moment *)
     if not assigned(ll^.head) then
        ll^.head:=idx_node
     else
     begin
          current:=ll^.head;
          previous:=nil;
          (* Find the position to insert the new key in *)
          while assigned(current) and compare_pos(current,idx_node) do
          begin
               previous:=current;
               current:=current^.next;
          end;
          (* current should now point to the appropriate node, which
             should be one *after* the pos to insert... or at end *)

             (* We're at the end of the list *)
          if not assigned(current) then
          begin
               previous^.next:=idx_node;
               idx_node^.previous:=previous;
          end
          else
              (* We're at the start of the list, which requires the
                 head node to be modified *)
              if current=ll^.head then
              begin
                   idx_node^.next:=current;
                   current^.previous:=idx_node;
                   ll^.head:=idx_node;
              end
              else
              begin
                   (* We've got a 'normal' insert *)
                   idx_node^.next:=current;
                   idx_node^.previous:=previous;
                   previous^.next:=idx_node;
                   current^.previous:=idx_node;
              end;

     end;
     (* Just so we don't have to count the list each time *)
     inc(ll^.no_nodes);

     (* Return the node *)
     add_index_hdr:=idx_node;
end;


procedure clear_keyarray(var k:keyarray);
(* Runs through a keyarray, resetting all entries *)

(* Ref. Section x.1.5 *)

var
   c:word;
begin
     for c:=1 to MAXKEYS do
         k[c]:='';
end;



procedure display_idx(i:index_Struct_ptr;
                      x:word);
(* Display all of the index information for a particular node *)

(* Ref. Section x.1.6 *)

var
   y:word;
   sx,sy:string;
begin
     system.str(x,sx);
     writeln_stdout(sx+' -> Relation: '+i^.index_data.relation);

     (* Print the primary keys *)
     for y:=1 to MAXKEYS do
     begin
          system.str(y,sy);
          writeln_stdout(sx+' -> Primary Key #'+sy+': '+i^.index_data.primary[y]);
     end;

     (* Print the secondary keys *)
     for y:=1 to MAXKEYS do
     begin
          system.str(y,sy);
          writeln_stdout(sx+' -> Secondary Key #'+sy+': '+i^.index_data.secondary[y]);
     end;
end;



procedure display_idx_struct(ll:index_struct_hdr_ptr);
(* Display all of the index entries for a given index structure *)

(* Ref. Section x.1.7 *)

var
   i:index_Struct_ptr;
   x,y:word;
begin
     (* Get the head of the list *)
     i:=ll^.head;
     x:=1;
     (* Walk through the list until we fall off the end *)
     while assigned(i) do
     begin
          (* Print the information *)
          display_idx(i,x);
          writeln;

          (* Get the next node *)
          i:=i^.next;
          inc(x);
     end;
end;



function idx_abs_findfirst(    idx:index_struct_hdr_ptr):index_struct_ptr;
(* Find the very first index reference entry *)

(* Ref. Section x.2.3 *)

var
   c:index_struct_ptr;
begin
     (* If the index structure is valid, return the head node *)
     if assigned(idx) then
        c:=idx^.head
     else
         (* Otherwise, return nil *)
         c:=nil;

     (* Return the value *)
     idx_abs_findfirst:=c;
end;



function idx_abs_findnext(    idx_node:index_struct_ptr):index_struct_ptr;
(* Find the very next index reference entry *)

(* Ref. Section x.2.4 *)

begin
     idx_abs_findnext:=idx_node^.next
end;




function idx_findfirst(    idx:index_struct_hdr_ptr;
                           relation:string):index_struct_ptr;
(* Find the first index reference entry matching the given relation *)

(* Ref. Section x.1.8 *)

var                         (* Check while loop condition for rels without index entry *)
   c:index_struct_ptr;
   cnt:byte;
begin
     (* Get the first index *)
     if assigned(idx) then
        c:=idx^.head
     else
         c:=nil;

     cnt:=0;
     (* Move through the structure until we reach the relation we're after *)
     while (assigned(c)) and (c^.index_data.relation<>relation) and (cnt<20) do
     begin
           c:=c^.next;
           inc(cnt);
     end;

     if cnt=20 then
        nonfatal_error(ERROR_UNDEFINED,'idx');

     if (assigned(c)) and (c^.index_data.relation=relation) then
        idx_findfirst:=c (* First idx of relation, or nil *)
     else
         idx_findfirst:=nil;
end;



function idx_findnext(    idx_node:index_struct_ptr):index_struct_ptr;
(* Find the next index reference structure with the same relation name *)

(* Ref. Section x.1.9 *)

var
   rel:string;
begin
     (* If the relation of the next entry matches the current entry, then
        return the next entry, otherwise we've exhausted the entries *)
     if (assigned(idx_node^.next))
     and (idx_node^.next^.index_data.relation=idx_node^.index_data.relation) then
        idx_findnext:=idx_node^.next
     else
         idx_findnext:=nil;
end;




procedure dispose_idx_hdr(var idx:index_struct_hdr_ptr);
(* Dispose of the main index structure header information *)

(* Ref. Section x.2.5 *)

begin
     if assigned(idx) then
     begin
          dispose(idx);
          idx:=nil;
     end;
end;



procedure dispose_idx_node(var isp:index_Struct_ptr);
(* Dispose of a index reference structure node *)

(* Ref. Section x.2.6 *)

begin
     if assigned(isp) then
     begin
          dispose(isp);
          isp:=nil;
     end;
end;




procedure dispose_idx_struct(var idx:index_Struct_hdr_ptr);

(* Dispose of the entire index reference structure accordingly *)

(* Ref. Section x.1.10 *)

var
   c,o:index_struct_ptr;
begin
     (* Get the head node *)
     c:=idx_abs_findfirst(idx);

     (* While we've got valid nodes *)
     while assigned(c) do
     begin
          o:=c;

          (* Get the next node *)
          c:=idx_abs_findnext(c);

          (* Dispose of the old node *)
          dispose_idx_node(o);
     end;

     (* Dispose of the header information *)
     dispose_idx_hdr(idx);
end;



function grab_tuple(var f:file;
                        pos:loc_type):string;
(* Physically pulls a tuple off of a disk, from the given file at the
   given position *)

(* Ref. Section x.1.11 *)

var
   data:array[1..MAX_FILE_DATA_SIZE] of char;
   noret:word;
   x,y:word;
   s:string;
begin
     (* Locate the position in the file *)
     seek(f,pos);

     (* Read block *)
     blockread(f,data,MAX_FILE_DATA_SIZE,noret);
     x:=2;

     (* Whilst we haven't encountered end-of-line markers *)
     while (data[x-1]<>#13) and (data[x]<>#10) do
           inc(x);

     s:='';
     (* Copy the information out of the buffer *)
     for y:=1 to x-2 do
         s:=concat(s,data[y]);

     (* Return the resulting string *)
     grab_tuple:=s;
end;



function loadfirst_tuple(var f:file;
                             pos:loc_type;
                             rel:relation):tuple_attributes;
(* Read a tuple for the first time, and create the tuple structure *)

(* Ref. Section x.1.12 *)

var
   s:string;
   ta:tuple_attributes;
   fld:field;
   nofields:word;
begin
     (* Reset the file ptr and the read the data *)
     reset(f,1);
     s:=grab_tuple(f,pos);

     (* Create the tuple *)
     ta:=tuple_prep;

     (* Fill the tuple with field information *)
     populate_fields(rel,ta,nofields);

     (* Break the data read off off the disk into the tuples *)
     break_down(s,ta,nofields);

     (* Return the tuple *)
     loadfirst_tuple:=ta;
end;




function loadnext_tuple(var f:file;
                        var ta:tuple_attributes;
                            pos:loc_type):tuple_attributes;
(* Read a tuple off of the disk, populating the tuple created by
   loadfirst_tuple *)

(* Ref. Section x.1.13 *)

var
   s:string;
   r:relation;
   nofields:word;
begin
     (* Read the data off of the disk *)
     s:=grab_tuple(f,pos);

     (* Split the information into the tuple *)
     break_down(s,ta,nofields);

     (* Return the tuple *)
     loadnext_tuple:=ta;
end;


(********************************************************************)
(*                        B-Tree routines                           *)
(********************************************************************)


function comparekeyless(    first,second:keytype):boolean;
var
   res:boolean;
   c:word;
begin
     res:=false;
     c:=0;
     while (c<=MAXKEYS) and not res do
     begin
          inc(c);
          if (first.primary[c]<second.primary[c]) then
             res:=true;
     end;
     c:=0;
     if not res then
        while (c<=MAXKEYS) and not res do
        begin
             inc(c);
             if (first.secondary[c]<second.secondary[c]) then
                res:=true;
        end;

     comparekeyless:=res;
end;



function comparekeyequal(    first,second:keytype):boolean;

(* Ref. Section x.1.25 *)

var
   res:boolean;
   pc,sc:word;
   c:word;
begin
     res:=false;
     c:=0;
     pc:=1;
     sc:=1;
     while (c<=MAXKEYS) and not res do
     begin
          inc(c);
          if (first.primary[c]=second.primary[c]) then
             inc(pc);
     end;
     c:=0;
     if not res then    {Equal so far}
        res:=false;
        while (c<=MAXKEYS) and not res do
        begin
             inc(c);
             if (first.secondary[c]=second.secondary[c]) then
                inc(sc);
        end;

     comparekeyequal:=(pc=MAXKEYS) and (sc=MAXKEYS);
end;



procedure searchnode(    target:keytype;
                         p:nodeptr;
                     var found:boolean;
                     var k:position);
(* Searches keys in node p^ for target; returns location k of target, or
   branch on which to continue search *)

(* Ref. Section x.2.7 *)

begin
     with p^ do
     begin
          if comparekeyless(target,key[1]) then
          begin
               found:=false;
               k:=0;
          end
          else
          begin              (* Start a sequential search through keys *)
               k:=count;
               while (comparekeyless(target,key[k])) and (k>1) do
                     dec(k);
               found:=comparekeyequal(target,key[k])
          end
     end;
end;



procedure nodesearch(    target:keytype;
                         root:nodeptr;
                     var found:boolean;
                     var targetnode:nodeptr;
                     var targetpos:position);
(* Search a given node from the B-Tree for a key *)

(* Ref. Section x.1.14 *)

begin
     if not assigned(root) then
        found:=false
     else
     begin
          SearchNode(target,root,found,targetpos);
          if found then
             targetnode:=root
          else
              nodesearch(target, root^.branch[targetpos],found,targetnode,targetpos)
     end
end;



procedure PushIn(    x:keytype;
                     xr,p:nodeptr;
                     k:position);
(* Inserts key x and pointer xr into node p^ at position k *)
var
   i:0..nodemax;        (* Index to move keys to make room for x *)

begin
     with p^ do
     begin
          for i:=count downto k+1 do    (* Shift all keys and branches
                                           to the right *)
          begin
               key[i+1]:=key[i];
               branch[i+1]:=branch[i];
          end;
          key[k+1]:=x;
          branch[k+1]:=xr;
          count:=count+1;
     end
end;



procedure Split(    x:keytype;
                    xr,p:nodeptr;
                    k:position;
                var y:keytype;
                var yr:nodeptr;
                    parent:nodeptr);
(* Splits node p^ with key x and pointer xr at position k into nodes
   p^ and yr^ with median key y *)

var
   i:0..nodemax;    (* Used for compying from p^ to new node *)
   median:position;

begin
     if k<=nodemin then
        median:=nodemin             (* new key x goes to left half *)
     else
         median:=nodemin+1;
     new(yr);                       (* get new node and put it on the right *)
     check_assign(yr,'index.split');
     with p^ do
     begin
          for i:=median+1 to nodemax do
          begin
               yr^.key[i-median]:=key[i];
               yr^.branch[i-median]:=branch[i];
          end;
          yr^.count:=nodemax-median;
          count:=median;
          if k<=nodemin then
             PushIn(x,xr,p,k)
          else
              PushIn(x,xr,yr,k-median);
          y:=key[count];
          yr^.branch[0]:=branch[count];
          dec(count);
     end
end;



procedure PushDown(    newkey:keytype;
                       p:nodeptr;
                   var pushup:boolean;
                   var x:keytype;
                   var xr:nodeptr;
                       parent:nodeptr);
var
   k:position;          (* Branch on which to continue the search *)
   found:boolean;       (* Is newkey already in the tree (error)? *)
begin
     if not assigned(p) then
     begin              (* Cannot insert into empty tree; recursion terminates *)
          pushup:=true;
          x:=newkey;
          xr:=nil;
     end
     else
     begin              (* Search current node *)
          SearchNode(newkey,p,found,k);
          if found then
          begin
               writeln('Error: inserting duplicate key');
               exit;
          end;
          PushDown(newkey,p^.branch[k],pushup,x,xr,p);
          if pushup then(* Reinsert median key *)
          with p^ do
          if count<nodemax then
          begin
               pushup:=false;
               PushIn(x,xr,p,k)
          end
          else
          begin
               pushup:=true;
               Split(x,xr,p,k,x,xr,parent);
          end
     end
end;




procedure keyinsert(    newkey:keytype;
                    var root:nodeptr);
(* Inserts newkey into the B-tree with given root. Requires that newkey
   is not already present in the tree. *)

var
   pushup:boolean;      (* Has the height of the tree increased? *)
   x:keytype;           (* Node to be re-inserted as new root *)
   xr,p:nodeptr;        (* xr = subtree on right of x *)
                        (* p = temporary pointer *)

begin
     PushDown(newkey,root,pushup,x,xr,root);
     if pushup then     (* Tree grows in height *)
     begin              (* Make a new root *)
          new(p);
          check_Assign(p,'index.keyinsert');
          with p^ do
          begin
               count:=1;
               key[1]:=x;
               branch[0]:=root;
               branch[1]:=xr;
               root:=p;
          end;
     end;
end;




procedure Remove(    p:nodeptr;
                     k:position);

(* Removes key[k] and branch[k] from p^ *)

var
   i:position;          (* Index to move entries *)
begin
     with p^ do
     begin
          for i:=k+1 to count do
          begin
               key[i-1]:=key[i];
               branch[i-1]:=branch[i]
          end;
          dec(count);
     end
end;




procedure Successor(    p:nodeptr;
                        k:position);

(* Replaces p^.key[k] by its immediate successor under natural order *)

var
   q:nodeptr;   (* Used to move down the tree to a leaf *)
begin
     q:=p^.branch[k];
     while assigned(q^.branch[0]) do
           q:=q^.branch[0];
     p^.key[k]:=q^.key[1];
end;



procedure MoveRight(    p:nodeptr;
                        k:position);
var
   c:position;

begin
     with p^.branch[k]^ do
     begin
          for c:=count downto 1 do
          begin
               key[c+1]:=key[c];
               branch[c+1]:=branch[c]
          end;
          branch[1]:=branch[0];
          inc(count);
          key[1]:=p^.key[k]
     end;
     with p^.branch[k-1]^ do
     begin
          p^.key[k]:=key[count];
          p^.branch[k]^.branch[0]:=branch[count];
          dec(count);
     end
end;



procedure MoveLeft(    p:nodeptr;
                       k:position);
var
   c:position;
begin
     with p^.branch[k-1]^ do
     begin
          inc(count);
          key[count]:=p^.key[k];
          branch[count]:=p^.branch[k]^.branch[0];
     end;
     with p^.branch[k]^ do
     begin
          p^.key[k]:=key[1];
          branch[0]:=branch[1];
          dec(count);
          for c:=1 to count do
          begin
               key[c]:=key[c+1];
               branch[c]:=branch[c+1];
          end
     end
end;




procedure Combine(    p:nodeptr;
                      k:position);
var
   c:position;
   q:nodeptr;
begin
     q:=p^.branch[k];
     with p^.branch[k-1]^ do
     begin
          inc(count);
          key[count]:=p^.key[k];
          branch[count]:=q^.branch[0];
          for c:=1 to q^.count do
          begin
               inc(count);
               key[count]:=q^.key[c];
               branch[count]:=q^.branch[c]
          end
     end;
     with p^ do
     begin
          for c:=k to count-1 do
          begin
               key[c]:=key[c+1];
               branch[c]:=branch[c+1];
          end;
          dec(count);
     end;
     dispose(q);
end;




procedure Restore(    p:nodeptr;
                      k:position);

(* Finds a key and inserts it into p^.branch[k]^ so as to restore minimum *)

begin
     if k>0 then
        if p^.branch[k-1]^.count>nodemin then       (* Move key to right *)
           MoveRight(p,k)
        else
            Combine(p,k)
     else                               (* Case: k=0 *)
         if p^.branch[1]^.count > nodemin then
            MoveLeft(p,1)
         else
             Combine(p,1)
end;



procedure RecDelete(    target:keytype;
                        p:nodeptr;
                    var found:boolean);
var
   k:position;  (* Location of target, or of branch on which to search *)

begin
     if not assigned(p) then
        found:=false    (* Hitting an empty tree is an error *)
     else
     with p^ do
     begin
          SearchNode(target,p,found,k);
          if found then
             if not assigned(branch[k-1]) then   (* Case: p^ is a leaf *)
                Remove(p,k)               (* Removes key from position k
                                             of p *)
             else
             begin
                  Successor(p,k);       (* Replaces key[k] by its successor *)
                  RecDelete(key[k],branch[k],found);
                  if not found then
                     writeln('Error occursed in recDelete');
                  end
             else
                 RecDelete(target,branch[k],found);

                 (* At this point, procedure has returned from recursive call *)

             if assigned(branch[k]) then
                if branch[k]^.count<nodemin then
                   Restore(p,k)
     end
end;




procedure Delete(    target:keytype;
                 var root:nodeptr);

(* Deletes the key target from the B-tree with the given root *)

var
   found:boolean;       (* Has the target been found in a subtree? *)
   p:nodeptr;           (* used to dispose of empty root *)
begin
     RecDelete(target,root,found);
     if not found then
        writeln('Error: Delete node not found in tree')
     else if root^.count=0 then     (* Root is now empty *)
     begin
          p:=root;
          root:=root^.branch[0];
          dispose(p)
     end
end;




procedure insert_tuple(    ta:tuple_attributes;
                       var isp:index_struct_ptr;
                           diskAddr:loc_type);

(* Insert a tuple into an index structure *)

(* Ref. Section x.2.8 *)

var
   btree:nodeptr;
   key:keytype;
   c:word;
begin

     (* Get the btree we're going to be inserting in *)
     btree:=isp^.index_data.index_struct;

     (* Populate the key information into the key data to be inserted *)
     for c:=1 to MAXKEYS do
     begin
          if (isp^.index_data.primary[c]<>'') then
             key.primary[c]:=ta^.tdatum[tuple_findfield(ta,isp^.index_data.primary[c])]^.data
          else
              key.primary[c]:='';

          if (isp^.index_data.secondary[c]<>'') then
             key.secondary[c]:=ta^.tdatum[tuple_findfield(ta,isp^.index_data.secondary[c])]^.data
          else
              key.secondary[c]:='';

     end;

     key.diskAddr:=diskAddr;

     (* Insert the key into the btree *)
     keyinsert(key,isp^.index_data.index_struct);

end;



procedure insert_tuple_idx(    ta:tuple_attributes;
                               rel:relation;
                               diskaddr:loc_type;
                               idx:index_struct_hdr_ptr);
(* Insert a given tuple into all of the indexes that exist for the
   specified relation *)

(* Ref. Section x.2.9 *)

var
   isp:index_struct_ptr;
   success,idxexist:boolean;
begin
     (* Get the first index *)
     isp:=idx_findfirst(idx,rel^.name);

     (* While its a valid node *)
     while assigned(isp) do
     begin
          (* Insert the tuple into the structure for this index *)
          insert_tuple(ta,isp,diskaddr);

          (* Get the next index *)
          isp:=idx_findnext(isp);
     end;
end;



function np_create_stack:np_stack;
(* Create a stack for traversal of the B-tree structure *)

(* Ref. Section x.2.10 *)

var
   ns:np_stack;
begin
     (* Create the node *)
     new(ns);
     check_assign(ns,'index.np_create_stack');

     (* Reset the information within the node *)
     ns^.head_node:=nil;
     ns^.no_items:=0;
     np_create_stack:=ns;
end;



procedure np_push_stack(    st:np_stack;
                            data:nodeptr;
                            pos:word);

(* Push a node onto the structure with the current position in the
   node *)

(* Ref. Section x.2.11 *)

var
   st_node:np_stack_node;
begin
     (* Create a new node *)
     new(st_node);
     check_assign(st_node,'index.np_push_stack');

     (* Populate the node with information in the parameter *)
     st_node^.data:=data;
     st_node^.next:=st^.head_node;
     st_node^.pos:=pos;

     (* Update the stack header information *)
     st^.head_node:=st_node;
     inc(st^.no_items);
end;



function np_pop_stack(    st:np_stack;
                      var pos:word):nodeptr;
(* Pop a node off of the structure *)

(* Ref. Section x.2.12 *)

var
   node:np_Stack_node;
begin
     (* If there is a head node *)
     if assigned(st^.head_node) then
     begin

          (* Get the head node from the stack *)
          np_pop_stack:=st^.head_node^.data;

          node:=st^.head_node;
          pos:=node^.pos;
          st^.head_node:=node^.next;

          (* Dispose of the node *)
          if assigned(node) then
             dispose(node);

          (* Update the stack header information *)
          dec(st^.no_items);
     end
        else
            (* If there is no head node, then return nil and report an error *)
            begin
                 np_pop_stack:=nil;
                 nonfatal_Error(ERROR_INTERNAL_STACK,'Attempt pop empty stack');
            end;
end;



function np_stack_size(    st:np_stack):word;
(* Return the number of nodes in the specified stack *)

(* Ref. Section x.2.13 *)

begin
     np_Stack_size:=st^.no_items;
end;




function np_stack_empty(    st:np_stack):boolean;
(* Return TRUE if the stack is empty, ie. 0 nodes *)
(* FALSE otherwise *)

(* Ref. Section x.2.14 *)

begin
     np_stack_empty:=(np_stack_size(st)=0) or (st=nil);
end;




procedure np_stack_dispose(var st:np_stack);
(* Dispose of the specified node *)

(* Ref. Section x.2.15 *)

begin
     (* If the node is assigned *)
     if assigned(st) then
     begin
          (* Dispose of the stack and reset to nil *)
          dispose(st);
          st:=nil;
     end;
end;



procedure np_flush_stack(var st:np_stack);
(* Dispose of the specified structure in its entirety *)

(* Ref. Section x.1.15 *)

var
   t:word;
begin
     (* Whilst the stack is not empty *)
     while not(np_stack_empty(st)) do
           (* Pop a node off of the stack *)
           np_pop_stack(st,t);

     (* Dispose of the stack header structure *)
     np_stack_dispose(st);
end;



function btree_findfirst(    idx:nodeptr;
                         var item:word;
                         var searchstack:np_stack):nodeptr;
(* Locate the first node in the btree, and return the node *)

(* Ref. Section x.1.16 *)

var
   c:nodeptr;
begin

     c:=idx;

     (* While we're in a valid node, and we can go down to the left
        tree in the current node, push the current position onto the
        stack, and move down *)
     while assigned(c) and assigned(c^.branch[0]) do
     begin
          np_push_stack(searchstack,c,1);
          c:=c^.branch[0];
     end;

     (* Return the node, and the position of the first key (1) *)
     item:=1;
     btree_findfirst:=c;
end;



function btree_findnext(    idx:nodeptr;
                        var item:word;
                        var searchstack:np_stack):nodeptr;
(* Locate the next node in the btree *)

(* Ref. Section x.1.17 *)

var
   c:nodeptr;
   pos:word;
begin
     c:=idx;

     (* If there is a key that is available in the current node
        then simply increment the counter *)
     if item<idx^.count then   {We're still within the current node}
     begin
          (* If the key has a tree beneath it, then get the left most
             node of that *)
          if assigned(idx^.branch[item]) then {Tree to traverse}
          begin
               np_push_stack(searchstack,c,item+1); {Push current pos}
               c:=btree_findfirst(idx^.branch[item],item,searchstack);
                                  {Get the first item in the sub-tree}
          end
             else
                 inc(item)
     end
     else
         (* We've exhausted the current node *)
     begin

          (* If the stack isn't empty, then take a node off of that
             and set the key pointer accordingly *)
         if not np_stack_empty(searchstack) then
         begin
              c:=np_pop_stack(searchstack,pos);
              item:=pos;
         end
         else
             (* No more nodes in current node, and stack exhausted *)
         if item=idx^.count then
         begin
               np_push_stack(searchstack,c,item+1); {Push current pos}
               c:=btree_findfirst(idx^.branch[item],item,searchstack);
         end
     end;

     (* We've reached the end of the search *)
     if (item>c^.count) then
     begin
          (* Return nil *)
          btree_findnext:=nil;
          item:=0;

          (* Flush the search stack *)
          np_flush_Stack(searchstack);
     end
     else
         (* Return the node *)
         btree_findnext:=c
end;



procedure btree_destroy(var idx:nodeptr);
(* Destroy the entire b-tree structure to recover memory *)

(* Ref. Section x.1.18 *)

var
   np:np_stack;
   c,item:word;
begin
     (* Create a stack *)
     np:=np_create_stack;

     (* Whilst we haven't finished the traversal *)
     while assigned(idx) do
     begin

          c:=0;
          (* Get all sub-trees of the current node *)
          while (c<=idx^.count) do
          begin
               (* If it is a valid sub-tree, then push it onto the
                  stack *)
               if assigned(idx^.branch[c]) then
                  np_push_Stack(np,idx^.branch[c],item);
               inc(c);
          end;

          (* Dispose of the node *)
          dispose(idx);

          (* If the stack is not empty get the next node,
             otherwise return a nil, which will terminate the
             search *)
          if not np_stack_empty(np) then
             idx:=np_pop_stack(np,item)
          else
              idx:=nil;
     end;

     (* Flush out the stack *)
     np_flush_stack(np);
end;

procedure build_index(    cidx:index_struct_ptr;
                          db:database;
                          rel:string);
(* Builds index *)

(* Ref. Section x.1.19 *)

var
   crel:relation;
   ct:tuple_Attributes;
   t:file;
   spos,epos:word;
begin
     (* Locate the relation *)
     crel:=relation_find(db,rel);

     writeln_stdout('--- Building Index for '+rel+' ---');

     spos:=0;

     (* If we have a valid relation *)
     if assigned(crel) then
     begin
          (* Read the first tuple from the disk *)
          ct:=readfirst_tuple_b(crel,t,crel^.nofields,spos,epos);

          (* Whilst we have a valid tuple *)
          while assigned(ct) do
          begin
               (* Print the tuple *)
               tuple_print(ct,false);

               (* Insert the tuple into index *)
               insert_tuple(ct,cidx,spos);

               (* Set the 'start' position from which to read the
                  next tuple as the 'end' position from the last read *)
               spos:=epos;

               (* Read the next tuple *)
               readnext_tuple_b(ct,t,crel^.nofields,spos,epos);
          end;
     end;

end;


procedure rebuild_indexes(    idx:index_struct_hdr_ptr;
                              db:database);
(* Rebuilds all indexes the *long* way (load all relations into memory
   and insert into indexes).

   Quick and dirty method of testing indexes
*)

(* Ref. Section x.1.20 *)

var
   cidx:index_struct_ptr;
   crel:relation;
   ct:tuple_Attributes;
   t:file;
   spos,epos:word;
begin
     (* Locate the first relation *)
     crel:=relation_findfirst(db);

     spos:=0;
     writeln('--- ReBuilding Indexes ---');

     (* While we have a valid relation *)
     while assigned(crel) do
     begin
          (* Read the first tuple from the disk *)
          ct:=readfirst_tuple_b(crel,t,crel^.nofields,spos,epos);

          (* Whilst we have a valid tuple *)
          while assigned(ct) do
          begin
               (* Print the tuple *)
               tuple_print(ct,false); writeln;

               (* Insert the tuple into all of the relations indexes *)
               insert_tuple_idx(ct,crel,spos,idx);

               (* Set the 'start' position from which to read the
                  next tuple as the 'end' position from the last read *)
               spos:=epos;

               (* Read the next tuple *)
               readnext_tuple_b(ct,t,crel^.nofields,spos,epos);
          end;

          (* Get the next relation *)
          crel:=relation_findnext(crel);
     end;

     (* Close the file that was in use for the reads *)
     close(t);
end;




procedure st_indexes(var idx:index_Struct_hdr_ptr;
                     var db:database);
(* Store the indexes, so that they may be retrieved at a later date
   thus avoiding the necessity to rebuild the entire structure through
   reading every tuple *)

(* Ref. Section x.1.21 *)

var
   cidx:index_struct_ptr;
   cidptr:nodeptr;
   crel:relation;
   ct:tuple_Attributes;
   f:file of keytype;
   fdes:text;
   spos,epos,rcount,item,c:word; (* rcount used to identify different
                                    indexes for the same relation *)
   scontrol:np_stack;
begin
     rcount:=1;
     spos:=0;

     (* Get the first relation *)
     crel:=relation_findfirst(db);
     writeln('--- Storing Indexes ---');

     (* Whilst we have a relation to deal with *)
     while assigned(crel) do
     begin
          (* Find the first index for the given relation *)
          cidx:=idx_findfirst(idx,relation_name(crel));

          (* Create a stack for traversal *)
          scontrol:=np_create_stack;

          (* Whilst we have a valid index to store *)
          while assigned(cidx) do
          begin

               (* Write out the primary and secondary keys to
                  store the description of the index *)
               assign(fdes,DIR_MASK+DATA_DIR+RELATION_DIR+'\'+relation_name(crel)+'.D'+pad_zeros(rcount));
               rewrite(fdes);
               writeln(fdes,relation_name(crel));
               for c:=1 to MAXKEYS do
               begin
                    writeln(fdes,cidx^.index_data.primary[c]);
                    writeln(fdes,cidx^.index_data.secondary[c]);
               end;
               close(fdes);


               (* Write out the index information *)
               assign(f,DIR_MASK+DATA_DIR+RELATION_DIR+'\'+relation_name(crel)+'.I'+pad_zeros(rcount));
               rewrite(f);

               (* Get the first node *)
               cidptr:=btree_findfirst(cidx^.index_data.index_struct,item,scontrol);

               (* Whilst the node is valid *)
               while assigned(cidptr) do
               begin
                    (* Store only the key information, ie. address
                       and key information *)
                    write(f,cidptr^.key[item]);

                    (* Locate the next entry in the btree *)
                    cidptr:=btree_findnext(cidptr,item,scontrol);
               end;
               (* Close the file *)
               close(f);

               (* Locate the next index *)
               cidx:=idx_findnext(cidx);

               (* Dispose of the stack in its entirety *)
               np_flush_stack(scontrol);

               (* Increment the identifier, so we can have multiple
                  indexes in the given relation *)
               inc(rcount);
          end;

          (* We're starting with a new relation, so reset the
             rcount variable *)
          rcount:=1;

          (* Retrieve the next relation *)
          crel:=relation_findnext(crel);
     end;
end;



procedure rd_indexes(    db:database);

(* Ref. Section x.1.22 *)

var
   cidx:index_struct_ptr;
   cidptr:nodeptr;
   crel:relation;
   ct:tuple_Attributes;
   f:file of keytype;
   fdes:text;
   ext,rname:string;
   spos,epos,rcount,item,c:word;
   scontrol:np_stack;
   dirinfo,dirinfodes:searchrec;
   kt:keytype;
   pi,si:keyarray;
begin
     (* Create a master index structure which is the default
        structure for most operations, some need may arise for
        a seperate index structure, this saves time *)
     idx_master:=create_idx_struct;

     rcount:=1;
     spos:=0;

     (* Get the first relation *)
     crel:=relation_findfirst(db);
     writeln_stdout('--- Retrieving Indexes ---');

     (* While we have a valid relation *)
     while assigned(crel) do
     begin
          (* Get the first file *)
          findfirst(get_basedir(db)+RELATION_DIR+'\'+relation_name(crel)+'.D*',anyfile,dirinfodes);

          (* While a file exists *)
          while (doserror=0) do
          begin
               (* Open the descriptor file *)
               assign(fdes,get_basedir(db)+RELATION_DIR+'\'+dirinfodes.name);
               reset(fdes);

               c:=0;

               (* Load the relation name *)
               readln(fdes,rname);

               (* Load the primary and secondary key descriptors *)
               while not eof(fdes) do
               begin
                    inc(c);
                    readln(fdes,pi[c]);
                    readln(fdes,si[c]);
               end;
               (* Close the file *)
               close(fdes);

               (* Determine the name of the file containing the index
                  information, ie. same name, but with .Ixx extension *)
               ext:=dirinfodes.name;
               cut_token(ext,'.');
               ext[1]:='I';
               ext:=concat('.',ext);

               (* Get the file and open it *)
               findfirst(get_basedir(db)+RELATION_DIR+'\'+relation_name(crel)+ext,anyfile,dirinfo);
               if (doserror=0) then
               begin
                    assign(f,get_basedir(db)+RELATION_DIR+'\'+dirinfo.name);
                    reset(f);
                    cidptr:=nil;
                    (* While we're not at the end of the file *)
                    while not eof(f) do
                    begin
                         (* Read the information *)
                         read(f,kt);

                         (* Insert the key into the b-tree *)
                         keyinsert(kt,cidptr);
                    end;

                    (* Add the index to the index reference structure *)
                    add_index_hdr(idx_master,relation_name(crel),pi,si,cidptr);

                    (* Close the file *)
                    close(f);
               end
                  else
                      (* We'd opened the descriptor file, but
                         the file that contains the important information
                         is causing an error... abort *)
                      fatal_error(1,get_basedir(db)+RELATION_DIR+'\'+dirinfo.name);

               findnext(dirinfodes);
          end;

          (* Locate the next relation *)
          crel:=relation_findnext(crel)

     end
end;



function get_index(    idx:index_struct_hdr_ptr;
                       rel:relation;
                       pk:string):index_struct_ptr;
(* Locate an index for a given key within a relation *)

(* Ref. Section x.1.23 *)

var
   cisp:index_struct_ptr;
   done:boolean;
begin
     (* Locate the first index within the index structure *)
     cisp:=idx_findfirst(idx,relation_name(rel));

     done:=false;
     (* Whilst there is an index node to be found, and we haven't completed
        the search successfully *)
     while assigned(cisp) and not done do
     begin
          (* If the current index is the node we're searching for *)
          if (cisp^.index_Data.primary[1]=pk) then
             (* Set done to true, to terminate the search *)
             done:=true
          else
              (* Otherwise, get the next index *)
              cisp:=idx_findnext(cisp)
     end;

     (* Return what we've found *)
     get_index:=cisp;
end;

procedure idx_Destroy(    db:database);

(* Disposes of an entire master index structure and indexes contained within *)

(* Ref. Section x.1.24 *)

var
   isp:index_struct_ptr;
   crel:relation;
begin
     (* Report to the user what's going on *)
     writeln_stdout('--- Disposing Indexes ---');

     (* Get the first relation *)
     crel:=relation_findfirst(db);

     (* Whilst the relation is valid *)
     while assigned(crel) do
     begin

          (* Get the first index for the current relation *)
          isp:=idx_findfirst(idx_master,relation_name(crel));

          (* Whilst the index is valid *)
          while assigned(isp) do
          begin
               (* Destroy the btree *)
               btree_destroy(isp^.index_data.index_struct);

               (* Get the next index *)
               isp:=idx_findnext(isp);
          end;

          (* Get the next relation *)
          crel:=relation_findnext(crel);
     end;

     (* Dispose of the master index structure *)
     dispose_idx_struct(idx_master);
end;

end.