unit hashing;

(*
Unit: hashing
Initial Date: 16/01/1995
Description: Hashing operators
Notes: Hashing algorithms extensively modified from examples in:
               'Data Structures with Abstract Data Types and Pascal'
               (2nd Edition) by Daniel F.Stubbs and Neil W.Webre;
               Published by Brooks/Cole Publishing Company, 1989,1985.
Revision History:
	 Placed under GNU General Public License: 22/06/1996
         Hash Table Load & Save added: 09/12/1995
         Commenting: 01/02/1995
         Chaining Modifications: 31/01/1995
         Initial Development: 16/01/1995
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)

interface

{$O+}           (* The {$O+} and overlay usage enables overlays
                   to be enabled at some later date if necessary.
                   Each unit must make use of this compiler directive
                   and unit if overlaying is to be supported *)

uses dtypes;

procedure Hashing_Create(var HT:HashTable);
procedure Hashing_Insert(var HT:HashTable;
                             e:hash_keytype;
                             hk:word);
procedure Hashing_Retrieve(    HT:HashTable;
                            tkey:hash_KeyType;
                        var e:hash_keytype;
                        var retrieved:boolean);
procedure Hashing_Terminate(var HT:HashTable);
procedure Hashing_Save(    HT:HashTable; filen:string);
procedure Hashing_Load(var HT:HashTable; filen:string);

implementation


uses utils,index;

type
    Position=0..TABLETOP;       (* The possible positions of elements in
                                   the hashing table *)
    StrSet=HashTable;            (* A hashtable used to implement a strset *)

    (* Structure for the read/writing of the hash tables *)
    nodestr=record
                  hash_key:0..TABLESIZE;
                  item:Hash_keytype;
                  pos:word;
    end;



function Hash(    k:hash_keytype):Position;
(* Return a hash key, using folding *)
var
   s:string;
   c,h:word;
begin
     h:=ord(k[1]);
     for c:=2 to length(k) do
         h:=((h*128)+ord(k[c])) mod TABLESIZE;
     Hash:=h;
end;




function ReHash(    p:Position):Position;
(* Rehash a value *)
begin
     ReHash:=(p+1) mod TABLESIZE;
end;




procedure Hashing_Insert(var HT:HashTable;
                             e:hash_keytype;
                             hk:word);
(* Insert a key into the specified hashing table *)
   (* If hk=REQ_CALC then calculate the hash key,
      otherwise use the value specified. Should only
      be specified when loading off disk, it will corrupt
      the hash table in ANY OTHER CIRCUMSTANCE *)
var
   posn:0..TABLETOP;
   nchn,pchnn,cchnn:chain;

       function chain_node_create(e:hash_keytype):chain;
       (* Create a chain node *)
       var
          chn:chain;
       begin
            (* Create the node *)
            new(chn);
            check_assign(chn,'hashing_insert/chain_node_create');

            (* Reset values *)
            chn^.next:=nil;
            chn^.elt:=e;

            (* Return the result *)
            chain_node_create:=chn;
       end;

begin
     (* This should save CPU time when the hash table is being
        loaded from disk - It will corrupt the hash table if
        used in any other circumstance *)

     (* If the hash key has not been provided *)
     if (hk=REQ_CALC) then
        (* Calculate the hash key *)
        posn:=Hash(e)
     else
         (* Else use the hash key provided *)
         posn:=hk;

     (* Create a blank node to insert *)
     nchn:=chain_node_create(e);

     (* If there is no chain node at the head *)
     if not assigned(HT^[posn].hd) then
        HT^[posn].hd:=nchn
     else
         begin
              (* Otherwise, start searching the list *)
              pchnn:=nil;
              cchnn:=HT^[posn].hd;

              (* Locate the node *after* the destination *)
              while assigned(cchnn) and (e>cchnn^.elt) do
              begin
                   pchnn:=cchnn;
                   cchnn:=cchnn^.next;
              end;

              (* If end of the list has been reached, add the new node
                 to the end *)
              if not assigned(cchnn) then
                 pchnn^.next:=nchn
              else

              (* If the node to be inserted should go before the first node,
                 adjust the header information *)
              if cchnn^.elt=HT^[posn].hd^.elt then
              begin
                   nchn^.next:=HT^[posn].hd;
                   HT^[posn].hd:=nchn;
              end
              else
              (* Otherwise insert the new node before the current node *)
              if (e<cchnn^.elt) then
              begin
                   nchn^.next:=cchnn;
                   pchnn^.next:=nchn;
              end
              else
              (* Unless its already in the list, in which case we may
                 as well just dispose of the memory and report an
                 error *)
              if (e=cchnn^.elt) then
              begin
                   writeln('Duplicate entry!');
                   dispose(nchn);
              end;
         end;
end;



procedure FindPosn(    HT:HashTable;
                       tkey:hash_KeyType;
                   var posn:Position;
                   var chn:chain;
                   var found:boolean);
(* Results         : If HT contains an element with key value tkey then
                     found is true and posn is that element's position;
                     otherwise found is false *)

(* Locates a node in the specified hash table, and returns the position
   and boolean value for the search result *)

var
   home:Position;
   cchnn:chain;
   finished:boolean;
begin
     found:=false;

     (* Calculate the hash key for the node if it were being inserted *)
     home:=Hash(tkey);
     posn:=home;
     finished:=false;
     cchnn:=HT^[posn].hd;

     (* Locate the chain node on the specified hash table *)
     while not(finished) do
           if not (assigned(cchnn)) or (cchnn^.elt=tkey) then
              finished:=true
           else
               cchnn:=cchnn^.next;

     (* Return TRUE or false depending on whether it was found or not,
        and return the node itself, just incase *)
     found:=(assigned(cchnn));
     chn:=cchnn;
{     repeat
           with HT^[posn] do
                if (HT^[posn].elt=tkey) and (status=occupied) then
                   found:=true
                else
                if status<>empty then
                   posn:=ReHash(posn);
     until (found) or (posn=home) or (HT^[posn].status=empty)}
end;




procedure Delete(var HT:HashTable;
                     tkey:hash_KeyType;
                 var del:boolean);
var
   posn:Position;
   found:Boolean;
   chn:chain;
begin
{     del:=false;
     FindPosn(HT,tkey,posn,chn,found);
     if found then
     begin

          del:=true;
     end}
     writeln(#7,#7,'Unexecpected code executed - hash/delete');
     halt;
end;



procedure Update(var HT:HashTable;
                     e:hash_keytype;
                 var updated:boolean);
(* Locate a given node in the hash table, and update its contents to
   those specified *)
var
   posn:position;
   found:boolean;
   chn:chain;
begin
     updated:=false;

     (* Locate the node *)
     FindPosn(HT,e,posn,chn,found);

     (* If the search was successful *)
     if found then
     begin
          (* Update the elements value *)
          chn^.elt:=e;

          (* Return true for updating *)
          updated:=true;
     end;
end;



procedure Hashing_Retrieve(    HT:HashTable;
                               tkey:hash_KeyType;
                           var e:hash_keytype;
                           var retrieved:boolean);

(* Retrieve a given node from the hash table *)

var
   posn:Position;
   found:boolean;
   chn:chain;
begin
     retrieved:=false;

     (* Locate the given node in the hash table *)
     FindPosn(HT,tkey,posn,chn,found);

     (* If the node exists in the node *)
     if found then
     begin
          (* Return the element's contents *)
          e:=chn^.elt;

          (* And update the success flag *)
          retrieved:=true;
     end
end;

{procedure Traverse(    HT:HashTable);
var
   k:integer;
begin
     for k:=0 to TABLETOP do
         with HT^[k] do
              if status=occupied then
                 Proc(elt)       (* Proc is a user-supplied procedure. *)
end;
}



procedure Hashing_Create(var HT:HashTable);
(* Create a new hashing table and return the pointer *)
var
   k:integer;
begin
     (* Create the hash table *)
     new(HT);
     check_assign(HT,'hashing.create');

     (* Initialise its contents *)
     for k:=0 to TABLESIZE-1 do
         HT^[k].hd:=nil;
end;




procedure Hashing_Terminate(var HT:HashTable);
(* Dispose of the chains associated with a hashing table, and
   the hashing table itself *)

var
   k:integer;
   ochn,cchnn:chain;
begin

     (* Move through the hashing table *)
     for k:=0 to TABLESIZE-1 do
     begin
          cchnn:=HT^[k].hd;

          (* Whilst valid chain node *)
          while assigned(cchnn) do
          begin
               (* Mark it *)
               ochn:=cchnn;

               (* Go to the next chain element *)
               cchnn:=cchnn^.next;

               (* Dispose of the previous node *)
               dispose(ochn);
          end;

     end;

     (* Dispose of the hashing table itself *)
     dispose(HT);
end;

procedure Hashing_Save(    HT:HashTable; filen:string);
var
   counter:word;
   nfile:file of nodestr;
   node:nodestr;
   curr_chain_node:chain;
begin
     (* Assign the file as a hashing file *)
     assign(nfile,filen);

     (* Reset it for writing *)
     rewrite(nfile);

     (* Loop through the entire hash table *)
     for counter:=0 to TABLESIZE-1 do
     begin
          (* Get the hash key into the write record *)
          node.hash_key:=counter;

          (* Get the head chain node *)
          curr_chain_node:=HT^[counter].hd;
          node.pos:=0;

          (* Whilst the chain node is valid *)
          while assigned(curr_chain_node) do
          begin
               (* Get the data stored *)
               node.item:=curr_chain_node^.elt;

               (* Increment the position *)
               inc(node.pos);

               (* Write the node to the file *)
               write(nfile,node);

               (* Get the next node *)
               curr_chain_node:=curr_chain_node^.next;
          end;
     end;

     (* Close the output file *)
     close(nfile);
end;

procedure Hashing_Load(var HT:HashTable; filen:string);
var
   nfile:file of nodestr;
   node:nodestr;
   ior:integer;
begin
     Hashing_Create(HT);
     assign(nfile,filen);
     {$I-}
     reset(nfile);
     {I+}
     ior:=IOResult;
     if (ior<>0) then
        nonfatal_error(ERROR_FILE_OPENING,'File: '+ filen)
     else
     begin
         while not eof(nfile) do
         begin
              read(nfile,node);
              Hashing_Insert(HT,node.item,node.hash_key);
         end;
         close(nfile);
     end;
end;

end.