unit fields;
(*
Unit: Fields
Description: Field maintenance and retrieval operations
References: References to the documentation is to the file fields.doc
            and section numbers are given as relative to this document.
Revision History:
         13/03/1995 - Extended field_findnext and field_find to dispose of
                      field ptr if appropriate. Previously just assigned
                      ptr to nil
         09/03/1995 - Documentated (Fields.doc)
         30/01/1995 - Modified field_findnext to stop creating a new
                      pointer each and every time - it was becoming lost
                      and not recovered. Caused an error in readnext_tuple
                      because all flds now the same, though different.
                      Rectified by param TRUE for new node, FALSE if same
         18/12/1994 - Post pointer create check and polite shutdown

To-do: Document: procedure flds_print_rel(    rel:relation);
       Document: field_findnext (Changed requirements)
       Document: field_find (Disposes of mem if passed node)
*)

(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)
interface

{$O+}

uses relations,tuples,dtypes,utils;


procedure field_create(    rel:relation;
                           field_name:string;
                           data_type:data_types);
function field_findfirst(    rel:relation):field;
function field_findnext(    rel:relation;
                        var fld:field;
                            newnode,node_dispose:boolean):field;
function field_type(     fld:field):data_types;
function field_name(    fld:field):string;
function field_find(    rel:relation;
                       name:string):field;
procedure field_print(fld:field);
procedure flds_print(    rel:relation);
procedure flds_print_rel(    rel:relation);
procedure fields_print(    ta:tuple_attributes);
procedure field_display(rel:relation);
procedure field_dispose(var fld:field);
procedure populate_fields(    rel:relation;
                         var ta:tuple_attributes;
                         var nofields:word);

implementation

procedure field_create(    rel:relation;
                           field_name:string;
                           data_type:data_types);
(* Create a field for the relation *)
(* Ref. Section x.1.3 *)

begin
     (* Write the field name to the file pointed to by the relation structure *)
     writeln(rel^.fieldptr,field_name);
     case data_type of
          string_t:writeln(rel^.fieldptr,'STRING');
          int:writeln(rel^.fieldptr,'INTEGER');
     end;
     inc(rel^.nofields); (* This should be calculated as well when
                            a field is read in *)
end;

procedure field_populate(    rel:relation;
                             fld:field);
(* Ref. Section x.1.6 *)
var
   data:string;
begin
     readln(rel^.fieldptr,fld^.name);
     readln(rel^.fieldptr,data);
         if data=STRING_TXT then
            fld^.data_type:=string_t
         else
         if data=INTEGER_TXT then
            fld^.data_type:=int
{         else
         if data=REAL_TXT then
            fld^.data_type:=real;}
end;

function field_findfirst(    rel:relation):field;
(* Ref. Section x.1.4 *)
var
   fld:field;
   data:string;
   s:string;
   I:integer;
begin
{     check_Avail_Mem(sizeof(fld));}
     new(fld);
     check_Assign(fld,'field.field_findfirst');
     {$I-}
     relation_reset_fldfile(rel);
     I:=IOResult;
     if I<>0 then
     begin
          system.str(I,S);
          report('IOResult Error #'+s+' occured when attempting to open '+rel^.fieldname);
     end;
     {$I+}
     if I<>0 then
        fatal_error(2,'File: '+DIR_MASK+DATA_DIR+RELATION_DIR+rel^.fieldname);
     field_populate(rel,fld);
     field_findfirst:=fld;
end;

procedure field_dispose(var fld:field);
(* Ref. Section x.1.11 *)
begin
     if assigned(fld) then
     begin
          dispose(fld);
          fld:=nil;
     end;
end;

function field_findnext(    rel:relation;
                        var fld:field;
                            newnode,node_dispose:boolean):field;
(* Ref. Section x.1.5 *)
var
   nfld:field;
begin
     if not eof(rel^.fieldptr) then
     begin
          if newnode then
          begin
               (* Create a new node for each item *)
               new(nfld);
               check_assign(nfld,'(field_findnext)');
               field_populate(rel,nfld);
               fld:=nfld;
          end
          else
              (* Populate current node *)
              field_populate(rel,fld);
     end
     else
     begin
          relation_close_fldfile(rel);
          if node_dispose then
             dispose(fld);
          fld:=nil;
     end;
     field_findnext:=fld;
end;

function field_name(    fld:field):string;
(* Ref. Section x.1.2 *)
begin
     field_name:=fld^.name;
end;

function field_type(     fld:field):data_types;
(* Ref. Section x.1.1 *)
begin
     field_type:=fld^.data_type;
end;


procedure field_print(fld:field);
(* Print the fields descriptions *)
(* Ref. Section x.2.1 *)
begin
     (* Check its valid! *)
     if assigned(fld) then
     begin
          write_stdout(fld^.name+'   (');
          case field_type(fld) of
               string_t:writeln_stdout('STRING)');
               int:writeln_stdout('NUMBER)');
          end;
     end;
end;


(* Display all of the fields in the relation *)

procedure field_display(rel:relation);
(* Ref. Section x.1.10 *)
var
   currentFld:field;
begin
     writeln('Name  (Type)');

     (* Get the first field *)
     currentFld:=field_findfirst(rel);
     while assigned(currentFld) do
     begin

          (* Print the current field *)
          field_print(currentFld);

          (* Find the next field *)
          currentFld:=field_findnext(rel,currentFld,FALSE,TRUE);
     end;
end;

(* Locate a field, and return ptr to it given its name *)

function field_find(    rel:relation;
                  name:string):field;
(* Ref. Section x.1.7 *)
var
   currentFld:field;
begin
     name:=upstring(name);
     (* Locate the first field *)
     currentFld:=field_findfirst(rel);

     (* Get the next field until found or end of list *)
     while assigned(currentFld) and (field_name(currentFld)<>name) do
           field_findnext(rel,currentFld,FALSE,TRUE);

     if (assigned(currentFld)) and (field_name(currentFld)=name) then
        field_find:=currentFld
     else
     begin
         (* Nothing found... *)
         if assigned(currentFld) then
            dispose(currentFld);
         field_find:=nil;
         searchError:=SEARCH_UNSUCCESSFUL
     end;
end;




procedure flds_print(    rel:relation);
(* Prints the field descriptions given a relation *)
(* Ref. Section x.1.8 *)
var
   cf:field;
begin
     (* Load the first field *)
     if assigned(rel) then
     begin
          cf:=field_findfirst(rel);

          (* Whilst valid field *)
          while assigned(cf) do
          begin
               (* Print the field name *)
               field_print(cf);

               (* Get the next field *)
               cf:=field_findnext(rel,cf,FALSE,TRUE);
          end;
     end
     else
         nonfatal_error(ERROR_DISPLAY_NONEX_REL,'');
end;

procedure flds_print_rel(    rel:relation);
(* Prints the field descriptions given a relation *)
(* Ref. Section x.1.8 *)
var
   cf:field;
   c,c2,nf:byte;
begin
     (* Load the first field *)
     cf:=field_findfirst(rel);

     nf:=0;

     (* Whilst valid field *)
     while assigned(cf) do
     begin
          inc(nf);
          (* Print the field name *)
          write_stdout(field_name(cf));

          for c:=length(field_name(cf)) to MAX_SIZE_DATUM_STRING do
              write_stdout(' ');

          (* Get the next field *)
          cf:=field_findnext(rel,cf,FALSE,TRUE);

     end;
     writeln_stdout_nl;
     for c:=1 to nf do
         for c2:=1 to MAX_SIZE_DATUM_STRING do
             write_stdout('-');
     writeln_stdout_nl;
end;




procedure fields_print(    ta:tuple_attributes);
(* Prints the fields given a tuple *)
(* Ref. Section x.1.1.9 *)
var
   count,c:word;
begin
     count:=1;

     (* Whilst the indicator is at a valid field ptr *)
     while (assigned(ta)) and (assigned(ta^.tdatum[count]^.fieldptr)) do
     begin
          (* Print the name to the standard outout *)
          write_stdout(ta^.tdatum[count]^.fieldptr^.name);

          (* Pad the line a bit more *)
          for c:=length(ta^.tdatum[count]^.fieldptr^.name) to 25 do
              write_stdout(' ');

          (* Increment the counter *)
          inc(count);
     end;

     (* Start a new line *)
     writeln_stdout_nl;

     (* Print a seperator *)
     for c:=1 to 79 do
         write_stdout('-');

     (* Start a new line *)
     writeln_stdout_nl;
end;




procedure populate_fields(    rel:relation;
                          var ta:tuple_attributes;
                          var nofields:word);
(* Ref. Section x.1.12 *)
var
   fld:field;
begin
     fld:=field_findfirst(rel);
     nofields:=0;
     while (assigned(fld)) do
     begin
          inc(nofields);
          ta^.tdatum[nofields]^.fieldptr:=fld;
          ta^.tdatum[nofields]^.relationPtr:=rel;
          fld:=field_findnext(rel,fld,TRUE,FALSE);
     end;
end;

end.