unit dbase;

(*
Unit: Dbase
Description: Provides operations to create & destroy master structure.
References: References to the documentation is to the file fields.doc
            and section numbers are given as relative to this document.
Revision History:
         18/12/1994 - Post pointer create check and polite shutdown
         26/03/1995 - Extensive commenting.
         27/03/1995 - Documented (dbase.doc)
         10/12/1995 - Implemented support for multiple databases.
	 22/05/1996 - Put under GNU General Public License
*)
(*   LEAP - An extensible relational algebra processor/RDBMS (v0.10.1)
 *   Copyright (C) 1996 Richard Leyton
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)

interface

{$O+}

uses dtypes;

function create_db(    name:string):database;
procedure destroy_db(var db:database);
procedure create_new_db(name,info:string);
function open_db(    name:string):database;
function get_basedir(    db:database):string;
function get_dbname(    db:database):string;

implementation

uses utils,tuples,relations;

function create_db(    name:string):database;
(* Create a database with the specified name *)

(* Ref. Section x.1.1 *)

var
   tempDb:database;
begin
     (* Create the database pointer *)
     new(tempDb);
     check_assign(tempDb,'dbase.create_db');

     (* Populate the database information *)
     tempDb^.name:=name;
     tempDb^.firstRelation:=nil;
     tempDb^.lastRelation:=nil;

     (* Return the database pointer *)
     create_db:=tempDb;
end;

procedure destroy_db(var db:database);
(* Destroy the specified database *)

(* Ref. Section x.1.2 *)
begin
     (* Release the memory *)
     dispose(db);

     (* Reset the pointer *)
     db:=nil;
end;


procedure create_new_db(name,info:string);
var
   ddrel,trel:relation;
   t:text;
   ok:boolean;
begin
     writeln_stdout('Creating database '+name+'...');

     ok:=true;

     {$I-}
     mkdir(DIR_MASK+DATABASE_DIR+'\'+name);
     ok:=ok and (IOResult=0);
     mkdir(DIR_MASK+DATABASE_DIR+'\'+name+ERROR_DIR);
     ok:=ok and (IOResult=0);
     mkdir(DIR_MASK+DATABASE_DIR+'\'+name+REPORT_DIR);
     ok:=ok and (IOResult=0);
     mkdir(DIR_MASK+DATABASE_DIR+'\'+name+DATA_DIR);
     ok:=ok and (IOResult=0);
     mkdir(DIR_MASK+DATABASE_DIR+'\'+name+SCRIPT_DIR);
     ok:=ok and (IOResult=0);
     {I+}

     if ok then
     begin
          upstring(name);
          ddrel:=relation_find(master_db,LEAP_DATABASES);
          trel:=tuple_flex_add(ddrel,name+' '+name,' ');
          if assigned(trel) then
          begin
               assign(t,DIR_MASK+DATABASE_DIR+'\'+name+DB_DESCRIPTOR_FILE);
               rewrite(t);
               strip_leading_spaces(info);
               writeln(t,info);
               close(t);
          end
          else
              nonfatal_error(ERROR_CANNOT_CREATE_DB,name);
     end
     else
         nonfatal_error(ERROR_CANNOT_CREATE_DB,name);

end;

function open_db(name:string):database;
var
   db_ptr:database;
begin
     writeln_stdout('Opening '+name+' database...');
     if (name=MASTER_DB_NAME) then
     begin
          writeln_stdout('This is the master database...');
          db_ptr:=create_db('master');

          db_ptr^.base_dir:=DIR_MASK+DATABASE_DIR+DATABASE_MASTER_DIR;

          (* We might want to check for the existance
             of certain relations to make sure everything
             is ok. Wouldn't want to get some data, and
             find the relation has gone walk about... *)
     end
     else
     begin
          writeln_stdout('This is a user ['+name+'] database...');
          db_ptr:=create_db(name);

          (* When master database is setup correctly, we should
             perform a select on "LEAPDATABASES" to get the directory
             name from the data dictionary *)

          (* For now, however, this will do *)
          db_ptr^.base_dir:=DIR_MASK+DATABASE_DIR+'\'+name;
     end;

     (* Return the "new" database *)
     open_db:=db_ptr;
end;

function get_basedir(    db:database):string;
(* Return the base directory for the specified database *)
begin
     get_basedir:=db^.base_dir;
end;

function get_dbname(    db:database):string;
(* Return the database name *)
begin
     get_dbname:=upstring(db^.name);
end;

end.