/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tprogram.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TProgram class.
*
* $Id: tprogram.cpp 1.2 1994/03/09 11:50:43 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tprogram.hpp"
#include "tmouse.hpp"
#include "msgbox.hpp"
#include <stdlib.h>
#include <iostream.h>

/*---------------------------- Global Variables ---------------------------*/

TProgram *TProgram::application = NULL;
TStatusLine	*TProgram::statusLine = NULL;
TMenuBar *TProgram::menuBar = NULL;
TDeskTop *TProgram::deskTop = NULL;

/*----------------------------- Implementation ----------------------------*/

TProgram::TProgram(char *pathToDriver)
	: TGroup(TRect(0,0,0,0)), pathToDriver(pathToDriver), systemPal(NULL),
	  TProgInit(TProgram::initStatusLine,
				TProgram::initMenuBar,
				TProgram::initDeskTop,
				TProgram::initVideoMode)
/****************************************************************************
*
* Function:		TProgram::TProgram
* Parameters:	pathToDriver	- Path to the graphics driver files
*				systemFont		- Font to use as system font
*
* Description:	Constructor for the TProgram class. Attempts to initialise
*				the graphics subsystem, and setup for normal operation.
*
****************************************************************************/
{
	initGraphics();
	state = sfVisible | sfFocused | sfActive | sfModal | sfExposed;
	modalState = 0;
	globalRepaint = false;
	pending.what = evNothing;

	// Change the system font depending on the video mode that has been
	// selected. We always use fonts from the same family, but different
	// sizes for different resolution.

	int 		maxy = MGL_sizey()+1;
	FontRecord	fontRec(fmHelvetica);

	if (maxy < 350)
		fontRec.sizey = 11;
	else if (maxy <= 350)
		fontRec.sizey = 13;
	else if (maxy <= 400)
		fontRec.sizey = 13;
	else if (maxy <= 480)
		fontRec.sizey = 15;
	else if (maxy <= 600)
		fontRec.sizey = 17;
	else if (maxy <= 768)
		fontRec.sizey = 22;
	else fontRec.sizey = 29;
	if (!fontManager.setSystemFont(fontRec)) {
		MGL_exit();
		cout << "Unable to load system font: "
			 << (char*)MGL_errorMsg(MGL_result());
		exit(1);
		}

	// Set the default system line width
	_MVIS_sysLineWidth = 1;
	if (maxy >= 768)
		_MVIS_sysLineWidth = 2;
	_MVIS_sysScrollBarWidth = fontRec.sizey;

	// Adjust the size of the deskTop given the sizes of the menu bar and
	// status line.

	TRect	extent;	getExtent(extent);
	TRect	deskExtent(extent);

	lock();
	if (createMenuBar && (menuBar = createMenuBar(extent)) != NULL) {
		insert(menuBar);
		menuBar->setState(sfActive,true);
		deskExtent.top() = menuBar->getBounds().botRight.y;
		}

/*	if (createStatusLine && (statusLine = createStatusLine(extent)) != NULL) {
		insert(statusLine);
		statusLine->setState(sfActive,true);
		deskExtent.bottom() = statusLine->getBounds().top();
		}*/

	if (createDeskTop && (deskTop = createDeskTop(deskExtent)) != NULL) {
		insert(deskTop);
		deskTop->setState(sfActive | sfFocused,true);
		}

	if (valid(cmValid))
		application = this;

	resetClip();
}

TProgram::~TProgram()
/****************************************************************************
*
* Function:		TProgram::~TProgram
*
* Description:	Destructor for the TProgram class
*
****************************************************************************/
{
	delete systemPal;		// Kill internal objects
/*	if (statusLine)
		delete statusLine;*/
	if (menuBar)
		delete menuBar;
	if (deskTop)
		delete deskTop;
	fontManager.shutDown();	// Shutdown the font manager
	MGL_exit();				// Shutdown the graphics system
}

bool TProgram::valid(ushort command)
/****************************************************************************
*
* Function:		TProgram::valid
* Parameters:	command	- Command ending the modal operation
* Returns:		True if the group is valid
*
* Description:	Checks to see that the standard objects such as the
*				status line, menu bar and desktop were created successfully,
*				and then check the items in the group.
*
****************************************************************************/
{
	if (command == cmValid) {
/*		if (createStatusLine && statusLine == NULL)
			return false;*/
		if (createMenuBar && menuBar == NULL)
			return false;
		if (createDeskTop && deskTop == NULL)
			return false;
		return true;
		}
	else if (command == cmQuit) {
		// The user is attempting to quit the program, so check to see if
		// this is valid and return true if it is.

		return exitProgram();
		}
	else return TGroup::valid(command);
}

TStatusLine *TProgram::initStatusLine(const TRect&)
/****************************************************************************
*
* Function:		TProgram::initStatusLine
* Parameters:	rect	- bounding rectangle for entire program view
* Returns:		Pointer to created status line.
*
* Description:	By default we have no status line
*
****************************************************************************/
{
	return NULL;
}

TMenuBar *TProgram::initMenuBar(const TRect& bounds)
/****************************************************************************
*
* Function:		TProgram::initMenuBar
* Parameters:	bounds	- bounding rectangle for entire program view
* Returns:		Pointer to created menu bar.
*
* Description:	Set up a very simple menu bar to allow the user to quit.
*
****************************************************************************/
{
	TMenu	*fileMenu = new TMenu();

	*fileMenu
		+ new TMenuItem("~Q~uit",cmQuit,HotKey(kbX,mdAlt),hcNoContext,"Alt+X");
	fileMenu->doneDefinition();

	TMenuBar	*menuBar = new TMenuBar(bounds);

	*menuBar
		+ new TSubMenuItem("~F~ile",fileMenu);
	menuBar->doneDefinition();

	return (TMenuBar*)validView(menuBar);
}

TDeskTop *TProgram::initDeskTop(const TRect& bounds)
/****************************************************************************
*
* Function:		TProgram::initDeskTop
* Parameters:	bounds	- bounding rectangle for entire program view
* Returns:		Pointer to created desktop.
*
* Description:	By default we draw a solid desktop.
*
****************************************************************************/
{
	return (TDeskTop*)validView(new TDeskTop(bounds,new TBackGround(bounds)));
}

void TProgram::initVideoMode(int&,int&)
/****************************************************************************
*
* Function:		TProgram::initVideoMode
* Parameters:	driver	- Detected graphics driver
*				mode	- Recommended graphics mode
*
* Description:	This does nothing, but can be used to change the default
*				graphics mode selected depending on the video driver
*				values.
*
****************************************************************************/
{
}

void TProgram::initSystemPalette()
/****************************************************************************
*
* Function:		TProgram::initSystemPalette
*
* Description:	Initialises the system palette, loading the default system
*				colors.
*
****************************************************************************/
{
	// Determine the screen size and resize the program view

	setBounds(TRect(0,0,MGL_sizex()+1,MGL_sizey()+1));

	// Allocate space for the palette, load the default palette, load
	// the default system colors and program it.

	if (systemPal != NULL)
		delete systemPal;

	palette *pal = systemPal = new palette[MGL_getPaletteSize()];
	MGL_getDefaultPalette(pal);

	if (graphDriver > grEGA) {
		// Set up the default VGA and SuperVGA palette

		pal[0].red = 0;		pal[0].green = 0;	pal[0].blue = 0;	// Black
		pal[1].red = 255;	pal[1].green = 255;	pal[1].blue = 255;	// White
		pal[2].red = 204;	pal[2].green = 204;	pal[2].blue = 204;	// LightGray
		pal[3].red = 128;	pal[3].green = 128;	pal[3].blue = 128;	// DarkGray
		pal[4].red = 0;		pal[4].green = 0;	pal[4].blue = 128;	// Blue
		pal[5].red = 0;		pal[5].green = 128;	pal[5].blue = 128;	// Jade
		pal[6].red = 255;	pal[6].green = 255;	pal[6].blue = 128;	// Yellow
		pal[7].red = 255;	pal[7].green = 255;	pal[7].blue = 204;	// LightYellow
		}
	else {
		// Set up the default EGA palette

		pal[0].red = 0;		pal[0].green = 0;	pal[0].blue = 0;	// Black
		pal[1].red = 192;	pal[1].green = 192;	pal[1].blue = 192;	// White
		pal[2].red = 128;	pal[2].green = 128;	pal[2].blue = 128;	// LightGray
		pal[3].red = 64;	pal[3].green = 64;	pal[3].blue = 64;	// DarkGray
		pal[4].red = 0;		pal[4].green = 0;	pal[4].blue = 128;	// Blue
		pal[5].red = 0;		pal[5].green = 128;	pal[5].blue = 128;	// Jade
		pal[6].red = 192;	pal[6].green = 192;	pal[6].blue = 128;	// Yellow
		pal[7].red = 192;	pal[7].green = 192;	pal[7].blue = 128;	// LightYellow
		}

	setSystemPalette();
	MGL_setCursorColor(MGL_realColor(scWhite));
}

void TProgram::initGraphics()
/****************************************************************************
*
* Function:		TProgram::initGraphics
*
* Description:	Attempts to initialise the graphics subsystem, exiting
*				with an error message if things did not go correctly.
*
****************************************************************************/
{
	MGL_detectGraph(&graphDriver,&graphMode);
	if (adjustVideoMode) adjustVideoMode(graphDriver,graphMode);

	_installMouse = true;				/* Install mouse handling stuff	*/
	MGL_init(&graphDriver,&graphMode,pathToDriver);

	int err = MGL_result();
	if (err != grOK) {
		cerr << "Graphics error: " << (char*)MGL_errorMsg(err) << endl;
		cerr << "Driver: " << MGL_driverName(graphDriver)
			 << ", Mode: " << MGL_modeName(graphMode) << endl;
		exit(EXIT_FAILURE);
		}

	initSystemPalette();
}

bool TProgram::exitProgram()
/****************************************************************************
*
* Function:		TProgram::exitProgram
* Returns:		True if program should be terminated.
*
* Description:	When the user attempts to terminate the program, this
*				method will be called to verify the command. By default
*				it just returns true, but you should override it to
*				prompt for saving files etc.
*
****************************************************************************/
{
	return true;
}

void TProgram::idle()
/****************************************************************************
*
* Function:		TProgram::idle
*
* Description:	Default idle routine. Updates the status line periodically
*				and posts messages when the command set is modified.
*
****************************************************************************/
{
	TRect	oldView;

	MGL_getViewport(&oldView);
	MGL_setViewport(bounds);
	TGroup::idle();
	MGL_setViewport(oldView);

	// Do all the above :-)
}

void TProgram::outOfMemory()
/****************************************************************************
*
* Function:		TProgram::outOfMemory
*
* Description:	Default out of memory routine - pops up a dialog box to
*				this effect.
*
****************************************************************************/
{
	messageBox("Out of memory: Could not complete the last operation.",
		mfError | mfOKButton | mfOKDefault);
}

void TProgram::handleEvent(TEvent& event,phaseType)
/****************************************************************************
*
* Function:		TProgram::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Main event handling routine for the program group. Note
*				before delegating the event down the chain of command, we
*				set the viewport to the bounds of the entire program
*				view.
*
****************************************************************************/
{
	// Check for Alt-1 to Alt-9 for window selection and post the
	// appropriate messages here

	TGroup::handleEvent(event);
	if (event.what == evCommand &&
			(event.message.command == cmQuit ||
			 event.message.command == cmRestart)) {
		endModal(event.message.command);
		clearEvent(event);
		}

	if (event.what == evKeyDown && event.key.keyCode == kbF10) {
		event.what = evCommand;
		event.message.command = cmMenu;
		event.message.infoPtr = NULL;
		putEvent(event);
		clearEvent(event);
		}
}

void TProgram::putEvent(TEvent& event)
/****************************************************************************
*
* Function:		TProgram::putEvent
* Parameters:	event	- Event to post
*
* Description:	Posts an event by storing it in the pending variable.
*
****************************************************************************/
{
	pending = event;
}

bool TProgram::getEvent(TEvent& event,ushort mask)
/****************************************************************************
*
* Function:		TProgram::getEvent
* Parameters:	event	- Place to store the event
* Returns:		True if an event was pending
*
* Description:	Gets the next pending event, looking in the event queue
*				if there are no pending messages.
*
****************************************************************************/
{
	// Check for repaint events first, and handle if one is found.

	if (eventQueue.getNext(event,evRepaint)) {
		doRepaint();
		clearEvent(event);
		return false;
		}

	if (mask == evRepaint)
		return false;

	if (pending.what != evNothing) {
		event = pending;
		pending.what = evNothing;
		}
	else {
		if (!eventQueue.getNext(event,mask))
			idle();
		}

	if (statusLine) {
		// Handle the special case of global status line events in here
		}

	return (event.what != evNothing);
}

bool TProgram::peekEvent(TEvent& event,ushort mask)
/****************************************************************************
*
* Function:		TProgram::peekEvent
* Parameters:	event	- Place to store the event
* Returns:		True if an event is pending
*
* Description:
*
****************************************************************************/
{
	if (pending.what != evNothing)
		event = pending;
	else
		eventQueue.peekNext(event,mask & ~evRepaint);
	return (event.what != evNothing);
}

void TProgram::eventError(TEvent&)
/****************************************************************************
*
* Function:		TProgram::eventError
* Parameters:	event	- Event causing the error
*
* Description:	End of the line for missing event routines. By default we
*				do nothing, but you can override this to bring up a
*				dialog box etc.
*
*				Maybe this could bring up a box by default.
*
****************************************************************************/
{
}

void TProgram::invalidRect(TRect& rect,bool global)
/****************************************************************************
*
* Function:		TProgram::invalidRect
* Parameters:	rect	- Rectangle to invalidate.
*				global	- True if repaint event should be handled globally
*
* Description:	Invalidates the specified rectangle, requiring it to be
*				refreshed. This is the TProgram view, so we save the invalid
*				rectangle for the pending repaint event (posted by
*				the TProgram object).
*
****************************************************************************/
{
	if (state & sfLockRepaint)
		return;

	if (invalid.isEmpty()) {
		// This is the first invalidRect call, so save the rectangle and
		// post an evRepaint message. Note that we post it to the event
		// queue, so that it will not be lost over time as other events
		// are processed.

		invalid = rect;
		TEvent event;
		event.what = evRepaint;
		eventQueue.post(event);
		if (!globalRepaint)
			globalRepaint = global;
		}
	else {
		// Another repaint event is already pending, so combine the
		// two repaints to be handled as one. At the moment we simply
		// repaint the union of the two rectangles being updated.

		invalid += rect;
		}
}

void TProgram::doRepaint()
/****************************************************************************
*
* Function:		TProgram::doRepaint
*
* Description:	Performs a repaint operation for the entire program view.
*
****************************************************************************/
{
	setClip(invalid);			// Clip to invalid rectangle
	mouse.obscure();
	invalid.empty();			// Flag the repaint has now occurred
	if (globalRepaint)
		redraw();				// Redraw everything
	else {
		// Find the currently active modal view, and repaint it. Note that
		// to perform the repaint, we must work out the absolute viewport
		// that must be set for the modal view.

		TView	*modal = findModalView();

		modal->setupOwnerViewport();
		modal->redraw();
		modal->resetViewport();
		}
	mouse.unobscure();
	resetClip();				// Reset clipping rectangle
	globalRepaint = false;		// Default to local repaint events
}

void TProgram::draw(const TRect& clip)
/****************************************************************************
*
* Function:		TProgram::draw
* Parameters:	clip	- Clip rectangle for drawing the program
*
****************************************************************************/
{
	TRect	oldView;

	MGL_getViewport(&oldView);
	MGL_setViewport(bounds);
	TGroup::draw(clip);
	MGL_setViewport(oldView);
}

void TProgram::redraw()
/****************************************************************************
*
* Function:		TProgram::redraw
*
****************************************************************************/
{
	TRect	oldView;

	MGL_getViewport(&oldView);
	MGL_setViewport(bounds);
	TGroup::redraw();
	MGL_setViewport(oldView);
}

ushort TProgram::run()
/****************************************************************************
*
* Function:		TProgram::run
*
* Description:	Runs the program until completion.
*
****************************************************************************/
{
	if (valid(cmValid)) {
		select(deskTop);			// Select the desktop
		mouse.show();				// Show the mouse cursor
		unlock();					// Unlock repaint events
		ushort retval = execute();	// Begin processing events
		mouse.hide();
		return retval;
		}
	else {
		MGL_exit();
		cerr << "Not enough memory to start program\n";
		exit(EXIT_FAILURE);
		return cmQuit;
		}
}

void TProgram::suspend()
/****************************************************************************
*
* Function:		TProgram::suspend
*
* Description:	Suspend execution of the program to allow shelling to DOS
*				etc.
*
****************************************************************************/
{
	// systemError.syspend();
	eventQueue.suspend();
	mouse.suspend();
}

void TProgram::resume()
/****************************************************************************
*
* Function:		TProgram::resume
*
* Description:	Resumes execution of the program after shelling to DOS etc.
*
****************************************************************************/
{
	mouse.resume();
	eventQueue.resume();
	// systemError.resume();
	mouse.show();
}

TPalette& TProgram::getPalette() const
/****************************************************************************
*
* Function:		TProgram::getPalette
* Returns:		Pointer to the standard palette for all programs.
*
* Description:	For the moment we simply map the colors directly onto the
*				normal system colors. This needs to be changed to reflect
*				the color organisation of pre-built components (windows,
*				menus, dialogs etc).
*
****************************************************************************/
{
	static char cpProgram[] = {
		scBlack,scJade,					// TBackGround

		scWhite,scDarkGray,				// TMenu and TStatusLine
		scLightGray,scLightGray,
		scBlack,scDarkGray,scBlack,

		scWhite,scDarkGray,				// TWindow
		scYellow,scYellow,scLightGray,
		scLightGray,
		scJade,scLightGray,
		scWhite,scDarkGray,
		scBlack,
		scLightGray,scBlack,scWhite,
		scDarkGray,scBlack,

		scWhite,scDarkGray,				// TDialog
		scYellow,scJade,scLightGray,
		scLightGray,
		scJade,scLightGray,
		scWhite,scDarkGray,
		scBlack,
		scLightGray,scBlack,scWhite,
		scDarkGray,scBlack,
		scLightGray,scLightGray,
		scBlack,scJade,scDarkGray,
		scBlack,
		scBlue,
		scLightYellow,scBlack,scWhite,
		scDarkGray,scBlack,
		scLightYellow,
		scBlack,scWhite,scDarkGray,
		scWhite,
		};

	static TPalette palette(cpProgram,sizeof(cpProgram));
	return palette;
}
