/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tinputl.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TInputLine class.
*
* $Id: tinputl.cpp 1.2 1994/03/09 11:50:36 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tinputl.hpp"
#include "tfontmgr.hpp"
#include "techlib.hpp"
#include "tgroup.hpp"
#include "tmouse.hpp"
#include "tkeys.hpp"
#include <string.h>
#include <ctype.h>
#include <dos.h>

#define	OFFSETX		4
#define	OFFSETY		4

/*----------------------------- Implementation ----------------------------*/

ushort	TInputLine::cursorRate = 9;			// Cursor blink rate
char	buf[2] = "1";						// Single character buffer

TInputLine::TInputLine(const TRect& bounds,int maxLen,const char *defaultText)
	: TView(bounds), maxLen(maxLen), text(new char[maxLen+1]), flags(0)
/****************************************************************************
*
* Function:		TInputLine::TInputLine
* Parameters:	bounds		- Bounding box for the input line
*				maxLen		- Maximum number of characters in input line
*				defaultText	- Text to use as default (NULL for none)
*
* Description:	Constructor for the TInputLine class.
*
****************************************************************************/
{
	setBounds(bounds);
	curPos = firstPos = selStart = selEnd = 0;
	setText(defaultText);
	options |= ofSelectable | ofFirstClick;
	oldTicks = 0;
}

TInputLine::~TInputLine()
/****************************************************************************
*
* Function:		TInputLine::~TInputLine
*
* Description:	Destructor for the TInputLine. Simply deletes the text.
*
****************************************************************************/
{
	delete text;
}

void TInputLine::setBounds(const TRect& bounds)
/****************************************************************************
*
* Function:		TInputLine::setBounds
* Parameters:	bounds	- New bounding box for the input line
*
* Description:	Computes the bounding box for the input line. The bounding
*				box is always resized to fit the current system font
*				(the bottom coordinate is changed).
*
****************************************************************************/
{
	TRect	b(bounds);

	metrics	m;
	fontManager.useFont(fmSystemFont);
	MGL_getFontMetrics(&m);

	starty = max(2 - m.descent,OFFSETY);
	b.bottom() = b.top() + starty + (m.ascent - m.descent + 1) + 3;
	TView::setBounds(b);

	// Work out the number of characters to scroll left or right, to be
	// approximately 1/3 of the input line width.

	short width = scrollChars = 0;

	while (width < size.x/3) {
		width += MGL_charWidth('M');
		scrollChars++;
		}
}

void TInputLine::deleteChars(short start,short count)
/****************************************************************************
*
* Function:		TInputLine::deleteChars
* Parameters:	start	- Starting character to delete
*				count	- Number of characters to delete
*
* Description:	Deletes the specified number of characters starting at
*				start from the text.
*
****************************************************************************/
{
	if (start >= len || len == 0)
		return;

	short	copyfrom = start + count;

	if (copyfrom >= len) {
		text[start] = '\0';			// Just null terminate the string
		len = start;
		}
	else {
		memmove(&text[start],&text[copyfrom],(len-copyfrom)+1);
		len -= count;
		}
	flags |= tlDirty;
}

void TInputLine::insertChar(char ch)
/****************************************************************************
*
* Function:		TInputLine::insertChar
* Parameters:	ch		- Character to insert
*
* Description:	Inserts a character into the string. If the string has
*				reached the maximum length, we beep the speaker.
*
****************************************************************************/
{
	if (flags & tlOverwriteMode) {
		if (curPos == len)
			len++;
		}
	else
		len++;

	if (len > maxLen) {
		len--;
		beep();
		return;
		}

	if (!(flags & tlOverwriteMode))
		memmove(&text[curPos+1],&text[curPos],len-curPos);
	text[curPos++] = ch;
	flags |= tlDirty;
}

bool TInputLine::deleteHighlight()
/****************************************************************************
*
* Function:		TInputLine::deleteHighlight
* Returns:		True if the highlight was deleted, false if not.
*
* Description:	Deletes the highlighted characters
*
****************************************************************************/
{
	if (selStart < selEnd) {
		deleteChars(selStart,selEnd-selStart);
		curPos = selStart;
		clearHighlight();
		return true;
		}
	return false;
}

void TInputLine::clearHighlight()
/****************************************************************************
*
* Function:		TInputLine::clearHighlight
*
* Description:	Clears the highlight if present.
*
****************************************************************************/
{
	if (selStart < selEnd) {
		selStart = selEnd = 0x7FFF;
		flags |= tlDirty;
		}
}

short TInputLine::findPos(const TPoint& p)
/****************************************************************************
*
* Function:		TInputLine::findPos
* Parameters:	p	- Point to find position from
* Returns:		Index of the character hit
*
* Description:	Attempts to find the character index hit by the point
*				p. If the point is halfway between two characters, we
*				find the closest selection point on either side.
*
****************************************************************************/
{
	short	startx = bounds.left() + OFFSETX;
	short	pos = firstPos;
	short	width;

	fontManager.useFont(fmSystemFont);
	while (startx + (width = MGL_charWidth(text[pos]))/2 < p.x) {
		startx += width;
		if (++pos == len)
			break;
		}
	return pos;
}

void TInputLine::selectLeft(short count,ushort modifiers)
/****************************************************************************
*
* Function:		TInputLine::selectLeft
* Parameters:	count	- Number of characters to select
*				modifiers	- keyboard shift modifiers
*
* Description:	Selects the next character on the left. If the shift
*				modifier's are set, we also extend the selection.
*
****************************************************************************/
{
	drawCursor(false);

	fontManager.useFont(fmSystemFont);
	short oldPos = curPos;

	for (int i = 0; i < count; i++) {
		if (curPos <= 0)
			break;
		curLoc -= MGL_charWidth(text[--curPos]);
		}

	if (oldPos != curPos) {
		if (modifiers & mdShift) {
			if (selStart == selEnd) {
				selEnd = curPos+count;
				selStart = curPos;
				}
			else if (selStart > curPos) {
				if (selStart < oldPos)
					selEnd = selStart;
				selStart = curPos;
				}
			else
				selEnd = curPos;
			flags |= tlDirty;
			}
		else
			clearHighlight();
		}

	repositionText();
	refresh();
}

void TInputLine::selectRight(short count,ushort modifiers)
/****************************************************************************
*
* Function:		TInputLine::selectRight
* Parameters:	count	- Number of characters to select
*				modifiers	- keyboard shift modifiers
*
* Description:	Selects the next character on the right. If the shift
*				modifier's are set, we also extend the selection.
*
****************************************************************************/
{
	drawCursor(false);

	fontManager.useFont(fmSystemFont);
	short oldPos = curPos;

	for (int i = 0; i < count; i++) {
		if (curPos == len)
			break;
		curLoc += MGL_charWidth(text[curPos++]);
		}

	if (oldPos != curPos) {
		if (modifiers & mdShift) {
			if (selStart == selEnd) {
				selStart = curPos-count;
				selEnd = curPos;
				}
			else if (selEnd < curPos) {
				if (selEnd > oldPos)
					selStart = selEnd;
				selEnd = curPos;
				}
			else
				selStart = curPos;
			flags |= tlDirty;
			}
		else
			clearHighlight();
		}

	repositionText();
	refresh();
}

void TInputLine::handleEvent(TEvent& event,phaseType phase)
/****************************************************************************
*
* Function:		TInputLine::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Event handling routines for the TInputLine class.
*
****************************************************************************/
{
	TView::handleEvent(event,phase);

	switch (event.what) {
		case evKeyDown:
		case evKeyAuto:
			switch (event.key.keyCode) {
				case kbLeft:
					selectLeft(1,event.key.modifiers);
					break;
				case kbRight:
					selectRight(1,event.key.modifiers);
					break;
				case kbHome:
					selectLeft(curPos,event.key.modifiers);
					break;
				case kbEnd:
					selectRight(len-curPos,event.key.modifiers);
					break;
				case kbBack:
					if (!deleteHighlight() && curPos > 0)
						deleteChars(--curPos,1);
					repositionText();
					refresh();
					break;
				case kbDel:
					if (event.what == evKeyAuto)
						break;
					if (!deleteHighlight())
						deleteChars(curPos,1);
					refresh();
					break;
				case kbIns:
					if (event.what == evKeyAuto)
						break;
					drawCursor(false);
					flags ^= tlOverwriteMode;
					drawCursor(true);
					break;
				default:
					// We have a normal key press here, so insert the
					// character into the text

					if (isprint(event.key.charScan.charCode)) {
						deleteHighlight();
						insertChar(event.key.charScan.charCode);
						repositionText();
						refresh();
						}
					else
						return;
				}
			clearEvent(event);
			break;
		case evMouseDown:
			if (event.mouse.doubleClick) {
				// A double click selects all of the text in the input line

				setText(text);
				}
			else if (event.mouse.buttons & mbLeftButton) {
				bool 	oldMove = eventQueue.mouseMove(true);
				bool	firstSelect = true;
				ushort	oldRepeat = eventQueue.getAutoRepeat();

				eventQueue.setAutoRepeat(2);

				while (event.what != evMouseUp) {
					switch (event.what) {
						case evMouseDown:
							clearHighlight();
							refresh();
						case evMouseAuto:
						case evMouseMove:
							Point p(event.where);
							globalToLocal(p);
							if (bounds.includes(p)) {
								// Selection within the window

								short newPos = findPos(p);
								if (newPos != curPos) {
									if (newPos < curPos)
										selectLeft(curPos-newPos,
											firstSelect ? 0 : mdShift);
									else
										selectRight(newPos-curPos,
											firstSelect ? 0 : mdShift);
									}
								firstSelect = false;
								}
							else if (event.what == evMouseAuto) {
								// Auto selection outside window to scroll

								if (p.x < bounds.left()) {
									if (curPos > 0)
										selectLeft(scrollChars,mdShift);
									}
								else if (curPos < len)
									selectRight(scrollChars,mdShift);
								}
							break;
						}
					getEvent(event);
					}

				eventQueue.mouseMove(oldMove);
				eventQueue.setAutoRepeat(oldRepeat);
				}
			clearEvent(event);
			break;
		}
}

void TInputLine::repaint(const TRect& clip)
/****************************************************************************
*
* Function:		TInputLine::repaint
* Parameters:	clip	- Clipping rectangle
*
* Description:	Repaints the input line in the current state. This assumes
*				that the entire input line is visible, and that the
*               viewport is currently set up for the owning view.
*				We simply display the text in the input line, without
*				redrawing the borders.
*
****************************************************************************/
{
	mouse.obscure();

	// Clip the text at input line boundary

	TRect	r(bounds);
	r.top() += 1;		r.left() += 1;
	r.right() -= 2;		r.bottom() -= 2;
    MGL_setClipRect(r &= clip);
	MGL_setColor(getColor(3));
	MGL_fillRect(bounds);

	fontManager.useFont(fmSystemFont);

	// Draw each letter of the text string individually, highlighting
	// if necessary.

	int 	i,startx = bounds.left() + OFFSETX;
	bool	highlight = false;
	char	*p;
	color_t	textColor = getColor(4);

	r.top() = bounds.top() + 2;
	r.bottom() = bounds.bottom() - 3;
	r.left() = startx;
	curLoc = -1;

	if ((state & sfFocused) && selStart < firstPos) {
		highlight = true;
		textColor = getColor(5);
		}

	for (i = firstPos,p = text + firstPos; i < len; i++) {
		buf[0] = *p++;
		r.right() = startx + MGL_charWidth(buf[0]);

		if (state & sfFocused) {
			if (i == selStart) {
				highlight = true;
				textColor = getColor(5);
				}
			if (i == selEnd) {
				highlight = false;
				textColor = getColor(4);
				}
			if (highlight) {
				MGL_setColor(getColor(6));
				MGL_fillRect(r);
				}
			}

		// Save the cursor insertion point location for later

		if (i == curPos)
			curLoc = startx;

		MGL_setColor(textColor);
		MGL_drawStrXY(startx,bounds.top()+starty,buf);
		if ((startx = r.left() = r.right()) >= bounds.right()-2)
			break;
		}
	if (curLoc == -1)
		curLoc = startx;
	flags &= ~tlDirty;
	mouse.unobscure();
}

void TInputLine::refresh()
/****************************************************************************
*
* Function:		TInputLine::refresh
*
* Description:	Refreshes the display if dirty.
*
****************************************************************************/
{
	if (flags & tlDirty)
		repaint(bounds);
	drawCursor(true);
}

void TInputLine::draw(const TRect& clip)
/****************************************************************************
*
* Function:		TInputLine::draw
* Parameters:	clip	- Clipping rectangle to draw the view with
*
* Description:	Draws the input line in the current state.
*
****************************************************************************/
{
	mouse.obscure();
	MGL_setClipRect(clip);
	MGL_setColor(getColor(2));
	drawRectCoord(bounds.left(),bounds.top(),bounds.right()-1,bounds.bottom()-1);
	MGL_setColor(getColor(1));
	MGL_lineCoord(bounds.left(),bounds.bottom()-1,bounds.right()-1,bounds.bottom()-1);
	MGL_lineCoord(bounds.right()-1,bounds.top(),bounds.right()-1,bounds.bottom()-1);
	repaint(clip);
	mouse.unobscure();
}

void TInputLine::redraw()
/****************************************************************************
*
* Function:		TInputLine::redraw
*
* Description:	Redraws the input line, and turns on the cursor.
*
****************************************************************************/
{
	oldTicks = 0;
	flags &= ~tlCursorVisible;
	TView::redraw();
}

void TInputLine::drawCursor(bool visible)
/****************************************************************************
*
* Function:		TInputLine::drawCursor
* Parameters:	visible	- True if the cursor is visible
*
* Description:	Draws the cursor in the specified state, and resets the
*				time to change the state of the cursor.
*
****************************************************************************/
{
	if (visible)
		flags |= tlCursorVisible;
	else
		flags &= ~tlCursorVisible;

	TRect	c(curLoc,bounds.top()+2,curLoc+1,bounds.bottom()-3);
	bool	selected = (selStart <= curPos && curPos < selEnd);

	if (flags & tlOverwriteMode && curPos < len)
		c.right() = curLoc + MGL_charWidth(text[curPos]);

	// Draw the cursor at the correct location

	mouse.obscure();
	MGL_setClipRect(c);
	MGL_setColor(getColor(visible ? 7 : (selected ? 6 : 3)));
	MGL_fillRect(c);
	MGL_setColor(getColor(visible ? 5 :	(selected ? 5 : 4)));
	buf[0] = text[curPos];
	MGL_drawStrXY(curLoc,bounds.top()+starty,buf);
	mouse.unobscure();

	// Reset the time till the cursor flips again

	oldTicks = (*((short*)FP(0x40,0x6C)));
}

void TInputLine::idle()
/****************************************************************************
*
* Function:		TInputLine::idle
*
* Description:	This routine gets called periodically during idle moments
*				when the input line is focused. It will flash the cursor
*				at the specified time interval.
*
****************************************************************************/
{
	short ticks = (*((short*)FP(0x40,0x6C)));
	if (ticks - oldTicks > cursorRate)
		drawCursor((flags ^ tlCursorVisible) & tlCursorVisible);
}

cursor *TInputLine::getCursor(const TPoint&)
/****************************************************************************
*
* Function:		TInputLine::getCursor
* Returns:		Pointer to the IBEAM cursor.
*
* Description:	Sets the cursor definition to the IBEAM cursor when the
*				cursor is within the input line.
*
****************************************************************************/
{
	return (state & sfActive) ? &IBEAM_CURSOR : NULL;
}

void TInputLine::repositionText()
/****************************************************************************
*
* Function:		TInputLine::repositionText
*
* Description:	Repositions the text within the current window, so that the
*				cursor insertion point is within the set of characters
*				currently within view.
*
****************************************************************************/
{
	fontManager.useFont(fmSystemFont);

	if (curPos < firstPos) {
		if ((firstPos = curPos - scrollChars*2) < 0)
			firstPos = 0;
		flags |= tlDirty;
		}
	else if (curLoc + MGL_charWidth(text[curPos]) >= bounds.right()-2) {
		if ((firstPos = curPos - scrollChars) < 0)
			firstPos = 0;
		flags |= tlDirty;
		}
}

void TInputLine::setText(const char *text)
/****************************************************************************
*
* Function:		TInputLine::setText
* Parameters:	text		- text to copy
*
* Description:	Sets the current text for the input line.
*
****************************************************************************/
{
	if (text)
		strncpy(TInputLine::text,text,maxLen);
	else
		TInputLine::text[0] = '\0';
	TInputLine::text[maxLen] = '\0';
	len = strlen(TInputLine::text);

	// Select all of the text

	selStart = firstPos = 0;
	curPos = selEnd = len;
	fontManager.useFont(fmSystemFont);
	curLoc = bounds.left() + OFFSETX + MGL_textWidth(text);
	repositionText();
	if (owner) {
		if (state & sfExposed) {
			if (!(owner->getState() & sfLockRepaint)) {
				setupOwnerViewport();
				repaint(bounds);
				resetViewport();
				}
			}
		else
			owner->repaint(bounds);
		}
}

TPalette& TInputLine::getPalette() const
/****************************************************************************
*
* Function:		TInputLine::getPalette
* Returns:		Pointer to the standard palette for input line's
*
****************************************************************************/
{
	static char cpInputLine[] = {1,2,24,25,26,27,28};
	static TPalette palette(cpInputLine,sizeof(cpInputLine));
	return palette;
}
