/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tgroup.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TGroup class.
*
* $Id: tgroup.cpp 1.2 1994/03/09 11:50:36 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tgroup.hpp"
#include "tmouse.hpp"
#include "techlib.hpp"
#include "tprogram.hpp"

/*----------------------------- Implementation ----------------------------*/

TGroup::TGroup(const TRect& bounds)
	: TView(bounds), inner(bounds), selected(NULL)
/****************************************************************************
*
* Function:		TGroup::TGroup
* Parameters:	bounds	- Bounding rectangle for the group
*
* Description:	Constructor for the TGroup class
*
****************************************************************************/
{
	resetClip();
}

TGroup::~TGroup()
/****************************************************************************
*
* Function:		TGroup::~TGroup
*
* Description:	Destructor for the TGroup class. We dont need to do
*				anything as the list of children will automatically
*				destruct itself correctly.
*
****************************************************************************/
{
}

bool TGroup::valid(ushort command)
/****************************************************************************
*
* Function:		TGroup::valid
* Parameters:	command	- Command ending the modal operation
* Returns:		True if the group is valid
*
* Description:	Simply checks that all children in the view are valid.
*
****************************************************************************/
{
	for (DListIterator<TView> i(children); i; i++)
		if (i.node()->valid(command) == false)
			return false;
	return true;
}

void TGroup::doEvent(TView& view,TEvent& event,phaseType phase)
/****************************************************************************
*
* Function:		TGroup
* Parameters:	view	- View to send event to
*				event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Handles the event for a single view. We check to see
*				if the view can recieve the event, send the event to
*				it if it can.
*
****************************************************************************/
{
	// If the view is disabled, don't send any events to it

	if (view.getState() & sfDisabled)
		return;

	switch (phase) {
		case phPreProcess:
			if (!(view.getOptions() & ofPreProcess))
				return;
			break;
		case phPostProcess:
			if (!(view.getOptions() & ofPostProcess))
				return;
			break;
		}
	view.handleEvent(event,phase);
}

void TGroup::eventError(TEvent& event)
/****************************************************************************
*
* Function:		TGroup::eventError
* Parameters:	event	- Unhandled event
*
* Description:	Handles the case of unhandled events. Simply calls the
*				owners routine if an owner exists.
*
****************************************************************************/
{
	if (owner)
		owner->eventError(event);
}

void TGroup::handleEvent(TEvent& event,phaseType phase)
/****************************************************************************
*
* Function:		TGroup::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Event handling routine for the TGroup class. Here we
*				delegate the event to all children of the group, depending
*				on the type of event (focused etc).
*
*				Note that before the event is sent to any of the children
*				of the group, the viewport is set to that of the owning
*				group.
*
****************************************************************************/
{
	DListIterator<TView>	i;

	TView::handleEvent(event,phase);		// Default handling first

	setupViewport();

	// Handle the case of the cmGrabModal and cmReleaseModal messages
	// specially for groups.

	if (event.what == evBroadcast && (state & sfModal)) {
		if (event.message.command == cmGrabModal) {
			modalState++;
			}
		else if (event.message.command == cmReleaseModal) {
			PRECONDITION(modalState > 0);
			modalState--;
			}
		}

	if (event.what & focusedEvents) {
		// The event is one that can be focused, so send the event to
		// all views with the ofPreProcess flag set, then to the focused
		// view, then to all views with the ofPostProcess flag set.

		for (i = children; i; i++)
			doEvent(*i.node(),event,phPreProcess);

		if (selected)
			doEvent(*selected,event,phFocused);

		if (event.what != evNothing) {
			for (i = children; i; i++)
				doEvent(*i.node(),event,phPostProcess);
			}
		}
	else {
		if (event.what & positionalEvents) {
			// The event is a positional type event, so work out which
			// child view the event lies in, and send the event to it.

			bool handled = false;
			for (i = children,i.restartTail(); i; i--) {
				if (i.node()->includes(event.where)) {
					doEvent(*i.node(),event,phFocused);
					handled = true;
					}
				}
			if (!handled && (state & sfModal) && event.what == evMouseDown) {
				// Positional event outside of the modal viewport, so
				// beep the speaker to indicate this.

				beep();
				}
			}
		else {
			// Must be a broadcast or some other type of event, so send
			// it to all children

			for (i = children; i; i++)
				doEvent(*i.node(),event,phFocused);
			}
		}

	resetViewport();
}

ushort TGroup::execView(TGroup *view)
/****************************************************************************
*
* Function:		TGroup::execView
* Parameters:	view	- Pointer to view to execute
* Returns:		Command that caused the view to complete execution
*
* Description:	Pop's up the group on the desktop, and executes
*				it until completion. When the group has finished
*				interacting with the user, the display under the group
*				is restored by posting a cmRepaint message with an
*				invalid rectangle set to the area affected by the
*				group.
*
****************************************************************************/
{
	TEvent	event;
	ushort	retval = cmCancel;

	if (view) {
		// Save the state of the view before executing it.

		bool 	selectable = view->getOptions() & ofSelectable;
		TView	*oldSelected = selected;

		view->setOption(ofSelectable,false);
		view->setState(sfModal,true);

		lock();						// Lock repaint events
		insert(view);
		select(view);
		unlock();
		retval = view->execute();
		remove(view);
		select(oldSelected);

		view->setState(sfModal,false);
		view->setOption(ofSelectable,selectable);

		// Repaint the area behind the dialog box by checking for
		// the pending repaint event.

		getEvent(event,evRepaint);
		}
	return retval;
}

void TGroup::changeCursor()
/****************************************************************************
*
* Function:		TGroup::changeCursor
*
* Description:	Changes the cursor definition if need be.
*
****************************************************************************/
{
	TPoint where;
	mouse.pos(where);
	cursor *c = getCursor(where);

	if (c == NULL) {
		if (_MVIS_currentCursor != &DEF_CURSOR) {
			mouse.setCursor(&DEF_CURSOR);
			_MVIS_currentCursor = &DEF_CURSOR;
			}
		}
	else if (c != _MVIS_currentCursor) {
		mouse.setCursor(c);
		_MVIS_currentCursor = c;
		}
}

ushort TGroup::execute()
/****************************************************************************
*
* Function:		TGroup::execute
* Returns:		Command the caused the view to complete execution
*
* Description:	Handles event processing for the group until the modal
*				view has finished executing.
*
****************************************************************************/
{
	TEvent 	event;

	// Save the old viewport and set the new viewport to be equal to the
	// owner's viewport, as all events will be handled relative to this
	// viewport.

	setupOwnerViewport();

	// Draw the group before executing it. Modal group's will always
	// be the topmost view, so we can avoid doing an expensive repaint
	// by locking all repaint events and simply drawing it on top
	// of everything.

	mouse.obscure();
	draw(bounds);
	mouse.unobscure();
	do {
		endState = cmValid;
		do {
			if (getEvent(event)) {
				handleEvent(event);
				if (event.what != evNothing)
					eventError(event);
				}
			else
				changeCursor();
			} while (endState == cmValid);
		} while (!valid(endState));

	resetViewport();
	return endState;
}

void TGroup::doInsert(TView *view)
/****************************************************************************
*
* Function:		TGroup::doInsert
* Parameters:	view	- View being inserted into the group
*
* Description:	Finished the insertion operation, doing all things common
*				to all insert operations.
*
****************************************************************************/
{
	view->setOwner(this);			// Set the owner field to point to us

	// Now check to see if the view should be centered, and if so
	// then center it. Note that we center it within the inner portion
	// of the group.

	if (view->getOptions() & ofCentered) {
		TRect	b(view->getBounds());
		TPoint	p(b.topLeft);

		if (view->getOptions() & ofCenterX) {
			p.x = ((inner.right()-inner.left()) - (b.right()-b.left())) / 2
					+ inner.left() - bounds.left();
			}
		if (view->getOptions() & ofCenterY) {
			p.y = ((inner.bottom()-inner.top()) - (b.bottom()-b.top())) / 2
					+ inner.top() - bounds.top();
			}

		view->moveTo(p);
		}

	// If the owner is exposed, then expose the child, causing a
	// repaint event to be posted.

	view->setState(sfExposed,state & sfExposed);
}

void TGroup::insert(TView *view)
/****************************************************************************
*
* Function:		TGroup::insert
* Parameters:	view	- View to insert into group
*
* Description:	Inserts the view at the head of the list of children.
*
****************************************************************************/
{
	if (view) {
		children.addToHead(view);		// Add to head of the list
		doInsert(view);
		}
}

void TGroup::insertAfter(TView *view,TView *after)
/****************************************************************************
*
* Function:		TGroup::insertAfter
* Parameters:	view	- View to insert into group
*				after	- View to insert after
*
* Description:	Inserts the view into the child list after the specified
*				child.
*
****************************************************************************/
{
	if (view) {
		children.addAfter(view,after);
		doInsert(view);
		}
}

void TGroup::remove(TView *view)
/****************************************************************************
*
* Function:		TGroup::remove
* Parameters:	view	- View to remove from the group
*
* Description:	Removes the view from the child list.
*
****************************************************************************/
{
	children.remove(view);
	view->setOwner(NULL);			// View has no owner, so clear field
	if (selected && view == selected)
		selected = NULL;

	// Request a repaint event for the spot where the view was removed
	// from, and unexpose the view and all children

	view->setState(sfExposed,false);
	globalRepaint(view->getBounds());
}

void TGroup::select(TView *view)
/****************************************************************************
*
* Function:		TGroup::select
* Parameters:	view	- View to select
*
* Description:	Make the view the new selected view. If view is NULL,
*				then the current view is deselected and no view is
*				selected.
*
****************************************************************************/
{
	if (view != selected) {
		// We are changing the currently selected view, so deselect
		// the old view and select the new one

		if (view) {
			if (!(view->getState() & sfActive))
				view->setState(sfActive,true);
			}

		if (selected) {
			selected->setState(sfFocused,false);
			repaint(selected->getBounds());
			}
		selected = view;
		if (view) {
			view->setState(sfFocused,true);
			repaint(view->getBounds());
			}
		}
}

TView* TGroup::getNext(TView *view)
/****************************************************************************
*
* Function:		TGroup::getNext
* Parameters:	view	- View to start with
* Returns:		Next view in tab order.
*
****************************************************************************/
{
	if (view == NULL)
		return children.peekHead();
	TView *next = children.next(view);
	if (next == NULL)
		return children.peekHead();
	return next;
}

TView* TGroup::getPrev(TView *view)
/****************************************************************************
*
* Function:		TGroup::getPrev
* Parameters:	view	- View to start with
* Returns:		Previous view in tab order
*
****************************************************************************/
{
	if (view == NULL)
		return children.peekHead();
	TView *prev = children.prev(view);
	if (prev == NULL)
		return children.peekTail();
	return prev;
}

void TGroup::selectNext()
/****************************************************************************
*
* Function:		TGroup::selectNext
*
* Description:	Selects the next view in the group, in tab order.
*
****************************************************************************/
{
	TView	*next = selected;

	do {
		if ((next = getNext(next)) == NULL) {
			selected = NULL;
			return;
			}
		} while ((next != selected) &&
				 (!(next->getState() & sfActive) ||
				  !(next->getOptions() & ofSelectable)));
	select(next);
}

void TGroup::selectPrev()
/****************************************************************************
*
* Function:		TGroup::selectPrev
*
* Description:	Selects the previous view in the group in tab order.
*
****************************************************************************/
{
	TView	*prev = selected;
	do {
		if ((prev = getPrev(prev)) == NULL) {
			selected = NULL;
			return;
			}
		} while ((prev != selected) &&
				 (!(prev->getState() & sfActive) ||
				  !(prev->getOptions() & ofSelectable)));
	select(prev);
}

void TGroup::setState(ushort aState,bool set)
/****************************************************************************
*
* Function:		TGroup::setState
* Parameters:	aState	- State flag to set
*				set		- True if flag should be set, false if cleared
*
****************************************************************************/
{
	if ((aState & sfModal) && set) {
		// The group is grabbing the modal state

		PRECONDITION(!(state & sfModal));
		message(TProgram::application,evBroadcast,cmGrabModal);
		modalState = 0;
		}

	TView::setState(aState,set);

	DListIterator<TView>	i;

	if (aState & sfActive) {
		// The group just became active, so activate all of the views in
		// the group, and request a repaint for the entire group

		for (i = children; i; i++)
			i.node()->setState(aState,set);
		repaint();
		}
	if (aState & sfFocused) {
		// The group just came into focus or went out of focus, so
		// change the focus on the currently selected view.

		if (selected)
			selected->setState(sfFocused,set);
		}
	if (aState & sfExposed) {
		// The group was just exposed or un-exposed, so change the
		// exposure of all the visible children and request a repaint
		// for the entire group.

		for (i = children; i; i++) {
			if (i.node()->getState() & sfVisible)
				i.node()->setState(sfExposed,set);
			}
		repaint();
		}
	if ((aState & sfModal) && !set) {
		// The group is releasing the modal state.

		message(TProgram::application,evBroadcast,cmReleaseModal);
		}
}

void TGroup::setBounds(const TRect& bounds)
/****************************************************************************
*
* Function:		TGroup::setBounds
* Parameters:	bounds	- New bounding rectangle for the view
*
* Description:	Sets the bounding rectangle for the view. This may need to
*				be overridden by derived classes to handle the case when
*				the view is moved when inserted into a group.
*
****************************************************************************/
{
	TView::setBounds(bounds);
	resetClip();
}

void TGroup::moveTo(int x,int y)
/****************************************************************************
*
* Function:		TGroup::moveTo
* Parameters:	x,y	- New position to move view to
*
****************************************************************************/
{
	TView::moveTo(x,y);
	resetClip();
}

bool TGroup::getClipRect(TRect& c) const
/****************************************************************************
*
* Function:		TGroup::getClipRect
* Parameters:	c	- Place to store the clip rectangle
* Returns:		True if the clip rectangle is non-empty.
*
* Description:	Obtains the current clip rectangle for the group. This
*				clip rectangle will be relative to the owner's viewport.
*
****************************************************************************/
{
	if (TView::getClipRect(c)) {
		TRect oClip(bounds.left() + clip.left(),
					bounds.top() + clip.top(),
					bounds.left() + clip.right(),
					bounds.top() + clip.bottom());
        c &= oClip;
		return !c.isEmpty();
		}
	else
		return false;
}

void TGroup::drawBackground(const TRect&)
/****************************************************************************
*
* Function:		TGroup::drawBackground
* Parameters:	clip	- Clipping rectangle to use when drawing
*
* Description:	Draws the background for the group. By default group's
*				have no background.
*
****************************************************************************/
{
}

TView *TGroup::findModalView()
/****************************************************************************
*
* Function:		TGroup::findModalView
* Returns:		Pointer to the currently active modal view (NULL if none).
*
* Description:	Checks to see if this group is the modal group, returning
*				a pointer to this if it is, otherwise we search all
*				children for the modal group.
*
****************************************************************************/
{
	if ((state & sfModal) && modalState == 0)
		return this;

	for (DListIterator<TView> i(children); i; i++) {
		TView *view = i.node()->findModalView();
		if (view)
			return view;
		}
	return NULL;
}

void TGroup::idle()
/****************************************************************************
*
* Function:		TGroup::idle
*
* Description:	Routine called when the view is idle. We simply call the
*				currently focused view's idle routine.
*
****************************************************************************/
{
	if (selected) {
		setupViewport();
		selected->idle();
		resetViewport();
		}
}

cursor *TGroup::getCursor(const TPoint& p)
/****************************************************************************
*
* Function:		TGroup::getCursor
* Parameters:	p	- Current location of mouse cursor (global coords)
* Returns:		Pointer to the cursor definition for the view.
*
* Description:	Here we work out which child view contains the cursor
*				and call it to obtain the cursor definition.
*
****************************************************************************/
{
	setupViewport();
	for (DListIterator<TView> i = children; i; i++)
		if (i.node()->includes(p)) {
			resetViewport();
			return i.node()->getCursor(p);
			}
	resetViewport();
	return NULL;
}

void TGroup::draw(const TRect&)
/****************************************************************************
*
* Function:		TGroup::draw
*
* Description:	Draws the group by drawing all of the children in the
*				group, starting with the last child and drawing in front
*				to back order.
*
****************************************************************************/
{
	DListIterator<TView> i(children);

	setupViewport();

	// Compute intersection between clipping rectangle and clipping
	// rectangle stored in the group, and draw all the views.

	TRect	c(0,0,size.x,size.y);
    c &= clip;
	drawBackground(c);
	for (i.restartTail(); i; i--) {
		if (i.node()->getState() & sfVisible && i.node()->getState() & sfExposed)
			i.node()->draw(c);
		}
	resetViewport();
}

void TGroup::redraw()
/****************************************************************************
*
* Function:		TGroup::redraw
*
* Description:	Redraws the group by redrawing all of the children in the
*				group, starting with the last child and drawing in front
*				to back order.
*
*				Redrawing always honours the clipping region.
*
****************************************************************************/
{
	TRect	c(bounds);

	if (getClipRect(c)) {
		if (c == bounds)
			draw(c);
		else {
			DListIterator<TView> i(children);

			setupViewport();
			c.offset(-bounds.left(),-bounds.top());
			drawBackground(c);
			for (i.restartTail(); i; i--)
				i.node()->redraw();
			resetViewport();
			}
		}
}

ushort TGroup::getHelpCtx() const
/****************************************************************************
*
* Function:		TGroup::getHelpCtx
*
* Description:	Returns the help context number for the group. If the
*				currently active view has no context, we return the
*				context of the entire group.
*
****************************************************************************/
{
	ushort h;

	if (selected && (h = selected->getHelpCtx()) == hcNoContext)
		h = TView::getHelpCtx();
	return h;
}
