/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tfontmgr.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Module to implement the loadable font manager routines.
*
* $Id: tfontmgr.cpp 1.2 1994/03/09 11:50:36 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tfontmgr.hpp"
#include "tprogram.hpp"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <dir.h>
#include <io.h>

/*---------------------------- Global Variables ---------------------------*/

const ushort MAXFONTS = 100;	// Maximum number of registered fonts

// Table of all currently registered font families

struct FontRegEntry {
	ushort		family;			// Family of the font
	const char	*fileprefix;	// Filename prefix for the font
	};

FontRegEntry registeredFonts[MAXFONTS] = {
	{ fmVecAstrology,			"astrolog"	},
	{ fmVecCyrillic,			"cyrillic"	},
	{ fmVecGothicEnglish,		"gotheng"	},
	{ fmVecGothicGerman,		"gothger"	},
	{ fmVecGothicItalian,		"gothita"	},
	{ fmVecGreekComplex,		"greekc"	},
	{ fmVecGreekComplexSmall,	"greekcs"	},
	{ fmVecGreekSimplex,		"greeks"	},
	{ fmVecItalicComplex,		"italicc"	},
	{ fmVecItalicComplexSmall,	"italiccs"	},
	{ fmVecItalicTriplex,		"italict"	},
	{ fmVecJapanese,			"japanese"	},
	{ fmVecLittle,				"little"	},
	{ fmVecMathLarge,			"mathbig"	},
	{ fmVecMathSmall,			"mathsmal"	},
	{ fmVecMusical,				"musical"	},
	{ fmVecRomanComplex,		"romanc"	},
	{ fmVecRomanComplexSmall,	"romancs"	},
	{ fmVecRomanDuplex,			"romand"	},
	{ fmVecRomanSimplex,		"romans"	},
	{ fmVecRomanTriplex,		"romant"	},
	{ fmVecScriptComplex,		"scriptc"	},
	{ fmVecScriptSimplex,		"scripts"	},
	{ fmVecSymbol,				"symbol"	},

	{ fmCharter,				"char"		},
	{ fmCourier,				"cour"		},
	{ fmHelvetica,				"helv"		},
	{ fmLucidaBright,			"lucb"		},
	{ fmLucidaSans,				"lucs"		},
	{ fmLucidaTypewriter,		"luct"		},
	{ fmNewCenturySchoolbook,	"ncen"		},
	{ fmSymbol,					"symb"		},
	{ fmTimes,					"tms"		},
	{ fmFixed,					"fix"		},
	{ fmPC,						"pc"		},

	{ fmUserFont,				NULL		},
	};

TFontManager	fontManager;

/*----------------------------- Implementation ----------------------------*/

TFontManager::TFontManager()
/****************************************************************************
*
* Function:		TFontManager::TFontManager
*
* Description:	Constructor for the font manager class. Here we setup the
*				initial global variables etc, but do not load any fonts.
*				The system font is set to the built in 8x8 bitmap font,
*				and can be re-set to something else after the graphics
*				system has been initialised.
*
****************************************************************************/
{
	currentFont = systemFont = FontRecord(fmFont8x8);
	lastRegistered = 0;
	while (registeredFonts[lastRegistered].family != fmUserFont)
		lastRegistered++;
	lastRegistered--;
	lastUserFont = fmUserFont-1;
	fonts = NULL;
	setMaximumFonts(fmDefaultMaximum);
}

TFontManager::~TFontManager()
/****************************************************************************
*
* Function:		TFontManager::~TFontManager
*
* Description:	Destructor for the font manager class. Purge all of the
*				currently loaded fonts and delete the font table.
*
****************************************************************************/
{
	shutDown();
	delete fonts;
}

void TFontManager::shutDown()
/****************************************************************************
*
* Function:		TFontManager::shutDown
*
* Description:	Shuts down the font manager, by resetting the system font
*				to the built in 8x8 bitmap font and purging all loaded
*				fonts.
*
****************************************************************************/
{
	setSystemFont(fmFont8x8);
	useFont(fmSystemFont);
	while (purgeLRUFont() != -1)
		;
}

void TFontManager::setMaximumFonts(ushort max)
/****************************************************************************
*
* Function:		TFontManager::setMaximumFonts
* Parameters:	max	- New maximum number of fonts
*
* Description:	Sets a new maximum number of loaded fonts. All the currently
*				loaded fonts will be purged, and will require reloading
*				again.
*
****************************************************************************/
{
	if (fonts) {
		while (purgeLRUFont() != -1)
			;
		delete fonts;
		}

	fonts = new FontEntry[maximumFonts = max];

	// Fill in the first entry with the built in 8x8 bitmap font, and
	// clear all other entries.

	fonts[0].rec.family = fmFont8x8;
	fonts[0].rec.attributes = fmNonPurgeable;
	fonts[0].fptr = DEFAULT_FONT;
	fonts[0].LRU = 0xFFFF;

	for (int i = 1; i < maximumFonts; i++) {
		fonts[i].LRU = 0;
		fonts[i].fptr = NULL;
		}
}

const char *TFontManager::fontFilename(const FontRecord& fontRec)
/****************************************************************************
*
* Function:		TFontManager::fontFilename
* Parameters:	fontRec	- Font record for the font
* Returns:		Pointer to the font's filename, or NULL if invalid.
*
* Description:	Builds the name of the font file to be loaded in a static
*				buffer. If the font record is invalid, we return NULL.
*
****************************************************************************/
{
	for (int i = 0; i <= lastRegistered; i++) {
		if (registeredFonts[i].family == fontRec.family)
			break;
		}
	if (i > lastRegistered)
		return NULL;

	// Build the name of the font file to load, from the font's file prefix
	// and the size and attributes of the font.

	static char name[MAXPATH];
	if (fontRec.sizey == -1) {
		sprintf(name,"%s%s%s",
			FMGR_FONTPATH,
			registeredFonts[i].fileprefix,
			FMGR_FONTEXT);
		}
	else if (fontRec.sizex == -1) {
		sprintf(name,"%s%s%d%s%s%s",
			FMGR_FONTPATH,
			registeredFonts[i].fileprefix,
			fontRec.sizey,
			(fontRec.attributes & fmBold) ? "b" : "",
			(fontRec.attributes & fmItalic) ? "i" : "",
			FMGR_FONTEXT);
		}
	else {
		sprintf(name,"%s%s%dx%d%s%s%s",
			FMGR_FONTPATH,
			registeredFonts[i].fileprefix,
			fontRec.sizex,
			fontRec.sizey,
			(fontRec.attributes & fmBold) ? "b" : "",
			(fontRec.attributes & fmItalic) ? "i" : "",
			FMGR_FONTEXT);
		}
	return name;
}

int TFontManager::loadFont(const FontRecord& fontRec,bool purgeable)
/****************************************************************************
*
* Function:		TFontManager::loadFont
* Parameters:	fontRec		- Font record for font to load
*				purgeable	- True if font should be purgeable
* Returns:		Index of loaded font in font table, or -1 on error.
*
* Description:	Attempts to load the specified font. We first check the
*				font table to see if it is already loaded, otherwise we
*				attempt to load it.
*
****************************************************************************/
{
	int		i,freeSlot,errorCode;

	if (fontRec.family == fmFont8x8)
		return 0;

	// Search the table of loaded fonts to see if it is already loaded,
	// also finding the first available slot.

	for (i = 0,freeSlot = maximumFonts; i < maximumFonts; i++) {
		if (fonts[i].fptr == NULL)
			freeSlot = min(i,freeSlot);
		else if (fonts[i].rec == fontRec) {
			if (fontRec != systemFont) {
				fonts[i].rec.attributes &= ~fmNonPurgeable;
				fonts[i].rec.attributes |= (purgeable ? 0 : fmNonPurgeable);
				}
			return i;
			}
		}

	// The font is not currently loaded, so attempt to load it in.

	const char *name = fontFilename(fontRec);
	if (name == NULL) {
		errorCode = cmInvalidFont;
		goto error;
		}

	if (freeSlot == maximumFonts)
		freeSlot = purgeLRUFont();
	fonts[freeSlot].rec = fontRec;
	fonts[freeSlot].rec.attributes &= ~fmNonPurgeable;
	fonts[freeSlot].rec.attributes |= (purgeable ? 0 : fmNonPurgeable);

	while (fonts[freeSlot].fptr == NULL) {
		fonts[freeSlot].fptr = MGL_loadFont(name);
		switch (MGL_result()) {
			case grLoadMem:
				if (purgeLRUFont() == -1) {
					errorCode = cmLowFontMem;
					goto error;
					}
				break;
			case grBadFontFile:
				errorCode = cmBadFontFile;
				goto error;
			case grFontNotFound:
				errorCode = cmFontNotFound;
				goto error;
			}
		}
	return freeSlot;

error:
	TEvent event;
	event.what = evCommand;
	event.message.infoPtr = NULL;
	event.message.command = errorCode;
	TProgram::deskTop->putEvent(event);
	return -1;
}

bool TFontManager::useFont(const FontRecord& fontRec)
/****************************************************************************
*
* Function:		TFontManager::useFont
* Parameters:	fontRec	- Font record of the font to use
* Returns:		True if the font is now active, false on error.
*
* Description:	Attempts to use the specified font. This may fail for
*				a number of reasons:
*
*					1. The font record is invalid
*					2. There is not enough memory to load the font
*					3. The font file does not exist
*
*				In either case, a cmInvalidFont or cmLowFontMem message
*				will be posted to the desktop, and the routine will
*				return false. The currently active font will be set to
*				the system font.
*
****************************************************************************/
{
	int		i,index;
	FontRecord	rec = fontRec;

	if (rec.family == fmSystemFont)
		rec = systemFont;

	if (rec != currentFont) {
		currentFont = rec;
		if ((index = loadFont(rec)) == -1) {
			useFont(fmSystemFont);
			return false;
			}
		PRECONDITION(index < maximumFonts);
		MGL_useFont(fonts[index].fptr);
		MGL_setSpaceExtra(0);
		if (currentFont.family < fmBitmapFont) {
			// Font is a vector font, so always reset the vector font to
			// the default size of (1,1)

			MGL_setTextSize(1,1,1,1);
        	}
		fonts[index].LRU = 0xFFFF;
		}

	// Adjust the LRU counts for all loaded fonts.

	for (i = 0; i < maximumFonts; i++) {
		if (fonts[i].LRU-- == 0)
			fonts[i].LRU = 0;
		}
	return true;
}

void TFontManager::purgeFont(const FontRecord& fontRec)
/****************************************************************************
*
* Function:		TFontManager::purgeFont
* Parameters:	fontRec	- Font record for the font to purge
*
* Description:	Forcibly purges a specified font. The system font cannot
*				be forcibly purged.
*
****************************************************************************/
{
	if (fontRec.family == fmSystemFont)
		return;

	for (int i = 0; i < maximumFonts; i++) {
		if (fonts[i].rec == fontRec)
			break;
		}
	if (i == maximumFonts)
		return;

	if (fonts[i].rec == systemFont || fonts[i].rec.family == fmFont8x8)
		return;

	MGL_unloadFont(fonts[i].fptr);
	fonts[i].fptr = NULL;
	fonts[i].LRU = 0;
	if (fontRec == currentFont)
		useFont(fmFont8x8);
}

int TFontManager::purgeLRUFont()
/****************************************************************************
*
* Function:		TFontManager::purgeLRUFont
* Returns:		Index of the purged font, or -1 if not more purgeable fonts.
*
* Description:	Finds and purges the least recently used font. If there
*				are no more purgeable fonts, we return -1.
*
****************************************************************************/
{
	int		index = -1;
	ushort	minLRU = 0xFFFF;

	for (int i = 0; i < maximumFonts; i++) {
		if (fonts[i].fptr != NULL && !(fonts[i].rec.attributes & fmNonPurgeable)) {
			if (fonts[i].LRU < minLRU) {
				index = i;
				minLRU = fonts[i].LRU;
				}
			}
		}
	if (index != -1)
		purgeFont(fonts[index].rec);
	return index;
}

ushort TFontManager::registerFont(const char *fileprefix)
/****************************************************************************
*
* Function:		TFontManager::registerFont
* Parameters:	fileprefix	- Filename prefix for the font file.
* Returns:		Font family for the registered font, 0xFFFF if no more room.
*
* Description:	Registers a user font family for use with the font manager.
*
****************************************************************************/
{
	if (++lastRegistered == MAXFONTS)
		return 0xFFFF;
	registeredFonts[lastRegistered].family = ++lastUserFont;
	registeredFonts[lastRegistered].fileprefix = fileprefix;
	return lastUserFont;
}

short TFontManager::availableFont(const FontRecord& fontRec)
/****************************************************************************
*
* Function:		TFontManager::availableFont
* Parameters:	fontRec	- Font record for font to check for
* Returns:		0 if available, error code if not.
*
* Description:	Checks to see if the specified font is available for use.
*				Returns one of the following:
*
*					0				- Font is available for use
*					cmInvalidFont	- Font record is invalid
*					cmFontNotFound	- Font file is unavailable
*
****************************************************************************/
{
	if (fontRec.family == fmFont8x8)
		return 0;

	const char *name = fontFilename(fontRec);
	if (name == NULL)
		return cmInvalidFont;
	if (!MGL_availableFont(name))
		return cmFontNotFound;
	return 0;
}

bool TFontManager::setSystemFont(const FontRecord& fontRec)
/****************************************************************************
*
* Function:		TFontManager::setSystemFont
* Parameters:	fontRec	- Font record of the new system font
* Returns:		True if system font was changed, false if font number invalid
*
* Description:	Attempts the change the system font setting to a new font.
*
****************************************************************************/
{
	if (systemFont == fontRec)
		return true;

	int	i;
	if (fontRec.family == fmFont8x8)
		i = 0;
	else {
		for (i = 0; i < lastRegistered; i++)
			if (registeredFonts[i].family == fontRec.family)
				break;
		}
	if (i != lastRegistered && availableFont(fontRec) == 0) {
		FontRecord oldSystemFont = systemFont;
		systemFont = fontRec;
		purgeFont(oldSystemFont);
		return (loadFont(fontRec,false) != -1);
		}
	return false;
}
