/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tfiledlg.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TFIleDialog class.
*
* $Id: tfiledlg.cpp 1.2 1994/03/09 11:50:36 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tfiledlg.hpp"
#include "tfontmgr.hpp"
#include "tsttext.hpp"
#include "tbutton.hpp"
#include "tkeys.hpp"
#include "msgbox.hpp"
#include <dos.h>
#include <ctype.h>
#include <direct.h>

/*----------------------------- Implementation ----------------------------*/

#define	LINESPERLIST	6

const uint	findAttr = FA_RDONLY | FA_ARCH;

TFileDialog::TFileDialog(const char *defFile,const char *title,
	const char *inputName,ushort flags)
	: TDialog(TRect(0,0,100,100),title), flags(flags),
      filenames(5,5), directories(5,5),
	  TWindowInit(TWindow::initFrame,TWindow::initTitleBar)
/****************************************************************************
*
* Function:		TFileDialog::TFileDialog
* Parameters:	defFile		- Text to put up as the default file
*				title		- Title for the dialog box
*				inputName	- Name for the filename input box
*				flags		- Creation flags for the dialog box
*
* Description:	Constructor for the TFileDialog.
*
****************************************************************************/
{
	TLabel	*label,*label2;

	options |= ofCentered;

	// Find the height of the system font, and compute the location of
	// all the elements of the input dialog.

	fontManager.useFont(fmSystemFont);
	metrics m;
	MGL_getFontMetrics(&m);
	int height = MGL_textHeight();
	int cellHeight = m.ascent - m.descent + 1 + 4;
	int buttonHeight = height + 12;
	int dialogWidth = (MGL_sizex()*2)/3;
	int	linesPerList = LINESPERLIST;

	if (MGL_sizey() < 349)
		linesPerList /= 2;

	TRect r(inner);
	int	adjustBottom = bounds.bottom() - r.bottom();

	// Create the filename input line and label

	r.top() += height;
	r.bottom() = r.top() + height;
	r.left() = 15;
	r.right() = 15 + MGL_textWidth(inputName);
	insert(label = new TLabel(r,inputName,NULL));

	r.top() = r.bottom();
	r.bottom() = r.top() + height;
	r.right() = dialogWidth-15;
	fileLine = new TInputLine(r,MAXPATH);
	label->setAttached(fileLine);

	// Create the file type and drive pick lists and labels

	r = fileLine->getBounds();
	TRect r2(r);
	r.right() = dialogWidth/2 - 5;
	r2.left() = dialogWidth/2 + 5;
	r2.top() = r.top() = r.bottom() + 8;
	r2.bottom() = r.bottom() = r.top() + height;
	insert(label = new TLabel(r,typeText,NULL));
	insert(label2  = new TLabel(r2,driveText,NULL));

	r2.top() = r.top() = r.bottom();
	r2.bottom() = r.bottom() = r.top() + height;
	TInputLine *fileType = new TInputLine(r,11,"<All Files>");
	fileType->setState(sfDisabled,true);		// Disable for now :-)
	driveSel = new TInputLine(r2,MAXDRIVE,NULL);
	driveSel->setState(sfDisabled,true);
	label->setAttached(fileType);
	label2->setAttached(driveSel);

	// Create the file and directory list boxes

	r = fileType->getBounds();
	r2 = driveSel->getBounds();
	r2.top() = r.top() = r.bottom() + 8;
	r2.bottom() = r.bottom() = r.top() + height;
	insert(label = new TLabel(r,fileText,NULL));
	insert(label2  = new TLabel(r2,directoryText,NULL));

	r2.top() = r.top() = r.bottom();
	r2.bottom() = r.bottom() = r.top() + cellHeight*linesPerList + 2*_MVIS_sysLineWidth;
	r.right() -= _MVIS_sysScrollBarWidth;
	r2.right() -= _MVIS_sysScrollBarWidth;
	fileList = new TList(r,TPoint(r.right()-r.left()-2*_MVIS_sysLineWidth,cellHeight));
	directoryList = new TList(r2,TPoint(r2.right()-r2.left()-2*_MVIS_sysLineWidth,cellHeight));
	label->setAttached(fileList);
	label2->setAttached(directoryList);

	// Now create the scroll bars for the lists and attach them to the
	// appropriate lists.

	TRect	r3(r),r4(r2);
	r3.left() = r3.right()-1;
	r3.right() += _MVIS_sysScrollBarWidth;
	r3.bottom() -= 1;
	r4.left() = r4.right()-1;
	r4.right() += _MVIS_sysScrollBarWidth;
	r4.bottom() -= 1;
	TScrollBar *vScroll = new TScrollBar(r3);
	TScrollBar *vScroll2 = new TScrollBar(r4);

	r.top() = r.bottom()-1;
	r.bottom() = r.top() + _MVIS_sysScrollBarWidth;
	r.right() -= 1;
	r2.top() = r2.bottom()-1;
	r2.bottom() = r2.top() + _MVIS_sysScrollBarWidth;
	r2.right() -= 1;
	TScrollBar *hScroll = new TScrollBar(r);
	TScrollBar *hScroll2 = new TScrollBar(r2);

	fileList->setHScroll(hScroll);
	fileList->setVScroll(vScroll);
	directoryList->setHScroll(hScroll2);
	directoryList->setVScroll(vScroll2);

	// Adjust the bounds of the box to fit, including the buttons

	bounds.right() = dialogWidth;
	bounds.bottom() = r.bottom() + adjustBottom + buttonHeight + 20;
	setBounds(bounds);

	// Now add the buttons to the dialog box

	int width = MGL_textWidth(cancelText) + 16;
	int sizex = width + 15;
	r.top() = bounds.bottom() - buttonHeight - 10;
	r.bottom() = r.top() + buttonHeight;
	r.right() =  r.left() + width;
	TButton *okBut = (flags & fdOpenButton) ?
		new TButton(r,openText,cmFileOpen,bfDefault) :
		new TButton(r,saveText,cmFileSave,bfDefault);

	r.left() += sizex;	r.right() +=  sizex;
	TButton *cancelBut = new TButton(r,cancelText,cmCancel,bfNormal);
	TButton *helpBut = NULL;

	if (flags & fdHelpButton) {
		r.left() += sizex;	r.right() +=  sizex;
		helpBut = new TButton(r,helpText,cmHelp,bfNormal);
		}

	// Now insert all of the items into the dialog box in the correct
	// tab-ing order

	if (helpBut)
		insert(helpBut);
	insert(cancelBut);
	insert(okBut);
	insert(directoryList);
	insert(fileList);
	insert(driveSel);
	insert(fileType);
	insert(fileLine);

	// Now insert the scroll bars on top of everything else
	insert(vScroll);	insert(vScroll2);
	insert(hScroll);	insert(hScroll2);

	// Now set the default filename for the dialog box

	directory[0] = filename[0] = '\0';
	setFilename(defFile,true);
}

void TFileDialog::updateFileLine()
/****************************************************************************
*
* Function:		TFileDialog::updateFileLine
*
* Description:	Updates the value in the file input line, with the
*				filename currently selected in the file list box.
*
****************************************************************************/
{
	TPoint		cursor;
	const char	*name;

	fileList->getCurrentCell(cursor);
	fileList->getCell(cursor.x,cursor.y,name);
	setFilename(name);
}

void TFileDialog::changeDirectory()
/****************************************************************************
*
* Function:		TFileDialog::changeDirectory
*
* Description:	Changes to the directory specified by the directory list
*				box, reads in the new filenames for the directory.
*
****************************************************************************/
{
	TPoint		cursor;
	const char	*dir;
	char		path[MAXPATH];

	directoryList->getCurrentCell(cursor);
	directoryList->getCell(cursor.x,cursor.y,dir);

	// Build the name for the new directory to change to

	strcpy(path,driveSel->getText());
	strcat(path,directory);
	if (dir[0] == '.' && dir[1] == '.') {
		// Go back one directory. We do this by removing the last directory
		// entry from the path that we have built so far

		for (int index = strlen(path)-2; index >= 2; index--)
			if (path[index] == '\\' || path[index] == '/') {
				path[index+1] = '\0';
				break;
				}
		}
	else {
		strcat(path,dir);
		strcat(path,"\\");
		}
	strcat(path,fileLine->getText());
	setFilename(path);
	loadFilenames();
}

bool TFileDialog::valid(ushort command)
/****************************************************************************
*
* Function:		TFileDialog::valid
* Parameters:	command	- Command causing the dialog to complete
* Returns:		True if the dialog is valid, false if not
*
* Description:	Determines if the value in the file input box is a valid
*				filename or not. We will not allow the user to quit
*				correctly unless the filename is valid.
*
****************************************************************************/
{
	if (command == cmValid || command == cmCancel)
		return true;

	if (TDialog::valid(command)) {
		if (directoryList->getState() & sfFocused)
			changeDirectory();
		else if (fileList->getState() & sfFocused) {
			updateFileLine();
			return !hasWilds;
			}
		else {
			if (validFilename(fileLine->getText())) {
				setFilename(fileLine->getText());
				if (hasWilds) {
					loadFilenames();
					fileList->select();
					}
				else
					return true;
				}
			}
		}
	return false;
}

void TFileDialog::handleEvent(TEvent& event,phaseType phase)
/****************************************************************************
*
* Function:		TFileDialog::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Event handling routine for the TFileDialog class. Attempts
*				to end the modal operation if the open or save buttons
*				are hit.
*
****************************************************************************/
{
	TDialog::handleEvent(event,phase);

	switch (event.what) {
		case evCommand:
			switch (event.message.command) {
				case cmFileOpen:
				case cmFileSave:
					endModal(event.message.command);
					clearEvent(event);
					break;
				case cmHelp:
					messageBox("Help not implemented yet",
						mfInformation | mfOKButton | mfOKDefault);
					clearEvent(event);
					break;
				}
			break;
		case evBroadcast:
			switch (event.message.command) {
				case cmListCursorChanged:
					if (event.message.infoPtr == fileList) {
						updateFileLine();
						clearEvent(event);
						}
					break;
				case cmListItemSelected:
					if (event.message.infoPtr == fileList) {
						endModal((flags & fdOpenButton) ?
							cmFileOpen : cmFileSave);
						clearEvent(event);
						}
					else if (event.message.infoPtr == directoryList) {
						changeDirectory();
						clearEvent(event);
						}
					break;
				}
			break;
		}
}

ushort TFileDialog::execute()
/****************************************************************************
*
* Function:		TFileDialog::execute
* Returns:		Command ending the execution.
*
* Description:	We overload the execute method to automatically set things
*				up for us. If the contents of the current directory are
*				not loaded, we load them first.
*
****************************************************************************/
{
	lock();							// Lock repaint events
	if (!(flags & fdDirLoaded))
		loadFilenames();
	select(fileLine);				// Focus on the file input line
	unlock();
	return TDialog::execute();
}

void TFileDialog::loadFilenames()
/****************************************************************************
*
* Function:		TFileDialog::loadFilenames
*
* Description:	Loads the dialog box with all the filenames in the
*				current directory.
*
****************************************************************************/
{
	ffblk	blk;
	char	buf[MAXPATH];
	int		i,result;
	bool 	subDir = (strlen(directory) != 1);

	if (flags & fdDirLoaded)	// Don't reload if already loaded
		return;

	filenames.empty();			// Empty filename and directory collections
	directories.empty();

	// Load all of the filenames for the current directory

	result = findfirst(filename,&blk,findAttr);
	while (result == 0) {
		if (!(blk.ff_attrib & FA_DIREC)) {
			filenames.add(new DynStr(blk.ff_name));
			if (lowMemory())
				goto memError;
			}
		result = findnext(&blk);
		}

	fnmerge(buf,driveSel->getText(),directory,"*",".*");
	result = findfirst(buf,&blk,FA_DIREC);
	while (result == 0) {
		if ((blk.ff_attrib & FA_DIREC) && blk.ff_name[0] != '.') {
			directories.add(new DynStr(blk.ff_name));
			if (lowMemory())
				goto memError;
			}
		result = findnext(&blk);
		}

	// Sort all of the filenames and directories, and insert the data
	// into the appropriate list boxes.

	filenames.sort();
	directories.sort();

	fileList->setDataBounds(TRect(0,0,1,filenames.numberOfItems()));
	for (i = 0; i < filenames.numberOfItems(); i++)
		fileList->setCell(0,i,*filenames[i]);
	fileList->setHRange(0,0);
	fileList->setVRange(0,max(0U,filenames.numberOfItems()-1));

	directoryList->setDataBounds(TRect(0,0,1,directories.numberOfItems()+subDir));
	if (subDir)
		directoryList->setCell(0,0,"..");
	for (i = 0; i < directories.numberOfItems(); i++)
		directoryList->setCell(0,i+subDir,*directories[i]);
	directoryList->setHRange(0,0);
	directoryList->setVRange(0,directories.numberOfItems()-1+subDir);

	fileList->clearSelection();
	fileList->selectCell(0,0);
	fileList->setCurrentCell(0,0);
	directoryList->clearSelection();
	directoryList->selectCell(0,0);
	directoryList->setCurrentCell(0,0);

	// Update the file and directory lists

	fileList->update();
	directoryList->update();

	flags |= fdDirLoaded;		// Flag that directory is now loaded
	return;

memError:
	// Flag that a memory error occurred when loading filenames

	filenames.empty();
	directories.empty();
	messageBox(tooManyFilesText,mfError | mfOKButton | mfOKDefault);
	endModal(cmCancel);
}

static bool driveValid(char drive)
/****************************************************************************
*
* Function:		driveValid
* Parameters:	drive	- Drive to check
* Returns:		True if the drive is valid, false if not.
*
* Description:	Simple utility routine to check if a drive is valid.
*
****************************************************************************/
{
	union REGS	regs;

	regs.h.dl = drive - 'A' + 1;	// Drive number (1 = a:, 2 = b: etc)
	regs.h.ah = 0x36;				// Get disk information service
	int86(0x21,&regs,&regs);
	return regs.x.ax != 0xFFFF;		// AX = 0xFFFF if disk is invalid
}

bool TFileDialog::validFilename(const char *filename)
/****************************************************************************
*
* Function:		TFileDialog::validFilename
* Parameters:	filename	- Filename to check for validity
* Returns:		True if the filename is valid.
*
* Description:	To determine this, we first check if the directory is
*				valid. If the 'fdMustExist' flag is set, then we also check
*				to ensure that the file selected also exists. If any errors
*				occur, we pop up a dialog box to that effect and return
*				false.
*
****************************************************************************/
{
	char	path[MAXPATH];
	char	drive[MAXDRIVE];
	char	dir[MAXDIR];
	char	name[MAXFILE];
	char	ext[MAXEXT];
	ffblk	blk;
	bool	invalid = false;

	int flags = fnsplit(filename,drive,dir,name,ext);

	if (!(flags & DRIVE)) {
		strcpy(drive,driveSel->getText());
		if (!(flags & DIRECTORY))
			strcpy(dir,directory);
		}

	// Check to see that the drive/directory exists

	strcpy(path,drive);		strcat(path,dir);
	int len = strlen(path);
	if (path[len-1] == '\\' || path[len-1] == '/')
		path[--len] = '\0';
	if (path[len-1] == ':')
		invalid = !driveValid(toupper(path[len-2]));
	else
		invalid = (findfirst(path,&blk,FA_DIREC) != 0
				  || !(blk.ff_attrib & FA_DIREC));
	if (invalid) {
		messageBox(invalidDriveText,mfError | mfOKButton | mfOKDefault);
		fileLine->select();
		return false;
		}

	// Check that the file exists if required. If the filename has wildcards,
	// then it is valid.

	if (!(flags & WILDCARDS) && (TFileDialog::flags & fdMustExist)) {
		strcat(path,"\\");
		strcat(path,name);
		strcat(path,ext);
		if (findfirst(path,&blk,findAttr) != 0 || (blk.ff_attrib & FA_DIREC)) {
			messageBox(invalidFileText,mfError | mfOKButton | mfOKDefault);
			fileLine->select();
			fileLine->setText(fileLine->getText());
			return false;
			}
		}

	return true;
}

void TFileDialog::setFilename(const char *filename,bool loadCWD)
/****************************************************************************
*
* Function:		TFileDialog::setFilename
* Parameters:	filename	- New filename.
*				loadCWD		- True if we should load the CWD
*
* Description:	Sets the current filename for the dialog box to 'name'. We
*				split up the name into it's component parts, storing the
*				file and extension information in the file input line,
*				and the directory information in the directory string.
*
*				If 'loadCWD' is true, and no directory is specified in the
*				filename, the current working directory will be loaded
*				and used to build the full pathname, otherwise the already
*				loaded values will be used.
*
****************************************************************************/
{
	char	cwd[MAXPATH];
	char	cwd_drive[MAXDRIVE];
	char	cwd_dir[MAXDIR];
	char	drive[MAXDRIVE];
	char	name[MAXFILE];
	char	ext[MAXEXT];
	ffblk	blk;

	hasWilds = false;
	int flags = fnsplit(filename,cwd_drive,cwd_dir,name,ext);

	if (flags & DRIVE && stricmp(drive,cwd_drive) != 0) {
		strcpy(drive,cwd_drive);
		loadCWD = true;
		}
	else
		strcpy(drive,driveSel->getText());
	if (flags & DIRECTORY)
		strcpy(directory,cwd_dir);

	if (loadCWD && (!(flags & DRIVE) || !(flags & DIRECTORY))) {
		if (drive[0] == '\0')
			_getdcwd(0,cwd,MAXPATH);
		else
			_getdcwd(toupper(drive[0])-'A'+1,cwd,MAXPATH);

		// _getdcwd will not append the / character to the path if a
		// directory component is present. If the directory is the root
		// directory, it will store the backslash.

		int len = strlen(cwd);
		if (cwd[len-1] == '\\' || cwd[len-1] == '/')
			cwd[--len] = '\0';
		strcat(cwd,"\\*.*");
		fnsplit(cwd,cwd_drive,cwd_dir,NULL,NULL);
		if (!(flags & DRIVE))
			strcpy(drive,cwd_drive);
		if (!(flags & DIRECTORY)) {
			strcpy(drive,cwd_drive);
			strcpy(directory,cwd_dir);
			}
		}

	if (!(flags & FILENAME)) {
		name[0] = '*';					// Insert default extension
		name[1] = '\0';
		ext[0] = '.';
		ext[1] = '*';
		ext[2] = '\0';
		hasWilds = true;
		}
	hasWilds = hasWilds || (flags & WILDCARDS);

	// Now set the filename and extension of the file input line.

	strupr(drive);	strupr(directory);
	strupr(name); 	strupr(ext);
	strcpy(cwd,name);
	strcat(cwd,ext);
	fileLine->setText(cwd);
	driveSel->setText(drive);

	// Save the complete filename in the filename variable. If the name
	// has changed, then flag that the directory contents will need to be
	// reloaded.

	fnmerge(cwd,drive,directory,name,ext);
	if (strcmp(cwd,TFileDialog::filename) != 0) {
		TFileDialog::flags &= ~fdDirLoaded;
		strcpy(TFileDialog::filename,cwd);
		}

	if (!hasWilds && (findfirst(cwd,&blk,findAttr | FA_DIREC) == 0) && (blk.ff_attrib & FA_DIREC)) {
		// The selected file is actually a directory, so load all the
		// files in this directory

		strcat(cwd,"\\*.*");
		setFilename(cwd);
		}
}
