/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: safetyp.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Module to overload the global operator new for MegaVision.
*               This provides a simple saftey pool mechanism for allocating
*				main memory.
*
* $Id: safetyp.cpp 1.2 1994/03/09 11:50:31 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tfontmgr.hpp"
#include <iostream.h>
#include <stdlib.h>

//---------------------------------------------------------------------------
// Class to manage the safety pool. A single global instantiation of this
// class is provided to automatically initialise the safety pool mechanism
// and program inception.
//---------------------------------------------------------------------------

class SafetyPool {
protected:
	void	*safetyPool;			// Pointer to allocated safety pool

public:
			// Constructor
			SafetyPool()	{ resize(); };

			// Method to re-size the safety pool
			void resize(size_t size = 4096);

			// Method to determine if safety pool is exhausted
			bool exhausted()
				{ return safetyPool == NULL; };
	};

/*----------------------------- Implementation ----------------------------*/

PRIVATE	SafetyPool		safetyPool;
PRIVATE	bool			_majorConsumer = false;

void SafetyPool::resize(size_t size)
/****************************************************************************
*
* Function:		SafetyPool::resize
* Parameters:	size	- New size for the safety pool.
*
****************************************************************************/
{
	if (safetyPool != NULL)
		free(safetyPool);
	safetyPool = (size == 0) ? NULL : malloc(size);
}

void *operator new(size_t size)
/****************************************************************************
*
* Function:		operator new
* Parameters:	size	- Size of block to allocate
* Returns:		Pointer to the allocated block
*
* Description:	Overloaded memory allocation routine. We attempt to allocate
*				the memory initially by calling malloc. If this fails, we
*				free the safety pool and try again. If this also fails
*				we bomb out with an error message.
*
****************************************************************************/
{
	void *temp = malloc(size);

	// Only dip into the safety pool if the current memory allocation is
	// _not_ by a major consumer. Major consumer's will always allocate
	// more than the safety pool, and require the result of the operator
	// new to return NULL if the memory allocation fails.

	if (!_majorConsumer) {
		while (temp == NULL) {
			if (safetyPool.exhausted()) {
				MGL_exit();
				cerr << "Out of memory - saftey pool exhausted!\n";
				exit(EXIT_FAILURE);
				}

			if (fontManager.purgeLRUFont() == -1) {
				// Resize the safety pool and try again

				safetyPool.resize(0);
				temp = malloc(size);
				}
			}
		}

	return temp;
}

void operator delete(void *p)
/****************************************************************************
*
* Function:		operator delete
* Parameters:	p	- Pointer to block to delete
*
* Description:	Overloaded memory deletion routine. We free the memory for
*				the block, and attempt to reallocate the safety pool if
*				it was freed.
*
****************************************************************************/
{
	free(p);
	if (safetyPool.exhausted())
		safetyPool.resize();
}

void majorConsumer()
/****************************************************************************
*
* Function:		majorConsumer
*
* Description:	Sets the major consumer flag to true, so memory allocations
*				will not dip into the safety pool but return NULL on
*				failure.
*
****************************************************************************/
{
	_majorConsumer = true;
}

void normalConsumer()
/****************************************************************************
*
* Function:		normalConsumer
*
* Description:	Resets the major consumer flag to normal.
*
****************************************************************************/
{
	_majorConsumer = false;
}

bool lowMemory()
/****************************************************************************
*
* Function:		lowMemory
*
* Description:	Returns true if memory allocation reached into the safety
*				pool.
*
****************************************************************************/
{
	return safetyPool.exhausted();
}
