             
/*
**  Keyboard Emulation Functions.
**
**  Copyright 1993-97 by Paul D. Burgin. All rights reserved.
*/

/* Includes actions for some function keys too. */

#include "conio.h"
#include "dos.h"
#include "bios.h"
#include "graphics.h"
#include "stdio.h"
#include "dir.h"
#include "string.h"
#include "ctype.h"
#include "setjmp.h"

#include "build.h"
#include "types.h"
#include "extern.h"
#include "key.h"
#include "macros.h"

/* Last scan code included in offset/mask tables (+1). */
#define LAST_USED 84

/* State of dragon shift lock. */
unsigned char last_shift = 0;

/* Prototype for hard reset jump buffer. */
extern jmp_buf hard_reset;

/* Offsets from start of rollover tables for IBM scan codes. */
unsigned char keylocs[LAST_USED] =
{
	0,	3,	2,	3,	4,	5,	6,	7,
	8,	1, 	2,	1,	6,	6,	6,	7,
	2,	8,	6,	3,	5,	2,	6,	2,
	8,	1,	5,	7,	1,	0,	2,	4,
	5,	7,	8,	1,	3,	4,	5,	4,
	1,	3,	0,	4,	3,	1,	4,	7,
	3,	7,	6,	5,	7,	8,	0,	3,
	0,	8,	0,	0,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	0,	2,
	4,	1,	6,	6,	6,	7,	4,	3,
	5,	1,	2,	5
};

/* Rollover table masks for all IBM scan codes. */
unsigned char keyands[LAST_USED] =
{
	255,	191,	254,	254,	254,	254,	254,	254,
	254,	253,	253,	254,	253,	253,	223,	223,
	239,	239,	251,	239,	239,	223,	239,	247,
	247,	239,	223,	223,	191,	255,	251,	239,
	251,	251,	251,	247,	247,	247,	247,	253,
	251,	253,	255,	254,	223,	223,	251,	239,
	251,	247,	247,	253,	253,	253,	255,	253,
	255,	223,	255,	255,	255,	255,	255,	255,
	255,	255,	255,	255,	255,	255,	255,	191,
	223,	251,	253,	223,	254,	223,	253,	191,
	223,	191,	247,	251
};

/* History buffer. */
unsigned char history_len = 0;
unsigned char history_temp[PRESET_LEN+1];
unsigned char history_buffer[PRESET_LEN+1]
	= "if(peek(49152)=32)thenclear200,20000:exec49152^m"; /* Run Alldream! */

/* Length of activated keyboard buffer. */
unsigned char key_buf = 0;

/* Pointer to current string of bufferred keys. */
unsigned char *buf_ptr;

/* Flags used to differenciate between certain keys. */
boolean rightkey, endkey, pgdnkey, upkey;

/* Use bios to read keyboard (also maintains and activates buffering). */
unsigned int getkey(void)
{
	register unsigned char	return_val;
	register unsigned int	bios_return;

	/* Any keys in an activated buffer? */
	if (key_buf > 0)
	{
		/* Get next key from buffer. */
		key_buf--;

		/* Translate special characters. */
		switch(return_val = *buf_ptr++)
		{
			case '_':	return_val = ' ';
						break;

			case '^':   if (key_buf > 0)
							key_buf--;
						return_val = toupper(*buf_ptr++);
						if ((return_val >= '@') && (return_val <= '['))
							return_val -= '@';
						break;
		}

		if (return_val == 0x00)
		{
			/* Read next char from buffer for extended characters. */
			if (key_buf > 0)
				key_buf--;
			bios_return = ((*buf_ptr++) << 8);
		}
		else
		{
			/* Normal keys; set scancode as 0x1c (return key). */
			bios_return = (0x1c00 | return_val);
		}
	}
	else
	{
		/* No buffer active so read keyboard for next key. */
		bios_return = bioskey(0);
	}

	/* Queue history or user defined macros (also check for quit). */
	switch (bios_return)
	{
		case 0x1c0d:	/* Return */
						if (history_len > 0)
						{
							history_temp[history_len] = '\0';
							strcpy(history_buffer,history_temp);
							history_len = 0;
						}
						return(bios_return);

		case 0x1c0a:	/* Ctrl-Return */
						buf_ptr = history_buffer;
						goto do_history;

		case 0x6800:
		case 0x6900:
		case 0x6a00:
		case 0x6b00:
		case 0x6c00:
		case 0x6d00:
		case 0x6e00:
		case 0x6f00:
		case 0x7000:
		case 0x7100:	/* Shift-F1 to Shift-F10 */
						buf_ptr = presets[(bios_return >> 8) - 0x68];
						goto do_history;

		case 0x4000:	/* F6 */
						if (strlen(presets[10]) > 0)
						{
							buf_ptr = presets[10];
							goto do_history;
						}
						break;

		/* Basic keywords - top row of keyboard. */
		case 0x1000:	buf_ptr = keywords[16];	goto do_history;
		case 0x1100:	buf_ptr = keywords[22];	goto do_history;
		case 0x1200:	buf_ptr = keywords[4];	goto do_history;
		case 0x1300:	buf_ptr = keywords[17];	goto do_history;
		case 0x1400:	buf_ptr = keywords[19];	goto do_history;
		case 0x1500:	buf_ptr = keywords[24];	goto do_history;
		case 0x1600:	buf_ptr = keywords[20];	goto do_history;
		case 0x1700:	buf_ptr = keywords[8];	goto do_history;
		case 0x1800:	buf_ptr = keywords[14];	goto do_history;
		case 0x1900:	buf_ptr = keywords[15];	goto do_history;

		/* Basic keywords - middle row of keyboard. */
		case 0x1e00:	buf_ptr = keywords[0];	goto do_history;
		case 0x1f00:	buf_ptr = keywords[18];	goto do_history;
		case 0x2000:	buf_ptr = keywords[3];	goto do_history;
		case 0x2100:	buf_ptr = keywords[5];	goto do_history;
		case 0x2200:	buf_ptr = keywords[6];	goto do_history;
		case 0x2300:	buf_ptr = keywords[7];	goto do_history;
		case 0x2400:	buf_ptr = keywords[9];	goto do_history;
		case 0x2500:	buf_ptr = keywords[10];	goto do_history;
		case 0x2600:	buf_ptr = keywords[11];	goto do_history;

		/* Basic keywords - bottom row of keyboard. */
		case 0x2c00:	buf_ptr = keywords[25];	goto do_history;
		case 0x2d00:	buf_ptr = keywords[23];	goto do_history;
		case 0x2e00:	buf_ptr = keywords[2];	goto do_history;
		case 0x2f00:	buf_ptr = keywords[21];	goto do_history;
		case 0x3000:	buf_ptr = keywords[1];	goto do_history;
		case 0x3100:	buf_ptr = keywords[13];	goto do_history;
		case 0x3200:	buf_ptr = keywords[12];	goto do_history;

		do_history:		key_buf = strlen(buf_ptr);
						return(getkey());
	}

	/* Check for ^Q and ^X. */
	switch (bios_return & 0xff)
	{
		case 17:
		case 24:	quit("",4);
	}

	/* Enter character into history buffer. */
	if (history_len < PRESET_LEN)
	{
		/* Translate special characters back to internal format. */
		switch(return_val = (bios_return & 0x00ff))
		{
			case 0x00:	if ((history_len + 2) < PRESET_LEN)
						{
							history_temp[history_len++] = '^';
							history_temp[history_len++] = '@';
							history_temp[history_len++] = (bios_return >> 8);
						}
						break;

			case '^':
			case '_':	if (history_len >= (PRESET_LEN-1))
							break;
						history_temp[history_len++] = '^';
			default:	history_temp[history_len++] = return_val;
						break;
		}
	}

	return(bios_return);
}

/* Wait for Ctrl and Alt keys to be released. */
void end_control()
{
	if (new_int9_set)
		while (keys[kLEFTCTRL] | keys[kRIGHTCTRL]
			| keys[kLEFTALT] | keys[kRIGHTALT]);
	else
		while ((bioskey(2) & 0x0c) != 0x00);
}

/* Emulator function keys and menus. */
boolean emu_keys(unsigned char localkey)
{
	/* Don't do anything until Ctrl & Alt keys are released. */
	end_control();

	/* Action keys. */
	switch(localkey)
	{
		/* F1 = help */
		case 0x3b:	help_screen();
					break;

		/* F2 = TextView menu */
		case 0x3c:	switch (selection_box(9,25,7,FALSE,FALSE,FALSE,99,
						"     TEXT VIEWER MENU","",
						" 1 - Reference Manual",
						" 2 - Cassette Directory",
						" 3 - Archive Directory",
						" 4 - Games Directory",
						" 5 - Snapshot Directory",
						" 6 - Cartridge Directory",
						" 7 - Base Directory"))
					{
						case 1:	shell_escape(4); break;
						case 2:	shell_escape(2); break;
						case 3:	shell_escape(1); break;
						case 4:	shell_escape(0); break;
						case 5:	shell_escape(5); break;
						case 6:	shell_escape(6); break;
						case 7:	shell_escape(7); break;
					}
					break;

		/* F3 = file management menu */
		case 0x3d:  file_menu();
					break;

		/* F4 = options */
		case 0x3e:  opt_screen();
					break;

		/* F5 = video menu */
		case 0x3f:	video_menu();
					break;

		/* F7 = function keys */
		case 0x41:	function_opt();
					break;

		/* F8 = control menu */
		case CONTROL_MENU:
					switch (selection_box(6,16,4,FALSE,FALSE,FALSE,99,
						"  CONTROL MENU","",
						"1 - Warm Reboot",
						"2 - Cold Reboot",
						"3 - MS-DOS Shell",
						"4 - Debugger"))
					{
						case 2:	memory[0x71] = 0;
						case 1:	reset_hardware();
								longjmp(hard_reset,1);
								break;

						case 3:	shell_escape(3);
								break;

						case 4: emu_keys(0x86);
								break;
					}
					break;

		/* F9 = assembler menu */
		case 0x85:	edit_key(FALSE);
					break;

		/* F10 = switch to debugger */
		case 0x86:	in_debug = TRUE;
					await_breakpoint = FALSE;
					break;

		/* Ctrl-F1 = display pmode1:screen1,0 graphic screen at $600 */
		case 0x5e:	new_screen_size = 4;
					new_screen_base = 0x600;
					memory[0xff22] &= 0xc7;
					memory[0xff22] |= 0xc0;
					execute_vmode(TRUE); /* stuff the scroll lock! */
					break;

		/* Ctrl-F2 = display pmode2:screen1,1 graphic screen at $600 */
		case 0x5f:	new_screen_size = 5;
					new_screen_base = 0x600;
					memory[0xff22] &= 0xdf;
					memory[0xff22] |= 0xd8;
					execute_vmode(TRUE); /* stuff the scroll lock! */
					break;

		/* Ctrl-F3 = display pmode3:screen1,0 graphic screen at $600 */
		case 0x60:	new_screen_size = 6;
					new_screen_base = 0x600;
					memory[0xff22] &= 0xe7;
					memory[0xff22] |= 0xe0;
					execute_vmode(TRUE); /* stuff the scroll lock! */
					break;

		/* Ctrl-F4 = display pmode4:screen1,1 graphic screen at $600 */
		case 0x61:	new_screen_size = 6;
					new_screen_base = 0x600;
					memory[0xff22] |= 0xf8;
					execute_vmode(TRUE); /* stuff the scroll lock! */
					break;

		/* Ctrl-F5 = toggle video palette */
		case 0x62:	set_vmode(memory[0xff22] ^= 0x08);
					break;

		/* Ctrl-F6 = toggle video 2 or 4 colours */
		case 0x63:	toggle_2or4_colours();
					break;

		/* Ctrl-F7 = toggle pixel width */
		case 0x64:	toggle_pixel_width();
					break;

		/* Ctrl-F8 = display text screen at $400 */
		case 0x65:  new_screen_size = 0;
					new_screen_base = 0x400;
					memory[0xff22] &= 0x0f;
					execute_vmode(TRUE); /* stuff the scroll lock! */
					break;

		/* Ctrl-F9 = call editor */
		case 0x66:	edit_key(TRUE);
					break;

		/* Ctrl-F10 = info screen */
		case 0x67:	inf_screen();
					break;

		/* Shift-F5 = decrease VDG memory address (by 0x0200) */
		case 0x58:	new_screen_base -= 0x0200;
					break;

		/* Shift-F6 = increase VDG memory address (by 0x0200) */
		case 0x59:	new_screen_base += 0x0200;
					break;

		/* Shift-F7 = decrease VDG memory address (by 1 screen) */
		case 0x5a:	new_screen_base -= vdg_sizes[new_vmode + 1];
					break;

		/* Shift-F8 = increase VDG memory address (by 1 screen) */
		case 0x5b:	new_screen_base += vdg_sizes[new_vmode + 1];
					break;

		/* Key not executed. */
		default:	return(FALSE);
	}
	return (TRUE);
}

/* Service pressed function key F1 to F10, or CONTROL (in real mode). */
void service_emu_keys(void)
{
	/* Get state of special keys. */
	boolean shft  = keys[kLEFTSHIFT] | keys[kRIGHTSHIFT];
	boolean alt   = keys[kLEFTALT]   | keys[kRIGHTALT];
	boolean cntrl = keys[kLEFTCTRL]  | keys[kRIGHTCTRL];

	/* Note: We need to wait for keys to be released before changing   */
	/* the keyboard interrupt handler or DOS gets rather confused and  */
	/* thinks keys are pressed when they aren't (as does new handler). */

	/* Check for Ctrl-Alt combination. */
	if (cntrl && alt)
		emu_keys(CONTROL_MENU);

	/* Check for quit. */
	else if ((keys[kQ] && (cntrl)) || (keys[kX] && (cntrl)))
	{
		while (keys[kQ]);
		while (keys[kX]);
		end_control();
		quit("",4);
	}

	/* Call handler subroutine using key scan codes. */
	else if (keys[kF1])
	{
		while (keys[kF1]);
		if (cntrl)
			emu_keys(0x5e);
		else
			emu_keys(0x3b);
	}
	else if (keys[kF2])
	{
		while (keys[kF2]);
		if (cntrl)
			emu_keys(0x5f);
		else
			emu_keys(0x3c);
	}
	else if (keys[kF3])
	{
		while (keys[kF3]);
		if (cntrl)
			emu_keys(0x60);
		else
			emu_keys(0x3d);
	}
	else if (keys[kF4])
	{
		while (keys[kF4]);
		if (cntrl)
			emu_keys(0x61);
		else
			emu_keys(0x3e);
	}
	else if (keys[kF5])
	{
		while (keys[kF5]);
		if (cntrl)
			emu_keys(0x62);
		else if (shft)
			emu_keys(0x58);
		else
			emu_keys(0x3f);
	}
	else if (keys[kF6])
	{
		while (keys[kF6]);
		if (cntrl)
			emu_keys(0x63);
		else if (shft)
			emu_keys(0x59);
	}
	else if (keys[kF7])
	{
		while (keys[kF7]);
		if (cntrl)
			emu_keys(0x64);
		else if (shft)
			emu_keys(0x5a);
		else
			emu_keys(0x41);
	}
	else if (keys[kF8])
	{
		while (keys[kF8]);
		if (cntrl)
			emu_keys(0x65);
		else if (shft)
			emu_keys(0x5b);
		else
			emu_keys(0x42);
	}
	else if (keys[kF9])
	{
		while (keys[kF9]);
		if (cntrl)
			emu_keys(0x66);
		else
			emu_keys(0x85);
	}
	else if (keys[kF10])
	{
		while (keys[kF10]);
		if (cntrl)
			emu_keys(0x67);
		else
			emu_keys(0x86);
	}

	/* Wait for keys to be released, so that we don't just come in again. */
	if (new_int9_set)
		while (keys[kF1] | keys[kF2] | keys[kF3] | keys[kF4] | keys[kF5]
			| keys[kF6] | keys[kF7] | keys[kF8] | keys[kF9] | keys[kF10]);
}

/* Create keyboard rollover table for hardware keyboard emulation. */
void create_matrix(void)
{
	unsigned char tandytemp;
	unsigned char shift = keys[kLEFTSHIFT] | keys[kRIGHTSHIFT]
						| keys[kLEFTALT]   | keys[kRIGHTALT];

	/* Refer to indivitual key booleans to generate state table. */
	memory[MATRIX+1] = ~(keys[k0]
						| keys[kKEYPAD0]
						| (keys[k8] << 1)
						| (keys[kTICK] << 2) /* @ */
						| (keys[kKEYPAD7] << 3)
						| (keys[kH] << 3)
						| (keys[kP] << 4)
						| (keys[kX] << 5)
						| (keys[kPGDN] << 6)
						| (keys[kKEYPADENTER] << 6)
						| (keys[kENTER] << 6));

	memory[MATRIX+2] = ~(keys[k1]
						| (keys[k9] << 1)
						| (keys[kA] << 2)
						| (keys[kI] << 3)
						| (keys[kINS] << 3)
						| (keys[kQ] << 4)
						| (keys[kY] << 5)
						| (keys[kTAB] << 6)
						| (keys[kHOME] << 6));

	memory[MATRIX+3] = ~(keys[k2]
						| (keys[kAPOS] << 1)
						| (keys[kBACKSLASH] << 1)
						| (keys[kB] << 2)
						| (keys[kJ] << 3)
						| (keys[kR] << 4)
						| (keys[kZ] << 5)
						| (keys[kEND] << 6)
						| (keys[kESC] << 6));

	memory[MATRIX+4] = ~(keys[k3]
						| (keys[kSEMICOLON] << 1)
						| (keys[kC] << 2)
						| (keys[kK] << 3)
						| (keys[kS] << 4)
						| (keys[kKEYPAD8] << 5)
						| (keys[kUARROW] << 5));

	memory[MATRIX+5] = ~(keys[k4]
						| (keys[kCOMMA] << 1)
						| (keys[kDEL] << 2)
						| (keys[kD] << 2)
						| (keys[kL] << 3)
						| (keys[kT] << 4)
						| (keys[kKEYPAD2] << 5)
						| (keys[kDARROW] << 5));

	memory[MATRIX+6] = ~(keys[k5]
						| (keys[kMINUS] << 1)
						| (keys[kKEYPADMINUS] << 1)
						| (keys[kKEYPAD9] << 2)
						| (keys[kE] << 2)
						| (keys[kM] << 3)
						| (keys[kU] << 4)
						| (keys[kBACKSPACE] << 5)
						| (keys[kKEYPAD4] << 5)
						| (keys[kLARROW] << 5));

	memory[MATRIX+7] = ~(keys[k6]
						| (keys[kPERIOD] << 1)
						| (keys[kKEYPADDEL] << 1)
						| (keys[kKEYPAD3] << 2)
						| (keys[kF] << 2)
						| (keys[kN] << 3)
						| (keys[kV] << 4)
						| (keys[kKEYPAD6] << 5)
						| (keys[kRARROW] << 5));

	memory[MATRIX+8] = ~(keys[k7]
						| (keys[kSLASH] << 1)
						| (keys[kKEYPADSLASH] << 1)
						| (keys[kKEYPAD1] << 2)
						| (keys[kG] << 2)
						| (keys[kO] << 3)
						| (keys[kW] << 4)
						| (keys[kSPACE] << 5));

	/* Keys which automatically press SHIFT key. */
	if (keys[kEQUAL])
	{
		memory[MATRIX+6] &= 253;
		shift = TRUE;
	}
	else if (keys[kLBRACKET])
	{
		memory[MATRIX+1] &= 253;
		shift = TRUE;
	}
	else if (keys[kRBRACKET])
	{
		memory[MATRIX+2] &= 253;
		shift = TRUE;
	}
	else if (keys[kHASH])
	{
		memory[MATRIX+4] &= 254;
		shift = TRUE;
	}
	else if (keys[kPGUP])
	{
		memory[MATRIX+1] &= 251;
		shift = TRUE;
	}
	else if (keys[kCAPSLOCK])
	{
		memory[MATRIX+1] &= 254;
		shift = TRUE;
	}
	else if (keys[kNUMLOCK])
	{
		memory[MATRIX+2] &= 191;
		shift = TRUE;
	}
	else if (keys[kKEYPADSTAR])
	{
		memory[MATRIX+3] &= 253;
		shift = TRUE;
	}
	else if (keys[kKEYPADPLUS])
	{
		memory[MATRIX+4] &= 253;
		shift = TRUE;
	}

	/* Add shift state. */
	memory[MATRIX+8] &= ~(shift << 6);

	/* Change column ordering for Tandy keyboard; a bit slow this. */
	if (arch == tandy)
	{
		for (tandytemp = 1; tandytemp < 9; tandytemp++)
		{
			shift = memory[MATRIX+tandytemp];
			memory[MATRIX+tandytemp] = (((shift >> 2) & 0x0f)
				| ((shift << 4) & 0x30) | (shift & 0xc0));
		}
	}
}

/* Keyboard input function (translates some keys to Dragon codes). */
unsigned char getch2(boolean just_getch)
{
	unsigned char localkey;
	unsigned int  biosk;

	/* Restore original INT9 handler before doing this. */
	if (new_int9_set)
		Set_Old_Int9();

	/* Wipe BASIC key scan table. */
	if (!just_getch)
	{
		far_set_mem8(338,255);
		far_set_mem8(339,255);
		far_set_mem8(340,255);
		far_set_mem8(341,255);
		far_set_mem8(342,255);
		far_set_mem8(343,255);
		far_set_mem8(344,255);
		far_set_mem8(345,255);
	}

	/* Get scan code of next key. */
	biosk = getkey();
	localkey = biosk & 0xff;
	biosk >>= 8;

	/* Create entry in BASIC scan table. */
	if (!just_getch && (localkey < LAST_USED))
		far_set_mem8(337 + keylocs[biosk], keyands[biosk]);

	/* Re-map some extended keys. */
	if (localkey == 0)
	{
		switch (localkey = biosk)
		{
			/* Re-map arrow keys. */
			case 0x4b:	localkey = 8;
						break;
			case 0x4d:	localkey = 9;
						rightkey = TRUE;
						break;
			case 0x50:	localkey = 10;
						break;
			case 0x48:	localkey = 94;
						upkey = TRUE;
						break;

			/* Re-map SHIFT+F1-F4 = shift+arrow keys. */
			case 0x54:	localkey = 21;
						break;
			case 0x55:	localkey = 95;
						break;
			case 0x56:	localkey = 91;
						break;
			case 0x57:	localkey = 93;
						break;

			/* END = break key mimic */
			case 0x4f:	localkey = 3;
						endkey = TRUE;
						break;

			/* HOME = clear key mimic */
			case 0x47:	localkey = 12;
						break;

			/* PG UP = pause (shift @) key mimic */
			case 0x49:	localkey = 19;
						break;

			/* PG DN = another enter key mimic */
			case 0x51:	localkey = 13;
						pgdnkey = TRUE;
						break;

			/* INS = edit insert (another I key) mimic */
			case 0x52:	localkey = 'i';
						break;

			/* DEL = edit delete (another D key) mimic */
			case 0x53:	localkey = 'd';
						break;

			/* SHIFT-TAB = ESC mimic */
			case 0x0f:	localkey = 0x1b;
						break;

			/* Transform F9 and F10 to unprintable chars. */
			case 0x43:
			case 0x44:
			case 0x70:
			case 0x71:	localkey += 0x42;

			/* Interpret emulator special keys. */
			default:    if (!just_getch)
						{
							if (emu_keys(localkey))
								localkey = 0;
						}
						break;
		}
	}
	/* ESC = break key mimic */
	else if (localkey == 0x1b)
		localkey = 3;

	/* Set overall AND entry in BASIC table. */
	if (!just_getch)
	{
		far_set_mem8(337,
			memory[338] & memory[339] & memory[340] & memory[341]
				& memory[342] & memory[343] & memory[344] & memory[345]);

		/* Read shift keys - set entry in BASIC table. */
		last_shift = (bioskey(2) & 0x43);
		if (last_shift)
			far_set_mem8(337, memory[337] & 191);

		/* Set dragon shift lock. */
		if (last_shift & 0x40)
		{
			if (memory[0x0149] != 0)
				memory[0x0149] = 0;
		}
		else
		{
			if (memory[0x0149] != 255)
				memory[0x0149] = 255;
		}
	}

	return(localkey);
}

/*                 Pseudo Hardware Keyboard Emulation                 */
/*                                                                    */
/* This assembles a rollover state table based upon the IBM keyboard  */
/* scan codes. All keys received within a certain duration are placed */
/* into the table, and then it is cleared after the Dragon has read   */
/* from it a specified number of times. After this it remains dead    */
/* awhile to avoid debounce. Of course every program needs different  */
/* settings because they may read different numbers of keys, and use  */
/* different scanning algorithms.                                     */

/* Create keyboard matrix for use with pseudo-hardware emulation. */
void old_create_matrix(void)
{
	unsigned int biosk;

	/* Wipe matrix before making new entries. */
	memory[MATRIX+1] = memory[MATRIX+2]
		= memory[MATRIX+3] = memory[MATRIX+4]
		= memory[MATRIX+5] = memory[MATRIX+6]
		= memory[MATRIX+7] = memory[MATRIX+8]
		= 0xff;

	/* Any key waiting? */
	if (iskey())
	{
		/* Get keypress. */
		drag_usedkey = TRUE;
		biosk = getkey();
		keyin = biosk & 0xff;
		biosk >>= 8;

		/* Action any special keys. */
		if (keyin == 0)
		{
			switch (keyin = biosk)
			{
				/* Transform F9 and F10 to unprintable chars. */
				case 0x43:
				case 0x44:
				case 0x70:
				case 0x71:	keyin += 0x42;
							break;
			}
			if (emu_keys(keyin))
				keyin = 0;
		}

		/* Clear appropriate bit in matrix. */
		if (biosk < LAST_USED)
			memory[MATRIX+keylocs[biosk]] &= keyands[biosk];
	}

	/* Set-up overall AND of all matrix entries. */
	memory[MATRIX] = memory[MATRIX+1]
						& memory[MATRIX+2]
						& memory[MATRIX+3]
						& memory[MATRIX+4]
						& memory[MATRIX+5]
						& memory[MATRIX+6]
						& memory[MATRIX+7]
						& memory[MATRIX+8];
}

/* Scan keyboard at PIA level and set shift status. */
void old_scan_keys(void)
{
	old_create_matrix();

	/* Read shift keys. */
	last_shift = (bioskey(2) & 0x43);
	if (last_shift)
	{
		memory[MATRIX]   &= 191;
		memory[MATRIX+8] &= 191;
	}
}
