indexing

   title:       "Tables: containers with two element entries (element or",
                "item itself and the associated key) implementing a FIFO",
                "policy and supporting traversal in the both directions";
   cluster:     "containers [contners]";
   project:     "The Universal Simple Container Library (USCL)";
   copyright:   "Frieder Monninger & Alexei Shestialtynov, 1995";
   author:      "Frieder Monninger (fm@eiffel.de)",
                "Alexei Shestialtynov (alexei@eiffel.ru)";
   original:    31,Aug,95;
   version:     1.0;
   last_change:
   approved_by:
   approved:
   key:         doubly_linked_list, two_way_traversable;
   done_at:     "SiG Computer (info@eiffel.ru)";
   extrnl_name: "table.e"

class TABLE [G, H]

   inherit
      KEYED_CONTAINER [G, H]
         end;

      TRAVERSABLE
         end

   creation
      make

   ---------------------------------------------------------------------------
   feature -- Operations

      allocate_automatically is
            -- switches the table to the mode of automatic memory allocation
            -- for new slots; it can be necessary only if 'allocate_manually'
            -- was called before because, by default, all types of the tables
            -- allocate memory automatically
         do
            no_reallocation := false
         end;

      allocate_manually (new_capacity: INTEGER) is
            -- forbids the table to allocate memory for new slots automatically
            -- and sets the total amount of slots allocated in the table to be
            -- equal to 'new_capacity' - being negligent after that in manual
            -- allocating extra slots for new entries to be put might lead to
            -- the exception of type SCIF ("container is full")
            --
         require
            no_obstacle:     not is_protected;
            proper_argument: new_capacity >= count and then new_capacity > 0
         do
            if not is_protected and then new_capacity >= count then

               no_reallocation := true;
               if new_capacity > capacity then
                  extend (new_capacity)
               elseif new_capacity < capacity then
                  shrink (new_capacity)
               end

            elseif is_protected then
               fault (f_allocate, errhlg.sc_is_protected)
            else
               fault (f_allocate, errhlg.sc_loss_of_data)
            end
         end;

      make (is_unique: BOOLEAN) is
            -- tells the container if the keys of its entries must be unigue
         local
            init_size: INTEGER

         do
            unique_keys := is_unique;

            init_size := default_size + counterweight;
            check init_size > 0 end;

            !!store.make (1, init_size);
            !CHAINING_WITH_KEYS [H]!guide.make (Current, init_size)
         end;

      merge (x: like Current) is
            -- puts all entries from 'other' into the table
         require
            no_obstacle:    not is_protected;
            valid_argument: x /= Void
         local
            csr: CURSOR

         do
            if not is_protected then

               from
                  csr := x.cursor
               until
                  csr.is_finished
               loop
                  put (x.item_at (csr), x.key_at (csr));
                  csr.forth
               end

            else
               fault (f_merge, errhlg.sc_is_protected)
            end

         ensure
            well_done: count = old count + x.count
                       and then count = guide.count
         end;

      put (x: G; k: H) is
            -- adds the new entry into the structure
         require else
            valid_item:  x /= Void;
            valid_key:   k /= Void;
            no_obstacle: not is_protected;
            uniqueness:  unique_keys implies not has_key (k)
         local
            pos: INTEGER

         do
            if not is_protected then
               if unique_keys then
                  pos := find_first (k)
               end;

               If pos = 0 then
                  guide.get_free;
                  pos := guide.free;

                  if pos = 0 then
                     extend (0);
                     guide.get_free;
                     pos := guide.free;

                     if pos = 0 then
                        fault (f_put, errhlg.sc_is_full)
                     end
                  end;

                  store.put (x, pos);
                  guide.put_key (k, pos);

                  count := count + 1
               else
                  fault (f_put, errhlg.sc_key_duplication)
               end

            else
               fault (f_put, errhlg.sc_is_protected)
            end

         ensure then
            well_done: not empty and then has_key (k)
                       and then count = guide.count
         end;

      remove (x: G; k: H) is
            -- deletes the entry from the table
         require
            no_obstacle: not is_protected;
            valid_item:  x /= Void;
            proper_key:  k /= Void and then has_key (k)
         local
            pos: INTEGER

         do
            if not is_protected then

               from
                  pos := find_first (k)
               invariant
                  pos >= 0 and then pos <= capacity
               until
                  pos = 0 or else x.is_equal (store.item (pos))
               loop
                  pos := find_next (k, pos)
               end;

               if pos /= 0 then
                  guide.remove (pos);
                  store.put (Void, pos);
                  count := count - 1;
                  shrink (0)
               else
                  fault (f_remove, errhlg.sc_key_not_found)
               end

            else
               fault (f_remove, errhlg.sc_is_protected)
            end;

         ensure
            well_done: count = old count - 1
                       and then count = guide.count
         end;

      remove_all (k: H) is
            -- deletes all the entries items of which associated with key 'k'
         require
            no_obstacle: not is_protected;
            proper_key:  k /= Void and then has_key (k)
         local
            pos, next_pos: INTEGER

         do
            if not is_protected then

               pos := find_first (k);
               if pos /= 0 then

                  from
                  invariant
                     pos >= 0 and then pos <= capacity
                  until
                     pos = 0
                  loop
                     next_pos := find_next (k, pos);
                     guide.remove (pos);
                     store.put (Void, pos);
                     count := count - 1;
                     pos := next_pos
                  end;
                  shrink (0)

               else
                  fault (f_remove_all, errhlg.sc_key_not_found)
               end

            else
               fault (f_remove_all, errhlg.sc_is_protected)
            end

         ensure
            well_done: old count < count and then not has_key (k)
                       and then count = guide.count
         end;

      search (k: H) is
            -- looks for the key: after every successful attempt 'found' is
            -- set to true and 'found_item' - to an element (item) associated
            -- with key 'k'; in case of failure 'found' is set to false
         require
            valid_argument: k /= Void
         local
            pos: INTEGER

         do
            pos := find_first (k);
            found := (pos /= 0);
            if found then
               found_item := store.item (pos)
            end
         end;

      wipe_out is
            -- remove all entries from the table
         require
            no_obstacle: not is_protected

         do
            if not is_protected then
               if no_reallocation then
                  counterweight := capacity - default_size
               else
                  counterweight := 0
               end;
               make (unique_keys);
               count := 0
            else
               fault (f_wipe_out, errhlg.sc_is_protected)
            end

         ensure
            well_done: empty and then count = guide.count
         end

   ---------------------------------------------------------------------------
   feature -- Queries

      count: INTEGER; -- number of the entries currently stored in the table or
                      -- equally the number of slots used


      found: BOOLEAN; -- was the last 'search' successful ?

      found_item:  G; -- the element (item) found by the last call to 'search';
                      -- the value is valid if and only if 'found = true'

      unique_keys: BOOLEAN; -- must the keys of the entries be unique ?

      capacity: INTEGER is
            -- gives the number of all slots allocated in the table (all types
            -- of the tables may have not only slots occupied by the entries
            -- already put into but unused ones also)
         do
            Result := store.count
         end;

      count_of (k: H): INTEGER is
            -- gives the number of the entries items of which associated with
            -- key 'k'; if 'unique_items = true' then the number can be equal
            -- only to zero or one
         require
            valid_key: k /= Void
         local
            pos: INTEGER

         do
            from
               pos := find_first (k)
            until
               pos = 0
            loop
               Result := Result + 1;
               pos := find_next (k, pos)
            end

         ensure
            well_done: Result <= count
         end;

      current_keys: LIST [H] is
            -- delivers all the keys (without duplicates) currently stored in
            -- the table
         local
            csr: CURSOR

         do
            !!Result.make (false);
            from
               csr := cursor
            until
               csr.is_finished
            loop

               if not Result.has_item (key_at (csr)) then
                  Result.put (key_at (csr))
               end;
               csr.forth
            end
         end;

      cursor_at (x: G; k: H): CURSOR is
            -- delivers a new cursor positioned to the entry; if there is no
            -- such entry currently stored in the table Void is delivered
         require
            valid_item: x /= Void;
            proper_key: k /= Void
         local
            pos: INTEGER

         do
            from
               pos := find_first (k)
            invariant
               pos >= 0 and then pos <= capacity
            until
               pos = 0 or else x.is_equal (store.item (pos))
            loop
               pos := find_next (k, pos)
            end;

            if pos /= 0 then
               !!Result.make (Current);
               Result.set_position (pos);
               Result.activate
            end

         ensure
            well_done: Result /= Void implies is_protected
         end;

      has_key (k: H): BOOLEAN is
            -- is the entry currently stored in the table ?
         require
            valid_key: k /= Void
         do
            Result := (find_first (k) /= 0)
         end;

      item_at (cs: CURSOR): G is
            -- delivers the element or item of the entry the cursor is
            -- positioned to
         require
            proper_cursor: is_inside (cs)

         do
            Result := store.item (cs.position)
         end;

      items_at_key (k: H): LIST [G] is
            -- delivers all the elements (items) currently stored in the
            -- table and associated with key 'k'
         require
            proper_key: k /= Void
         local
            pos: INTEGER

         do
            !!Result.make (false);
            pos := find_first (k);

            if pos /= 0 then
               from
               invariant
                  pos >= 0 and then pos <= capacity
               until
                  pos = 0
               loop
                  Result.put (store.item (pos));
                  pos := find_next (k, pos)
               end
            end
         end;

      key_at (cs: CURSOR): H is
            -- the key of the entry the cursor is positioned to
         require
            proper_cursor: is_inside (cs)

         do
            Result := guide.get_key (cs.position)
         end

   ---------------------------------------------------------------------------
   feature {CURSOR}

      back (cs: CURSOR) is
            -- moves one item backwards
         local
            pos: INTEGER

         do
            if not cs.is_finished then
               pos := guide.back (cs.position);
               if pos /= 0 then
                  cs.set_position (pos)
               else
                  cs.stop
               end

            else
               errhlg.stop (Current, cs, f_back, errhlg.sc_out_of_range)
            end

         ensure then
            well_done: not cs.is_finished implies has_key (key_at (cs))
         end;

      first (cs: CURSOR) is
            -- positions on the first item
         do
            if not empty then
               cs.set_position (guide.first)
            else
               cs.stop
            end;

         ensure then
            well_done: not empty implies has_key (key_at (cs))
         end;

      forth (cs: CURSOR) is
            -- moves one item forward
         local
            pos: INTEGER

         do
            if not cs.is_finished then
               pos := guide.forth (cs.position);
               if pos /= 0 then
                  cs.set_position (pos)
               else
                  cs.stop
               end

            else
               errhlg.stop (Current, cs, f_forth, errhlg.sc_out_of_range)
            end

         ensure then
            well_done: not cs.is_finished implies has_key (key_at (cs))
         end;

      last (cs: CURSOR) is
            -- positions on the last item
         do
            if not empty then
               cs.set_position (guide.last)
            else
               cs.stop
            end

         ensure then
            well_done: not empty implies has_key (key_at (cs))
         end

   ---------------------------------------------------------------------------
   feature {NONE} -- Implementation

      counterweight: INTEGER; -- to change initial capacity while using 'make'
                              -- more than once (after creation)

      guide: POOL [H]; -- the ordering structure of the container

      no_reallocation: BOOLEAN; -- is automatic memory allocation is allowed ?

      store: ARRAY [G]; -- the real receptacle for entries to be stored

      default_size: INTEGER is
            -- the initial capacity of the container
         once
            Result := 16
         end;

      extend (new_capacity: INTEGER) is
            -- allocates memory for new slots
         require
            proper_capacity: new_capacity >= 0
         local
            old_capacity: INTEGER

         do
            if new_capacity > 0 then
               old_capacity := capacity;
               store.resize (1, new_capacity);
               guide.extend (new_capacity - old_capacity)

            elseif new_capacity = 0 then
               if not no_reallocation then
                  -- ...
                  old_capacity := capacity;
                  store.resize (1, old_capacity * 2);
                  guide.extend (old_capacity)
               end

            else
               fault (f_extend, errhlg.sc_unexpected)
            end

         ensure
            well_done: count = old count
         end;

      find_first (k: H): INTEGER is
            -- finds the first copy of the entry the element or item of which
            -- associated with key 'k' in the table
         require
            valid_key: k /= Void

         do
            Result := guide.search (k);
            if Result < 0 then
               Result := 0
            end

         ensure
            well_done: Result >= 0 and then Result <= capacity
         end;

      find_next (k: H; pos: INTEGER): INTEGER is
            -- finds the next entry (the current one is storing at 'pos')
            -- the element or item of which is associated with key 'k';
            -- traversal is towards the end of the table
         require
            proper_key:      k /= Void and then has_key (k);
            proper_position: pos > 0 and then pos <= capacity
         do
            from
               Result := guide.forth (pos)
            until
               Result = 0 or else k.is_equal (guide.get_key (Result))
            loop
               Result := guide.forth (Result)
            end

         ensure
            well_done: Result >= 0 and then Result <= capacity
         end;

      restructure (new_capacity: INTEGER) is
            -- creates a new table (with 'new_capacity' slots allocated,
            -- copies all the entries of the old table to a new one and
            -- saves the latter one as the current table
         require
            proper_size: new_capacity > 0 and then new_capacity >= count

         local
            csr:   CURSOR;
            pos,
            cnt:   INTEGER;
            items: ARRAY [G];
            keys:  ARRAY [H]

         do
            if new_capacity < count then
               fault (f_restructure, errhlg.sc_unexpected)
            else
               -- Saving the data of the container
               csr := cursor;

               if not csr.is_finished then
                  cnt := count;
                  !!items.make (1, cnt);
                  !!keys.make (1, cnt);

                  from
                     pos := 0
                  variant
                     saving: count - pos
                  until
                     csr.is_finished
                  loop
                     pos := pos + 1;
                     items.put (item_at (csr), pos);
                     keys.put (key_at (csr), pos);
                     csr.forth
                  end
               end;

               -- Modifying 'capacity' ...
               counterweight := new_capacity - default_size;
               make (unique_keys);
               count := 0;

               -- Restoring the data
               if items /= Void then

                  from
                     pos := 0
                  variant
                     restoring: cnt - count
                  until
                     pos = cnt
                  loop
                     pos := pos + 1;
                     put (items.item (pos), keys.item (pos))
                  end
               end
            end

         ensure
            well_done: count = old count and then count = guide.count
         end;

      shrink (new_capacity: INTEGER) is
            -- selects the algorithm for shrinking if it's necessary
         require
            proper_capacity: new_capacity >= 0

         do
            if new_capacity = 0 then
               if not no_reallocation and then
                  count * 4 <= capacity and then
                  capacity >= default_size * 2 then

                  restructure (capacity // 2)
               end
            else
               restructure (new_capacity)
            end
         end

   ---------------------------------------------------------------------------
   invariant

      nothing_lost:   capacity = count + guide.stock;
      properly_tuned: capacity = guide.capacity;
      is_ordered:     guide /= Void and then guide.target = Current

end -- class TABLE [G, H]