indexing

   title:       "Simple lists: containers with one element entries",
                "implementing a first-in, first-out policy and",
                "supporting only restricted traversal in one direction";
   cluster:     "containers [contners]";
   project:     "The Universal Simple Container Library (USCL)";
   copyright:   "Frieder Monninger & Alexei Shestialtynov, 1995";
   author:      "Frieder Monninger (fm@eiffel.de)",
                "Alexei Shestialtynov (alexei@eiffel.ru)";
   original:    31,Aug,95;
   version:     1.0;
   last_change:
   approved_by:
   approved:
   key:         singly_linked_list, FIFO, one_way_traversable;
   done_at:     "SiG Computer (info@eiffel.ru)";
   extrnl_name: "spl_list.e"

class SIMPLE_LIST [G]

   inherit
      CONTAINER [G]
         end;

      SIMPLE_TRAVERSABLE
         end

   creation
      make

   ---------------------------------------------------------------------------
   feature -- Operations

      allocate_automatically is
            -- switches the list to the mode of automatic memory allocation
            -- for new slots; it can be necessary only if 'allocate_manually'
            -- was called before because, by default, all types of the lists
            -- allocate memory automatically
         do
            no_reallocation := false
         end;

      allocate_manually (new_capacity: INTEGER) is
            -- forbids the list to allocate memory for new slots automatically
            -- and sets the total amount of slots allocated in the list to be
            -- equal to 'new_capacity' - being negligent after that in manual
            -- allocating extra slots for new entries to be put might lead to
            -- the exception of type SCIF ("container is full")
            --
         require
            no_obstacle:     not is_protected;
            proper_capacity: new_capacity >= count and then new_capacity > 0

         do
            if not is_protected and then new_capacity >= count then

               no_reallocation := true;
               if new_capacity > capacity then
                  extend (new_capacity)
               elseif new_capacity < capacity then
                  shrink (new_capacity)
               end

            elseif is_protected then
               fault (f_allocate, errhlg.sc_is_protected)
            else
               fault (f_allocate, errhlg.sc_loss_of_data)
            end
         end;

      make (is_unique: BOOLEAN) is
            -- tells the container if its entries must be unigue
         local
            init_size: INTEGER

         do
            unique_items := is_unique;
            init_size := default_size + counterweight;

            !!store.make (1, init_size);
            !SIMPLE_CHAIN!guide.make (Current, init_size)
         end;

      merge (other: like Current) is
            -- puts all entries from 'other' into the list
         require
            no_obstacle:    not is_protected;
            valid_argument: other /= Void
         local
            input: SIMPLE_CURSOR

         do
            if not is_protected then

               from
                  input := other.cursor
               until
                  input.is_finished
               loop
                  put (other.item_at (input));
                  input.forth
               end

            else
               fault (f_merge, errhlg.sc_is_protected)
            end

         ensure
            well_done: count = old count + other.count
                       and then count = guide.count
         end;

      put (x: G) is
            -- adds the entry into the list
         require else
            valid_argument: x /= Void;
            no_obstacle:    not is_protected;
            uniqueness:     unique_items implies not has_item (x)
         local
            pos: INTEGER

         do
            if not is_protected then
               if unique_items then
                  pos := find_first (x)
               end;

               If pos = 0 then
                  guide.get_free;
                  pos := guide.free;

                  if pos = 0 then
                     extend (0);
                     guide.get_free;
                     pos := guide.free;

                     if pos = 0 then
                        fault (f_put, errhlg.sc_is_full)
                     end
                  end

                  store.put (x, pos);
                  guide.put (pos);

                  count := count + 1

               else
                  fault (f_put, errhlg.sc_item_duplication)
               end

            else
               fault (f_put, errhlg.sc_is_protected)
            end

         ensure then
            well_done: not empty and then has_item (x)
                       and then count = guide.count
         end;

      remove (x: G) is
            -- deletes the entry from the list
         require
            no_obstacle:    not is_protected;
            valid_argument: x /= Void;
            valid_target:   has_item (x)

         do
            if not is_protected then
               if has_item (x) then
                  delete (x, only_one);
                  shrink (0)
               else
                  fault (f_remove, errhlg.sc_item_not_found)
               end
            else
               fault (f_remove, errhlg.sc_is_protected)
            end

         ensure
            well_done: count = old count - 1
                       and then count = guide.count
         end;

      remove_all (x: G) is
            -- deletes the entry from the list together with all its
            -- duplicates if more than one copy is currently stored
         require
            no_obstacle:    not is_protected;
            valid_argument: x /= Void;
            valid_target:   has_item (x)

         do
            if not is_protected then
               if has_item (x) then
                  delete (x, all_of_them);
                  shrink (0)
               else
                  fault (f_remove_all, errhlg.sc_item_not_found)
               end
            else
               fault (f_remove_all, errhlg.sc_is_protected)
            end

         ensure
            well_done: old count < count and then not has_item (x)
                       and then count = guide.count
         end;

      search (x: G) is
            -- looks for the element: after every successful attempt 'found'
            -- is set to true and 'found_item' - to the found element;
            -- in case of failure 'found' is set to false
         require
            valid_argument: x /= Void
         local
            pos: INTEGER

         do
            pos := find_first (x);
            found := (pos /= 0);
            if found then
               found_item := store.item (pos)
            end
         end;

      wipe_out is
            -- remove all entries from the list
         require
            no_obstacle: not is_protected

         do
            if not is_protected then
               if no_reallocation then
                  counterweight := capacity - default_size
               else
                  counterweight := 0
               end;

               make (unique_items);
               count := 0
            else
               fault (f_wipe_out, errhlg.sc_is_protected)
            end

         ensure
            well_done: empty and then count = guide.count
         end;

   ---------------------------------------------------------------------------
   feature -- Queries

      count: INTEGER; -- number of the entries currently stored in the list or
                      -- equally the number of slots used

      found: BOOLEAN; -- was the last 'search' successful ?

      found_item: G;  -- the entry found by the very last call to 'search';
                      -- the value is valid if and only if 'found = true'

      unique_items: BOOLEAN; -- must the entries be unique ?

      capacity: INTEGER is
            -- gives the number of all slots allocated in the list (all types
            -- of the lists may have not only slots occupied by the entries
            -- already put into but unused ones also)
         do
            Result := store.count
         end;

      count_of (x: G): INTEGER is
            -- gives the number of the duplicates of the entry inclusive; if
            -- 'unique_items = true' then the number can be only zero or one
         require
            valid_argument: x /= Void
         local
            pos: INTEGER

         do
            from
               pos := find_first (x)
            until
               pos = 0
            loop
               Result := Result + 1;
               pos := find_next (x, pos)
            end

         ensure
            well_done: Result <= count
         end;

      cursor_at (x: G): SIMPLE_CURSOR is
            -- delivers a new cursor positioned to the entry; if there is no
            -- such entry currently stored in the list Void is delivered
         require
            valid_argument: x /= Void
         local
            pos: INTEGER

         do
            pos := find_first (x);

            if pos /= 0 then
               !!Result.make (Current);
               Result.set_position (pos);
               Result.activate
            end

         ensure
            well_done: Result /= Void implies is_protected
         end;

      has_item (x: G): BOOLEAN is
            -- is the entry currently stored in the list ?
         require
            valid_argument: x /= Void
         do
            Result := (find_first (x) /= 0)
         end;

      item_at (cs: SIMPLE_CURSOR): G is
            -- delivers the entry of the list the cursor is positioned to
         require
            proper_cursor: is_inside (cs)

         do
            Result := store.item (cs.position)
         end;

   ---------------------------------------------------------------------------
   feature {SIMPLE_CURSOR}

      first (cs: SIMPLE_CURSOR) is
            -- positions on the first item
         do
            if not empty then
               cs.set_position (guide.first)
            else
               cs.stop
            end

         ensure then
            well_done: not empty implies has_item (item_at (cs))
         end;

      forth (cs: SIMPLE_CURSOR) is
            -- moves one item forward
         local
            pos: INTEGER

         do
            if not cs.is_finished then
               pos := guide.forth (cs.position);
               if pos /= 0 then
                  cs.set_position (pos)
               else
                  cs.stop
               end

            else
               errhlg.stop (Current, cs, f_forth, errhlg.sc_out_of_range)
            end

         ensure then
            well_done: not cs.is_finished implies has_item (item_at (cs))
         end

   ---------------------------------------------------------------------------
   feature {ON_ITEMS_FOR_ORDERING} -- Implementation

      store: ARRAY [G] -- the real receptacle for entries to be stored

   ---------------------------------------------------------------------------
   feature {NONE} -- Implementation

      only_one:    BOOLEAN is false;
      all_of_them: BOOLEAN is true;

      counterweight: INTEGER;   --  to change initial capacity while
                                --  using 'make' more than once (at creation)
      guide: SIMPLE_CHAIN_CORE; --  the ordering structure of the container

      no_reallocation: BOOLEAN; -- is automatic memory allocation is allowed ?

      default_size: INTEGER is
            -- the initial capacity of the container
         once
            Result := 16
         end;

      delete (x: G, erase_the_same: BOOLEAN) is
            -- actually deletes the entry with or without its duplicates
         require
            valid_argument: x /= Void;
            valid_target:   has_item (x)
         local
            pos, next_pos: INTEGER

         do
            from
               pos := find_first (x)
            variant
               count
            until
               pos = 0
            loop
               if erase_the_same then
                  next_pos := find_next (x, pos)
               end;
               guide.remove (pos);
               store.put (Void, pos);
               count := count - 1;
               pos := next_pos
            end
         end;

      extend (new_capacity: INTEGER) is
            -- allocates memory for new slots
         require
            proper_capacity: new_capacity >= 0
         local
            old_capacity: INTEGER

         do
            if new_capacity > 0 then
               old_capacity := capacity;
               store.resize (1, new_capacity);
               guide.extend (new_capacity - old_capacity)

            elseif new_capacity = 0 then
               if not no_reallocation then
                  -- ...
                  old_capacity := capacity;
                  store.resize (1, old_capacity * 2);
                  guide.extend (old_capacity)
               end

            else
               fault (f_extend, errhlg.sc_unexpected)
            end

         ensure
            well_done: count = old count
         end;

      find_first (x: G): INTEGER is
            -- finds the first copy of the entry in the list
         require
            valid_argument: x /= Void

         do
            from
               Result := guide.first
            until
               Result = 0 or else x.is_equal (store.item (Result))
            loop
               Result := guide.forth (Result)
            end

         ensure
            well_done: Result >= 0 and then Result <= capacity
         end;

      find_next (x: G; pos: INTEGER): INTEGER is
            -- finds the duplicates of the entry which is stored at 'pos';
            -- traversal is towards the end of the list
         require
            valid_argument:  x /= Void;
            proper_position: pos > 0 and then pos <= capacity

         do
            from
               Result := guide.forth (pos)
            until
               Result = 0 or else x.is_equal (store.item (Result))
            loop
               Result := guide.forth (Result)
            end

         ensure
            well_done: Result >= 0 and then Result <= capacity
         end;

      restructure (new_capacity: INTEGER) is
            -- creates a new list (with 'new_capacity' slots allocated,
            -- copies all the entries of the old list to a new one and
            -- saves the latter one as the current list
         require
            proper_size: new_capacity >= count

         local
            csr: SIMPLE_CURSOR;
            tmp: ARRAY [G];
            pos,
            cnt: INTEGER
         do
            if new_capacity < count then
               fault (f_restructure, errhlg.sc_unexpected)
            else
               -- Saving the data of the container
               csr := cursor;

               if not csr.is_finished then
                  cnt := count;
                  !!tmp.make (1, count);

                  from
                     pos := 0
                  variant
                     saving: count - pos
                  until
                     csr.is_finished
                  loop
                     pos := pos + 1;
                     tmp.put (item_at (csr), pos);
                     csr.forth
                  end
               end;

               -- Modifying 'capacity' ...
               counterweight := new_capacity - default_size;
               make (unique_items);
               count := 0;

               -- Restoring the data
               if tmp /= Void then
                  from
                     pos := 0
                  variant
                     restoring: cnt - count
                  until
                     pos = cnt
                  loop
                     pos := pos + 1;
                     put (tmp.item (pos))
                  end
               end
            end

         ensure
            well_done: count = old count and then count = guide.count
         end;

      shrink (new_capacity: INTEGER) is
            -- selects the algorithm for shrinking if it's necessary
         require
            proper_capacity: new_capacity >= 0

         do
            if new_capacity = 0 then
               if not no_reallocation and then
                  count * 4 <= capacity and then
                  capacity >= default_size * 2 then

                  restructure (capacity // 2)
               end
            else
               restructure (new_capacity)
            end
         end

   ---------------------------------------------------------------------------
   invariant

      nothing_lost:   capacity = count + guide.stock;
      properly_tuned: capacity = guide.capacity;
      is_ordered:     guide /= Void and then guide.target = Current

end -- class SIMPLE_LIST