indexing

   title:       "Key priority queues: dispensers with two element entries",
                "in which only the entry with the largest key is accessed";
   cluster:     "containers [contners]";
   project:     "The Universal Simple Container Library (USCL)";
   copyright:   "Frieder Monninger & Alexei Shestialtynov, 1995";
   author:      "Frieder Monninger (fm@eiffel.de)",
                "Alexei Shestialtynov (alexei@eiffel.ru)";
   original:    31,Aug,95;
   version:     1.0;
   last_change:
   approved_by:
   approved:
   key:         priority_queue, the_largest_key;
   done_at:     "SiG Computer (info@eiffel.ru)";
   extrnl_name: "key_pr_q.e"

class KEY_PRIORITY_QUEUE [G, H -> COMPARABLE]

   inherit
      KEYED_DISPENSER [G, H]
         redefine
            pool
         end

   ---------------------------------------------------------------------------
   feature -- Operations

      put (x: G; k: H) is
            -- adds the new entry into the dispenser
         do
            if pool = Void then
               !!pool.make (false);
               cs := pool.cursor
            end;

            check cs.is_finished end;

            pool.put (x, k);

            key  := pool.last_key_head;
            item := pool.item_at_last_key_head

         ensure then
            well_done: count = old count + 1
         end;

      remove is
            -- deletes the "active entry" and sets 'item' to the element
            -- associated with the largest key among those still existing in
            -- the dispenser after the removal and 'key' to this key; if there
            -- are no more entries then 'item' and 'key' are both set to Void
         do
            if not empty then

               check item /= Void and then key /= Void end;
               pool.remove (item, key);

               key  := pool.last_key_head;
               item := pool.item_at_last_key_head

            else
               fault (f_remove, errhlg.sc_is_empty)
            end

         ensure then
            well_done: cs.is_finished and then count = old count - 1
         end

   ---------------------------------------------------------------------------
   feature -- Queries

      table_of_elements: TABLE [G, H] is
            -- delivers all the entries of the dispenser in the order in
            -- which they would be repeatedly removed if it really happened
         local
            item_stack:    STACK [G];
            key_stack:     STACK [H];
            current_key:   H;
            pop_stack_out,
            push_on_stack: BOOLEAN

         do
            !!Result.make (false);
            if not empty then
               check pool /= Void and then count > 0  end;

               from
                  !!item_stack;
                  !!key_stack;

                  -- allocate slots in advance to avoid possible
                  -- frequent automatic reallocation ...
                  Result.allocate_manually (count);

                  cs.last
               until
                  Result.count = count
               loop
                  push_on_stack := true;
                  pop_stack_out := false;

                  if cs.is_finished then
                     pop_stack_out := true;
                     push_on_stack := false
                  else
                     current_key := pool.key_at (cs);

                     -- all keys and items go into the stacks
                     -- to reverse the order of duplicates ...
                     if not (key_stack.empty or else
                           current_key.is_equal (key_stack.item)) then

                        pop_stack_out := true
                     end
                  end;

                  if pop_stack_out then
                     check not key_stack.empty end;

                     from
                     invariant
                        key_stack.count = item_stack.count
                     until
                        key_stack.empty
                     loop
                        Result.put (item_stack.item, key_stack.item);
                        item_stack.remove;
                        key_stack.remove
                     end
                  end;

                  if push_on_stack then
                     item_stack.put (pool.item_at (cs));
                     key_stack.put (current_key);
                     cs.back
                  end
               end;

               -- return to automatic memory reallocation
               Result.allocate_automatically
            end

         ensure then
            well_done: cs /= Void implies cs.is_finished
         end

   ---------------------------------------------------------------------------
   feature {NONE} -- Implementation

      pool: SORTED_TABLE [G, H] -- the real receptacle for entries to be stored

end -- class KEY_PRIORITY_QUEUE [G, H -> COMPARABLE]