indexing

   title:       "Some universal common facilities of the guides for classes",
                "SORTED_LIST and SORTED_TABLE (used without duplicates);",
                "---------------------------------------------------------",
                " All algorithms are taken (with minor changes) from book ",
                " Introduction to Algorithms by Thomas H. Cormen, Charles ",
                " E. Leiserson and Ronald L. Rivest (c) The Massachusetts ",
                " Institute of Technology, 1990 --------------------------";
   cluster:     "containers [contners]";
   project:     "The Universal Simple Container Library (USCL)";
   copyright:   "Frieder Monninger & Alexei Shestialtynov, 1995";
   author:      "Frieder Monninger (fm@eiffel.de)",
                "Alexei Shestialtynov (alexei@eiffel.ru)";
   original:    31,Aug,95;
   version:     1.0;
   last_change:
   approved_by:
   approved:
   key:         binary_search_tree, red_black_tree;
   done_at:     "SiG Computer (info@eiffel.ru)";
   extrnl_name: "is_order.e"

deferred class SIMPLE_ORDER [G -> COMPARABLE]

   inherit
      POOL [G]
         rename
            entrance as root, prev as left, next as right
         redefine
            put, remove, first, forth, last, back, count
         end

   ---------------------------------------------------------------------------
   feature {SIMPLE_CONTAINER} -- Operations

      put (pos: INTEGER) is
            -- puts the new entry in order of the guide
         local
            x, y, z: INTEGER;
            stream:  ARRAY [INTEGER];
            turn:    BOOLEAN

         do
            -- Inserting a new node ... --

            left.put (0, pos);
            right.put (0, pos);
            color.put (red, pos);

            if root = 0 then
               root := pos;
               color.put (black, pos);
               parent.put (0, pos)

            else
               from
                  x := root

               invariant
                  pos_is_not_used_yet: pos /= x;
                  proper_steps:        x >= 0 and then x <= capacity

               until
                  x = 0

               loop
                  y := x;

                  if get_key (pos) < get_key (x) then
                     x := left.item (x)

                  elseif get_key (pos) > get_key (x) then
                     x := right.item (x)

                  else
                     duplicate (pos, y);
                     x := 0;
                     y := 0
                  end
               end;

               if y > 0 then
                  -- inserting a new unique node - not a duplicate copy

                  if get_key (pos) < get_key (y) then
                     left.put (pos, y);
                     parent.put (y, pos)

                  else
                     right.put (pos, y);
                     parent.put (y, pos)
                  end
               end
            end;

            -- Balancing the tree ... --

            if color.item (pos) = red then

               from
                  x := pos

               until
                  not (x /= root and then color.item (get_parent (x)) = red)

               loop
                  z := get_parent (x);
                  check
                     -- the root must be only black -> it means that if the
                     -- color of a node is red - it cannot be the root ...
                     --
                     the_assumption_is_true: get_parent (z) /= 0
                  end;
                  if z = left.item (get_parent (z)) then
                     stream := right;
                     turn   := to_the_left
                  else
                     stream := left;
                     turn   := to_the_right
                  end;

                  y := stream.item (get_parent (z));

                  if color.item (y) = red then
                     color.put (black, z);
                     color.put (black, y);
                     color.put (red, get_parent (z));
                     x := get_parent (z)

                  else
                     if x = stream.item (z) then
                        x := z;
                        --
                        -- possible options:
                        --    stream = right -> turn = to_the_left
                        --    stream = left  -> turn = to_the_right
                        --
                        rotate (turn, x)
                     end;

                     z := get_parent (x);
                     color.put (black, z);
                     color.put (red, get_parent (z));
                     --
                     -- possible options:
                     --    stream = right -> not turn = to_the_right
                     --    stream = left  -> not turn = to_the_left
                     --
                     rotate (not turn, get_parent (z))
                  end
               end
            end;

            color.put (black, root)
         end;

      search (x: G): INTEGER is
            -- gives the position (a positive integer) of the entry if it
            -- exists; otherwise - a non-positive integer
         local
            shadow: INTEGER

         do
            if root /= 0 then
               from
                  Result := root

               invariant
                  right: Result /= 0 implies ((right.item (Result) >= 0)
                         and then (right.item (Result) <= capacity));
                  left:  Result /= 0 implies ((left.item (Result) >= 0)
                         and then (left.item (Result) <= capacity))

               until
                  not (Result /= 0 and then not x.is_equal (get_key (Result)))

               loop
                  shadow := Result;
                  if x < get_key (Result) then
                     Result := left.item (Result)
                  else
                     Result := right.item (Result)
                  end
               end;

               if Result = 0 then
                  Result := - shadow
               end
            end
         end;

      remove, frozen remove_tree_node (pos: INTEGER) is
            -- brings the entry out of the order of the guide
         local
            p, w, x, y: INTEGER;
            main, subs: ARRAY [INTEGER];
            clry, turn: BOOLEAN

         do
            -- Removing a node ... --

            if left.item (pos) = 0 or else right.item (pos) = 0 then
               y := pos
            else
               y := next_node (up, pos)
            end;

            x := left.item (y);
            if x = 0 then
               x := right.item (y)
            end;

            p := get_parent (y);
            put_parent (p, x);

            if p = 0 then
               root := x
            else
               if y = left.item (p) then
                  left.put (x, p)
               else
                  right.put (x, p)
               end
            end;

            clry := color.item (y);

            if y /= pos then
               substitute (pos, y)
            end;

            if clry = black then

               -- Balancing the tree ... --

               from
               until
                  not (x /= root and then color.item (x) = black)

               loop
                  p := get_parent (x);

                  if x = left.item (p) then
                     main := right;
                     subs := left;
                     turn := to_the_left
                  else
                     main := left;
                     subs := right;
                     turn := to_the_right
                  end;

                  w := main.item (p);
                  if color.item (w) = red then
                     color.put (black, w);
                     color.put (red, p);
                     --
                     -- possible options:
                     --    main = right -> turn = to_the_left
                     --    main = left  -> turn = to_the_right
                     --
                     rotate (turn, p);
                     w := main.item (p)
                  end;

                  if (color.item (subs.item (w)) = black) and then
                     (color.item (main.item (w)) = black) then

                     color.put (red, w);
                     x := p

                  else
                     if color.item (main.item (w)) = black then

                        color.put (black, subs.item (w));
                        color.put (red, w);
                        --
                        -- possible options:
                        --    main = right -> not turn = to_the_right
                        --    main = left  -> not turn = to_the_left
                        --
                        rotate (not turn, w);
                        w := main.item (p)
                     end;

                     color.put (color.item (p), w);
                     color.put (black, p);
                     color.put (black, main.item (w));
                     --
                     -- possible options:
                     --    main = right -> turn = to_the_left
                     --    main = left  -> turn = to_the_right
                     --
                     rotate (turn, p);
                     x := root
                  end
               end;

               color.put (black, x)
            end;

            frozen_put_in_stock (pos);
            remove_key (pos)
         end

   ---------------------------------------------------------------------------
   feature {SIMPLE_CONTAINER} -- Queries

      first: INTEGER is
            -- gives the position of the first entry
         do
            Result := subtree_leaf (down, root)
         end;

      forth, tree_forth (pos: INTEGER): INTEGER is
            -- gives the position of the next (about 'pos') entry
         do
            if internal_node (pos) then
               target.halt (Current, "tree_forth")
            else
               Result := next_node (up, pos)
            end
         end;

      last, frozen tree_last: INTEGER is
            -- gives the position of the last entry
         do
            Result := subtree_leaf (up, root)
         end;

      back, tree_back (pos: INTEGER): INTEGER is
            -- gives the position of the previous (about 'pos') entry
         do
            if internal_node (pos) then
               target.halt (Current, "tree_back")
            else
               Result := next_node (down, pos)
            end
         end;

      count: INTEGER is
            -- gives the number of used slots or equally the number of
            -- entries kept in the guide
         do
            Result := count_tree_leaves (root);
         end

   ---------------------------------------------------------------------------
   feature {NONE} -- Implementation

      red,   up,  to_the_right: BOOLEAN is true;
      black, down, to_the_left: BOOLEAN is false;

      parent: ARRAY [INTEGER]; -- ... of parents in a binary search tree
      color:  ARRAY [BOOLEAN]; -- ... of colors in a red-black tree ...
                               --     'false' means black and 'true' - red

      get_parent (index: INTEGER): INTEGER is
            -- gives the item at position 'index' of array 'parent'
         require
            proper_index: index >= 0 and then index <= capacity

         do
            Result := parent.item (index);
            if Result < 0 then
               target.halt (Current, "get_parent")
            end

         ensure
            well_done: Result >= 0 and then Result <= capacity
         end;

      put_parent (element, index: INTEGER) is
            -- puts 'element' at position 'index' into array 'parent'
         require
            proper_index:   index >= 0 and then index <= capacity;
            proper_element: element >= 0 and then element <= capacity

         do
            if parent.item (index) < 0 then
               target.halt (Current, "put_parent")
            else
               parent.put (element, index)
            end
         end;

      duplicate (new, existed: INTEGER) is
            -- ... should not be invoked in class 'SIMPLE_ORDER' !!
         require
            proper_new:     new > 0 and then new <= capacity;
            proper_existed: existed > 0 and then existed <= capacity

         do
            target.halt (Current, "duplicate")

         ensure
            well_done: color.item (new) = black
         end;

      resizing (new_size: INTEGER) is
            -- changes the amount of slots allocated
         do
            left.resize (1, new_size);
            right.resize (1, new_size);
            color.resize (0, new_size);
            parent.resize (0, new_size)
         end;

      subsequent_init (size: INTEGER) is
            -- initializes the internal arrays other than 'next' or 'right';
            -- it's the second outstanding part of creation procedure 'make'
         do
            !!left.make (1, size);

            -- if a child of a node does not exist, the corresponding pointer
            -- field of the node contains the value NIL (actually zero) - we
            -- shall regard these NIL's as being pointers to external nodes
            -- which are only always black colored ...

            !!color.make (0, size);
            color.put (black, 0);

            !!parent.make (0, size)
         end;

      count_tree_leaves (pos: INTEGER): INTEGER is
            -- the number of nodes in the tree
         require
            proper_position: pos >= 0 and then pos <= capacity

         do
            if pos /= 0 then
               Result := count_tree_leaves (left.item (pos)) +
                         count_tree_leaves (right.item (pos)) + 1
            end

         ensure
            well_done: Result <= capacity
         end;

      subtree_leaf (rightmost: BOOLEAN; pos: INTEGER): INTEGER is
            -- finds the position of leaves in the tree
         require
            proper_position: pos >= 0 and then pos <= capacity
         local
            next_pos: INTEGER;
            stream:   ARRAY [INTEGER]

         do
            if rightmost then
               stream := right
            else
               stream := left
            end;

            from
               next_pos := pos
            invariant
               next_pos >= 0 and then next_pos <= capacity
            until
               next_pos = 0
            loop
               Result := next_pos;
               next_pos := stream.item (Result)
            end
         end;

      next_node (right_hand: BOOLEAN; pos: INTEGER): INTEGER is
            -- finds the next or previous node (about 'pos') in the tree
         require
            proper_position: pos > 0 and then pos <= capacity
         local
            direct: BOOLEAN;
            interm: INTEGER;
            stream: ARRAY [INTEGER]

         do
            if right_hand then
               direct := down;
               stream := right
            else
               direct := up;
               stream := left
            end;

            Result := stream.item (pos);
            if Result /= 0 then
               Result := subtree_leaf (direct, Result)
            else
               from
                  interm := pos;
                  Result := get_parent (interm)

               invariant
                  Result >= 0 and then Result <= capacity

               until
                  not (Result /= 0 and then interm = stream.item (Result))

               loop
                  interm := Result;
                  Result := get_parent (Result)
               end
            end
         end;

      substitute (junk, new: INTEGER) is
            -- substitutes node 'junk' for node 'new' ...
         require
            proper_junk:   junk > 0 and then junk <= capacity;
            proper_new:    new > 0 and then new <= capacity;

            proper_parent: get_parent (junk) /= 0 implies
                           (left.item (get_parent (junk)) = junk or else
                           right.item (get_parent (junk)) = junk);

            either_or:     parent.item (new) >= 0 or else
                           parent.item (junk) >= 0;

            proper_right:  right.item (junk) /= 0 implies
                           get_parent (right.item (junk)) = junk;
            proper_left:   left.item (junk) /= 0 implies
                           get_parent (left.item (junk)) = junk
         local
            tmp: INTEGER

         do
            if parent.item (new) < 0 then
               put_parent (get_parent (junk), new)

               if parent.item (junk) < 0 then
                  target.halt (Current, "substitute")
               end
            else
               parent.put (parent.item (junk), new)
            end;

            left.put (left.item (junk), new);
            right.put (right.item (junk), new);
            color.put (color.item (junk), new);

            check
               proper_left:  left.item (new) >= 0;
               proper_right: right.item (new) >= 0
            end;

            if left.item (new) > 0 then
               put_parent (new, left.item (new))
            end;

            if right.item (new) > 0 then
               put_parent (new, right.item (new))
            end;

            tmp := get_parent (new);
            if tmp /= 0 then
               if left.item (tmp) = junk then
                  left.put (new, tmp)
               else
                  right.put (new, tmp)
               end
            else
               root := new
            end;

            if parent.item (0) = junk then
               parent.put (new, 0)
            end
         end;

      rotate (clockwise: BOOLEAN; pos: INTEGER) is
            -- performs 'Left-Rotate' (LR) or 'Right-Rotate' (RR) described in
            -- ITA - for the full name of the book refer to the Indexing part
            -- of the class header;
            --
            -- if 'clockwise = true' it performs RR action otherwise - LR one:
            -- LR turns |     to     |   and RR turns |     to     |
            --         (z)          (z)              (z)          (z)
            --          |            |                |            |
            --         pos           y               pos           y
            --         / \          / \              / \          / \
            --        a   y       pos  c            y   c        a  pos
            --           / \      / \              / \              / \
            --      (z) b   c    a   b (z)        a   b (z)    (z) b   c
            --
            -- z - a temporary assignment to node b and the parent of 'pos' -
            -- it '(z)' is placed here only to simplify reading the source ...
            --
         require
            valid_position:  pos > 0 and then pos <= capacity;
            proper_position: (clockwise implies left.item (pos) /= 0) and then
                             (not clockwise implies right.item (pos) /= 0)
         local
            y, z: INTEGER;   main, subs: ARRAY[INTEGER]

         do
            if clockwise then
               -- RR action --
               main := left;
               subs := right
            else
               -- LR action --
               main := right;
               subs := left
            end;

            y := main.item (pos);  -- set y
            z := subs.item (y);    -- z is used as a temporary local attribute
                                   -- and only for some optimization
            main.put (z, pos);     -- LR: turn y's left subtree
                                   --     into pos's right subtree
                                   -- RR: turn y's right subtree
                                   --     into pos's left subtree
            if z /= 0 then
               put_parent (pos, z)
            end;

            z := get_parent (pos);
            put_parent (z, y);     -- link pos's parent to y
            if z = 0 then
               root := y
            elseif pos = left.item (z) then
               left.put (y, z)
            else
               right.put (y, z)
            end;

            subs.put (pos, y);     -- LR: put pos on y's left
                                   -- RR: put pos on y's right
            put_parent (y, pos)
         end;

      is_a_red_black_tree: BOOLEAN is
            -- is the tree a red-black tree ?
         do
            if color.item (root) = black then
               Result := checking_tree (root)
            end
         end;

      checking_tree (pos: INTEGER): BOOLEAN is
            -- checks if the tree has all the necessary properties of
            -- a red-black tree
         require
            proper_position: pos >= 0 and then pos <= capacity

         do
            if pos /= 0 then

               Result := true;

               if (color.item (pos) = red) and then
                  (color.item (left.item (pos)) /= black or else
                  color.item (right.item (pos)) /= black) then

                  Result := false
               end;

               if black_height (down, left.item (pos)) /=
                  black_height (up, right.item (pos)) then

                  Result := false
               end;

               if Result then
                  Result := checking_tree (left.item (pos)) and then
                            checking_tree (right.item (pos))
               end

            else

               Result := color.item (pos) = black
            end
         end;

      black_height (rightpath: BOOLEAN; pos: INTEGER): INTEGER is
            -- gives the number of black nodes on the path from 'pos' but
            -- not including ...
         require
            proper_position: pos >= 0 and then pos <= capacity
         local
            next_pos:   INTEGER;
            main, subs: ARRAY [INTEGER]

         do
            if rightpath then
               main := right;
               subs := left
            else
               main := left;
               subs := right
            end;

            from
               next_pos := pos;
               if color.item (next_pos) = black then
                  Result := 1
               end

            invariant
               next_pos >= 0 and then next_pos <= capacity

            until
               next_pos = 0

            loop
               if main.item (next_pos) = 0 then
                  next_pos := subs.item (next_pos)
               else
                  next_pos := main.item (next_pos)
               end;

               if color.item (next_pos) = black then
                  Result := Result + 1
               end
            end
         end;

      internal_node (pos: INTEGER): BOOLEAN is
            -- is the node an internal one (a duplicate) ?
         require
            proper_position: pos > 0 and then pos <= capacity

         do
            Result := (parent.item (pos) = 0 and then root /= pos)
         end

   ---------------------------------------------------------------------------
   invariant

      proper_tree_structure: is_a_red_black_tree

end -- deferred class SIMPLE_ORDER [G -> COMPARABLE]