/***********************************************************************/
/* SHOW.C - Functions involving displaying the data.                   */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * 36 David Road                 Phone:                    +617 3849 7731
 * Holland Park                  http://www.gu.edu.au/gwis/the/markh.html
 * Brisbane                      **** Maintainer PDCurses & REXX/SQL ****
 * QLD 4121                      ************* Author of THE ************
 * Australia                     ************* Member RexxLA ************
 */

/*
$Id: show.c 2.1 1995/06/24 16:31:13 MH Rel MH $
*/

#include <stdio.h>
#include <time.h>

#if defined(USE_EXTCURSES)
# include <cur04.h>
#endif

#include "the.h"
#include "proto.h"

/*------------------------ function definitions -----------------------*/
#ifdef PROTO
static void build_lines(short,LINE *,short,short);
static void show_lines(void);
static void show_a_line(short);
static void set_prefix_contents(LINE *,short,LINETYPE);
static void show_hex_line(short);
#else
static void build_lines();
static void show_lines();
static void show_a_line();
static void set_prefix_contents();
static void show_hex_line();
#endif

static LINE *hexshow_curr=NULL;

/***********************************************************************/
#ifdef PROTO
void show_heading(void)
#else
void show_heading()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE dir_path[MAX_FILE_NAME+1];
 extern CHARTYPE dir_files[MAX_FILE_NAME+1];
 extern CHARTYPE rexx_macro_name[MAX_FILE_NAME+1];
 extern bool horizontal;
 extern CHARTYPE display_screens;
 extern short compatible;
/*--------------------------- local data ------------------------------*/
 short x=0,fpath_len=0,max_name=0;
 LINETYPE line_number=0L;
 CHARTYPE buffer[60];
 CHARTYPE display_path[MAX_FILE_NAME+1];
 CHARTYPE *fpath = display_path;
 short num_to_delete=0,num_to_start=0,col=0;
 short size_col=0,line_col=0;
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_heading");
#endif
/*---------------------------------------------------------------------*/
/* If IDLINE is not to be displayed, exit now.                         */
/*---------------------------------------------------------------------*/
 if (!CURRENT_VIEW->id_line)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Reset idline to blank.                                              */
/*---------------------------------------------------------------------*/
 wattrset(CURRENT_WINDOW_IDLINE,set_colour(CURRENT_FILE->attr+ATTR_IDLINE));
 wmove(CURRENT_WINDOW_IDLINE,0,0);
 my_wclrtoeol(CURRENT_WINDOW_IDLINE);
/*---------------------------------------------------------------------*/
/* Get line,col values only if POSITION is ON...                       */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->position_status)
    (void)get_current_position(&line_number,&x);
/*---------------------------------------------------------------------*/
/* Set up buffer for line,col,size and alt values for vertical screens.*/
/*---------------------------------------------------------------------*/
 if (display_screens != 1 && !horizontal)
   {
    if (CURRENT_VIEW->position_status)
      {
       if (compatible == COMPAT_XEDIT)
         {
          sprintf(buffer,"S=%-.1ld L=%-.1ld C=%-.1d A=%d,%d",
                      CURRENT_FILE->number_lines,
                      line_number,x,
                      CURRENT_FILE->autosave_alt,
                      CURRENT_FILE->save_alt);
         }
       else
         {
          sprintf(buffer,"L=%-.1ld C=%-.1d S=%-.1ld A=%d,%d",
                      line_number,x,
                      CURRENT_FILE->number_lines,
                      CURRENT_FILE->autosave_alt,
                      CURRENT_FILE->save_alt);
         }
      }
    else
      {
       sprintf(buffer,"S=%-.1ld A=%d,%d",
                      CURRENT_FILE->number_lines,
                      CURRENT_FILE->autosave_alt,
                      CURRENT_FILE->save_alt);
      }
    max_name = (CURRENT_SCREEN.screen_cols-1) - strlen(buffer);
   }
 else
   {
    if (CURRENT_VIEW->position_status)
       max_name = (CURRENT_SCREEN.screen_cols-48);
    else
       max_name = (CURRENT_SCREEN.screen_cols-27);
    if (compatible == COMPAT_XEDIT)
      {
       size_col = CURRENT_SCREEN.screen_cols-46;
       line_col = CURRENT_SCREEN.screen_cols-46+12;
      }
    else
      {
       size_col = CURRENT_SCREEN.screen_cols-46+21;
       line_col = CURRENT_SCREEN.screen_cols-46;
      }
   }
/*---------------------------------------------------------------------*/
/* Determine which portion of filename can be displayed.               */
/*---------------------------------------------------------------------*/
 switch(CURRENT_FILE->pseudo_file)
   {
    case PSEUDO_DIR:
         strcpy(display_path,"DIR: ");
         strcat(display_path,dir_path);
         strcat(display_path,dir_files);
         break;
    case PSEUDO_REXX:
         strcpy(display_path,"Output from: ");
         strcat(display_path,rexx_macro_name);
         break;
    default:
         strcpy(display_path,CURRENT_FILE->fpath);
         strcat(display_path,CURRENT_FILE->fname);
         break;
   }
 fpath = strtrans(display_path,ISLASH,ESLASH);
 fpath_len = strlen(fpath);
 if (fpath_len > max_name)
   {
    num_to_delete = fpath_len - max_name + 2;
    num_to_start = (strlen(fpath)/2)-(num_to_delete/2);
    for (i=0;i<num_to_start;i++)
      {
       mvwaddch(CURRENT_WINDOW_IDLINE,0,i,display_path[i]);
      }
    col = i+2;
    waddstr(CURRENT_WINDOW_IDLINE,"<>");
    for (i=num_to_start+num_to_delete;i<strlen(fpath);i++,col++)
      {
       mvwaddch(CURRENT_WINDOW_IDLINE,0,col,display_path[i]);
      }
   }
 else
  {
   wmove(CURRENT_WINDOW_IDLINE,0,0);
   wprintw(CURRENT_WINDOW_IDLINE,"%s",fpath);
  }
 if (display_screens != 1 && !horizontal)
   {
    wmove(CURRENT_WINDOW_IDLINE,0,max_name+1);
    wprintw(CURRENT_WINDOW_IDLINE,"%-s",buffer);
   }
 else
   {
    if (CURRENT_VIEW->position_status)
      {
       wmove(CURRENT_WINDOW_IDLINE,0,line_col);
       wprintw(CURRENT_WINDOW_IDLINE,"Line=%-6.1ld Col=%-5.1d",line_number,x);
      }
    wmove(CURRENT_WINDOW_IDLINE,0,size_col);
    wprintw(CURRENT_WINDOW_IDLINE,"Size=%-6.1ld",CURRENT_FILE->number_lines);
    wmove(CURRENT_WINDOW_IDLINE,0,CURRENT_SCREEN.screen_cols-46+32);
    wprintw(CURRENT_WINDOW_IDLINE,"Alt=          ");
    wmove(CURRENT_WINDOW_IDLINE,0,CURRENT_SCREEN.screen_cols-46+36);
    wprintw(CURRENT_WINDOW_IDLINE,"%d,%d",CURRENT_FILE->autosave_alt,
                             CURRENT_FILE->save_alt);
   }
#ifdef MSWIN
 Win31Scroll(x,line_number,CURRENT_FILE->number_lines,CURRENT_FILE->max_line_length);
#endif
 wnoutrefresh(CURRENT_WINDOW_IDLINE);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void show_footing(void)
#else
void show_footing()
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE *rec;
 extern CHARTYPE *cmd_rec;
 extern CHARTYPE *pre_rec;
 extern bool ETMODEx;
 extern CHARTYPE *the_version;
 extern CHARTYPE *the_release;
 extern CHARTYPE *the_copyright;
 extern bool CLOCKx;
 extern bool HEXDISPLAYx;
 extern bool INSERTMODEx;
 extern CHARTYPE number_of_files;
 extern WINDOW *foot;
 extern bool colour_support;
 extern bool initial;
 extern bool rexx_support;
 extern ROWTYPE STATUSLINEx;
/*--------------------------- local data ------------------------------*/
 register int i=0;
 short y=0,x=0;
 short key=0;
 WINDOW *w=NULL;
 time_t timer;
 struct tm *tblock=NULL;
 char rel[8];                                  /* not initialised - OK */
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_footing");
#endif
/*---------------------------------------------------------------------*/
/* If the status line is off, just exit...                             */
/*---------------------------------------------------------------------*/
 if (STATUSLINEx == 'O')
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* If GUI option set for status line...                                */
/*---------------------------------------------------------------------*/
 if (STATUSLINEx == 'G')
   {
#ifdef MSWIN
   Show_GUI_footing();
# ifdef TRACE
    trace_return();
# endif
    return;
#endif

#ifdef TRACE
    trace_return();
#endif
    return;
   }
 w = CURRENT_WINDOW;
/*---------------------------------------------------------------------*/
/* Display THE version.                                                */
/*---------------------------------------------------------------------*/
 strcpy(rel,"THE ");
 strcat(rel,the_version);
/* strcat(rel,the_release+10);*/
 mvwaddstr(foot,0,0,rel);
/*---------------------------------------------------------------------*/
/* Clear from files to clock position...                               */
/*---------------------------------------------------------------------*/
 for (i=12;i<53;i++)
   {
    wmove(foot,0,i);
    waddch(foot,' ');
   }
/*---------------------------------------------------------------------*/
/* Display number of files or copyright on startup.                    */
/*---------------------------------------------------------------------*/
 wmove(foot,0,12);
 if (initial)
   {
    waddstr(foot,the_copyright);
    initial = FALSE;
   }
 else
    wprintw(foot,"Files=%d ",number_of_files);
/*---------------------------------------------------------------------*/
/* Display any pending prefix command warning                          */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->first_ppc != NULL
 &&  CURRENT_FILE->first_ppc->ppc_cmd_idx != (-1)
 &&  CURRENT_FILE->first_ppc->ppc_cmd_idx != (-2))
   {
    wmove(foot,0,20);
    wprintw(foot,"'%s' pending...",get_prefix_command(CURRENT_FILE->first_ppc->ppc_cmd_idx));
   }
/*---------------------------------------------------------------------*/
/* Display CLOCK.                                                      */
/*---------------------------------------------------------------------*/
 if (CLOCKx)
   {
    timer = time(NULL);
    tblock = localtime(&timer);
    wmove(foot,0,53);

    wprintw(foot,"%2d:%02.2d%s",
           (tblock->tm_hour > 12) ? (tblock->tm_hour-12) : (tblock->tm_hour),
            tblock->tm_min,
           (tblock->tm_hour >= 12) ? ("pm") : ("am"));
   }
/*---------------------------------------------------------------------*/
/* Display HEXDISPLAY.                                                 */
/*---------------------------------------------------------------------*/
 if (HEXDISPLAYx)
   {
    getyx(CURRENT_WINDOW,y,x);
    switch(CURRENT_VIEW->current_window)
      {
       case WINDOW_MAIN:
            key = (short)(*(rec+CURRENT_VIEW->verify_col-1+x) & A_CHARTEXT);
            break;
       case WINDOW_COMMAND:
            key = (short)(*(cmd_rec+x) & A_CHARTEXT);
            break;
       case WINDOW_PREFIX:
            key = (short)(*(pre_rec+x) & A_CHARTEXT);
            break;
      }
    wmove(foot,0,61);
    if (key == '\0')
       wprintw(foot,"' '=00/000 ");
    else
      {
       if (ETMODEx)
          wprintw(foot,"'%1.1c'=%2.2X/%3.3d ",key,key,key);
       else
          wprintw(foot,"'%s'=%2.2X/%3.3d ",(key >127) ? " " : unctrl(key),key,key);
      }
   }
/*---------------------------------------------------------------------*/
/* Display colour setting.                                             */
/*---------------------------------------------------------------------*/
 wmove(foot,0,73);
#ifdef A_COLOR
 if (colour_support)
    waddch(foot,'C');
 else
    waddch(foot,'c');
#else
 waddch(foot,'M');
#endif
/*---------------------------------------------------------------------*/
/* Display REXX support character.                                     */
/*---------------------------------------------------------------------*/
 wmove(foot,0,74);
 if (rexx_support)
    waddch(foot,'R');
 else
    waddch(foot,' ');
/*---------------------------------------------------------------------*/
/* Display INSERTMODE toggle.                                          */
/*---------------------------------------------------------------------*/
 wmove(foot,0,76);
 if (INSERTMODEx)
    waddstr(foot,"Ins");
 else
    waddstr(foot,"   ");
/*---------------------------------------------------------------------*/
/* Refresh the STATUS LINE.                                            */
/*---------------------------------------------------------------------*/
 wnoutrefresh(foot);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void clear_footing(void)
#else
void clear_footing()
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern WINDOW *foot;
 extern ROWTYPE STATUSLINEx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    clear_footing");
#endif
/*---------------------------------------------------------------------*/
/* If the status line is not displayed, don't do anything.             */
/*---------------------------------------------------------------------*/
 switch(STATUSLINEx)
   {
    case 'T':
    case 'B':
         wmove(foot,0,0);
         my_wclrtoeol(foot);
         break;
    default:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void redraw_window(WINDOW *win)
#else
void redraw_window(win)
WINDOW *win;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0,j=0;
 chtype ch=0;
 short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    redraw_window");
#endif
 getyx(win,y,x);
 for (i=0;i<getmaxx(win);i++)
     for (j=0;j<getmaxy(win);j++)
        {
         wmove(win,j,i);
         ch = (chtype)(winch(win) & A_CHARTEXT);
         put_char(win,ch,ADDCHAR);
        }
 wmove(win,y,x);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void repaint_screen(void)
#else
void repaint_screen()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    repaint_screen");
#endif

 getyx(CURRENT_WINDOW,y,x);
 y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                            CURRENT_VIEW->current_row);
 if (x > CURRENT_SCREEN.cols[WINDOW_MAIN])
    x = 0;
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();
 cleanup_command_line();
/* show_heading();*/
 wmove(CURRENT_WINDOW,y,x);

#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void build_current_screen(void)
#else
void build_current_screen()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 LINE *curr=NULL;
 LINE *save_curr=NULL;
 short crow = CURRENT_VIEW->current_row;
 LINETYPE cline = CURRENT_VIEW->current_line;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    build_current_screen");
#endif
 hexshow_curr = save_curr = curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,
                                            cline,CURRENT_FILE->number_lines);
/*---------------------------------------------------------------------*/
/* Build   the file contents from the current line to the bottom of the*/
/* window.                                                             */
/*---------------------------------------------------------------------*/
 build_lines(DIRECTION_FORWARD,curr,CURRENT_SCREEN.rows[WINDOW_MAIN]-crow,crow);
/*---------------------------------------------------------------------*/
/* Build   the file contents from the current line to the top of the   */
/* window.                                                             */
/*---------------------------------------------------------------------*/
 curr = save_curr->prev;
 build_lines(DIRECTION_BACKWARD,curr,crow,crow-1);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void build_other_screen(void)
#else
void build_other_screen()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 CHARTYPE save_current_screen=0;
 VIEW_DETAILS *save_current_view=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    build_other_screen");
#endif
 save_current_view = CURRENT_SCREEN.screen_view;
 save_current_screen = current_screen;
 CURRENT_VIEW  = OTHER_SCREEN.screen_view;
 current_screen = (current_screen==0)?1:0;
 build_current_screen();
 current_screen = save_current_screen;
 CURRENT_VIEW = save_current_view;
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void display_current_screen(void)
#else
void display_current_screen()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_macro;
 extern bool in_profile;
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 WINDOW *previous_window=NULL;
 unsigned short x=0,y=0;
 unsigned short savex=0,savey=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    display_current_screen");
#endif
/*---------------------------------------------------------------------*/
/* We don't display the screen if we are in a macro, in the profile or */
/* curses hasn't started yet...                                        */
/*---------------------------------------------------------------------*/
 if (in_profile
 || in_macro
 || !curses_started)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Turn off the cursor.                                                */
/*---------------------------------------------------------------------*/
 draw_cursor(FALSE);
/*---------------------------------------------------------------------*/
/* Display the IDLINE window...                                        */
/*---------------------------------------------------------------------*/
 show_heading();
/*---------------------------------------------------------------------*/
/* Display the ARROW and CMDLINE if on...                              */
/*---------------------------------------------------------------------*/
 if (CURRENT_WINDOW_COMMAND != NULL)
   {
    wattrset(CURRENT_WINDOW_COMMAND,set_colour(CURRENT_FILE->attr+ATTR_CMDLINE));
    redraw_window(CURRENT_WINDOW_COMMAND);
    touchwin(CURRENT_WINDOW_COMMAND);
    wnoutrefresh(CURRENT_WINDOW_COMMAND);
   }
 if (CURRENT_WINDOW_ARROW != NULL)
   {
    wattrset(CURRENT_WINDOW_ARROW,set_colour(CURRENT_FILE->attr+ATTR_ARROW));
    redraw_window(CURRENT_WINDOW_ARROW);
    touchwin(CURRENT_WINDOW_ARROW);
    wnoutrefresh(CURRENT_WINDOW_ARROW);
   }
/*---------------------------------------------------------------------*/
/* Save the position of previous window if on command line.            */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
    getyx(PREVIOUS_WINDOW,savey,savex);
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* Display the built lines...                                          */
/*---------------------------------------------------------------------*/
 show_lines();
/*---------------------------------------------------------------------*/
/* Refresh the windows.                                                */
/*---------------------------------------------------------------------*/
 if (CURRENT_WINDOW_PREFIX != NULL)
     wnoutrefresh(CURRENT_WINDOW_PREFIX);
 wrefresh(CURRENT_WINDOW_MAIN);
/*---------------------------------------------------------------------*/
/* Lastly, turn the cursor back on again.                              */
/*---------------------------------------------------------------------*/
 draw_cursor(TRUE);
/*---------------------------------------------------------------------*/
/* Restore the position of previous window if on command line.         */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
    wmove(PREVIOUS_WINDOW,savey,savex);
 wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void display_other_screen(void)
#else
void display_other_screen()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 CHARTYPE save_current_screen=0;
 VIEW_DETAILS *save_current_view=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    display_other_screen");
#endif
 save_current_view = CURRENT_SCREEN.screen_view;
 save_current_screen = current_screen;
 CURRENT_VIEW  = OTHER_SCREEN.screen_view;
 current_screen = (current_screen==0)?1:0;
 display_current_screen();
 current_screen = save_current_screen;
 CURRENT_VIEW = save_current_view;
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
static void build_lines(short direction,LINE *curr,
                         short rows,short start_row)
#else
static void build_lines(direction,curr,rows,start_row)
short direction;
LINE *curr;
short rows,start_row;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern VIEW_DETAILS *vd_mark;
 extern short compatible;
/*--------------------------- local data ------------------------------*/
 LINETYPE cline = CURRENT_VIEW->current_line;
 RESERVED *curr_rsrvd=NULL;
 short num_shadow_lines=0;
 bool marked=FALSE,current=FALSE;
 short tab_actual_row=0;
 short scale_actual_row=0;
 short hexshow_actual_start_row=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    build_lines");
#endif
/*---------------------------------------------------------------------*/
/* Determine the row that is the focus line.                           */
/*---------------------------------------------------------------------*/
 if (direction == DIRECTION_BACKWARD)
    cline--;
 num_shadow_lines = 0;
 while(rows)
   {
    CURRENT_SCREEN.sl[start_row].full_length = FALSE;
    CURRENT_SCREEN.sl[start_row].number_lines_excluded = 0;
    CURRENT_SCREEN.sl[start_row].other_start_col = 0;
    CURRENT_SCREEN.sl[start_row].other_end_col = 0;
    CURRENT_SCREEN.sl[start_row].line_type = LINE_LINE;
    CURRENT_SCREEN.sl[start_row].main_enterable = TRUE;
    CURRENT_SCREEN.sl[start_row].prefix_enterable = TRUE;
/*---------------------------------------------------------------------*/
/* If HEXSHOW is ON...                                                 */
/*---------------------------------------------------------------------*/
    if (CURRENT_VIEW->hexshow_on)
      {
       hexshow_actual_start_row=calculate_actual_row(CURRENT_VIEW->hexshow_base,CURRENT_VIEW->hexshow_off,CURRENT_SCREEN.rows[WINDOW_MAIN]);
       if (hexshow_actual_start_row == start_row
       ||  hexshow_actual_start_row+1 == start_row)
         {
          LINETYPE line_length;
          LINE *line_ptr;
          CURRENT_SCREEN.sl[start_row].line_type = LINE_HEXSHOW;
          CURRENT_SCREEN.sl[start_row].full_length = TRUE;
          CURRENT_SCREEN.sl[start_row].line_number = (-1L);
          CURRENT_SCREEN.sl[start_row].main_enterable = FALSE;
          CURRENT_SCREEN.sl[start_row].prefix_enterable = FALSE;
          CURRENT_SCREEN.sl[start_row].normal_colour = set_colour(CURRENT_FILE->attr+ATTR_SHADOW);
          if (CURRENT_VIEW->current_line == CURRENT_VIEW->focus_line)
            {
             CURRENT_SCREEN.sl[start_row].contents = rec;
             CURRENT_SCREEN.sl[start_row].length = rec_len;
            }
          else
            {
             CURRENT_SCREEN.sl[start_row].contents = hexshow_curr->line;
             CURRENT_SCREEN.sl[start_row].length = hexshow_curr->length;
            }
          if (hexshow_actual_start_row == start_row)
             CURRENT_SCREEN.sl[start_row].other_start_col = 0;
          else
             CURRENT_SCREEN.sl[start_row].other_start_col = 1;
          if (CURRENT_VIEW->prefix) /* display prefix if on */
            {
             memset(CURRENT_SCREEN.sl[start_row].prefix,' ',PREFIX_WIDTH);
             CURRENT_SCREEN.sl[start_row].prefix[PREFIX_WIDTH] = '\0';
             CURRENT_SCREEN.sl[start_row].prefix_colour = set_colour(CURRENT_FILE->attr+ATTR_PREFIX);
            }
          start_row += direction;
          rows--;
          continue;
         }
      }
/*---------------------------------------------------------------------*/
/* If the current line is a reserved line...                           */
/*---------------------------------------------------------------------*/
    if ((curr_rsrvd = find_reserved_line(TRUE,start_row,0,0)) != NULL)
      {
       CURRENT_SCREEN.sl[start_row].line_type = LINE_RESERVED;
       CURRENT_SCREEN.sl[start_row].full_length = TRUE;
       CURRENT_SCREEN.sl[start_row].line_number = (-1L);
       CURRENT_SCREEN.sl[start_row].current = (LINE *)NULL;
       CURRENT_SCREEN.sl[start_row].main_enterable = FALSE;
       CURRENT_SCREEN.sl[start_row].prefix_enterable = FALSE;
       if (CURRENT_VIEW->prefix)               /* display prefix if on */
         {
          CURRENT_SCREEN.sl[start_row].prefix_colour = set_colour(curr_rsrvd->attr);
          if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_LEFT)
            {
             memset(CURRENT_SCREEN.sl[start_row].prefix,' ',PREFIX_WIDTH);
             memcpy(CURRENT_SCREEN.sl[start_row].prefix,curr_rsrvd->line,min(curr_rsrvd->length,PREFIX_WIDTH));
             CURRENT_SCREEN.sl[start_row].prefix[PREFIX_WIDTH] = '\0';
             if (curr_rsrvd->length <= PREFIX_WIDTH)
               {
                CURRENT_SCREEN.sl[start_row].contents = NULL;
                CURRENT_SCREEN.sl[start_row].length = 0;
               }
             else
               {
                CURRENT_SCREEN.sl[start_row].contents = curr_rsrvd->line+PREFIX_WIDTH;
                CURRENT_SCREEN.sl[start_row].length = curr_rsrvd->length-PREFIX_WIDTH;
               }
            }
          else
            {
             CURRENT_SCREEN.sl[start_row].contents = curr_rsrvd->line;
             CURRENT_SCREEN.sl[start_row].length = min(curr_rsrvd->length,CURRENT_SCREEN.cols[WINDOW_MAIN]);
             memset(CURRENT_SCREEN.sl[start_row].prefix,' ',PREFIX_WIDTH);
             CURRENT_SCREEN.sl[start_row].prefix[PREFIX_WIDTH] = '\0';
             if (curr_rsrvd->length >= CURRENT_SCREEN.cols[WINDOW_MAIN])
               {
                memcpy(CURRENT_SCREEN.sl[start_row].prefix,
                       curr_rsrvd->line+CURRENT_SCREEN.cols[WINDOW_MAIN],
                       min(curr_rsrvd->length-CURRENT_SCREEN.cols[WINDOW_MAIN],
                           PREFIX_WIDTH));
               }
            }
         }
       else
         {
          CURRENT_SCREEN.sl[start_row].contents = curr_rsrvd->line;
          CURRENT_SCREEN.sl[start_row].length = curr_rsrvd->length;
         }
       CURRENT_SCREEN.sl[start_row].normal_colour = set_colour(curr_rsrvd->attr);
       CURRENT_SCREEN.sl[start_row].other_colour = set_colour(curr_rsrvd->attr);
       start_row += direction;
       rows--;
       continue;
      }
/*---------------------------------------------------------------------*/
/* If the current line is the scale or tab line...                     */
/*---------------------------------------------------------------------*/
    tab_actual_row=calculate_actual_row(CURRENT_VIEW->tab_base,CURRENT_VIEW->tab_off,CURRENT_SCREEN.rows[WINDOW_MAIN]);
    scale_actual_row=calculate_actual_row(CURRENT_VIEW->scale_base,CURRENT_VIEW->scale_off,CURRENT_SCREEN.rows[WINDOW_MAIN]);
    if ((CURRENT_VIEW->scale_on && scale_actual_row == start_row)
    || (CURRENT_VIEW->tab_on && tab_actual_row == start_row))
      {
       CURRENT_SCREEN.sl[start_row].contents = NULL;
       CURRENT_SCREEN.sl[start_row].length = 0;
       CURRENT_SCREEN.sl[start_row].line_number = (-1L);
       CURRENT_SCREEN.sl[start_row].current = (LINE *)NULL;
       CURRENT_SCREEN.sl[start_row].main_enterable = FALSE;
       CURRENT_SCREEN.sl[start_row].prefix_enterable = FALSE;
       if (CURRENT_VIEW->prefix) /* display prefix if on */
         {
          memset(CURRENT_SCREEN.sl[start_row].prefix,' ',PREFIX_WIDTH);
          CURRENT_SCREEN.sl[start_row].prefix[PREFIX_WIDTH] = '\0';
          CURRENT_SCREEN.sl[start_row].prefix_colour = set_colour(CURRENT_FILE->attr+ATTR_PREFIX);
         }
       if (CURRENT_VIEW->tab_on && tab_actual_row == start_row)
         {
          CURRENT_SCREEN.sl[start_row].line_type |= LINE_TABLINE;
          CURRENT_SCREEN.sl[start_row].normal_colour = set_colour(CURRENT_FILE->attr+ATTR_TABLINE);
         }
       if (CURRENT_VIEW->scale_on && scale_actual_row == start_row)
         {
          CURRENT_SCREEN.sl[start_row].line_type |= LINE_SCALE;
          CURRENT_SCREEN.sl[start_row].normal_colour = set_colour(CURRENT_FILE->attr+ATTR_SCALE);
         }
       start_row += direction;
       rows--;
       continue;
      }
/*---------------------------------------------------------------------*/
/* If the current line is above or below TOF or EOF, set all to blank. */
/*---------------------------------------------------------------------*/
    if (curr == NULL)
      {
       CURRENT_SCREEN.sl[start_row].contents = NULL;
       CURRENT_SCREEN.sl[start_row].length = 0;
       CURRENT_SCREEN.sl[start_row].normal_colour = set_colour(CURRENT_FILE->attr+ATTR_FILEAREA);
       CURRENT_SCREEN.sl[start_row].line_type = (direction == DIRECTION_BACKWARD) ? LINE_OUT_OF_BOUNDS_ABOVE : LINE_OUT_OF_BOUNDS_BELOW;
       CURRENT_SCREEN.sl[start_row].line_number = (-1L);
       CURRENT_SCREEN.sl[start_row].current = (LINE *)NULL;
       CURRENT_SCREEN.sl[start_row].main_enterable = FALSE;
       CURRENT_SCREEN.sl[start_row].prefix_enterable = FALSE;
       if (CURRENT_VIEW->prefix) /* display prefix if on */
         {
          memset(CURRENT_SCREEN.sl[start_row].prefix,' ',PREFIX_WIDTH);
          CURRENT_SCREEN.sl[start_row].prefix[PREFIX_WIDTH] = '\0';
          CURRENT_SCREEN.sl[start_row].prefix_colour = set_colour(CURRENT_FILE->attr+ATTR_PREFIX);
         }
       start_row += direction;
       rows--;
       continue;
      }
/*---------------------------------------------------------------------*/
/* If the current line is excluded, increment a running total.         */
/* Ignore the line if on TOF or BOF.                                   */
/*---------------------------------------------------------------------*/
    if (curr->next != NULL                           /* Bottom of file */
    &&  curr->prev != NULL)                             /* Top of file */
      {
       if (in_scope(CURRENT_VIEW,curr)
       || cline == CURRENT_VIEW->current_line
       || curr->pre != NULL)
          ;
       else
         {
          if (num_shadow_lines == 0
          && direction == DIRECTION_FORWARD)
            {
             set_prefix_contents(curr,start_row,cline);
             CURRENT_SCREEN.sl[start_row].line_number = cline;
             CURRENT_SCREEN.sl[start_row].current = curr;
            }
          num_shadow_lines++;
          cline += (LINETYPE)direction;
          if (direction == DIRECTION_FORWARD)
             curr = curr->next;
          else
             curr = curr->prev;
          continue;
         }
      }
/*---------------------------------------------------------------------*/
/* If we get here, we have to determine if a shadow line is to be      */
/* displayed or not.                                                   */
/*---------------------------------------------------------------------*/
    if (CURRENT_VIEW->shadow
    && num_shadow_lines > 0)
      {
       CURRENT_SCREEN.sl[start_row].length = 0;
       if (direction != DIRECTION_FORWARD)
         {
          set_prefix_contents(curr->next,start_row,cline+1);
          CURRENT_SCREEN.sl[start_row].line_number = cline+1;
          CURRENT_SCREEN.sl[start_row].current = curr;
         }
       CURRENT_SCREEN.sl[start_row].normal_colour = set_colour(CURRENT_FILE->attr+ATTR_SHADOW);
       CURRENT_SCREEN.sl[start_row].full_length = TRUE;
       CURRENT_SCREEN.sl[start_row].number_lines_excluded = num_shadow_lines;
       CURRENT_SCREEN.sl[start_row].line_type = LINE_SHADOW;
       if (compatible == COMPAT_XEDIT)
          CURRENT_SCREEN.sl[start_row].main_enterable = FALSE;
       num_shadow_lines = 0;
       start_row += direction;
       rows--;
       continue;
      }
/*---------------------------------------------------------------------*/
/* Determine if line being processed is the current line.              */
/*---------------------------------------------------------------------*/
    if (cline == CURRENT_VIEW->current_line)
       current = TRUE;
    else
       current = FALSE;
/*---------------------------------------------------------------------*/
/* Determine if line being processed is in a marked block.             */
/*---------------------------------------------------------------------*/
    if (MARK_VIEW != (VIEW_DETAILS *)NULL
    &&  MARK_VIEW == CURRENT_VIEW)
      {
       if (cline >= MARK_VIEW->mark_start_line
       &&  cline <= MARK_VIEW->mark_end_line)
          marked = TRUE;
       else
          marked = FALSE;
      }
/*---------------------------------------------------------------------*/
/* The remainder is for lines that are to be displayed.                */
/*---------------------------------------------------------------------*/
    CURRENT_SCREEN.sl[start_row].line_number = cline;
    CURRENT_SCREEN.sl[start_row].current = curr;
/*---------------------------------------------------------------------*/
/* If the current row to be displayed is the focus line, display       */
/* the working area, rec and rec_len instead of the entry in the LL.   */
/*---------------------------------------------------------------------*/
    if (cline == CURRENT_VIEW->focus_line)
      {
       CURRENT_SCREEN.sl[start_row].contents = rec;
       CURRENT_SCREEN.sl[start_row].length = rec_len;
      }
    else
      {
       CURRENT_SCREEN.sl[start_row].contents = curr->line;
       CURRENT_SCREEN.sl[start_row].length = curr->length;
      }
    set_prefix_contents(curr,start_row,cline);
/*---------------------------------------------------------------------*/
/* Set up TOF and EOF lines...                                         */
/*---------------------------------------------------------------------*/
    if (curr->next == NULL                           /* Bottom of file */
    ||  curr->prev == NULL)                             /* Top of file */
      {
       CURRENT_SCREEN.sl[start_row].normal_colour = (current) ? set_colour(CURRENT_FILE->attr+ATTR_CTOFEOF) :
                                                   set_colour(CURRENT_FILE->attr+ATTR_TOFEOF);
       CURRENT_SCREEN.sl[start_row].line_type = LINE_TOF_EOF;
       if (compatible == COMPAT_XEDIT)
          CURRENT_SCREEN.sl[start_row].main_enterable = FALSE;
      }
    else
      {
       if (marked)
         {
          switch(MARK_VIEW->mark_type)
            {
             case M_LINE:
                  CURRENT_SCREEN.sl[start_row].other_start_col = (-1);
                  CURRENT_SCREEN.sl[start_row].other_end_col = (-1);
                  CURRENT_SCREEN.sl[start_row].normal_colour = (current) ? set_colour(CURRENT_FILE->attr+ATTR_CBLOCK) :
                                                               set_colour(CURRENT_FILE->attr+ATTR_BLOCK);
                  CURRENT_SCREEN.sl[start_row].full_length = TRUE;
                  break;
             case M_BOX:
             case M_COLUMN:
             case M_WORD:
                  CURRENT_SCREEN.sl[start_row].other_start_col = MARK_VIEW->mark_start_col - 1;
                  CURRENT_SCREEN.sl[start_row].other_end_col = MARK_VIEW->mark_end_col - 1;
                  CURRENT_SCREEN.sl[start_row].normal_colour = (current) ? set_colour(CURRENT_FILE->attr+ATTR_CURLINE) :
                                                               set_colour(CURRENT_FILE->attr+ATTR_FILEAREA);
                  CURRENT_SCREEN.sl[start_row].other_colour = (current) ? set_colour(CURRENT_FILE->attr+ATTR_CBLOCK) :
                                                              set_colour(CURRENT_FILE->attr+ATTR_BLOCK);
                  break;
             case M_STREAM:
                  break;
            }
         }
       else
         {
          CURRENT_SCREEN.sl[start_row].normal_colour = (current) ? set_colour(CURRENT_FILE->attr+ATTR_CURLINE) :
                                                      set_colour(CURRENT_FILE->attr+ATTR_FILEAREA);
          CURRENT_SCREEN.sl[start_row].other_colour = CURRENT_SCREEN.sl[start_row].normal_colour;
         }
      }
    start_row += direction;
    rows--;

    cline += (LINETYPE)direction;
    if (direction == DIRECTION_FORWARD)
       curr = curr->next;
    else
       curr = curr->prev;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
static void show_lines(void)
#else
static void show_lines()
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool ETMODEx;
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 register short i=0,j=0;
 short true_col=0;
 short max_cols = min(CURRENT_SCREEN.cols[WINDOW_MAIN],CURRENT_VIEW->verify_end-CURRENT_VIEW->verify_start+1);
 short off=0,num_tens=0;
 unsigned short y=0,x=0;
 CHARTYPE tens[5];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_lines");
#endif
 for (i=0;i<CURRENT_SCREEN.rows[WINDOW_MAIN];i++)
   {
/*---------------------------------------------------------------------*/
/* Display the contents of the prefix area (if on).                    */
/*---------------------------------------------------------------------*/
    if (CURRENT_VIEW->prefix)
      {
       wattrset(CURRENT_WINDOW_PREFIX,CURRENT_SCREEN.sl[i].prefix_colour);
       wmove(CURRENT_WINDOW_PREFIX,i,0);
       my_wclrtoeol(CURRENT_WINDOW_PREFIX);
       if (ETMODEx)
          waddstr(CURRENT_WINDOW_PREFIX,CURRENT_SCREEN.sl[i].prefix);
       else
          put_string(CURRENT_WINDOW_PREFIX,i,0,CURRENT_SCREEN.sl[i].prefix,strlen(CURRENT_SCREEN.sl[i].prefix));
      }
    wmove(CURRENT_WINDOW_MAIN,i,0);
    wattrset(CURRENT_WINDOW_MAIN,CURRENT_SCREEN.sl[i].normal_colour);
/*---------------------------------------------------------------------*/
/* Display any shadow line. No need to test to see if SHADOW is ON as  */
/* number_excluded_lines would not be > 0 if SHADOW OFF.               */
/*---------------------------------------------------------------------*/
    if (CURRENT_SCREEN.sl[i].number_lines_excluded > 0)
      {
       for (j=0;j<CURRENT_SCREEN.cols[WINDOW_MAIN];j++)
          mvwaddch(CURRENT_WINDOW_MAIN,i,j,'-');
       wmove(CURRENT_WINDOW_MAIN,i,max(0,(CURRENT_SCREEN.cols[WINDOW_MAIN]/2)-14));
       wprintw(CURRENT_WINDOW_MAIN,"%4d line(s) not displayed ",CURRENT_SCREEN.sl[i].number_lines_excluded);
       continue;
      }
/*---------------------------------------------------------------------*/
/* Display SCALE and/or TABLINE...                                     */
/*---------------------------------------------------------------------*/
    if (CURRENT_SCREEN.sl[i].line_type & LINE_SCALE
    ||  CURRENT_SCREEN.sl[i].line_type & LINE_TABLINE)
      {
       my_wclrtoeol(CURRENT_WINDOW_MAIN);
       for (j=0;j<max_cols;j++)
         {
          wmove(CURRENT_WINDOW_MAIN,i,j);
          true_col = j + CURRENT_VIEW->verify_col-1;
/*---------------------------------------------------------------------*/
/* Display '|' in current column position if this is the scale line.   */
/*---------------------------------------------------------------------*/
          if (CURRENT_SCREEN.sl[i].line_type & LINE_SCALE
          &&  CURRENT_VIEW->current_column == true_col+1)
            {
             waddch(CURRENT_WINDOW_MAIN,'|');
             continue;
            }
/*---------------------------------------------------------------------*/
/* Display 'T' in each tab column. This overrides all other characters */
/* except column position.                                             */
/*---------------------------------------------------------------------*/
          if (CURRENT_SCREEN.sl[i].line_type & LINE_TABLINE)
            {
             if (is_tab_col(true_col+1))
               {
                waddch(CURRENT_WINDOW_MAIN,'T');
                continue;
               }
            }
/*---------------------------------------------------------------------*/
/* Only display the following if it is a scale line...                 */
/*---------------------------------------------------------------------*/
          if (CURRENT_SCREEN.sl[i].line_type & LINE_SCALE)
            {
             if (CURRENT_VIEW->margin_left-1 == true_col)
               {
                waddch(CURRENT_WINDOW_MAIN,'[');
                continue;
               }
             if (CURRENT_VIEW->margin_right-1 == true_col)
               {
                waddch(CURRENT_WINDOW_MAIN,']');
                continue;
               }
             if (CURRENT_VIEW->margin_indent_offset
             &&  CURRENT_VIEW->margin_left+CURRENT_VIEW->margin_indent-1 == true_col)
               {
                waddch(CURRENT_WINDOW_MAIN,'p');
                continue;
               }
             if (!CURRENT_VIEW->margin_indent_offset
             &&  CURRENT_VIEW->margin_indent-1 == true_col)
               {
                waddch(CURRENT_WINDOW_MAIN,'p');
                continue;
               }
             if (CURRENT_VIEW->zone_start-1 == true_col)
               {
                waddch(CURRENT_WINDOW_MAIN,'<');
                continue;
               }
             if (CURRENT_VIEW->zone_end-1 == true_col)
               {
                waddch(CURRENT_WINDOW_MAIN,'>');
/*              break;*/
                continue;
               }
             if (true_col % 10 == 9)
               {
                sprintf(tens,"%d",(true_col / 10) + 1);
                num_tens = strlen(tens);
                getyx(CURRENT_WINDOW_MAIN,y,x);
                if ((short)x-(num_tens-1) < 0)
                  {
                   off = -((short)x-(num_tens-1));
                   x = 0;
                  }
                else
                  {
                   off = 0;
                   x -= (num_tens-1);
                  }
                wmove(CURRENT_WINDOW_MAIN,y,x);
                waddstr(CURRENT_WINDOW_MAIN,tens+off);
                continue;
               }
             if (true_col % 5 == 4)
               {
                waddch(CURRENT_WINDOW_MAIN,'+');
                continue;
               }
             waddch(CURRENT_WINDOW_MAIN,'.');
            }
         }
       continue;
      }
/*---------------------------------------------------------------------*/
/* Display HEXSHOW line...                                             */
/*---------------------------------------------------------------------*/
    if (CURRENT_SCREEN.sl[i].line_type & LINE_HEXSHOW)
      {
       my_wclrtoeol(CURRENT_WINDOW_MAIN);
       show_hex_line(i);
       continue;
      }
/*---------------------------------------------------------------------*/
/* Display TOF or EOF line.                                            */
/*---------------------------------------------------------------------*/
    if (CURRENT_SCREEN.sl[i].line_type == LINE_TOF_EOF)
      {
       my_wclrtoeol(CURRENT_WINDOW_MAIN);
       waddstr(CURRENT_WINDOW_MAIN,CURRENT_SCREEN.sl[i].contents);
       continue;
      }
/*---------------------------------------------------------------------*/
/* Display marked LINE block line(s).                                  */
/*---------------------------------------------------------------------*/
    show_a_line(i);
   }
 if (CURRENT_WINDOW_PREFIX != NULL)
    wattrset(CURRENT_WINDOW_PREFIX,set_colour(CURRENT_FILE->attr+ATTR_PENDING));
 wattrset(CURRENT_WINDOW_MAIN,set_colour(CURRENT_FILE->attr+ATTR_FILEAREA));
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
static void show_a_line(short row)
#else
static void show_a_line(row)
short row;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 chtype ch=0;
 LENGTHTYPE vcol=0,vend=0,vlen=0;
 LENGTHTYPE length=0;
 CHARTYPE *line=NULL;
#if defined(USE_EXTCURSES)
 chtype attr=0,old_attr=0;
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_a_line");
#endif
/*---------------------------------------------------------------------*/
/* If the line to be displayed is a reserved line, set the columns to  */
/* be displayed so that the full line is displayed.                    */
/*---------------------------------------------------------------------*/
 if (CURRENT_SCREEN.sl[row].line_type == LINE_RESERVED)
   {
    vcol = 0;
    vend = CURRENT_SCREEN.cols[WINDOW_MAIN];
    vlen = CURRENT_SCREEN.cols[WINDOW_MAIN];
   }
 else
   {
    vcol = CURRENT_VIEW->verify_col - 1;
    vend = CURRENT_VIEW->verify_end - 1;
    vlen = CURRENT_VIEW->verify_end - CURRENT_VIEW->verify_start + 1;
   }
 length = CURRENT_SCREEN.sl[row].length;
 line = CURRENT_SCREEN.sl[row].contents;
/*---------------------------------------------------------------------*/
/* If the contents are NULL then clear to eol in normal colour.        */
/*---------------------------------------------------------------------*/
 if (line == NULL)
   {
    my_wclrtoeol(CURRENT_WINDOW_MAIN);
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 wmove(CURRENT_WINDOW_MAIN,row,0);
 for (i=0;i<CURRENT_SCREEN.cols[WINDOW_MAIN];i++)
   {
/*---------------------------------------------------------------------*/
/* If the last character in the line has been displayed, display blank.*/
/*---------------------------------------------------------------------*/
    if (i+vcol < length)
       ch = *(line+i+vcol);
    else
       ch = ' ';
/*---------------------------------------------------------------------*/
/* Determine colour of character to be displayed. BOX blocks are sorted*/
/* out here.                                                           */
/*---------------------------------------------------------------------*/
    if (vcol+i >= CURRENT_SCREEN.sl[row].other_start_col
    &&  vcol+i <= CURRENT_SCREEN.sl[row].other_end_col)
#if defined(USE_EXTCURSES)
       attr = CURRENT_SCREEN.sl[row].other_colour;
#else
       ch = ch | CURRENT_SCREEN.sl[row].other_colour;
#endif
    else
#if defined(USE_EXTCURSES)
       attr = CURRENT_SCREEN.sl[row].normal_colour;
#else
       ch = ch | CURRENT_SCREEN.sl[row].normal_colour;
#endif
/*---------------------------------------------------------------------*/
/* If we have gone past the VERIFY END column, display a blank in the  */
/* normal colour.                                                      */
/*---------------------------------------------------------------------*/
    if (i >= vlen)
      {
#if defined(USE_EXTCURSES)
       ch = ' ';
       attr = CURRENT_SCREEN.sl[row].normal_colour;
#else
       ch = ' ' | CURRENT_SCREEN.sl[row].normal_colour;
#endif
      }
/*---------------------------------------------------------------------*/
/* Go and display the character...                                     */
/*---------------------------------------------------------------------*/
#if defined(USE_EXTCURSES)
    if (attr != old_attr)
      {
       wattrset(CURRENT_WINDOW_MAIN,attr);
       old_attr = attr;
      }
#endif
    put_char(CURRENT_WINDOW_MAIN,ch,ADDCHAR);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
static void set_prefix_contents(LINE *curr,short start_row,LINETYPE cline)
#else
static void set_prefix_contents(curr,start_row,cline)
LINE *curr;
short start_row;
LINETYPE cline;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 short offset=0;
 CHARTYPE *ptr=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    set_prefix_contents");
#endif
 ptr = CURRENT_SCREEN.sl[start_row].prefix;
/*---------------------------------------------------------------------*/
/* Determine in which column we are to start displaying prefix contents*/
/*---------------------------------------------------------------------*/
 if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
 &&  prefix_width != PREFIX_WIDTH)
   {
    offset = 1;
    *(ptr) = ' ';
   }
 else
    offset = 0;
 if (CURRENT_VIEW->prefix)
   {
    if (curr->pre != NULL)                /* prefix command pending... */
/*  && !blank_field(curr->pre->ppc_command))*/    /* ... and not blank */
      {
       strcpy(ptr+offset,curr->pre->ppc_command);
       CURRENT_SCREEN.sl[start_row].prefix_colour = set_colour(CURRENT_FILE->attr+ATTR_PENDING);
      }
    else                             /* no prefix command on this line */
      {
       memset(ptr,' ',PREFIX_WIDTH);
       CURRENT_SCREEN.sl[start_row].prefix[PREFIX_WIDTH] = '\0';
       CURRENT_SCREEN.sl[start_row].prefix_colour = set_colour(CURRENT_FILE->attr+ATTR_PREFIX);
       if (CURRENT_VIEW->number)
         {
          if ((CURRENT_VIEW->prefix&PREFIX_STATUS_MASK) == PREFIX_ON)
             sprintf(ptr+offset,"%*.*ld",prefix_width,prefix_width,cline);
          else
             sprintf(ptr+offset,"%*ld",prefix_width,cline);
         }
       else
         {
          if ((CURRENT_VIEW->prefix&PREFIX_STATUS_MASK) == PREFIX_ON)
             memset(ptr+offset,'=',prefix_width);
         }
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
bool line_in_view(LINETYPE line_number)
#else
bool line_in_view(line_number)
LINETYPE line_number;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 bool result=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    line_in_view");
#endif
 for (i=0;i<CURRENT_SCREEN.rows[WINDOW_MAIN];i++)
   {
    if (CURRENT_SCREEN.sl[i].line_number == line_number)
      {
       result = TRUE;
       break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(result);
}
/***********************************************************************/
#ifdef PROTO
bool column_in_view(LENGTHTYPE column_number)
#else
bool column_in_view(column_number)
LENGTHTYPE column_number;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 bool result=FALSE;
 LENGTHTYPE min_file_col=0,max_file_col=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    column_in_view");
#endif
 min_file_col = CURRENT_VIEW->verify_col - 1;
 max_file_col = CURRENT_VIEW->verify_col + CURRENT_SCREEN.cols[WINDOW_MAIN] - 2;

 if (column_number >= min_file_col
 &&  column_number <= max_file_col)            /* new column in display */
    result = TRUE;
#ifdef TRACE
 trace_return();
#endif
 return(result);
}
/***********************************************************************/
#ifdef PROTO
LINETYPE find_next_current_line(LINETYPE num_pages,short direction)
#else
LINETYPE find_next_current_line(num_pages,direction)
LINETYPE num_pages;
short direction;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 LINETYPE cline = CURRENT_VIEW->current_line;
 short rows=0,num_display_lines=0,num_shadow_lines=0;
 LINE *curr=NULL;
 RESERVED *curr_reserved=CURRENT_FILE->first_reserved;
 short tab_actual_row=calculate_actual_row(CURRENT_VIEW->tab_base,CURRENT_VIEW->tab_off,CURRENT_SCREEN.rows[WINDOW_MAIN]);
 short scale_actual_row=calculate_actual_row(CURRENT_VIEW->scale_base,CURRENT_VIEW->scale_off,CURRENT_SCREEN.rows[WINDOW_MAIN]);
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    find_next_current_line");
#endif
/*---------------------------------------------------------------------*/
/* Determine the number of file lines displayed...                     */
/*---------------------------------------------------------------------*/
 num_display_lines = (CURRENT_SCREEN.rows[WINDOW_MAIN]) - 1;
 for (i=0;curr_reserved!=NULL;i++)
     curr_reserved = curr_reserved->next;
 num_display_lines -= i;
 if (CURRENT_VIEW->scale_on)
    num_display_lines--;
 if (CURRENT_VIEW->tab_on)
    num_display_lines--;
 if (CURRENT_VIEW->hexshow_on)
    num_display_lines = num_display_lines - 2;
 if (CURRENT_VIEW->scale_on
 &&  CURRENT_VIEW->tab_on
 &&  tab_actual_row == scale_actual_row)
    num_display_lines++;

 curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,cline,CURRENT_FILE->number_lines);
 while(num_pages)
   {
    rows = num_display_lines;
    while(rows)
      {
/*---------------------------------------------------------------------*/
/* If the current line is above or below TOF or EOF, set all to blank. */
/*---------------------------------------------------------------------*/
       if (curr == NULL)
         {
          cline = (direction == DIRECTION_FORWARD) ? CURRENT_FILE->number_lines + 1L : 0L;
          num_pages = 1L;
          break;
         }
/*---------------------------------------------------------------------*/
/* If the current line is excluded, increment a running total.         */
/* Ignore the line if on TOF or BOF.                                   */
/*---------------------------------------------------------------------*/
       if (curr->next != NULL                           /* Bottom of file */
       &&  curr->prev != NULL)                             /* Top of file */
         {
          if (!in_scope(CURRENT_VIEW,curr))
            {
             num_shadow_lines++;
             cline += (LINETYPE)direction;
             if (direction == DIRECTION_FORWARD)
                curr = curr->next;
             else
                curr = curr->prev;
             continue;
            }
         }
/*---------------------------------------------------------------------*/
/* If we get here, we have to determine if a shadow line is to be      */
/* displayed or not.                                                   */
/*---------------------------------------------------------------------*/
       if (CURRENT_VIEW->shadow
       && num_shadow_lines > 0)
         {
          num_shadow_lines = 0;
          rows--;
          continue;
         }
       rows--;
       cline += (LINETYPE)direction;
       if (direction == DIRECTION_FORWARD)
          curr = curr->next;
       else
          curr = curr->prev;
      }
    num_pages--;
   }
 if (direction == DIRECTION_FORWARD
 &&  cline > CURRENT_FILE->number_lines+1L)
    cline = CURRENT_FILE->number_lines+1L;
 if (direction == DIRECTION_BACKWARD
 &&  cline < 0L)
    cline = 0L;
 cline = find_next_in_scope(CURRENT_VIEW,(LINE *)NULL,cline,direction);
#ifdef TRACE
 trace_return();
#endif
 return(cline);
}
/***********************************************************************/
#ifdef PROTO
short get_row_for_focus_line(LINETYPE fl,short cr)
#else
short get_row_for_focus_line(fl,cr)
LINETYPE fl;
short cr;
#endif
/***********************************************************************/
/*---------------------------------------------------------------------*/
/* Returns the row within the main window where the focus line is      */
/* placed. If the focus line is off the screen, or out of bounds of the*/
/* current size of the file; <0 or >number_lines, this returns the     */
/* current row.                                                        */
/*---------------------------------------------------------------------*/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    get_row_for_focus_line");
#endif
 for (i=0;i<CURRENT_SCREEN.rows[WINDOW_MAIN];i++)
   {
    if (CURRENT_SCREEN.sl[i].line_number == fl)
      {
#ifdef TRACE
       trace_return();
#endif
       return(i);
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(cr);
}
/***********************************************************************/
#ifdef PROTO
LINETYPE get_focus_line_in_view(LINETYPE fl,unsigned short row)
#else
LINETYPE get_focus_line_in_view(fl,row)
LINETYPE fl;
unsigned short row;
#endif
/***********************************************************************/
/*---------------------------------------------------------------------*/
/* Returns a new focus line if the specified focus line is no longer   */
/* in view, or the same line number if that line is still in view.     */
/*---------------------------------------------------------------------*/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register unsigned short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    get_focus_line_in_view");
#endif
 for (i=row;i<CURRENT_SCREEN.rows[WINDOW_MAIN];i++)
   {
    if (CURRENT_SCREEN.sl[i].line_number != (-1L))
      {
#ifdef TRACE
       trace_return();
#endif
       return(CURRENT_SCREEN.sl[i].line_number);
      }
   }
 for (i=row;i>0;i--)
   {
    if (CURRENT_SCREEN.sl[i].line_number != (-1L))
      {
#ifdef TRACE
       trace_return();
#endif
       return(CURRENT_SCREEN.sl[i].line_number);
      }
   }
/*---------------------------------------------------------------------*/
/* We should never get here as there would be no editable lines in view*/
/*---------------------------------------------------------------------*/
#ifdef TRACE
 trace_return();
#endif
 return(fl);
}
/***********************************************************************/
#ifdef PROTO
LINETYPE calculate_focus_line(LINETYPE fl,LINETYPE cl)
#else
LINETYPE calculate_focus_line(fl,cl)
LINETYPE fl,cl;
#endif
/***********************************************************************/
/*---------------------------------------------------------------------*/
/* Returns the new focus line. If the focus line is still in the       */
/* window, it stays as is. If not,the focus   line becomes the current */
/* line.                                                               */
/*---------------------------------------------------------------------*/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 LINETYPE new_fl=(-1L);
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    calculate_focus_line");
#endif
 for (i=0;i<CURRENT_SCREEN.rows[WINDOW_MAIN];i++)
  {
   if (CURRENT_SCREEN.sl[i].line_number == fl
   &&  (CURRENT_SCREEN.sl[i].line_type == LINE_LINE
     || CURRENT_SCREEN.sl[i].line_type == LINE_TOF_EOF))
     {
      new_fl = fl;
      break;
     }
  }
 if (new_fl == (-1L))
    new_fl = cl;
#ifdef TRACE
 trace_return();
#endif
 return(new_fl);
}
/***********************************************************************/
#ifdef PROTO
static void show_hex_line(short row)
#else
static void show_hex_line(row)
short row;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 chtype ch=0;
 LENGTHTYPE vcol=0,vend=0,vlen=0;
 LENGTHTYPE length=0;
 CHARTYPE *line=NULL;
 int num=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_hex_line");
#endif
/*---------------------------------------------------------------------*/
/* Set up columns to display...                                        */
/*---------------------------------------------------------------------*/
 vcol = CURRENT_VIEW->verify_col - 1;
 vend = CURRENT_VIEW->verify_end - 1;
 vlen = CURRENT_VIEW->verify_end - CURRENT_VIEW->verify_start + 1;
 length = CURRENT_SCREEN.sl[row].length;
 line = CURRENT_SCREEN.sl[row].contents;
#if defined(USE_EXTCURSES)
 wattrset(CURRENT_WINDOW_MAIN,CURRENT_SCREEN.sl[row].normal_colour);
#endif
 wmove(CURRENT_WINDOW_MAIN,row,0);
 for (i=0;i<CURRENT_SCREEN.cols[WINDOW_MAIN];i++)
   {
/*---------------------------------------------------------------------*/
/* If the last character in the line has been displayed, display blank.*/
/*---------------------------------------------------------------------*/
    if (i+vcol < length)
       ch = *(line+i+vcol);
    else
       ch = ' ';
/*---------------------------------------------------------------------*/
/* If we have gone past the VERIFY END column, display a blank.        */
/*---------------------------------------------------------------------*/
    if (i >= vlen)
       ch = ' ';
/*---------------------------------------------------------------------*/
/* Calculate the HEX character to display based on which HEXSHOW line  */
/* is being displayed.                                                 */
/*---------------------------------------------------------------------*/
    if (CURRENT_SCREEN.sl[row].other_start_col == 0)
       num = (int)((ch / 16) + (int)'0');
    else
       num = (int)((ch % 16) + (int)'0');
    num = (num > (int)'9') ? num + 7 : num;
#if defined(USE_EXTCURSES)
    ch = (chtype)num;
#else
    ch = (chtype)num | CURRENT_SCREEN.sl[row].normal_colour;
#endif
/*---------------------------------------------------------------------*/
/* Go and display the character...                                     */
/*---------------------------------------------------------------------*/
    put_char(CURRENT_WINDOW_MAIN,ch,ADDCHAR);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void get_current_position(LINETYPE *line,short *col)
#else
void get_current_position(line,col)
LINETYPE *line;
short *col;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    get_current_position");
#endif
 getyx(CURRENT_WINDOW,y,x);
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_COMMAND:
         *line = CURRENT_VIEW->current_line;
         *col = x+1;
         break;
    case WINDOW_MAIN:
         *line = CURRENT_VIEW->focus_line;
         *col = x + CURRENT_VIEW->verify_col;
         break;
    case WINDOW_PREFIX:
         *line = CURRENT_VIEW->focus_line;
         if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
         &&  prefix_width != PREFIX_WIDTH)
            ;
         else
            x++;
         *col = x;
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
void calculate_new_column(COLTYPE current_screen_col,LENGTHTYPE current_verify_col,
                          LENGTHTYPE new_file_col,COLTYPE *new_screen_col, LENGTHTYPE *new_verify_col)
#else
void calculate_new_column(current_screen_col,current_verify_col,new_file_col,new_screen_col,new_verify_col)
COLTYPE current_screen_col;
LENGTHTYPE current_verify_col,new_file_col;
COLTYPE *new_screen_col;
LENGTHTYPE *new_verify_col;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 LINETYPE x=0,verify_width=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    calculate_new_column");
#endif
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
   {
    *new_screen_col = (LENGTHTYPE)(new_file_col);
#ifdef TRACE
    trace_return();
#endif
    return;
   }
#if 0
 min_file_col = CURRENT_VIEW->verify_col - 1;
 max_file_col = CURRENT_VIEW->verify_col + CURRENT_SCREEN.cols[WINDOW_MAIN] - 2;

 if (new_file_col >= min_file_col
 &&  new_file_col <= max_file_col)            /* new column in display */
#else
 if (column_in_view(new_file_col))
#endif
   {
    *new_screen_col = (LENGTHTYPE)(new_file_col - (current_verify_col - 1));
    *new_verify_col = current_verify_col;
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* To get here, we have new verify column...                           */
/*---------------------------------------------------------------------*/
 x = CURRENT_SCREEN.cols[WINDOW_MAIN] / 2;
 *new_verify_col = (LENGTHTYPE)max(1L,(LINETYPE)new_file_col - x + 2L);
 *new_screen_col = (LENGTHTYPE)((*new_verify_col == 1) ? new_file_col : x - 1);
#ifdef TRACE
 trace_return();
#endif
 return;
}
