/***********************************************************************/
/* CURSOR.C - CURSOR commands                                          */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * 36 David Road                 Phone:                    +617 3849 7731
 * Holland Park                  http://www.gu.edu.au/gwis/the/markh.html
 * Brisbane                      **** Maintainer PDCurses & REXX/SQL ****
 * QLD 4121                      ************* Author of THE ************
 * Australia                     ************* Member RexxLA ************
 */

/*
$Id: cursor.c 2.1 1995/06/24 16:29:25 MH Rel MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/***********************************************************************/
#ifdef PROTO
short cursor_cmdline(short col)
#else
short cursor_cmdline(col)
short col;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  cursor_cmdline");
#endif
/*---------------------------------------------------------------------*/
/* If CMDLINE is OFF return without doing anything.                    */
/*---------------------------------------------------------------------*/
 if (CURRENT_WINDOW_COMMAND == (WINDOW *)NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    CURRENT_VIEW->previous_window = CURRENT_VIEW->current_window;
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
    CURRENT_VIEW->current_window = WINDOW_COMMAND;
   }
 wmove(CURRENT_WINDOW,0,col-1);

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short cursor_column(void)
#else
short cursor_column()
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  cursor_column");
#endif
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_COMMAND:
         rc = cursor_home(FALSE);
         break;
    case WINDOW_PREFIX:
         getyx(CURRENT_WINDOW,y,x);
         CURRENT_VIEW->current_window = WINDOW_MAIN;
         wmove(CURRENT_WINDOW,y,0);
         break;
   }
 rc = execute_move_cursor(CURRENT_VIEW->current_column-1);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short cursor_down(bool escreen)
#else
short cursor_down(escreen)
bool escreen;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE CMDARROWSTABCMDx;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  cursor_down");
#endif
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_PREFIX:
    case WINDOW_MAIN:
         rc = scroll_line(DIRECTION_FORWARD,1L,FALSE,escreen);
         break;
    case WINDOW_COMMAND:
/*---------------------------------------------------------------------*/
/* Cycle forward  through the command list or tab to first line.       */
/*---------------------------------------------------------------------*/
         if (CMDARROWSTABCMDx)
            rc = Sos_topedge("");
         else
            rc = Retrieve((CHARTYPE *)"+");
         break;
    default:
         display_error(2,(CHARTYPE *)"",FALSE);
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short cursor_file(LINETYPE line,LENGTHTYPE col)
#else
short cursor_file(line,col)
LINETYPE line;
LENGTHTYPE col;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 LENGTHTYPE left_col=0,right_col=0;
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  cursor_file");
#endif
/*---------------------------------------------------------------------*/
/* If line is not in display, error.                                   */
/*---------------------------------------------------------------------*/
 if (!line_in_view(line))
   {
    display_error(63,"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If column is not 0 and not in display, error.                       */
/*---------------------------------------------------------------------*/
 if (col == 0)
    x = 1;
 else
   {
    left_col = CURRENT_VIEW->verify_start;
    right_col = min(CURRENT_VIEW->verify_end,CURRENT_SCREEN.cols[WINDOW_MAIN]);
    if (col < left_col
    ||  col > right_col)
      {
       display_error(63,"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    x = col - CURRENT_VIEW->verify_start + 1;
   }
 y = get_row_for_focus_line(line,CURRENT_VIEW->current_line);
 rc = cursor_move(FALSE,y+1,x);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short cursor_home(bool save)
#else
short cursor_home(save)
bool save;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 CHARTYPE last_win=0;
 unsigned short x=0,y=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  cursor_home");
#endif
/*---------------------------------------------------------------------*/
/* If CMDLINE is OFF return without doing anything.                    */
/*---------------------------------------------------------------------*/
 if (CURRENT_WINDOW_COMMAND == (WINDOW *)NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 last_win = CURRENT_VIEW->previous_window;
 CURRENT_VIEW->previous_window =
              CURRENT_VIEW->current_window;
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
    {
     post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
     CURRENT_VIEW->current_window = WINDOW_COMMAND;
     wmove(CURRENT_WINDOW,0,0);
    }
 else
    {
     if (!line_in_view(CURRENT_VIEW->focus_line))
        CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line;
     pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
     if (save)
        CURRENT_VIEW->current_window = last_win;
     else
        CURRENT_VIEW->current_window = WINDOW_MAIN;
     getyx(CURRENT_WINDOW,y,x);
     y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                                CURRENT_VIEW->current_row);
     if (CURRENT_VIEW->current_window == WINDOW_PREFIX
     &&  (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
     &&  prefix_width != PREFIX_WIDTH
     &&  x == 0)
        x = 1;
     wmove(CURRENT_WINDOW,y,x);
    }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short cursor_left(bool escreen,bool kedit_defaults)
#else
short cursor_left(escreen,kedit_defaults)
bool escreen,kedit_defaults;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern short prefix_width;
 extern short compatible;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
 short col=0,old_col=0;
 short left_column=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  cursor_left");
#endif
/*---------------------------------------------------------------------*/
/* The following should be a temporary fix for KEDIT compatability...  */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->prefix
 &&  kedit_defaults)
    escreen = FALSE;
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* If we are in the prefix area and the prefix is on the right and the */
/* current prefix width is not the maximum(PREFIX_WIDTH) then the last */
/* enterable column is column 1.                                       */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_PREFIX
 &&  (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
 &&  prefix_width != PREFIX_WIDTH)
    left_column = 1;
 else
    left_column = 0;
/*---------------------------------------------------------------------*/
/* For all windows, if we are not at left column, move 1 pos to left.  */
/*---------------------------------------------------------------------*/
 if (x > left_column)
   {
    wmove(CURRENT_WINDOW,y,x-1);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* For all windows, determine if CMDARROWSTABLRx is set for tabbing or */
/* scrolling and act accordingly.                                      */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
         if (!escreen)
           {
            if (CURRENT_VIEW->prefix)
               rc = Sos_prefix("");
            rc = Sos_lastcol("");
           }
         else
           {
#ifdef VERSHIFT
            if (CURRENT_VIEW->verify_col != 1)
              {
               rc = execute_move_cursor(CURRENT_VIEW->verify_col-2);
#else
/*           if (CURRENT_VIEW->verify_start != CURRENT_VIEW->verify_col)*/
            if (CURRENT_VIEW->verify_col != 1)
              {
               old_col = (CURRENT_VIEW->verify_col-1);
               x = min(CURRENT_SCREEN.cols[WINDOW_MAIN],CURRENT_VIEW->verify_end-CURRENT_VIEW->verify_start+1) / 2;
               col = max(1,old_col - (short)x);
               CURRENT_VIEW->verify_col = col;
               build_current_screen();
               display_current_screen();
               x = old_col - (CURRENT_VIEW->verify_col-1);
               wmove(CURRENT_WINDOW,y,x-1);
#endif

              }
            else
              {
               extern short compatible;
               if (compatible == COMPAT_KEDIT)
                 {
                  if (CURRENT_VIEW->prefix)
                     if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_LEFT)
                        rc = Sos_prefix("");
                  rc = Sos_lastcol("");
                 }
              }
           }
         break;
    case WINDOW_PREFIX:
    case WINDOW_COMMAND:
         if ((escreen
         &&  (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT)
         ||  !escreen)
             rc = Sos_rightedge("");
         break;
    default:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short cursor_right(bool escreen,bool kedit_defaults)
#else
short cursor_right(escreen,kedit_defaults)
bool escreen,kedit_defaults;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern short prefix_width;
 extern short compatible;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0,tempx=0;
 short right_column=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  cursor_right");
#endif
/*---------------------------------------------------------------------*/
/* The following should be a temporary fix for KEDIT compatability...  */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->prefix
 &&  kedit_defaults)
    escreen = FALSE;
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* Check for going past end of line - max_line_length                  */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->verify_col+x+1 > max_line_length)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If we are in the prefix area and the prefix is on the left and the  */
/* current prefix width is not the maximum(PREFIX_WIDTH) then the last */
/* enterable column is 1 column less.                                  */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_PREFIX
 &&  (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_LEFT
 &&  prefix_width != PREFIX_WIDTH)
    right_column = getmaxx(CURRENT_WINDOW)-2;
 else
    right_column = getmaxx(CURRENT_WINDOW)-1;
/*---------------------------------------------------------------------*/
/* For all windows, if we are not at right column, move 1 pos to right.*/
/*---------------------------------------------------------------------*/
 if (x < right_column)
   {
    wmove(CURRENT_WINDOW,y,x+1);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* For all windows, determine if CMDARROWSTABLRx is set for tabbing or */
/* scrolling and act accordingly.                                      */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
         if (!escreen)
           {
            if (CURRENT_VIEW->prefix)
               rc = Sos_prefix("");
            else
               wmove(CURRENT_WINDOW,y,0); /* this should move down a line too */
           }
         else
           {
#ifdef VERSHIFT
            rc = execute_move_cursor(right_column+1);
#else
            tempx = getmaxx(CURRENT_WINDOW);
            if (x == tempx-1)
              {
               x = (tempx / 2);
               CURRENT_VIEW->verify_col += x;
               build_current_screen();
               display_current_screen();
               wmove(CURRENT_WINDOW,y,x+(tempx % 2));
              }
#endif

           }
         break;
    case WINDOW_PREFIX:
    case WINDOW_COMMAND:
         if (!escreen)
            rc = Sos_leftedge("");
         else 
            if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_LEFT)
               Tabpre("");
         break;
    default:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short cursor_up(bool escreen)
#else
short cursor_up(escreen)
bool escreen;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE CMDARROWSTABCMDx;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 CHARTYPE *current_command=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  cursor_up");
#endif
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
    case WINDOW_PREFIX:
         rc = scroll_line(DIRECTION_BACKWARD,1L,FALSE,escreen);
         break;
    case WINDOW_COMMAND:
/*---------------------------------------------------------------------*/
/* Cycle backward through the command list or tab to last line.        */
/*---------------------------------------------------------------------*/
         if (CMDARROWSTABCMDx)
            rc = Sos_bottomedge("");
         else
           {
            current_command = get_next_command(DIRECTION_FORWARD);
            wmove(CURRENT_WINDOW_COMMAND,0,0);
            my_wclrtoeol(CURRENT_WINDOW_COMMAND);
            if (current_command != (CHARTYPE *)NULL)
               Cmsg(current_command);
           }
         break;
    default:
         display_error(2,(CHARTYPE *)"",FALSE);
         rc = RC_INVALID_OPERAND;
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short cursor_move(bool escreen,short row,short col)
#else
short cursor_move(escreen,row,col)
bool escreen;
short row,col;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 register int i=0,j=0;
 short rc=RC_OK;
 unsigned short x=0,y=0;
 unsigned short max_row=0,min_row=0,max_col=0;
 short idx=(-1);
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  cursor_move");
#endif
 getyx(CURRENT_WINDOW_MAIN,y,x);
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (!escreen)
   {
    if (find_last_focus_line(&max_row) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
    if (find_first_focus_line(&min_row) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
    if (row == 0)
       row = y;
    else
      {
       if (row > max_row)
          row = max_row;
       else
         {
          if (row < min_row)
             row = min_row;
          else
            {
             if (CURRENT_SCREEN.sl[row-1].main_enterable)
                row--;
             else
               {
                display_error(63,"",FALSE);
#ifdef TRACE
                trace_return();
#endif
                return(RC_TOF_EOF_REACHED);/* this is a strange RC :-( */
               }
            }
         }
      }
    max_col = CURRENT_SCREEN.cols[WINDOW_MAIN];
    if (col == 0)
       col = x;
    else
       if (col > max_col)
          col = max_col - 1;
       else
          col--;
    switch(CURRENT_VIEW->current_window)
      {
       case WINDOW_COMMAND:
            rc = cursor_home(FALSE);
            break;
       case WINDOW_PREFIX:
            CURRENT_VIEW->current_window = WINDOW_MAIN;
            break;
      }
    wmove(CURRENT_WINDOW_MAIN,row,col);
    CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[row].line_number;
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
   }
 else
/*---------------------------------------------------------------------*/
/* For CURSOR ESCREEN...                                               */
/*---------------------------------------------------------------------*/
   {
/*---------------------------------------------------------------------*/
/* Convert supplied row/col to 0 based offset...                       */
/*---------------------------------------------------------------------*/
    if (row == 0)
       row = CURRENT_SCREEN.start_row[CURRENT_VIEW->current_window] + y;
    else
       row--;
    if (col == 0)
       col = CURRENT_SCREEN.start_col[CURRENT_VIEW->current_window] + x;
    else
       col--;
    max_row = CURRENT_SCREEN.screen_rows - 1;
    max_col = CURRENT_SCREEN.screen_cols - 1;
/*---------------------------------------------------------------------*/
/* If row/col outside maximum screen size, exit...                     */
/*---------------------------------------------------------------------*/
    if (row > max_row
    ||  col > max_col)
      {
       display_error(63,"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_TOF_EOF_REACHED);/* this is a strange RC :-( */
      }
/*---------------------------------------------------------------------*/
/* Determine which window the cursor will end up in...                 */
/*---------------------------------------------------------------------*/
    for (i=0;i<VIEW_WINDOWS;i++)
      {
       if (row >= CURRENT_SCREEN.start_row[i]
       &&  row <= (CURRENT_SCREEN.start_row[i] + CURRENT_SCREEN.rows[i] - 1)
       &&  col >= CURRENT_SCREEN.start_col[i]
       &&  col <= (CURRENT_SCREEN.start_col[i] + CURRENT_SCREEN.cols[i] - 1))
         {
          idx = i;
          break;
         }
      }
    row = row - CURRENT_SCREEN.start_row[idx];
    col = col - CURRENT_SCREEN.start_col[idx];
    switch(idx)
      {
       case WINDOW_MAIN:
            row = get_row_for_tof_eof(row,current_screen);
            if (!CURRENT_SCREEN.sl[row].main_enterable)
              {
               display_error(63,"",FALSE);
#ifdef TRACE
               trace_return();
#endif
               return(RC_TOF_EOF_REACHED);/* this is a strange RC :-( */
              }
            rc = Sos_current("");
            wmove(CURRENT_WINDOW_MAIN,row,col);
            CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[row].line_number;
            pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
            break;
       case WINDOW_PREFIX:
            row = get_row_for_tof_eof(row,current_screen);
            if (!CURRENT_SCREEN.sl[row].prefix_enterable)
              {
               display_error(63,"",FALSE);
#ifdef TRACE
               trace_return();
#endif
               return(RC_TOF_EOF_REACHED);/* this is a strange RC :-( */
              }
            if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_LEFT
            &&  prefix_width != PREFIX_WIDTH
            &&  col == prefix_width)
              {
               display_error(63,"",FALSE);
#ifdef TRACE
               trace_return();
#endif
               return(RC_TOF_EOF_REACHED);/* this is a strange RC :-( */
              }
            if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
            &&  prefix_width != PREFIX_WIDTH
            &&  col == 0)
              {
               display_error(63,"",FALSE);
#ifdef TRACE
               trace_return();
#endif
               return(RC_TOF_EOF_REACHED);/* this is a strange RC :-( */
              }
            rc = Sos_current("");
            rc = Sos_prefix("");
            wmove(CURRENT_WINDOW_PREFIX,row,col);
            CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[row].line_number;
            pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
            break;
       case WINDOW_COMMAND:
            rc = cursor_cmdline(col+1);
            break;
       default:
            display_error(63,"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_TOF_EOF_REACHED);/* this is a strange RC :-( */
            break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
long where_now(void)
#else
long where_now()
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE display_screens;
/*--------------------------- local data ------------------------------*/
 long rc=0L;
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  where_now");
#endif
 getyx(CURRENT_WINDOW,y,x);
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
         rc |= WHERE_WINDOW_MAIN;
         break;
    case WINDOW_PREFIX:
         if ((CURRENT_VIEW->prefix & PREFIX_LOCATION_MASK) == PREFIX_LEFT)
            rc |= WHERE_WINDOW_PREFIX_LEFT;
         else
            rc |= WHERE_WINDOW_PREFIX_RIGHT;
         break;
    case WINDOW_COMMAND:
         if (CURRENT_VIEW->cmd_line == 'B')
            rc |= WHERE_WINDOW_CMDLINE_BOTTOM;
         else
            rc |= WHERE_WINDOW_CMDLINE_TOP;
         break;
   }
 if (display_screens == 1)
    rc |= WHERE_SCREEN_ONLY;
 else
   {
    if (current_screen == 0)
       rc |= WHERE_SCREEN_FIRST;
    else
       rc |= WHERE_SCREEN_LAST;
   }
 rc |= (long)y;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
long what_current_now(void)
#else
long what_current_now()
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 long rc=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  what_current_now");
#endif
 if (CURRENT_WINDOW_PREFIX != NULL)
   {
    if ((CURRENT_VIEW->prefix & PREFIX_LOCATION_MASK) == PREFIX_LEFT)
       rc |= WHERE_WINDOW_PREFIX_LEFT;
    else
       rc |= WHERE_WINDOW_PREFIX_RIGHT;
   }
 if (CURRENT_VIEW->cmd_line == 'B')
    rc |= WHERE_WINDOW_CMDLINE_BOTTOM;
 else
    if (CURRENT_VIEW->cmd_line == 'T')
       rc |= WHERE_WINDOW_CMDLINE_TOP;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
long what_other_now(void)
#else
long what_other_now()
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE display_screens;
/*--------------------------- local data ------------------------------*/
 long rc=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  what_other_now");
#endif
if (display_screens == 1)
  {
#ifdef TRACE
   trace_return();
#endif
   return(rc);
  }
 if (OTHER_SCREEN.win[WINDOW_PREFIX] != NULL)
   {
    if ((OTHER_SCREEN.screen_view->prefix & PREFIX_LOCATION_MASK) == PREFIX_LEFT)
       rc |= WHERE_WINDOW_PREFIX_LEFT;
    else
       rc |= WHERE_WINDOW_PREFIX_RIGHT;
   }
 if (OTHER_SCREEN.screen_view->cmd_line == 'B')
    rc |= WHERE_WINDOW_CMDLINE_BOTTOM;
 else
    if (OTHER_SCREEN.screen_view->cmd_line == 'T')
       rc |= WHERE_WINDOW_CMDLINE_TOP;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
long where_next(long where,long what_current,long what_other)
#else
long where_next(where,what_current,what_other)
long where,what_current,what_other;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 long where_row=0L,where_window=0L,where_screen=0L;
 long what_current_window=0L;
 long what_other_window=0L,what_other_screen=0L;
 long rc=0L;
 unsigned short current_top_row=0,current_bottom_row=0;
/*--------------------------- processing ------------------------------*/
 where_row = where & WHERE_ROW_MASK;
 where_window = where & WHERE_WINDOW_MASK;
 where_screen = where & WHERE_SCREEN_MASK;
 what_current_window = what_current & WHERE_WINDOW_MASK;
 what_other_window = what_other & WHERE_WINDOW_MASK;
 find_first_focus_line(&current_top_row);
 find_last_focus_line(&current_bottom_row);

 switch(where_window)
   {
    case WHERE_WINDOW_MAIN:
/*---------------------------------------------------------------------*/
/* In filearea.                                                        */
/*---------------------------------------------------------------------*/
         if (what_current_window & WHERE_WINDOW_PREFIX_RIGHT)
/*---------------------------------------------------------------------*/
/* In filearea and there is prefix on right.                           */
/* Result: same row,same screen,go to prefix.                          */
/*---------------------------------------------------------------------*/
            return(where_row | where_screen | WHERE_WINDOW_PREFIX_RIGHT);
         switch(where_screen)
           {
            case WHERE_SCREEN_FIRST:
            case WHERE_SCREEN_LAST:
                     /* the two cases above will be separate in future */
            case WHERE_SCREEN_ONLY:
/*---------------------------------------------------------------------*/
/* In filearea and only screen.                                        */
/*---------------------------------------------------------------------*/
                 if (where_row == (long)current_bottom_row)
                   {
/*---------------------------------------------------------------------*/
/* In filearea, prefix on left or off, on bottom line.                 */
/*---------------------------------------------------------------------*/
                    if (what_current_window & WHERE_WINDOW_CMDLINE_TOP)
/*---------------------------------------------------------------------*/
/* In filearea,prefix on left or off, on bottom line.                  */
/* Result: row irrelevant,same screen,go to cmdline.                   */
/*---------------------------------------------------------------------*/
                       return(where_row | where_screen | WHERE_WINDOW_CMDLINE_TOP);
                    if (what_current_window & WHERE_WINDOW_CMDLINE_BOTTOM)
/*---------------------------------------------------------------------*/
/* In filearea,prefix on left or off, on bottom line.                  */
/* Result: row irrelevant,same screen,go to cmdline.                   */
/*---------------------------------------------------------------------*/
                       return(where_row | where_screen | WHERE_WINDOW_CMDLINE_BOTTOM);
/*---------------------------------------------------------------------*/
/******************* To get here, there is no cmdline. *****************/
/*---------------------------------------------------------------------*/
                    if (what_current_window & WHERE_WINDOW_PREFIX_LEFT)
/*---------------------------------------------------------------------*/
/* In filearea,prefix on left or off, on bottom line.                  */
/* Result: first row,same screen,go to prefix.                         */
/*---------------------------------------------------------------------*/
                       return((long)current_top_row | where_screen | WHERE_WINDOW_PREFIX_LEFT);
                   }
/*---------------------------------------------------------------------*/
/******************* To get here, we are not on last row. **************/
/*---------------------------------------------------------------------*/
                 if (what_current_window & WHERE_WINDOW_PREFIX_LEFT)
/*---------------------------------------------------------------------*/
/* In filearea, prefix on left, not on bottom line.                    */
/* Result: next row,same screen,go to prefix.                          */
/*---------------------------------------------------------------------*/
                    return(where_row+1L | where_screen | WHERE_WINDOW_PREFIX_LEFT);
/*---------------------------------------------------------------------*/
/* In filearea, no prefix, not on bottom line.                         */
/* Result: next row,same screen,same window.                           */
/*---------------------------------------------------------------------*/
                 return(where_row+1L | where_screen | WHERE_WINDOW_MAIN);
                 break;
           }
         break;
    case WHERE_WINDOW_PREFIX_LEFT:
         rc = where_row | where_screen | WHERE_WINDOW_MAIN;
         break;
    case WHERE_WINDOW_PREFIX_RIGHT:
         switch(where_screen)
           {
            case WHERE_SCREEN_FIRST:
            case WHERE_SCREEN_LAST:
                     /* the two cases above will be separate in future */
            case WHERE_SCREEN_ONLY:
/*---------------------------------------------------------------------*/
/* In right prefix and only screen.                                    */
/*---------------------------------------------------------------------*/
                 if (where_row != (long)current_bottom_row)
/*---------------------------------------------------------------------*/
/* In right prefix and not on bottom line.                             */
/* Result: next row,same screen,go to filearea.                        */
/*---------------------------------------------------------------------*/
                    return(where_row+1L | where_screen | WHERE_WINDOW_MAIN);
                 if (what_current_window & WHERE_WINDOW_CMDLINE_BOTTOM)
/*---------------------------------------------------------------------*/
/* In right prefix, cmdline on bottom, on bottom line.                 */
/* Result: row irrelevant,same screen,go to cmdline.                   */
/*---------------------------------------------------------------------*/
                    return(where_row | where_screen | WHERE_WINDOW_CMDLINE_BOTTOM);
                 if (what_current_window & WHERE_WINDOW_CMDLINE_TOP)
/*---------------------------------------------------------------------*/
/* In right prefix, cmdline on top, on bottom line.                    */
/* Result: row irrelevant,same screen,go to cmdline.                   */
/*---------------------------------------------------------------------*/
                    return(where_row | where_screen | WHERE_WINDOW_CMDLINE_TOP);
/*---------------------------------------------------------------------*/
/* In right prefix, no cmdline, on bottom line.                        */
/* Result: first row,same screen,go to filearea.                       */
/*---------------------------------------------------------------------*/
                 return((long)current_top_row | where_screen | WHERE_WINDOW_MAIN);
                 break;
           }
         break;
    case WHERE_WINDOW_CMDLINE_TOP:
         switch(where_screen)
           {
            case WHERE_SCREEN_FIRST:
            case WHERE_SCREEN_LAST:
                     /* the two cases above will be separate in future */
            case WHERE_SCREEN_ONLY:
/*---------------------------------------------------------------------*/
/* In cmdline, and only screen.                                        */
/*---------------------------------------------------------------------*/
                 if (what_current_window & WHERE_WINDOW_PREFIX_LEFT)
/*---------------------------------------------------------------------*/
/* In cmdline, and only screen and prefix on left.                     */
/* Result: first row, same screen, go to prefix.                       */
/*---------------------------------------------------------------------*/
                    return((long)current_top_row | where_screen | WHERE_WINDOW_PREFIX_LEFT);
/*---------------------------------------------------------------------*/
/* In cmdline, and prefix on right or none.                            */
/* Result: first row, same screen, go to filearea.                     */
/*---------------------------------------------------------------------*/
                 return((long)current_top_row | where_screen | WHERE_WINDOW_MAIN);
                 break;
           }
         break;
    case WHERE_WINDOW_CMDLINE_BOTTOM:
         switch(where_screen)
           {
            case WHERE_SCREEN_FIRST:
            case WHERE_SCREEN_LAST:
                     /* the two cases above will be separate in future */
            case WHERE_SCREEN_ONLY:
/*---------------------------------------------------------------------*/
/* In cmdline, and only screen.                                        */
/*---------------------------------------------------------------------*/
                 if (what_current_window & WHERE_WINDOW_PREFIX_LEFT)
/*---------------------------------------------------------------------*/
/* In cmdline, and only screen and prefix on left.                     */
/* Result: first row, same screen, go to prefix.                       */
/*---------------------------------------------------------------------*/
                    return((long)current_top_row | where_screen | WHERE_WINDOW_PREFIX_LEFT);
/*---------------------------------------------------------------------*/
/* In cmdline, and prefix on right or none.                            */
/* Result: first row, same screen, go to filearea.                     */
/*---------------------------------------------------------------------*/
                 return((long)current_top_row | where_screen | WHERE_WINDOW_MAIN);
                 break;
           }
         break;
   }
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
long where_before(long where,long what_current,long what_other)
#else
long where_before(where,what_current,what_other)
long where,what_current,what_other;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 long where_row=0L,where_window=0L,where_screen=0L;
 long what_current_window=0L;
 long what_other_window=0L,what_other_screen=0L;
 long rc=0L;
 unsigned short current_top_row=0,current_bottom_row=0;
/*--------------------------- processing ------------------------------*/
 where_row = where & WHERE_ROW_MASK;
 where_window = where & WHERE_WINDOW_MASK;
 where_screen = where & WHERE_SCREEN_MASK;
 what_current_window = what_current & WHERE_WINDOW_MASK;
 what_other_window = what_other & WHERE_WINDOW_MASK;
 find_first_focus_line(&current_top_row);
 find_last_focus_line(&current_bottom_row);

 switch(where_window)
   {
    case WHERE_WINDOW_MAIN:
/*---------------------------------------------------------------------*/
/* In filearea.                                                        */
/*---------------------------------------------------------------------*/
         if (what_current_window & WHERE_WINDOW_PREFIX_LEFT)
/*---------------------------------------------------------------------*/
/* In filearea and there is prefix on left.                            */
/* Result: same row,same screen,go to prefix.                          */
/*---------------------------------------------------------------------*/
            return(where_row | where_screen | WHERE_WINDOW_PREFIX_LEFT);
         switch(where_screen)
           {
            case WHERE_SCREEN_FIRST:
            case WHERE_SCREEN_LAST:
                     /* the two cases above will be separate in future */
            case WHERE_SCREEN_ONLY:
/*---------------------------------------------------------------------*/
/* In filearea and only screen.                                        */
/*---------------------------------------------------------------------*/
                 if (where_row == (long)current_top_row)
                   {
/*---------------------------------------------------------------------*/
/* In filearea, prefix on right or off, on top    line.                */
/*---------------------------------------------------------------------*/
                    if (what_current_window & WHERE_WINDOW_CMDLINE_BOTTOM)
/*---------------------------------------------------------------------*/
/* In filearea,prefix on right or off, on top    line.                 */
/* Result: row irrelevant,same screen,go to cmdline.                   */
/*---------------------------------------------------------------------*/
                       return(where_row | where_screen | WHERE_WINDOW_CMDLINE_BOTTOM);
                    if (what_current_window & WHERE_WINDOW_CMDLINE_TOP)
/*---------------------------------------------------------------------*/
/* In filearea,prefix on right or off, on top    line.                 */
/* Result: row irrelevant,same screen,go to cmdline.                   */
/*---------------------------------------------------------------------*/
                       return(where_row | where_screen | WHERE_WINDOW_CMDLINE_TOP);
/*---------------------------------------------------------------------*/
/******************* To get here, there is no cmdline. *****************/
/*---------------------------------------------------------------------*/
                    if (what_current_window & WHERE_WINDOW_PREFIX_RIGHT)
/*---------------------------------------------------------------------*/
/* In filearea,prefix on right or off, on top    line.                 */
/* Result: last  row,same screen,go to prefix.                         */
/*---------------------------------------------------------------------*/
                       return((long)current_bottom_row | where_screen | WHERE_WINDOW_PREFIX_RIGHT);
                   }
/*---------------------------------------------------------------------*/
/******************* To get here, we are not on top  row. **************/
/*---------------------------------------------------------------------*/
                 if (what_current_window & WHERE_WINDOW_PREFIX_RIGHT)
/*---------------------------------------------------------------------*/
/* In filearea, prefix on right, not on top    line.                   */
/* Result: prior row,same screen,go to prefix.                         */
/*---------------------------------------------------------------------*/
                    return(where_row-1L | where_screen | WHERE_WINDOW_PREFIX_RIGHT);
/*---------------------------------------------------------------------*/
/* In filearea, no prefix, not on top    line.                         */
/* Result: prior row,same screen,same window.                          */
/*---------------------------------------------------------------------*/
                 return(where_row-1L | where_screen | WHERE_WINDOW_MAIN);
                 break;
           }
         break;
    case WHERE_WINDOW_PREFIX_RIGHT:
         rc = where_row | where_screen | WHERE_WINDOW_MAIN;
         break;
    case WHERE_WINDOW_PREFIX_LEFT:
         switch(where_screen)
           {
            case WHERE_SCREEN_FIRST:
            case WHERE_SCREEN_LAST:
                     /* the two cases above will be separate in future */
            case WHERE_SCREEN_ONLY:
/*---------------------------------------------------------------------*/
/* In left  prefix and only screen.                                    */
/*---------------------------------------------------------------------*/
                 if (where_row != (long)current_top_row)
/*---------------------------------------------------------------------*/
/* In left  prefix and not on top    line.                             */
/* Result: prior row,same screen,go to filearea.                       */
/*---------------------------------------------------------------------*/
                    return(where_row-1L | where_screen | WHERE_WINDOW_MAIN);
                 if (what_current_window & WHERE_WINDOW_CMDLINE_BOTTOM)
/*---------------------------------------------------------------------*/
/* In left  prefix, cmdline on bottom, on top    line.                 */
/* Result: row irrelevant,same screen,go to cmdline.                   */
/*---------------------------------------------------------------------*/
                    return(where_row | where_screen | WHERE_WINDOW_CMDLINE_BOTTOM);
                 if (what_current_window & WHERE_WINDOW_CMDLINE_TOP)
/*---------------------------------------------------------------------*/
/* In left  prefix, cmdline on top, on top    line.                    */
/* Result: row irrelevant,same screen,go to cmdline.                   */
/*---------------------------------------------------------------------*/
                    return(where_row | where_screen | WHERE_WINDOW_CMDLINE_TOP);
/*---------------------------------------------------------------------*/
/* In left  prefix, no cmdline, on top    line.                        */
/* Result: last  row,same screen,go to filearea.                       */
/*---------------------------------------------------------------------*/
                 return((long)current_bottom_row | where_screen | WHERE_WINDOW_MAIN);
                 break;
           }
         break;
    case WHERE_WINDOW_CMDLINE_TOP:
         switch(where_screen)
           {
            case WHERE_SCREEN_FIRST:
            case WHERE_SCREEN_LAST:
                     /* the two cases above will be separate in future */
            case WHERE_SCREEN_ONLY:
/*---------------------------------------------------------------------*/
/* In cmdline, and only screen.                                        */
/*---------------------------------------------------------------------*/
                 if (what_current_window & WHERE_WINDOW_PREFIX_RIGHT)
/*---------------------------------------------------------------------*/
/* In cmdline, and only screen and prefix on right.                    */
/* Result: last  row, same screen, go to prefix.                       */
/*---------------------------------------------------------------------*/
                    return((long)current_bottom_row | where_screen | WHERE_WINDOW_PREFIX_RIGHT);
/*---------------------------------------------------------------------*/
/* In cmdline, and prefix on left  or none.                            */
/* Result: last  row, same screen, go to filearea.                     */
/*---------------------------------------------------------------------*/
                 return((long)current_bottom_row | where_screen | WHERE_WINDOW_MAIN);
                 break;
           }
         break;
    case WHERE_WINDOW_CMDLINE_BOTTOM:
         switch(where_screen)
           {
            case WHERE_SCREEN_FIRST:
            case WHERE_SCREEN_LAST:
                     /* the two cases above will be separate in future */
            case WHERE_SCREEN_ONLY:
/*---------------------------------------------------------------------*/
/* In cmdline, and only screen.                                        */
/*---------------------------------------------------------------------*/
                 if (what_current_window & WHERE_WINDOW_PREFIX_RIGHT)
/*---------------------------------------------------------------------*/
/* In cmdline, and only screen and prefix on right.                    */
/* Result: last  row, same screen, go to prefix.                       */
/*---------------------------------------------------------------------*/
                    return((long)current_bottom_row | where_screen | WHERE_WINDOW_PREFIX_RIGHT);
/*---------------------------------------------------------------------*/
/* In cmdline, and prefix on left  or none.                            */
/* Result: last  row, same screen, go to filearea.                     */
/*---------------------------------------------------------------------*/
                 return((long)current_bottom_row | where_screen | WHERE_WINDOW_MAIN);
                 break;
           }
         break;
   }
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
bool enterable_field(long where)
#else
bool enterable_field(where)
long where;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 bool rc=TRUE;
 ROWTYPE row=0;
 long where_screen=0L;
 CHARTYPE scrn=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  enterable_field");
#endif
 where_screen = where & WHERE_SCREEN_MASK;
 row = (ROWTYPE)(where & WHERE_ROW_MASK);
 scrn = (where_screen == WHERE_SCREEN_LAST) ? 1 : 0;
 switch(where & WHERE_WINDOW_MASK)
   {
    case WHERE_WINDOW_MAIN:
         if (!screen[scrn].sl[row].main_enterable)
            rc = FALSE;
         break;
    case WHERE_WINDOW_PREFIX_LEFT:
    case WHERE_WINDOW_PREFIX_RIGHT:
         if (!screen[scrn].sl[row].prefix_enterable)
            rc = FALSE;
         break;
    case WHERE_WINDOW_CMDLINE_TOP:
    case WHERE_WINDOW_CMDLINE_BOTTOM:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short go_to_new_field(long save_where,long where)
#else
short go_to_new_field(save_where,where)
long save_where,where;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 long save_where_screen=0L,where_screen=0L;
 long save_where_window=0L,where_window=0L;
 unsigned short x=0;
 ROWTYPE where_row=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  go_to_new_field");
#endif
 save_where_screen = save_where & WHERE_SCREEN_MASK;
 where_screen = where & WHERE_SCREEN_MASK;
 save_where_window = save_where & WHERE_WINDOW_MASK;
 where_window = where & WHERE_WINDOW_MASK;
 where_row = (ROWTYPE)(where & WHERE_ROW_MASK);
 if (save_where_screen != where_screen)
   {
   }
 if (save_where_window == where_window)
/*---------------------------------------------------------------------*/
/* No change to screen or window...                                    */
/*---------------------------------------------------------------------*/
    CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[where_row].line_number;
 else
   {
    switch(save_where_window)
      {
       case WHERE_WINDOW_MAIN:
            switch(where_window)
              {
               case WHERE_WINDOW_PREFIX_LEFT:
               case WHERE_WINDOW_PREFIX_RIGHT:
                    CURRENT_VIEW->current_window = WINDOW_PREFIX;
                    CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[where_row].line_number;
                    break;
               case WHERE_WINDOW_CMDLINE_TOP:
               case WHERE_WINDOW_CMDLINE_BOTTOM:
                    CURRENT_VIEW->previous_window = CURRENT_VIEW->current_window;
                    CURRENT_VIEW->current_window = WINDOW_COMMAND;
                    where_row = 0;
                    break;
              }
            break;
       case WHERE_WINDOW_PREFIX_LEFT:
       case WHERE_WINDOW_PREFIX_RIGHT:
            switch(where_window)
              {
               case WHERE_WINDOW_MAIN:
                    CURRENT_VIEW->current_window = WINDOW_MAIN;
                    CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[where_row].line_number;
                    break;
               case WHERE_WINDOW_CMDLINE_TOP:
               case WHERE_WINDOW_CMDLINE_BOTTOM:
                    CURRENT_VIEW->previous_window = CURRENT_VIEW->current_window;
                    CURRENT_VIEW->current_window = WINDOW_COMMAND;
                    where_row = 0;
                    break;
              }
            break;
       case WHERE_WINDOW_CMDLINE_TOP:
       case WHERE_WINDOW_CMDLINE_BOTTOM:
            switch(where_window)
              {
               case WHERE_WINDOW_PREFIX_LEFT:
               case WHERE_WINDOW_PREFIX_RIGHT:
                    CURRENT_VIEW->current_window = WINDOW_PREFIX;
                    CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[where_row].line_number;
                    break;
               case WHERE_WINDOW_MAIN:
                    CURRENT_VIEW->current_window = WINDOW_MAIN;
                    CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[where_row].line_number;
                    break;
              }
            break;
      }
   }
 if (CURRENT_VIEW->current_window == WINDOW_PREFIX
 &&  (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
 &&  prefix_width != PREFIX_WIDTH)
    x = 1;
 else
    x = 0;
 wmove(CURRENT_WINDOW,where_row,x);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
void get_cursor_position(LINETYPE *screen_line,LINETYPE *screen_column,LINETYPE *file_line,LINETYPE *file_column)
#else
void get_cursor_position(screen_line,screen_column,file_line,file_column)
LINETYPE *screen_line,*screen_column,*file_line,*file_column;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
 unsigned short begy=0,begx=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  get_cursor_position");
#endif
 if (curses_started)
   {
    getyx(CURRENT_WINDOW,y,x);
    getbegyx(CURRENT_WINDOW,begy,begx);
    *screen_line = (LINETYPE)(y + begy + 1L);
    *screen_column = (LINETYPE)(x + begx + 1L);
   }
 else
    *screen_line = *screen_column = (-1L);
 if (CURRENT_VIEW->current_window != WINDOW_MAIN)
    *file_line = *file_column = (-1L);
 else
   {
    *file_line = CURRENT_VIEW->focus_line;
    *file_column = (LINETYPE)x + (LINETYPE)CURRENT_VIEW->verify_start
                               + (LINETYPE)CURRENT_VIEW->verify_col - 1L;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
short advance_focus_line(LINETYPE num_lines)
#else
short advance_focus_line(num_lines)
LINETYPE num_lines;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
 LINE *curr=NULL;
 LINETYPE actual_lines=num_lines;
 short direction=DIRECTION_FORWARD;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  advance_focus_line");
#endif
 if (num_lines < 0L)
   {
    actual_lines = -num_lines;
    direction = DIRECTION_BACKWARD;
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,CURRENT_VIEW->focus_line,CURRENT_FILE->number_lines);
 while(actual_lines>0)
    {
     if (direction == DIRECTION_BACKWARD)
        curr = curr->prev;
     else
        curr = curr->next;
     if (curr == NULL)
        break;
     CURRENT_VIEW->focus_line += (LINETYPE)direction;
     if (in_scope(CURRENT_VIEW,curr)
     || CURRENT_VIEW->scope_all)
        actual_lines--;
    }
 if (!line_in_view(CURRENT_VIEW->focus_line))
    CURRENT_VIEW->current_line = CURRENT_VIEW->focus_line;
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();
 if (curses_started)
   {
    getyx(CURRENT_WINDOW,y,x);
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_row);
    wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 if (CURRENT_TOF || CURRENT_BOF)
    return(RC_TOF_EOF_REACHED);
 else
    return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short advance_current_line(LINETYPE num_lines)
#else
short advance_current_line(num_lines)
LINETYPE num_lines;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 LINE *curr=NULL;
 LINETYPE actual_lines=num_lines;
 short direction=DIRECTION_FORWARD;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("cursor.c:  advance_current_line");
#endif
 if (num_lines < 0L)
   {
    actual_lines = -num_lines;
    direction = DIRECTION_BACKWARD;
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,CURRENT_VIEW->current_line,CURRENT_FILE->number_lines);
 while(actual_lines>0)
    {
     if (direction == DIRECTION_BACKWARD)
        curr = curr->prev;
     else
        curr = curr->next;
     if (curr == NULL)
        break;
     CURRENT_VIEW->current_line += (LINETYPE)direction;
     if (in_scope(CURRENT_VIEW,curr)
     || CURRENT_VIEW->scope_all)
        actual_lines--;
    }
 build_current_screen();
 if (!line_in_view(CURRENT_VIEW->focus_line))
    CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line;
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 if (CURRENT_TOF || CURRENT_BOF)
    return(RC_TOF_EOF_REACHED);
 else
    return(RC_OK);
}
