/***********************************************************************/
/* COMMSOS.C - sos commands.                                           */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * 36 David Road                 Phone:                    +617 3849 7731
 * Holland Park                  http://www.gu.edu.au/gwis/the/markh.html
 * Brisbane                      **** Maintainer PDCurses & REXX/SQL ****
 * QLD 4121                      ************* Author of THE ************
 * Australia                     ************* Member RexxLA ************
 */

/*
$Id: commsos.c 2.1 1995/06/24 16:29:14 MH Rel MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/

/*man-start*********************************************************************



========================================================================
SOS COMMAND REFERENCE
========================================================================
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     sos addline - add blank line after focus line

SYNTAX
     SOS ADDline

DESCRIPTION
     The SOS ADDLINE command inserts a blank line in the file following
     the focus line. The cursor is placed in the column under the first
     non-blank in the focus line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SOS LINEADD, SOS DELLINE

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_addline(CHARTYPE *params)
#else
short Sos_addline(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_addline");
#endif
 rc = Add("1");
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos bottomedge - move cursor to bottom edge of FILEAREA

SYNTAX
     SOS BOTTOMEdge

DESCRIPTION
     The SOS BOTTOMEDGE command moves the cursor to the last 
     enterable line in the FILEAREA or PREFIX area. If the cursor
     is on the command line, the cursor moves to the first 
     enterable line of the FILEAREA.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Comaptible.

SEE ALSO
     SOS TOPEDGE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_bottomedge(CHARTYPE *params)
#else
short Sos_bottomedge(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short y=0,x=0,row=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_bottomedge");
#endif
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* Get the last enterable row. If an error, stay where we are...       */
/*---------------------------------------------------------------------*/
 if (find_last_focus_line(&row) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* For each window determine what needs to be done...                  */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_COMMAND:
         if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) != PREFIX_LEFT)
            x += PREFIX_WIDTH;
         CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[row].line_number;
         pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         CURRENT_VIEW->current_window = WINDOW_MAIN;
         wmove(CURRENT_WINDOW,row,x);
         break;
    case WINDOW_MAIN:
    case WINDOW_PREFIX:
            if (row != y)                            /* different rows */
              {
               post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
               CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[row].line_number;
               pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
               wmove(CURRENT_WINDOW,row,x);
              }
            break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos current - move cursor to current line

SYNTAX
     SOS CURRent

DESCRIPTION
     The SOS CURRENT command moves the cursor to the current column
     of the cursor line from any window.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_current(CHARTYPE *params)
#else
short Sos_current(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short x=0,y=0;
 bool same_line=TRUE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_current");
#endif
 getyx(CURRENT_WINDOW_MAIN,y,x);
 switch (CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
         if (CURRENT_VIEW->focus_line != CURRENT_VIEW->current_line)
           {
            post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
            CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line;
            same_line = FALSE;
           }
         y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                                    CURRENT_VIEW->current_row);
         wmove(CURRENT_WINDOW_MAIN,y,x);
         if (!same_line)
            pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         break;
    case WINDOW_PREFIX:
    case WINDOW_COMMAND:
         CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line;
         y = get_row_for_focus_line(CURRENT_VIEW->focus_line,
                                    CURRENT_VIEW->current_row);
         CURRENT_VIEW->current_window = WINDOW_MAIN;
         wmove(CURRENT_WINDOW_MAIN,y,x);
         pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         break;
    default:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos cursoradj - move first non-blank character to cursor

SYNTAX
     SOS CURSORAdj

DESCRIPTION
     The SOS CURSORADJ command moves text in the focus line so that
     the first non-blank character appears under the cursor position.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_cursoradj(CHARTYPE *params)
#else
short Sos_cursoradj(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *rec;
/*--------------------------- local data ------------------------------*/
 short num_cols=0,first_non_blank_col=0,col=0,rc=RC_OK;
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_cursoradj");
#endif
 getyx(CURRENT_WINDOW,y,x);
 switch (CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
         if (FOCUS_TOF || FOCUS_BOF)
           {
            display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_INVALID_ENVIRON);
           }
         col = x + CURRENT_VIEW->verify_col - 1;
         first_non_blank_col = strzne(rec,' ');
         if (first_non_blank_col == (-1))
            first_non_blank_col = 0;
         num_cols = first_non_blank_col - col;
         if (num_cols < 0)
            rc = execute_shift_command(FALSE,-num_cols,CURRENT_VIEW->focus_line,1,FALSE);
         else
            if (num_cols > 0)
               rc = execute_shift_command(TRUE,num_cols,CURRENT_VIEW->focus_line,1,FALSE);
         break;
    default:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos delback - delete the character to the left of the cursor

SYNTAX
     SOS DELBAck

DESCRIPTION
     The SOS DELBACK command moves the cursor one character to the left
     and deletes the character now under the cursor.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     SOS DELCHAR

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_delback(CHARTYPE *params)
#else
short Sos_delback(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern CHARTYPE *pre_rec;
 extern unsigned short pre_rec_len;
 extern bool prefix_changed;
 extern bool readonly;
 extern bool CMDARROWSTABLRx;
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_delback");
#endif
 getyx(CURRENT_WINDOW,y,x);
 switch (CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
/*---------------------------------------------------------------------*/
/* If running in read-only mode and an attempt is made to execute this */
/* command in the MAIN window, then error...                           */
/*---------------------------------------------------------------------*/
         if (readonly)
           {
            display_error(56,"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_INVALID_ENVIRON);
           }
         if (CURRENT_SCREEN.sl[y].line_type != LINE_LINE)
           {
            display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_INVALID_ENVIRON);
           }
         break;
    case WINDOW_COMMAND:
         if (x == 0)
           {
#ifdef TRACE
            trace_return();
#endif
            return(RC_OK);
           }
         wmove(CURRENT_WINDOW,y,x-1);
         my_wdelch(CURRENT_WINDOW);
         if (x <= cmd_rec_len)
           {
            memdelchr(cmd_rec,x-1,cmd_rec_len,1);
            cmd_rec_len--;
           }
#ifdef TRACE
         trace_return();
#endif
         return(RC_OK);
         break;
    case WINDOW_PREFIX:
         if (x == 0)
           {
#ifdef TRACE
            trace_return();
#endif
            return(RC_OK);
           }
         prefix_changed = TRUE;
         wmove(CURRENT_WINDOW,y,x-1);
         my_wdelch(CURRENT_WINDOW);
         if (x <= pre_rec_len)
           {
            memdelchr(pre_rec,x-1,pre_rec_len,1);
            pre_rec_len --;
           }
#ifdef TRACE
            trace_return();
#endif
         return(RC_OK);
         break;
    default:
         break;
   }
/*---------------------------------------------------------------------*/
/* Remainder of processing is only for WINDOW_MAIN.                    */
/*---------------------------------------------------------------------*/
 if (x == 0 && CURRENT_VIEW->verify_start == CURRENT_VIEW->verify_col)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 cursor_left(TRUE,FALSE);
/*---------------------------------------------------------------------*/
/* If we are after the last character of the line, exit.               */
/*---------------------------------------------------------------------*/
 if (x+CURRENT_VIEW->verify_col-1 > rec_len)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }

 getyx(CURRENT_WINDOW,y,x);
 my_wdelch(CURRENT_WINDOW);

 memdelchr(rec,CURRENT_VIEW->verify_col-1+x,rec_len,1);
 rec_len--;
/*---------------------------------------------------------------------*/
/* If there is a character off the right edge of the screen, display it*/
/* in the last character of the main window.                           */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->verify_col-1+CURRENT_SCREEN.cols[WINDOW_MAIN]-1 < rec_len)
   {
    wmove(CURRENT_WINDOW,y,CURRENT_SCREEN.cols[WINDOW_MAIN]-1);
    put_char(CURRENT_WINDOW,rec[CURRENT_VIEW->verify_col-1+CURRENT_SCREEN.cols[WINDOW_MAIN]-1],ADDCHAR);
    wmove(CURRENT_WINDOW,y,x);
   }
/*---------------------------------------------------------------------*/
/* If the character being deleted is on a line which is in the marked  */
/* block, redisplay the window.                                        */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW == MARK_VIEW)
   {
    if (CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line
    &&  CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
      {
       build_current_screen();
       display_current_screen();
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     sos delchar - delete character under cursor

SYNTAX
     SOS DELChar

DESCRIPTION
     The SOS DELCHAR command deletes the character under the cursor.
     Text to the right is shifted to the left.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     SOS DELBACK

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_delchar(CHARTYPE *params)
#else
short Sos_delchar(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern CHARTYPE *pre_rec;
 extern unsigned short pre_rec_len;
 extern bool prefix_changed;
 extern bool readonly;
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_delchar");
#endif
/*---------------------------------------------------------------------*/
/* If running in read-only mode and an attempt is made to execute this */
/* command in the MAIN window, then error...                           */
/*---------------------------------------------------------------------*/
 if (readonly && CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    display_error(56,"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
 getyx(CURRENT_WINDOW,y,x);
 switch (CURRENT_VIEW->current_window)
   {
    case WINDOW_COMMAND:
         my_wdelch(CURRENT_WINDOW);
         if (x < cmd_rec_len)
           {
            memdelchr(cmd_rec,x,cmd_rec_len,1);
            cmd_rec_len--;
           }
#ifdef TRACE
         trace_return();
#endif
         return(RC_OK);
         break;
    case WINDOW_PREFIX:
         my_wdelch(CURRENT_WINDOW);
         if (x < pre_rec_len)
           {
            prefix_changed = TRUE;
            memdelchr(pre_rec,x,pre_rec_len,1);
            pre_rec_len--;
           }
#ifdef TRACE
         trace_return();
#endif
         return(RC_OK);
         break;
    case WINDOW_MAIN:
/*---------------------------------------------------------------------*/
/* Do not allow this command on the top or bottom of file lines or on  */
/* shadow lines.                                                       */
/*---------------------------------------------------------------------*/
         if (CURRENT_SCREEN.sl[y].line_type != LINE_LINE)
           {
            display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_INVALID_ENVIRON);
           }
         my_wdelch(CURRENT_WINDOW);
         break;
    default:
         break;
   }
/*---------------------------------------------------------------------*/
/* If we are not after the last character of the line...               */
/*---------------------------------------------------------------------*/
 if (x+CURRENT_VIEW->verify_col <= rec_len)
   {
    memdelchr(rec,CURRENT_VIEW->verify_col-1+x,rec_len,1);
    rec_len--;
/*---------------------------------------------------------------------*/
/* If there is a character off the right edge of the screen, display it*/
/* in the last character of the main window.                           */
/*---------------------------------------------------------------------*/
    if (CURRENT_VIEW->verify_col-1+CURRENT_SCREEN.cols[WINDOW_MAIN]-1 < rec_len)
      {
       wmove(CURRENT_WINDOW,y,CURRENT_SCREEN.cols[WINDOW_MAIN]-1);
       put_char(CURRENT_WINDOW,rec[CURRENT_VIEW->verify_col-1+CURRENT_SCREEN.cols[WINDOW_MAIN]-1],ADDCHAR);
       wmove(CURRENT_WINDOW,y,x);
      }
   }
/*---------------------------------------------------------------------*/
/* If the character being deleted is on a line which is in the marked  */
/* block, redisplay the window.                                        */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW == MARK_VIEW)
   {
    if (CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line
    &&  CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
      {
       build_current_screen();
       display_current_screen();
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     sos delend - delete to end of line

SYNTAX
     SOS DELEnd

DESCRIPTION
     The SOS DELEND command deletes all characters from the current
     column to the end of line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_delend(CHARTYPE *params)
#else
short Sos_delend(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern CHARTYPE *pre_rec;
 extern unsigned short pre_rec_len;
 extern bool prefix_changed;
 extern bool readonly;
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 unsigned short col=0,x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_delend");
#endif
 getyx(CURRENT_WINDOW,y,x);
 switch (CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
/*---------------------------------------------------------------------*/
/* If running in read-only mode and an attempt is made to execute this */
/* command in the MAIN window, then error...                           */
/*---------------------------------------------------------------------*/
         if (readonly)
           {
            display_error(56,"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_INVALID_ENVIRON);
           }
         if (CURRENT_SCREEN.sl[y].line_type != LINE_LINE)
           {
            display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_INVALID_ENVIRON);
           }
         col = x + CURRENT_VIEW->verify_col - 1;
         for (i=col;i<max_line_length;i++)
             rec[i] = ' ';
         if (rec_len > col)
            rec_len = col;
         my_wclrtoeol(CURRENT_WINDOW);
         break;
    case WINDOW_COMMAND:
         for (i=x;i<COLS;i++)
             cmd_rec[i] = ' ';
         if (cmd_rec_len > x)
            cmd_rec_len = x;
         my_wclrtoeol(CURRENT_WINDOW);
         break;
    case WINDOW_PREFIX:
         prefix_changed = TRUE;
         for (i=x;i<PREFIX_WIDTH;i++)
             pre_rec[i] = ' ';
         if (pre_rec_len > x)
            pre_rec_len = x;
         my_wclrtoeol(CURRENT_WINDOW);
         break;
    default:
         break;
   }
/*---------------------------------------------------------------------*/
/* If the character being deleted is on a line which is in the marked  */
/* block, and we are in the filearea, redisplay the screen.            */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW == MARK_VIEW
 &&  CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    if (CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line
    &&  CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
      {
       build_current_screen();
       display_current_screen();
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     sos delline - delete focus line

SYNTAX
     SOS DELLine

DESCRIPTION
     The SOS DELLINE command deletes the focus line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SOS LINEDEL, SOS ADDLINE

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_delline(CHARTYPE *params)
#else
short Sos_delline(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short x=0,y=0;
 LINETYPE true_line=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_delline");
#endif
 if (CURRENT_VIEW->current_window == WINDOW_MAIN
 ||  CURRENT_VIEW->current_window == WINDOW_PREFIX)
   {
    getyx(CURRENT_WINDOW,y,x);
    if (CURRENT_SCREEN.sl[y].line_type != LINE_LINE
    &&  !CURRENT_VIEW->scope_all)
      {
       display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_ENVIRON);
      }
   }
 true_line = CURRENT_VIEW->focus_line;
 rc = rearrange_line_blocks(COMMAND_DELETE,SOURCE_COMMAND,true_line,
                            true_line,true_line,1,CURRENT_VIEW,CURRENT_VIEW,FALSE);
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    if (curses_started)
       getyx(CURRENT_WINDOW,y,x);
    CURRENT_VIEW->focus_line = get_focus_line_in_view(CURRENT_VIEW->focus_line,y);
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,CURRENT_VIEW->current_row);
    if (curses_started)
       wmove(CURRENT_WINDOW,y,x);
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
   }
/* rc = DeleteLine("1");*/
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos delword - delete word at or right of cursor

SYNTAX
     SOS DELWord

DESCRIPTION
     The SOS DELWORD command deletes the word at or to the right
     of the current cursor position and any spaces following the 
     word.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_delword(CHARTYPE *params)
#else
short Sos_delword(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern bool readonly;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short rc=RC_OK;
 LENGTHTYPE first_col=0,last_col=0;
 unsigned short x=0,y=0,temp_rec_len=0;
 short num_cols=0,left_col=0,col_pos=0;
 CHARTYPE *temp_rec=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_delword");
#endif
/*---------------------------------------------------------------------*/
/* This function is not applicable to the PREFIX window.               */
/*---------------------------------------------------------------------*/
 getyx(CURRENT_WINDOW,y,x);
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_PREFIX:
         display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
         trace_return();
#endif
         return(RC_INVALID_ENVIRON);
         break;
    case WINDOW_MAIN:
/*---------------------------------------------------------------------*/
/* If running in read-only mode and an attempt is made to execute this */
/* command in the MAIN window, then error...                           */
/*---------------------------------------------------------------------*/
         if (readonly)
           {
            display_error(56,"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_INVALID_ENVIRON);
           }
         if (CURRENT_SCREEN.sl[y].line_type != LINE_LINE)
           {
            display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(RC_INVALID_ENVIRON);
           }
         temp_rec = rec;
         temp_rec_len = rec_len;
         left_col = CURRENT_VIEW->verify_col-1;
         break;
    case WINDOW_COMMAND:
         temp_rec = (CHARTYPE *)cmd_rec;
         temp_rec_len = cmd_rec_len;
         left_col = 0;
         break;
   }
 if (get_word(temp_rec,temp_rec_len,x+left_col,&first_col,&last_col) == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return(0);
   }
/*---------------------------------------------------------------------*/
/* Delete from the field the number of columns calculated above        */
/* and adjust the appropriate record length.                           */
/*---------------------------------------------------------------------*/
 num_cols = last_col-first_col+1;
 memdelchr(temp_rec,first_col,temp_rec_len,num_cols);
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
         rec_len -= num_cols;
         rc = execute_move_cursor(first_col);
         build_current_screen(); 
         display_current_screen();
#if 0
         col_pos = x;
         if (first_col >= left_col)
           {
            build_current_screen(); 
            display_current_screen();
            wmove(CURRENT_WINDOW,y,first_col);
           }
         else
           {
            x = CURRENT_SCREEN.cols[WINDOW_MAIN] / 2;
            CURRENT_VIEW->verify_col = max(1,first_col - x + 2);
            x = (CURRENT_VIEW->verify_col == 1) ? first_col : x - 1;
            wmove(CURRENT_WINDOW,y,x);
            build_current_screen(); 
            display_current_screen();
           }
#endif
         break;
    case WINDOW_COMMAND:
         cmd_rec_len -= num_cols;
         wmove(CURRENT_WINDOW,y,first_col);
         for (i=0;i<num_cols;i++)
            my_wdelch(CURRENT_WINDOW);
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos doprefix - execute any pending prefix commands

SYNTAX
     SOS DOPREfix

DESCRIPTION
     The SOS DOPREFIX command executes any pending prefix commands.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_doprefix(CHARTYPE *params)
#else
short Sos_doprefix(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_doprefix");
#endif
/*---------------------------------------------------------------------*/
/*                                                                     */
/*---------------------------------------------------------------------*/
/* post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);*/
 rc = execute_prefix_commands();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos edit - edit a file from directory list

SYNTAX
     SOS EDIT

DESCRIPTION
     The SOS EDIT command allows the user to edit a file, chosen from
     a directory list.(the file DIR.DIR).

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible with default definition for Alt-X key.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_edit(CHARTYPE *params)
#else
short Sos_edit(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE sp_path[MAX_FILE_NAME+1];
 extern CHARTYPE sp_fname[MAX_FILE_NAME+1];
 extern CHARTYPE dir_path[MAX_FILE_NAME+1];
 extern unsigned short file_start;
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
 LINE *curr=NULL;
 CHARTYPE edit_fname[MAX_FILE_NAME];
 unsigned short y=0,x=0;
 short rc=RC_OK;
 LINETYPE true_line=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_edit");
#endif
/*---------------------------------------------------------------------*/
/* If the current file is not the special DIR.DIR file exit.           */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->pseudo_file != PSEUDO_DIR)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* Determine which line contains a vaild file to edit. TOF and EOF are */
/* invalid positions.                                                  */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
   {
    if (CURRENT_TOF || CURRENT_BOF)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_ENVIRON);
      }
    true_line = CURRENT_VIEW->current_line;
   }
 else
   {
    getyx(CURRENT_WINDOW,y,x);
    if (FOCUS_TOF || FOCUS_BOF)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_ENVIRON);
      }
    true_line = CURRENT_VIEW->focus_line;
   }
/*---------------------------------------------------------------------*/
/* Find the current LINE pointer for the focus_line.                   */
/*---------------------------------------------------------------------*/
 curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,true_line,CURRENT_FILE->number_lines);
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Validate that the supplied file is valid.                           */
/*---------------------------------------------------------------------*/
 strcpy(edit_fname,dir_path);
 strcat(edit_fname,(CHARTYPE *)curr->line+file_start);
 if ((rc = splitpath(edit_fname)) != RC_OK)
   {
    display_error(10,edit_fname,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Edit the file.                                                      */
/*---------------------------------------------------------------------*/
 strcpy(edit_fname,sp_path);
 strcat(edit_fname,sp_fname);
 rc = Xedit(edit_fname);
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos endchar - move cursor to end of focus line

SYNTAX
     SOS ENDChar

DESCRIPTION
     The SOS ENDCHAR command moves the cursor to the position after
     the last character displayed in the current window.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     SOS STARTENDCHAR

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_endchar(CHARTYPE *params)
#else
short Sos_endchar(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern LENGTHTYPE rec_len;
 extern LENGTHTYPE cmd_rec_len;
 short rc=RC_OK;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_endchar");
#endif
 getyx(CURRENT_WINDOW,y,x);
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_PREFIX:
         rc = RC_INVALID_ENVIRON;
         break;
    case WINDOW_COMMAND:
         wmove(CURRENT_WINDOW,y,cmd_rec_len);
         rc = RC_OK;
         break;
    case WINDOW_MAIN:
         rc = execute_move_cursor(rec_len);
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos execute - move cursor to command line and execute command

SYNTAX
     SOS EXecute

DESCRIPTION
     The SOS EXECUTE command moves the cursor to the command line
     and executes any command that is on the command line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible

STATUS
     Complete. 
**man-end**********************************************************************/
#ifdef PROTO
short Sos_execute(CHARTYPE *params)
#else
short Sos_execute(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_execute");
#endif

 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
    rc = cursor_cmdline(1);
 if (rc == RC_OK)
   {
    for (i=0;i<cmd_rec_len;i++)
       temp_cmd[i] = cmd_rec[i];
    temp_cmd[cmd_rec_len] = '\0';
    strip(temp_cmd,'L',' ');
    add_command(temp_cmd);
    rc = command_line(temp_cmd,COMMAND_ONLY_FALSE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos firstchar - move cursor to first non-blank of field

SYNTAX
     SOS FIRSTCHar

DESCRIPTION
     The SOS FIRSTCHAR command moves the cursor to the first
     non-blank character of the cursor field

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible

SEE ALSO
     SOS FIRSTCOL

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_firstchar(CHARTYPE *params)
#else
short Sos_firstchar(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK,new_col=0;
 unsigned short y=0,x=0;
 LINE *curr=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_firstchar");
#endif
/*---------------------------------------------------------------------*/
/* For the command line and prefix area, just go to the first column...*/
/*---------------------------------------------------------------------*/
 getyx(CURRENT_WINDOW,y,x);
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND
 ||  CURRENT_VIEW->current_window == WINDOW_PREFIX)
   {
    wmove(CURRENT_WINDOW,y,0);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* For the filearea, we have to do a bit more...                       */
/*---------------------------------------------------------------------*/
 curr = CURRENT_SCREEN.sl[y].current;
 new_col = memne(curr->line,' ',curr->length);
 if (new_col == (-1))
    new_col = 0;
 rc = execute_move_cursor(new_col);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos firstcol - move cursor to first column of field

SYNTAX
     SOS FIRSTCOl

DESCRIPTION
     The SOS FIRSTCOL command moves the cursor to the first
     column of the cursor field

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible

SEE ALSO
     SOS FIRSTCHAR, SOS LASTCOL

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_firstcol(CHARTYPE *params)
#else
short Sos_firstcol(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_firstcol");
#endif
/*---------------------------------------------------------------------*/
/* For the command line and prefix area, just go to the first column...*/
/*---------------------------------------------------------------------*/
 getyx(CURRENT_WINDOW,y,x);
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND
 ||  CURRENT_VIEW->current_window == WINDOW_PREFIX)
   {
    wmove(CURRENT_WINDOW,y,0);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* For the filearea, we have to do a bit more...                       */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->verify_col != 1)
   {
    rc = execute_move_cursor(0);
#if 0
    CURRENT_VIEW->verify_col = 1;
    build_current_screen();
    display_current_screen();
#endif
   }
 wmove(CURRENT_WINDOW,y,0);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos lastcol - move cursor to last column of field

SYNTAX
     SOS LASTCOl

DESCRIPTION
     The SOS LASTCOL command moves the cursor to the last column
     of the cursor field.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     SOS FIRSTCOL

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_lastcol(CHARTYPE *params)
#else
short Sos_lastcol(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_lastcol");
#endif
/*---------------------------------------------------------------------*/
/* For the command line and filearea, just go to the last column...    */
/*---------------------------------------------------------------------*/
 getyx(CURRENT_WINDOW,y,x);
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND
 ||  CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    x = getmaxx(CURRENT_WINDOW)-1;
    wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* For the prefix area we have to do a bit more...                     */
/*---------------------------------------------------------------------*/
 if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
 &&  prefix_width != PREFIX_WIDTH)
     x = PREFIX_WIDTH - 1;
 else
     x = prefix_width - 1;
 wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos leftedge - move cursor to left edge of window

SYNTAX
     SOS LEFTEdge

DESCRIPTION
     The SOS LEFTEDGE command moves the cursor to the leftmost edge
     of the filearea if not on the command line or to the leftmost
     edge of the command line if on the command line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible

SEE ALSO
     SOS RIGHTEDGE, SOS PREFIX

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_leftedge(CHARTYPE *params)
#else
short Sos_leftedge(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_leftedge");
#endif
 getyx(CURRENT_WINDOW,y,x);
 if (CURRENT_VIEW->current_window == WINDOW_PREFIX)
    CURRENT_VIEW->current_window = WINDOW_MAIN;
 wmove(CURRENT_WINDOW,y,0);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     sos lineadd - add blank line after focus line

SYNTAX
     SOS LINEAdd

DESCRIPTION
     The SOS LINEADD command inserts a blank line in the file following
     the focus line. The cursor is placed in the column under the first
     non-blank in the focus line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SOS ADDLINE, SOS LINEDEL

STATUS
     Complete
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     sos linedel - delete focus line

SYNTAX
     SOS LINEDel

DESCRIPTION
     The SOS LINEDEL command deletes the focus line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SOS DELLINE, SOS LINEADD

STATUS
     Complete
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     sos makecurr - make focus line the current line

SYNTAX
     SOS MAKECURR

DESCRIPTION
     The SOS MAKECURR command set the current line to the current focus
     line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_makecurr(CHARTYPE *params)
#else
short Sos_makecurr(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_makecurr");
#endif
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
    rc = execute_makecurr(CURRENT_VIEW->focus_line);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos marginl - move cursor to the left margin column

SYNTAX
     SOS MARGINL

DESCRIPTION
     The SOS MARGINL command moves the cursor to the left margin
     column.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.
            Although, when issued from the command line, nothing
            happens.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_marginl(CHARTYPE *params)
#else
short Sos_marginl(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_marginl");
#endif
 if (Sos_leftedge("") == RC_OK)
    rc = execute_move_cursor(CURRENT_VIEW->margin_left-1);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos marginr - move cursor to the right margin column

SYNTAX
     SOS MARGINR

DESCRIPTION
     The SOS MARGINR command moves the cursor to the right margin
     column.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.
            Although, when issued from the command line, nothing
            happens.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_marginr(CHARTYPE *params)
#else
short Sos_marginr(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_marginr");
#endif
 if (Sos_leftedge("") == RC_OK)
    rc = execute_move_cursor(CURRENT_VIEW->margin_right-1);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos parindent - move cursor to the paragraph indent column

SYNTAX
     SOS PARINDent

DESCRIPTION
     The SOS PARINDENT command moves the cursor to the paragraph
     indent column.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.
            Although, when issued from the command line, nothing
            happens.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Sos_parindent(CHARTYPE *params)
#else
short Sos_parindent(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 COLTYPE parindent=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_parindent");
#endif
 if (CURRENT_VIEW->margin_indent_offset)
    parindent = CURRENT_VIEW->margin_left + CURRENT_VIEW->margin_indent - 1;
 else
    parindent = CURRENT_VIEW->margin_indent - 1;
 if (Sos_leftedge("") == RC_OK)
    rc = execute_move_cursor(parindent);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos prefix - move cursor to leftmost edge of prefix area

SYNTAX
     SOS PREfix

DESCRIPTION
     The SOS PREFIX command moves the cursor to the rightmost edge
     of the prefix area.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible

SEE ALSO
     SOS LEFTEDGE, SOS RIGHTEDGE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_prefix(CHARTYPE *params)
#else
short Sos_prefix(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_prefix");
#endif
/*---------------------------------------------------------------------*/
/* If the cursor is in the command line or there is no prefix on, exit.*/
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND
 ||  !CURRENT_VIEW->prefix)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 getyx(CURRENT_WINDOW,y,x);
 if (CURRENT_VIEW->current_window == WINDOW_MAIN)
    CURRENT_VIEW->current_window = WINDOW_PREFIX;
 if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
 &&  prefix_width != PREFIX_WIDTH)
    x = 1;
 else
    x = 0;
 wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos qcmnd - move cursor to command line and clear

SYNTAX
     SOS QCmnd

DESCRIPTION
     The SOS QCMND command moves the cursor to the first column of
     the command line and clears the command line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible

SEE ALSO
     SOS EXECUTE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_qcmnd(CHARTYPE *params)
#else
short Sos_qcmnd(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_qcmnd");
#endif
 if ((rc = cursor_cmdline(1)) == RC_OK)
   {
    if (CURRENT_WINDOW_COMMAND != (WINDOW *)NULL)
      {
       wmove(CURRENT_WINDOW_COMMAND,0,0);
       my_wclrtoeol(CURRENT_WINDOW_COMMAND);
      }
    memset(cmd_rec,' ',max_line_length);
    cmd_rec_len = 0;
  }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos rightedge - move cursor to right edge of window

SYNTAX
     SOS RIGHTEdge

DESCRIPTION
     The SOS RIGHTEDGE command moves the cursor to the rightmost edge
     of the filearea if not on the command line or to the rightmost
     edge of the command line if on the command line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible

SEE ALSO
     SOS LEFTEDGE, SOS PREFIX

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_rightedge(CHARTYPE *params)
#else
short Sos_rightedge(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_rightedge");
#endif
 getyx(CURRENT_WINDOW,y,x);
 if (CURRENT_VIEW->current_window == WINDOW_PREFIX)
    CURRENT_VIEW->current_window = WINDOW_MAIN;
 x = getmaxx(CURRENT_WINDOW)-1;
 wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos startendchar - move cursor to end/start of focus line

SYNTAX
     SOS STARTENDChar

DESCRIPTION
     The SOS STARTENDCHAR command moves the cursor to the first character
     displayed in the current window, if the cursor is after the last
     character displayed in the current window, or to the position after
     the last character displayed in the current window, if the cursor is
     anywhere else.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     SOS ENDCHAR

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_startendchar(CHARTYPE *params)
#else
short Sos_startendchar(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern LENGTHTYPE rec_len;
 extern unsigned short cmd_rec_len;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0,line_col=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_startendchar");
#endif
 getyx(CURRENT_WINDOW,y,x);
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_PREFIX:
         rc = RC_INVALID_ENVIRON;
         break;
    case WINDOW_COMMAND:
         if (x >= cmd_rec_len)
            wmove(CURRENT_WINDOW,y,0);
         else
            wmove(CURRENT_WINDOW,y,cmd_rec_len);
         break;
    case WINDOW_MAIN:
         if (x + CURRENT_VIEW->verify_col > min(rec_len,CURRENT_VIEW->verify_end))
            rc = Sos_firstcol("");
         else
            rc = Sos_endchar("");
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos tabb - move cursor to previous tab stop

SYNTAX
     SOS TABB

DESCRIPTION
     The SOS TABB command causes the cursor to move to the previous tab
     column as set by the SET TABS command.
     If the resulting column is beyond the left hand edge of the main
     window, the window will scroll half a window.

COMPATIBILITY
     XEDIT: Does not allow arguments.
     KEDIT: Compatible. See below.
     Does not line tab to next line if before the left hand tab column.

SEE ALSO
     SET TABS, SOS TABF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_tabb(CHARTYPE *params)
#else
short Sos_tabb(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
 LENGTHTYPE prev_tab_col=0,current_col=0;
 COLTYPE new_screen_col=0;
 LENGTHTYPE new_verify_col=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_tabb");
#endif
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* Determine action depending on current window...                     */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_PREFIX:
#ifdef TRACE
         trace_return();
#endif
         return(RC_OK);
         break;
    case WINDOW_MAIN:
         current_col = x+CURRENT_VIEW->verify_col;
         break;
    case WINDOW_COMMAND:
         current_col = x+1;
         break;
   }
/*---------------------------------------------------------------------*/
/* First determine the next tab stop column...                         */
/*---------------------------------------------------------------------*/
 prev_tab_col = find_prev_tab_col(current_col);
/*---------------------------------------------------------------------*/
/* If no prev tab column, stay where we are and return...              */
/*---------------------------------------------------------------------*/
 if (prev_tab_col == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* For all windows, if the new cursor position does not exceed the     */
/* right edge, move there.                                             */
/*---------------------------------------------------------------------*/
 prev_tab_col--;                               /* zero base the column */

#ifdef VERSHIFT
 rc = execute_move_cursor(prev_tab_col);
#else
 calculate_new_column(x,CURRENT_VIEW->verify_col,prev_tab_col,&new_screen_col,&new_verify_col);
 if (CURRENT_VIEW->verify_col != new_verify_col
 &&  CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    CURRENT_VIEW->verify_col = new_verify_col;
    build_current_screen(); 
    display_current_screen();
   }
 wmove(CURRENT_WINDOW,y,new_screen_col);
#endif

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos tabf - move cursor to next tab stop

SYNTAX
     SOS TABf

DESCRIPTION
     The SOS TABF command causes the cursor to move to the next tab column
     as set by the SET TABS command.
     If the resulting column is beyond the right hand edge of the main
     window, the window will scroll half a window.

COMPATIBILITY
     XEDIT: Does not allow arguments.
     KEDIT: Compatible. See below.
     Does not line tab to next line if after the right hand tab column.

SEE ALSO
     SET TABS, SOS TABB

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_tabf(CHARTYPE *params)
#else
short Sos_tabf(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool INSERTMODEx;
 extern CHARTYPE tabkey_insert,tabkey_overwrite;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
 LENGTHTYPE next_tab_col=0,current_col=0;
 COLTYPE new_screen_col=0;
 LENGTHTYPE new_verify_col=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_tabf");
#endif
/*---------------------------------------------------------------------*/
/* If the actual tab character is to be display then exit so that      */
/* editor() can process it as a raw key.                               */
/*---------------------------------------------------------------------*/
 if (INSERTMODEx && tabkey_insert == 'C')
   {
#ifdef TRACE
    trace_return();
#endif
    return(RAW_KEY);
   }
 if (!INSERTMODEx && tabkey_overwrite == 'C')
   {
#ifdef TRACE
    trace_return();
#endif
    return(RAW_KEY);
   }
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* Determine action depending on current window...                     */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_PREFIX:
#ifdef TRACE
         trace_return();
#endif
         return(RC_OK);
         break;
    case WINDOW_MAIN:
         current_col = x+CURRENT_VIEW->verify_col;
         break;
    case WINDOW_COMMAND:
         current_col = x+1;
         break;
   }
/*---------------------------------------------------------------------*/
/* First determine the next tab stop column...                         */
/*---------------------------------------------------------------------*/
 next_tab_col = find_next_tab_col(current_col);
/*---------------------------------------------------------------------*/
/* If no next tab column, stay where we are and return...              */
/*---------------------------------------------------------------------*/
 if (next_tab_col == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Check for going past end of line - max_line_length                  */
/*---------------------------------------------------------------------*/
 if (next_tab_col > max_line_length)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_TRUNCATED);
   }
/*---------------------------------------------------------------------*/
/* For all windows, if the new cursor position does not exceed the     */
/* right edge, move there.                                             */
/*---------------------------------------------------------------------*/
 next_tab_col--;                               /* zero base the column */

#ifdef VERSHIFT
 rc = execute_move_cursor(next_tab_col);
#else
 calculate_new_column(x,CURRENT_VIEW->verify_col,next_tab_col,&new_screen_col,&new_verify_col);
 if (CURRENT_VIEW->verify_col != new_verify_col
 &&  CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    CURRENT_VIEW->verify_col = new_verify_col;
    build_current_screen(); 
    display_current_screen();
   }
 wmove(CURRENT_WINDOW,y,new_screen_col);
#endif

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos tabfieldb - move cursor to previous enterable field

SYNTAX
     SOS TABFIELDB

DESCRIPTION
     The SOS TABFIELDB command causes the cursor to move to the first
     column of the current enterable field. If the cursor is already
     in the first column of the current field the cursor moves to the
     first column of the previous enterable field on the screen. 
     This command is intended to mimic the behavior of the SHIFT-TAB 
     key on a 3270 terminal.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     SOS TABFIELDF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_tabfieldb(CHARTYPE *params)
#else
short Sos_tabfieldb(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 long save_where=0L,where=0L,what_current=0L,what_other=0L;
 unsigned short y=0,x=0,left_col=0;
 bool stay_in_current_field=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_tabfieldb");
#endif
/*---------------------------------------------------------------------*/
/* Determine if the cursor is in the left-most column of the current   */
/* field...                                                            */
/*---------------------------------------------------------------------*/
 getyx(CURRENT_WINDOW,y,x);
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
    case WINDOW_COMMAND:
         if (x != left_col)
            stay_in_current_field = TRUE;
         break;
    case WINDOW_PREFIX:
         if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_RIGHT
         &&  prefix_width != PREFIX_WIDTH)
            left_col = 1;
         else
            left_col = 0;
         if (x != left_col)
            stay_in_current_field = TRUE;
         break;
   }
/*---------------------------------------------------------------------*/
/* If the cursor was not in the left-most column of the current field, */
/* move it there now...                                                */
/*---------------------------------------------------------------------*/
 if (stay_in_current_field)
   {
    wmove(CURRENT_WINDOW,y,left_col);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* ... otherwise determine which is the previous enterable filed and   */
/* move the cursor there.                                              */
/*---------------------------------------------------------------------*/
 save_where = where = where_now();
 what_current = what_current_now();
 what_other = what_other_now();
 while(1)
   {
    where = where_before(where,what_current,what_other);
    if (where == save_where)
       break;
    if (enterable_field(where))
       break;
   }
/*---------------------------------------------------------------------*/
/* If we can't go anywhere, stay where we are...                       */
/*---------------------------------------------------------------------*/
 if (where == save_where)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 rc = go_to_new_field(save_where,where);
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos tabfieldf - move cursor to next enterable field

SYNTAX
     SOS TABFIELDf

DESCRIPTION
     The SOS TABFIELDF command causes the cursor to move to the next 
     enterable field on the screen. This command is intended to
     mimic the behavior of the TAB key on a 3270 terminal.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     SOS TABFIELDB

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_tabfieldf(CHARTYPE *params)
#else
short Sos_tabfieldf(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 long save_where=0L,where=0L,what_current=0L,what_other=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_tabfieldf");
#endif
 save_where = where = where_now();
 what_current = what_current_now();
 what_other = what_other_now();
 while(1)
   {
    where = where_next(where,what_current,what_other);
    if (where == save_where)
       break;
    if (enterable_field(where))
       break;
   }
/*---------------------------------------------------------------------*/
/* If we can't go anywhere, stay where we are...                       */
/*---------------------------------------------------------------------*/
 if (where == save_where)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 rc = go_to_new_field(save_where,where);
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos tabwordb - move cursor to beginning of previous word

SYNTAX
     SOS TABWORDB

DESCRIPTION
     The SOS TABWORDB command causes the cursor to move to the first character
     of the word to the left or to the start of the line if no more
     words precede.
     If the resulting column is beyond the left hand edge of the 
     FILEAREA, the window will scroll half a window.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     SOS TABWORDF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_tabwordb(CHARTYPE *params)
#else
short Sos_tabwordb(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
 short start_word_col=0;
 unsigned short word_break=0;
 CHARTYPE *temp_rec=NULL;
 register short i=0;
 short num_cols=0,col_pos=0,left_col=0;
 COLTYPE new_screen_col=0;
 LENGTHTYPE new_verify_col=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_tabwordb");
#endif
/*---------------------------------------------------------------------*/
/* This function is not applicable to the PREFIX window.               */
/*---------------------------------------------------------------------*/
 getyx(CURRENT_WINDOW,y,x);
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_PREFIX:
         display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
         trace_return();
#endif
         return(RC_INVALID_ENVIRON);
         break;
    case WINDOW_MAIN:
         temp_rec = rec;
         left_col = CURRENT_VIEW->verify_col-1;
         break;
    case WINDOW_COMMAND:
         temp_rec = (CHARTYPE *)cmd_rec;
         left_col = 0;
         break;
   }
/*---------------------------------------------------------------------*/
/* Determine the start of the prior word, or go to the start of the    */
/* line if already at or before beginning of prior word.               */
/*---------------------------------------------------------------------*/
 word_break = 0;
 start_word_col = (-1);
 for (i=left_col+x;i>0;i--)
   {
    switch(word_break)
      {
       case 0:  /* still in current word */
            if (*(temp_rec+i) == ' ')
               word_break++;
            break;
       case 1:  /* in first blank space */
            if (*(temp_rec+i) != ' ')
               word_break++;
            break;
       case 2:  /* in previous word */
            if (*(temp_rec+i) == ' ')
              {
               start_word_col = i+1;
               word_break++;
              }
            break;
       default: /* should not get here */
            break;
      }
    if (word_break == 3)
       break;
   }
 if (start_word_col == (-1))
    start_word_col = 0;

#ifdef VERSHIFT
 rc = execute_move_cursor(start_word_col);
#else
 calculate_new_column(x,CURRENT_VIEW->verify_col,start_word_col,&new_screen_col,&new_verify_col);
 if (CURRENT_VIEW->verify_col != new_verify_col
 &&  CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    CURRENT_VIEW->verify_col = new_verify_col;
    build_current_screen(); 
    display_current_screen();
   }
 wmove(CURRENT_WINDOW,y,new_screen_col);
#endif

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos tabwordf - move cursor to start of next word

SYNTAX
     SOS TABWORDf

DESCRIPTION
     The SOS TABWORDF command causes the cursor to move to the first character
     of the next word to the right or to the end of the line if no more
     words follow.
     If the resulting column is beyond the right hand edge of the
     FILEAREA, the window will scroll half a window.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     SOS TABWORDB

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_tabwordf(CHARTYPE *params)
#else
short Sos_tabwordf(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0,temp_rec_len=0,num_cols=0;
 short start_word_col=0,left_col=0;
 bool word_break=FALSE;
 CHARTYPE *temp_rec=NULL;
 register short i=0;
 COLTYPE new_screen_col=0;
 LENGTHTYPE new_verify_col=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_tabwordf");
#endif
/*---------------------------------------------------------------------*/
/* This function is not applicable to the PREFIX window.               */
/*---------------------------------------------------------------------*/
 getyx(CURRENT_WINDOW,y,x);
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_PREFIX:
         display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
         trace_return();
#endif
         return(RC_INVALID_ENVIRON);
         break;
    case WINDOW_MAIN:
         temp_rec = rec;
         temp_rec_len = rec_len;
         left_col = CURRENT_VIEW->verify_col-1;
         break;
    case WINDOW_COMMAND:
         temp_rec = (CHARTYPE *)cmd_rec;
         temp_rec_len = cmd_rec_len;
         left_col = 0;
         break;
   }
/*---------------------------------------------------------------------*/
/* If we are after the last column of the line, then just ignore the   */
/* command and leave the cursor where it is.                           */
/*---------------------------------------------------------------------*/
 if ((x + left_col) > temp_rec_len)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Determine the start of the next word, or go to the end of the line  */
/* if already at or past beginning of last word.                       */
/*---------------------------------------------------------------------*/
 word_break = FALSE;
 start_word_col = (-1);
 for (i=left_col+x;i<temp_rec_len;i++)
   {
    if (*(temp_rec+i) == ' ')
       word_break = TRUE;
    else
      {
       if (word_break)
         {
          start_word_col = i;
          break;
         }
      }
   }
 if (start_word_col == (-1))
    start_word_col = temp_rec_len;

#ifdef VERSHIFT
 rc = execute_move_cursor(start_word_col);
#else
 calculate_new_column(x,CURRENT_VIEW->verify_col,start_word_col,&new_screen_col,&new_verify_col);
 if (CURRENT_VIEW->verify_col != new_verify_col
 &&  CURRENT_VIEW->current_window == WINDOW_MAIN)
   {
    CURRENT_VIEW->verify_col = new_verify_col;
    build_current_screen(); 
    display_current_screen();
   }
 wmove(CURRENT_WINDOW,y,new_screen_col);
#endif

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos topedge - move cursor to top edge of filearea

SYNTAX
     SOS TOPEdge

DESCRIPTION
     The SOS TOPEDGE command moves the cursor to the first
     enterable line in the FILEAREA or PREFIX area. If the cursor
     is on the command line, the cursor moves to the first 
     enterable line of the FILEAREA.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Comaptible.

SEE ALSO
     SOS BOTTOMEDGE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_topedge(CHARTYPE *params)
#else
short Sos_topedge(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 unsigned short y=0,x=0,row=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_topedge");
#endif
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* Get the last enterable row. If an error, stay where we are...       */
/*---------------------------------------------------------------------*/
 if (find_first_focus_line(&row) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* For each window determine what needs to be done...                  */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_COMMAND:
         if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) != PREFIX_LEFT)
            x += PREFIX_WIDTH;
         CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[row].line_number;
         pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         CURRENT_VIEW->current_window = WINDOW_MAIN;
         wmove(CURRENT_WINDOW,row,x);
         break;
    case WINDOW_MAIN:
    case WINDOW_PREFIX:
            if (row != y)                            /* different rows */
              {
               post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
               CURRENT_VIEW->focus_line = CURRENT_SCREEN.sl[row].line_number;
               pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
               wmove(CURRENT_WINDOW,row,x);
              }
            break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos undo - undo changes to the current line

SYNTAX
     SOS UNDO

DESCRIPTION
     The SOS UNDO command causes the contents of the focus line (or the
     command line) to be reset to the contents before the cursor was
     positioned there.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible. 

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos_undo(CHARTYPE *params)
#else
short Sos_undo(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern CHARTYPE *pre_rec;
 extern unsigned short pre_rec_len;
 extern bool prefix_changed;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commsos.c: Sos_undo");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 switch (CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
         pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
         build_current_screen(); 
         display_current_screen();
         break;
    case WINDOW_COMMAND:
         memset(cmd_rec,' ',max_line_length);
         cmd_rec_len = 0;
         wmove(CURRENT_WINDOW,0,0);
         my_wclrtoeol(CURRENT_WINDOW);
         break;
    case WINDOW_PREFIX:
         prefix_changed = TRUE;
         memset(pre_rec,' ',PREFIX_WIDTH);
         pre_rec_len = 0;
         getyx(CURRENT_WINDOW,y,x);
         wmove(CURRENT_WINDOW,y,0);
         my_wclrtoeol(CURRENT_WINDOW);
         break;
    default:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
