;  TMS370 Debug monitor for use with NoICE370
;
;  Copyright (c) 1995 by John Hartman
;
;  Thanks to Brian Mohlman for building and lending a test board with 
;  which I could test MON370 and NOICE370.
;
;  Modification History:
;       23-Sep-95 JLH Version 1.0 release
;	19-Jan-96 JLH Correct bug in WriteReg: return status only.  ver 1.1
;
;============================================================================
;
;  To customize for a given target, you must change code in the
;  hardware equates, the string TSTG, and the routine RESET
;  You may or may not need to change GETCHAR, PUTCHAR, depending on
;  how peculiar your UART is.
;
;  This file has been assembled with the Texas Instruments free
;  TMS370 assembler SA370.
;
;  CAUTION:  SA370 DOES NOT DETECT OUT OF RANGE CONDITIONAL
;  BRANCH INSTRUCTIONS!  BE VERY CAREFUL IF YOU ADD CODE!
;
;  Support for PACT interrupts is commented out.  SA370 has no conditional
;  capability, so this is the best I can do.  Look for the symbol
;  $PACT$ below
;
;  NOTES ON MEMORY USAGE:
;  As written, this monitor does not require the permanent use of any
;  memory on page zero.  Locations 0 through 11 are saved into off-chip
;  RAM upon entry to the monitor, and restored before user program
;  execution begins.  The functions READ_BYTE and WRITE_BYTE contain
;  code to simulate reading and writing R0-R11 via NoICE commands.
;
;  Because the monitor has no permanent page zero locations, the
;  interrupt re-vectoring adds 142 cycles to interrupt service time.
;  If you can spare two bytes of page zero RAM for permanent use by
;  the monitor, this could be reduced to 21 cycles.  Refer to the
;  interrupt re-vectoring code near the bottom of this file.
;
;  The monitor places at most 5 bytes on the user stack when entered via
;  interrupt, trap, or breakpoint: ST, user PC, and two bytes more.
;
;  To add mapped memory support:
;       1) Define map port MAPREG here
;       2) Define map port RAM image MAPIMG here if MAPREG is write only
;       3) Search for and modify MAPREG, MAPIMG, and REG_PAGE usage below
;       4) In TSTG below edit "LOW AND HIGH LIMIT OF MAPPED MEM"
;          to appropriate range
;
;============================================================================
;  HARDWARE PLATFORM CUSTOMIZATIONS
;============================================================================
CHIP_RAM        .EQU    0000H           ;START OF ON-CHIP RAM/REGISTERS
;
;  Stack grows upwards.  The monitor uses at most 5 bytes of stack
;  for entry from an interrupt or TRAP.
STACK           .EQU    0001H           ;BASE OF PROCESSOR STACK AT RESET
MON_STACK       .EQU    7               ;STACK INIT VALUE IN MONITOR
SAVE_TOP        .EQU    12              ;A,B,6 REGS, 4 BYTES STACK
;
NOICE_CODE      .EQU    7800H           ;START OF MONITOR CODE
HARD_VECT       .EQU    7FC0H           ;START OF HARDWARE VECTORS
;$PACT$ if PACT is used, then
;;HARD_VECT       .EQU    7F9CH         ;START OF HARDWARE VECTORS (PACT)
;
NOICE_RAM       .EQU    08000H          ;START OF MONITOR RAM
USER_CODE       .EQU    09000H          ;START OF USER CODE
USER_VECT       .EQU    0FFC0H          ;START OF USER VECTORS
;$PACT$ if PACT is used, then
;;USER_VECT       .EQU    0FF9CH        ;START OF USER VECTORS (PACT)
;
;============================================================================
;
;  TMS370 perhipherals
SCCR0   .EQU  P010   ;System control register 0; mode pin
SCCR1   .EQU  P011   ;System control register 1; memory configuration
SCCR2   .EQU  P012   ;System control register 2; powerdown states
;
APORT1  .EQU  P020   ;Port A control 1
APORT2  .EQU  P021   ;Port A control 2
ADATA   .EQU  P022   ;Port A data
ADIR    .EQU  P023   ;Port A data direction 1=output
BPORT1  .EQU  P024   ;Port B control 1
BPORT2  .EQU  P025   ;Port B control 2
BDATA   .EQU  P026   ;Port B data
BDIR    .EQU  P027   ;Port B data direction 1=output
CPORT1  .EQU  P028   ;Port C control 1
CPORT2  .EQU  P029   ;Port C control 2
CDATA   .EQU  P02A   ;Port C data
CDIR    .EQU  P02B   ;Port C data direction 1=output
DPORT1  .EQU  P02C   ;Port D control 1
DPORT2  .EQU  P02D   ;Port D control 2
DDATA   .EQU  P02E   ;Port D data
DDIR    .EQU  P02F   ;Port D data direction 1=output
;
WDRST   .EQU  P048   ;Watchdog timer reset key
T1CTL1  .EQU  P049   ;Timer 1 control registers
T1CTL2  .EQU  P04A
;
SCICCR  .EQU  P050   ;Mode control
SCICTL  .EQU  P051   ;Operation control register
BAUDMSB .EQU  P052   ;Baud control: high byte
BAUDLSB .EQU  P053   ;Baud control: low byte
TXCTL   .EQU  P054   ;TX interrupt control and status register
RXCTL   .EQU  P055   ;RX interrupt control and status register
RXBUF   .EQU  P057   ;RX data buffer
TXBUF   .EQU  P059   ;TX data buffer
SCIPC1  .EQU  P05D   ;Port control reg 1
SCIPC2  .EQU  P05E   ;Port control reg 2
SCIPRI  .EQU  P05F   ;Priority select
;
;  Bit equates
TXRDY   .EQU    80H     ;TRANSMIT BUFFER EMPTY
RXRDY   .EQU    40H     ;RECEIVE BUFFER FULL

;==========================================================================
;
;  Communications function codes.
FN_GET_STATUS         .equ    0FFH    ;reply with device info
FN_READ_MEM           .equ    0FEH    ;reply with data
FN_WRITE_MEM          .equ    0FDH    ;reply with status (+/-)
FN_RD_REGS            .equ    0FCH    ;reply with registers
FN_WR_REGS            .equ    0FBH    ;reply with status
FN_RUN_TARGET         .equ    0FAH    ;reply (delayed) with registers
FN_SET_BYTES          .equ    0F9H    ;reply with data (truncate if error)
FN_IN                 .equ    0F8H    ;input from port
FN_OUT                .equ    0F7H    ;output to port
;
FN_MIN                .equ    0F7H    ;MINIMUM RECOGNIZED FUNCTION CODE
FN_ERROR              .equ    0F0H    ;error reply to unknown op-code
;
;==========================================================================
;  Non-page zero Monitor RAM definitions
        .DATA    NOICE_RAM
;
;  Target registers:  order must match that in TRG370.C
;  Knowledge of order is also assumed by DEFAULT_INT, below
;
TASK_REGS:
REG_STATE:     .BLOCK  1
REG_PAGE:      .BLOCK  1
REG_FLAGS:     .BLOCK  1
REG_PCL:       .BLOCK  1
REG_PCH:       .BLOCK  1
REG_SP:        .BLOCK  1
REG_A:         .BLOCK  1
REG_B:         .BLOCK  1
;
T_REGS_SIZE     .EQU    $-TASK_REGS
;
;  Save page-0 "registers" used by NoICE monitor here
;  For simplest code in READ_BYTE and WRITE_BYTE, this must immediately 
;  follow REG_B
;  - save 6 bytes we use for registers (R2-R7)
;  - save 4 bytes we use for stack (R8-R11)
SAVE_REG:       .BLOCK  SAVE_TOP-2
;
;  Communications buffer
;  (Must be at least as long as the longer of TASK_REG_SZ or TSTG_SIZE.
;  At least 19 bytes recommended.  Larger values may improve speed of NoICE
;  download and memory move commands.)
COMBUF_SIZE     .EQU    64              ;DATA SIZE FOR COMM BUFFER
COMBUF:         .BLOCK  2+COMBUF_SIZE+1 ;BUFFER ALSO HAS FN, LEN, AND CHECK
;
;===========================================================================
        .TEXT NOICE_CODE                        ;reset start address
;
;  Power on reset
;
;  Reset clears A and B, and sets SP = 01.  Other registers unchanged.
;  Causes of reset:
;  1) power on.  Cold start bit (bit7) in SCCR0 set
;  2) OSC out of range.  Cold start = 0, OSC bit (bit4) in SCCR0 set
;  3) watchdog timeout.  Cold start = 0, WDT bit (bit5) in T1CTL2 set
;  4) reset pin.  None of the above
;
RESET:  LDST    #0                      ;disable interrupts (be sure)
;
;  Save user RAM we wish to use for scratch registers
;  (After WDT or OSC fail, these may have meaning)
        CLR     B
SAVER0: MOV     2(B),A                  ;register 2 thru SAVE_TOP-1
        MOV     A,SAVE_REG(B)
        INC	B
        CMP     #SAVE_TOP,B
        JLO     SAVER0
;
        MOV     #MON_STACK,B            ;set up stack pointer
        LDSP
;
        OR      #10H,SCCR1              ;Disable auto wait states
;
	MOV	#0FFH,A
        MOV     A,APORT2            	;A=1 B=1, OUTPUT ENABLE=1
        MOV     A,ADIR
        MOV     A,BPORT2
        MOV     A,BDIR
        MOV     A,CPORT2
        MOV     A,CDIR
        MOV     A,DPORT1
        MOV     A,DPORT2
        MOV     A,DDIR
;
;  Determine cause of reset, and set REG_STATE appropriately.
;  state 0 = reset by pin
;  (state 1 = breakpoint, standard for all processors)
;  state 2 = power up
;  state 3 = OSC fail
;  state 4 = WDT
        MOV     #2,A
        BTJO    #80H,SCCR0,RR10         ;JIF POWER ON RESET (BIT SET)
        MOV     #3,A
        BTJO    #10H,SCCR0,RR10         ;JIF OSC FAIL RESET (BIT SET)
        MOV     #4,A
        BTJO    #20H,T1CTL2,RR10        ;JIF WDT RESET (BIT SET)
        MOV     #0,A                    ;ELSE RESET PIN RESET
RR10:   MOV     A,REG_STATE             ;SAVE STATE OF RESET
        AND     #01101111B,SCCR0        ;CLEAR POWER ON, OSC FAIL BITS
        AND     #11011111B,T1CTL2       ;CLEAR WDT BIT
;
;  Stall, if necessary (such as external UART which may remain in reset
;  longer than the micro.  In this case, init UART AFTER this delay)
;  Wait also lets MAXIM charge pumps get up to speed before we transmit.
	MOV     #10,A
        CLR     B
LLPP:   DJNZ    B,LLPP        ;LONG DELAY IN CASE EXTERNAL UART IN RESET
	DJNZ    A,LLPP
;
;============================================================================
;  Initialize your UART and other hardware here
        MOV     #00010000B,SCICTL
        ;          0                    ;reset SCI
        ;           1                   ;internal SCLK
        ;             0                 ;sleep disabled
        ;              00               ;disable TX and RX
        MOV     #00000000B,SCIPC1
        ;              0                ;SCICLK is general purpose pin
        ;               0               ;SCICLK is input
        MOV     #00100010B,SCIPC2
        ;          1                    ;Enable Tx pin
        ;              1                ;Enable Rx pin
        MOV     #00000000B,TXCTL
        ;               0               ;disable TX interrupts
        MOV     #00000000B,RXCTL
        ;               0               ;disable RX interrupts
        MOV     #00010111B,SCICCR       ;8-bits, even parity, 1 stop bit
        ;        0                      ;1 stop bit
        ;         0                     ;odd parity (not used)
        ;          0                    ;no parity
        ;           1                   ;asynchronous mode
        ;            0                  ;idle line protocol
        ;             111               ;8 data bits
;
;  Init baud rate:  rate = CLKIN/( (divisor + 1)*128 )
;  With 20 Mhz crystal, 9600 baud: DIV = 15 (1.73 % error)
;  With 20 Mhz crystal, 19200 baud: DIV = 7 (1.73 % error)
;  With 20 Mhz crystal, 38400 baud: DIV = 3 (1.73 % error)
        MOV     #0,A
        MOV     A,BAUDMSB               ;initialize baud rate
        MOV     #7,A
        MOV     A,BAUDLSB
;
        MOV     #00110011B,SCICTL
        ;          1                    ;enable SCI (not reset)
        ;           1                   ;internal SCLK
        ;             0                 ;sleep disabled
        ;              11               ;enable TX and RX
;
        MOV     RXBUF,A                 ;clear out garbage from SCI
;
;  If Watchdog timer is desired, enable it here.
;  (GetChar and PutChar renew the WDT)

;
;  Initialize other user hardware
;  (Or defer to application code)

;============================================================================
;
;  Initialize user's trap vectors to default values.
;  (Should be overlaid by user's code, but catch the shlitz)
        MOVW    #USER_VECT,R5           ;ADDRESS OF USER VECTORS
        MOV     #NTRAP/2,B              ;NUMBER OF TRAP VECTORS
TRAPV:  MOV     #DEFAULT_TRAP/256,A
        MOV     A,@R5                   ;SET VECTOR WITH DEFAULT HANDLER
        MOV     #DEFAULT_TRAP,A
        MOV     A,1(R5)
        INCW    #2,R5
        DJNZ    B,TRAPV
;
;  Initialize user's interrupt vectors to default values.
;  (Should be overlaid by user's code, but catch the shlitz)
;  (R4,R5) points at next RAM vector
        MOV     #NVEC/2,B               ;NUMBER OF VECTORS
INITV:  MOV     #DEFAULT_INT/256,A
        MOV     A,@R5                   ;SET VECTOR WITH DEFAULT HANDLER
        MOV     #DEFAULT_INT,A
        MOV     A,1(R5)
        INCW    #2,R5
        DJNZ    B,INITV
;
;  Initialize user's registers
;  (REG_STATE was set above as per cause of RESET)
        CLR     A
        MOV     A,REG_PAGE              ;INITIAL USER MEMORY PAGE
        MOV     A,REG_FLAGS             ;reset clears ST
        MOV     A,REG_A                 ;reset clears A
        MOV     A,REG_B                 ;reset clears B
        MOV     #USER_CODE/256,A        ;INITIAL PC IN USER CODE
        MOV     A,REG_PCH
        MOV     #USER_CODE,A
        MOV     A,REG_PCL
        MOV     #STACK,A                ;reset sets SP=1
        MOV     A,REG_SP
;
;  Set function code for "GO".  Then if we are here because of a reset
;  (such as a watchdog timeout) after being told to GO, we will come
;  back with registers so user can see the reset
        MOV     #FN_RUN_TARGET,A
        MOV     A,COMBUF
        JMPL    RETURN_REGS             ;DUMP REGS, ENTER MONITOR
;
;===========================================================================
;  Get a character to A
;
;  Return A=char, CY=0 if data received
;         CY=1 if timeout (0.5 seconds)
;
;  Uses 2 bytes of stack including return address
;
GETCHAR:
;
;  (Add timeout (return CY=1) if no byte recieved in 500 msec, if desired)
GC10:
;
;  If your application uses the Watchdog Timer, enable this code.
        MOV     #055H,WDRST
        MOV     #0AAH,WDRST
;
        BTJZ    #RXRDY,RXCTL,GC10       ;loop til ready
;
;  Data received:  return CY=0. data in A
        MOV     RXBUF,A
        TST     A                       ;clear carry
        RTS
;
;===========================================================================
;  Output character in A
;
;  Uses 2 bytes of stack including return address
;
PUTCHAR:
;
;  If your application uses the Watchdog Timer, enable this code.
        MOV     #055H,WDRST
        MOV     #0AAH,WDRST
;
        BTJZ     #TXRDY,TXCTL,PUTCHAR  ;loop til ready for output
        MOV      A,TXBUF               ;place the character into the buffer
        RTS
;
;===========================================================================
;
;  Response string for GET TARGET STATUS request
;  Reply describes target:
TSTG:   .byte   9                       ;2: PROCESSOR TYPE = TMS370
        .byte   COMBUF_SIZE             ;3: SIZE OF COMMUNICATIONS BUFFER
        .byte   0                       ;4: NO OPTIONS
        .word   0,0                     ;5-8: LOW AND HIGH LIMIT OF MAPPED MEM (NONE)
        .byte   B1-B0                   ;9 BREAKPOINT INSTRUCTION LENGTH
B0:     TRAP    15                      ;10+ BREKAPOINT INSTRUCTION (TRAP)
B1:     .string "TMS370 monitor V1.1" 	;DESCRIPTION, ZERO
	.byte	0
BRK_SIZE        .equ    B1-B0           ;DEFINE LENGTH OF A BREAKPOINT INSTR.
TSTG_SIZE       .equ    $-TSTG          ;SIZE OF STRING
;
;===========================================================================
;  Read byte from memory (R2,R3) to A
;
;  Allows address translation and protection
;
READ_BYTE:
;
;  Test for access to memory below R8
;  Use REG_A, REG_B, SAVE_R2, etc. instead
        CMP     #0,R2
        JNZ     RB80                    ;JIF ABOVE REGISTERS
        CMP     #SAVE_TOP,R3
        JHS     RB80                    ;JIF ABOVE REGISTERS AND STACK
        PUSH    B
        MOV     R3,B
        MOV     REG_A(B),A             ;GET SAVED REGISTER INSTEAD
        POP     B
        RTS
;
RB80:   MOV     0(R3),A
        RTS
;
;===========================================================================
;  Write byte from A to memory (R2,R3)
;
;  Allows address translation and protection
;
WRITE_BYTE:
;
;  Test for access to memory below R8
;  Use REG_A, REG_B, SAVE_R2, etc. instead
        CMP     #0,R2
        JNZ     WB80                    ;JIF ABOVE REGISTERS
        CMP     #SAVE_TOP,R3
        JHS     WB80                    ;JIF ABOVE REGISTERS AND STACK
        PUSH    B
        MOV     R3,B
        MOV     A,REG_A(B)              ;SET SAVED REGISTER INSTEAD
        POP     B
        RTS
;
WB80:   MOV     A,0(R3)
        RTS
;
;===========================================================================
;  HARDWARE PLATFORM INDEPENDENT EQUATES AND CODE
;
;===========================================================================
;  Enter here via TRAP or CALL for breakpoint.
;
;  PC is stacked; A, B, and FLAGS are not saved; interrupt state unchanged
;
BREAKPOINT_ENTRY:
        PUSH    ST                      ;3 bytes on user stack
        LDST    #0                      ;DISABLE INTERRUPTS
	MOV     A,REG_A			;save register A
        POP     A
        MOV     A,REG_FLAGS             ;save pre-trap flags
        MOV	B,A
        MOV     A,REG_B			;save register B
;
        POP     B                       ;Get interrupted PC
        POP     A
        INCW    #-BRK_SIZE,R1           ;BACK UP PC TO POINT AT BREAKPOINT
        MOV     A,REG_PCH
        MOV     B,A
        MOV     A,REG_PCL
;
        MOV     #1,A                    ;state = breakpoint
        MOV     A,REG_STATE
        JMP     INT_COMMON

;===========================================================================
;
;  Default trap handler:  enter with REG_STATE = interrrupt number
;
;  FLAGS not saved, interrupt state unchanged
;
;  REG_A and REG_B have saved registers (also in A and B)
DEFAULT_TRAP:
        PUSH    ST                      ;3 bytes on user stack
        LDST    #0                      ;DISABLE INTERRUPTS
        POP     A
        MOV     A,REG_FLAGS             ;save pre-trap flags
;
        POP     A                       ;Get interrupted PC
        MOV     A,REG_PCL
        POP     A
        MOV     A,REG_PCH
        JMP     INT_COMMON

;===========================================================================
;
;  Default interrupt handler:  enter with REG_STATE = interrrupt number
;
;  REG_A and REG_B have saved registers (also in A and B)
DEFAULT_INT:
        POP     A                       ;Get interrupted PC
        MOV     A,REG_PCL
        POP     A
        MOV     A,REG_PCH
;
        POP     A
        MOV     A,REG_FLAGS             ;save pre-interrupt flags
        JMP     INT_COMMON
;
;===========================================================================
;  Common code for breakpoint, trap, or interrupt entry
;
INT_COMMON:
;
;  Save user stack pointer
        STSP
        XCHB    A
        MOV     A,REG_SP                ;value prior to interrupt/trap
;
;  Switch to monitor stack
        MOV     #MON_STACK,B
        LDSP
;
;  Save memory mapping register (if used)
        CLR     A                       ;fake bank, if unmapped
;;;     MOV     MAPREG,A                ;or bank, if mapped
        MOV     A,REG_PAGE              ;save PAGE, if any, else 0
;
;  Save user RAM we wish to use for scratch registers
        CLR     B
SAVER:  MOV     2(B),A                  ;register 2 thru SAVE_TOP-1
        MOV     A,SAVE_REG(B)
        INC	B
        CMP     #SAVE_TOP,B
        JLO     SAVER
;
;  Return registers to master
        JMPL    RETURN_REGS
;
;===========================================================================
;  Main loop:  wait for command frame from master
MAIN:   MOV     #MON_STACK,B            ;A CLEAN STACK IS A HAPPY STACK
        LDSP
        MOVW    #COMBUF,R5              ;BUILD MESSAGE HERE
;
;  First byte is a function code
        CALL    GETCHAR                 ;GET A FUNCTION (uses 2 bytes of stack)
        JC      MAIN                    ;JIF TIMEOUT: RESYNC
        CMP     #FN_MIN,A
        JLO     MAIN                    ;JIF BELOW MIN: ILLEGAL FUNCTION
        MOV     A,@R5                   ;SAVE FUNCTION CODE
        INCW    #1,R5
;
;  Second byte is data byte count (may be zero)
        CALL    GETCHAR                 ;GET A LENGTH BYTE
        JC      MAIN                    ;JIF TIMEOUT: RESYNC
        CMP     #COMBUF_SIZE+1,A
        JHS     MAIN                    ;JIF TOO LONG: ILLEGAL LENGTH
        MOV     A,@R5                   ;SAVE LENGTH
        INCW    #1,R5
        MOV     A,R6                    ;R6 = COUNT
        JZ      MA80                    ;SKIP DATA LOOP IF LENGTH IS 0
;
;  Loop for data
MA10:   CALL    GETCHAR                 ;GET A DATA BYTE
        JC      MAIN                    ;JIF TIMEOUT: RESYNC
        MOV     A,@R5                   ;SAVE DATA BYTE
        INCW    #1,R5
        DJNZ    R6,MA10
;
;  Get the checksum
MA80:   CALL    GETCHAR                 ;GET THE CHECKSUM
        JC      MAIN                    ;JIF TIMEOUT: RESYNC
        MOV     A,R6                    ;SAVE CHECKSUM
;
;  Compare received checksum to that calculated on received buffer
;  (Sum should be 0)
        CALL    CHECKSUM                ;GET CHECKSUM TO A
        ADD     R6,A
        JNE     MAIN                    ;JIF BAD CHECKSUM (A != 0)
;
;  Process the message.
        MOVW    #COMBUF,R5
        MOV     1(R5),A                 ;GET THE LENGTH
        MOV     A,B
        MOV     @R5,A                   ;GET THE FUNCTION CODE
        INCW    #2,R5
;
        CMP     #FN_GET_STATUS,A
        JEQ     TARGET_STATUS
        CMP     #FN_READ_MEM,A
        JEQ     READ_MEM
        CMP     #FN_WRITE_MEM,A
        JEQ     WRITE_MEM
        CMP     #FN_RD_REGS,A
        JEQ     JREAD_REGS
        CMP     #FN_WR_REGS,A
        JEQ     JWRITE_REGS
        CMP     #FN_RUN_TARGET,A
        JEQ     JRUN_TARGET
        CMP     #FN_SET_BYTES,A
        JEQ     JSET_BYTES
        CMP     #FN_IN,A
        JEQ     JIN_PORT
        CMP     #FN_OUT,A
        JEQ     JOUT_PORT
;
;  Error: unknown function.  Complain
        MOV     #FN_ERROR,A
        MOV     A,COMBUF                ;SET FUNCTION AS "ERROR"
        MOV     #1,B
        JMPL    SEND_STATUS             ;VALUE IS "ERROR"

JREAD_REGS:   JMPL   READ_REGS
JWRITE_REGS:  JMPL   WRITE_REGS
JRUN_TARGET:  JMPL   RUN_TARGET
JSET_BYTES:   JMPL   SET_BYTES
JIN_PORT:     JMPL   IN_PORT
JOUT_PORT:    JMPL   OUT_PORT

;===========================================================================
;
;  Target Status:  FN, len
;
;  Enter with A=function, B=length, (R4,R5) points at first data byte
;
TARGET_STATUS:
;
        MOVW    #TSTG,R7                ;STATUS STRING
        MOV     #TSTG_SIZE,A            ;LENGTH OF REPLY
        MOV     A,-1(R5)                ;SET SIZE IN REPLY BUFFER
        MOV	A,B
;
;  Loop on data in string
TS10:   MOV     @R7,A
        INCW    #1,R7
        MOV     A,@R5
        INCW    #1,R5
        DJNZ    B,TS10
;
;  Compute checksum on buffer, and send to master, then return
        JMPL    SEND

;===========================================================================
;
;  Read Memory:  FN, len, page, Alo, Ahi, Nbytes
;
;  Enter with A=function, B=length, (R4,R5) points at first data byte
;
READ_MEM:
;
;  Set map
;;      MOV     @R5,A
;;      MOV     A,MAPREG
;
;  Get address to (R2,R3)
        MOV     1(R5),A                 ;LSB ADDRESS
        MOV     A,R3
        MOV     2(R5),A                 ;MSB ADDRESS
        MOV     A,R2
;
;  Get byte count
        MOV     3(R5),A                 ;BYTE COUNT
        MOV     A,B
;
;  Prepare return buffer: FN (unchanged), LEN, DATA
        MOV     A,-1(R5)                ;RETURN LENGTH = REQUESTED DATA
        TST     A
        JEQ     GLP90                   ;JIF NO BYTES TO GET (A=0)
;
;  Read the requested bytes from local memory
GLP:    CALL    READ_BYTE
        INCW    #1,R3
        MOV     A,@R5
        INCW    #1,R5
        DJNZ    B,GLP
;
;  Compute checksum on buffer, and send to master, then return
GLP90:  JMPL    SEND

;===========================================================================
;
;  Write Memory:  FN, len, page, Alo, Ahi, (len-3 bytes of Data)
;
;  Enter with A=function, B=length, (R4,R5) points at first data byte
;
WRITE_MEM:
;
;  Set map
;;      MOV     @R5,A
;;      MOV     A,MAPREG
;
;  Get address to (R2,R3)
        MOV     1(R5),A                 ;LSB ADDRESS
        MOV     A,R3
        MOV     2(R5),A                 ;MSB ADDRESS
        MOV     A,R2
;
;  Get length to write
        SUB     #3,B                    ;BYTE COUNT LESS PAGE, ADDRESS
        JEQ     WLP80                   ;EXIT OF NO BYTES REQUESTED (A=0)
        MOV     B,R7                    ;SAVE COPY IN R7 FOR COMPARE LOOP
        INCW    #3,R5
;
;  Write the requested bytes to local memory
WLP:    MOV     @R5,A                   ;GET BYTE TO WRITE
        INCW    #1,R5
        CALL    WRITE_BYTE              ;TO MEMORY
        INCW    #1,R3
        DJNZ    B,WLP                   ;LOOP ON WRITING BYTES
;
;  Compare to see if the write worked
;
;  Get memory address again, point (R4,R5) at source data
        MOVW    #COMBUF+2,R5
        MOV     1(R5),A                 ;LSB ADDRESS
        MOV     A,R3
        MOV     2(R5),A                 ;MSB ADDRESS
        MOV     A,R2
        INCW    #3,R5
;
;  Compare the requested bytes to local memory
WLP50:  CALL    READ_BYTE               ;GET BYTE FROM MEMORY
        INCW    #1,R3
        CMP     @R5,A                   ;COMPARE TO BYTE WE WROTE
        JNE     WLP90                   ;JIF NOT SAME AS WRITTEN VALUE
        INCW    #1,R5
        DJNZ    R7,WLP50                ;LOOP ON COMPARING BYTES
;
;  Write succeeded:  return status = 0
WLP80:  CLR     A                       ;RETURN STATUS = 0
        JMP     WLP100
;
;  Write failed:  return status = 1
WLP90:  MOV     #1,A
;
;  Return OK status
WLP100: JMPL    SEND_STATUS

;===========================================================================
;
;  Read registers:  FN, len=0
;
;  Enter with A=function, B=length, (R4,R5) points at first data byte
;
READ_REGS:
;
;  Enter here from int after "RUN" and "STEP" to return task registers
;  All registers unknown
;  COMBUF+0 will contain either FN_RD_REGS or FN_RUN_TARGET
RETURN_REGS:
        MOVW    #COMBUF+2,R5	        ;COMMUNICATIONS BUFFER: 1ST DATA BYTE
        MOVW    #TASK_REGS,R3           ;REGISTER LIVE HERE
        MOV     #T_REGS_SIZE,B          ;NUMBER OF BYTES
;
;  Prepare return buffer: FN (unchanged), LEN, DATA
        MOV     B,A
        MOV     A,-1(R5)                ;SAVE DATA LENGTH
;
;  Copy the registers
GRLP:   MOV     @R3,A
        INCW    #1,R3
        MOV     A,@R5
        INCW    #1,R5
        DJNZ    B,GRLP
;
;  Compute checksum on buffer, and send to master, then return
        JMPL    SEND

;===========================================================================
;
;  Write registers:  FN, len, (register image)
;
;  Enter with A=function, B=length, (R4,R5) points at first data byte
;
WRITE_REGS:
;
        CMP     #0,B
        JEQ     WRR80                   ;JIF NO REGISTERS
        MOVW    #TASK_REGS,R3
;
;  Copy the registers
WRLP:   MOV     @R5,A
        INCW    #1,R5
        MOV     A,@R3
        INCW    #1,R3
        DJNZ    B,WRLP
;
;  Return OK status
WRR80:  CLR     A
        JMPL    SEND_STATUS

;===========================================================================
;
;  Run Target:  FN, len
;
;  Uses 3 bytes of user stack, 0 bytes of monitor stack
;
;  Enter with A=function, B=length, (R4,R5) points at first data byte
;
RUN_TARGET:
;
;  Restore user RAM we used for scratch registers
        CLR     B
RESER:  MOV     SAVE_REG(B),A
        MOV     A,2(B)                  ;register 2 thru SAVE_TOP-1
        INC	B
        CMP     #SAVE_TOP,B
        JLO     RESER
;
;  Restore user's map
;;      MOV     REG_PAGE,A
;;      MOV     A,MAPREG
;
;  Switch to user stack
        MOV     REG_SP,A
        MOV     A,B
        LDSP
;
;  Push user's FLAGS for RTI
        MOV     REG_FLAGS,A
        PUSH    A
;
;  Push user PC for RTI
        MOV     REG_PCH,A
        PUSH    A
        MOV     REG_PCL,A
        PUSH    A
;
;  Restore final user registers
        MOV     REG_B,A
        MOV     A,B
        MOV     REG_A,A
;
;  Return to user
        RTI
;
;===========================================================================
;
;  Set target byte(s):  FN, len { (page, alow, ahigh, data), (...)... }
;
;  Return has FN, len, (data from memory locations)
;
;  If error in insert (memory not writable), abort to return short data
;
;  This function is used primarily to set and clear breakpoints
;
;  Enter with A=function, B=length, (R4,R5) points at first data byte
;
;  Uses 3 bytes of stack
;
SET_BYTES:
        CMP     #0,B
        JEQ     SB90            ;JIF NO BYTES (A=0)
        MOV     R4,R6           ;POINTER TO REPLY BUFFER
        MOV     R5,R7
;
;  Loop on inserting bytes
SB10:
;
;  Set map
;;      MOV     @R5,A
;;      MOV     A,MAPREG
;
;  Get address to (R2,R3)
        MOV     1(R5),A
        MOV     A,R3
        MOV     2(R5),A
        MOV     A,R2
;
;  Read current data at byte location
        CALL    READ_BYTE               ;READ CURRENT DATA FROM (R2,R3)
        MOV     A,@R7                   ;SAVE IN REPLY BUFFER
;
;  Insert new data at byte location
        MOV     3(R5),A
        CALL    WRITE_BYTE              ;WRITE BYTE TO (R2,R3)
        CALL    READ_BYTE               ;READ IT BACK
        CMP     3(R5),A                 ;COMPARE TO WHAT WE WROTE
        JNE     SB90                    ;JIF INSERT FAILED: ABORT
;
;  Advance to next breakpoint
        INCW    #4,R5                   ;NEXT BYTE TO INSERT
        INCW    #1,R7                   ;NEXT REPLY BYTE
        SUB     #4,B
        JNE     SB10                    ;LOOP FOR ALL BYTES
;
;  Return buffer with data from byte locations
;  R7 points at next free byte in return buffer.  Compute length
SB90:   MOV     R7,A
        SUB     #COMBUF+2,A             ;A = BYTES WRITTEN SUCCESSFULLY
        MOV     A,COMBUF+1              ;SET COUNT OF RETURN BYTES
;
;  Compute checksum on buffer, and send to master, then return
        JMPL    SEND

;===========================================================================
;
;  Input from port:  FN, len, PortAddressLow, PortAddressHigh
;
;  Enter with A=function, B=length, (R4,R5) points at first data byte
;
IN_PORT:
;
;  Get port address
        MOV     @R5,A                   ;LSB ADDRESS
        MOV     A,R3
        MOV     1(R5),A                 ;MSB ADDRESS
        MOV     A,R2
;
;  Read port value
        CALL    READ_BYTE               ;READ BYTE (R2,R3)
;
;  Return byte read as "status"
        JMPL    SEND_STATUS

;===========================================================================
;
;  Output to port:  FN, len, PortAddressLo, PAhi, data
;
;  Enter with A=function, B=length, (R4,R5) points at first data byte
;
OUT_PORT:
;
;  Get port address
        MOV     @R5,A                   ;LSB ADDRESS
        MOV     A,R3
        MOV     1(R5),A                 ;MSB ADDRESS
        MOV     A,R2
;
        MOV     2(R5),A                 ;BYTE TO WRITE
        CALL    WRITE_BYTE              ;OUTPUT!
;
;  Return status of OK
        CLR     A
        JMP     SEND_STATUS
;
;===========================================================================
;  Build status return with value from "A"
;
SEND_STATUS:
        MOV     A,COMBUF+2              ;SET STATUS
        MOV     #1,A
        MOV     A,COMBUF+1              ;SET LENGTH
        JMP     SEND

;===========================================================================
;  Append checksum to COMBUF and send to master
;
;  Uses 2 bytes of stack (jumped to, not called)
;
SEND:   CALL    CHECKSUM                ;GET A=CHECKSUM, (R4,R5)->CHECKSUM LOCATION
        COMPL   A
        MOV     A,@R5                   ;STORE NEGATIVE OF CHECKSUM
;
;  Send buffer to master
        MOVW    #COMBUF,R5              ;POINTER TO OUTPUT BUFFER
        MOV     COMBUF+1,A              ;LENGTH OF DATA
        ADD     #3,A                    ;PLUS FUNCTION, LENGTH, CHECKSUM
        MOV     A,B
SND10:  MOV     @R5,A
        INCW    #1,R5
        CALL    PUTCHAR                 ;SEND A BYTE
        DJNZ    B,SND10
        JMPL    MAIN                    ;BACK TO MAIN LOOP

;===========================================================================
;  Compute checksum on COMBUF.  COMBUF+1 has length of data,
;  Also include function byte and length byte
;
;  Returns:
;       A, R7 = checksum
;       (R4,R5) = pointer to next byte in buffer (checksum location)
;       B scratched
;
;  Uses 2 bytes of stack including return address
;
CHECKSUM:
        MOVW    #COMBUF,R5              ;POINTER TO BUFFER
        MOV     COMBUF+1,A              ;LENGTH OF DATA
        ADD     #2,A                    ;PLUS FUNCTION AND LENGTH BYTES
        MOV     A,B
        CLR     R7                      ;INIT CHECKSUM TO ZERO
CHK10:  MOV     @R5,A
        ADD     A,R7
        INCW    #1,R5
        DJNZ    B,CHK10                 ;loop for all
        MOV     R7,A
        RTS                             ;return with checksum in A

;===========================================================================
;
;  Interrupt handler to re-vector interrupts through user RAM
;
;  For interrupt, flags and PC are on stack, and flags are cleared.
;  For trap, PC is on stack, flags are unaffected
;
;  This code is carefully crafted to use no page zero RAM.
;  If you can allocate two bytes of page zero RAM, Rn, for permanent
;  use by the monitor, replace each xxx_H handler by code:
;
;       xxx_H:  MOVW    #USER_VECT+nn,Rn+1      ;13 LOAD USER'S VECTOR
;               BR      @Rn+1                   ; 8 JUMP THRU VECTOR
;                                               ;--
;                                               ;21 cycles, 6 bytes
;
;  This would decrease the latency added by RAM re-vectoring from 142
;  cycles to 21 cycles.  In addition to the xxx_H handlers, you would 
;  need to add default handlers to process interrupts not handled by user 
;  code.  The RAM vectors would be initialized with the specific
;  default_xxx_H addresses, rather than all being initialized with 
;  generic DEFAULT_TRAP or DEFAULT_INT.
;
;       default_xxx_H:
;               MOV     A,REG_A                 ;save A
;               MOV     #X,A                    ;interrupt number
;               JMPL    INT_ENTRY               ;join common code
;						;8 bytes each
;
;  Caution: xxx_H above is not exactly equivalent to the direct connection
;  of an interrupt handler to a vector:  condition code bits in flag 
;  register are affected.  This is not true of the longer form below,
;  but should cause few problems, as interrupt routines have no business
;  looking at their entry condition codes anyway!
;
;===========================================================================
;
;  Vector traps thru RAM.  State numbers must match TRG370.C
;
TRAP15_H:       PUSH    ST                      ; 8 save pre-trap flags
                MOV     A,REG_A                 ;10
                MOV     #5,A                    ; 6 processor state
                JMP     TRAP_ENTRY              ; 7 join common code
;                                               ;---
;                                               ;31 cycles
TRAP14_H:       PUSH    ST
                MOV     A,REG_A
                MOV     #6,A
                JMP     TRAP_ENTRY
;
TRAP13_H:       PUSH    ST
                MOV     A,REG_A
                MOV     #7,A
                JMP     TRAP_ENTRY
;
TRAP12_H:       PUSH    ST
                MOV     A,REG_A
                MOV     #8,A
                JMP     TRAP_ENTRY
;
TRAP11_H:       PUSH    ST
                MOV     A,REG_A
                MOV     #9,A
                JMP     TRAP_ENTRY
;
TRAP10_H:       PUSH    ST
                MOV     A,REG_A
                MOV     #10,A
                JMP     TRAP_ENTRY
;
TRAP9_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #11,A
                JMP     TRAP_ENTRY
;
TRAP8_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #12,A
                JMP     TRAP_ENTRY
;
TRAP7_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #13,A
                JMP     TRAP_ENTRY
;
TRAP6_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #14,A
                JMP     TRAP_ENTRY
;
TRAP5_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #15,A
                JMP     TRAP_ENTRY
;
TRAP4_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #16,A
                JMP     TRAP_ENTRY
;
TRAP3_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #17,A
                JMP     TRAP_ENTRY
;
TRAP2_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #18,A
                JMP     TRAP_ENTRY
;
TRAP1_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #19,A
                JMP     TRAP_ENTRY
;
TRAP0_H:        PUSH    ST
                MOV     A,REG_A
                MOV     #20,A
                JMP     TRAP_ENTRY
;
;
;  - PC is on stack, flags are above them (pushed above).
;  - Pre-trap "A" has been saved in REG_A.
;  - "A" contains processor state corresponding to interrupt.
;  - No page 0 registers are yet available, as they may be in use by the user
TRAP_ENTRY:                             ;31 cycles from trap to here
        LDST    #0                      ; 6 DISABLE INTERRUPTS
        MOV     A,REG_STATE             ;10 SAVE PROCESSOR STATE/INT NUMBER
        RL      A                       ; 8 MAKE WORD INDEX
        XCHB    A                       ;10 B = INDEX; A = INTERRUPTED B
        MOV     A,REG_B                 ;10
        MOV     USER_VECT-10(B),A       ;12 MSB of RAM vector
        PUSH    A                       ; 9
        MOV     USER_VECT-9(B),A        ;12 LSB
        PUSH    A                       ; 9
        MOV     REG_B,A                 ;10 Restore A and B
        XCHB    A                       ;10
        MOV     REG_A,A                 ;10
        RTI                             ;12 "jump" thru vector, restore flags
;                                       ;-----
;                                       ;153 cycles added to trap
;
;===========================================================================
;
;  Vector interrupts thru RAM.  State numbers must match TRG370.C
;
UNUSED1_H:      MOV     A,REG_A                 ;10
                MOV     #21,A                   ; 6 processor state
                JMP     INT_ENTRY               ; 7 join common code
;                                               ;---
;                                               ;23 cycles
UNUSED2_H:      MOV     A,REG_A
                MOV     #22,A
                JMP     INT_ENTRY
;
UNUSED3_H:      MOV     A,REG_A
                MOV     #23,A
                JMP     INT_ENTRY
;
UNUSED4_H:      MOV     A,REG_A
                MOV     #24,A
                JMP     INT_ENTRY
;
UNUSED5_H:      MOV     A,REG_A
                MOV     #25,A
                JMP     INT_ENTRY
;
UNUSED6_H:      MOV     A,REG_A
                MOV     #26,A
                JMP     INT_ENTRY
;
ADC_H:          MOV     A,REG_A
                MOV     #27,A
                JMP     INT_ENTRY
;
TIMER2_H:       MOV     A,REG_A
                MOV     #28,A
                JMP     INT_ENTRY
;
SCITX_H:        MOV     A,REG_A
                MOV     #29,A
                JMP     INT_ENTRY
;
SCIRX_H:        MOV     A,REG_A
                MOV     #30,A
                JMP     INT_ENTRY
;
TIMER1_H:       MOV     A,REG_A
                MOV     #31,A
                JMP     INT_ENTRY
;
SPI_H:          MOV     A,REG_A
                MOV     #32,A
                JMP     INT_ENTRY
;
INT3_H:         MOV     A,REG_A
                MOV     #33,A
                JMP     INT_ENTRY
;
INT2_H:         MOV     A,REG_A
                MOV     #34,A
                JMP     INT_ENTRY
;
INT1_H:         MOV     A,REG_A
                MOV     #35,A
                JMP     INT_ENTRY
;
;
;  - Flags and PC are on stack, and flags are cleared.
;  - Pre-interrupt "A" has been saved in REG_A
;  - "A" contains processor state corresponding to interrupt.
;  - No page 0 registers are yet available, as they may be in use by the user
INT_ENTRY:                              ;23 cycles from int to here
        MOV     A,REG_STATE             ;10 SAVE PROCESSOR STATE/INT NUMBER
        RL      A                       ; 8 MAKE WORD INDEX
        XCHB    A                       ;10 B = INDEX; A = INTERRUPTED B
        MOV     A,REG_B                 ;10
        MOV     USER_VECT-10(B),A       ;12 MSB of RAM vector
        PUSH    A                       ; 9
        MOV     USER_VECT-9(B),A        ;12 LSB
        PUSH    A                       ; 9
        MOV     REG_B,A                 ;10 Restore A and B
        XCHB    A                       ;10
        MOV     REG_A,A                 ;10
        RTS                             ; 9 "jump" thru RAM vector
;                                       ;-----
;                                       ;142 cycles added to interrupt
;
;$PACT$ if PACT is used, then
;  (These are AFTER INT_ENTRY lest they exceed the range of JMP -
;  SA370 does not validate this!)
;;P_SCITX_H:    MOV     A,REG_A
;;              MOV     #36,A    
;;              JMP     INT_ENTRY
;;
;;P_SCIRX_H:    MOV     A,REG_A
;;              MOV     #37,A    
;;              JMP     INT_ENTRY
;;
;;P_CD0_H:      MOV     A,REG_A
;;              MOV     #38,A    
;;              JMP     INT_ENTRY
;;
;;P_CD1_H:      MOV     A,REG_A
;;              MOV     #39,A    
;;              JMP     INT_ENTRY
;;
;;P_CD2_H:      MOV     A,REG_A
;;              MOV     #40,A    
;;              JMP     INT_ENTRY
;;
;;P_CD3_H:      MOV     A,REG_A
;;              MOV     #41,A    
;;              JMP     INT_ENTRY
;;
;;P_CD4_H:      MOV     A,REG_A
;;              MOV     #42,A    
;;              JMP     INT_ENTRY
;;
;;P_CD5_H:      MOV     A,REG_A
;;              MOV     #43,A    
;;              JMP     INT_ENTRY
;;
;;P_CD6_H:      MOV     A,REG_A
;;              MOV     #44,A    
;;              JMP     INT_ENTRY
;;
;;P_CD7_H:      MOV     A,REG_A
;;              MOV     #45,A    
;;              JMP     INT_ENTRY
;;
;;P_BUF_H:      MOV     A,REG_A
;;              MOV     #46,A    
;;              JMP     INT_ENTRY
;;
;;P_CP6_H:      MOV     A,REG_A
;;              MOV     #47,A    
;;              JMP     INT_ENTRY
;;
;;P_CP5_H:      MOV     A,REG_A
;;              MOV     #48,A    
;;              JMP     INT_ENTRY
;;
;;P_CP4_H:      MOV     A,REG_A
;;              MOV     #49,A    
;;              JMPL    INT_ENTRY
;;
;;P_CP3_H:      MOV     A,REG_A
;;              MOV     #50,A    
;;              JMPL    INT_ENTRY
;;
;;P_CP2_H:      MOV     A,REG_A
;;              MOV     #51,A    
;;              JMPL    INT_ENTRY
;;
;;P_CP1_H:      MOV     A,REG_A
;;              MOV     #52,A    
;;              JMPL    INT_ENTRY
;;
;;P_TOV_H:      MOV     A,REG_A
;;              MOV     #53,A    
;;              JMPL    INT_ENTRY
;;
;===========================================================================
        .TEXT   HARD_VECT
;
;$PACT$ if PACT is used, then
;;	.WORD	P_SCITX_H		;7F9C
;;	.WORD	P_SCIRX_H		;7F9E
;;	.WORD	P_CD0_H			;7FA0
;;	.WORD	P_CD1_H			;7FA2
;;	.WORD	P_CD2_H			;7FA4
;;	.WORD	P_CD3_H			;7FA6
;;	.WORD	P_CD4_H			;7FA8
;;	.WORD	P_CD5_H			;7FAA
;;	.WORD	P_CD6_H			;7FAC
;;	.WORD	P_CD7_H			;7FAE
;;	.WORD	P_BUF_H			;7FB0
;;	.WORD	P_CP6_H			;7FB2
;;	.WORD	P_CP5_H			;7FB4
;;	.WORD	P_CP4_H			;7FB6
;;	.WORD	P_CP3_H			;7FB8
;;	.WORD	P_CP2_H			;7FBA
;;	.WORD	P_CP1_H			;7FBC
;;	.WORD	P_TOV_H			;7FBE
;
;  Using one of the TRAP instructions for breakpoint gives a single-byte
;  breakpoint instruction.  However, if all 16 TRAPs are required by
;  user software, then CALL may be used for breakpoint:  Change
;  BREAKPOINT_ENTRY to TRAP_15 below, and change the definition of the
;  instruction to be used for breakpoint at label B0 in TSTG from
;  "TRAP 15" to "Call BREAKPOINT_ENTRY"
;
;  Note that any other TRAP instruction may be used, if desired.
TRAP_V: .WORD   BREAKPOINT_ENTRY        ;7FC0  or TRAP15_H
        .WORD   TRAP14_H                ;7FC2
        .WORD   TRAP13_H                ;7FC4
        .WORD   TRAP12_H                ;7FC6
        .WORD   TRAP11_H                ;7FC8
        .WORD   TRAP10_H                ;7FCA
        .WORD   TRAP9_H                 ;7FCC
        .WORD   TRAP8_H                 ;7FCE
        .WORD   TRAP7_H                 ;7FD0
        .WORD   TRAP6_H                 ;7FD2
        .WORD   TRAP5_H                 ;7FD4
        .WORD   TRAP4_H                 ;7FD6
        .WORD   TRAP3_H                 ;7FD8
        .WORD   TRAP2_H                 ;7FDA
        .WORD   TRAP1_H                 ;7FDC
        .WORD   TRAP0_H                 ;7FDE

NTRAP   .EQU    $-TRAP_V
;
;  Unused vectors
INT_V:  .WORD   UNUSED1_H               ;7FE0
        .WORD   UNUSED2_H               ;7FE2
        .WORD   UNUSED3_H               ;7FE4
        .WORD   UNUSED4_H               ;7FE6
        .WORD   UNUSED5_H               ;7FE8
        .WORD   UNUSED6_H               ;7FEA
;
;  Interrupts
        .WORD   ADC_H                   ;7FEC
        .WORD   TIMER2_H                ;7FEE
        .WORD   SCITX_H                 ;7FF0
        .WORD   SCIRX_H                 ;7FF2
        .WORD   TIMER1_H                ;7FF4
        .WORD   SPI_H                   ;7FF6
        .WORD   INT3_H                  ;7FF8
        .WORD   INT2_H                  ;7FFA
        .WORD   INT1_H                  ;7FFC
        .WORD   RESET                   ;7FFE
NVEC    .EQU    $-INT_V
;
;       .end    RESET
