;  Z8 Debug monitor for use with NoICEZ8
;
;  Copyright (c) 1992-1994 by John Hartman
;
;  Modification History:
;	14-Oct-92 by John Hartman:  created from MON740
;	29-DEC-92 JCO   Conversion to Z8
;	16-Feb-93 JLH	Update to latest NOICE features
;	23-Mar-93 JLH	Correct missing code in memory paging
;	24-Aug-93 JLH   Bad constant for COMBUF length compare
;	10-Jan-94 JLH   Add suggested code for interrupt enable (V1.2)
;	 2-Feb-94 JLH 	Cleanup, verify interrupts
;
;============================================================================
;
;  To customize for a given target, you must change code in the
;  hardware equates, the string TSTG, and the routines RESET and REWDT.
;  You may or may not need to change GETCHAR, PUTCHAR, depending on
;  how peculiar your UART is.
;
;  This file has been assembled with the ASM800 assembler which Zilog
;  includes with various Z8 evaluation boards.  Note that symbols are
;  CASE SENSITIVE:  in particular, "a", "b", etc are NOT the same as
;  "A", "B", etc.  The former represent "registers" with one byte
;  addresses.  The latter represent registers in the monitor's
;  regiater set (addresses 0 to 15)
;
;  To add mapped memory support:
;	1) Define map port MAPREG here
;	2) Define map port RAM image MAPIMG here if MAPREG is write only
;	3) Search for and modify MAPREG, MAPIMG, and REG_PAGE usage below
;       4) In TSTG below edit "LOW AND HIGH LIMIT OF MAPPED MEM"
;	   to appropriate range
;============================================================================
.z800
;
;  User code in external RAM.  Interrupts are re-vectored via "vectors"
;  beginning here.  Thus, you only need to change the ORG or link address
;  of your application program between debug and ROM versions.
;  The value of "USER_CODE" will, of course, depend on where your target
;  has RAM.
USER_CODE  .EQU 0A000H		;START OF USER INTERRUPTS/CODE
;
;  User's initial RP
USER_RP	   .EQU 070H
;
;  The ERF, or Expanded Register File, is one more example of how much
;  schlitz Zilog engineers can fit into one address.  (These people must
;  build ships in bottles as a hobby).  The symbol below controls access
;  to ERFs using NoICE.  Refer to the code READ_BYTE and WRITE_BYTE for
;  details.
;
;  - If your Z8 has no ERFs (Z80C00/10/20 etc), define ERF_FILES as 0
;  - If your Z8 has 1 file of ERFs (Z86C06/09/19 etc), define ERF_FILES as 1
;  - If your Z8 has n files of ERFs (Z86C95 etc), define ERF_FILES as n
ERF_FILES  .EQU 4		;target is Z8 with DSP

;
;  Monitor's register/scratch RAM location.  The monitor uses 16 bytes
;  of registers beginning at this address.  They may NOT be changed
;  during user program execution, or unusual things may happen.
MONRP	   .EQU	40H
;
;  Location of monitor code (0, except during debug under another monitor)
NOICE_CODE .EQU 0000H      	;START OF NOICE INTERRUPTS/CODE
;
;  Tuck the monitor RAM at the top of external memory, out of the way.
;  See below for the amount of RAM reqyured.
RAM_START  .EQU 0F800H          ;START OF MONITOR RAM
;
;==========================================================================
;include z8 i/o and register equates
	.LIST	OFF
	.include	Z8EQU.INC
	.LIST	ON

;============================================================================
;  Monitor RAM definitions
		.ABS
		.ORG	RAM_START
;
;  Initial user stack.
;  (Size and location is user option,  If desired, user stack may be in
;  on-chip memory)
		.BLOCK	64
INITSTACK:        
;
;  Monitor stack
;  (Calculated use is at most ?? bytes.  Leave plenty of spare)
;  While this could be on-chip RAM, moving the monitor stack off chip
;  frees the maximal amount of on-chip RAM for application use.
;  Note that using internal stack would require changing the initialization
;  of the register P01M below.
		.BLOCK	32
MONSTACK:        
;
;  Target registers:  order must match that in TRGZ8.C
TASK_REGS:
 MREG_STATE:	.BLOCK  1
 MREG_PAGE:	.BLOCK	1
 MREG_IMR:	.BLOCK	1
 MREG_FLAGS:	.BLOCK	1
 MREG_PCL:	.BLOCK	1
 MREG_PCH:	.BLOCK	1
 MREG_RP:	.BLOCK	1
 MREG_WREGS:	.BLOCK	16
 MREG_SPL:	.BLOCK	1
 MREG_SPH:	.BLOCK	1
 ;
TASK_REGS_SIZE	.EQU	$-TASK_REGS
; Caution:  don't put parenthesis around the above:
; The parenthesis in ($-TASK_REGS) are "remembered", such that 
; LD BC,TASK_REGS_SIZE is the same as LD BC,(TASK_REGS_SIZE)
; It is OK to use parenthesis around the difference if the difference 
; is to be divided - just not around the entire expression!!!!!
;
;  Communications buffer
;  (Must be at least as long as TASK_REG_SIZE.  At least 19 bytes recommended.
;  Larger values may improve speed of NoICE memory move commands.)
COMBUF_SIZE	.EQU	64			;DATA SIZE FOR COMM BUFFER
COMBUF:		.BLOCK	2+COMBUF_SIZE+1		;BUFFER ALSO HAS FN, LEN, AND CHECK
;
;===========================================================================
;  Interrupt vectors
	.ORG	NOICE_CODE
        .word   IRQ0		;Jump through our skip-to-user
        .word   IRQ1
        .word   IRQ2
        .word   IRQ3
        .word   IRQ4
        .word   IRQ5
;
;----------------------------------------------------------
;
;   MN_PowerOn  -  Initialize the Z8 MCU
;
;----------------------------------------------------------

	.org NOICE_CODE+0CH    	;reset start address

MN_PowerOn:
;
;  See Zilog Application Brief on Z86XX Interrupt Request Register for
;  details of the following obscure sequence.
        DI                      ;initialize interrupt flip flops
        CLR     IMR             ;make sure nobody's enabled
        CLR     IRQ             ;kill any spurrious interrupts
        EI                      ;enable writes to IRQ (so we can use UART)
        DI                      ;disable interrupts
;
;  Initialize ports 2 and 3
        CLR     P2M             ;Port 2 all outputs
        LD      P3M,#01000001B  ;Port 3 UART/ no parity.  No
				;handshaking.  Port 2 pullups active
;
;  Enable the address/data bus, 16 address lines
	LD      P01M,#11100011B ;Port 1 AD7-AD0, Port 2 A15-A8, EXTENDED
;                                ; memory timing, EXTERNAL stack.
	LD	SPH,#^HB(MONSTACK)
	LD	SPL,#^LB(MONSTACK)
;
;  Set baud rate to 9600 bps
        LD      PRE0,#05h       ;Modulo-n, prescaler = 1
        LD      T0,#16          ;20M/2/4/16/9600=16
        LD      TMR,#03h        ;start brg running
;
;  Initialize user registers
	SRP	#MONRP
	LD	H,#^HB(REG_INIT)	;ADDRESS OF INITIALIZATION TABLE
	LD	L,#^LB(REG_INIT)
	LD	D,#^HB(TASK_REGS)	;POINTER TO REG IMAGES
	LD	E,#^LB(TASK_REGS)
	LD	B,#TASK_REGS_SIZE
INIT_LP:
	LDE	A,@HL
	LDE	@DE,A
	INCW	DE
	INCW	HL
	DJNZ	B,INIT_LP
;
;  Initialize user's interrupt vectors to default values.
;  (Should be overlaid by user's code, but catch the shlitz)
        LD      R0,#^HB(INT_ENTRY)	;DEFAULT HANDLER
        LD      R1,#^LB(INT_ENTRY)
	LD	H,#^HB(USER_CODE)	;ADDRESS OF USER CODE SPACE
	LD	L,#^LB(USER_CODE)
        LD	B,#6			;NUMBER OF VECTOR: IRQ0 TO IRQ5
INITV:	LDE	@HL,R0			;SET VECTOR WITH DEFAULT HANDLER
	INCW	HL
        LDE	@HL,R1
        INCW	HL
        DJNZ	B,INITV        
;
;  Stall, if necessary (such as external UART which may remain in reset
;  longer than the micro.  In this case, init UAR AFTER this delay)
;;	LD	H,#0
;;LLPP	INCW	HL		;LONG DELAY IN CASE EXTERNAL UART IN RESET
;;	JR	NZ,LLPP
;
	JP	RETURN_REGS		;DUMP REGS, ENTER MONITOR
;
REG_INIT:
	.BYTE	0			;PROCESSOR STATE (0=RESET)
	.BYTE	0			;PAGE NUMBER (UNUSED)
        .BYTE   0	  		;IMR
        .BYTE   0                       ;FLAGS
	.BYTE	^LB(USER_CODE+0CH)	;INITIAL PCL
	.BYTE	^HB(USER_CODE+0CH)	;INITIAL PCH
	.BYTE	USER_RP			;INITIAL RP
	.BYTE	0,0,0,0,0,0,0,0		;R0-R7
	.BYTE	0,0,0,0,0,0,0,0		;R8-R15
	.BYTE	^LB(INITSTACK)		;INITIAL VALUE FOR USER STACK
	.BYTE	^HB(INITSTACK)

;===========================================================================
;  Get a character to A
;
;  Return A=char, CY=0 if data received
;	  CY=1 if timeout (0.5 seconds or longer, if used)
;
GETCHAR:
       	PUSH	D
       	PUSH	E
        LD	D,#0FFH			;LONG TIMEOUT
GC10:	DECW	DE
	LD	A,D
        OR	A,E
        JR	Z,GC90			;EXIT IF TIMEOUT
	TM	IRQ,#00001000B		;TEST FOR RXD
	JR	Z,GC10
	LD	A,SIO
	AND	IRQ,#11110111B
	RCF
	POP	E
        POP	D
        RET
;
;  Timeout:  return CY=1
GC90:	SCF				;CY=1
	POP	E
	POP	D
        RET 
;
;===========================================================================
;  Output character in A
;
PUTCHAR:
	LD	SIO,A
PC11:	TCM	IRQ,#00010000B
	JR	NZ,PC11
	AND	IRQ,#11101111B
	RET
        
;===========================================================================
;
;  Target Status:  FN, len
;
;  Reply describes target:
TSTG:	.BYTE	2			;2: PROCESSOR TYPE = Z8
	.BYTE	COMBUF_SIZE		;3: SIZE OF COMMUNICATIONS BUFFER
	.BYTE	0			;4: NO OPTIONS
        .BYTE	0,0,0,0			;5-8: LOW AND HIGH LIMIT OF MAPPED MEM (NONE)
        .BYTE	B1-B0			;9 BREAKPOINT INSTR LENGTH
;
;  The following two byte breakpoint instruction allows greatest
;  flexibility in placing breakpoints, but requires that two bytes of
;  on-chip RAM remain set to the address of the breakpoints handler.
;  (The monitor sets these bytes before running the user program.)
;  As an alternative, one could use a three byte absolute CALL to
;  BREAKPOINT_ENTRY.  This would also require changing the code at
;  BREAKPOINT_ENTRY to decrement the PC by 3, rather than the current 2.
B0:	CALL	@MONRP			;10+ BREKAPOINT INSTRUCTION
B1:     .BYTE	'NoICE Z8 monitor V1.4',0	;DESCRIPTION, ZERO

TSTG_SIZE	.EQU	$-TSTG		;SIZE OF STRING

;===========================================================================
;  Read the byte at DE from register, ERF, or external memory
;  0000-00ff is internal register
;  0100-0nff is ERF, where n = ERF_FILES
;  0m00-ffff is external memory (code/data space)
READ_BYTE:
	OR	D,D			;IF D=0, GET FROM REGS
	JR	Z,RB_REG
	CP	D,#ERF_FILES+1		;IF <n GET FROM ERFS
	JR	C,RB_ERF
;
;  External memory
	LDE	A,@DE			;BYTE OF EXTERNAL MEMORY
	RET
;        
;  Read from register 0-FF
RB_REG:	LD	A,@E			;BYTE OF REGISTER
	RET
;
;  Read from extended register file
;  Addresses 0WXY map to ERF file "W", group "X", register "Y"
RB_ERF:	PUSH	D
        PUSH	E
;
;  Processors with only one file of ERFs can skip this code
  .if (ERF_FILES > 1)
;
;  Select ERF file "W"
;  In all ERF files, group F register 8 bit 6 and 5 are ERF-file-selectors
	DEC	D			;01 to 0n -> 00 to n-1
	SWAP	D			;00 to m0
	RL	D			;00 to 11 (bits 6 and 5)
	PUSH	RP
	LD	RP,#(MONRP+0FH)		;SELECT CONTROL REGISTERS (GROUP F)
	PUSH	8			;SAVE PREVIOUS ERF FILE SELECTOR
	AND	8,#10011111B		;ERF FILE SELECT
	OR	8,D
  .endif
;
;  Read extended register @L (group,reg) from current ERF file
	LD	D,E
	SWAP	D
	AND	D,#00001111B		;MASK OFF GROUP
	AND	E,#00001111B		;MASK OFF REGISTER WITHIN GROUP
	ADD	D,#MONRP
	LD	RP,D			;LOAD REGISTER POINTER
	LD	A,@E			;LOAD BYTE
;
;  Processors with only one file of ERFs can skip this code
  .if (ERF_FILES > 1)
	LD	RP,#(MONRP+0FH)
	POP	8
	POP	RP
  .endif
;
	POP	E
        POP	D
	RET
	
;===========================================================================
;  Write A to the byte at DE from register, ERF, or external memory
;  0000-00ff is internal register
;  0100-0nff is ERF, where n = ERF_FILES
;  0m00-ffff is external memory (code/data space)
WRITE_BYTE:
	OR	D,D			;IF D=0, GET FROM REGS
	JR	Z,WB_REG
	CP	D,#ERF_FILES+1		;IF <n GET FROM ERFS
	JR	C,WB_ERF
;
;  External memory
	LDE	@DE,A			;BYTE OF EXTERNAL MEMORY
	RET
;        
;  Read from register 0-FF
WB_REG:	LD	@E,A			;BYTE OF REGISTER
	RET
;
;  Read from extended register
;  Addresses 0WXY map to ERF file "W", group "X", register "Y"
WB_ERF:	PUSH	D
	PUSH	E
;
;  Processors with only one file of ERFs can skip this code
  .if (ERF_FILES > 1)
;
;  Select ERF file "W"
;  In all ERF files, group F register 8 bit 6 and 5 are ERF-file-selectors
	DEC	D			;01 to 0n -> 00 to n-1
	SWAP	D			;00 to m0
	RL	D			;00 to 11 (bits 6 and 5)
	PUSH	RP
	LD	RP,#(MONRP+0FH)		;SELECT CONTROL REGISTERS (GROUP F)
	PUSH	8			;SAVE PREVIOUS ERF FILE SELECTOR
	AND	8,#10011111B		;ERF FILE SELECT
	OR	8,D
  .endif
;
;  Write extended register @L (group,reg) from current ERF file
	LD	D,E
	SWAP	D
	AND	D,#00001111B		;MASK OFF GROUP
	AND	E,#00001111B		;MASK OFF REGISTER WITHIN GROUP
	ADD	D,#MONRP
	LD	RP,D			;LOAD REGISTER POINTER
	LD	A,@E			;LOAD BYTE
;
;  Processors with only one file of ERFs can skip this code
  .if (ERF_FILES > 1)
	LD	RP,#(MONRP+0FH)
	POP	8
	POP	RP
  .endif
;
	POP	E
        POP	D
	RET
;
;===========================================================================
;  HARDWARE PLATFORM INDEPENDENT EQUATES AND CODE
;
;  Communications function codes.
FN_GET_STATUS	.EQU	0FFH	;reply with device info
FN_READ_MEM	.EQU	0FEH	;reply with data
FN_WRITE_MEM	.EQU	0FDH	;reply with status (+/-)
FN_READ_REGS	.EQU	0FCH	;reply with registers
FN_WRITE_REGS	.EQU	0FBH	;reply with status
FN_RUN_TARGET	.EQU	0FAH	;reply (delayed) with registers
FN_SET_BYTES	.EQU	0F9H	;reply with data (truncate if error)
FN_IN		.EQU	0F8H	;input from port
FN_OUT		.EQU	0F7H	;output to port
;
FN_MIN		.EQU	0F7H	;MINIMUM RECOGNIZED FUNCTION CODE
FN_ERROR	.EQU	0F0H	;error reply to unknown op-code
;
;===========================================================================
;
;  Enter here via interrupt:  PC, FLAGS are stacked.
;  Interrupts are disabled
;
;  Monitor register "b" has target state value corresponding to interrupt
;
INT_ENTRY:
        DI			;DISABLE INTERRUPTS (SHOULD ALREADY BE OFF)
	POP	c		;save user's flags for later
;
	LD	a,IMR		;GET INTERRUPT MASK REG
        OR	a,#80H		;SET MASTER BIT: IT WAS SET BEFORE INT OCCURRED
	PUSH	a		;SAVE INTERRUPT ENABLE
	JR	BE10
;
;===========================================================================
;
;  Enter here via JSR nn for breakpoint:  PC is stacked.
;  Value is CALL + 2 (or +3 if absolute CALL is used)
;  Interrupt status is not changed from user program
BREAKPOINT_ENTRY:
;
;  Interrupts may be on
	PUSH	IMR		;SAVE INTERRUPT ENABLE
        DI			;DISABLE INTERRUPTS
        LD	b,#1		;state = 1 = breakpoint
	LD	c,FLAGS		;save user's flags for later
;
;  Save registers in reg block for return to master
BE10:	LD	a,RP		;save current RP in monitor's A
	SRP	#MONRP		;change to monitor reg group
;
	LD	D,#^HB(TASK_REGS)
	LD	E,#^LB(TASK_REGS)
	LDE	@DE,B		;SAVE STATE
	INCW	DE
;
	LD	H,#0
	LDE	@DE,H
	INCW	DE		;SET MEMORY PAGE TO ZERO (OR SAVE CURRENT VALUE)
;
	POP	H		;GET SAVED IMR
	LDE	@DE,H
	INCW	DE
;
	LDE	@DE,C		;SAVE FLAG REGISTER
	INCW	DE
;
	POP	H		;get program counter
	POP	L
;
;  If entry is breakpoint, then back up PC to point at instruction
	CP	B,#1
        JR	NZ,BE20		;JIF NOT BREAKPOINT
	DECW	HL		;back up to point at breakpoint CALL @MONRP
	DECW	HL
;
BE20:	LDE	@DE,L		;SAVE PROGRAM COUNTER
	INCW	DE
	LDE	@DE,H
	INCW	DE
;
	LDE	@DE,A		;SAVE USER RP
	INCW	DE
;
	AND	A,#11110000B
	LD	B,#10H		;number of registers = 16
EMLP:	LD	C,@A
	LDE	@DE,C
	INCW	DE
	INC	A
	DJNZ	B,EMLP
;
	LD	B,SPL		;SAVE USER STACK POINTER
	LDE	@DE,B
	INCW	DE
	LD	B,SPH
	LDE	@DE,B
	INCW	DE
;
;  If on-chip stack is used, don't change stack
        LD	SPL,#^LB(MONSTACK)	;AND USE OURS INSTEAD
	LD	SPH,#^HB(MONSTACK)
;
;  Return registers to master
	JP	RETURN_REGS
;
;===========================================================================
;  Main loop:  wait for command frame from master
MAIN:	DI
	LD	SPH,#^HB(MONSTACK)	;CLEAN STACK IS HAPPY STACK
	LD	SPL,#^LB(MONSTACK)
	LD	H,#^HB(COMBUF)		;BUILD MESSAGE HERE
	LD	L,#^LB(COMBUF)
;
;  First byte is a function code
	CALL	GETCHAR			;GET A FUNCTION
        JR	C,MAIN			;JIF TIMEOUT: RESYNC
	CP	A,#FN_MIN
        JR	C,MAIN			;JIF BELOW MIN: ILLEGAL FUNCTION
        LDE	@HL,A			;SAVE FUNCTION CODE
        INCW	HL
;
;  Second byte is data byte count (may be zero)
	CALL	GETCHAR			;GET A LENGTH BYTE
        JR	C,MAIN			;JIF TIMEOUT: RESYNC
	CP	A,#COMBUF_SIZE+1
        JR	NC,MAIN			;JIF TOO LONG: ILLEGAL LENGTH
        LDE	@HL,A			;SAVE LENGTH
        INCW	HL
        OR	A,A
        JR	Z,MA80			;SKIP DATA LOOP IF LENGTH = 0
;
;  Loop for data
        LD	B,A			;SAVE LENGTH FOR LOOP
MA10:	CALL	GETCHAR			;GET A DATA BYTE
        JR	C,MAIN			;JIF TIMEOUT: RESYNC
        LDE	@HL,A			;SAVE DATA BYTE
        INCW	HL
        DJNZ	B,MA10
;
;  Get the checksum
MA80:	CALL	GETCHAR			;GET THE CHECKSUM
        JR	C,MAIN			;JIF TIMEOUT: RESYNC
        LD	C,A			;SAVE CHECKSUM
;
;  Compare received checksum to that calculated on received buffer
;  (Sum should be 0)
        CALL	CHECKSUM
	ADD	A,C
        JR	NZ,MAIN			;JIF BAD CHECKSUM
;
;  Process the message.
	LD	H,#^HB(COMBUF)
	LD	L,#^LB(COMBUF)
	LDE	A,@HL			;GET THE FUNCTION CODE
        INCW	HL
        LDE	B,@HL			;GET THE LENGTH
        INCW	HL			;HL POINTS TO MESSAGE DATA
        CP	A,#FN_GET_STATUS
        JP	Z,TARGET_STATUS
        CP	A,#FN_READ_MEM
        JP	Z,READ_MEM
        CP	A,#FN_WRITE_MEM
        JP	Z,WRITE_MEM
        CP	A,#FN_READ_REGS
        JP	Z,READ_REGS
        CP	A,#FN_WRITE_REGS
        JP	Z,WRITE_REGS
        CP	A,#FN_RUN_TARGET
        JP	Z,RUN_TARGET
        CP	A,#FN_SET_BYTES
        JP	Z,SET_BYTES
        CP	A,#FN_IN
        JP	Z,IN_PORT
        CP	A,#FN_OUT
        JP	Z,OUT_PORT
;
;  Error: unknown function.  Complain
	LD	A,#FN_ERROR
        LDE	@HL,A		;SET FUNCTION AS "ERROR"
	LD	A,#1
        JP	SEND_STATUS	;VALUE IS "ERROR"
;
;===========================================================================
;
;  Enter with A=function, B=length, HL points at first data byte
;
TARGET_STATUS: 
;
	LD	D,#^HB(TSTG)		;DATA FOR REPLY
	LD	E,#^LB(TSTG)		;DATA FOR REPLY
        LD	B,#TSTG_SIZE		;LENGTH OF REPLY
        DECW	HL
        LDE	@HL,B			;SET SIZE IN REPLY BUFFER
	INCW	HL
TAR_LP:	LDE	A,@DE			;TRANSFER STATUS TO COMM BUFFER
	LDE	@HL,A
	INCW	DE
	INCW	HL
	DJNZ	B,TAR_LP
;
;  Compute checksum on buffer, and send to master, then return
	JP	SEND
;
;===========================================================================
;
;  Read Memory:  FN, len, page, Alo, Ahi, Nbytes
;
;  Enter with A=function, B=length, HL points at first data byte
;
READ_MEM: 
;
;  Set map
	INCW	HL			;SKIP PAGE
;
;  Set address
	LDE	E,@HL
	INCW	HL
	LDE	D,@HL
	INCW	HL

	LDE	B,@HL			;NUMBER OF BYTES TO GET
;
;  Prepare return buffer: FN (unchanged), LEN, DATA
        DECW    HL
        DECW    HL
        DECW    HL
        DECW    HL
	LDE	@HL,B			;RETURN LENGTH = REQUESTED DATA
        INCW	HL
        OR	B,B
        JR	Z,GLP90			;JIF NO BYTES TO GET
;
;  Read the requested bytes from local memory
GLP:	CALL	READ_BYTE		;GET BYTE @DE TO A
	LDE	@HL,A			;STORE TO RETURN BUFFER
        INCW	DE
	INCW	HL
	DJNZ	B,GLP
;
;  Compute checksum on buffer, and send to master, then return
GLP90:	JP	SEND

;===========================================================================
;
;  Write Memory:  FN, len, page, Alo, Ahi, (len-3 bytes of Data)
;
;  Enter with A=function, B=length, HL points at first data byte
;
WRITE_MEM: 
;
;  Set map
	INCW	HL			;JUST SKIP PAGE FOR NOW
;
;  Get Address
	LDE	E,@HL			;POINTER TO DESTINATION
	INCW	HL
	LDE	D,@HL			;POINTER TO DESTINATION
	INCW	HL
        SUB	B,#3			;LESS PAGE, ADDRLO, ADDRHI
	JR	Z,PLP50			;EXIT IF NONE REQUESTED
;
;  Write the specified bytes to local memory
	PUSH	B
	PUSH	H
	PUSH	L
	PUSH	D
	PUSH	E
;
WLP:	LDE	A,@HL		;GET DATA FROM HOST BUFFER
	CALL	WRITE_BYTE	;WRITE TO REG/ERF/MEMORY
	INCW	DE
	INCW	HL
	DJNZ	B,WLP
;
;  Compare to see if the write worked
	POP	E
	POP	D
	POP	L
	POP	H
	POP	B
;
;  Compare the specified bytes to external memory
PLP20:	CALL	READ_BYTE		;GET BYTE FROM MEM/REG TO A
	LDE	C,@HL
	CP	A,C
	JR	NZ,PLP80		;JIF WRITE FAILED
        INCW	DE
        INCW	HL
        DJNZ	B,PLP20
;
;  Write succeeded:  return status = 0
PLP50:	XOR	A,A			;RETURN STATUS = 0
	JR	PLP90
;
;  Write failed:  return status = 1
PLP80:	LD	A,#1
;
;  Return OK status
PLP90:	JP	SEND_STATUS
;
;===========================================================================
;
;  Read registers:  FN, len
;
;  Enter with A=function, B=length, HL points at first data byte
;
READ_REGS: 
	LD	B,#FN_READ_REGS		;FUNCTION IS "READ REG REPLY"
	JR	RRCOM
;
;  Enter here from "RUN" and "STEP" to return task registers
RETURN_REGS:
	LD	B,#FN_RUN_TARGET	;FUNCTION IS "RUN TARGET REPLY"
;
RRCOM:	LD	H,#^HB(TASK_REGS)	;REGISTER LIVE HERE
	LD	L,#^LB(TASK_REGS)	;REGISTER LIVE HERE
	LD	A,#TASK_REGS_SIZE	;NUMBER OF BYTES
;
;  Prepare return buffer: FN, LEN, DATA
        LD	D,#^HB(COMBUF)		;POINTER TO FN, LEN, DATA
        LD	E,#^LB(COMBUF)		;POINTER TO FN, LEN, DATA
	LDE	@DE,B			;SAVE FN
	INCW	DE
	LDE	@DE,A			;SAVE DATA LENGTH
        INCW	DE
;
;  Copy the registers
	LD	B,A
GRLP:	LDE	A,@HL			;GET BYTE TO A
	LDE	@DE,A			;STORE TO RETURN BUFFER
	INCW	HL
        INCW	DE
	DJNZ	B,GRLP
;
;  Compute checksum on buffer, and send to master, then return
	JP	SEND

;===========================================================================
;
;  Write registers:  FN, len, (register image)
;
;  Enter with A=function, B=length, HL points at first data byte
;
WRITE_REGS: 
;
        OR	B,B
        JR	Z,WRR80			;JIF NO REGISTERS
;
;  Copy the registers
	LD	D,#^HB(TASK_REGS)	;OUR REGISTERS LIVE HERE
	LD	E,#^LB(TASK_REGS)	;OUR REGISTERS LIVE HERE
WRRLP:	LDE	A,@HL			;GET BYTE TO A
	LDE	@DE,A			;STORE TO REGISTER IMAGE
	INCW	HL
        INCW	DE
	DJNZ	B,WRRLP
;
;  Return OK status
WRR80:	XOR	A,A
	JP	SEND_STATUS

;===========================================================================
;
;  Run Target:  FN, len
;
;  Enter with A=function, B=length, HL points at first data byte
;
RUN_TARGET: 
;
;  Switch to user stack (skip this if using on-chip stack, if we didn't
;  switch to a monitor stack upon entry)
	LD	H,#^HB(MREG_SPL)
	LD	L,#^LB(MREG_SPL)
	LDE	E,@HL		;USER STACK POINTER
	INCW	HL
	LDE	D,@HL
	LD	SPL,E
	LD	SPH,D
;
;  Load rest of user's registers
	LD	H,#^HB(TASK_REGS+2)  ;SKIP STATE AND PAGE
	LD	L,#^LB(TASK_REGS+2)
;
	LDE	A,@HL		;IMR, SAVE FOR LATER
	INCW	HL
;
	LDE	C,@HL		;FLAGS, SAVE FOR LATER
	INCW	HL
;
	LDE	E,@HL		;PROGRAM COUNTER
	INCW	HL
	LDE	D,@HL
	INCW	HL
	PUSH	E		;PC TO USER STACK FOR RET LATER
	PUSH	D
        PUSH	C		;SAVE FLAGS
;
;  Restore working registers
;  ("A" HAS IMR, USED BELOW - DON'T TRASH IT!!!)
	LDE	B,@HL		;USER RP
	INCW	HL
	LD	C,B
	AND	C,#11110000B	;FIRST WORKING REGISTER
	CP	C,#MONRP
	JR	Z,RNSP		;DON'T WONK OUR OWN REGISTERS!
	LD	D,#10H		;16 registers to load
RNLP	LDE	E,@HL
	LD	@C,E
	INC	C
	INCW	HL
	DJNZ	D,RNLP
;
RNSP:	
;
;  Initialize breakpoint address (so we can use 2-byte BP instruction)
	LD	R0,#^HB(BREAKPOINT_ENTRY)
	LD	R1,#^LB(BREAKPOINT_ENTRY)
;
;  Return to user with interrupts as per IMR
	TM	A,#80H
        JR	Z,RNOINT	;JIF USER'S INTS WERE OFF
	AND	A,#7FH		;CLEAR MASTER INT ENABLE
        LD	IMR,A		;RESTORE IMR, EXCEPT FOR MASTER BIT
	LD	RP,B		;RESTORE USER REG GROUP
        IRET			;RESTORE FLAGS, PC, ENABLE INTS
;
;  (We don't use IRET, as it will ALWAYS enable interrupts.)
RNOINT:	LD	IMR,A		;RESTORE IMR
	LD	RP,B		;RESTORE USER REG GROUP
	POP	FLAGS		;RELOAD FLAGS
	RET			;BACK TO USER, INTS DISABLED
;
;===========================================================================
;
;  Set target byte(s):  FN, len { (page, alow, ahigh, data), (...)... }
;
;  Return has FN, len, (data from memory locations)
;
;  If error in insert (memory not writable), abort to return short data
;
;  This function is used primarily to set and clear breakpoints
;
;  Enter with A=function, B=length, HL points at first data byte
;
SET_BYTES: 
;
	PUSH	R2
	PUSH	R3
	PUSH	R9
	AND	B,#11111100B		;AVOID INFINITE LOOP
        INC	B
        DEC	B
	LD	C,#0			;C GETS COUNT OF INSERTED BYTES
        JR	Z,SB90			;JIF NO BYTES (C=0)
	LD	R2,H			;SET UP REPLY BUFFER
	LD	R3,L
;
;  Loop on inserting bytes
SB10:   INCW	HL
	LDE	E,@HL			;ADDRESS TO DE
        INCW	HL
        LDE	D,@HL
        INCW	HL
;
;  Read current data at byte location
	LDE	A,@DE			;READ CURRENT DATA
        LDE	@RR2,A			;SAVE IN RETURN BUFFER
	INCW	RR2
;
;  Insert new data at byte location
	LDE	A,@HL
	LDE	@DE,A			;SET BYTE
        LDE	R9,@DE			;READ IT BACK
        CP	A,R9			;COMPARE TO DESIRED VALUE
        JR	NZ,SB90			;BR IF INSERT FAILED: ABORT
        INCW	HL
	INC	C			;ELSE COUNT ONE BYTE TO RETURN
;
	DEC	B
	DEC	B
	DEC	B
        DJNZ	B,SB10			;LOOP FOR ALL BYTES
;
;  Return buffer with data from byte locations
SB90:	LD	H,#^HB(COMBUF+1)
	LD	L,#^LB(COMBUF+1)
	LDE	@HL,C			;SET COUNT OF RETURN BYTES
;
;  Compute checksum on buffer, and send to master, then retURN
	POP	R9
	POP	R3
	POP	R2
SEND1:	JP	SEND

;===========================================================================
;
;  Input from port:  FN, len, PortAddressLo, PAhi (=0)
;
;  Enter with A=function, B=length, HL points at first data byte
;
IN_PORT: 
;
;  Get port address
	LDE	E,@HL
        INCW	HL
        LDE	D,@HL
;
;  Read port value
	CALL	READ_BYTE	;IN FROM "PORT"
;
;  Return byte read as "status"
	JP	SEND_STATUS

;===========================================================================
;
;  Output to port:  FN, len, PortAddressLo, PAhi (=0), data
;
;  Enter with A=function, B=length, HL points at first data byte
;
OUT_PORT: 
;
;  Get port address
	LDE	E,@HL
        INCW	HL
        LDE	D,@HL
        INCW	HL
;
;  Get data
	LDE	A,@HL
;
;  Write value to port
	CALL	WRITE_BYTE		;OUT FROM "PORT"
;
;  Return status of OK
	XOR	A,A
	JP	SEND_STATUS
;
;===========================================================================
;  Build status return with value from "A"
;
SEND_STATUS:
	LD	H,#^HB(COMBUF+2)
	LD	L,#^LB(COMBUF+2)
	LDE	@HL,A			;SET STATUS
	DECW	HL
	LD	A,#1
        LDE	@HL,A			;SET LENGTH
	JR	SEND

;===========================================================================
;  Append checksum to COMBUF and send to master
;
SEND:	CALL	CHECKSUM		;GET A=CHECKSUM, HL->checksum location
	LD	B,A			;NEG
	LD	A,#0
	SUB	A,B
	LDE	@HL,A			;STORE NEGATIVE OF CHECKSUM
;
;  Send buffer to master
	LD	H,#^HB(COMBUF)		;POINTER TO DATA
	LD	L,#^LB(COMBUF)		;POINTER TO DATA
	INCW	HL
        LDE	A,@HL			;LENGTH OF DATA
	DECW	HL
        ADD	A,#3			;PLUS FUNCTION, LENGTH, CHECKSUM
        LD	B,A			;save count for loop
SND10:	LDE	A,@HL
	CALL	PUTCHAR			;SEND A BYTE
	INCW	HL
	DJNZ	B,SND10
	JP	MAIN			;BACK TO MAIN LOOP

;===========================================================================
;  Compute checksum on COMBUF.  COMBUF+1 has length of data,
;  Also include function byte and length byte
;
;  Returns:
;	A = checksum
;	HL = pointer to next byte in buffer (checksum location)
;  	B is scratched
;
CHECKSUM:
	LD	H,#^HB(COMBUF)		;pointer to buffer
	LD	L,#^LB(COMBUF)		;pointer to buffer
	INCW	HL
        LDE	A,@HL			;length of message
	DECW	HL
        ADD	A,#2			;plus function, length
        LD	B,A			;save count for loop
	XOR	A,A			;init checksum to 0
	PUSH	C
CHK10:	LDE	C,@HL
	ADD	A,C
	INCW	HL
        DJNZ	B,CHK10			;loop for all
	POP	C
	RET				;return with checksum in A
;===========================================================================
;  Hop via RAM vectors to user interrupts
;
;  The first 6 words of USER_RAM are assumed to contain the addresses
;  of the user's interrupt handlers.  These are initialized to dummy
;  handlers at monitor reset, in case the user doesn't init them.
;
;  At entry here PC and FLAGS are on stack.
;  At entry to user handler, flags may be changed (but stack is OK
;  for IRET).
;
IRQ0:	LD	b,#2			;Monitor state for IRQ0
	LD	h,#^HB(USER_CODE+0)
	LD	l,#^LB(USER_CODE+0)
        JR	IRQX
;        
IRQ1:	LD	b,#3			;Monitor state for IRQ1
	LD	h,#^HB(USER_CODE+2)
	LD	l,#^LB(USER_CODE+2)
        JR	IRQX
;        
IRQ2:	LD	b,#4			;Monitor state for IRQ2
	LD	h,#^HB(USER_CODE+4)
	LD	l,#^LB(USER_CODE+4)
        JR	IRQX
;        
IRQ3:	LD	b,#5			;Monitor state for IRQ3
	LD	h,#^HB(USER_CODE+6)
	LD	l,#^LB(USER_CODE+6)
        JR	IRQX
;        
IRQ4:	LD	b,#6			;Monitor state for IRQ4
	LD	h,#^HB(USER_CODE+8)
	LD	l,#^LB(USER_CODE+8)
        JR	IRQX
;        
IRQ5:	LD	b,#7			;Monitor state for IRQ5
	LD	h,#^HB(USER_CODE+10)
	LD	l,#^LB(USER_CODE+10)
;;;	JR	IRQX
;        
IRQX:   PUSH	RP			;Need short regs for LDE
	SRP	#MONRP
	LDE	D,@HL			;Get user's handler address
        INCW	HL
        LDE	E,@HL
        POP	RP
        JP	@de			;Jump to user code
;
	.END
