;  80(1)96 Debug monitor for use with NoICE96
;
;  Copyright (c) 1992-1994 by John Hartman
;
;  Thanks to Gene Pfister for testing and reporting problems in the
;  80x96 version of NoICE.
;
;  Modification History:
;       31-Oct-94 by John Hartman: created from MON8051
;        4-Nov-94 JLH continued
;       27-Nov-94 JLH add support for internal UART
;       13-Dec-94 JLH fix bugs reported by Gene Pfister
;        8-Jan-95 JLH continued hardware init and usage tweaks
;       26-Jan-95 JLH interrupt linkage and formalized memory access
;	19-Jan-96 JLH bug near WRR80: should be SEND_STATUS, not SEND. v1.1
;
;============================================================================
;
;  To customize for a given target, you must change code in the
;  hardware equates, the string TSTG, and the routines RESET and REWDT.
;  You may or may not need to change GETCHAR, PUTCHAR, depending on
;  how peculiar your UART is.
;
;  This file has been assembled with the PseudoSam 8051 assembler A51.
;  This code will operate on either 8096 or 80196.  It does not use any
;  80196 instructions (BMOVI etc)
;
;  To add mapped memory support:
;       1) Define map port MAPREG here
;       2) Define map port RAM image MAPIMG here if MAPREG is write only
;       3) Search for and modify MAPREG, MAPIMG, and REG_PAGE usage below
;       4) In TSTG below edit "LOW AND HIGH LIMIT OF MAPPED MEM"
;          to appropriate range
;============================================================================
;
;  User code in external RAM.  Interrupts are re-vectored via vectors
;  beginning here.  Thus, you only need to change the ORG or link address
;  of your application program between debug and ROM versions.
;  The value of "USER_CODE" will, of course, depend on where your target
;  has RAM.
        .EQU    USER_CODE,H'C000        ;START OF USER INTERRUPTS/CODE
;
;  Initial user stack.
;  Size and location is user option.  May be in on-chip or off-chip memory.
;  (We initialize it just below NoICE RAM)
        .EQU    USER_STACK_TOP,H'FF80
;
;  Location of monitor code (2000, except during debug under another monitor)
        .EQU    NOICE_CODE,H'2000       ;START OF NOICE INTERRUPTS/CODE
;
;  Location of non-page zero monitor RAM.
;  See below for the amount of RAM reqyured.
        .EQU    NOICE_RAM,H'FF80        ;START OF NOICE RAM
;
;============================================================================
;  HARDWARE PLATFORM CUSTOMIZATIONS
;============================================================================
        .EQU    INT_MASK,H'08   	;Interrupt mask
        .EQU    WDT,     H'0A   	;Watchdog timer
        .EQU    WSR,     H'14   	;Window select
        .EQU    IOC1,    H'16   	;I/O CONTROL 1
;
;  Put you UART equates here
;  (Set for on-chip UART)
        .EQU    SER_BAUD,  H'0E         ;UART baud rate timer
        .EQU    SER_CONTRL,H'11         ;UART control register
        .EQU    SER_STATUS,H'11         ;UART status register
          .EQU    RXRDY,6               ;Receive Ready bit in SER_STATUS
          .EQU    TXRDY,5               ;Transmit Ready bit in SER_STATUS
        .EQU    SER_RXDATA,H'07         ;UART read data
        .EQU    SER_TXDATA,H'07         ;UART transmit data
;
;==========================================================================
;  Define "registers" used by NoICE monitor
        .EQU    SP,H'18
        .EQU    AX,H'1A
        .EQU    BX,H'1C
        .EQU    CX,H'1E
        .EQU    DX,H'20
        .EQU    END_REGS,H'22
;
        .EQU    AL,AX
        .EQU    AH,AX+1
        .EQU    BL,BX
        .EQU    BH,BX+1
        .EQU    CL,CX
        .EQU    CH,CX+1
;
;==========================================================================
;  Communications function codes.
        .equ    FN_GET_STATUS, H'FF    ;reply with device info
        .equ    FN_READ_MEM,   H'FE    ;reply with data
        .equ    FN_WRITE_MEM,  H'FD    ;reply with status (+/-)
        .equ    FN_RD_REGS,    H'FC    ;reply with registers
        .equ    FN_WR_REGS,    H'FB    ;reply with status
        .equ    FN_RUN_TARGET, H'FA    ;reply (delayed) with registers
        .equ    FN_SET_BYTES,  H'F9    ;reply with data (truncate if error)
        .equ    FN_IN,         H'F8    ;input from port
        .equ    FN_OUT,        H'F7    ;output to port
;
        .equ    FN_MIN,        H'F7    ;MINIMUM RECOGNIZED FUNCTION CODE
        .equ    FN_ERROR,      H'F0    ;error reply to unknown op-code
;
;==========================================================================
;  Non-page zero Monitor RAM definitions
        .ORG    NOICE_RAM
;
;  Monitor stack
;  (Calculated use is at most 2 bytes.  Leave plenty of spare)
;  While this could be on-chip RAM, moving the monitor stack off chip
;  frees the maximal amount of on-chip RAM for application use.
                .RS  32
MONSTACK:
;
SAVE_WSR:       .RS  1          ;SAVE USER'S WSR
;
;  Target registers:  order must match that in TRG8096.C
;  Knowledge of order is also assumed by INT_ENTRY, below
;
;  Caution: TASK_REGS must be at an odd address, so that REG_PC and REG_SP
;  are even.  Add a dummy byte here if required (or a .ODD directive, if
;  your assembler has one)
TASK_REGS:
 REG_STATE:    .RS  1
 REG_PAGE:     .RS  1
 REG_PSW:      .RS  1
 REG_PC:       .RS  2           ;MUST BE EVEN ADDRESS!
 REG_SP:       .RS  2           ;MUST BE EVEN ADDRESS!
 ;
        .EQU    T_REGS_SIZE, *-TASK_REGS
;
;  Save page-0 "registers" used by NoICE monitor here
;  For simplest code, these must immediately follow REG_SP
SAVE_AX:        .RS  2           ;MUST BE EVEN ADDRESS!
SAVE_BX:        .RS  2           ;MUST BE EVEN ADDRESS!
SAVE_CX:        .RS  2           ;MUST BE EVEN ADDRESS!
SAVE_DX:        .RS  2           ;MUST BE EVEN ADDRESS!
;
;  Communications buffer
;  (Must be at least as long as the longer of TASK_REG_SZ or TSTG_SIZE.
;  At least 19 bytes recommended.  Larger values may improve speed of NoICE
;  download and memory move commands.)
        .EQU    COMBUF_SIZE, 64         ;DATA SIZE FOR COMM BUFFER
COMBUF: .RS     2+COMBUF_SIZE+1         ;BUFFER ALSO HAS FN, LEN, AND CHECK
;
;===========================================================================
;  Interrupt vectors
        .ORG    NOICE_CODE
VEC1:   .DW     INT_TIMER_OVR           ; 2 (int 2000)
        .DW     INT_A_D_COMPLETE        ; 3 (int 2002)
        .DW     INT_HSI_DATA_AV         ; 4 (int 2004)
        .DW     INT_HSO                 ; 5 (int 2006)
        .DW     INT_HSIO                ; 6 (int 2008)
        .DW     INT_SW_TIMERS           ; 7 (int 200A)
        .DW     INT_SERIAL_PORT         ; 8 (int 200C)
        .DW     INT_EXTINT              ; 9 (int 200E)
        .DW     BREAKPOINT_ENTRY        ; 1 (breakpoint) or 10 (TRAP) (int 2010)
        .DW     INT_BAD_OPCODE          ; 11 (int 2012)
        .equ    NVEC_1, (*-VEC1)/2      ;NUMBER OF VECTORS IN FIRST TABLE
;
;  Data following interrupt vectors
;  (Change these as required by your hardware)
        .ORG    NOICE_CODE+H'0014
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (2014-2017)
        .DB     B'11101110              ;CCB (2018)
        ;         11                      ;no protection
        ;           10                    ;add at most 3 wait states (don't hang!)
        ;             1                   ;ALE (Standard Bus)
        ;              1                  ;WR and BHE (Standard Bus)
        ;               1                 ;dynmaic bus width (BUSWIDTH pin)
        ;                0                ;disable powerdown during development
        .DB     H'20                    ;"must contain 20h" (2019)
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (201A-201D)
        .DB     H'FF, H'FF              ;reserved (201E-201F)
        .DB     H'FF, H'FF, H'FF, H'FF  ;security key (2020-2023)
        .DB     H'FF, H'FF, H'FF, H'FF  ;security key (2024-2027)
        .DB     H'FF, H'FF, H'FF, H'FF  ;security key (2028-202B)
        .DB     H'FF, H'FF, H'FF, H'FF  ;security key (202C-202F)
        .DB
;
;  Upper Interrupt vectors
        .ORG    NOICE_CODE+H'0030
VEC2:   .DW     INT_TI                  ; 12 (int 2030)
        .DW     INT_RI                  ; 13 (int 2032)
        .DW     INT_4TH_IN_HSI_FIFO     ; 14 (int 2034)
        .DW     INT_TIMER2_CAPTURE      ; 15 (int 2036)
        .DW     INT_TIMER2_OVERFLOW     ; 16 (int 2038)
        .DW     INT_EXTINT1             ; 17 (int 203A)
        .DW     INT_HSI_FIFO_FULL       ; 18 (int 203C)
        .DW     INT_NMI                 ; 19 (int 203E)
        .equ    NVEC_2, (*-VEC2)/2      ;NUMBER OF VECTORS IN 2ND TABLE
;
;  PTS vectors
;  These cannot be re-directed to RAM PTS vectors.  If you use PTS, you
;  will need to make a custom version of this monitor with PTS vectors
;  pointing at your PTS blocks in register RAM.
        .ORG    NOICE_CODE+H'0040

;  Data following PTS vectors
        .ORG    NOICE_CODE+H'005E
        .DB     H'FF, H'FF              ;reserved (205E-205F)
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (2060-2063)
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (2064-2067)
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (2068-206B)
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (206C-206F)
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (2070-2073)
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (2074-2077)
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (2078-207B)
        .DB     H'FF, H'FF, H'FF, H'FF  ;reserved (207C-207F)
;
;===========================================================================
        .org NOICE_CODE+H'0080          ;reset start address
;  Power on reset
RESET:  DI                              ;disable interrupts
        LDB     WSR,#0                  ;Disable HOLD, select HWindow 0
        LD      SP,#MONSTACK            ;initialize to monitor stack
;
;  Stall, if necessary (such as external UART which may remain in reset
;  longer than the micro.  In this case, init UART AFTER this delay)
;;      LD      BX,#0
;;LLPP  DEC     BX                      ;LONG DELAY IN CASE EXTERNAL UART IN RESET
;;      JNE     LLPP
;
;============================================================================
;  Initialize your UART and other hardware here
;
;  For on-chip UART, set IOC1.5 for output
        ORB     IOC1,#B'00100000        ;SET BIT 5 FOR TXD OUTPUT
;
;  Init baud rate:  rate = XTAL/(16*(DIV+1); or DIV = XTAL/(16*rate) - 1
;  With 16 Mhz crystal, 9600 baud: DIV = 103 (9615 baud: +0.2 percent)
;  With 16 Mhz crystal, 19200 baud: DIV = 51 (19230 baud: +0.2 percent)
;  With 16 Mhz crystal, 38400 baud: DIV = 25 (38461 baud: +0.2 percent)
        LDB     SER_BAUD,#25            ;38400 baud from 16 Mhz
        LDB     SER_BAUD,#H'80          ;start baud rate generator
;
;  Init baud rate:  rate = XTAL/(16*(DIV+1); or DIV = XTAL/(16*rate) - 1
;  With 11.0592 Mhz crystal, 9600 baud: DIV = 71
;  With 11.0592 Mhz crystal, 19200 baud: DIV = 35
;;;     LDB     SER_BAUD,#35            ;19200 baud from 11.0592 Mhz
;;;     LDB     SER_BAUD,#H'80          ;start baud rate generator
;
;  Initialize UART
        LDB     SER_CONTRL,#B'00001001  ;RX ENABLE, MODE 1
;
;  Initialize other hardware

;============================================================================
;
;  Initialize user's interrupt vectors to default values.
;  (Should be overlaid by user's code, but catch the shlitz)
        LD      BX,#DEFAULT_INT         ;FIRST DEFAULT HANDLER
        LD      DX,#USER_CODE           ;ADDRESS OF USER VECTORS
        LDB     CL,#NVEC_1              ;NUMBER OF VECTORS
INITV:  ST      BX,[DX]+                ;SET VECTOR WITH DEFAULT HANDLER
        ADD     BX,#DEFAULT_INT_SIZE	;ADDRESS OF NEXT DEFAULT HANDLER
        DJNZ    CL,INITV
;
        LD      DX,#USER_CODE+H'0030    ;ADDRESS OF SECONDARY VECTORS
        LDB     CL,#NVEC_2              ;NUMBER OF VECTORS
INITV2: ST      BX,[DX]+                ;SET VECTOR WITH DEFAULT HANDLER
        ADD     BX,#DEFAULT_INT_SIZE	;ADDRESS OF NEXT DEFAULT HANDLER
        DJNZ    CL,INITV2
;
;  Initialize user's registers
        CLR     AX
        STB     AL,REG_STATE[0]         ;STATE 0 = RESET
        STB     AL,REG_PAGE[0]          ;INITIAL USER MEMORY PAGE
        STB     AL,REG_PSW[0]
        LD      AX,#USER_CODE+H'80      ;INITIAL PC IN USER CODE
        ST      AX,REG_PC[0]
        LD      AX,#USER_STACK_TOP      ;INITIAL USER STACK
        ST      AX,REG_SP[0]
;
;  Set function code for "GO".  Then if we are here because of a reset
;  (such as a watchdog timeout) after being told to GO, we will come
;  back with registers so user can see the reset
        LDB     AL,#FN_RUN_TARGET
        STB     AL,COMBUF[0]
        SJMP    RETURN_REGS             ;DUMP REGS, ENTER MONITOR
;
;===========================================================================
;  Get a character to AL
;
;  Return AL=char, CY=0 if data received
;         CY=1 if timeout (0.5 seconds)
;
;  Uses 2 bytes of stack including return address
;
GETCHAR:
;
;  Be sure WSR allows access to UART
        STB     WSR,SAVE_WSR[0]
        LDB     WSR,#0
;
;  (Add timeout (return CY=1) if no byte recieved in 500 msec, if desired)
GC10:
;
;  If your application uses the Watchdog Timer, enable this code.
;  Otherwise, leave it out:  once you write to the WDT, you MUST
;  continue doing so forever.
;;;     LDB     WDT,#H'1E
;;;     LDB     WDT,#H'E1

        LDB     AL,SER_STATUS[0]
        JBC     AL,RXRDY,GC10           ;loop til ready
;
;  Data received:  return CY=0. data in A
        CLRC
        LDB     AL,SER_RXDATA[0]
;
;  Put WSR back where it was
        LDB     WSR,SAVE_WSR[0]
        RET
;
;===========================================================================
;  Output character in AL
;
;  Uses 2 bytes of stack including return address
;  Destroys AH
;
PUTCHAR:
;
;  Be sure WSR allows access to UART
        STB     WSR,SAVE_WSR[0]
        LDB     WSR,#0
;
;  If your application uses the Watchdog Timer, enable this code.
;  Otherwise, leave it out:  once you write to the WDT, you MUST
;  continue doing so forever.
;;;     LDB     WDT,#H'1E
;;;     LDB     WDT,#H'E1
;
;  Use the following code for 8096 on-chip UART: TXDRY bit may have been
;  cleared by reads of the SER_STATUS register in GETCHAR.  Thus, rather
;  than waiting for TXRDY BEFORE sending, we send, and wait AFTERWARD
        STB     AL,SER_TXDATA[0]        ;send data
PC10:   LDB     AH,SER_STATUS[0]        ;read status (and clear TX and RX bits)
        JBC     AH,TXRDY,PC10           ;loop til output complete
;
;  Use the following code for most "normal" UARTs
;;;PC10:     
;;;     LDB     AH,SER_STATUS[0]
;;;     JBC     AH,TXRDY,PC10           ;loop til previous output complete
;;;     STB     AL,SER_TXDATA[0]        ;send data
;
;  Put WSR back where it was
        LDB     WSR,SAVE_WSR[0]
        RET
;
;===========================================================================
;
;  Response string for GET TARGET STATUS request
;  Reply describes target:
TSTG:   .db     8                       ;2: PROCESSOR TYPE = 8096
        .db     COMBUF_SIZE             ;3: SIZE OF COMMUNICATIONS BUFFER
        .db     0                       ;4: NO OPTIONS
        .dw     0,0                     ;5-8: LOW AND HIGH LIMIT OF MAPPED MEM (NONE)
        .db     B1-B0                   ;9 BREAKPOINT INSTRUCTION LENGTH
B0:     .db     H'F7                    ;10+ BREKAPOINT INSTRUCTION (TRAP)
B1:     .db     "80(1)96 monitor V1.1",0 ;DESCRIPTION, ZERO
        .equ    TSTG_SIZE, *-TSTG       ;SIZE OF STRING
        .equ    BRK_SIZE, B1-B0         ;DEFINE LENGTH OF A BREAKPOINT INSTR.
;
;===========================================================================
;  Read byte from memory [BX] to CH
;
;  Allows address translation and protection
;
;  Vertical Windowing is not supported.
;
READ_BYTE:
;
;  Test for access to SP, AX, BX, CX, or DX
;  Use REG_SP, SAVE_AX, SAVE_BX, SAVE_CX, SAVE_DX instead
        CMP     BX,#SP-1
        JNH     RB80                    ;JIF BELOW REGISTERS
        CMP     BX,#END_REGS-1
        JH      RB80                    ;JIF ABOVE REGISTERS
        LDB     CH,REG_SP-SP[BX]        ;GET SAVED REGISTER INSTEAD
        RET
;
RB80:   LDB     CH,[BX]
        RET
;
;===========================================================================
;  Write byte from CH to memory [BX]
;
;  Allows address translation and protection
;
;  Vertical Windowing is not supported.
;
WRITE_BYTE:
;
;  Test for access to SP, AX, BX, CX, or DX
;  Use REG_SP, SAVE_AX, SAVE_BX, SAVE_CX, SAVE_DX instead
        CMP     BX,#SP-1
        JNH     WB80                    ;JIF BELOW REGISTERS
        CMP     BX,#END_REGS-1
        JH      WB80                    ;JIF ABOVE REGISTERS
        STB     CH,REG_SP-SP[BX]        ;GET SAVED REGISTER INSTEAD
        RET
;
WB80:   STB     CH,[BX]
        RET
;
;===========================================================================
;  HARDWARE PLATFORM INDEPENDENT EQUATES AND CODE
;
;===========================================================================
;  Enter here via TRAP for breakpoint.  PC is stacked
BREAKPOINT_ENTRY:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine (dummy)
        PUSH    #1	             	;breakpoint
;
;  Enter here for unserviced interrupt.
;  - Enter with TOS = interrupt code = processor state
;  - Below that is a return to INT_EXIT
;  - Below that is saved (PSW/INT_MASK)
;  - Below that is the PC of the interrupted code
;
;  None of our "registers" are available, as they may be in use by the user
INT_ENTRY:
;
;  Save user RAM we wish to use for scratch registers
        ST      AX,SAVE_AX[0]
        ST      BX,SAVE_BX[0]
        ST      CX,SAVE_CX[0]
        ST      DX,SAVE_DX[0]
;
;  Save processor registers in reg block for return to master
        LD      DX,#TASK_REGS
        POP	BX			;get state from stack
        STB     BL,[DX]+                ;save state (and keep in BL)
        LDB     AL,#0                   ;fake bank, if unmapped
;;;     LDB     AL,MAPREG[0]            ;or bank, if mapped
        STB     AL,[DX]+                ;save PAGE, if any, else 0
;
        POP     AX                      ;delete dummy return to INT_EXIT
        POP     AX			;get pre-interrupt PSW/INT_MASK
        STB     AH,[DX]+                ;save PSW bits as "register"
        STB     AL,INT_MASK             ;restore int mask, for user inspection
;
;  If entry here was by breakpoint (state=1), then back up the program
;  counter to point at the breakpoint/TRAP instruction.  Else leave PC alone.
        POP     AX                      ;PC
        CMPB    BL,#1
        JNE     NOTBP                   ;JIF NOT A BREAKPOINT
        SUB     AX,#BRK_SIZE            ;BACK UP PC TO POINT AT BREAKPOINT
NOTBP:  ST      AX,[DX]+		;user's PC
;
        ST      SP,[DX]+                ;save user's SP
;
;  Switch to monitor stack, to be sure of a real stack if the user progam
;  loaded something bad into SP.
        LD      SP,#MONSTACK            ;initial stack pointer
;
;  Return registers to master
        SJMP    RETURN_REGS
;
;===========================================================================
;  Main loop:  wait for command frame from master
MAIN:   LD      DX,#COMBUF              ;BUILD MESSAGE HERE
;
;  First byte is a function code
        SCALL   GETCHAR                 ;GET A FUNCTION (uses 2 bytes of stack)
        JC      MAIN                    ;JIF TIMEOUT: RESYNC
        CMPB    AL,#FN_MIN-1
        JNH     MAIN                    ;JIF BELOW MIN: ILLEGAL FUNCTION
        STB     AL,[DX]+                ;SAVE FUNCTION CODE
;
;  Second byte is data byte count (may be zero)
        SCALL   GETCHAR                 ;GET A LENGTH BYTE
        JC      MAIN                    ;JIF TIMEOUT: RESYNC
        CMPB    AL,#COMBUF_SIZE
        JH      MAIN                    ;JIF TOO LONG: ILLEGAL LENGTH
        STB     AL,[DX]+                ;SAVE LENGTH
        ADDB    CL,AL,#0                ;CL = AL, TEST VALUE FOR ZERO
        JE      MA80                    ;SKIP DATA LOOP IF LENGTH IS 0
;
;  Loop for data
MA10:   SCALL   GETCHAR                 ;GET A DATA BYTE
        JC      MAIN                    ;JIF TIMEOUT: RESYNC
        STB     AL,[DX]+                ;SAVE DATA BYTE
        DJNZ    CL,MA10
;
;  Get the checksum
MA80:   SCALL   GETCHAR                 ;GET THE CHECKSUM
        JC      MAIN                    ;JIF TIMEOUT: RESYNC
        LDB     BL,AL                   ;SAVE CHECKSUM
;
;  Compare received checksum to that calculated on received buffer
;  (Sum should be 0)
        SCALL   CHECKSUM
        ADDB    AL,BL
        JNE     MAIN                    ;JIF BAD CHECKSUM (AL != 0)
;
;  Process the message.
        LD      DX,#COMBUF
        LDB     AL,[DX]+                ;GET THE FUNCTION CODE
        LDB     CL,[DX]+                ;GET THE LENGTH
;
        CMPB    AL,#FN_GET_STATUS
        JE      TARGET_STATUS
        CMPB    AL,#FN_READ_MEM
        JE      READ_MEM
        CMPB    AL,#FN_WRITE_MEM
        JE      WRITE_MEM
        CMPB    AL,#FN_RD_REGS
        JE      JREAD_REGS
        CMPB    AL,#FN_WR_REGS
        JE      JWRITE_REGS
        CMPB    AL,#FN_RUN_TARGET
        JE      JRUN_TARGET
        CMPB    AL,#FN_SET_BYTES
        JE      JSET_BYTES
        CMPB    AL,#FN_IN
        JE      JIN_PORT
        CMPB    AL,#FN_OUT
        JE      JOUT_PORT
;
;  Error: unknown function.  Complain
        LDB     AL,#FN_ERROR
        LD      DX,#COMBUF
        STB     AL,[DX]                 ;SET FUNCTION AS "ERROR"
        LDB     AL,#1
        SJMP    SEND_STATUS             ;VALUE IS "ERROR"

JREAD_REGS:   SJMP    READ_REGS
JWRITE_REGS:  SJMP    WRITE_REGS
JRUN_TARGET:  SJMP    RUN_TARGET
JSET_BYTES:   SJMP    SET_BYTES
JIN_PORT:     SJMP    IN_PORT
JOUT_PORT:    SJMP    OUT_PORT

;===========================================================================
;
;  Target Status:  FN, len
;
;  Enter with AL=function, CL=length, DX points at first data byte
;
TARGET_STATUS:
;
        LD      BX,#TSTG                ;STATUS STRING
        LDB     CL,#TSTG_SIZE           ;LENGTH OF REPLY
        STB     CL,-1[DX]               ;SET SIZE IN REPLY BUFFER
;
;  Loop on data in string
TS10:   LDB     AL,[BX]+
        STB     AL,[DX]+
        DJNZ    CL,TS10
;
;  Compute checksum on buffer, and send to master, then return
        SJMP    SEND

;===========================================================================
;
;  Read Memory:  FN, len, page, Alo, Ahi, Nbytes
;
;  Enter with AL=function, CL=length, DX points at first data byte
;
READ_MEM:
;
;  Set map
;;      LDB     AL,[DX]
;;      STB     AL,MAPREG[0]
;
;  Get address
        LDB     BL,1[DX]                ;LSB ADDRESS
        LDB     BH,2[DX]                ;MSB ADDRESS
;
;  Get byte count
        LDB     CL,3[DX]                ;BYTE COUNT
;
;  Prepare return buffer: FN (unchanged), LEN, DATA
        STB     CL,-1[DX]               ;RETURN LENGTH = REQUESTED DATA
        ORB     CL,CL
        JE      GLP90                   ;JIF NO BYTES TO GET (A=0)
;
;  Read the requested bytes from local memory
GLP:    LCALL   READ_BYTE
        INC     BX
        STB     CH,[DX]+
        DJNZ    CL,GLP
;
;  Compute checksum on buffer, and send to master, then return
GLP90:  SJMP    SEND

;===========================================================================
;
;  Write Memory:  FN, len, page, Alo, Ahi, (len-3 bytes of Data)
;
;  Enter with AL=function, CL=length, DX points at first data byte
;
WRITE_MEM:
;
;  Set map
        LDB     AL,[DX]+                ;PAGE, ADVANCE DX
;;      STB     AL,MAPREG[0]
;
;  Get address
        LDB     BL,[DX]+                ;LSB ADDRESS, ADVANCE DX
        LDB     BH,[DX]+                ;MSB ADDRESS, ADVANCE DX
;
;  Get length to write
        SUBB    CL,#3                   ;LESS PAGE, ADDRESS
        JE      WLP80                   ;EXIT OF NO BYTES REQUESTED (A=0)
        LDB     AH,CL                   ;SAVE COPY IN AH FOR COMPARE LOOP
;
;  Write the requested bytes to local memory
WLP:    LDB     CH,[DX]+                ;GET BYTE TO WRITE
        LCALL   WRITE_BYTE              ;TO MEMORY
        INC     BX
        DJNZ    CL,WLP                  ;LOOP ON WRITING BYTES
;
;  Compare to see if the write worked
;
;  Get memory address again, point DX at source data
        LD      DX,#COMBUF+3
        LDB     BL,[DX]+                ;LSB MEMORY ADDRESS, ADVANCE DX
        LDB     BH,[DX]+                ;MSB MEMORY ADDRESS, ADVANCE DX
;
;  Compare the requested bytes to local memory
WLP50:  LCALL   READ_BYTE               ;GET BYTE FROM MEMORY
        INC     BX
        CMPB    CH,[DX]+                ;COMPARE TO BYTE WE WROTE
        JNE     WLP90                   ;JIF NOT SAME AS WRITTEN VALUE
        DJNZ    AH,WLP50                ;LOOP ON COMPARING BYTES
;
;  Write succeeded:  return status = 0
WLP80:  CLRB    AL                      ;RETURN STATUS = 0
        SJMP    WLP100
;
;  Write failed:  return status = 1
WLP90:  LDB     AL,#1
;
;  Return OK status
WLP100: SJMP    SEND_STATUS

;===========================================================================
;
;  Read registers:  FN, len=0
;
;  Enter with AL=function, CL=length, DX points at first data byte
;
READ_REGS:
;
;  Enter here from int after "RUN" and "STEP" to return task registers
;  All registers unknown
;  COMBUF+0 will contain either FN_RD_REGS or FN_RUN_TARGET
RETURN_REGS:
        LD      DX,#COMBUF+1            ;RETURN BUFFER: LENGTH BYTE
        LD      BX,#TASK_REGS           ;REGISTER LIVE HERE
        LDB     CL,#T_REGS_SIZE         ;NUMBER OF BYTES
;
;  Prepare return buffer: FN (unchanged), LEN, DATA
        STB     CL,[DX]+                ;SAVE DATA LENGTH
;
;  Copy the registers
GRLP:   LDB     AL,[BX]+
        STB     AL,[DX]+
        DJNZ    CL,GRLP
;
;  Compute checksum on buffer, and send to master, then return
        SJMP    SEND

;===========================================================================
;
;  Write registers:  FN, len, (register image)
;
;  Enter with AL=function, CL=length, DX points at first data byte
;
WRITE_REGS:
;
        ORB     CL,CL
        JE      WRR80                   ;JIF NO REGISTERS
        LD      BX,#TASK_REGS
;
;  Copy the registers
WRLP:   LDB     AL,[DX]+
        STB     AL,[BX]+
        DJNZ    CL,WRLP
;
;  Return OK status
WRR80:  CLRB    AL
        SJMP    SEND_STATUS

;===========================================================================
;
;  Run Target:  FN, len
;
;  Uses 4 bytes of user stack, 0 bytes of monitor stack
;
;  Enter with AL=function, CL=length, DX points at first data byte
;
RUN_TARGET:
;
;  Restore user's map
;;      LDB     AL,REG_PAGE[0]
;;      STB     AL,MAPREG[0]
;
;  Switch to user stack
        LD      SP,REG_SP[0]
;
;  Push user PC for RET
        PUSH    REG_PC[0]
;
;  Push user's PSW
        LDB     AH,REG_PSW[0]
        LDB     AL,INT_MASK
        PUSH    AX
;
;  Restore user RAM we used for scratch registers
        LD      AX,SAVE_AX[0]
        LD      BX,SAVE_BX[0]
        LD      CX,SAVE_CX[0]
        LD      DX,SAVE_DX[0]
;
;  Return to user
        POPF                            ; SET PSW
        RET                             ; 0 bytes on stack - back to user
;
;===========================================================================
;
;  Set target byte(s):  FN, len { (page, alow, ahigh, data), (...)... }
;
;  Return has FN, len, (data from memory locations)
;
;  If error in insert (memory not writable), abort to return short data
;
;  This function is used primarily to set and clear breakpoints
;
;  Enter with AL=function, CL=length, DX points at first data byte
;
;  Uses 0 bytes of stack
;
SET_BYTES:
        ORB     CL,CL
        JE      SB90            ;JIF NO BYTES (A=0)
        LD      AX,DX           ;POINTER TO REPLY BUFFER
;
;  Loop on inserting bytes
SB10:
;
;  Set map
        LDB     CH,[DX]+
;;      STB     CH,MAPREG[0]
;
;  Get address
        LDB     BL,[DX]+
        LDB     BH,[DX]+
;
;  Read current data at byte location
        LCALL   READ_BYTE               ;READ CURRENT DATA
        STB     CH,[AX]                 ;SAVE IN REPLY BUFFER
;
;  Insert new data at byte location
        LDB     CH,[DX]
        LCALL   WRITE_BYTE              ;WRITE BYTE TO [BX]
        LCALL   READ_BYTE               ;READ IT BACK
        CMPB    CH,[DX]+                ;COMPARE TO WHAT WE WROTE
        JNE     SB90                    ;JIF INSERT FAILED: ABORT
;
;  Advance to next breakpoint
        INC     AX                      ;NEXT REPLY BYTE
        SUBB    CL,#4
        JNE     SB10                    ;LOOP FOR ALL BYTES
;
;  Return buffer with data from byte locations
;  AX points at next free byte in return buffer.  Compute length
SB90:   SUB     CX,AX,#COMBUF+2         ;CX = BYTES WRITTEN SUCCESSFULLY
        STB     CL,COMBUF+1[0]          ;SET COUNT OF RETURN BYTES
;
;  Compute checksum on buffer, and send to master, then return
        SJMP    SEND

;===========================================================================
;
;  Input from port:  FN, len, PortAddressLow, PortAddressHigh
;
;  Enter with AL=function, CL=length, DX points at first data byte
;
IN_PORT:
;
;  Get port address
        LDB     BL,[DX]+                ;LSB ADDRESS
        LDB     BH,[DX]+                ;MSB ADDRESS
;
;  Read port value
        LCALL   READ_BYTE               ;READ BYTE [BX]
        LDB     AL,CH                   ;BYTE IN AL FOR RETURN
;
;  Return byte read as "status"
        SJMP    SEND_STATUS

;===========================================================================
;
;  Output to port:  FN, len, PortAddressLo, PAhi, data
;
;  Enter with AL=function, CL=length, DX points at first data byte
;
OUT_PORT:
;
;  Get port address
        LDB     BL,[DX]+                ;LSB ADDRESS
        LDB     BH,[DX]+                ;MSB ADDRESS
        LDB     CH,[DX]                 ;BYTE TO WRITE
        LCALL   WRITE_BYTE              ;OUTPUT!
;
;  Return status of OK
        CLRB    AL
        SJMP    SEND_STATUS
;
;===========================================================================
;  Build status return with value from "AL"
;
SEND_STATUS:
        LD      DX,#COMBUF+1
        LDB     AH,#1
        STB     AH,[DX]+                ;SET LENGTH
        STB     AL,[DX]                 ;SET STATUS
        SJMP    SEND

;===========================================================================
;  Append checksum to COMBUF and send to master
;
;  Uses 2 bytes of stack
;
SEND:   SCALL   CHECKSUM                ;GET AL=CHECKSUM, DX->CHECKSUM LOCATION
        NEGB    AL
        STB     AL,[DX]                 ;STORE NEGATIVE OF CHECKSUM
;
;  Send buffer to master
        LD      DX,#COMBUF              ;POINTER TO OUTPUT BUFFER
        LDB     CL,1[DX]                ;LENGTH FO DATA
        ADDB    CL,#3                   ;PLUS FUNCTION, LENGTH, CHECKSUM
SND10:  LDB     AL,[DX]+
        SCALL   PUTCHAR                 ;SEND A BYTE
        DJNZ    CL,SND10
        SJMP    MAIN                    ;BACK TO MAIN LOOP

;===========================================================================
;  Compute checksum on COMBUF.  COMBUF+1 has length of data,
;  Also include function byte and length byte
;
;  Returns:
;       AL = checksum
;       DX = pointer to next byte in buffer (checksum location)
;       CL scratched
;
;  Uses 2 bytes of stack including return address
;
CHECKSUM:
        LD      DX,#COMBUF              ;POINTER TO BUFFER
        LDB     CL,1[DX]                ;LENGTH BYTE
        ADDB    CL,#2                   ;PLUS FUNCTION AND LENGTH BYTES
        CLR     AL                      ;INIT CHECKSUM TO ZERO
CHK10:  ADDB    AL,[DX]+
        DJNZ    CL,CHK10                ;loop for all
        RET                             ;return with checksum in AL

;===========================================================================
;
;  Interrupt handlers to re-vector interrupts through user RAM
;
;  PC is stacked by the interrupt.  Interrupts are still enabled at
;  entry to this code, but execution of the first instruction (ONLY) 
;  is guaranteed.  Since we cannot pass this exact condition to the user, 
;  we save interrupt status and disable interrupts here via PUSHF,
;  and CALL the user.  The user may then PUSHF/PUSHA...POPF/POPA,RET
;  as if she was directly connected to the interrupt.
;
;  The only difference from direct interrupt to user is that PSW.1 (I)
;  and INT_MASK are clear upon entry to the user's routine, and there 
;  are 2 more words (return to our handler, and INT_MASK/PSW) on the stack.
;
;  This code is carefully crafted to use no page zero RAM.
;
INT_TIMER_OVR:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0000[0]     ;address of user's interrupt handler
        RET                             ;"call" the user's handler

INT_A_D_COMPLETE:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0002[0]     ;address of user's interrupt handler
        RET                             ;"call" the user's handler
;
INT_HSI_DATA_AV:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0004[0]
        RET                             ;"call" the user's handler

INT_HSO:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0006[0]
        RET                             ;"call" the user's handler

INT_HSIO:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0008[0]
        RET                             ;"call" the user's handler

INT_SW_TIMERS:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'000A[0]
        RET                             ;"call" the user's handler

INT_SERIAL_PORT:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'000C[0]
        RET                             ;"call" the user's handler

INT_EXTINT:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'000E[0]
        RET                             ;"call" the user's handler

;  TRAP is normally vectored directly to BREAKPOINT_ENTRY, rather
;  than here.  This could be changed to allow user code to handle TRAP
;  if a CALL is used for breakpoint:  Change BREAKPOINT_ENTRY to INT_TRAP
;  in the table VEC1, and change the definition of the instruction to be 
;  used for breakpoint at label B0 in TSTG.
INT_TRAP:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'00010[0]
        RET                             ;"call" the user's handler

;  BAD OPCODE is normally passed directly to the monitor, rather than 
;  thru RAM.  This could be changed to allow user code to handle BAD
;  OPCODE.  Replace (PUSH #11/LJMP) with (PUSH USER_CODE/RET)
INT_BAD_OPCODE:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    #11             	;int 2012
        LJMP    INT_ENTRY
;;;     PUSH    USER_CODE+H'000E[0]
;;;     RET                             ;"call" the user's handler

INT_TI:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0030[0]
        RET                             ;"call" the user's handler

INT_RI:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0032[0]
        RET                             ;"call" the user's handler

INT_4TH_IN_HSI_FIFO:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0034[0]
        RET                             ;"call" the user's handler

INT_TIMER2_CAPTURE:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0036[0]
        RET                             ;"call" the user's handler

INT_TIMER2_OVERFLOW:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'0038[0]
        RET                             ;"call" the user's handler

INT_EXTINT1:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'003A[0]
        RET                             ;"call" the user's handler

INT_HSI_FIFO_FULL:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        PUSH    USER_CODE+H'003C[0]
        RET                             ;"call" the user's handler

;  NMI is normally passed directly to the monitor, rather than 
;  thru RAM.  This could be changed to allow user code to handle NMI
;  Replace (PUSH #19/LJMP) with (PUSH USER_CODE/RET)
INT_NMI:
        PUSHF                           ;disable interrupts (must be 1st instruction)
        PUSH    #INT_EXIT               ;address of clean-up routine
        LDB     BL,#19          	;19 (int 203E)
        LJMP    INT_ENTRY
;;;     PUSH    USER_CODE+H'003E[0]
;;;     RET                             ;"call" the user's handler

;==========================================================================
;  Default Interrupt Handlers, used to catch interrupts and traps not
;  initialized by the user.
;
;  These must all be the same length, unless the RAM vector initialization
;  code is changed.  Thus, we include TRAP, BAD OPCODE, and NMI even
;  if they will not actually be used.
;
DEFAULT_INT:
;
;  INT_TIMER_OVR
        PUSH    #2              ;int 2000
        LJMP    INT_ENTRY
	.equ	DEFAULT_INT_SIZE, *-DEFAULT_INT

;  INT_A_D_COMPLETE
        PUSH    #3              ;int 2002
        LJMP    INT_ENTRY

;  INT_HSI_DATA_AV
        PUSH    #4              ;int 2004
        LJMP    INT_ENTRY

;  INT_HSO
        PUSH    #5              ;int 2006
        LJMP    INT_ENTRY

;  INT_HSIO
        PUSH    #6              ;int 2008
        LJMP    INT_ENTRY

;  INT_SW_TIMERS
        PUSH    #7              ;int 200A
        LJMP    INT_ENTRY

;  INT_SERIAL_PORT
        PUSH    #8              ;int 200C
        LJMP    INT_ENTRY

;  INT_EXTINT
        PUSH    #9              ;int 200E
        LJMP    INT_ENTRY

;  INT_TRAP
        PUSH    #10             ;int 2010
        LJMP    INT_ENTRY

;  INT_BAD_OPCODE
        PUSH    #11             ;int 2012
        LJMP    INT_ENTRY

;  INT_TI
        PUSH    #12             ;int 2030
        LJMP    INT_ENTRY

;  INT_RI
        PUSH    #13             ;int 2032
        LJMP    INT_ENTRY

;  INT_4TH_IN_HSI_FIFO
        PUSH    #14             ;int 2034
        LJMP    INT_ENTRY

;  INT_TIMER2_CAPTURE
        PUSH    #15             ;int 2036
        LJMP    INT_ENTRY

;  INT_TIMER2_OVERFLOW
        PUSH    #16             ;int 2038
        LJMP    INT_ENTRY

;  INT_EXTINT1
        PUSH    #17             ;int 203A
        LJMP    INT_ENTRY

;  INT_HSI_FIFO_FULL
        PUSH    #18             ;int 203C
        LJMP    INT_ENTRY

;  INT_NMI
        PUSH    #19             ;int 203E
        LJMP    INT_ENTRY
;
;  Interrupt exit
INT_EXIT:
        POPF                    ;Restore interrupt state
        RET                     ;Back to interrupted code

        .end    RESET
