/* 
VXDLIST.C
Display Windows 3.x Enhanced mode VxD chain
Andrew Schulman, September 1992
CIS 76320,302

Usage: vxdlist [-verbose] [vxd_name or #hex_id]
Examples:
vxdlist 
vxdlist -verbose
vxdlist DOSMGR
vxdlist -verbose DOSMGR
vxdlist #0c

The name is case-sensitive!

bcc -DDPMI_APP -2 -P- vxdlist.c vxdchain.c map_lin.c dpmishel.c ctrl_c.asm

bcc -W -2 -P- vxdlist.c vxdchain.c map_lin.c
*/

#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dos.h>
#ifdef DPMI_APP
#include "dpmishel.h"
#else
#include "windows.h"
#endif
#include "map_lin.h"
#include "vxdchain.h"

#ifndef DPMI_APP
#define _dos_exit(err) exit(err)
#endif

static DDB far *glob_ddb = (DDB far *) 0;

#define FREE_MAPPED_LINEAR(fp) { free_mapped_linear(fp); fp = 0; }

void fail(const char *s) 
{ 
    if (glob_ddb)
        FREE_MAPPED_LINEAR(glob_ddb);    // clean up any left over
    fputs(s, stderr); 
    fputs("\n", stderr);
    _dos_exit(1);
}

int iswin3e(void)	// make sure Windows Enhanced mode
{					// by calling INT 2Fh AX=1600h
    WORD maj=0, min=0;
    _asm {
        mov ax, 1600h
        int 2fh
        mov byte ptr maj, al
        mov byte ptr min, ah
        }
    return ((maj > 1) && (maj != 0x80));
}

unsigned long axton(char *s)
{
    if ((s[0] == '0') && (s[1] == 'x'))
    {
        unsigned long ret;
        sscanf(&s[2], "%lx", &ret);
        return ret;
    }
    else
        return atol(s);
}

static int verbose = 0;
static char *vxd_name = (char *) 0;
static WORD vxd_id = 0;

int v86_main(int argc, char *argv[])
{
    int i;

	puts("VXDLIST version 1.10\n"
    "Displays Windows Enhanced mode Virtual Device Driver (VxD) Chain\n"
    "Copyright (c) 1992-1993 Andrew Schulman. All rights reserved.\n");
    
    if (! iswin3e())
        fail("This program requires Windows Enhanced mode");
    
    for (i=1; i<argc; i++)
        if ((argv[i][0] == '-') || (argv[i][0] == '/'))
        {
            char *arg = strupr(&argv[i][1]);
            if (strcmp(arg, "VERBOSE") == 0)
                verbose = 1;
            else
                fail("usage: vxdlist [-verbose] [vxd_name or #hex_id]");
        }
		else if (argv[i][0] == '#')
			sscanf(&argv[i][1], "%04X", &vxd_id);
		else
		{
			vxd_name = "        ";	// 8 spaces
			memcpy(vxd_name, argv[i], strlen(argv[i]));
		}
    
    return 0;   // okay to switch to pmode
}

void spaces(int x)
{
    char buf[80], *s=buf;
    while (x--) *s++ = ' ';
    *s = '\0';
    fputs(buf, stdout);
}

#ifdef DPMI_APP
#define CHECK_CTRL_C()  { if (ctrl_c_hit) fail("Ctrl-C detected"); }
#else
#define CHECK_CTRL_C()	/**/
#endif

void list_vxd_func_tab(DDB far *pddb)
{
	char buf[128];
	int i;
	char *s;
	DWORD far *p;
	DWORD far *srvc_tab = (DWORD far *) 
		map_linear(pddb->DDB_Service_Table_Ptr,
				   pddb->DDB_Service_Table_Size * sizeof(DWORD));
	if (! srvc_tab)
		fail("Can't map in DDB Service Table");

	if (Get_VxD_Desc(pddb->DDB_Req_Device_Number, buf, 127) == 0)
		puts(buf);	// VxD description string

	for (i=0, p=srvc_tab; i<pddb->DDB_Service_Table_Size; i++, p++)
	{
		CHECK_CTRL_C();
		printf("   %02X%04Xh @ %08lXh", pddb->DDB_Req_Device_Number, i, *p);
		if (Get_VxD_Proc_Name(pddb->DDB_Req_Device_Number, i, buf, 127)==0)
			printf("   %s", buf);
		printf("\n");
	}

	FREE_MAPPED_LINEAR(srvc_tab);
	printf("\n");
}
	
void display(DWORD vxd_lin)
{
    unsigned char name[9];
    static int did_banner = 0;
    
	DDB far *pddb;
	if (Get_VxD(vxd_lin, &pddb) != 0)
		fail("Can't get VxD");
	glob_ddb = pddb;	// for cleanup by fail()
	
    if (! did_banner)
    {
        did_banner = 1;
        puts("\n"
"Name      Vers   ID      DDB        Control    V86 API    PM API     #Srvc\n"
"--------  ----   -----   --------   --------   --------   --------   -----");
    }

    /* printf %8.8Fs should work, but doesn't seem to! */
    _fstrncpy(name, pddb->DDB_Name, 8);
    name[8] = '\0';
    printf("%s  ", name);
    
    printf("%u.%02u   ", 
        pddb->DDB_Dev_Major_Version, pddb->DDB_Dev_Minor_Version);
    
    if (pddb->DDB_Req_Device_Number) 
        printf("%04Xh", pddb->DDB_Req_Device_Number);
    else
        spaces(5);
    spaces(3);
    
    printf("%08lX   ", vxd_lin);
    printf("%08lX   ", pddb->DDB_Control_Proc);
    
    if (pddb->DDB_V86_API_Proc) printf("%08lX", pddb->DDB_V86_API_Proc);
    else spaces(8);
	putchar( (pddb->DDB_V86_API_CSIP) ? '*' : ' ');
    spaces(2);
        
    if (pddb->DDB_PM_API_Proc) printf("%08lX", pddb->DDB_PM_API_Proc);
    else spaces(8);
	putchar( (pddb->DDB_PM_API_CSIP) ? '*' : ' ');
    spaces(2);
    
    printf("%u\n", pddb->DDB_Service_Table_Size);
    if (verbose)
	{
		// Callback established first time an app calls 2f/1684/ID in
		// specified mode
		void (far *fp)();
		if (fp = pddb->DDB_V86_API_CSIP) printf("V86 callback @ %Fp\n", fp);
		if (fp = pddb->DDB_PM_API_CSIP)  printf("PM callback @ %Fp\n", fp);
		
		// VMM keeps VxD chain in Init order
		printf("Init order: %08lX\n", pddb->DDB_Init_Order);
		
		list_vxd_func_tab(pddb);
	}
		
	Free_VxD(pddb);
}

int pmode_main(int argc, char *argv[])
{
    DWORD vxd_lin, next;
    int err;
	
	if (vxd_id || vxd_name)
	{
		if (vxd_id && (err = Find_VxD_ID(vxd_id, &vxd_lin)))
			{ printf("Can't find #%04X\n", vxd_id); _dos_exit(2); }
		if (vxd_name && (err = Find_VxD(vxd_name, &vxd_lin)))
			{ printf("Can't find \"%s\"\n", vxd_name); _dos_exit(2); }
		display(vxd_lin);
		return 0;
	}
	
	if ((err = Get_First_VxD(&vxd_lin)) != 0)
		fail("Can't locate VxD chain");
	for (;;)
	{
		CHECK_CTRL_C();
		display(vxd_lin);
		if ((err = Get_Next_VxD(vxd_lin, &next)) == 0)
			vxd_lin = next;
		else if (err == ERROR_END_OF_CHAIN)
			break;	// successfully got to end of VxD chain
		else
			fail("Can't walk VxD chain");
	}
	
	return 0;
}

#ifndef DPMI_APP
// pretend it's a DPMI app
void main(int argc, char *argv[])
{
	if (v86_main(argc, argv) == 0)
		return pmode_main(argc, argv);
}
#endif

