/****************************************************************************
  WIN_MAIN.C -- Ray Valdes, August 1993.
  
 DBGTRACE.EXE


    This is a sample program that accompanies the article
    "Debugging Windows Programs" by Ray Valdes in the Nov 93 issue
    of Dr Dobbs Journal.

    The program here is a simple text editor that invokes Windows built-in 
    EDIT control to handle on-screen editing.  The editor is not a real
    program but is based on one of the sample apps provided with the
    C compiler.  This sample program does not even save files to disk,
    so it is purely for illustrative purposes.
    
    I modified the original sample code in order to show three things:

    1. Using my debug tracing facility, i.e., dbg_Trace() and friends.
       See files DBGTRACE.C and DBGTRACE.H for more information.

    2. Using the Avanti PinPoint trace facility, which is similar
       to my. Unfortunately, we can't include the Avanti .H, .DDL and .LIB
       modules here, so do not #define PINPOINT_ON because it
       won't compile with that option.

    3. Subclassing an edit control so as to trace message from it.
  
    Currently, there are make files for both Borland BCC3.1 and Microsoft
    C8 compiler (a.k.a. Visual C++).  However, the code is very vanilla
    C, so it should work with any compiler that can create Windows apps.
    
    To use with Microsoft C8, type: make_msc
    To use with Borland BCC3.1, type: make_bor
    
****************************************************************************/

#include "config.h"
#include "windows.h"
#include "std_defs.h"
#include "rc_defs.h"
#include "dbgtrace.h"

#undef PINPOINT_ON
#ifdef PINPOINT_ON
#include "pp_api.h"	/* to demonstrate PinPoint trace facility */
#else
#define PINPOINT(x)     /* this removes calls to PinPoint at compile time*/
#endif

/****************************************************************/
#define EXPORTED FAR PASCAL
#define PRIVATE static

#define APP_NAME "EditApp: "
/****************************************************************/

int PASCAL WinMain(HANDLE, HANDLE, LPSTR, int);
long EXPORTED MainWndProc(HWND, UINT, WPARAM, LPARAM);
BOOL EXPORTED AboutDlgProc(HWND, UINT, WPARAM, LPARAM);

/****************************************************************/

PRIVATE BOOL app_InitFirstInstance(HANDLE);
PRIVATE BOOL app_InitEveryInstance(HANDLE,int);

/****************************************************************/

PRIVATE HANDLE   theAppInstance;
PRIVATE HANDLE   theAppAccelTable;
PRIVATE HWND     theEditWindow;
PRIVATE HWND     theAppMainWindow;

#define SUBCLASS_EDITPROC
#ifdef SUBCLASS_EDITPROC
PRIVATE VOID subclass_editwnd(VOID);
LONG EXPORTED app_SubclassedWndProc(HWND,UINT,WPARAM,LPARAM);
#endif

/****************************************************************************
    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)
    PURPOSE: calls initialization function, processes message loop
****************************************************************************/
int PASCAL WinMain(
    HANDLE hInstance,
    HANDLE hPrevInstance,
    LPSTR lpCmdLine,
    int nCmdShow)
{
    MSG msg;

    dbg_Trace(APP_NAME "WinMain start");
    
#undef PINPOINT_ON
    PINPOINT(ppFopen("edit_ctl",FALSE));
    PINPOINT(ppBeginProgram()); //enabled by "#define PINPOINT_ON"

    if (!hPrevInstance)
    {
	dbg_Trace(APP_NAME "First-instance initialization");
	if (!app_InitFirstInstance(hInstance))
	    return  FALSE;
    }
    dbg_Trace(APP_NAME "Every-instance initialization");
    if (!app_InitEveryInstance(hInstance, nCmdShow))
        return  FALSE;

    while (GetMessage(&msg, NULL, NULL, NULL)) 
    {
	/* Only translate message if it is not an accelerator message */
        if (!TranslateAccelerator(theAppMainWindow, theAppAccelTable, &msg)) 
	{
            TranslateMessage(&msg);
            DispatchMessage(&msg); 
        }
    }
    PINPOINT(ppEndProgram());
    PINPOINT(ppFclose());
    dbg_Trace("WinMain end");
    return msg.wParam;
}

/****************************************************************/
PRIVATE BOOL app_InitFirstInstance(HANDLE hInstance)
{
    WNDCLASS  wc;
    BOOL result;
    
    PINPOINT(ppBeginFunction(" ","InitApp"," "));

    wc.style       = NULL;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra  = 0;
    wc.cbWndExtra  = 0;
    wc.hInstance   = hInstance;
    wc.hIcon       = LoadIcon(NULL, "AppIcon");
    wc.hCursor     = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
    wc.lpszMenuName  =  "AppMainMenu";
    wc.lpszClassName = "AppMainWindowClass";
    result = RegisterClass(&wc);
    if(!result)
    {
	PINPOINT(ppEndFunction("InitApp"));
	return FALSE;
    }
#ifdef DEBUG_TRACE_ENABLED
    result = dbg_RegisterDebugWindow(hInstance,0);
    if(!result)
    {
	PINPOINT(ppEndFunction("InitApp"));
	return FALSE;
    }
#endif    
    return TRUE;
}

/****************************************************************/
PRIVATE BOOL app_InitEveryInstance( 
    HANDLE          hInstance,
    int             nCmdShow)
{
    RECT            rect;
    int x,y;
    
    x = GetSystemMetrics(SM_CXSCREEN);
    y = GetSystemMetrics(SM_CYSCREEN);

    PINPOINT(ppBeginFunction(" ","InitInst"," " ));

    theAppInstance = hInstance;

    theAppAccelTable = LoadAccelerators(theAppInstance, "EditCntlAcc");

    theAppMainWindow = CreateWindow(
        "AppMainWindowClass",
        "EditCntl Sample Application",
        WS_OVERLAPPEDWINDOW,
        25,
	25,
	(x - 50), ((y * 2)/3)-25,
        NULL,
        NULL,
	theAppInstance,
        NULL
    );
    if (!theAppMainWindow)
    {
	PINPOINT(ppEndFunction("InitInst"));
        return  FALSE;
    }
    PINPOINT(ppPrintf("InitInst","theAppMainWindow=%d",theAppMainWindow));
    
    GetClientRect(theAppMainWindow, (LPRECT) &rect);

    /* Create a child window */

    theEditWindow = CreateWindow("Edit",
	NULL,
	WS_CHILD | WS_VISIBLE |
	ES_MULTILINE |
	WS_VSCROLL | WS_HSCROLL |
	ES_AUTOHSCROLL | ES_AUTOVSCROLL,
	0,
	0,
	(rect.right-rect.left),
	(rect.bottom-rect.top),
	theAppMainWindow,
	IDC_EDIT,                          /* Child control i.d. */
	theAppInstance,
	NULL
	);

    if (!theEditWindow) 
    {
	DestroyWindow(theAppMainWindow);
	PINPOINT(ppEndFunction("InitInst"));
	return FALSE;
    }
#ifdef DEBUG_TRACE_ENABLED
    {
	HWND hWnd = dbg_CreateDebugWindow(theAppInstance,
			    theAppMainWindow,0);
	if(!hWnd)
	{
	    DestroyWindow(theEditWindow);
	    DestroyWindow(theAppMainWindow);
	    PINPOINT(ppEndFunction("InitInst"));
	    return FALSE;
	}	    
    }
#endif
#ifdef SUBCLASS_EDITPROC
    subclass_editwnd();
#endif
    ShowWindow(theAppMainWindow, nCmdShow);
    UpdateWindow(theAppMainWindow);
    PINPOINT(ppEndFunction("InitInst"));
    return  TRUE;
}
/****************************************************************/

long EXPORTED MainWndProc(
    HWND	hWnd,
    UINT	message,
    WPARAM	wParam,
    LPARAM	lParam)
{
    switch (message) 
    {
	case WM_LBUTTONDOWN:
	    dbg_Trace(fmtStr(APP_NAME "mouse-up event h=%d v=%d",
			 (short) LOWORD(lParam), (short) HIWORD(lParam)));
	    break;
	    
	case WM_COMMAND:
	    PINPOINT(ppPrintf("MainWndProc","CMD=%d",wParam));
            switch (wParam) 
	    {
                case IDM_ABOUT:
		{
		    FARPROC lpProcAbout;
    
                    lpProcAbout = MakeProcInstance(AboutDlgProc, 
						theAppInstance);
                    DialogBox(theAppInstance, "AboutBox", hWnd, lpProcAbout);
                    FreeProcInstance(lpProcAbout);
		} break;

                /* file menu commands */
                case IDM_NEW:
                case IDM_OPEN:
                case IDM_SAVE:
                case IDM_SAVEAS:
                case IDM_PRINT:
                    MessageBox (
                          GetFocus(),
                          "Command not implemented",
                          "EditCntl Sample Application",
                          MB_ICONASTERISK | MB_OK);
                    break;  
                case IDM_EXIT:
                    DestroyWindow(hWnd);
                    break;
    
                /* edit menu commands */
                case IDM_UNDO:
                case IDM_CUT:
                case IDM_COPY:
                case IDM_PASTE:
                case IDM_CLEAR:
                    MessageBox (
                          GetFocus(),
                          "Command not implemented",
                          "EditCntl Sample Application",
                          MB_ICONASTERISK | MB_OK);
                    break;  
                case IDC_EDIT:
		    PINPOINT(ppPrintf("MainWndProc","msg from IDC_EDIT"));
                    if (HIWORD (lParam) == EN_ERRSPACE) 
		    {
                        MessageBox (
                              GetFocus ()
                            , "Out of memory."
                            , "EditCntl Sample Application"
                            , MB_ICONHAND | MB_OK
                        );
                    }
                    break;
            } 
            break;

        case WM_SETFOCUS:
	    dbg_Trace(APP_NAME "MainWndProc: WM_SETFOCUS");
            SetFocus (theEditWindow);
            break;

        case WM_SIZE:
            MoveWindow(theEditWindow, 
			0, 0, LOWORD(lParam), HIWORD(lParam), 
			TRUE);
            break;

	case WM_DESTROY:
	    PostQuitMessage(0);
	    break;
	default:
	    return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}

/****************************************************************/
BOOL EXPORTED AboutDlgProc(
    HWND hDlg,
    UINT message,
    WPARAM wParam,
    LPARAM lParam)
{
    switch (message) 
    {
	case WM_INITDIALOG:   return  TRUE;

	case WM_COMMAND:
	    if (   wParam == IDOK
                || wParam == IDCANCEL) 
	    {
		    EndDialog(hDlg, TRUE);
		    return  TRUE;
	    }
	    break;
    }
    return  FALSE;
}
/****************************************************************/

#ifdef SUBCLASS_EDITPROC
static FARPROC theSavedEditWndProc;

PRIVATE VOID subclass_editwnd(void)
{
    theSavedEditWndProc = (FARPROC) 
	GetWindowLong(theEditWindow, GWL_WNDPROC);
    
    SetWindowLong(theEditWindow, GWL_WNDPROC,
	(LONG) MakeProcInstance(
		(FARPROC) app_SubclassedWndProc,theAppInstance));
}

LONG EXPORTED app_SubclassedWndProc(HWND hWnd,UINT msg,
    WPARAM wParam, LPARAM lParam)
{
    if(msg==WM_LBUTTONUP)
    {
	dbg_Trace(fmtStr(APP_NAME "EditCtl: mouse-up at h=%d v=%d",
	    (short) LOWORD(lParam), (short) HIWORD(lParam)));
    }
    else if(msg==WM_CHAR)
    {
	dbg_Trace(fmtStr(APP_NAME "EditCtl: char = 0x%x <%c>",
	    wParam,wParam));
    }
    return CallWindowProc(theSavedEditWndProc, hWnd,msg,wParam,lParam);
}
#endif
