;---------------------------------------------------------------    
;intisr - Interrupt Service Routines for Int Monitor           |
;--------------------------------------------------------------|
;Copyright 1991, 1993 ASMicro Co.                              |
;--------------------------------------------------------------|
; 09/25/93                      Rick Knoblaugh                 |
;--------------------------------------------------------------|
;include files                                                 |
;---------------------------------------------------------------    
                .386P
                include intequ.inc
                include intmac.inc
                include intstruc.inc


data            segment para public 'data16' use16
                extrn   old_user_int:DWORD 
                extrn   store_buf:byte
                extrn   store_buf_end:ABS
                extrn   store_ptr:word
                extrn   display_loc:word
                extrn   intm_ctrl:BYTE

data            ends                   
           
gdt_seg         segment para public 'data16' use16
                extrn   sel_databs:byte      
                extrn   sel_video:byte
                extrn   sel_data:byte
                extrn   sel_tss_alias:byte
gdt_seg         ends                              

isrcode         segment para public 'icode16' use16
                assume cs:isrcode, ds:nothing, es:nothing
;--------------------------------------------------------------
;PUBLICS                                                      |
;--------------------------------------------------------------
                public  user_int_isr     
                public  int_0
                public  int_1
                public  int_2
                public  int_3
                public  int_4
                public  int_5
                public  int_6
                public  int_7
                public  except_8
                public  except_9
                public  except_0ah
                public  except_0bh
                public  except_0ch
                public  except_0dh
                public  except_0eh
                public  except_0fh
                public  int_20h
                public  int_21h
                public  int_22h
                public  int_23h
                public  int_24h
                public  int_25h
                public  int_26h
                public  int_27h

                public  int_70h
                public  int_71h
                public  int_72h
                public  int_73h
                public  int_74h
                public  int_75h
                public  int_76h
                public  int_77h
                public  display_it

        irp     z, <0, 1, 2, 3, 4, 5, 6, 7>
        DOINT   &z
        endm

        irp     z, <8, 9, 0ah, 0bh, 0ch, 0dh, 0eh, 0fh>
        DOEXCP  &z
        endm

        irp     z, <20h, 21h, 22h, 23h, 24h, 25h, 26h, 27h>
        DOEXCPH &z 
        endm

        irp     z, <70h, 71h, 72h, 73h, 74h, 75h, 76h, 77h>
        DOINT   &z 
        endm     


;--------------------------------------------------------------
;except_handler - Process as follows:                         |
;                                                             |
;                      Int 0dh - Go look for software int.    |
;                                                             |
;                    Any other                                |
;                    exception - go display exception number  |
;                                and halt.                    |
;--------------------------------------------------------------
except_handler  proc    near
                mov     bp, sp
                cmp     [bp].e_pushed_int , GEN_PROT_EXCEP
                je      gen_prot_isr
                mov     ax, [bp].e_pushed_int ;int in ax, go display
                jmp short fatal_error
except_handler  endp

fatal_error     proc    near
                call    display_it
                jmp     $
fatal_error     endp

;--------------------------------------------------------------
;display_it - Display hex number on screen at next display    |
;             offset.                                         |
;                                                             |
;             Enter:  number in AX                            |
;                     processor in protected mode             |
;                                                             |
;             All registers saved                             |
;--------------------------------------------------------------
display_it      proc    near
                pusha
                push    ds
                push    es

                mov     dx, offset gdt_seg:sel_data
                mov     ds, dx                  ;get our data segment
                assume  ds:data
                mov     dx, offset gdt_seg:sel_video
                mov     es, dx                  ;and video segment
                mov     di, display_loc
                xchg    al, ah                  ;print MSB first
                call    put_hex_digit
                xchg    al, ah                  ;get LSB
                call    put_hex_digit
                add     di, 2                   ;past space and attribute
                cmp     di, VID_PAGE_SIZE
                jb      display_i100
                xor     di, di                  ;back to start of page
display_i100:
                mov     display_loc, di
                pop     es
                pop     ds
                popa
                ret
display_it      endp        

put_hex_digit   proc    near
                push    ax
                mov     cx, 2                   ;2 digits in al
                mov     ah, al
put_hex_100:
                shr     al, 4
                cmp     al, 9
                ja      put_hex_200
                add     al, '0'
                jmp     short put_hex_300
put_hex_200:
                add     al, 'A' - 10
put_hex_300:
                cld
                stosb
                inc     di                      ;past attrib
                mov     al, ah
                shl     al, 4        
                loop    put_hex_100
                pop     ax
                ret
put_hex_digit   endp                  

;--------------------------------------------------------------
;pass_thru - This procedure is JMPed to by any interrupt      |
;            handler which wishes to pass control to the      |
;            original ISR per the interrupt vector table.     |
;                                                             |
;            Enter:                                           |
;                  See stack_area struc for stack layout      |
;                  Any error code has been removed from stack.|
;                  EIP on stack has been adjusted if          |
;                  necessary.                                 |
;--------------------------------------------------------------
pass_thru       proc    near
                mov     bp, sp
                pushad
                mov     cx, offset gdt_seg:sel_databs 
                mov     ds, cx                  ;address all base memory
                call    adjust_ustack           ;adjust user stack
;
;returns with [esi][edx] pointing to user stack area
;
                mov     cx, [bp].s_cs           ;put on user cs
                mov     [esi][edx].user_cs, cx
                mov     ecx, [bp].s_eip         ;put on ip
                mov     [esi][edx].user_ip, cx
                movzx   ebx, [bp].s_pushed_int  ;get int number
                call    determin_store          ;store if requested
                movzx   ecx, [ebx * 4].d_offset  ;offset portion
                mov     [bp].s_eip, ecx                     
                mov     cx, [ebx * 4].d_segment  ;segment portion
                mov     [bp].s_cs, cx
                call    determin_brk            ;gen break point if requested
                popad
                add     sp, 2                   ;get rid of int number
                pop     bp
                iretd                  
pass_thru       endp        

;--------------------------------------------------------------
;adjust_ustack - Make room for flags and a far address on the |
;               user stack.  Place flags from PL0 stack       |
;               on user stack.  The caller of this routine    |
;               will then place the far address on the user   |
;               stack.                                        |
;                                                             |
;            Enter:                                           |
;                  bp=base of PL0 stack                       |
;                  ds=absolute zero selector                  |
;             Exit:                                           |
;                 esi=linear address of user stack segment    |
;                 edx=adjusted user sp                        |
;                                                             |
;            No registers saved.                              |
;--------------------------------------------------------------
adjust_ustack   proc    near
                movzx   esi, [bp].s_ss          ;user stack
                shl     esi, 4                  ;make linear
                mov     edx, [bp].s_esp         ;user stack pointer
                sub     edx, 6                  ;flags, cs, ip
                mov     [bp].s_esp, edx         ;adjust it
                mov     ecx, [bp].s_eflags      ;put on flags
                mov     [esi][edx].user_flags, cx
;
;Change flags on stack so that original ISR will be entered with 
;interrupts cleared and trap flag cleared to be consistent with their
;state upon entering an ISR (the normal way).
;
                and     ecx, not ( (mask resumef) OR (mask inter) OR (mask trapf)) 
                mov     [bp].s_eflags, ecx      ;put back flags
                ret
adjust_ustack   endp


;--------------------------------------------------------------
;determin_brk - Determine if a break point is to be generated |
;               for this interrupt.  If it is, adjust PL0     |
;               stack so that IRETD will go to desired        |
;               break point interrupt (NMI, int 3, or int 1). |
;               In this case, INT ISR will also be placed on  |
;               the user stack.                               |
;                                                             |
;            Enter:                                           |
;                  bx=interupt number.                        |
;                  ax=ax when int occurred                    |
;                  bp=base of PL0 stack                       |
;                                                             |
;             Exit:                                           |
;                  user stack adjusted                        |
;                                                             |
;            No registers saved.                              |
;--------------------------------------------------------------
determin_brk    proc    near
                mov     cx, offset gdt_seg:sel_data           
                mov     ds, cx
                mov     fs, cx

                cmp     intm_ctrl.enable_break, TRUE  ;do break points?
                jne     determin_b999           ;if not, exit

;
;Store number of interrupt that is occurring.  This could possibly be
;used by a custom user breakpoint routine which might get control
;on more than one specified interrupt (i.e. if you have written a custom
;interrupt routine and have specified that it get control during various
;interrupts, your routine can look at this byte and determine which
;interrupt caused the "break point").
;
;
                mov     intm_ctrl.int_now, bl   

                mov     si, offset intm_ctrl.brk_each
                call    ck_bit_map              ;gen break point for this int
                jnc     determin_b999           ;if not, exit


;determine if user has indicated that ah/al be equal to a specified 
;value as condition for generating a break point.
;
                call    ck_cond_ax              
                jnz     determin_b999           ;if condition not met

determin_b300: 
                mov     cx, offset gdt_seg:sel_databs 
                mov     ds, cx                  ;address all base memory
                call    adjust_ustack
;
;returns with [esi][edx] pointing to user stack area
;

                mov     cx, [bp].s_cs           ;get cs of int
                mov     [esi][edx].user_cs, cx  ;put on user stack
                mov     ecx, [bp].s_eip         ;get eip of int
                mov     [esi][edx].user_ip, cx  ;put on user stack

;
;make address to which IRETD will transfer control, the interrupt
;with which to enter the debugger (e.g. int 1, or NMI)
;
                movzx   ebx, fs:intm_ctrl.brk_action
                movzx   eax, [ebx * 4].d_offset  ;offset portion
                mov     [bp].s_eip, eax                     
                mov     ax, [ebx * 4].d_segment  ;segment portion
                mov     [bp].s_cs, ax

determin_b999:
                ret
determin_brk    endp


;--------------------------------------------------------------
;determin_store - Determine if this interrupt is to be stored.|
;                 If it is, store it.                         |
;                                                             |
;            Enter:                                           |
;                  bx=interupt number.                        |
;                  ax=ax value when int occurred              |
;   ds register saved.                                        |
;--------------------------------------------------------------
determin_store  proc     near
                push    ds

                mov     cx, offset gdt_seg:sel_data           
                mov     ds, cx

                cmp     intm_ctrl.enable_log, TRUE  ;log any?
                jne     determin_s999           ;if not, exit

                mov     si, offset intm_ctrl.log_each
                call    ck_bit_map              ;log this int?
                jnc     determin_s999           ;if not, exit


;determine if user has indicated that ah/al be equal to a specified 
;value as condition for logging.
;
                call    ck_cond_ax              
                jnz     determin_s999           ;if condition not met

determin_s300: 

                mov     si, store_ptr
                mov     [si].buf_int, bl       ;store int number 
                mov     [si].buf_ax, ax        ;store ax value   

                add     si, size buf_record     ;advance buf ptr

                cmp     si, store_buf_end - size buf_record  ;at end?
                jb      determin_s500   
                mov     si, offset store_buf 
determin_s500: 
                mov     store_ptr, si           ;save new ptr


determin_s999: 
                pop     ds
                ret
determin_store  endp
;--------------------------------------------------------------
;ck_cond_ax - Determine if it has been specified that ah/al   |
;             be equal to a certain value as a condition for  |
;             logging and/or generating a break point.        |
;                                                             |
;            Enter:                                           |
;                 ebx=interupt number.                        |
;                  ax=ax value when int occurred              |
;                  ds=selector for our local data             |
;             Exit:                                           |
;                  zero flag set if ax matches condition or   |
;                  no condition is configured.                |
;                                                             |
;            No registers saved.                              |
;--------------------------------------------------------------
ck_cond_ax      proc    near
                mov     dx, intm_ctrl.values_ax[ebx*2] ;get value to cmp
                cmp     dh, DONT_COMP           ;cmp ah portion?
                je      ck_cond200              ;if not, continue

                cmp     dh, ah                  ;meets user criterion?
                jne     ck_cond999              ;if not, report it
ck_cond200: 
                cmp     dl, DONT_COMP           ;cmp al portion?
                je      ck_cond999              ;if not, exit      

                cmp     dl, al                  ;meets user criterion?
                                                ;if not, nz returned
ck_cond999: 

                ret
ck_cond_ax      endp        


;--------------------------------------------------------------
;gen_prot_isr - JMP here if int 0dh.  Process as follows:     | 
;                                                             |
;               Look for software int.  If found, go route to |
;               appropriate ISR.                              |
;                                                             |
;               If other than software int, go display 0dh    |
;               and halt.                                     |
;                                                             |
;--------------------------------------------------------------
gen_prot_isr    proc    near
                pushad        
                mov     bx, offset gdt_seg:sel_databs 
                mov     ds, bx
                movzx   ebx, [bp].e_cs  ;get cs of user instruction
                shl     ebx, 4          ;make linear
                add     ebx, [bp].e_eip ;add ip 
                mov     bx, [ebx]       ;get bytes at cs:ip
                cmp     bl, INT_OPCODE
                jne     gen_prot100
                inc     [bp].e_eip      ;get past the 0cdh
gen_prot050:                
                inc     [bp].e_eip 

                cmp     bh, 15h                 ;int 15?
                jne     short gen_prot095
                cmp     ah, 89h                 ;request for protected mode?
                jne     short gen_prot090
                                                ;if so, can't allow
                bts     [bp].e_eflags, carry    ;set carry
                popad
                jmp     gen_prot299             ;and return
gen_prot090:
                cmp     ah, 87h                 ;request for extended move? 
                jne     short gen_prot095
                call    emulate_blk_mov         ;if so, we must do it
                popad
                mov     ah, 0                   ;default to success
                jnz     gen_prot299             ;exit if success
                mov     ah, 3                   ;indicate a20 gate failed
                jmp     gen_prot299             ;and return

gen_prot095:

;
;Adjust stack so that error code goes away and int number retrieved from
;instruction goes in spot on stack where pushed int number is (for stacks
;with no error code).  Stack will be the way pass_thru routine likes it.
;
                mov     ax, bx
                mov     bx, [bp].e_pushed_bp
                shl     ebx, 16         ;get into high word
                mov     bl, ah          ;interrupt number
                mov     [bp].e_errcode, ebx
                popad       
                add     sp, 4           ;error code gone
                jmp     pass_thru
gen_prot100:
                cmp     bl, INT3_OPCODE
                jne     gen_prot150
                mov     bh, 3           ;interrupt 3
                jmp     short gen_prot050         
gen_prot150:
;
;Also need to add code here to check for a few other exceptions (e.g.  
;the HLT instruction).
;
                mov     ax, [bp].e_cs   ;get cs of user instruction
                call    display_it
                mov     eax, [bp].e_eip ;add ip 
                call    display_it
                popad      
                mov     ax, [bp].e_pushed_int
                jmp     fatal_error

gen_prot299:

                add     sp, 2                   ;int number pushed
                pop     bp
                add     sp, 4                   ;error code
                iretd
gen_prot_isr    endp

;--------------------------------------------------------------
;emulate_blk_mov - Called from gen_prot_isr when an int 15h   |
;                  function 87h (extended memory block move)  |
;                  is detected.  Perform the function and     |
;                  return.                                    |
;                                                             |
;                  Enter:                                     |
;                         es on PL0 stack and SI form a       |
;                               pointer to user's GDT         |
;                                                             |
;                            cx=number of words to move       |
;                                                             |
;                   Exit: Indicate success by affecting flags |
;                         image on PL0 stack.                 |
;                                                             |
;                         Zero flag set if error (for now     |
;                         the only error checked for is       |
;                         an A20 failure - other error        |
;                         checks should be added).            |
;                                                             |
;--------------------------------------------------------------
emulate_blk_mov proc    near
                mov     ax, offset gdt_seg:sel_databs ;absolute zero base
                mov     ds, ax
                mov     es, ax
;First, determine if a20 is already set
                mov     ebx, 100000h            ;1 meg                
                xor     dl, dl                  ;use dl for a20 flag
                mov     eax, [ebx]              ;get contents at "1 meg"
                mov     dword ptr [ebx], 0ffffffffh       ;move test value there
                xor     edi, edi
                cmp     dword ptr [edi], 0ffffffffh       ;did it wrap to zero
                mov     [ebx], eax              ;restore value        
                je      short emulate_b025      ;if so, a20 not set
                mov     dl, 1                   ;indicate a20 already set
                jmp     short emulate_b100     
emulate_b025:

                mov     al, 1                   ;set a20 function
                call    do_a20
                jnz     short emulate_b100
emulate_b050:
                bts     [bp].e_eflags, carry    ;indicate error
                btr     [bp].e_eflags, zero     
                ret
emulate_b100:
                movzx   ebx, [bp].e_es   ;get user's es
                shl     ebx, 4           ;make linear
                movzx   esi, si

                mov     eax, 00ffffffh   ;24 bit address only
;get destination address
                mov     edi, dword ptr [ebx][esi].ex_mdest.seg_base_low   
                and     edi, eax         ;24 bit address only

;get source address
                mov     esi, dword ptr [ebx][esi].ex_msource.seg_base_low   
                and     esi, eax         ;24 bit address only

                cld

                movzx   ecx, cx         ;word count
                shr     cx, 1           ;convert to dword count
                jnc     short emulate_b500  ;if evenly div by 4

                db      67h             ;force 32 bit addressing
                movsw                   ;move odd one
                jcxz    emulate_b600    ;if only 1 word was requested
emulate_b500:
                db      67h             ;force 32 bit addressing
                rep     movsd           ;move dwords
emulate_b600:
                test    dl, 1           ;a20 already set when entered?
                jnz     short emulate_b999

                sub     al, al                  ;clear a20 function
                call    do_a20
                jz      short emulate_b050      ;if error

emulate_b999:
                btr     [bp].e_eflags, carry    ;indicate success
                bts     [bp].e_eflags, zero     
                ret
emulate_blk_mov endp

;--------------------------------------------------------------
;do_a20 -  Turn A20 on or off.                                |
;                                                             |
;         Enter:                                              |
;                al=1 turn it on.                             |
;                al=0 turn it off.                            |
;                                                             |
;          Exit: Zero set if error                            |
;                                                             |
;     No registers saved.                                     |
;--------------------------------------------------------------
do_a20          proc    near
                mov     ah,0dfh        ;a20 on
                or      al,al
                jnz     short do_a20_100 
                mov     ah,0ddh        ;a20 off
do_a20_100:
                call    key_wait
                jz      short do_a20_999
                mov     al,0d1h
                out     64h,al
                call    key_wait
                mov     al,ah
                out     60h,al
                call    key_wait
                jz      short do_a20_999        
                mov     al,0ffh
                out     64h,al
                call    key_wait
do_a20_999:                      
                ret
do_a20          endp

key_wait        proc    near
                push    cx
                xor     cx,cx             ;max time out
key_w100:
                dec     cx
                jz      short key_w999
                in      al,64h
                and     al,2
                jnz     short key_w100    ;loop if keyboard controller not ready
key_w999:       or      cx,cx             ;return with zero set if error
                pop     cx
                ret
key_wait        endp

;--------------------------------------------------------------
;ck_bit_map - Determine if corresponding bit in bitmap for    | 
;             a given interrupt is set.                       |
;                                                             |
;             Enter:  ds:si points to bitmap                  |
;                     bx = interrupt                          |
;                                                             |
;              Exit:  Carry set if bit is set                 |
;                                                             |
;  All registers saved.                                       |
;--------------------------------------------------------------
ck_bit_map      proc    near
                push    bx
                push    dx
                mov     dx, bx                  ;int                
                and     dx, 7                   ;get non byte boundary
                shr     bx, 3                   ;int/8 
                bt      [si+ bx], dx            ;first bit position
                pop     dx
                pop     bx
                ret
ck_bit_map      endp        

;--------------------------------------------------------------
;user_int_isr - Allow user to control logging and breakpoint  | 
;               actions.  Process requests as follows:        |
;                                                             |
;    Function                                                 |
;   Enter with                                                |
;                                                             |
;      ah=0     Return far pointer to storage buffer          |
;               Exit:  dx:bx far pointer to buffer            |
;                                                             |
;      ah=1     Return far pointer to structure which         |
;               controls all logging/breakpoint activity.     |
;               Exit:  dx:bx far pointer to structure         |
;                                                             |
;      ah=2     Enable all logging/breakpoint actions         |
;               Enter:  al=0 disable logging of all ints      |
;                       al=1 enable logging of all ints       |
;                       dl=0 disable all breakpoint actions   |
;                       dl=1 enable all breakpoint actions    |
;                                                             |
;      ah=3     Indicate that an interrupt should be logged   |
;               and/or generate a breakpoint.                 |
;               Enter:  al=int number                         |
;                       dl=0 don't log this int               |
;                       dl=1 log this int                     |
;                       bh=0 don't generate a breakpoint      |
;                       bh=1 generate a break point           |
;                       bl=breakpoint int to generate         |
;                       cx=qualifier for logging/breakpoints  |
;                          to be contained in ax (0ffffh if   |
;                          none).                             |
;                                                             |
;                                                             |
;                                                             |
;     All registers saved.                                    |
;--------------------------------------------------------------
user_int_isr    proc    near
                push    ax
                push    esi
                push    ds

                push    data
                pop     ds                      ;get ds=local data
                assume  ds:data

                cmp     ah, 3                   ;enable/disable int?
                je      user_int300
                ja      user_int900             ;invalid request
                cmp     ah, 1 
                ja      user_int200             ;enable/disable all ints

                mov     dx, data                ;segment for ptrs
                mov     bx, offset intm_ctrl    ;return control structure ptr
                je      user_int900             ;return control ptr 

                mov     bx, offset store_buf    ;return storage ptr
                jmp     short user_int900
user_int200:                
                cmp     al, 1                   ;only 0,1 valid
                ja      user_int900
                cmp     dl, 1                   ;only 0,1 valid
                ja      user_int900
                mov     intm_ctrl.enable_log, al   
                mov     intm_ctrl.enable_break, dl   
                jmp     short user_int900
user_int300:                
                sub     ah, ah
                movzx   esi, ax                 ;get int number
                mov     intm_ctrl.values_ax[esi * 2], cx ;store cmp value
                shr     si, 3                   ;get bitmap offset
                and     al, 7                   ;get bit number
                bts     intm_ctrl.log_each[si], al ;default to log
                or      dl, dl                  ;user wants log?
                jnz     user_int400        
                btr     intm_ctrl.log_each[si], al ;if not, turn off
user_int400:
                bts     intm_ctrl.brk_each[si], al ;default to break action
                or      bh, bh                  ;user wants breakpoint?
                jnz     user_int500                     
                btr     intm_ctrl.brk_each[si], al ;if not, turn off
user_int500:
                mov     intm_ctrl.brk_action, bl ;action when break
user_int900:
                pop     ds
                pop     esi
                pop     ax
                iret
user_int_isr    endp        
isrcode         ends
                end
