/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
/*
	The editor operates by associating an internal object with a
	window.  The association takes the form of a sequence of logical
	lines.  The total number of lines is available to the editor.

	Each line is numbered, from 0 through the number of lines - 1.
	The editor acts by requesting the current value of a logical line,
	which is returned as a null-terminated string of bytes.
	The editor then formats the line into physical lines for display 
	to the window system.

	The editor will request enough logical lines to fill the current
	screen.  The editor will maintain a cursor position within the
	current window.  That position corresponds in turn to one of the
	displayed logical lines, which becomes the 'active line'.

	Edit events modify the current active line in various ways, such
	as adding or deleting characters, splitting a line or deleting
	it altogether.
 */
include	console, string;
include	forms;
include window;
include	sound;
include	regexp;
include	file;
include	filesys;
include	error;

filePos:	public	type	{
	public:

	line:	lineNum;
	column:	int;
	offset:	int;
	lpos:	textPos;

isEqual:	(x: * filePos) boolean =
	{
	if	(line == x->line &&
		 column == x->column)
		return TRUE;
	else
		return FALSE;
	}

	};

findAction:	type	char = {
	findCurrentColumn,
	findEndOfLine,
	adjustNewCharacter,
	adjustNewline,
	adjustDeleteCharacter,
	adjustDeleteNewline,
	adjustDeleteLine,
	};

EditMap:	public	keyMap;

editor:	public	type	inherit	viewer {
	public:

	cursor:		filePos;
	textColor:	char;
	buffer:		* editBuffer;

constructor:	(titl: [:] char, sz: point, mm: * menu) =
	{
	super constructor(titl, sz, mm);
	scrap = 0;
	textColor = WHITE;
	}

connect:	(buf: * editBuffer) =
	{
	buffer = buf;
	}

formatLine:	dynamic	(ln: int, colorBar: char) =
	{
	i:		int;
	col:		int;
	k:		int;
	cp:		* char;
	p:		point;
	x:		int;
	ch:		char;
	attr:		char;
	outp:		ref colorChar;
	lineBuffer:	[80] colorChar;

	cp = buffer fetchLine(ln, 0);
	if	(cp == 0){
		super setColor(textColor);
		memSet(lineBuffer, textColor, sizeof lineBuffer);
		lineBuffer[0].character = '\xb0';
		for	(i = 1; i < size.x; i++)
			lineBuffer[i].character = ' ';
		write([ 0, ln - display.line ], lineBuffer[:size.x]);
		return;
		}
	if	(colorBar)
		memSet(lineBuffer, colorBar, sizeof lineBuffer);
	else if	(ln < startBlock.line ||
		 ln > endBlock.line)
		memSet(lineBuffer, textColor, sizeof lineBuffer);
	else	{
		j:		int;

		j = COLOR(HIGH|YELLOW, BLUE);
		memSet(lineBuffer, j, sizeof lineBuffer);
		col = endBlock.column - display.column;
		if	(ln == endBlock.line &&
			 col < size.x){
			if	(col < 0)
				col = 0;
			j = col;
			i = (size.x - j) * sizeof colorChar;
			outp = &lineBuffer[j];
			memSet(outp, textColor, i);
			}
		col = startBlock.column - display.column;
		if	(ln == startBlock.line &&
			 col > 0){
			i = col * sizeof colorChar;
			memSet(lineBuffer, textColor, i);
			}
		}
	col = 0;
	outp = lineBuffer;
	while	(col < display.column){
		ch = *cp;
		if	(ch == '\n' ||
			 ch == EOF_MARK)
			break;
		cp++;
		if	(ch == 0xFF){
			ch = *cp;
			cp++;
			if	(ch == 'c')
				cp++;
			}
		if	(ch == '\t'){
			x2:	int;

			x2 = (col + 8) & ~7;
			if	(x2 <= display.column)
				col = x2;
			else	{
				col = display.column;
				cp--;
				break;
				}
			}
		else if	(ch == '\r')
			;
		else
			col++;
		}

	x = 0;
	do	{
		ch = *cp;
		if	(ch == '\n' ||
			 ch == 0 ||
			 ch == EOF_MARK){
			while	(x < size.x){
				outp->character = ' ';
				outp++;
				x++;
				}
			break;
			}
		cp++;
		if	(ch == 0xFF){
			ch = *cp;
			cp++;
			if	(ch == 'c'){
				ch = *cp;
				i = (size.x - x) * sizeof colorChar;
				memSet(outp, ch, i);
				cp++;
				}
			continue;
			}
		if	(ch == '\t'){
			x2:	int;

			x2 = (col + 8) & ~7;
			while	(col < x2){
				outp->character = ' ';
				outp++;
				x++;
				col++;
				}
			}
		else if	(ch == '\r')
			;
		else	{
			outp->character = ch;
			outp++;
			x++;
			col++;
			}
		}
		while	(x < size.x);
	write([ 0, ln - display.line ], lineBuffer[:size.x]);
	}

centerCursor:	() =
	{
	centerDisplay();
	findCursorColumn(findCurrentColumn);
	}

changeLine:	(ln: lineNum) =
	{
	if	(ln < display.line)
		return;
	if	(ln >= display.line + size.y)
		return;
	lineStatus[ln - display.line] = -1;
	}

redraw:	dynamic	() =
	{
	memSet(&lineStatus, -1, sizeof lineStatus);
	updateCursor();
	repaintDisplay();
	}

setScrap:	(ns: * editBuffer) =
	{
	scrap = ns;
	}

close:	dynamic () int =
	{
	return super close();
	}

localKeys:	dynamic	() * keyMap =
	{
	return(&EditMap);
	}

beforeKeyPressed:	dynamic	() =
	{
	repaintDisplay();
	}

afterKeyPressed:	dynamic	() =
	{
	clearMessage();
	}

/*
	This function resets the state of the editor to be at the top of the
	file with no block active.  Then draw the buffer contents.
 */
resetEditor:	dynamic	() =
	{
	display.line = 0;
	display.column = 0;
	startBlock = [ 0, 0, 0, 0 ];
	endBlock = [ 0, 0, 0, 0 ];
	cursor = [ 0, 0, 0, 0 ];
	desiredColumn = 0;
	redraw();
	}

upArrow:	dynamic	() =
	{
	if	(cursor.line == 0){
		topAlarm();
		return;
		}
	cursor.line--;
	fixupColumn();
	if	(cursor.line < display.line)
		refreshDisplay(display.line - 1);
	else
		updateCursor();
	}

downArrow:	dynamic	() =
	{
	p:	point;

	if	(cursor.line == buffer lineCount() - 1){
		bottomAlarm();
		return;
		}
	else	{
		cursor.line++;
		fixupColumn();
		if	(cursor.line == display.line + size.y)
			refreshDisplay(display.line + 1);
		else
			updateCursor();
		}
	}

upScreen:	dynamic	() =
	{
	p:	point;
	i:	int;

	p = size;
	if	(cursor.line >= p.y)
		cursor.line -= p.y;
	else	{
		topAlarm();
		cursor.line = 0;
		}
	if	(display.line >= p.y)
		i = display.line - p.y;
	else
		i = 0;
	fixupColumn();
	refreshDisplay(i);
	}

downScreen:	dynamic	() =
	{
	p:		point;
	lastLine:	int;
	i:		int;

	p = size;
	lastLine = buffer lineCount();
	cursor.line += p.y;
	if	(cursor.line >= lastLine){
		cursor.line = lastLine - 1;
		bottomAlarm();
		}
	if	(p.y >= lastLine)
		i = 0;
	else if	(lastLine - cursor.line < p.y)
		i = lastLine - p.y;
	else
		i = display.line + p.y;
	fixupColumn();
	refreshDisplay(i);
	}

beginLine:	dynamic	() =
	{
	desiredColumn = cursor.column = 0;
	cursor.offset = 0;
	if	(display.column){
		display.column = 0;
		redraw();
		}
	updateCursor();
	}

endLine:	dynamic	() =
	{
	findCursorColumn(findEndOfLine);
	}

rightArrow:	dynamic	() =
	{
	cp:	* char;
	i:	int;
	ch:	char;

	cp = buffer fetchLine(cursor.line, cursor.offset);
	if	(cp == 0){
		bottomAlarm();
		return;
		}
	if	(*cp == '\r')
		cp++;
	if	(*cp == EOF_MARK)
		bottomAlarm();
	else if	(*cp == '\n')
		return;
	else	{
		cursor.offset++;
		findCursorColumn(findCurrentColumn);
		}
	}

leftArrow:	dynamic	() =
	{
	i:	int;

	if	(cursor.offset > 0){
		cursor.offset--;
		findCursorColumn(findCurrentColumn);
		}
	}

dragUp:		dynamic	() = 
	{
	if	(cursor isEqual(&startBlock)){
		if	(cursor.offset)
			changeLine(cursor.line);
		upArrow();
		startBlock = cursor;
		changeLine(cursor.line);
		}
	else if	(cursor isEqual(&endBlock) &&
		 isBlockDefined()){
		if	(cursor.offset)
			changeLine(cursor.line);
		upArrow();
		endBlock = cursor;

			/*
			   This case covers what happens when backing up over
			   the starting point of the block.  The two end points
			   switch roles.
			 */

		if	(!isBlockDefined()){
			endBlock = startBlock;
			startBlock = cursor;
			}
		changeLine(cursor.line);
		}
	else	{
		endBlock = cursor;
		upArrow();
		startOfBlock();
		}
	}

dragDown:	dynamic	() =
	{
	if	(cursor isEqual(&endBlock)){
		changeLine(cursor.line);
		downArrow();
		endBlock = cursor;
		if	(cursor.offset)
			changeLine(cursor.line);
		}
	else if	(cursor isEqual(&startBlock) &&
		 isBlockDefined()){
		changeLine(cursor.line);
		downArrow();
		startBlock = cursor;

			/*
			   This case covers what happens when backing up over
			   the starting point of the block.  The two end points
			   switch roles.
			 */

		if	(!isBlockDefined()){
			startBlock = endBlock;
			endBlock = cursor;
			}
		if	(cursor.offset)
			changeLine(cursor.line);
		}
	else	{
		startBlock = cursor;
		downArrow();
		endOfBlock();
		}
	}

dragRight:	dynamic	() =
	{
	if	(cursor isEqual(&endBlock)){
		rightArrow();
		endBlock = cursor;
		changeLine(cursor.line);
		}
	else if	(cursor isEqual(&startBlock) &&
		 isBlockDefined()){
		rightArrow();
		startBlock = cursor;
		changeLine(cursor.line);
		}
	else	{
		startBlock = cursor;
		rightArrow();
		endOfBlock();
		}
	}

dragLeft:	dynamic	() =
	{
	if	(cursor isEqual(&startBlock)){
		leftArrow();
		startBlock = cursor;
		changeLine(cursor.line);
		}
	else if	(cursor isEqual(&endBlock) &&
		 isBlockDefined()){
		leftArrow();
		endBlock = cursor;
		changeLine(cursor.line);
		}
	else	{
		endBlock = cursor;
		leftArrow();
		startOfBlock();
		}
	}

rightWord:	dynamic () =
	{
	cp:	* char;
	i:	int;
	ch:	char;

	cp = buffer fetchLine(cursor.line, cursor.offset);
	if	(cp == 0){
		bottomAlarm();
		return;
		}
	if	(*cp == '\r')
		cp++;
	if	(*cp == EOF_MARK){
		bottomAlarm();
		return;
		}
	else if	(*cp == '\n')
		return;

		/* If we are on a word, skip over it. */

	while	(isalpha(*cp) || *cp == '_' || isdigit(*cp)){
		cp++;
		cursor.offset++;
		}

		/* Now advance to the next word */

	for	(;;){
		if	(*cp == EOF_MARK ||
			 *cp == '\n' ||
			 *cp == '_' ||
			 isdigit(*cp) ||
			 isalpha(*cp))
			break;
		cp++;
		cursor.offset++;
		}
	findCursorColumn(findCurrentColumn);
	}

leftWord:	dynamic () =
	{
	cp:	* char;
	i:	int;
	ch:	char;

	cp = buffer fetchLine(cursor.line, cursor.offset);
	if	(cp == 0 ||
		 (cursor.line == 0 && cursor.offset == 0)){
		topAlarm();
		return;
		}
	if	(cursor.offset == 0)
		return;

		/* If we are on a word, back up at least 1 character */

	if	(*cp == '_' ||
		 isdigit(*cp) ||
		 isalpha(*cp)){
		cp--;
		cursor.offset--;
		}

		/* Now back up to the previous word */

	for	(;;){
		if	(cursor.offset == 0 ||
			 *cp == '_' ||
			 isdigit(*cp) ||
			 isalpha(*cp))
			break;
		cp--;
		cursor.offset--;
		}
	if	(cursor.offset){

			/* Now we are on a word, skip over it. */

		for	(;;){
			cp--;
			cursor.offset--;
			if	(*cp != '_' &&
				 !isdigit(*cp) &&
				 !isalpha(*cp))
				break;
			if	(cursor.offset == 0){
				findCursorColumn(findCurrentColumn);
				return;
				}
			}
		cursor.offset++;
		}
	findCursorColumn(findCurrentColumn);
	}

beginFile:	dynamic	() =
	{
	cursor.line = 0;
	cursor.lpos = 0;
	if	(display.column)
		redraw();
	display.column = desiredColumn = cursor.column = 0;
	cursor.offset = 0;
	refreshDisplay(0);
	}

endFile:	dynamic	() =
	{
	p:		point;
	lastLine:	int;
	i:		int;

	p = size;
	lastLine = buffer lineCount();
	cursor.line = lastLine - 1;
	cursor.lpos = buffer seek(cursor.line, 3);
	endLine();
	if	(p.y >= lastLine)
		refreshDisplay(0);
	else
		refreshDisplay(lastLine - p.y);
	}
/*
setChanged:	() =
	{

		/* A zero length delete still sets the changed flag */

	buffer seek(0, 0);
	buffer delete(0);
	}
 */
enterCharacter:	dynamic	(k: keystroke) =
	{
	changeLine(cursor.line);
	buffer seek(cursor.lpos + cursor.offset, 0);
	buffer insert(ref char(&k), 1);
	findColumn(&startBlock, adjustNewCharacter);
	findColumn(&endBlock, adjustNewCharacter);
	rightArrow();
	}

tab:	dynamic	() =
	{
	enterCharacter('\t');
	}

enterKey:	dynamic	() =
	{
	i:	int;
	j:	int;
	k:	int;
	cp:	* char;
	p:	point;
	x:	int;

	buffer seek(cursor.lpos + cursor.offset, 0);
	buffer insert("\r\n", 2);
	findColumn(&startBlock, adjustNewline);
	findColumn(&endBlock, adjustNewline);
	p = size;
	i = cursor.line - display.line;
	if	(i < p.y - 2){
		setColor(textColor);
		verticalScroll([ 0, i + 1 ], size, -1);
		}
	selectiveScroll(i + 1, p.y - 1, -1);
	changeLine(cursor.line);
	beginLine();
	downArrow();
	}

deleteCharacter:dynamic	() =
	{
	i:	int;
	j:	int;
	p:	point;
	cp:	* char;

	cp = buffer fetchLine(cursor.line, cursor.offset);
	if	(cp == 0)
		return;
	i = *cp;
	if	(i == EOF_MARK){
		bottomAlarm();
		return;
		}
	buffer seek(cursor.lpos + cursor.offset, 0);
	if	(i == '\r'){
		buffer delete(1);
		i = *cp;
		}
	j = cursor.line - display.line;
	buffer delete(1);
	if	(i == '\n'){
		p = size;
		if	(j < p.y - 2){
			setColor(textColor);
			verticalScroll([ 0, j + 1 ], size, 1);
			}
		selectiveScroll(j + 1, p.y - 1, 1);
		findColumn(&startBlock, adjustDeleteNewline);
		findColumn(&endBlock, adjustDeleteNewline);
		}
	else	{
		findColumn(&startBlock, adjustDeleteCharacter);
		findColumn(&endBlock, adjustDeleteCharacter);
		}
	changeLine(cursor.line);
	}

backspace:	dynamic	() =
	{
	if	(cursor.line == 0 && cursor.offset == 0)
		topAlarm();
	else if	(cursor.offset){
		leftArrow();
		deleteCharacter();
		}
	else	{
		upArrow();
		endLine();
		deleteCharacter();
		}
	}

deleteWord:	dynamic () =
	{
	cp:		* char;
	i:		int;
	ch:		char;
	delAmount:	int;

	cp = buffer fetchLine(cursor.line, cursor.offset);
	if	(cp == 0){
		bottomAlarm();
		return;
		}
	if	(*cp == '\r')
		cp++;
	if	(*cp == EOF_MARK){
		bottomAlarm();
		return;
		}
	else if	(*cp == '\n')
		return;
	cp++;
	delAmount = 1;

		/* If we are on a word, skip over it. */

	while	(isalpha(*cp) || *cp == '_' || isdigit(*cp)){
		cp++;
		delAmount++;
		}

		/* Now advance to the next word */

	for	(;;){
		if	(*cp != ' ' &&
			 *cp != '\t')
			break;
		cp++;
		delAmount++;
		}
	if	(delAmount){
		buffer seek(cursor.lpos + cursor.offset, 0);
		buffer delete(delAmount);
		changeLine(cursor.line);
		while	(delAmount){
			delAmount--;
			findColumn(&startBlock, adjustDeleteCharacter);
			findColumn(&endBlock, adjustDeleteCharacter);
			}
		}
	}

deleteLine:	dynamic	() =
	{
	i:		int;
	ch:		char;
	p:		point;
	delAmount:	int;
	cp:		* char;

	cp = buffer fetchLine(cursor.line, 0);
	if	(cp == 0)
		return;
	delAmount = 0;
	for	(;;){
		ch = *cp;
		cp++;
		if	(ch == EOF_MARK)
			break;
		delAmount++;
		if	(ch == '\n')
			break;
		}
	if	(delAmount){
		buffer seek(cursor.lpos, 0);
		buffer delete(delAmount);
		}
	p = size;
	i = cursor.line - display.line;
	if	(i < p.y - 1){
		setColor(textColor);
		verticalScroll( [ 0, i ], size, 1);
		}
	selectiveScroll(i, p.y - 1, 1);
	findColumn(&startBlock, adjustDeleteLine);
	findColumn(&endBlock, adjustDeleteLine);
	fixupColumn();
	updateCursor();
	}

startOfBlock:	dynamic	() =
	{
	startBlock = cursor;
	redraw();
	}

endOfBlock:	dynamic	() =
	{
	endBlock = cursor;
	redraw();
	}

updateBlockLocation:	private	() =
	{
	if	(endBlock.line == startBlock.line)
		endBlock.offset = cursor.offset +
			(endBlock.offset - startBlock.offset);
	endBlock.line = cursor.line + (endBlock.line - startBlock.line);
//	endBlock.lpos = buffer seek(endBlock.line, 3);
	startBlock = cursor;
	findColumn(&endBlock, findCurrentColumn);
	}

moveBlock:	dynamic	() =
	{
	i:	int;
	size:	int;
	spos:	textPos;

	i = relativeLocation();
	if	(i > -2 &&
		 i < 2)
		return;
	spos = startBlock.lpos + startBlock.offset;
	buffer seek(spos, 0);
	size = endBlock.lpos + endBlock.offset - spos;

	buffer extract(&Transfer, size);

		/* If i < 0 it means the cursor is above the block,
			otherwise the cursor is below the block. */

	if	(i < 0){
		buffer delete(size);
		buffer seek(cursor.lpos + cursor.offset, 0);
		buffer merge(&Transfer);
		}
	else	{
		buffer seek(cursor.lpos + cursor.offset, 0);
		buffer merge(&Transfer);
		buffer seek(spos, 0);
		buffer delete(size);
		cursor.line -= Transfer lineCount() - 1;
		cursor.lpos = buffer seek(cursor.line, 3);
		updateCursor();
		}
	updateBlockLocation();
	Transfer makeEmpty();
	redraw();
	centerDisplay();
	}

copyBlock:	dynamic	() =
	{
	i:	int;
	spos:	int;
	size:	int;

	i = relativeLocation();
	if	(i == 0)
		return;
	spos = startBlock.lpos + startBlock.offset;
	buffer seek(spos, 0);
	size = endBlock.lpos + endBlock.offset - spos;

	buffer extract(&Transfer, size);
	buffer seek(cursor.lpos + cursor.offset, 0);
	buffer merge(&Transfer);
	updateBlockLocation();
	Transfer makeEmpty();
	redraw();
	}

cutToScrap:	dynamic	() =
	{
	i:	int;
	size:	int;
	spos:	textPos;

	i = relativeLocation();
	if	(scrap == 0 ||
		 !isBlockDefined()){
		beep();
		return;
		}
	spos = startBlock.lpos + startBlock.offset;
	buffer seek(spos, 0);
	size = endBlock.lpos + endBlock.offset - spos;
	buffer extract(scrap, size);
	buffer delete(size);
	if	(i == 0){
		cursor = startBlock;
		centerDisplay();
		findCursorColumn(findCurrentColumn);
		}
	else if	(i > 0){
		cursor.line -= scrap lineCount() - 1;
		cursor.lpos = buffer seek(cursor.line, 3);
		centerDisplay();
		findCursorColumn(findCurrentColumn);
		}
	endBlock = startBlock;
	redraw();
	}

deleteBlock:	dynamic	() =
	{
	i:	int;
	size:	int;
	spos:	textPos;

	i = relativeLocation();
	if	(!isBlockDefined())
		return;
	spos = startBlock.lpos + startBlock.offset;
	buffer seek(spos, 0);
	size = endBlock.lpos + endBlock.offset - spos;
	buffer extract(&Transfer, size);
	buffer delete(size);
	if	(i == 0){
		cursor = startBlock;
		centerDisplay();
		findCursorColumn(findCurrentColumn);
		}
	else if	(i > 0){
		cursor.line -= Transfer lineCount() - 1;
		centerDisplay();
		findCursorColumn(findCurrentColumn);
		}
	Transfer makeEmpty();
	endBlock = startBlock;
	redraw();
	}

copyToScrap:	dynamic	() =
	{
	if	(scrap == 0 ||
		 !isBlockDefined()){
		beep();
		return;
		}

	spos:	textPos;
	size:	int;

	spos = startBlock.lpos + startBlock.offset;
	size = endBlock.lpos + endBlock.offset - spos;

	buffer seek(spos, 0);
	buffer extract(scrap, size);
	}

pasteFromScrap:	dynamic	() =
	{
	if	(scrap == 0){
		postMessage("No scrap");
		return;
		}
	buffer seek(cursor.lpos + cursor.offset, 0);
	buffer merge(scrap);
	startBlock = cursor;
	endBlock = cursor;
	redraw();
	}

gotoLine:	(line: unsigned) =
	{
	sp:	filePos;

	if	(line < 1 ||
		 line > buffer lineCount()){
		postMessage("Line number not in file");
		return;
		}
	sp.line = line - 1;
	sp.offset = 0;
	sp.lpos = buffer seek(sp.line, 3);
	cursor = sp;
	centerDisplay();
	findCursorColumn(findCurrentColumn);
	clearMessage();
	}

	private:

	desiredColumn:	int;
	lineOffset:	int;
	startBlock:	filePos;
	endBlock:	filePos;
	display:	filePos;
	lineStatus:	[50] int;
	scrap:		* editBuffer;

updateCursor:	() =
	{
	textCursor([ cursor.column - display.column, 
				cursor.line - display.line ]);
	}

isBlockDefined:		() int =
	{
	if	(startBlock.lpos + startBlock.offset <
			endBlock.lpos + endBlock.offset)
		return 1;
	else
		return 0;
	}

relativeLocation:	() int =
	{
	cpos:	int;
	spos:	int;
	epos:	int;

	startBlock.lpos = buffer seek(startBlock.line, 3);
	endBlock.lpos = buffer seek(endBlock.line, 3);
	spos = startBlock.lpos + startBlock.offset;
	epos = endBlock.lpos + endBlock.offset;
	if	(spos >= epos)
		return 0;
	cpos = cursor.lpos + cursor.offset;
	if	(cpos < spos)
		return -2;
	if	(cpos == spos)
		return -1;
	if	(cpos > epos)
		return 2;
	if	(cpos == epos)
		return 1;
	return 0;
	}

findColumn:	(f: * filePos, fa: findAction) =
	{
	i:	int;
	ch:	char;
	j:	int;
	cp:	* char;

	switch	(fa){
	case	adjustNewCharacter:
		if	(f->line == cursor.line &&
			 f->offset > cursor.offset)
			f->offset++;
		else
			return;
		break;

	case	adjustNewline:
		if	(f->line > cursor.line){
			f->line++;
			f->lpos = buffer seek(f->line, 3);
			return;
			}
		else if	(f->line == cursor.line &&
			 f->offset > cursor.offset){
			f->offset -= cursor.offset;
			f->line++;
			f->lpos = buffer seek(f->line, 3);
			}
		else
			return;
		break;

	case	adjustDeleteCharacter:
		if	(f->line == cursor.line &&
			 f->offset > cursor.offset)
			f->offset--;
		else
			return;
		break;

	case	adjustDeleteNewline:
		if	(f->line > cursor.line + 1){
			f->line--;
			f->lpos = buffer seek(f->line, 3);
			return;
			}
		else if	(f->line == cursor.line + 1){
			f->offset += cursor.offset;
			f->line--;
			f->lpos = buffer seek(f->line, 3);
			}
		else
			return;
		break;

	case	adjustDeleteLine:
		if	(f->line > cursor.line){
			f->line--;
			f->lpos = buffer seek(f->line, 3);
			}
		else if	(f->line == cursor.line)
			f->offset = f->column = 0;
		return;
		}
	cp = buffer fetchLine(f->line, 0);
	if	(cp == 0){
		f->column = 0;
		return;
		}
	if	(fa == findEndOfLine)
		j = 0x7FFFFFFF;
	else
		j = f->offset;
	f->column = 0;
	for	(;j;){
		ch = *cp;
		cp++;
		if	(fa == findEndOfLine &&
			 (ch == '\n' ||
			  ch == EOF_MARK)){
			f->offset = 0x7FFFFFFF - j;
			break;
			}
		if	(ch == '\t')
			f->column = (f->column + 8) & ~7;
		else if	(ch == '\n')
			return;
		else if	(ch == '\r')
			continue;
		else
			f->column++;
		j--;
		}
	}
/*
	relativeFlag:

	 0 = find end of line.
	 1 = find position relative to current location only, if we pass
	     over a \n, then wrap to a new line.
 */
findCursorColumn:	(relativeFlag: findAction) =
	{
	findColumn(&cursor, relativeFlag);
	if	(cursor.column < display.column){
		display.column = cursor.column;
		redraw();
		}
	else if	(cursor.column >= display.column + size.x){
		display.column = 1 + cursor.column - size.x;
		redraw();
		}
	desiredColumn = cursor.column;
	updateCursor();
	}

fixupColumn:	() =
	{
	i:	int;
	ch:	char;
	j:	int;
	cp:	* char;

	cp = buffer fetchLine(cursor.line, 0);
	cursor.lpos = buffer seek(0, 1);
	if	(cp == 0){
		cursor.column = 0;
		i = display.column;
		display.column = desiredColumn = cursor.column;
		if	(i)
			redraw();
		return;
		}
	cursor.column = 0;
	cursor.offset = 0;
	while	(cursor.column < desiredColumn){
		ch = *cp;
		cp++;
		if	(ch == '\n' || ch == EOF_MARK)
			break;
		else if	(ch == '\t')
			cursor.column = (cursor.column + 8) & ~7;
		else if	(ch == '\r')
			break;
		else
			cursor.column++;
		cursor.offset++;
		}
	if	(cursor.column < display.column){
		display.column = cursor.column;
		redraw();
		}
	else if	(cursor.column >= display.column + size.x){
		display.column = 1 + cursor.column - size.x;
		redraw();
		}
	}

selectiveScroll:	(from: int, to: int, by: int) =
	{
	i:	int;

	if	(by > 0){
		for	(i = from + by; i < to; i++)
			if	(lineStatus[i] == -1)
				lineStatus[i - by] = -1;
		lineStatus[to] = -1;
		}
	else	{
		for	(i = from; i < to + by; i++)
			if	(lineStatus[i] == -1)
				lineStatus[i - by] = -1;
		if	(from < size.y)
			lineStatus[from] = -1;
		}
	}

refreshDisplay:	(ln: int) =
	{
	i:	int;
	j:	int;
	k:	int;
	cp:	* char;
	p:	point;
	x:	int;

	p = size;
	j = p.y;
	i = 0;
	setColor(textColor);
	if	(ln <= display.line - p.y ||
		 ln >= display.line + p.y)
		clear();
	else if	(ln < display.line){
		j = display.line - ln;
		verticalScroll([ 0, 0 ], size, -j);
		memMove(&lineStatus[j], &lineStatus, (p.y - j) * sizeof int);
		}
	else if	(ln > display.line){
		verticalScroll([ 0, 0 ], size, ln - display.line);
		memMove(&lineStatus, &lineStatus[ln - display.line],
				(p.y - (ln - display.line)) * sizeof (int));
		i = p.y - (ln - display.line);
		}
	else
		j = 0;
	for	(; i < j; i++)
		lineStatus[i] = -1;
	display.line = ln;
	updateCursor();
	}

centerDisplay:	() =
	{
	p:	point;
	ln:	int;

	p = size;
	ln = cursor.line;
	if	(ln < display.line ||
		 ln >= display.line + p.y){
		if	(ln < p.y / 2)
			ln = 0;
		else
			ln -= p.y / 2;
		refreshDisplay(ln);
		}
	}

repaintDisplay:	() =
	{
//	l:	*line;
	i:	int;
	j:	int;
	k:	int;
	cp:	* char;
	p:	point;
	x:	int;

	p = size;
	for	(i = 0, j = display.line; i < p.y; i++, j++){
		if	(lineStatus[i] != j){
			if	(testKey() != -1)
				return;
			formatLine(j, 0);
			lineStatus[i] = j;
			}
		}
	}

};

topAlarm:	() =
	{
	postMessage("Top of file");
	}

bottomAlarm:	() =
	{
	postMessage("Bottom of file");
	}

lineNum:	public	type	int;
textPos:	public	type	long;

EOF_MARK:	public	const	char = 0;

editBuffer:	public	type	{
	public:

close:	dynamic	() =
	{
	}

makeEmpty:	dynamic	() =
	{
	}

lineCount:	dynamic	() lineNum =
	{
	}

insert:	dynamic	(* char, int) =		// buffer address and length
	{
	}

merge:	dynamic	(buff: * editBuffer) =	// edit buffer to merge
	{
	i:		int;
	j:		int;
	intermed:	[128] char;

	i = buff seek(0, 2);
	while	(i > 0){
		if	(i > 128)
			j = 128;
		else
			j = i;
		buff seek(i - j, 0);
		buff read(intermed, j);
		insert(intermed, j);
		i -= j;
		}
	}

seek:	dynamic	(textPos, int) textPos =
	{
	}

read:	dynamic	(pointer, int) int =	// buffer and len and actual copied
	{
	}

fetchLine:	dynamic	(lineNum, int) * char = 
					// line number and offset in line
	{
	}

getLineno:	dynamic	(* lineNum, * int) =
					// return current location in lines
	{
	}

extract:	dynamic	(* editBuffer, int) =
					// buffer to put the data in
	{
	}

beginExtract:	dynamic	(textPos) =	// size
	{
	}

write:	dynamic	(pointer, int) =	// block and length
	{
	}

delete:	dynamic	(int) =			// amount to delete
	{
	}

search:	dynamic	(* textPos,		// location where found
	 * searchPattern) int =		// search pattern (compiled regular
					// expression)
	{
	return 0;
	}

hasChanged:	dynamic	() int =
	{
	return 0;
	}

	};

Transfer:	transferBuffer;

transferBuffer:	public	type	inherit	editBuffer {
	public:

makeEmpty:	dynamic	() =
	{
	if	(data)
		free(data);
	data = 0;
	}

lineCount:	dynamic	() lineNum =
	{
	cp:	* char;
	i:	textPos;
	lc:	lineNum;

	lc = 1;
	for	(i = 0, cp = data; i < dataLen; i++, cp++){
		if	(*cp == '\n')
			lc++;
		}
	return lc;
	}

seek:	dynamic	(newPos: textPos, whence: int) textPos =
	{
	switch	(whence){
	case	0:			// SEEK_ABS
		offset = newPos;
		break;

	case	1:			// SEEK_CUR
		offset += newPos;
		break;

	case	2:
		offset = dataLen + newPos;
		}
	return offset;
	}

read:	dynamic	(buf: pointer, len: int) int =
					// buffer and len and actual copied
	{
	if	(offset >= dataLen)
		return 0;
	if	(offset + len > dataLen)
		len = dataLen - offset;
	memCopy(buf, data + offset, len);
	offset += len;
	return len;
	}

beginExtract:	dynamic	(size: textPos) =
	{
	if	(data)
		free(data);
	data = alloc(size);
	dataLen = size;
	offset = 0;
	}

write:	dynamic	(buf: pointer, len: int) =	// block and length
	{
	if	(offset >= dataLen)
		return;
	if	(offset + len > dataLen)
		len = dataLen - offset;
	memCopy(data + offset, buf, len);
	offset += len;
	return;
	}

	data:		* char;
	dataLen:	textPos;
	offset:		textPos;

	};

TEXT_ATTR:	const	char = HIGH|YELLOW;

fileEditor:	public	type	inherit	editor {
	public:

	filename:	[:] char;
	buffer:		textBuffer;

constructor:	(titl: [:] char, sz: point, mm: * menu) =
	{
	super constructor(titl, sz, mm);
//	stateFrame = [ titl, sz ];
	}

connect:	(file: [:] char) =
	{
	filename = new [|file] char;
	filename [:]= file;
	buffer = [ filename ];
	if	(buffer isNew())
		postMessage("New file %s", file);
	}

open:	dynamic	(grnd: * frame, p: point) int =
	{
	i:	int;

	super connect(&buffer);
	setColor(color);
	i = super open(grnd, p);
	resetEditor();
	return i;
	}

setBorder:	dynamic	(b: borderStyles) =
	{
	super setBorder(b);
	}

popup:	dynamic	(grnd: * frame, p: point) int =
	{
	i:	int;

//	super constructor([ 0 ], [ size.x, size.y - 1], mainMenu);
	super connect(&buffer);
	setColor(color);
	i = super popup(grnd, p);
	resetEditor();
	return i;
	}

close:	dynamic	() int =
	{
	if	(buffer hasChanged()){
		i:	int;

		i = confirm("File has changes, save it?", 0x31);

		if	(i < 0){
			postMessage("File not saved");
			return 0;
			}
		if	(i &&
			 buffer save(filename)){
			postMessage("Could not save");
			return 0;
			}
		}
//	stateFrame close();
	buffer close();
	return super close();
	}

redraw:	dynamic	() =
	{
/*
	stateFrame setColor(WHITE);
	stateFrame printf([ 0, 0 ], 
			"Line:       Column:           %s", &filename);
 */
	super redraw();
	}

saveFile:	dynamic	() =
	{
	if	(buffer save(filename))
		postMessage("Could not save");
	}

localKeys:	dynamic	() * keyMap =
	{
	return(&FileEditMap);
	}

beforeKeyPressed:	dynamic	() =
	{
/*
	c:	filePos;

	c = cursor;
	c.line++;
	c.column++;
	FileStateForm open(self, [ 0, 0 ]);
	FileStateForm draw(&c);
	FileStateForm close();
 */
	super beforeKeyPressed();
	}

switchFile:	() =
	{
	memSet(&StartData, 0, sizeof StartData);
	StartData.startString [:]= filename;
	StartForm popup(self, [ 0, 0 ]);
	StartForm redraw();
	if	(StartForm startup() != FS_SUCCESS)
		return;

	if	(StartData.startString[0] == 0)
		return;

	if	(buffer hasChanged()){
		i:	int;

		i = confirm("File has changes, save it?", 0x31);

		if	(i < 0){
			postMessage("File not saved");
			return;
			}
		if	(i &&
			 buffer save(filename)){
			postMessage("Could not save");
			return;
			}
		}
	buffer close();
	connect(StartData.startString);
	resetEditor();
	}

renameFile:	() =
	{
	memSet(&StartData, 0, sizeof StartData);
	StartData.startString [:]= filename;
	StartForm popup(self, [ 0, 0 ]);
	StartForm redraw();
	if	(StartForm startup() != FS_SUCCESS)
		return;

	if	(StartData.startString[0] == 0)
		return;

	s:	[:] char;
	i:	int;

	i = stringLength(StartData.startString);
	s = StartData.startString[:i];
	filename [:]= s;
	title = filename;
	drawTitle();
	buffer setChanged();
	}

gotoLineCmd:	() =
	{
	GotoData.line = cursor.line + 1;
	GotoForm popup(self, [ 0, 0 ]);
	GotoForm redraw();
	if	(GotoForm startup() != FS_SUCCESS)
		return;

	gotoLine(GotoData.line);
	}

search:	dynamic	() =
	{
	SearchForm popup(self, [ 0, 0 ]);
	SearchForm redraw();

	if	(SearchForm startup() != FS_SUCCESS)
		return;

	if	(SearchData.searchString[0] == 0)
		return;

	performSearchReplace(RO_SEARCH);
	}

replace:	dynamic	() =
	{
	SearchForm popup(self, [ 0, 0 ]);
	SearchForm redraw();
	if	(SearchForm startup() != FS_SUCCESS)
		return;

	if	(SearchData.searchString[0] == 0)
		return;

	ReplaceForm popup(self, [ 0, 0 ]);
	ReplaceForm redraw();
	if	(ReplaceForm startup() != FS_SUCCESS)
		return;

	performSearchReplace(RO_REPLACE);
	}

again:	dynamic	() =
	{
	switch	(LastOperation){
	case	RO_SEARCH:
	case	RO_REPLACE:
		performSearchReplace(LastOperation);
		break;
		}
	}

performSearchReplace:	(op: repeatableOps) =
	{
	sp:	filePos;
	sx:	textPos;

	LastOperation = op;
	SearchPattern dispose();
	SearchPattern = [ Sr_metaflag ];
	if	(!SearchPattern compile(SearchData.searchString)){
		postMessage("Improper regular expression\n");
		return;
		}
	for	(;;){
		showCursor();
		postMessage("Searching...");
		sp = cursor;

		foundLen:	int;

		sx = cursor.lpos + cursor.offset;
		buffer seek(sx, 0);
		foundLen = buffer search(&sx, &SearchPattern);
		if	(foundLen < 0){
			postMessage("Not found");
			return;
			}
		len:	int;

		buffer getLineno(&cursor.line, 
					&cursor.offset);
		cursor.lpos = sx - cursor.offset;
		cursor.offset += foundLen;
		centerCursor();
		if	(op == RO_REPLACE){
			if	(ReplaceState == RS_ASK){
				i:	replaceActions;

				i = askForReplaceAction();
				if	(i == RA_STOP)
					break;
				if	(i == RA_ALL)
					ReplaceState = RS_DOIT;
				if	(i == RA_SKIP)
					continue;
				}
			buffer seek(sx, 0);
			cursor.offset -= foundLen;
			buffer delete(foundLen);
			foundLen = stringLength(SearchData.replaceString);
			buffer insert(SearchData.replaceString, foundLen);
			cursor.offset += foundLen;
			centerCursor();
			changeLine(cursor.line);
			}
		else
			break;
		}
	clearMessage();
	}

	private:

askForReplaceAction:	() replaceActions =
	{
	i:	unsigned;
	k:	keystroke;

	postMessage("Replace?  Yes/No/All");
	for	(;;){
		beforeKeyPressed();
		k = getKey();
		afterKeyPressed();
		if	(k == 'y' || k == 'Y'){
			i = RA_YES;
			break;
			}
		else if	(k == 'n' || k == 'N'){
			i = RA_SKIP;
			break;
			}
		else if	(k == 'a' || k == 'A'){
			i = RA_ALL;
			break;
			}
		else if	(k == ESC || k == GREY_ESC){
			i = RA_STOP;
			break;
			}
		beep();
		}
	return i;
	}

//	stateFrame:	window;
	
};

beginSession:	public	(filename: [:] char, mm: * menu) int =
	{
	esp:	ref fileEditor;

	esp = new fileEditor[ filename, [ 80, 25 ], mm ];
	esp connect(filename);
	esp setBorder(WB_DBL_TOP);
	esp setColor(TEXT_ATTR);
	esp->textColor = TEXT_ATTR;
	esp->borderColor = BLUE;
	esp->titleColor = COLOR(HIGH|WHITE, BLACK);
	esp open(&Screen, [ 0, 0 ]);
	esp setScrap(&Scrap);
	setPrimaryViewer(esp);
	return 1;
	}

Scrap:	public	transferBuffer;

StartData:	{
	public:

	startString:	[71] char;
	};

StartForm:	form;

FileStateForm:	form;

init:	entry	() =
	{
	FileStateForm = [ "", [ 80, 1 ] ];
	FileStateForm integer([ 6, 0 ], [ 5, 1 ], offsetof filePos.line,
					sizeof lineNum, "", 0, WHITE);
	FileStateForm integer([ 20, 0 ], [ 3, 1 ], 
					offsetof filePos.column,
					sizeof int, "", 0, WHITE);
	StartForm = [ "", [ 80, 1 ] ];
	StartForm text([ 0, 0 ], WHITE, "File:");
	StartForm stringf([ 6, 0 ], [ 70, 1], 0, 71, "", 0, HIGH|WHITE);
	StartForm connect(&StartData);
	GotoForm = [ "", [ 80, 1 ] ];
	GotoForm text([ 0, 0 ], WHITE, "Goto:");
	GotoForm integer([ 6, 0 ], [ 10, 1], 0, 
				sizeof int, "", 0, HIGH|WHITE);
	GotoForm connect(&GotoData);
	SearchForm = [ "", [ 80, 1 ] ];
	SearchForm text([ 0, 0 ], WHITE, "Search:");
	SearchForm stringf([ 8, 0 ], [ 70, 1 ], 
					0, 71, "", 0, HIGH|WHITE);
	SearchForm connect(&SearchData);
	ReplaceForm = [ "", [ 80, 1 ] ];
	ReplaceForm text([ 0, 0 ], WHITE, "Replace:");
	ReplaceForm stringf([ 9, 0 ], [ 70, 1 ], 
					71, 71, "", 0, HIGH|WHITE);
	ReplaceForm connect(&SearchData);
	}

startSession:	public	(* viewer) =
	{
	memSet(&StartData, 0, sizeof StartData);

	StartForm popup(&Screen, [ 0, 1 ]);
	StartForm redraw();
	if	(StartForm startup() != FS_SUCCESS)
		return;

	if	(StartData.startString[0] == 0)
		return;

	if	(!beginSession(StartData.startString, 0))
		postMessage("Couldn't open file");
	}

saveSession:	public	(v: * viewer) =
	{
	v saveFile();
	}

/*
switchSession:	public	(v: * viewer) =
	{
	w:	* fileEditor;

	w = type * fileEditor(v);
	w switchFile();
	}
 */
renameSession:	public	(v: * viewer) =
	{
	w:	* fileEditor;

	w = ref fileEditor(v);
	w renameFile();
	}

GotoData:	{
	public:

	line:	int;
	};

GotoForm:	form;

gotoLine:	public	(v: * viewer) =
	{
	w:	* fileEditor;

	w = ref fileEditor(v);
	w gotoLineCmd();
	}

toggleRegularExpression:	public	(* viewer) =
	{
	if	(Sr_metaflag){
		postMessage("Regular expressions OFF");
		Sr_metaflag = 0;
		}
	else	{
		postMessage("Regular expressions ON");
		Sr_metaflag = 1;
		}
	}

Sr_metaflag:	char = TRUE;

SearchData:	{
	public:

	searchString:	[71] char;
	replaceString:	[71] char;
	};

SearchPattern:		searchPattern;

SearchForm:		form;
ReplaceForm:		form;

repeatableOps:	type	char = {
	RO_SEARCH = 1,
	RO_REPLACE
	};

replaceStates:	type	char = {
	RS_ASK,
	RS_DOIT
	};

replaceActions:	type	char = {
	RA_YES,
	RA_SKIP,
	RA_ALL,
	RA_STOP
	};

LastOperation:	repeatableOps;
ReplaceState:	replaceStates;
FileEditMap:	public	keyMap;

BLOCKSIZE:	const	int = 2048;

textBuffer:	public	type	inherit	editBuffer	{
	public:

constructor:	(fname: [:] char) int =
	{
	fd:	stream;
	i:	int;
	eb:	* fileBlock;
	cp:	* char;
	len:	int;

	flags = 0;
	blocks = 0;
	lastBlock = 0;
	lnCount = 0;
	textLength = 0;
	lnCount = 0;
	pos = 0;
	accessLines = 0;
	accessBlock = 0;
	accessOffset = 0;
	accessPtr = 0;
	if	(|fname)
		i = fd open(fname, AR_READ);
	else
		i = -1;
	if	(i){
		eb = new fileBlock;
		blocks = eb;
		eb->prev = 0;
		eb->next = 0;
		makeEmpty();
		flags |= NEWFILE;
		return SUCCESS;
		}
	eb = 0;
	for	(;;){
		if	(eb == 0){
			eb = self newBlock(0, 0);
			if	(eb == 0){
				fd close();
				return ERRNOMEMORY;
				}
			}
		s:	[:] char;

		s = eb->text[eb->length:];
		len = fd read(s);
		if	(len == 0){
			eb->text[eb->length] = EOF_MARK;
			eb->length++;
			eb->lineCount++;
			lnCount++;
			break;
			}
		i = eb->length;
		cp = &eb->text[i];
		len += i;
		for	(;;){
			i++;
			if	(*cp == '\n')
				eb->lineCount++;
			else if	(*cp == 26 ||
				 *cp == 0){
				*cp = EOF_MARK;
				eb->lineCount++;
				len = i;
				break;
				}
			if	(i >= len)
				break;
			cp++;			// assure that cp points at
						// the last character.
			}
		textLength += len;
		eb->length = len;
		lnCount += eb->lineCount;
		if	(*cp == EOF_MARK)
			break;
		if	(*cp == '\n')
			eb = 0;

			/* Split very large lines. */

		else if	(eb->lineCount == 0){
			eb->lineCount = 1;
			eb = self newBlock(cp - 1, 2);
			if	(eb == 0)
				return(ERRNOMEMORY);
			cp[-1] = '\r';
			*cp = '\n';
			}
		else	{
			do	{
				cp--;
				i--;
				}
				while	(*cp != '\n');
			eb->length = i;
			eb = self newBlock(cp + 1, len - i);
			}
		}
	fd close();
	return SUCCESS;
	}

setChanged:	() =
	{
	flags |= CHANGED;
	}

isNew:	() int =
	{
	return flags & NEWFILE;
	}

close:	dynamic	() =
	{
	eb:	* fileBlock;
	vp:	pointer;

	for	(eb = blocks; eb;){
		vp = eb;
		eb = eb->next;
		free(vp);
		}
	}

save:	(filename: [:] char) int =
	{
	fd:	stream;
	i:	int;
	j:	int;
	eb:	* fileBlock;
	fbase:	[:] char;
	fbak:	[MAXPATH] char;

		/* If the file already exists, rename it to .BAK */

	if	(|filename == 0)
		return ERRINVALIDDATA;
	if	(FileSystem access(filename, AR_WRITE) == SUCCESS){
		s:	[:] char;

		fbase = stripExtension(filename);
		s = makePath(fbak, "", fbase, ".bak");
		FileSystem unlink(s);
		FileSystem move(filename, s);
		}
	i = fd create(filename, FA_READ|FA_WRITE|FA_SEEKOK);
	if	(i)
		return i;
	for	(eb = blocks; eb; eb = eb->next){
		if	(eb->next == 0)
			j = eb->length - 1;	// Don't write the EOF mark
		else
			j = eb->length;
		i = fd write(eb->text[:j]);
		if	(i != j){
			fd close();
			return(i);
			}
		}
	fd close();
	flags &= ~CHANGED;
	return SUCCESS;
	}

makeEmpty:	dynamic	() =
	{
	eb:	* fileBlock;
	vp:	pointer;

	eb = blocks;
	lastBlock = eb;
	eb->length = 1;
	eb->lineCount = 1;
	eb->text[0] = EOF_MARK;
	lnCount = 1;
	textLength = 1;
	for	(eb = eb->next; eb;){
		vp = eb;
		eb = eb->next;
		free(vp);
		}
	blocks->next = 0;
	}

lineCount:	dynamic	() lineNum =
	{
	return lnCount;
	}

insert:	dynamic	(newData: * char, len: int) =
					// buffer address and length
	{
	i:		int;
	j:		int;
	cp:		* char;
	newlines:	int;
	eb:		* fileBlock;
	neweb:		* fileBlock;
	filleb:		* fileBlock;
	off:		int;

	eb = accessBlock;
	newlines = 0;
	for	(i = 0, cp = newData; i < len; i++, cp++)
		if	(*cp == '\n')
			newlines++;
	if	(eb->length + len > BLOCKSIZE){
		splitBlock(eb);
		if	(accessOffset >= eb->length){
			accessOffset -= eb->length;
			accessLines += eb->lineCount;
			eb = eb->next;
			accessBlock = eb;
			accessPtr = &eb->text[accessOffset];
			if	(eb == 0)
				return;
			}
		}
	insertIntoBlock(eb, accessOffset, newData, len, newlines);
	}

seek:	dynamic	(newPos: textPos, whence: int) textPos =
	{
	eb:	* fileBlock;
	a:	textPos;

	switch	(whence){
	case	0:
		pos = newPos;
		break;

	case	1:
		if	(newPos == 0)
			return pos;
		pos += newPos;
		break;

	case	2:
		pos = textLength + newPos;
		break;

	case	3:		// go to line number
		a = 0;
		for	(eb = blocks; eb; eb = eb->next){
			if	(eb->lineCount > newPos)
				break;
			a += eb->length;
			newPos -= eb->lineCount;
			}

		if	(eb == 0 || newPos == 0){
			pos = a;
			break;
			}

		cp:	* char;
		i:	int;

		i = 0;
		for	(cp = eb->text; i < eb->length; i++, cp++){
			if	(*cp == '\n'){
				newPos--;
				if	(newPos == 0)
					break;
				}
			}
		pos = a + i + 1;
		break;
		}

	accessLines = 0;
	a = pos;
	for	(eb = blocks; eb && eb->length <= a; eb = eb->next){
		a -= eb->length;
		accessLines += eb->lineCount;
		}
	accessBlock = eb;
	if	(eb){
		accessOffset = a;
		accessPtr = &eb->text[accessOffset];
		}
	else	{
		accessOffset = 0;
		accessPtr = 0;
		pos = textLength;
		}
	return pos;
	}

read:	dynamic	(buf: pointer, len: int) int =
					// buffer and len and actual copied
	{
	if	(accessBlock == 0)
		return 0;

	eb:	* fileBlock;
	off:	int;
	rem:	int;
	count:	int;

	off = accessOffset;
	count = 0;
	for	(eb = accessBlock; eb; eb = eb->next){
		rem = eb->length - off;
		if	(len < rem)
			rem = len;
		cp:	* char;

		cp = &eb->text[off];
		memCopy(buf, cp, rem);
		len -= rem;
		count += rem;
		if	(len == 0)
			break;
		off = 0;
		}
	return count;
	}

fetchLine:	dynamic	(ln: lineNum, off: int) * char = 
					// line number and offset in line
	{
	i:	int;

	seek(ln, 3);
	for	(i = 0; i < off; i++, accessPtr++){
		if	(*accessPtr == '\n' ||
			 *accessPtr == EOF_MARK)
			break;
		}
	accessOffset += i;
	pos += i;
	return accessPtr;
	}

extract:	dynamic	(buff: * editBuffer, len: int) =
	{
	i:	int;
	j:	int;
	cp:	* char;
	eb:	* fileBlock;
	ch:	char;
	size:	int;
	neweb:	* fileBlock;
	cutoff:	int;
	off:	int;

	if	(accessBlock == 0)
		return;
	buff makeEmpty();
	buff beginExtract(len);
	off = accessOffset;
	for	(eb = accessBlock; len && eb; ){
		i = eb->length - off;
		if	(i > len)
			i = len;
		buff write(&eb->text[off], i);
		off = 0;
		eb = eb->next;
		len -= i;
		}
	}

beginExtract:	dynamic	(textPos) =	// size
	{
	makeEmpty();
	}

write:	dynamic	(buf: pointer, len: int) =	// block and length
	{
	insert(buf, len);
	seek(len, 1);
	}

delete:	dynamic	(len: int) =			// amount to delete
	{
	vp:		pointer;
	residue:	int;
	newlines:	int;
	final:		int;
	i:		int;
	eb:		* fileBlock;
	neb:		* fileBlock;
	nneb:		* fileBlock;
	loff:		int;
	thislen:	int;
	nextlen:	int;
	off:		textPos;

	if	(accessBlock == 0)
		return;
	eb = accessBlock;
	off = accessOffset;
	textLength -= len;
	newlines = 0;
	flags |= CHANGED;
	while	(len > 0){
		final = off + len;
		if	(final > eb->length)
			final = eb->length;
		for	(i = off; i < final; i++)
			if	(eb->text[i] == '\n')
				newlines++;
		eb->lineCount -= newlines;
		if	(final < eb->length){
			memMove(&eb->text[off], &eb->text[final],
					eb->length - final);
			eb->length -= len;
			lnCount -= newlines;
			return;
			}
		len -= eb->length - off;
		eb->length = off;
		if	(eb->length == 0){
			if	(eb->prev)
				eb->prev->next = eb->next;
			else
				blocks = eb->next;
			if	(eb->next == 0){
				lastBlock = eb->prev;
				free(eb);
				break;
				}
			eb->next->prev = eb->prev;
			vp = eb;
			eb = eb->next;
			accessBlock = blocks;
			accessLines = 0;
			free(vp);
			}
		else
			eb = eb->next;
		off = 0;
		lnCount -= newlines;
		newlines = 0;
		}
	if	(eb == 0 ||
		 eb->prev == 0)
		return;
	neb = eb;
	eb = eb->prev;

		// Make sure the last block (where the last deletes occurred)
		// ends in a newline.  If it doesn't, we have to merge lines.

//	assert(eb->length);
	final = eb->length - 1;
	if	(eb->text[final] == '\n')
		return;

		// Ukk, we have to merge lines.

	accessBlock = blocks;
	accessLines = 0;
	if	(eb->length + neb->length <= BLOCKSIZE){
		memMove(&eb->text[final + 1], &neb->text[0], neb->length);
		eb->length += neb->length;
		eb->lineCount += neb->lineCount;
		vp = eb->next;
		eb->next = neb->next;
		if	(eb->next)
			eb->next->prev = eb;
		else
			lastBlock = eb;
		free(vp);
		}

		// Calculate the length of the splice pieces

	for	(thislen = 0; final >= 0; thislen++, final--)
		if	(eb->text[final] == '\n')
			break;
	for	(nextlen = 0; nextlen < neb->length; nextlen++)
		if	(neb->text[nextlen] == '\n'){
			nextlen++;
			break;
			}

		// Can we copy the next block's data into this one?

	if	(eb->length + nextlen < BLOCKSIZE){
//		assert(nextlen < neb->length);
		self insertIntoBlock(eb, eb->length, &neb->text[0], nextlen, 1);
		memMove(&neb->text[0], &neb->text[nextlen], 
						neb->length - nextlen);
		neb->length -= nextlen;
		return;
		}

		// How about this block's data to the next block?

	if	(neb->length + thislen < BLOCKSIZE){
//		assert(thislen < eb->length);
		self insertIntoBlock(neb, 0, &eb->text[eb->length - thislen], 
						thislen, 0);
		eb->length -= thislen;
		return;
		}

		// Double ukk, we've got to split the blocks!

	nneb = addBlock(eb, &eb->text[eb->length - thislen], thislen);

		// If we run out of memory, we're in big trouble

	if	(nneb == 0)
		return;
	eb->length -= thislen;
	nneb->lineCount = 0;
	self insertIntoBlock(nneb, nneb->length, &neb->text[0], nextlen, 1);
	neb->length -= nextlen;
	neb->lineCount--;
	memMove(&neb->text[0], &neb->text[nextlen], neb->length);
//	printf("eb = %lx neb = %lx eb len = %d neb len = %d this = %d next = %d\n",
//		eb, neb, eb->length, neb->length, thislen, nextlen);
//	printf("nneb = %lx next = %lx len = %x text = %s\n", nneb, nneb->next,
//			nneb->length, &nneb->text);
	}

getLineno:	dynamic	(line: * lineNum, col: * int) =
	{
	i:	int;
	cp:		* char;
	lastline:	* char;

	if	(accessBlock == 0){
		*line = 0;
		*col = 0;
		return;
		}
	i = accessLines;
	for	(lastline = cp = accessBlock->text; cp < accessPtr; cp++)
		if	(*cp == '\n'){
			i++;
			lastline = cp + 1;
			}
	*col = accessPtr - lastline;
	*line = i;
	}

search:	dynamic	(fndPos: * int,		// found line number and offset in line
	 sp: * searchPattern) int =	// search pattern (compiled regular
					// expression)
	{
	ln:		int;
	offset:		int;
	remainder:	int;
	eb:		* fileBlock;
	cp:		* char;
	endp:		* char;
	foundLen:	int;
	i:		int;
	match:		* char;
	xpos:		int;

	if	(accessBlock == 0)
		return -1;
	xpos = pos;
	eb = accessBlock;
	cp = accessPtr;
	endp = &eb->text[eb->length];
	for	(;;){
		foundLen = sp search(cp, endp, &match);
		if	(foundLen != -1)
			break;
		xpos += endp - cp;
		eb = eb->next;
		if	(eb == 0)
			return -1;
		offset = 0;
		cp = &eb->text[0];
		endp = cp + eb->length;
		}
	xpos += match - cp;
	seek(xpos, 0);
	*fndPos = xpos;
	return foundLen;
	}

hasChanged:	dynamic	() int =
	{
	return flags & CHANGED;
	}

	accessBlock:		* fileBlock;
	accessOffset:		int;
	accessPtr:		* char;
	accessLines:		unsigned;
	pos:			textPos;

private:

newBlock:	(newData: pointer, len: unsigned) * fileBlock =
	{
	cp:	* char;
	eb:	* fileBlock;

	eb = alloc(sizeof (fileBlock));
	if	(eb == 0)
		return(0);
	eb->next = 0;
	eb->prev = lastBlock;
	if	(lastBlock)
		lastBlock->next = eb;
	else
		blocks = eb;
	lastBlock = eb;
	eb->length = len;
	memMove(&eb->text[0], newData, len);
	eb->lineCount = 0;
	for	(cp = &eb->text[0]; len; len--, cp++){
		if	(*cp == '\n')
			eb->lineCount++;
		}
	return(eb);
	}

splitBlock:	(eb: * fileBlock) =
	{
	neweb:		* fileBlock;
	cp:		* char;
	off:		int;
	newlines:	int;
	lastline:	int;

	lastline = 0;
	cp = &eb->text[0];
	off = 0;
	newlines = 0;
	do	{
		if	(*cp == '\n'){
			if	(off >= BLOCKSIZE / 2){
				thisdiff:	int;
				lastdiff:	int;

				thisdiff = (off + 1) - (BLOCKSIZE / 2);
				lastdiff = (BLOCKSIZE / 2) - lastline;
				if	(thisdiff < lastdiff){
					off++;
					if	(off >= eb->length)
						return;
					newlines++;
					}
				else
					off = lastline;
				neweb = self addBlock(eb, &eb->text[off],
							eb->length - off);
				if	(neweb == 0)
					return;
				neweb->lineCount = eb->lineCount - newlines;
				eb->lineCount = newlines;
				eb->length = off;
				accessLines = 0x7fff;
				return;
				}
			newlines++;
			lastline = off + 1;
			}
		off++;
		cp++;
		}
		while	(off < eb->length);
	}

addBlock:	(eb: * fileBlock, newData: * char, len: int) * fileBlock =
	{
	neweb:	* fileBlock;

	neweb = alloc(sizeof (fileBlock));
	if	(neweb == 0)
		return(0);
	neweb->length = len;
	memMove(&neweb->text[0], newData, len);
	neweb->prev = eb;
	if	(eb){
		if	(eb->next)
			eb->next->prev = neweb;
		else
			lastBlock = neweb;
		neweb->next = eb->next;
		eb->next = neweb;
		}
	else	{
		blocks->prev = neweb;
		neweb->next = blocks;
		blocks = neweb;
		}
	return(neweb);
	}

insertIntoBlock: (eb: * fileBlock, off: unsigned,
				 newData: pointer, len: unsigned,
				 newlines: int) =
	{
	if	(off < eb->length)
		memMove(&eb->text[off + len], &eb->text[off],
						eb->length - off);
	memMove(&eb->text[off], newData, len);
	eb->length += len;
	textLength += len;
	eb->lineCount += newlines;
	lnCount += newlines;
	flags |= CHANGED;
	}

	lnCount:		lineNum;
	blocks:			* fileBlock;
	lastBlock:		* fileBlock;
	textLength:		textPos;
	flags:			editFlags;
	};

fileBlock:	public	type	{
	public:

	next:		* fileBlock;
	prev:		* fileBlock;
	length:		unsigned;
	lineCount:	unsigned;
	text:		[BLOCKSIZE] char;
	};

editFlags:	type	char = {
	CHANGED = 0x01,
	NEWFILE = 0x02,
	};

