/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	alys;
include	hardware, list;
include	hprocess;
include	arena;
include	object;
include	kprintf;
include	karena;

NPROCS:		public	const int = (_GDT_TSS_TOP - _GDT_TSS_BASE) >> 4;
MAXAGE:			const int = 7;		/* Maximum process aging */

ProcAcquire:	semaphore;
SuspendEvent:	semaphore;
ProcessTable:	public	[NPROCS] ref process = [ &Process0 ];
Process0:		process;
CurProc:	public	ref process = &Process0;
FloatProc:	public	ref process = 0;

processorIdle:	public	() boolean =
	{
	return RunQ isEmpty();
	}

RunQ:			queue;

displayProcesses:	public	() =
	{
	i:	int;

	for	(i = 0; i < NPROCS; i++){
		if	(ProcessTable[i] == 0)
			continue;
		p:	ref process;
		p = ProcessTable[i];

		if	(p->animates)
			kprintf("p %d: %8s animates %d(%p) arena %p resrc %p(%c) stack [%x-%x]\n", i,
				pStatusName[p->status],
				p->animates->me, p->animates, p->where, p->resource, p->alertable ? 'Y' : 'N',
				p->kernelStack, &p->kernelStack[KERNEL_STACK]);
		else
			kprintf("p %d: %8s arena %x resrc %p(%c) stack [%x-%x]\n", i,
				pStatusName[p->status],
				p->where, p->resource, p->alertable ? 'Y' : 'N',
				p->kernelStack, &p->kernelStack[KERNEL_STACK]);

		}
	}

pStatusName:	[] ref char = [
	"PFREE",
	"PNEW",
	"PRUN",
	"PSPIN",
	"PWAIT",
	"PDEAD",
	"PSTOP",
	"PINVALID"
	];

process:	public	type	inherit	queue	{
	public:

	index:		unsigned[16];

		// CPU scheduling parameters

	status:		pStatus_t;
	basepriority:	byte;
	cpu:		byte;
	aging:		byte;
	priority:	byte;

		// Event wait info

	alertable:	boolean;
	resource:	ref semaphore;

		// Timing info

	times:		processTimes_t;
	childtimes:	processTimes_t;

		// Process hardware state

	task:		task_t;
	kernelStack:	[KERNEL_STACK] byte;
	kernelThread:	threadContext;
	userThread:	vaddr_t;

	myJob:		ref far job;
	animates:	ref object;
	where:		ref arena;

create:	factory	() ref process =
	{
	i:	int;
	p:	ref process;

	ProcAcquire down(FALSE);
	for	(i = 0;; i++){
		if	(i >= _KERNEL_PROCS){
			ProcAcquire up();
			return 0;
			}
		if	(ProcessTable[i] == 0)
			break;
		}
	self = ProcessTable[i] = new process;
	index = i;
	ProcAcquire up();
	status = PNEW;
	basepriority = PR_NORMAL;
	priority = PR_NORMAL;
	cpu = 0;
	aging = 0;
	times = [ 0, 0 ];
	childtimes = [ 0, 0 ];
	task = [];
	animates = 0;
	myJob = CurProc->myJob;
	where = 0;
	userThread = 0;
	kernelThread = [ 0, 0, 0, &Heap ];
	return self;
	}

free:	() =
	{
	n:	threadLock;

	n lock();
	ProcessTable[index] = 0;
	n unlock();
	Heap free(self);
	}

describe:	(p: ref process_t) =
	{
	p->status = status;
	p->pid = index;
//	p->parent = parent;
	p->basepriority = basepriority;
	p->priority = priority;
	p->times = times;
	p->childtimes = childtimes;
	}

setPriority:	(nprio: byte) int =
	{
	n:		threadLock;
	b:		byte;

//	if	(userid &&
//		 nprio >= PR_LOW)
//		return ERRPERMISSION;
	b = basepriority;
	basepriority = nprio;
	n lock();
	extract();		// remove myself from the run queue
	setRunnable();		// and reschedule at new priority
	n unlock();
	return b;
	}

maskPriority:	(nprio: byte) int =
	{
	b:		byte;

	b = priority;
	priority = nprio;
	return b;
	}

process0constructor:	() =
	{
	basepriority = PR_NORMAL;
	loadProc0Hardware(&task);
	kernelThread = *_Thread;
	_Thread = &kernelThread;
	}

suspend:	public	() =
	{
	SuspendEvent down(TRUE);	// Only a signal can wake us
	}

raise:	() =
	{
	exit(EX_RAISE);
	}

abort:	(code: unsigned) =
	{
	if	(animates coreDump())
		code |= EX_CORE;
	exit(EX_ABORT | code);
	}

exit:	(code: unsigned) =
	{
	if	(animates->context)
		animates->context = animates->context close();
	KernelArena bind(self);
	setPriority(PR_HIGH);		// make uninterruptable
	animates->parent childExit(code);

		// At this point we have a scheduling problem.  The CurProc
		// memory must be freed, but we are still executing on it.
		// Therefore, the object switch has to kill the process and
		// not let it regain control.  If somehow, control gets here,
		// we drop our priority and idle.

	kprintf("exit done\n");
	setPriority(0);
	for	(;;)
		;
	}
/*
	The parent process calls this function to clean up the child.
	The child hopefully will never regain control.
 */
termination:	() =
	{
	if	(FloatProc == self)
		FloatProc = 0;
	free();
	}

ageProcess:	() =
	{
	if	(--cpu != 0 ||
		 priority >= PR_HIGH)
		return;
	cpu = 1 << aging;
	if	(aging < MAXAGE){
		aging++;
		if	(aging > basepriority)
			aging = basepriority;
		priority = basepriority - aging;
		extract();
		schedule();
		}
	}
/*
 *	FUNCTION:	setRunnable
 *
 *	DESCRIPTION:
 *		This function places a process on the run queue.
 *
 *	INPUTS:
 *		self		The process to be set runnable.
 *
 *	OUTPUTS:
 *		None.
 */
setRunnable:	public	() =
	{
	status = PRUN;
	priority = basepriority;
	aging = 0;
	cpu = 1;
	schedule();
	}
/*
 *	FUNCTION:	schedule
 *
 *	DESCRIPTION:
 *		This function places a process on the run queue.
 *
 *	INPUTS:
 *		self		The process to be put on the process
 *				queue.
 *
 *	OUTPUTS:
 *		None.
 */
schedule:	() =
	{
	sp:	* process;
	n:	threadLock;

	n lock();
//	kprintf("schedule %x pri = %d RunQ %x\n", self, priority, RunQ.next);
	for	(sp = pointer(RunQ.next); sp != &RunQ; sp = pointer(sp->next)){
		if	(sp->priority < priority)
			break;
		}
	sp insert(self);
	n unlock();
	processSwitch();		// switch processes if necessary
	}

arrangeCall:	(f: pointer) =
	{
	hardwarePushCall(f, &task);
	}

	};
/*
	This function does a process switch.  If the new
	process is not the same as the current process, an
	assortment of pointers are replaced.

	In short, a process context consists of the following:
		1. An kernel stack.
		2. An arena structure.
		3. A user stack.
		4. MMU settings.
		5. FPP (80x87) save area (not used).
		6. CurProc pointer.

 *		Note that this function contains the system idle loop.
 */
processSwitch:	public	() =
	{
	pindex:		int;
	n:		threadLock;

	if	(PanicButton ||
		 CurArena->latch)
		return;
	n lock();

		// ALYS idle loop

//	kprintf("i");
	while	(RunQ isEmpty())
		allowInterrupts();
//	kprintf(".");

	if	(pointer(RunQ.next) != CurProc){
//		kprintf("switch %d.->", CurProc->animates->me);
		CurProc->userThread = CurArena peekThread();
		CurProc = pointer(RunQ.next);
//		kprintf("%d.\n", CurProc->animates->me);
		CurArena = CurProc->where;
		_Thread = &CurProc->kernelThread;
		CurArena pokeThread(CurProc->userThread);
		hardwarePswitch(CurProc->index, &CurProc->task);
		}
	n unlock();
	}
/*
	This function establishes the RunQ, the queue of runnable
	processes, it then sculpts process 0 from scratch (remember that
	process 0 is pre-allocated above).  
 */
initializeProcesses:	entry	() =
	{
	RunQ constructor();
	Process0 process0constructor();
	ProcAcquire = [ 1 ];
	SuspendEvent = [ 0 ];
	}

serialRecursiveLock:	public	type	{
	counter:	semaphore;
	owner:		ref process;
	depth:		int;

	public:

constructor:	() =
	{
	owner = 0;
	counter = [ 1 ];
	depth = 0;
	}

down:	(alertable: boolean) boolean =
	{
	n:	threadLock;

	n lock();
	if	(CurProc == owner){
		depth++;
		n unlock();
		return TRUE;
		}
	if	(counter down(alertable)){
		owner = CurProc;
		depth = 1;
		n unlock();
		return TRUE;
		}
	else
		return FALSE;
	}

up:	() =
	{
	if	(CurProc == owner){
		depth--;
		if	(depth == 0){
			n:	threadLock;

			n lock();
			owner = 0;
			counter up();
			n unlock();
			}
		}
	else
		panic("Open of an open serialRecursiveLock: %p\n", self);
	}

	};

semaphore:	public	type	inherit	queue	{
	public:

	count:		int;

constructor:	(n: int) =
	{
	count = n;
	super constructor();
	}

/*
 *	FUNCTION:	down
 *
 *	DESCRIPTION:
 *		This function decrements a resource count and suspends
 *		the calling process until resources are available.
 */
down:	(alertable: boolean) boolean =
	{
	n:	threadLock;

	n lock();
	if	(--count < 0){
//		kprintf("%x->%x down count %d\n", CurProc, self, count);
//		kprintf("RunQ = %x:[%x,%x]->[%x,%x]\n",
//			&RunQ, RunQ.prev, RunQ.next, RunQ.next->prev,
//			RunQ.next->next);
		CurProc->alertable = alertable;
		CurProc->resource = self;
		CurProc->status = PWAIT;
		CurProc extract();
		enqueue(CurProc);
		processSwitch();
		n unlock();
		if	(CurProc->alertable == alertable)
			return TRUE;
		else
			return FALSE;
		}
	else	{
		n unlock();
		return TRUE;
		}
	}
/*
 *	FUNCTION:	downNowait
 *
 *	DESCRIPTION:
 *		This function decrements a resource count and returns
 *		failure if none are currently available.
 */
downNowait:	() boolean =
	{
	n:	threadLock;

	n lock();
	if	(count <= 0){
		n unlock();
		return FALSE;
		}
	else	{
		--count;
		n unlock();
		return TRUE;
		}
	}
/*
 *	FUNCTION:	up
 *
 *	DESCRIPTION:
 *		This function gives a resource back, freeing the first
 *		waiting process, if any.
 */
up:	() =
	{
	p:		ref process;
	n:		threadLock;

	n lock();
	if	(count++ < 0){
		p = pointer(dequeue());
//		kprintf("%x<-%x up count %d\n", p, self, count);
//		kprintf("RunQ = %x:[%x,%x]->[%x,%x]\n",
//			&RunQ, RunQ.prev, RunQ.next, RunQ.next->prev,
//			RunQ.next->next);
		p->resource = 0;
		p setRunnable();
		}
	n unlock();
	}
/*
 *	FUNCTION:	wakeup
 *
 *	DESCRIPTION:
 *		This function frees all processes currently blocked
 *		on the resource.
 */
wakeup:	() =
	{
	while	(count < 0)
		up();
	}
/*
 *	FUNCTION:	empty
 *
 *	DESCRIPTION:
 *		This function clears a resource, effectively
 *		emptying a partially filled queue.  Note that if there
 *		are waiting processes for an unavailable resource, this call
 *		has no effect.
 */
empty:	() =
	{
	n:	threadLock;

	n lock();
	if	(count > 0)
		count = 0;
	n unlock();
	}
/*
	This function aborts a process that is in an alertable wait.  By
	clearing the value of alertable, this is a signal that the wait
	aborted.
 */
abort:	(p: ref process) =
	{
	n:		threadLock;

	n lock();
	if	(p->alertable){
		count++;
		p->alertable = FALSE;
		p extract();
		p setRunnable();
		}
	n unlock();
	}
/*
	This function will break the waits on this semaphore.  If there is a
	process in an unbreakable wait, this function will only wake up
	processes ahead of it in the queue.
 */
startle:	() =
	{
	n:	threadLock;
	p:	ref process;

	n lock();
	if	(count < 0){
		p = ref process(next);
		if	(p->alertable)
			abort(p);
		}
	n unlock();
	}
};
