/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
/*
	There may be any number of implementation supported integral
	types.  The char, int and long sizes are guaranteed to exist.
	The char size is the smallest size that can be stored.  The
	int size is the smallest computational size.  The long size
	is the largest supported size.
 */
CHARBITS:	public	const	int = 8;	// bits per char
INTBITS:	public	const	int = 32;	// bits per int
LONGBITS:	public	const	int = 32;	// bits per long

FLOATBITS:	public	const	int = 32;
DOUBLEBITS:	public	const	int = 64;
EXTENDBITS:	public	const	int = 80;

CCLEN:		public	const	int = 4;	// max # of chars in character
						// constant
/*
	This function accepts a long integer value and reduces it to a
	value that is representable within the number of bits given by
	the bits parameter.  If the bits value is negative, the number
	should be treated as a signed quantity and the value should be
	truncated to a signed integer value.

	Limitations:

	This code only works for 2's complement signed integers.

	You can only truncate a value that can be represented as a
	long.  Obviously, larger values must be handled differently.
	This has significance when this code is used in a cross-compiler
	that targets a machine with a larger long than the host's.
 */
integerConstant:	public	(i: long, bits: int) long =
	{
	mask:	long;
	sign:	int;

	sign = bits;			// remember the signedness
	if	(bits < 0)
		bits = -bits;		// force bits to positive
	if	(bits >= 32)
		return i;
	mask = 1L << bits - 1;
	i &= mask;
	if	(sign < 0 && 
		 i & (1 << (bits - 1)))
		i |= ~mask;		// sign extend
	return i;
	}
/*
	This function determines whether the given value is representable
	in the number of bits given.

	Notes:

	This code only works using 2's complement signed integers on the
	host.  The target can use 1's complement.

	The number of bits is reduced by one for signed integers, since
	the extra bit is the sign.
 */
integerFits:	public	(i: long, bits: int) boolean =
	{
	mask:	long;
	sign:	int;

	sign = bits;			// remember the signedness
	if	(bits < 0)
		bits = -bits - 1;	// force bits to positive
	mask = ~(1L << bits - 1);
	i &= mask;
	if	(i == 0)
		return TRUE;		// a positive value!
	if	(sign && i == mask)
		return TRUE;		// a negative signed value!
	else
		return FALSE;		// not representable
	}
/*
/*
	SHIFT_BITS is the number of significant bits used in a shift
	operation.  In an 80386, this is 8 since shift instructions use
	8-bit registers.
 */
SHIFT_BITS:	public	const	int = 8;

UnsignedTypes:	public	[] numberTypeDescriptor = [
	[ 8,  8 ],					// char
	[ 16, 16 ],					// short
	[ 32, 32 ]					// long
	];

SignedTypes:	public	[] numberTypeDescriptor = [
	[ 8,  8 ],					// char
	[ 16, 16 ],					// short
	[ 32, 32 ]					// long
	];

FloatingTypes:	public	[] numberTypeDescriptor = [
	[ 32, 32 ],					// float
	[ 64, 64 ],					// double
	[ 80, 16 ]					// extend
	];
/*
	This following array describes the order in which members
	should be allocated in a structure.  In a union, all members
	have offset zero.  In a packed structure, all members are
	allocated in declaration order.  In an unpacked structure,
	all members are allocated in descending order of alignment.
	The following table indicates the alignments needed.  Note
	that an entry of -1 indicates where the dynamic vector pointer
	should be allocated.
 */
StructLayout:	public	[] short = [
	64,
	-1,
	32,
	16,
	8
	];
/*
	This function returns an updated bit offset given a starting
	bit offset and a type.  The assumption is that the next member
	of a structure to be allocated has type t, and the current
	cumulative bit offset is given by bitOffset.  The return value
	is the possibly padded bit offset.
 */
packedAdjustment:	public	(bitOffset: addr_t, t: ref type_s) addr_t =
	{
	width:		int;
	bytePart:	int;

	if	(t->topType != T_SIGNED &&
		 t->topType != T_UNSIGNED)
		return (bitOffset + 7) & ~7;
	bytePart = bitOffset & 7;
	width = t bitSizeOf();
	if	(width + bytePart >= LONGBITS)
		return (bitOffset + 7) & ~7;
	else
		return bitOffset;
	}

immediateSconstArgument: public	(ref tree_p, ref type_s) ref tree_p =
	{
	return 0;
	}

PUSH_TEMPS_ALLOWED:	public	const	boolean = TRUE;
 */