/*	This is a simple program to find the square root of a number
**	using a power series expansion.
**
**	sqrt(y) = sqrt(1 - x)		y = 1 - x, or x = 1 - y
**
**	f(x) = (1 - x) ^ (1/2)				f(0) = 1
**
**	f'(x)/1! = - (1/2) * (1 - x) ^ (-1/2)		f'(0)/1! = -(1/2)
**
**	f"(x)/2! = - (1/8) * (1 - x) ^ (-3/2)		f"(0)/2! = -(1/8)
**
**	f"'(x)/3! = -(1/16) * (1 - x) ^ (-5/2)		f"'(0)/3! = -(1/16)
**
**	f""(x)/4! = -(5/128) * (1 - x) ^ (-7/2)		f""(0)/4! = -(5/128)
**
**	Recursion relation:
**
**	f[n](0)/n! = ((2n - 3) / (2 * n)) * f[n-1](0)	n > 1
**
**	Thus:
**
**	sqrt(1 - x) = 1 - (1/2) * x - (1/8) * x ^ 2 - ... + f[n](0)/n! * x ^ n
**			+ ...
**
*/
#include <stdio.h>
#include "fmlib.h"

#define LENGTH 48		/* Up to 44 byte mantissa, 4 byte exponent */
#define TRIALS 10

void
usage()
{
    printf("Usage: root2ext value accuracy mant exp trials\n");
    printf("Where:          value = value to find square root of\n");
    printf("                accuracy = desired accuracy of result\n");
    printf("                mant = number of mantissa bits (23 to 384)\n");
    printf("                exp = number of exponent bits (8 to 31)\n");
    printf("                trials = number of repeat calcuations\n");
    exit(-1);
}

void
main(argc, argv)
int argc;
char *argv[];
{
    unsigned char y[LENGTH];	/* Value to find square root of, this
				 * should be between 0 and 2. */
    unsigned char x[LENGTH];	/* Value used in series expansion, this
				 * should between -1 and 1 */
    unsigned char term[LENGTH];	/* Value of specific term */
    unsigned char sum[LENGTH];	/* Sum of terms */
    unsigned char limit[LENGTH]; /* Accuracy limit */
    unsigned char two_n[LENGTH]; /* Temporary variable for (2 * n) */
    unsigned char three[LENGTH]; /* Value of 3 */
    unsigned char one[LENGTH];	/* Value of 1 */
    unsigned char temp[LENGTH];	/* Temporary variable */
    unsigned char cc;		/* Condition codes */

    char string_array[128];	/* String conversion area */

    int mant_bits;		/* Number of bits in the mantissa */
    int exp_bits;		/* Number of bits in the exponent */
    int trials;			/* Number of trials */
    int n;			/* Current term number */
    int twon;
    int i;			/* Copy index */
    int j;			/* Trial index */

    /* Check input arguments for acceptable values */
    if (argc != 6)
	usage();

    if (sscanf(argv[3], "%d", &mant_bits) != 1)
	usage();

    if (sscanf(argv[4], "%d", &exp_bits) != 1)
	usage();

    if (sscanf(argv[5], "%d", &trials) != 1)
	usage();

    if ((mant_bits < 23) || (mant_bits > 384) ||
	(exp_bits < 8) || (exp_bits > 31))
	usage();

    for (j = 0; j < trials; j++)
    {
	/* Convert value to find square root of and accuracy */
	strtoflt(argv[1], 128, y, mant_bits, exp_bits);
	strtoflt(argv[2], 128, limit, mant_bits, exp_bits);

	/* Calculate constants, constant 3 for recursion relation */
	strtoflt("3", 2, three, mant_bits, exp_bits);
	/* Sum = 1.0, initial sum */
	strtoflt("1.0", 12, sum, mant_bits, exp_bits);

	/* x = 1 - y, Calculate x value to use in series */
	for (i = 0; i < LENGTH; i++)
	    x[i] = sum[i];
	fsubm(x, y, exp_bits, mant_bits);

	/* Calculate initial term = 0.5 * x, n = 1 term */
	strtoflt("0.5", 4, term, mant_bits, exp_bits);
	cc = fmultm(term, x, exp_bits, mant_bits);
	n = 2;			/* Next term to calculate */

	/* Loop until term is less than limiting value.  Note that for
	 * an alternating series, only the positive values are tested. */
	while ((fcmpm(term, limit, mant_bits, exp_bits) > 0) ||
	      ((cc & FFNEG) != 0))
	{
	    /* sum -= term */
	    fsubm(sum, term, exp_bits, mant_bits);

	    /* term(n) = term(n-1) * (x * ((2 * n) - 3) / (2 * n)) */
	    twon = 2 * n;
	    intoflt(&twon, 4, two_n, mant_bits, exp_bits);
	    for (i = 0; i < LENGTH; i++)
		temp[i] = two_n[i];
	    fsubm(temp, three, exp_bits, mant_bits);
	    fdivm(temp, two_n, exp_bits, mant_bits);
	    fmultm(temp, x, exp_bits, mant_bits);
	    cc = fmultm(term, temp, exp_bits, mant_bits);
	    n++;
	}
    }
    printf("Done...\n");

    /* Print out the result */
    fltostr(sum, mant_bits, exp_bits, string_array, 128);
    printf("Iteration %d is: %s\n", n, string_array);

    return;
}
    
