/* Extended IEEE Compatible Floating Point Arithmetic Library
**
** Version 1.0
** Copyright (C) 1990, by Fred Motteler
** All Rights Reserved
**
** This is a simple extended floating point arithmetic package.
** It must be used with the extended integer arithmetic package.
**
** Format:
**
** |S| n bit signed exponent | m bit fraction |
**
** The format is compatible with IEEE Standard for Binary Floating
** when:
**		n = 8,   m = 23		single precision
**		n = 11,  m = 52		double precision
**
** If n = 15 and m = 64, this format is almost compatible with IEEE
** extended precision format.  The main difference are
**	1) IEEE extended precision format is a 12 byte format with 10 bytes of
**	   information.  Bits 64 thru 80 are zero.
**	2) The most significant bit of the mantissa (implied in single and
**	   double precision IEEE formats) is explicitly included.
**
** Each routine that returns a floating point result also returns a condition
** code value:
**
**	0 0 0 0 N Z I n
**	Where N = negative (FFNEG)
**	      Z = zero (FFZERO)
**	      I = infinite (FFINF)
**	      n = not a number (FFNAN)
**
** Each routine that returns an integer result also returns a condition
** code value:
**
**	0 0 0 0 Z V S C
**	Where Z = zero (ZERO)
**	      V = overflow (OVERFLOW)
**	      S = sign (SIGN)
**	      C = carry (CARRY)
**
** This is an include file containing the actual tester data.  There are
** tow data tables that contain test cases for:
**
**	string to integer conversions
**	string to float conversions
**
*/

/* Structures for testing numeric conversion operations. */
/* String to integer. */
struct sivals {
    char *operationS;			/* Pointer to operation type string */
    char *intinBP;			/* Pointer to input integer string */
    int intolenN;			/* Byte length of output integer */
    char *intoBP;			/* Pointer to expected output string */
    int ccresultN;			/* Resulting condition code */
    } sitableAH[] = {
	{"Basic string to integer conversion",
	 "+2547", WORDWIDE,
	 "2547", 0},

	{"Basic string to negative integer conversion",
	 "-12547", WORDWIDE,
	 "-12547", SIGN},

	{"Basic string to zero integer conversion",
	 "0", WORDWIDE,
	 "0", ZERO},

	{"Basic string to negative zero integer conversion",
	 "-0", WORDWIDE,
	 "0", ZERO},

	{"Basic string to almost integer overflow conversion",
	 "32767", WORDWIDE,
	 "32767", 0},

	{"Basic string to almost negative integer overflow conversion",
	 "-32767", WORDWIDE,
	 "-32767", SIGN},

	{"Basic string to almost negative integer overflow conversion",
	 "-32768", WORDWIDE,
	 "-32768", SIGN},

	{"Basic string to just integer overflow conversion",
	 "32768", WORDWIDE,
	 "32767", OVERFLOW},

	{"Basic string to just negative integer overflow conversion",
	 "-32769", WORDWIDE,
	 "-32768", (OVERFLOW | SIGN)},

	{"Basic string to integer overflow conversion",
	 "592874", WORDWIDE,
	 "32767", OVERFLOW},

	{"Basic string to negative integer overflow conversion",
	 "-55204", WORDWIDE,
	 "-32768", (OVERFLOW | SIGN)},

	{"Basic string to byte conversion",
	 "+25", BYTEWIDE,
	 "25", 0},

	{"Basic string to almost byte overflow conversion",
	 "127", BYTEWIDE,
	 "127", 0},

	{"Basic string to just byte overflow conversion",
	 "128", BYTEWIDE,
	 "127", OVERFLOW},

	{"Basic string to almost negative byte overflow conversion",
	 "-127", BYTEWIDE,
	 "-127", SIGN},

	{"Basic string to almost negative byte overflow conversion",
	 "-128", BYTEWIDE,
	 "-128", SIGN},

	{"Basic string to just negative byte overflow conversion",
	 "-129", BYTEWIDE,
	 "-128", (SIGN | OVERFLOW)},

	{"Basic string to long conversion",
	 "-13995725", LONGWIDE,
	 "-13995725", SIGN},

	{"Basic string to double long conversion",
	 "+28339681396995725", DOUBLEWIDE,
	 "28339681396995725", 0},

	{"Basic string to quad long conversion",
	 "-27893133966388133692679490587125", QUADWIDE,
	 "-27893133966388133692679490587125", SIGN},

	{"String with decimal point to integer conversion",
	 "+25.47", WORDWIDE,
	 "25", 0},

	{"String with non decimal digit to integer conversion",
	 "+25a47", WORDWIDE,
	 "25", 0},

	{"String with non decimal digit to integer conversion",
	 "+25-47", WORDWIDE,
	 "25", 0},

	{ NULL,
	  NULL, 0,
	  NULL, 0}};

/* String to floating point. */
struct sfvals {
    char *operationS;			/* Pointer to operation type string */
    char *fltinBP;			/* Pointer to input float string */
    int fltofracN;			/* Precision of float mantissa */
    int fltoexpN;			/* Precision of float exponent */
    int fltolenN;			/* Byte length required by float */
    char *fltoutBP;			/* Pointer to expected output float */
    int ccresultN;			/* Resulting condition code */
    } sftableAH[] = {
	{"Basic string to float conversion",
	 "105.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.10509995e+3", 0},

	{"Basic string to negative float conversion",
	 "-2105.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-.21050994e+4", FFNEG},

	{"Basic string to zero float conversion",
	 "0", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.0e+0", FFZERO},

	{"Basic string to underflow float conversion",
	 "+.35948681e-45", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.0e+0", FFZERO},

	{"Basic string to almost underflow float conversion",
	 "+.39999999e-37", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.39999972e-37", 0},

	{"Basic string to just underflow float conversion",
	 "+.19999999e-38", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.0e+0", FFZERO},

	{"Basic string to overflow float conversion",
	 "+.35948681e-45", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.0e+0", FFZERO},

	{"Basic string to almost overflow float conversion",
	 "+.23999992e+39", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.23999992e+39", 0},

	{"Basic string to just overflow float conversion",
	 "+.47999992e+39", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+Infinity", FFINF},

	{"Basic string with too many digits for float to float conversion",
	 "-3459.47999992e+24", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-.34594788e+28", FFNEG},

	{"Basic string with too many digits for int to float conversion",
	 "345947999992", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.34594784e+12", 0},

	{"Basic string to double conversion",
	 "1305.13677389", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+.1305136773889999e+4", 0},

	{"Basic string to negative double conversion",
	 "-12552105.13472", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-.1255210513471998e+8", FFNEG},

	{"Basic string to extended conversion",
	 "1569305.1336811677389", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 "+.15693051336811677377e+7", 0},

	{"Basic string to negative extended conversion",
	 "-122834619552105.13472", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 "-.12283461955210513465e+15", FFNEG},

	{"Basic string to sixteen conversion",
	 "1305.13677389", SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 "+.13051367738899999999999999999999e+4", 0},

	{"Basic string to negative sixteeen conversion",
	 "-12552105.13472", SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 "-.12552105134719999999999999999996e+8", FFNEG},

	{"Basic string with bad digit for float conversion",
	 "3459cx793599992", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.34589996e+4", 0},

	{"Basic string with white space for float conversion",
	 "3459 793599992", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.34589996e+4", 0},

	{"Basic string with bad exponent digit for float conversion",
	 "3459e+x", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.34589996e+4", 0},

	{"Basic string with no exponent sign for float conversion",
	 "3459e4", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.34589996e+8", 0},

	{"Basic string with bad exponent digit for float conversion",
	 "3459e+4qq", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.34589996e+8", 0},

	{"Basic string with no valid digits for float conversion",
	 "Hi there!", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.0e+0", 0},

	{ NULL,
	  NULL, 0, 0, 0,
	  NULL, 0}};
