/* Extended IEEE Compatible Floating Point Arithmetic Library
**
** Version 1.0
** Copyright (C) 1990, by Fred Motteler
** All Rights Reserved
**
** This is a simple extended floating point arithmetic package.
** It must be used with the extended integer arithmetic package.
**
** Format:
**
** |S| n bit signed exponent | m bit fraction |
**
** The format is compatible with IEEE Standard for Binary Floating
** when:
**		n = 8,   m = 23		single precision
**		n = 11,  m = 52		double precision
**
** If n = 15 and m = 64, this format is almost compatible with IEEE
** extended precision format.  The main difference are
**	1) IEEE extended precision format is a 12 byte format with 10 bytes of
**	   information.  Bits 64 thru 80 are zero.
**	2) The most significant bit of the mantissa (implied in single and
**	   double precision IEEE formats) is explicitly included.
**
** Each routine returns a condition code value:
**
**	0 0 0 0 N Z I n
**	Where N = negative
**	      Z = zero
**	      I = infinite
**	      n = not a number
**
** This is an include file containing the actual tester data.  There are
** three data tables that contain test cases for:
**
**	integer to float conversions
**	float to integer conversions
**	float to float conversions
**
*/

/* Structures for testing numeric conversion operations. */
/* Integer to floating point. */
struct ifvals {
    char *operationS;			/* Pointer to operation type string */
    char *intinBP;			/* Pointer to input integer string */
    int intinlenN;			/* Byte length of input integer */
    char *fltoutBP;			/* Pointer to expected output string */
    int fltoutfracN;			/* Precision of float mantissa */
    int fltoutexpN;			/* Precision of float exponent */
    int fltoutlenN;			/* Byte length required by float */
    int ccresultN;			/* Resulting condition code */
    } iftableAH[] = {
	{"Basic byte to float conversion",
	 "23", BYTEWIDE,
	 "+.22999998e+2", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Negative byte to float conversion",
	 "-67", BYTEWIDE,
	 "-.66999992e+2", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Basic word integer to float conversion",
	 "5", WORDWIDE,
	 "+.50000000e+1", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Negative integer to float conversion",
	 "-5", WORDWIDE,
	 "-.50000000e+1", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Maximum word integer to float conversion",
	 "32767", WORDWIDE,
	 "+.32766996e+5", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Zero word integer to float conversion",
	 "0", WORDWIDE,
	 "+.0e+0", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFZERO},

	{"Minimum integer to float conversion",
	 "-32768", WORDWIDE,
	 "-.32767998e+5", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Basic long to float conversion",
	 "458895", LONGWIDE,
	 "+.45889496e+6", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Negative long to float conversion",
	 "-132945", LONGWIDE,
	 "-.13294497e+6", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Basic double long to float conversion",
	 "4588959456822", DOUBLEWIDE,
	 "+.45889592e+13", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Negative double long to float conversion",
	 "-132945499234", DOUBLEWIDE,
	 "-.13294547e+12", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Basic quad long to float conversion",
	 "458895945682234885932", QUADWIDE,
	 "+.45889596e+21", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Negative quad long to float conversion",
	 "-1329454992343958174", QUADWIDE,
	 "-.13294550e+19", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Almost overflow conversion of quad wide into to float",
	 "170141183400000000000000000000000000000", QUADWIDE,
	 "+.17014118e+39", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Negative almost overflow conversion of quad wide into to float",
	 "-170141183400000000000000000000000000000", QUADWIDE,
	 "-.17014118e+39", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Basic word integer to double conversion",
	 "3285", WORDWIDE,
	 "+.3284999999999999e+4", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	  0},

	{"Negative integer to double conversion",
	 "-5188", WORDWIDE,
	 "-.5187999999999999e+4", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	  FFNEG},

	{"Basic word integer to extended conversion",
	 "3285", WORDWIDE,
	 "+.32849999999999999998e+4", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	  0},

	{"Negative integer to extended conversion",
	 "-5188", WORDWIDE,
	 "-.51879999999999999996e+4", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	  FFNEG},

	{"Basic word integer to sixteen conversion",
	 "3285", WORDWIDE,
	 "+.32849999999999999999999999999999e+4",
	  SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	  0},

	{"Negative integer to sixteen conversion",
	 "-5188", WORDWIDE,
	 "-.51879999999999999999999999999998e+4",
	  SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	  FFNEG},

	{ NULL,
	  NULL, 0,
	  NULL, 0, 0, 0,
	  0}};

/* Floating point to integer. */
struct fivals {
    char *operationS;			/* Pointer to operation type string */
    char *fltinBP;			/* Pointer to input float string */
    int fltinfracN;			/* Precision of float mantissa */
    int fltinexpN;			/* Precision of float exponent */
    int fltinlenN;			/* Byte length required by float */
    char *intoutBP;			/* Pointer to expected output int */
    int intoutlenN;			/* Byte length of input integer */
    int ccresultN;			/* Resulting condition code */
    } fitableAH[] = {
	{"Basic float to byte conversion",
	 "105.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "105", BYTEWIDE,
	  0},

	{"Negative float to byte conversion",
	 "-25.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-25", BYTEWIDE,
	  SIGN},

	{"Basic float to word integer conversion",
	 "5.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "5", WORDWIDE,
	  0},

	{"Negative float to word integer conversion",
	 "-5.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-5", WORDWIDE,
	  SIGN},

	{"Basic float to long conversion",
	 "1.384517e+8", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "138451696", LONGWIDE,
	  0},

	{"Negative float to long conversion",
	 "-1255.134", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-1255", LONGWIDE,
	  SIGN},

	{"Basic float to double long conversion",
	 "1.384517e+8", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "138451696", DOUBLEWIDE,
	  0},

	{"Negative float to double long conversion",
	 "-1255.134", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-1255", DOUBLEWIDE,
	  SIGN},

	{"Basic float to quad long conversion",
	 "1.384517e+18", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "1384516861445537792", QUADWIDE,
	  0},

	{"Negative float to quad long conversion",
	 "-1255.134e+10", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-12551339376640", QUADWIDE,
	  SIGN},

	{"Basic double to long conversion",
	 "1.3845172953668e+6", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "1384517", LONGWIDE,
	  0},

	{"Basic negative double to long conversion",
	 "-1255.13473489e+3", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-1255134", LONGWIDE,
	  SIGN},

	{"Basic extended to long conversion",
	 "1.3845172953668e+8", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 "138451729", LONGWIDE,
	  0},

	{"Basic extended to long conversion",
	 "-1255.1347348958e+2", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 "-125513", LONGWIDE,
	  SIGN},

	{"Basic sixteen sixteen to long conversion",
	 "1.384517295366849233e+8", SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 "138451729", LONGWIDE,
	  0},

	{"Basic sixteen byte to long conversion",
	 "-1255.13473489583955e+2", SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 "-125513", LONGWIDE,
	  SIGN},

	{"Float to long conversion overflow",
	 "1.384517e+18", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "2147483647", LONGWIDE,
	  OVERFLOW},

	{"Negative float to long conversion overflow",
	 "-1.384517e+18", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-2147483648", LONGWIDE,
	  (OVERFLOW | SIGN)},

	{"Float to long conversion underflow",
	 ".124e-5", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "0", LONGWIDE,
	  ZERO},

	{"Float to int conversion truncation",
	 ".12490e+2", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "12", LONGWIDE,
	  0},

	{"Negative float to int conversion truncation",
	 "-.12490e+2", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-12", LONGWIDE,
	  SIGN},

	{ NULL,
	  NULL, 0, 0, 0,
	  NULL, 0,
	  0}};

/* Floating point to floating point. */
struct ffvals {
    char *operationS;			/* Pointer to operation type string */
    char *fltinBP;			/* Pointer to input float string */
    int fltinfracN;			/* Precision of float mantissa */
    int fltinexpN;			/* Precision of float exponent */
    int fltinlenN;			/* Byte length required by float */
    char *fltoutBP;			/* Pointer to output float string */
    int fltoutfracN;			/* Precision of float mantissa */
    int fltoutexpN;			/* Precision of float exponent */
    int fltoutlenN;			/* Byte length required by float */
    int ccresultN;			/* Resulting condition code */
    } fftableAH[] = {
	{"Basic float to double conversion",
	 "5.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.5099998950958251e+1", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	  0},

	{"Negative float to double conversion",
	 "-5.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-.5099998950958251e+1", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	  FFNEG},

	{"Zero float to double conversion",
	 "0", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.0e+0", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	  FFZERO},

	{"Basic double to float conversion",
	 "5.1", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+.50999996e+1", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Negative double to float conversion",
	 "-5.1", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-.50999996e+1", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Zero double to float conversion",
	 "0", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+.0e+0", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFZERO},

	{"Big positive double to float overflow conversion",
	 "5.2e+45", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+Infinity", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFINF},

	{"Big negative double to float overflow conversion",
	 "-5.2e+45", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-Infinity", SINGLEFRAC, SINGLEXP, SINGLEN,
	  (FFINF | FFNEG)},

	{"Little positive double to float underflow conversion",
	 "5.2e-45", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+.0e+0", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFZERO},

	{"Little negative double to float underflow conversion",
	 "-5.2e-45", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-.0e+0", SINGLEFRAC, SINGLEXP, SINGLEN,
	  (FFZERO | FFNEG)},

	{"Positive double to float almost overflow conversion",
	 "2.2e+38", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+.22000000e+39", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Negative double to float almost overflow conversion",
	 "-2.2e+38", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-.22000000e+39", SINGLEFRAC, SINGLEXP, SINGLEN,
	 FFNEG},

	{"Positive double to float barely overflow conversion",
	 "4.8e+38", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+Infinity", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFINF},

	{"Negative double to float barely overflow conversion",
	 "-4.8e+38", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-Infinity", SINGLEFRAC, SINGLEXP, SINGLEN,
	  (FFINF | FFNEG)},

	{"Positive double to float almost underflow conversion",
	 "2e-38", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+.19999998e-37", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Negative double to float almost underflow conversion",
	 "-2e-38", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-.19999998e-37", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Positive double to float barely underflow conversion",
	 "1e-38", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+.0e+0", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFZERO},

	{"Negative double to float barely underflow conversion",
	 "-1e-38", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-.0e+0", SINGLEFRAC, SINGLEXP, SINGLEN,
	  (FFZERO | FFNEG)},

	{"Basic float to extended conversion",
	 "5.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "+.50999989509582519528e+1", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	  0},

	{"Basic float to sixteen conversion",
	 "-5.1", SINGLEFRAC, SINGLEXP, SINGLEN,
	 "-.50999989509582519531249999999998e+1",
	  SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	  FFNEG},

	{"Basic double to extended conversion",
	 "5.1", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "+.50999999999999969796e+1", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	  0},

	{"Basic double to sixteen conversion",
	 "-5.1", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 "-.50999999999999969801933730195740e+1",
	  SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	  FFNEG},

	{"Basic extended to float conversion",
	 "5.1", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 "+.50999996e+1", SINGLEFRAC, SINGLEXP, SINGLEN,
	  0},

	{"Basic sixteen to float conversion",
	 "-5.1", SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 "-.50999996e+1", SINGLEFRAC, SINGLEXP, SINGLEN,
	  FFNEG},

	{"Basic extended to double conversion",
	 "5.1", EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 "+.5099999999999998e+1", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	  0},

	{"Basic sixteen to double conversion",
	 "-5.1", SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 "-.5099999999999998e+1", DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	  FFNEG},

	{ NULL,
	  NULL, 0, 0, 0,
	  NULL, 0, 0, 0,
	  0}};
