/* Extended IEEE Compatible Floating Point Arithmetic Library
**
** Version 1.1
** Copyright (C) 1990, by Fred Motteler
** All Rights Reserved
**
** This is part 2 of a simple extended floating point arithmetic package
** tester.
**
** This is a table driven tester.  It does a comprehensive test of:
**	1. Integer / ASCII string conversions
**	2. Float / ASCII decimal conversions
**	3. Integer / Float conversions
**	4. Float / Integer conversions
**	5. Float / Float conversions
**
** Where reasonable, all numeric input and output values are specified
** as ASCII strings.  This greatly simplifies setting up test cases, and
** provides a good test of the string conversion routines.
** Note that these routines recognize zero, but NOT infinite, not a number,
** or denormalized input parameters.  They blindly assume that the arguments
** are valid floating point numbers.
**
** These routines will return valid zero, infinite, and not a number values.
*/
#include <stdio.h>
#ifndef MWC
#include <stdlib.h>
#endif
#include "imlib.h"
#include "ffmlib.h"
#include "fmlib.h"
#include "fmtest.h"

/* Include the tables containing the tester data.  There are three data
** tables that contain test cases for:
**
**	integer to float conversions
**	float to integer conversions
**	float to float conversions
*/
#include "fmtest2.h"

/*
** Function:	int fmtst2(int *testPN)
**
** This is a table driven tester.  It does a comprehensive test of:
**	1. Integer / ASCII string conversions
**	2. Float / ASCII decimal conversions
**	3. Integer / Float conversions
**	4. Float / Integer conversions
**	5. Float / Float conversions
**
** testPN is a pointer to an integer used to total the error count.
**
** Where reasonable, all numeric input and output values are specified
** as ASCII strings.  This greatly simplifies setting up test cases, and
** provides a good test of the string conversion routines.
*/
int
#ifdef PROTOTYPES
fmtst2(int *testPN)
#else
fmtst2(testPN)
int *testPN;
#endif
{
    int i, fracbitsN, expbitsN, totalenN, radixN;
    int intlenN;
    int infracbitsN, inexpbitsN, intotalenN;
    int outfracbitsN, outexpbitsN, outtotalenN;
#ifdef MEMTEST
    int memparcelN;
#endif
    unsigned char op1AB[128], op2AB[128];
    unsigned char condcodeB;
    char resultAB[128], quotientAB[128];
    int errcountN;

    errcountN = 0;

#ifdef MEMTEST
    fmallinit();
#endif

    printf("************************************************\n");
    printf("** Integer to floating point conversion tests **\n");
    printf("************************************************\n");
    i = 0;
    while (iftableAH[i].operationS != NULL)
    {
	/* Determine the byte length of the input integer */
	intlenN = iftableAH[i].intinlenN;

	/* Determine the length and type of the output floating point number */
	expbitsN = iftableAH[i].fltoutexpN;
	fracbitsN = iftableAH[i].fltoutfracN;
	totalenN = iftableAH[i].fltoutlenN;

	printf("Operation %d: %s\n", (*testPN)++, (iftableAH[i].operationS));
#ifdef TEST
	printf("Input integer length: %d\n", intlenN);
	printf("Output float exponent bits: %d, Mantissa bits: %d, Total byte length: %d\n",
		expbitsN, fracbitsN, totalenN);
#endif

	/* Convert the first operand from a string to an integer. */
	strtoint(iftableAH[i].intinBP, strlen(iftableAH[i].intinBP), op1AB,
		 intlenN, 10);
#ifdef TEST
	printf("Integer as string: %s\n", iftableAH[i].intinBP);
	foutput("Integer in binary: ", op1AB, intlenN);
#endif

	/* Perform the conversion. */
	condcodeB = intoflt(op1AB, intlenN, op2AB, fracbitsN, expbitsN);

#ifdef TEST
	foutput("Result as float: ", op2AB, totalenN);
#endif
	/* Convert the result back to a string. */
	fltostr(op2AB, fracbitsN, expbitsN, resultAB, 128);
#ifdef TEST
	printf("Result as string: %s\n", resultAB);
	printf("Result should be: %s\n", iftableAH[i].fltoutBP);
	printf("Result condition code: 0x%x\n", condcodeB);
	printf("Condition code should be: 0x%x\n", iftableAH[i].ccresultN);
#endif
	if ((strcmp(resultAB, iftableAH[i].fltoutBP) == 0) &&
	    (((int) condcodeB) == iftableAH[i].ccresultN))
	{
	    printf(" [(-:} OK {:-)]\n");
	}
	else
	{
	    printf(" [)-:}***** ERROR *****{:-(]\n");
	    errcountN++;
	}
#ifdef MEMTEST
	if ((memparcelN = fmallreport()) != 0)
	{
	    printf("%d memory blocks not free'd\n", memparcelN);
	    errcountN += memparcelN;
	}
#endif
	i++;
    }

    printf("************************************************\n");
    printf("** Floating point to integer conversion tests **\n");
    printf("************************************************\n");
    i = 0;
    while (fitableAH[i].operationS != NULL)
    {
	/* Determine the length and type of the input floating point number */
	expbitsN = fitableAH[i].fltinexpN;
	fracbitsN = fitableAH[i].fltinfracN;
	totalenN = fitableAH[i].fltinlenN;

	/* Determine the byte length of the output integer */
	intlenN = fitableAH[i].intoutlenN;

	printf("Operation %d: %s\n", (*testPN)++, (fitableAH[i].operationS));
#ifdef TEST
	printf("Input float exponent bits: %d, Mantissa bits: %d, Total byte length: %d\n",
		expbitsN, fracbitsN, totalenN );
	printf("Output integer length: %d\n", intlenN);
#endif

	/* Convert the first operand from a string to a float. */
	strtoflt(fitableAH[i].fltinBP, strlen(fitableAH[i].fltinBP), op1AB,
		 fracbitsN, expbitsN );
#ifdef TEST
	printf("Float as string: %s\n", fitableAH[i].fltinBP);
	foutput("Float in binary: ", op1AB, totalenN );
#endif

	/* Perform the conversion. */
	condcodeB = fltoint(op1AB, fracbitsN, expbitsN, op2AB, intlenN);

#ifdef TEST
	foutput("Result as integer: ", op2AB, intlenN);
#endif
	/* Convert the result back to a string. */
	intostr(op2AB, intlenN, resultAB, 128, 10);
#ifdef TEST
	printf("Result as string: %s\n", resultAB);
	printf("Result should be: %s\n", fitableAH[i].intoutBP);
	printf("Result condition code: 0x%x\n", condcodeB);
	printf("Condition code should be: 0x%x\n", fitableAH[i].ccresultN);
#endif
	if ((strcmp(resultAB, fitableAH[i].intoutBP) == 0) &&
	    (((int) condcodeB) == fitableAH[i].ccresultN))
	{
	    printf(" [(-:} OK {:-)]\n");
	}
	else
	{
	    printf(" [)-:}***** ERROR *****{:-(]\n");
	    errcountN++;
	}
#ifdef MEMTEST
	if ((memparcelN = fmallreport()) != 0)
	{
	    printf("%d memory blocks not free'd\n", memparcelN);
	    errcountN += memparcelN;
	}
#endif
	i++;
    }

    printf("*******************************************************\n");
    printf("** Floating point to floating point conversion tests **\n");
    printf("*******************************************************\n");
    i = 0;
    while (fftableAH[i].operationS != NULL)
    {
	/* Determine the length and type of the input floating point number */
	inexpbitsN = fftableAH[i].fltinexpN;
	infracbitsN = fftableAH[i].fltinfracN;
	intotalenN = fftableAH[i].fltinlenN;

	/* Determine the length and type of the output floating point number */
	outexpbitsN = fftableAH[i].fltoutexpN;
	outfracbitsN = fftableAH[i].fltoutfracN;
	outtotalenN = fftableAH[i].fltoutlenN;

	printf("Operation %d: %s\n", (*testPN)++, (fftableAH[i].operationS));
#ifdef TEST
	printf(" Input float exponent bits: %d, Mantissa bits: %d, Total byte length: %d\n",
		inexpbitsN, infracbitsN, intotalenN );
	printf("Output float exponent bits: %d, Mantissa bits: %d, Total byte length: %d\n",
		outexpbitsN, outfracbitsN, outtotalenN );
#endif

	/* Convert the first operand from a string to a float. */
	strtoflt(fftableAH[i].fltinBP, strlen(fftableAH[i].fltinBP), op1AB,
		 infracbitsN, inexpbitsN );
#ifdef TEST
	printf("Input float as string: %s\n", fftableAH[i].fltinBP);
	foutput("Input float in binary: ", op1AB, intotalenN);
#endif

	/* Perform the conversion. */
	condcodeB = fltoflt(op1AB, infracbitsN, inexpbitsN,
			    op2AB, outfracbitsN, outexpbitsN);

#ifdef TEST
	foutput("Result as float: ", op2AB, outtotalenN);
#endif
	/* Convert the result back to a string. */
	fltostr(op2AB, outfracbitsN, outexpbitsN, resultAB, 128);
#ifdef TEST
	printf("Result as string: %s\n", resultAB);
	printf("Result should be: %s\n", fftableAH[i].fltoutBP);
	printf("Result condition code: 0x%x\n", condcodeB);
	printf("Condition code should be: 0x%x\n", fftableAH[i].ccresultN);
#endif
	if ((strcmp(resultAB, fftableAH[i].fltoutBP) == 0) &&
	    (((int) condcodeB) == fftableAH[i].ccresultN))
	{
	    printf(" [(-:} OK {:-)]\n");
	}
	else
	{
	    printf(" [)-:}***** ERROR *****{:-(]\n");
	    errcountN++;
	}
#ifdef MEMTEST
	if ((memparcelN = fmallreport()) != 0)
	{
	    printf("%d memory blocks not free'd\n", memparcelN);
	    errcountN += memparcelN;
	}
#endif
	i++;
    }

    return(errcountN);
}
