/* DSP_CTRL.C */

#define DSP_CTRL

#include <stdio.h>
#include <conio.h>
#include <process.h>
#include <dos.h>

#include "dsp_ctrl.h"

#define ZPB32								0x00
#define ZPB34								0x01
#define CODYBOARD						0x02
#define DECOMP_ADDR_DSP32		0x500
#define WAVELET_ADDR_DSP32	0x0608
#define DECOMP_ADDR_DSP34		0x0350
#define WAVELET_ADDR_DSP34	0x0478

int _pcr;
unsigned _board_addr, _pcr_addr, _pcrh_addr, _par_addr, _pare_addr;
unsigned _pir_addr, _rbisr_addr, _sioc_addr, _id_reg_addr;

unsigned _wavelet_filt_addr, _decomp_coeff_addr;
unsigned *_trace_scale_addr;
unsigned *_decomp_data2, *_decomp_data4, *_decomp_data6;
unsigned _decomp_datalen;
unsigned _decomp_data2_DSP32[2] = {0x01e8, 0xfffc};
unsigned _decomp_data4_DSP32[2] = {0x01e0, 0xfff4};
unsigned _decomp_data6_DSP32[2] = {0x01d8, 0xffec};
unsigned _decomp_data2_DSP34[4] = {0x01e0, 0x0000, 0xfffc, 0xffff};
unsigned _decomp_data4_DSP34[4] = {0x01d8, 0x0000, 0xfff4, 0xffff};
unsigned _decomp_data6_DSP34[4] = {0x01d0, 0x0000, 0xffec, 0xffff};
unsigned _trace_scale_addrDSP32[8] = {0x0524, 0x0540, 0x055c, 0x0578,
																			0x0594, 0x05b0, 0x05cc, 0x05e8};
unsigned _trace_scale_addrDSP34[8] = {0x0390, 0x03ac, 0x03c8, 0x03e4,
																			0x0400, 0x041c, 0x0438, 0x0454};
float _scale_data[10] = {0.001155796991, 0.002311593982, 0.005778984955,
												 0.01155796991, 0.02311593982, 0.05778984955,
												 0.1155796991, 0.2311593982, 0.5778984955,
												 1.155796991};

char  codefileDSP32[] = "anlyzr32.dsp";
char  codefileDSP32C[] = "anlyz32c.dsp";

void RunZPB32(void);
void RunCODY(void);
void RunZPB34(void);
void HaltZPB32(void);
void HaltCODY(void);
void HaltZPB34(void);
void SetDMAaddrZPB32(unsigned long addr);
void SetDMAaddrZPB34(unsigned long addr);
void DownloadIntZPB34(unsigned addr, unsigned *data, unsigned count);
void DownloadIntZPB32(unsigned addr, unsigned *data, unsigned count);
unsigned long ConvIEEEtoDSP32(float a);
unsigned int ACKpir16bitZPB32 (void);
unsigned int ACKpir16bitZPB34(void);
unsigned WaitZPB32(unsigned count);
unsigned WaitZPB34(unsigned count);
void DownloadDSPexec(char *filename);
void (*DownloadInt)();

void DSPboardConfigure(void)
{
	FILE *cfg;
	size_t count;
	char DSPboardtype, answer;

	/* attempt to open the DSP board configuration file */
	if (!(cfg = fopen("analyzer.cfg", "rb")))
	{
		/* create a new configuration file and fill it */
		puts("\nConfiguration file ANALYZER.CFG not found");
		cfg = fopen("analyzer.cfg", "wb");
		do
		{
			puts("\nEnter type of DSP board to be used");
			printf("0 - ZPB32, 1 - ZPB34, 2 - CODYBOARD: ");
			DSPboardtype = (char) getche();
		} while ((DSPboardtype != '0') && (DSPboardtype != '1') &&
							(DSPboardtype != '2'));

		DSPboardtype -= 0x30;
		do
		{
			puts("\nEnter I/O bus address of the DSP board");
			printf("in hexidecimal (e.g. 0x300 entered as 300) : ");
			scanf("%x", &_board_addr);
			printf("\nIs the I/O bus addr %xH? (Y/N): ", _board_addr);
			answer = getche();
			answer &= 0xdf;
		} while (answer != 'Y');

		putc((int) DSPboardtype, cfg); /* write configuration data to file */
		putw((int) _board_addr, cfg);
	}
	else
	{
		/* read the DSP board type */
		count = fread((void *) &DSPboardtype, sizeof(char), 1, cfg);
		if (!count)
		{
			puts("\nUnexpected End-of-file encountered while");
			puts("reading configuration file ANALYZER.CFG");
			fclose(cfg);
			exit(1);
		}

		/* read the location of the DSP board in the I/O map */
		count = fread((void *) &_board_addr, sizeof(unsigned), 1, cfg);
		if (!count)
		{
			puts("\nUnexpected End-of-file encountered while");
			puts("reading configuration file ANALYZER.CFG");
			fclose(cfg);
			exit(2);
		}
	}

	fclose(cfg); /* done with the configuration file */

	/* set up DSP board register addresses, memory addresses,
		 data arrays, and I/O routines according the the DSP board type */
	switch (DSPboardtype)
	{
		case ZPB32:
			_par_addr = _board_addr;
			_pdr_addr = _board_addr + 0x02;
			_pcr_addr = _board_addr + 0x07;
			_pir_addr = _board_addr + 0x08;
			_pcrh_addr = 0;                   /* non-existent on DSP32 processor */
			_pare_addr = 0;                   /* non-existent on DSP32 processor */

			_sioc_addr = _board_addr + 0x0f;  /* miscellaneous registers */
			_id_reg_addr = _board_addr + 0x0f;
			_rbisr_addr = 0;                  /* non-existent on ZPB32 board */
			_bitmap0 = 0x0678L;
			_bitmap1 = 0x1fdcL;

			_wavelet_filt_addr = WAVELET_ADDR_DSP32;
			_trace_scale_addr = _trace_scale_addrDSP32;
			_decomp_coeff_addr = DECOMP_ADDR_DSP32;
			_decomp_data2 = _decomp_data2_DSP32;
			_decomp_data4 = _decomp_data4_DSP32;
			_decomp_data6 = _decomp_data6_DSP32;
			_decomp_datalen = 2;

			SetDMAaddr = SetDMAaddrZPB32;
			DownloadInt = DownloadIntZPB32;
			RunDSP = RunZPB32;
			HaltDSP = HaltZPB32;
			ACKpir16bit = ACKpir16bitZPB32;
			WaitDSP = WaitZPB32;
			DownloadDSPexec(codefileDSP32);
			break;

		case ZPB34:
			_par_addr = _board_addr;          /* bank 0 */
			_pdr_addr = _board_addr + 0x04;

			_pcr_addr = _board_addr + 0x06;   /* bank 1 */

			_pir_addr = _board_addr;          /* bank 2 */
			_pcrh_addr = _board_addr + 0x04;
			_pare_addr = _board_addr + 0x06;

			_rbisr_addr = _board_addr + 0x0b; /* miscellaneous registers */
			_sioc_addr = _board_addr + 0x0e;
			_id_reg_addr = _board_addr + 0x0f;
			_bitmap0 = 0x0528L;
			_bitmap1 = 0x1e8cL;

			_wavelet_filt_addr = WAVELET_ADDR_DSP34;
			_trace_scale_addr = _trace_scale_addrDSP34;
			_decomp_coeff_addr = DECOMP_ADDR_DSP34;
			_decomp_data2 = _decomp_data2_DSP34;
			_decomp_data4 = _decomp_data4_DSP34;
			_decomp_data6 = _decomp_data6_DSP34;
			_decomp_datalen = 4;

			SetDMAaddr = SetDMAaddrZPB34;
			DownloadInt = DownloadIntZPB34;
			RunDSP = RunZPB34;
			HaltDSP = HaltZPB34;
			ACKpir16bit = ACKpir16bitZPB34;
			WaitDSP = WaitZPB34;
			DownloadDSPexec(codefileDSP32C);
			break;

		case CODYBOARD:
			_par_addr = _board_addr;
			_pdr_addr = _board_addr + 0x02;
			_pcr_addr = _board_addr + 0x07;
			_pir_addr = _board_addr + 0x08;
			_pcrh_addr = 0;                   /* non-existent on DSP32 processor */
			_pare_addr = 0;                   /* non-existent on DSP32 processor */

			_sioc_addr = 0;                   /* non-existent on CODYBOARD */
			_id_reg_addr = _board_addr + 0x0f;
			_rbisr_addr = 0;                  /* non-existent on ZPB32 board */
			_bitmap0 = 0x0678L;
			_bitmap1 = 0x1fdcL;

			_wavelet_filt_addr = WAVELET_ADDR_DSP32;
			_trace_scale_addr = _trace_scale_addrDSP32;
			_decomp_coeff_addr = DECOMP_ADDR_DSP32;
			_decomp_data2 = _decomp_data2_DSP32;
			_decomp_data4 = _decomp_data4_DSP32;
			_decomp_data6 = _decomp_data6_DSP32;
			_decomp_datalen = 2;

			SetDMAaddr = SetDMAaddrZPB32;
			DownloadInt = DownloadIntZPB32;
			RunDSP = RunCODY;
			HaltDSP = HaltCODY;
			ACKpir16bit = ACKpir16bitZPB32;
			WaitDSP = WaitZPB32;
			DownloadDSPexec(codefileDSP32);
			break;

		default:
			puts("\nUndefined DSP board type in configuration file ANALYZER.CFG");
			exit(3);
		}
}


void RunZPB32(void)
{
	outportb(_sioc_addr, 0x05); /* set up serial I/O control for codec */
	outportb(_pcr_addr, 0x00); /* halt DSP32 */
	outportb(_pcr_addr, 0x01); /* reset DSP32 */
	outportb(_pcr_addr, 0x1f); /* run DSP32 with autoincrement DMA and ... */
}                            /* 16-bit PIR interrupt vector enabled */


void RunCODY(void)
{
	outportb(_pcr_addr, 0x00); /* halt DSP32 */
	__emit__(0x90);
	outportb(_pcr_addr, 0x01); /* reset DSP32 */
	__emit__(0x90);
	outportb(_pcr_addr, 0x1f); /* run DSP32 with autoincrement DMA and ... */
}                            /* 16-bit PIR interrupt vector enabled */


void RunZPB34(void)
{
	outportb(_sioc_addr, 0x05); /* set up serial I/O control for codec */
	outportb(_rbisr_addr, 1);
	outport(_pcr_addr, 0x00); /* halt DSP32C */
	outport(_pcr_addr, 0x01); /* reset DSP32C */
	outport(_pcr_addr, 0x07); /* set PCR register address map to PCR[1] = 1, */
	outportb(_rbisr_addr, 2);
	outport(_pcrh_addr, 0x02); /* and PCR[9] = 1 */
	outportb(_rbisr_addr, 1);
	outport(_pcr_addr, 0x021f); /* run DSP32C with autoincrement DMA and ... */
}                            /* 16-bit PIR interrupt vector enabled */


void HaltZPB32(void)
{
	outportb(_pcr_addr, 0x18); /* halt DSP and set up for post-increment DMA */
	outportb(_sioc_addr, 0x33); /* deactivate the serial I/O to the codec */
}


void HaltCODY(void)
{
	outportb(_pcr_addr, 0x18); /* halt DSP and set up for post-increment DMA */
}


void HaltZPB34(void)
{
	outportb(_rbisr_addr, 1);
	outport(_pcr_addr, 0x1e); /* halt DSP and set up for post-increment DMA */
	outportb(_rbisr_addr, 2);
	outport(_pcrh_addr, 0x02);
	outportb(_rbisr_addr, 0);
	outportb(_sioc_addr, 0x33);  /* deactivate the serial I/O to the codec */
}


void SetDMAaddrZPB32(unsigned long addr)
{
	outport(_par_addr, (unsigned) addr);
}


void SetDMAaddrZPB34(unsigned long addr)
{
	unsigned *addr_parts = (unsigned *) &addr;

	outportb(_rbisr_addr, 2);
	outport(_pare_addr, addr_parts[1]); /* set high word of address */
	outportb(_rbisr_addr, 0);
	outport(_par_addr, addr_parts[0]); /* set low word of address */
}


void DownloadIntZPB32(unsigned addr, unsigned *data, unsigned count)
{
	unsigned int i;

	_pcr = inportb(_pcr_addr); /* save current PCR value */
	__emit__(0x90);
	outportb(_pcr_addr, (unsigned char) _pcr | 0x18); /* post-increment DMA */
	__emit__(0x90);
	outport(_par_addr, addr);

	for (i = 0; i < count; i++)   /* download 16-bit integer */
		outport(_pdr_addr, data[i]);

	__emit__(0x90);
	outportb(_pcr_addr, (unsigned char) _pcr); /* restore original PCR value */
}


void DownloadIntZPB34(unsigned addr, unsigned *data, unsigned count)
{
	unsigned int i;

	outportb(_rbisr_addr, 1);
	_pcr = inport(_pcr_addr); /* save current PCR value */
	outport(_pcr_addr, (_pcr | 0x0018) & 0xfeff); /* post-increment DMA */
	outportb(_rbisr_addr, 2);
	outport(_pare_addr, 0x0000);
	outportb(_rbisr_addr, 0);
	outport(_par_addr, addr);

	for (i = 0; i < count; i++)   /* download 16-bit integer */
		outport(_pdr_addr, data[i]);

	outportb(_rbisr_addr, 1);
	outport(_pcr_addr, _pcr); /* restore original PCR value */
}


/* Convert an IEEE 754 format number to DSP32 floating point format */
unsigned long ConvIEEEtoDSP32(float a)
{
	unsigned long sn, ex, fr;
	union {
					float b;
					unsigned long i;
				} x;

	x.b = a;
	sn = x.i & 0x80000000L;        /* shift and mask off sign bit */
	ex = (x.i >> 23) & 0xffL;      /* shift and mask off bits of exponent */
	fr = (x.i << 8) & 0x7fffffffL; /* shift and mask off bits of mantissa */
	if ((!sn && ex) || (sn && fr)) /* twiddle exponent */
		ex += 1;

	if (sn)
		fr = -fr;  /* two's complement of the mantissa */

	return(sn | fr | ex);
}


void DownloadFiltCoeffs(float *filters)
{
	int i;
	unsigned long dsp32float[12];
	unsigned *dsp_data = (unsigned *) dsp32float;

	for (i = 0; i < 12; i++) /* download each float as two integers */
		dsp32float[i] = ConvIEEEtoDSP32(filters[i]);

	DownloadInt(_wavelet_filt_addr, dsp_data, 24);
}


void DownloadDecompCoeffs(int filt_len)
{
	/* select coefficient set according to length of filters */
	switch(filt_len)
	{
		case FILTER_LENGTH_2:
			DownloadInt(_decomp_coeff_addr, _decomp_data2, _decomp_datalen);
			break;
		case FILTER_LENGTH_4:
			DownloadInt(_decomp_coeff_addr, _decomp_data4, _decomp_datalen);
			break;
		case FILTER_LENGTH_6:
			DownloadInt(_decomp_coeff_addr, _decomp_data6, _decomp_datalen);
	}
}


void DownloadTraceScale(int trace_num, int scale_num)
{
	unsigned long dsp32float;
	unsigned *dsp_data = (unsigned *) &dsp32float;

	dsp32float = ConvIEEEtoDSP32(_scale_data[scale_num]);
	DownloadInt(_trace_scale_addr[trace_num], dsp_data, 2);
}


unsigned ACKpir16bitZPB32 (void)
{
	return(inport(_pir_addr));
}


unsigned ACKpir16bitZPB34(void)
{
	outportb(_rbisr_addr, 2);
	return (inport(_pir_addr));
}


unsigned WaitZPB32(unsigned count)
{
	do
	{
		count-- ;
	} while (!(inportb(_pcr_addr) & 0x40) && (count > 0));

	return (count); /* count reflects time left before timeout */
}


unsigned WaitZPB34(unsigned count)
{
	outportb(_rbisr_addr, 1);

	do
	{
		count--;
	} while (!(inport(_pcr_addr) & 0x0040) && (count > 0));

	return (count); /* count reflects time left before timeout */
}


void DownloadDSPexec(char *file_name)
{
	FILE *fp;
	int file_header[10];
	int i, j;
	unsigned raw_int[2];
	unsigned long sect_header[10];
	unsigned long sect_header_table_ptr;

	if ((fp = fopen(file_name, "rb")) == NULL)
	{
		fclose(fp);
		printf("\nDSP COFF file %s not found\n", file_name);
		exit(4);
	}

	/* Read the file header. */

	if (fread(file_header, sizeof(file_header), 1, fp) == 0)
	{
		fclose(fp);
		puts("\nUnexpected End-of-file encountered");
		printf("while reading DSP COFF file %s\n", file_name);
		exit(5);
	}

	/* Seek past the optional header. */

	fseek(fp, (long) file_header[8], 1);
	sect_header_table_ptr = ftell(fp);

	HaltDSP(); /* make sure DSP is halted and set up for DMA operation */

	for (i = 0; i < file_header[1]; i++)
	{
		fseek(fp, sect_header_table_ptr, 0);

		if (fread(sect_header, sizeof(sect_header), 1, fp) == 0)
		{
			fclose(fp);
			puts("\nUnexpected End-of-file encountered");
			printf("while reading DSP COFF file %s\n", file_name);
			exit(6);
		}

		sect_header_table_ptr = ftell(fp);
		fseek(fp, sect_header[5], 0);

		SetDMAaddr(sect_header[2]);

		for (j = 0; j < sect_header[4]; j += 4)
		{
			if (fread(raw_int, sizeof(raw_int), 1, fp) == 0)
			{
				fclose(fp);
				puts("\nUnexpected End-of-file encountered");
				printf("while reading DSP COFF file %s\n", file_name);
				exit(7);
			}

			outport(_pdr_addr, raw_int[0]);
			__emit__(0x90);
			outport(_pdr_addr, raw_int[1]);
		}
	}

	fclose(fp);
}
