/* -*- Mode: C -*- */
/* common.h - common include file - all other include files (and
 *	      programs) include this file
 * Created by Robert Heller on Fri Dec  6 19:42:59 1991
 *
 * ------------------------------------------------------------------
 * Home Libarian by Deepwoods Software
 * Common Header Files
 * ------------------------------------------------------------------
 * Modification History:
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 * 
 * 
 * Copyright (c) 1991,1992 by Robert heller (D/B/A Deepwoods Software)
 *        All Rights Reserved
 * 
 */

#ifndef _COMMON_
#define _COMMON_
#include <stdlib.h>		// standard C library functions
/*
 * O/S dependent stuff:
 *    1) we need file modes (passed as args to open and creat (create under
 *	 OSK)
 *    2) we need terminal mode hacking
 *    3) we need signal and trap interception stuff
 */
#ifdef OSK
#include <modes.h>		// file modes
// generic symbols (hide O/S dependent names)
#define ReadMode S_IREAD
#define ReadWriteMode S_IREAD|S_IWRITE
#define ReadFlags S_IREAD
#define ReadWriteFlags S_IREAD|S_IWRITE
// signal hackery
#include "/dd/defs/signal.h"	// the G++ signal.h is a UNIX style signal.h
// terminal modes
#include <sgstat.h>		// Get/Set tty modes (OSK)
// special library functions
typedef int (*icept)(int);
extern "MWC" {
	int _ss_opt(int,char*);
	int _gs_opt(int,char*);
	int _gs_rdy(int);
	int intercept(icept);
	char *strerror(int);
};
#else
#include <fcntl.h>
#include <sys/stat.h> 
// generic symbols (hide O/S dependent names)
#define ReadMode S_IREAD
#define ReadWriteMode S_IREAD|S_IWRITE
#define ReadFlags O_RDONLY|O_BINARY
#define ReadWriteFlags O_RDWR|O_BINARY
#endif
// Global constants:
// KeySize is how long key strings can be.  This includes the nul byte
const KeySize = 36;
// Tree order
const Order   =  10;
// Key type
typedef char Key[KeySize];
// with an order 10 tree and a 32 byte key string + standard overhead
// (pointers, size info, data record pointers), we will get a tree node
// (page) of just under 1024 bytes.  With a small filler, pages will
// be 1024 bytes in size.  This is a reasonable size for most O/Ss.
const SectorSize = 1024;

// Global enumerated types
// Your basic true/false type:
typedef int Boolean;
const Boolean false = 0;
const Boolean true  = 1;
// File directions:
enum Direction { in, out, inout };
// File open status
enum OpenStatus { failure, openold, opennew };
// LastSearchType.  Used by SearchXX and SearchXXAgain.
enum LSType { none, id, title, author, subj };
// open modes
enum OpenMode { ReadOnly = 0x001, ReadWrite = 0x003, ModeMask = 0x0FF,
		Create = 0x100 };
// types of cards
enum CardType { Book = 'B', Magazine = 'M', CD = 'D', AudioCassette = 'C',
		Album = 'A', LaserDisk = 'L', VHSVideo = 'V', BetaVideo = 'S',
		EightMM = '8', EightTrack = 'E', DAT = '4', Other = 'O' };
// Terminal command keys
enum  CommandKeys {
	escCmd	= -1,		// ESC key
	upCmd   = -2,		// Up Arrow (^P)
	downCmd = -3,		// Down Arrow (^N)
	leftCmd = -4,		// Left Arrow (^B)
	rightCmd = -5,		// Right Arrow (^F)
	backCmd = -6,		// BackSpace (^H,RUBOUT)
	deleCmd = -7		// DeleteLine (^U)
};
// Terminal drawing modes
enum Mode {
	defPen     = 0x00000,
	graphPen   = 0x00100,
	revsPen    = 0x00200,
	charMask   = 0x000FF,
	modeMask   = 0x0FF00
};
// Types of errors
enum ErrKind {
	memErr,  termErr, sysErr
};

// Basic structure classes

// A disk record contains a size and a file offset.
struct DiskRecord {
	long int record_size;
	long record_address;
	DiskRecord (long int size = 0, long addr = 0L) 
		   {record_size = size; record_address = addr;}
};

// A disk page is simular, except the size is fixed.
// there are some special operator and constructor methods
// used to make things convenient, since these are used like
// pointers
struct DiskPage {
	const long int record_size = SectorSize;
	long  record_address;
	DiskPage (DiskPage& a) {record_address = a.record_address;}
	DiskPage (long addr = 0L) {record_address = addr;}
	inline friend Boolean operator == (DiskPage& a,DiskPage& b)
				{return(a.record_address == b.record_address);}
	inline friend Boolean operator != (DiskPage& a,DiskPage& b)
				{return(a.record_address != b.record_address);}
	inline DiskPage& operator = (DiskPage& a)
				{record_address = a.record_address;return *this;}

};

// A data item - has a key, a data record "pointer", and a child page
// "pointer".
struct Item {
	Key key;
	DiskRecord data;
	DiskPage right;
};

// A page is a tree node.
// It has a size (number of used items), pointers to a child and parent,
// a vector of items, and some filler (used to make sure the page is
// exactly 1024 bytes in size).
struct Page {
	long int size;
	DiskPage left;
	DiskPage parent;
	long int parentidx;
	Item items[Order*2];
	char filler[SectorSize-(sizeof(long int) +
				(2*sizeof(DiskPage)) +
				sizeof(long int) +
				(sizeof(Item)*Order*2))];
};

// Page Table Entry
// Has a dirty flag, a pointer to the disk page, and a pointer to
// the core copy of the disk page.
struct PTEntry {
	Boolean isdirty;
	DiskPage diskpage;
	Page     *corepage;
};

// HomeBlock - this is a 1024 byte record at the start of the file.
// it contains a 8 byte magic header, pointers to the roots of the
// four trees and a vector of available pre-allocated pages.
const int MaxNumFree = (SectorSize - ( 8 + (4 * sizeof(DiskPage)) + 
					   sizeof(long int))) /
		       sizeof(DiskPage);

struct HomeBlock {
	const char  * const Magic = "LIBRV000";
	char magic[8];
	DiskPage IdRoot, TitleRoot, AuthorRoot, SubjRoot;
	long int numfree;
	DiskPage freepages[MaxNumFree];
};

// Core resident record - it has a size and a chunk of memory (the
// record itself).
// Lots of fun.  I wish C++ had a garbage collector...
struct Record {
	long int size;
	char *buffer;
	     Record () {size = 0; buffer = 0;}		// empty buffer
	     Record (long int size) {Record::size = size; // preallocated buffer
				     buffer = new char[size];}
	     Record (Record& rec)		// record from another record
		{
			size = rec.size;
			if (size > 0) {
				buffer = new char[size];
				memcpy(buffer,rec.buffer,size);
			}
		}
	    ~Record () {if (size > 0) delete buffer;}	// Oh for a GC!
	Record& operator = (Record& rec)		// copy a record
		{
			if (size < rec.size) {
				delete buffer;
				buffer = new char[rec.size];
			} else if (rec.size <= 0 && size > 0)
			    delete buffer;
			size = rec.size;
			if (rec.size > 0)
			    memcpy(buffer,rec.buffer,rec.size);
			return *this;
		}
	void NewBuffer(long int size)		// buffer re-allocator
		{
			if (Record::size > 0) delete buffer;
			Record::size = size;
			if (size > 0) buffer = new char[size];
		}			
};

// In core item.  Code outside of vBTree.cc never does actual disk I/O.
// Instead it passes in-core records and the insertion, searching, and
// traversal code reads and writes the data from and to the disk file.
struct CoreItem {
	Key key;
	Record data;
};

// Function pointer types
typedef void (*TravFunc)(CoreItem*,int);
typedef void (*ErrFun)(int,const char*);
typedef void (*InterruptFun)(void);

#endif
