/*     Regular expression library test code
       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
       vv. 1.01          M.I.Barlow 3-10-97
*/
/*{{{  includes */

#include <stdio.h>
#ifdef __TURBOC__
  #include <alloc.h>
#else
  #include <malloc.h>
#endif
#include "rxplib.h"

/*}}}   */
/*{{{  defines */

#define TRUE  1
#define FALSE 0

/*}}}   */
/*{{{  count_heap */

long
count_heap ( void )
{
  #ifdef __TURBOC__
    long total = 0L;

    if (heapcheck() != _HEAPOK)
    {
	printf("SERIOUS: heap tested as corrupt.\n");
	return (0L);
    }
    else
    {
	struct heapinfo hi;

	hi.ptr = NULL;
	while (heapwalk(&hi) != _HEAPEND)
	{
	    if (hi.in_use != 0)
		total += (long)(hi.size);
	}
	return (total);
    }
  #else
    return (0L);
  #endif
}

/*}}}   */
/*{{{  main */

void
main (void)
{
    /*{{{  declare */
    
    char txt[256], rep[256], pat[256] = "dummy";
    RxpError err;
    RxpPatn root;
    char *eloc;
    
    /*}}}   */

    printf("\nRXPLIB TESTING: enter an empty regexp to terminate...\n");
    while (*pat)
    {
	/*{{{  do a test */
	
	long before, after;
	
	/*{{{  get pattern */
	printf("\nregexpr ? > ");
	if (!gets(pat))
	    *pat = '\0';
	printf("%s\n",pat);
	/*}}}   */
	/*{{{  get replacement */
	printf("repmask ? > ");
	if (!gets(rep))
	    *rep = '\0';
	printf("%s\n",rep);
	/*}}}   */
	/*{{{  get text to scan */
	printf("text ?    > ");
	if (!gets(txt))
	    *txt = '\0';
	printf("%s\n",txt);
	/*}}}   */
	
	before = count_heap();
	
	err = rxp_create_pattern(pat,&root,&eloc);
	
	if (err == RxpOK)
	{
	    /*{{{  do the search & replace */
	    
	    RxpMatch match;
	    
	    err = rxp_find_match(root,txt,TRUE,&match);
	    
	    if (err == RxpOK)
	    {
		/*{{{  declare */
		char *cp = txt;
		/*}}}   */
	    
		/*{{{  display match status */
		printf("%s\n",txt);
		while (cp < match.fptr[0])
		{
		    printf(" ");
		    cp++;
		}
		printf("^");
		cp++;
		while (cp < (match.bptr[0] - 1))
		{
		    printf("-");
		    cp++;
		}
		if (match.fptr[0] >= (match.bptr[0] - 1))
		    printf("\nFound match.\n");
		else
		    printf("^\nFound match.\n");
		/*}}}   */
		if (*rep)
		{
		    /*{{{  do the replacement */
		    
		    char *repl;
		    
		    err = rxp_create_replacement(rep,match,&repl);
		    
		    if (err == RxpAllocFail)
			printf("SERIOUS: Memory allocation failure creating replacement.\n");
		    else if (err == RxpNullArg)
			printf("SERIOUS: Null pointer passed to function.\n");
		    else if (err == RxpBadEscape)
			printf("ERROR: Isolated trailing \\ in replacement mask.\n");
		    else if (err == RxpBadRange) /* only if rxp_extensions_flag != 0 */
			printf("ERROR: Malformed slice operation in replacement mask.\n");
		    else if (err == RxpBadHex) /* only if rxp_C_escapes_flag != 0 */
			printf("ERROR: Malformed hexadecimal escape in replacement mask.\n");
		    else
		    {
			printf("replacement: ===>%s<===\n",repl);
			free(repl);
		    }
		    
		    /*}}}   */
		}
	    }
	    else if (err == RxpFail)
		printf("No match found.\n");
	    else if (err == RxpNullArg)
		printf("SERIOUS: Null pointer passed to function.\n");
	    else
		printf("SERIOUS: Suspected heap corruption.\n");
	    
	    /*}}}   */
	}
	else if (err == RxpCorrupt)
	    printf("SERIOUS: Suspected heap corruption.\n");
	else if (err == RxpAllocFail)
	    printf("SERIOUS: Memory allocation failure.\n");
	else if (err == RxpNullArg)
	    printf("SERIOUS: Null pointer passed to function.\n");
	else if (err == RxpFail)
	    printf("FATAL: Empty pattern; terminating.\n");
	else
	{
	    /*{{{  explain the problem */
	    
	    char *cp = pat;
	    
	    printf("=> %s\n |-",pat);
	    
	    while (cp < eloc)
	    {
		cp++;
		printf("-");
	    }
	    printf("^\nERROR: ");
	    switch (err)
	    {
		case RxpBadSet:
		    printf("malformed set [] seen\n");
		    break;
		case RxpBadClosure:
		    printf("invalid closure (nothing to apply it to)\n");
		    break;
		case RxpBadRange:
		    printf("malformed explicit closure range \\{n,m\\}\n");
		    break;
		case RxpBigRange:
		    printf("in \\{n,m\\} neither n nor m may be be > 255\n");
		    break;
		case RxpBadGroup:
		    printf("no terminating ) seen after (\n");
		    break;
		case RxpBadEscape:
		    printf("solitary \\ seen at end of pattern\n");
		    break;
		case RxpBadHex: /* only if rxp_C_escapes_flag != 0 */
		    printf("saw \\x<hh> where <h> is not a hex digit\n");
		    break;
		case RxpBadCapture:
		    printf("\\0 is not a valid capture specifier\n");
		    break;
	    }
	    
	    /*}}}   */
	}
	
	if (rxp_destroy_pattern(&root) == RxpCorrupt)
	    printf("SERIOUS: Corrupt pattern!\n");
	
	after = count_heap();
	if (after != before)
	    printf("SERIOUS: memory leakage (%ld bytes).\n",after - before);
	
	/*}}}   */
    }
}

/*}}}   */
