/*     Regular expression library test code
       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
       vv. 1.01          M.I.Barlow 3-10-97
 */
/*{{{  includes */

#include <stdio.h>
#ifdef __TURBOC__
  #include <alloc.h>
#else
  #include <malloc.h>
#endif
#include "rxplib.h"

/*}}}   */
/*{{{  defines */

#define TRUE  1
#define FALSE 0

/*}}}   */
/*{{{  count_heap */

long
count_heap ( void )
{
  #ifdef __TURBOC__
    long total = 0L;

    if (heapcheck() != _HEAPOK)
    {
	printf("SERIOUS: heap tested as corrupt.\n");
	return (0L);
    }
    else
    {
	struct heapinfo hi;

	hi.ptr = NULL;
	while (heapwalk(&hi) != _HEAPEND)
	{
	    if (hi.in_use != 0)
		total += (long)(hi.size);
	}
	return (total);
    }
  #else
    return (0L);
  #endif
}

/*}}}   */
/*{{{  main */

void
main (int argc, char *argv[])
{
    if (argc != 3)
    {
	/*{{{  show help */
	
	fprintf(stderr,"\nUsage: gsub <regexp> <replacement>\n\n");
	fprintf(stderr,"       Filter which reads from <stdin>, finds all text matching <regexp>,\n");
	fprintf(stderr,"       replaces it with <replacement> and writes the result to <stdout>.\n");
	
	/*}}}   */
    }
    else
    {
	/*{{{  declare */
	
	char line[1024];
	RxpError err;
	RxpPatn root;
	char *eloc;
	long before, after;
	
	/*}}}   */

	before = count_heap();
	err = rxp_create_pattern(argv[1],&root,&eloc);
	if (err == RxpOK)
	{
	    /*{{{  do the search & replace */
	    
	    char line[1024];
	    RxpMatch match;
	    int going = TRUE;
	    
	    while (going)
	    {
		if (!gets(line))
		    going = FALSE;
		else
		{
		    char *cp = line;
		    char *rp;
		    char tmp;
		    int at_start = TRUE;
	    
		    err = RxpOK;
		    while (err == RxpOK)
		    {
			err = rxp_find_match(root,cp,at_start,&match);
	    
			switch (err)
			{
			    case RxpOK:
				/*{{{  output the unaffected part */
				
				tmp = *(match.fptr[0]);
				*(match.fptr[0]) = '\0';
				printf("%s",cp);
				*(match.fptr[0]) = tmp;
				
				/*}}}   */
				/*{{{  do the replacement */
				
				err = rxp_create_replacement(argv[2],match,&rp);
				
				switch (err)
				{
				    case RxpOK:
					printf("%s",rp);
					free(rp);
					break;
				    case RxpAllocFail:
					fprintf(stderr,"SERIOUS: Insufficient memory to do replacement.\n");
					going = FALSE;
					break;
				    case RxpNullArg:
					fprintf(stderr,"SERIOUS: Null pointer passed to function.\n");
					going = FALSE;
					break;
				    case RxpBadEscape:
					fprintf(stderr,"SERIOUS: Isolated trailing \\ in replacement mask.\n");
					going = FALSE;
					break;
				    case RxpBadRange:
					fprintf(stderr,"SERIOUS: Malformed slice in replacement.\n");
					going = FALSE;
					break;
				    case RxpBadHex:
					fprintf(stderr,"SERIOUS: Bad hex escape in replacement.\n");
					going = FALSE;
					break;
				}
				
				/*}}}   */
				/*{{{  prepare to scan the remainder */
				
				cp = match.bptr[0];
				
				/*}}}   */
				break;
			    case RxpFail:
				/*{{{  output the remainder of the line */
				
				printf("%s\n",cp);
				
				/*}}}   */
				break;
			    case RxpNullArg:
				fprintf(stderr,"SERIOUS: Null pointer passed to function.\n");
				going = FALSE;
				break;
			    case RxpCorrupt:
				fprintf(stderr,"SERIOUS: Suspected heap corruption.\n");
				going = FALSE;
				break;
			}
			at_start = FALSE;
		    }
		}
	    }
	    
	    /*}}}   */
	}
	else if (err == RxpCorrupt)
	    fprintf(stderr,"SERIOUS: Suspected heap corruption.\n");
	else if (err == RxpAllocFail)
	    fprintf(stderr,"SERIOUS: Memory allocation failure.\n");
	else if (err == RxpNullArg)
	    fprintf(stderr,"SERIOUS: Null pointer passed to function.\n");
	else if (err == RxpFail)
	    fprintf(stderr,"FATAL: Empty pattern; terminating.\n");
	else
	{
	    /*{{{  explain the problem */
	    
	    char *cp = argv[1];
	    
	    fprintf(stderr,"=> %s\n |-",argv[1]);
	    
	    while (cp < eloc)
	    {
		cp++;
		fprintf(stderr,"-");
	    }
	    fprintf(stderr,"^\nERROR: ");
	    switch (err)
	    {
		case RxpBadSet:
		    fprintf(stderr,"malformed set [] seen\n");
		    break;
		case RxpBadClosure:
		    fprintf(stderr,"invalid closure (nothing to apply it to)\n");
		    break;
		case RxpBadRange:
		    fprintf(stderr,"malformed explicit closure range \\{n,m\\}\n");
		    break;
		case RxpBigRange:
		    fprintf(stderr,"in \\{n,m\\} neither n nor m may be be > 255\n");
		    break;
		case RxpBadGroup:
		    fprintf(stderr,"no terminating ) seen after (\n");
		    break;
		case RxpBadEscape:
		    fprintf(stderr,"solitary \\ seen at end of pattern\n");
		    break;
		case RxpBadHex: /* only if rxp_C_escapes_flag != 0 */
		    fprintf(stderr,"saw \\x<hh> where <h> is not a hex digit\n");
		    break;
		case RxpBadCapture:
		    fprintf(stderr,"\\0 is not a valid capture specifier\n");
		    break;
	    }
	    
	    /*}}}   */
	}
	if (rxp_destroy_pattern(&root) == RxpCorrupt)
	    fprintf(stderr,"SERIOUS: Corrupt pattern!\n");

	after = count_heap();
	if (after != before)
	    fprintf(stderr,"SERIOUS: memory leakage (%ld bytes).\n",after - before);
    }
}

/*}}}   */
