
///////////////////////////////////////////////////////////////////
//                                                               //
//  ZMIN.C  --  program showing bare essentials for calling      //
//              zmodem transfer code.                            //
//  (c) Mike Dumdei, 6 Holly Lane, Texarkana TX 75503  USA       //
//                                                               //
//    bcc zmin.c zcmplr.c xyzgen.c zmlib_s.lib comm_s.lib        //
//    cl zmin.c zcmplr.c xyzgen.c /link zmlib_s comm_s           //
//                                                               //
//    Expects the following data on command line:                //
//     zmin com# { -B##### -L##### -H -E } -R                    //
//     zmin com# { -B##### -L##### -H } -S files.lst             //
//                                                               //
//   Items in braces are optional.  B = baud rate, L = locked    //
//   baud rate, H = use RTS/CTS handshake.  E = resume transfer  //
//   (default in this program is to overwrite -- see below).     //
//                                                               //
///////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "comm.h"
#include "extra.h"
#include "zmdos.h"

void main(int argc, char *argv[]);
int ProcCmdLine(int argc, char *argv[]);
void usage(void);
void ZMsg(int type, ...);

//
//  Following 2 longs REQUIRED for linker
//
long PrevHunds;
long PrevSecs;

ASYNC port;                 /* declare an ASYNC port structure */
int combase = 0x3f8;
int irq = IRQ4;
int vctr = VCTR4;           /* default port is COM1 */
char params[12] = "";       /* port parameters */
char lockedbaud[12] = "";   /* locked baud parameter */
int msrflow = 0;            /* hardware flow control */
char fnames[256] = "";      /* list of files to send if sending */

/*/////////////////////////////////////////////////////////
//                                                       //
//      Main                                             //
//                                                       //
//:mai////////////////////////////////////////////////// */
void main(int argc, char *argv[])
{
    int i, rval, FIFOretry = 0, openmask = 0;

    if (ProcCmdLine(argc, argv) == 0)   // process command line
        usage();
    tickhookset(1);                     // enable MCOMM timer functions
     //
     //  open the port -- more code than necessary here but this code
     //  overcomes serial port lockup problems on some PCI motherboards
     //
    AllocRingBuffer(&port, 2048, 4096, 0);
    while (1)
    {
        i = async_open(&port, combase, irq, vctr|openmask, params);
        if (i != 0)
        {
            if (i == 512 && (port.Stat3 & B_FIFO) && FIFOretry == 0
             && !(openmask & 0x4000))
            {
                openmask |= 0x4000;
                FIFOretry = 1;      /* FIFO retry code fixes some PCI */
                continue;           /* serial port lockup problems    */
            }
            tickhookset(0);
            printf("\nSerial port open error, Error code = %d\n\a", i);
            exit(i - 20); /* -20 so exit code don't clash with zm result */
        }
        if (FIFOretry == 1)
        {
            openmask &= ~0x4000;
            FIFOretry = 2;          /* More FIFO retry code.  Opening */
            tdelay(1);              /*  with FIFOs off, closing, and  */
            async_close(&port);     /*  reopening with FIFOs on seems */
            continue;               /*  to unlock the port.           */
        }
        if (ConnectBaud == 0L)
        {
            ConnectBaud = atol(port.BPDSstr);
            strcpy(params, port.BPDSstr);
        }
        break;
    }
    async_msrflow(&port, msrflow);      // if -H on cmd line, enable RTS h/s

     //
     //  Send/receive zmodem.  Requirements are: 1) async port opened,
     //   2) timer tickhook enabled, 3) flow control enabled if using
     //   hardware handshake, 4) ZMsg handler defined which can be an
     //   empty function (see below)
     //
    if (TFlag.F.Receiving == 1)
        rval = ZmodemRecv(&port);
    else
        rval = ZmodemSend(&port, fnames);

    async_close(&port);     // close the comm port (required)
    tickhookset(0);         // disable MCOMM timer hook (required)
    exit(rval);
}

/*/////////////////////////////////////////////////////////
//  ZMsg -- zmodem message handler                       //
//:zms///////////////////////////////////////////////:z: */
void ZMsg(int type, ...)
{                           // ignore all progress messages
}

/*/////////////////////////////////////////////////////////
//  ProcCmdLine -- process commnad line args             //
//:pro////////////////////////////////////////////////// */
int ProcCmdLine(int argc, char *argv[])
{
                        /*   COM1   COM2   COM3   COM4  */
    static int bases[4] = { 0x3f8, 0x2f8, 0x3e8, 0x2e8 };
    static int irqs[4]  = {  IRQ4,  IRQ3,  IRQ4,  IRQ3 };
    static int vctrs[4] = { VCTR4, VCTR3, VCTR4, VCTR3 };
    int parg = 0, i, j;
    char *p1;

    TFlag.F.ExistOpts = 3;      // overwrite file if it already exist
    for (i = 1; i < argc; i++)
    {
        p1 = strupr(argv[i]);
        if (*p1 == '-' || *p1 == '/')
            ++p1;
        if (parg == 0)
        {
            if (strncmp(argv[i], "COM", 3) == 0)
            {
                j = atoi(isdigit(*(p1 += 3)) ? p1 : argv[++i]);
                if (j < 1 && j > 4)
                    usage();
                --j, parg = 1;
                combase = bases[j], irq = irqs[j], vctr = vctrs[j];
                continue;
            }
        }
        switch (*p1)
        {
          case 'B':             /* connect baud rate */
            strcpy(params, (isdigit(*++p1)) ? p1 : argv[++i]);
            break;
          case 'L':             /* locked baud rate */
            strcpy(lockedbaud, (isdigit(*++p1)) ? p1 : argv[++i]);
            break;
          case 'H':             /* enable hardware handshake */
            msrflow = B_CTS | B_RTS;
            break;
          case 'E':             /* zmodem resume if file exists */
            TFlag.F.ExistOpts = 1;
            break;
          case 'R':             /* receiving */
            TFlag.F.Receiving = 1;
            return 1;
          case 'S':             /* sending -- file name(s) follow */
            TFlag.F.Receiving = 0;
            if (*++p1 > ' ')
                argv[i--] = p1;
            while (++i < argc)
            {
                strcat(fnames, " ");
                strcat(fnames, argv[i]);
            }
            return 1;
          default:
            return 0;
        }
    }
    return 0;
}

/*///////////////////////////////////////////////
//  Usage                                      //
//:usa//////////////////////////////////////// */
void usage(void)
{
    static char msg[] = "\n\
\
ZMIN -- Zmodem Protocol Driver 1.00\n\
 (c) 1994, Mike Dumdei, 6 Holly Lane, Texarkana Tx 75503\n\
\n\
** CODE DEMO ONLY!  DO NOT DISTRIBUTE AS AN .EXE!\n\
\n\
zmin com# { -B##### -L##### -H -E# } -R\n\
zmin com# { -B##### -L##### -H } -S file1 file2 ...\n\n";

    printf(msg);
    exit(100);
}


