
/*/////////////////////////////////////////////////////////////////////
//                                                                   //
//  XYZGEN.C -- functions common to xmodem, ymodem, zmodem           //
//                                                                   //
//    (c) 1991, Mike Dumdei, 6 Holly Lane, Texarkana TX, 75503       //
//                                                                   //
//////////////////////////////////////////////////////////////////// */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include <bios.h>
#include "comm.h"
#include "extra.h"
#include "zmdos.h"
#ifdef __TURBOC__
  #include <dir.h>
#endif

/*--------------------------------------------------------------------+
|                                                                     |
|                             Data Section                            |
|                                                                     |
+--------------------------------------------------------------------*/
#define GLOBAL
/*--------------------------------------------+
|  initialized globals used by x,y,zmodem     |
+--------------------------------------------*/
GLOBAL ASYNC *tPort = NULL;     /* pointer to port structure */
GLOBAL char *DfltPath = "";     /* default download path */
GLOBAL long ConnectBaud = 0L;   /* set to connect rate if using fixed link */
GLOBAL int RxTimeOut = 182;     /* receive char timeout in ticks (1/18 s.) */
GLOBAL int DefRxTimeOut = 182;  /* default value for RxTimeOut */
GLOBAL int VBufSize = 2048;     /* size of variable disk buffer to use */
GLOBAL WORD BufSize = 0;        /* 0 if overlaying I/O, else <= VBufSize */
GLOBAL TFLAGS TFlag = { 0x29 }; /* transfer flags (resume,keeptime,ign car) */

/*--------------------------------------------+
|  uninitialized globals used by x,y,zmodem   |
+--------------------------------------------*/
GLOBAL char *PathName = NULL;   /* full pathname of current file */
GLOBAL char *NameExt = NULL;    /* name and extension only of current file */
GLOBAL long FilePos = 0L;       /* current file offset */
GLOBAL long FileSize = 0L;      /* length of current file */
GLOBAL int BlkLen = 0;          /* current block length */
GLOBAL int BytesRxd = 0;        /* # bytes in last data packet received */
GLOBAL BYTE *IOBuf = NULL;      /* I/O buffer */
GLOBAL BYTE RxQueue[128] = "";  /* receive queue */
GLOBAL BYTE *RxQPtr = NULL;     /* pointer to current RxQueue position */
GLOBAL int RxBytesQd = 0;       /* bytes in receive queue */

/*--------------------------------------------+
|  static variables                           |
+--------------------------------------------*/
static FILE *tFh = NULL;        /* file handle of current file */
static BYTE *VBuf = NULL;       /* buffer for setvbuf */
static int XoffFlow = 0;        /* XOFF flow control active */
static int CTSFlow = B_CTS;     /* CTS flow control inactive (CTS high) */

/*--------------------------------------------+
|  16 bit CRC table                           |
+--------------------------------------------*/
GLOBAL WORD Crc16Tbl[256] =
{
    0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50A5, 0x60C6, 0x70E7, 
    0x8108, 0x9129, 0xA14A, 0xB16B, 0xC18C, 0xD1AD, 0xE1CE, 0xF1EF, 
    0x1231, 0x0210, 0x3273, 0x2252, 0x52B5, 0x4294, 0x72F7, 0x62D6, 
    0x9339, 0x8318, 0xB37B, 0xA35A, 0xD3BD, 0xC39C, 0xF3FF, 0xE3DE, 
    0x2462, 0x3443, 0x0420, 0x1401, 0x64E6, 0x74C7, 0x44A4, 0x5485, 
    0xA56A, 0xB54B, 0x8528, 0x9509, 0xE5EE, 0xF5CF, 0xC5AC, 0xD58D, 
    0x3653, 0x2672, 0x1611, 0x0630, 0x76D7, 0x66F6, 0x5695, 0x46B4, 
    0xB75B, 0xA77A, 0x9719, 0x8738, 0xF7DF, 0xE7FE, 0xD79D, 0xC7BC, 
    0x48C4, 0x58E5, 0x6886, 0x78A7, 0x0840, 0x1861, 0x2802, 0x3823, 
    0xC9CC, 0xD9ED, 0xE98E, 0xF9AF, 0x8948, 0x9969, 0xA90A, 0xB92B, 
    0x5AF5, 0x4AD4, 0x7AB7, 0x6A96, 0x1A71, 0x0A50, 0x3A33, 0x2A12, 
    0xDBFD, 0xCBDC, 0xFBBF, 0xEB9E, 0x9B79, 0x8B58, 0xBB3B, 0xAB1A, 
    0x6CA6, 0x7C87, 0x4CE4, 0x5CC5, 0x2C22, 0x3C03, 0x0C60, 0x1C41, 
    0xEDAE, 0xFD8F, 0xCDEC, 0xDDCD, 0xAD2A, 0xBD0B, 0x8D68, 0x9D49, 
    0x7E97, 0x6EB6, 0x5ED5, 0x4EF4, 0x3E13, 0x2E32, 0x1E51, 0x0E70, 
    0xFF9F, 0xEFBE, 0xDFDD, 0xCFFC, 0xBF1B, 0xAF3A, 0x9F59, 0x8F78, 
    0x9188, 0x81A9, 0xB1CA, 0xA1EB, 0xD10C, 0xC12D, 0xF14E, 0xE16F, 
    0x1080, 0x00A1, 0x30C2, 0x20E3, 0x5004, 0x4025, 0x7046, 0x6067, 
    0x83B9, 0x9398, 0xA3FB, 0xB3DA, 0xC33D, 0xD31C, 0xE37F, 0xF35E, 
    0x02B1, 0x1290, 0x22F3, 0x32D2, 0x4235, 0x5214, 0x6277, 0x7256, 
    0xB5EA, 0xA5CB, 0x95A8, 0x8589, 0xF56E, 0xE54F, 0xD52C, 0xC50D, 
    0x34E2, 0x24C3, 0x14A0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405, 
    0xA7DB, 0xB7FA, 0x8799, 0x97B8, 0xE75F, 0xF77E, 0xC71D, 0xD73C, 
    0x26D3, 0x36F2, 0x0691, 0x16B0, 0x6657, 0x7676, 0x4615, 0x5634, 
    0xD94C, 0xC96D, 0xF90E, 0xE92F, 0x99C8, 0x89E9, 0xB98A, 0xA9AB, 
    0x5844, 0x4865, 0x7806, 0x6827, 0x18C0, 0x08E1, 0x3882, 0x28A3, 
    0xCB7D, 0xDB5C, 0xEB3F, 0xFB1E, 0x8BF9, 0x9BD8, 0xABBB, 0xBB9A, 
    0x4A75, 0x5A54, 0x6A37, 0x7A16, 0x0AF1, 0x1AD0, 0x2AB3, 0x3A92, 
    0xFD2E, 0xED0F, 0xDD6C, 0xCD4D, 0xBDAA, 0xAD8B, 0x9DE8, 0x8DC9, 
    0x7C26, 0x6C07, 0x5C64, 0x4C45, 0x3CA2, 0x2C83, 0x1CE0, 0x0CC1, 
    0xEF1F, 0xFF3E, 0xCF5D, 0xDF7C, 0xAF9B, 0xBFBA, 0x8FD9, 0x9FF8, 
    0x6E17, 0x7E36, 0x4E55, 0x5E74, 0x2E93, 0x3EB2, 0x0ED1, 0x1EF0
};

/*--------------------------------------------------------------------+
|                                                                     |
|                             Code Section                            |
|                                                                     |
+--------------------------------------------------------------------*/

/*/////////////////////////////////////////////////////////
//  AsyncMode - save/set/restore async mode of operation //
/////////////////////////////////////////////////:async: */
void _C_ AsyncMode(int mode)
{
    static char params[10];
    static BYTE xmode, modeset = 0;
    char *n_params;
    BYTE n_xmode;

    if (mode != 0)                  /* 1 = xflow, 2 = no xflow */
    {                               /*  both 1 & 2 set "N81"   */
        if (modeset == 0)
        {
            xmode = tPort->Stat3 & (B_XUSET | B_XUSER);
            strcpy(params, tPort->BPDSstr);
            ++modeset;
        }
        n_xmode = (mode == 1) ? B_XUSE : 0;
        strcpy((strchr(n_params = tPort->BPDSstr, '\0') - 3), "N81");
    }
    else
    {
        if (modeset == 0)
            return;
        --modeset;
        n_xmode = xmode;            /* restore configuration   */
        n_params = params;
    }
    async_xflow(tPort, n_xmode);
    async_setbpds(tPort, n_params);
}

/*/////////////////////////////////////////////////////////
//  Blk16Crc - calculates 16 bit crc for a block of bytes//
//   returns: 16 bit crc                                 //
///////////////////////////////////////////////////:crc: */
WORD _C_ Blk16Crc(register BYTE *block, int bytes)
{
    register WORD crc = 0;

    while (bytes--)
        crc = Upd16Crc(crc, *block++);
    return (crc);
}

/*/////////////////////////////////////////////////////////
//  CloseFile -- close disk file                         //
//   statics: VBuf, tFh                                  //
//////////////////////////////////////////////////:disk: */
void _C_ CloseFile(void)
{
    if (tFh == NULL)
        return;
    if (TFlag.F.Receiving && TFlag.F.KeepTime)
    {
        fflush(tFh);
        UnixToDos(&DosTime, &DosDate, RxdUnixTime);
        DosSetFileTime(fileno(tFh), DosDate, DosTime);
    }
    fclose(tFh);
    tFh = NULL;
    if (VBuf != NULL)
        free(VBuf), VBuf = NULL;
}

/*/////////////////////////////////////////////////////////
//  FlushDisk -- flush file buffers                      //
//////////////////////////////////////////////////:disk: */
void _C_ FlushDisk(void)
{
    fflush(tFh);
}

/*/////////////////////////////////////////////////////////
//  GetNameExt -- get pointer to file name.ext component //
//   returns: pointer to name.ext                        //
/////////////////////////////////////////////////:parse: */
char * _C_ GetNameExt(char *FullPath)
{
    register char *p1 = strchr(FullPath, '\0');
    while (p1-- > FullPath)
    {
        if (*p1 == '\\' || *p1 == ':' || *p1 == '/')
            break;
    }
    return (++p1);
}

/*/////////////////////////////////////////////////////////
//  LoadRxQueue - block receive with time out            //
//   returns: received character, fatal error, TIMED_OUT //
/////////////////////////////////////////////////:async: */
int _C_ LoadRxQueue(void)
{
    long rxtimer, xontimer;

    RxQPtr = RxQueue;
    if (!async_rxcnt(tPort))
    {
        set_timeout(&rxtimer, RxTimeOut);
        set_timeout(&xontimer, XonTimeOut);
    }
    while (1)
    {
        ZMsg(M_IDLE);
        if (KBHIT && ((KBREAD & 0xff) == 0x1b))
            return LCL_ABORT;
        if (!async_carrier(tPort) && !TFlag.F.IgnCarrier)
            return LOST_CARRIER;
        if ((RxBytesQd =
         async_rxblk(tPort, (char *)RxQueue, sizeof(RxQueue))-1) >= 0)
            return (*RxQPtr++);
        if (timed_out(&rxtimer))
            return TIMED_OUT;
        if (async_xuse(tPort) && timed_out(&xontimer))
        {
            async_tx(tPort, XON);
            set_timeout(&xontimer, XonTimeOut);
        }
    }
}

/*/////////////////////////////////////////////////////////
//  OpenFile -- open disk file                           //
//   returns: 1 if file opened, 0 if open failed         //
//   statics: tFh, VBuf                                  //
//////////////////////////////////////////////////:disk: */
int _C_ OpenFile(char *name, char *mode)
{
    if ((tFh = fopen(name, mode)) == NULL)
        return 0;
    if (VBufSize > 512 && (VBuf = malloc(VBufSize)) != NULL)
        setvbuf(tFh, (char *)VBuf, _IOFBF, VBufSize);
    return 1;
}

/*/////////////////////////////////////////////////////////
//  ReadDisk -- read disk block padding with ctrl-z's    //
//   returns: number of bytes read                       //
//////////////////////////////////////////////////:disk: */
int _C_ ReadDisk(int bytes)
{
    int i;

    if ((i = fread(IOBuf, 1, bytes, tFh)) < bytes)
    {
        if (!feof(tFh))
            return FILE_ERR;
        memset(&IOBuf[i], '\x1a', bytes - i);
    }
    return (i);
}

/*/////////////////////////////////////////////////////////
//  Seek -- seek to file position                        //
//   returns: 0 if success, else FILE_ERR                //
//////////////////////////////////////////////////:disk: */
int _C_ Seek(long pos)
{
    return ((fseek(tFh, pos, SEEK_SET) == 0) ? 0 : FILE_ERR);
}

/*/////////////////////////////////////////////////////////
//  SendBreak - send ~100 ms break signal                //
/////////////////////////////////////////////////:async: */
void _C_ SendBreak(void)
{
    Delay(1);
    async_sndbrk(tPort, 1);
    Delay(2);
    async_sndbrk(tPort, 0);
}

/*/////////////////////////////////////////////////////////
//  WaitTxLevel - monitor transmit buffer level          //
//   returns: 1 = transmit buffer <= level               //
//            0 = transmit buffer > level                //
//   generates M_IDLE ZMsg if still waiting              //
//   generates M_FLOW ZMsgs with following args:         //
//            0 - delta XOFF (XOFF cleared)              //
//            1 - delta XOFF (got an XOFF)               //
//            2 - delta CTS (ok to send)                 //
//            3 - delta CTS (flow control enabled)       //
//            4 - port error (port restarted)            //
/////////////////////////////////////////////////////:z: */
int _C_ WaitTxLevel(int level)
{
    static long xofftimer;
    register int ch, count;
    long ticks;

    if (((ch = async_cts(tPort)) ^ CTSFlow) && !(tPort->MSRMask & B_CTS))
    {
        ZMsg(M_FLOW, (ch) ? 2 : 3);
        CTSFlow = ch;
    }
    if ((ch = async_xrxd(tPort)) ^ XoffFlow)
        ZMsg(M_FLOW, (ch) ? 1 : 0);
    if (ch)
    {
        if (XoffFlow == 0)
            set_timeout(&xofftimer, XoffTimeOut);
        else if (timed_out(&xofftimer))
            async_xoffclr(tPort);
    }
    XoffFlow = ch;
    if ((count = async_txcnt(tPort)) <= level)
        return 1;
    if (!async_carrier(tPort) && !TFlag.F.IgnCarrier)
        return LOST_CARRIER;
    if ((ch = KBHIT) != 0)
    {
        if ((ch & 0xff) == 0x1b)
            return LCL_ABORT;
        KBREAD;
    }
    set_timeout(&ticks, 4);
    while ((count == async_txcnt(tPort)) && !async_xrxd(tPort)
     && async_cts(tPort) && !timed_out(&ticks))
        ;
    if (timed_out(&ticks))
    {
        ZMsg(M_FLOW, 4);
        async_restart(tPort);
        return 1;
    }
    if (count <= level)
        return 1;
    ZMsg(M_IDLE);
    return 0;
}

/*/////////////////////////////////////////////////////////
//  WriteDisk -- write disk block                        //
//   returns: bytes written or FILE_ERR                  //
//////////////////////////////////////////////////:disk: */
int _C_ WriteDisk(int bytes)
{
    return (fwrite(IOBuf, 1, bytes, tFh) == bytes) ? bytes : FILE_ERR;
}

/*/////////////////////////////////////////////////////////
//  XYInitVars - init variables common to X,Y,Zmodem     //
//   returns: 0 if successful, else MEM_ERR              //
//////////////////////////////////////////////////////// */
int _C_ XYInitVars(ASYNC *comport)
{
    tPort = comport;
    if ((IOBuf = calloc(MAXPACKET + MAX_PATH + 50, 1)) == NULL)
        return MEM_ERR;
    PathName = (char *)&IOBuf[MAXPACKET + 44];
    BlkLen = (TFlag.F.KBlocks) ? 1024 : 128;
    tFh = NULL, VBuf = NULL;
    RxBytesQd = XoffFlow = 0;
    return 0;
}

