/*
   FILE : mapdemo.c

          a good example of how to use the game lib.

     Limitations:

      * tile indexes are bytes, so maximum 256 different tiles
      * tile files must fit in 64K,

*/

#include <stdio.h>
#include <io.h>
#include <stdlib.h>
#include <conio.h>
#include <alloc.h>
#include <dos.h>
#include <time.h>
#include <string.h>

#include <pr2.h>
#include <palette.h>
#include <readlbm.h>
#include <fstring.h>
#include <g_io.h>
#include <gui.h>
#include <erase.h>
#include <die.h>
#include <gmalloc.h>
#include <gsound.h>

#include <Xlib_all.h>

#include <colors.h>
#include <gtimer.h>

#include <sprite.h>
#include <sprbank.h>

#include <tilemap.h>

#define TICKS_PER_SECOND 60

#define MOVE_INCREMENT 24

#define NUM_TANK_ROTATIONS 12


//------- Constants ---------
#define MOVE_STEP 12
#define ENABLE_ERASE 1
#define NO_ERASE 0

#define NUM_GOODGUY_SPRITES 1
#define NUM_BADGUY_SPRITES 10
#define NUM_EXPLOSION_SPRITES 10
#define NUM_MISSILE_SPRITES 8
#define NUM_BADSHELL_SPRITES 15


// clipping rectangle
#define WIN_LEFT 20
#define WIN_RIGHT 343
#define WIN_TOP  23
#define WIN_BOTTOM 200

#define MOVE_DELAY_VARIATION 50
#define SHOOT_DELAY_VARIATION 80

#define MOVE_DELAY_MIN 0
#define SHOOT_DELAY_MIN 20

#define LEVEL_START_DELAY 200
#define LEVEL_END_DELAY 50

#define MAPMOVE_MARGIN 40

#define TANK_APPROX_WIDTH 20
#define TANK_APPROX_HEIGHT 20

#define SHELL_CENTER 4

#define TURN_DELAY 10  // turn delay in 1/18 sec ticks = 4/18 = 1/4 sec.

enum SpriteBankIDs {GGUY_ID, BADGUY_ID, BADSHELLS_ID, EXPLD_ID, MISSILE_ID};

enum controllerPos { X_LEFT, X_CENTER, X_RIGHT, Y_UP, Y_CENTER, Y_DOWN };

//--------- Local storage ----------
static short level;
static short  gameOver = 0, levelOver=0;
static USHORT badGuysHit=0, goodGuyHit=0;
static short badGuysAlive=0;
static long  nextTurn;

sprite_bank_t  far *badGuyBank, far *badShellsBank, far *explodeBank,
               far *missileBank, far *gguyBank;

static sprite_t far *gguy;
static short gguyRotation=0;
static short controlX = X_CENTER, controlY=Y_CENTER;

static map_t far *map;

static short tankFwdX[]= { 0, 4, 4, 4, 4, 4, 0, -4, -4, -4, -4, -4  };
static short tankFwdY[]= { -3, -2, -1, 0, 1, 2, 3, 2, 1, 0, -1, -2  };

static short tankRevX[]= { 0, -4, -4, -4, -4, -4, 0, 4,  4, 4, 4, 4  };
static short tankRevY[]= { 3, 2, 1, 0, -1,-2, -3, -2, -1, 0, 1, 2  };

static short shellVelX[]= { 0, 2, 4, 8, 4, 2, 0, -2, -4, -8, -4, -2  };
static short shellVelY[]= { -8, -4, -2, 0, 2, 4, 8, 4, 2, 0, -2, -4  };

static short badGuysRot[ NUM_BADGUY_SPRITES ];
static short badGuysRotDir[ NUM_BADGUY_SPRITES ];

static long badGuysShootNext[ NUM_BADGUY_SPRITES ];
static long badGuysTurnNext[ NUM_BADGUY_SPRITES ];

static BYTE pal[768];


// Local function prototypes
void drawScreen(void);
static void updateScreenInfo(short level);
short chgRotation( short rotation, short dRot );
void initBadGuysAI( void );
static void badGuysTurnShoot( void );
static void startNewBadGuy( void );
void eraseAndDraw( void );
void checkCollisions(void);
static void move(void);
static void userControl( void );
void startLevel( short level );
void endLevel( short level );
void playLevel( short level );


/*---------------------------------------------------------------------------

   main()

---------------------------------------------------------------------------*/
void main(void)
{
short i;

   randomize();

   init_timer();
   init_xmode_video();

   init_mem_list();  // crucial

   /* set palette */
   if ( load_palette("bruce.pal", (char far *)pal) )
      die("error getting palette");

   // set palette to dark intially
   for( i= 0; i < 768; i++ )
      palette[i]= 0;

   setvgapalette((char far *)palette);

   init_events("crshair.cbm");

   /* set aside mem etc.. for 50 erase rects, not to exceed 32k */
   init_gb_erase(50, 32000);

   page=0;

   map= load_tile_map( "mapdemo.map", WIN_LEFT, WIN_RIGHT,
                          WIN_TOP, WIN_BOTTOM );         // screen window

  // Read sprites from files, intialize...
  // NOTE: init order determines which sprites drawn over top of others
  //  ie: first initialized will be first drawn, therefore last initialized
  //  will be drawn over top of others
  init_sprites();


  badGuyBank= setup_spritebank("grtnkpb.spr", NUM_BADGUY_SPRITES, BADGUY_ID,
                            WIN_LEFT, WIN_RIGHT, WIN_TOP, WIN_BOTTOM,
                            map->min_map_x, map->max_map_x, map->min_map_y, map->max_map_y,
                            ENABLE_ERASE, NUM_BADGUY_SPRITES);

  gguyBank= setup_spritebank("tankpbm.spr",   NUM_GOODGUY_SPRITES, GGUY_ID,
                            WIN_LEFT, WIN_RIGHT, WIN_TOP, WIN_BOTTOM,
                            map->min_map_x, map->max_map_x, map->min_map_y, map->max_map_y,
                            ENABLE_ERASE, NUM_GOODGUY_SPRITES);

  missileBank= setup_spritebank("eball.spr", NUM_MISSILE_SPRITES, MISSILE_ID,
                            WIN_LEFT, WIN_RIGHT, WIN_TOP, WIN_BOTTOM,
                            map->min_map_x, map->max_map_x, map->min_map_y, map->max_map_y,
                            ENABLE_ERASE, NUM_MISSILE_SPRITES);

  badShellsBank= setup_spritebank("eball.spr", NUM_BADSHELL_SPRITES, BADSHELLS_ID,
                            WIN_LEFT, WIN_RIGHT, WIN_TOP, WIN_BOTTOM,
                            map->min_map_x, map->max_map_x, map->min_map_y, map->max_map_y,
                            ENABLE_ERASE, NUM_BADSHELL_SPRITES);

  explodeBank= setup_spritebank("expl_pbm.spr", NUM_EXPLOSION_SPRITES, EXPLD_ID,
                            WIN_LEFT, WIN_RIGHT, WIN_TOP, WIN_BOTTOM,
                            map->min_map_x, map->max_map_x, map->min_map_y, map->max_map_y,
                            ENABLE_ERASE, NUM_EXPLOSION_SPRITES);

   gguy = gguyBank->sprites;

   start_sprite(gguy, SPR_CHG_STOP, SPR_NO_LIMITS, WIN_LEFT+100, WIN_TOP+100, 0,0);
   set_bitmap( gguy, gguyRotation );

   initBadGuysAI();

   level= 1;
   while ( !gameOver )
      {
      startLevel( level );
      playLevel( level );
      endLevel( level );
      level++;
      }

   fade_out(0);

   deinit_sprites();
   deinit_tile_map(map);
   deinit_gb_erase();
   deinit_events();
   deinit_xmode_video();
   deinit_timer();
}


/*---------------------------------------------------------------------------

   drawScreen()

---------------------------------------------------------------------------*/
void drawScreen(void)
{
short clip_left=WIN_LEFT, clip_right=WIN_RIGHT+1, clip_top=WIN_TOP, clip_bott=WIN_BOTTOM+1;

   // draw clipping rectangle
   x_line(clip_left-1,clip_top-1,clip_right,clip_top-1,EGA_WHITE,VisiblePageOffs);
   x_line(clip_right,clip_top-1,clip_right,clip_bott, EGA_WHITE,VisiblePageOffs);
   x_line(clip_right,clip_bott,clip_left-1,clip_bott,EGA_WHITE,VisiblePageOffs);
   x_line(clip_left-1,clip_bott,clip_left-1,clip_top-1,EGA_WHITE,VisiblePageOffs);

   x_line(clip_left-1,clip_top-1,clip_right,clip_top-1,EGA_WHITE,HiddenPageOffs);
   x_line(clip_right,clip_top-1,clip_right,clip_bott, EGA_WHITE,HiddenPageOffs);
   x_line(clip_right,clip_bott,clip_left-1,clip_bott,EGA_WHITE,HiddenPageOffs);
   x_line(clip_left-1,clip_bott,clip_left-1,clip_top-1,EGA_WHITE,HiddenPageOffs);

   // print instructions
   x_printf(7,221,VisiblePageOffs,EGA_DARKGRAY,
    (char far *)"  <ArrowKeys,Joystick=Move><SpaceBar=Shoot>");
   x_printf(6,220,VisiblePageOffs,EGA_RED,
    (char far *)"  <ArrowKeys,Joystick=Move><SpaceBar=Shoot>");

   x_printf(7,221,HiddenPageOffs,EGA_DARKGRAY,
    (char far *)"  <ArrowKeys,Joystick=Move><SpaceBar=Shoot>");
   x_printf(6,220,HiddenPageOffs,EGA_RED,
    (char far *)"  <ArrowKeys,Joystick=Move><SpaceBar=Shoot>");

   draw_all_tiles(map);
   draw_low_dressings(map);
   draw_high_dressings(map);
   updateScreenInfo(level);

   /* page flip */
   x_page_flip(0,0);
   page= page ? 0 : 1;

   draw_all_tiles(map);
   draw_low_dressings(map);
   draw_high_dressings(map);
   updateScreenInfo(level);
}

/*---------------------------------------------------------------------------

    updateScreenInfo()

---------------------------------------------------------------------------*/
static void updateScreenInfo(short level)
{
static short frameCount, framesPerSecond;
static long frameTimer;

    frameCount++;
     if( fast_tick < frameTimer )
        return;

     frameTimer = fast_tick + TICKS_PER_SECOND;
     framesPerSecond = frameCount;
     frameCount = 0;

      x_bgprintf(6,205,VisiblePageOffs,EGA_YELLOW,EGA_BLACK,
       (char far *)"  Level:%2d   BadGuys hit:%d   GoodGuy hit:%d",
      level, badGuysHit, goodGuyHit );
      x_bgprintf(6,205,HiddenPageOffs,EGA_YELLOW,EGA_BLACK,
       (char far *)"  Level:%2d   BadGuys hit:%d   GoodGuy hit:%d",
      level, badGuysHit, goodGuyHit );

}

/*---------------------------------------------------------------------------

   chgRotation()

---------------------------------------------------------------------------*/
short chgRotation( short rotation, short dRot )
{
   rotation+=dRot;

   if ( dRot >= 0 )
      {
      if ( rotation >= NUM_TANK_ROTATIONS)
         rotation = 0;
      }
   else
      {
      if ( rotation < 0 )
         rotation = NUM_TANK_ROTATIONS-1;
      }
   return rotation;
}

/*---------------------------------------------------------------------------

   initBadGuysAI()

---------------------------------------------------------------------------*/
void initBadGuysAI( void )
{
short i;

   for(i = 0; i < NUM_BADGUY_SPRITES; i++)
      {
      badGuysTurnNext[i] = fast_tick + random( MOVE_DELAY_VARIATION )+ MOVE_DELAY_MIN;
      badGuysShootNext[i] = fast_tick + random( SHOOT_DELAY_VARIATION) + SHOOT_DELAY_MIN;
      badGuysRotDir[i]= -1;
      }
}

/*---------------------------------------------------------------------------

   badGuysTurnShoot()

---------------------------------------------------------------------------*/
static void badGuysTurnShoot( void )
{
short i, shellDx, shellDy;
sprite_t far *sPtr;

   for(i = 0; i < NUM_BADGUY_SPRITES; i++)
      {
      sPtr = badGuyBank->sprites+i;
      if( sPtr->change_mode != SPR_CHG_DEAD )
         {

         // Turn bad guy if time to do so
         if ( fast_tick > badGuysTurnNext[ i ] )
            {
//          badGuysRot[i]= chgRotation( badGuysRot[i], badGuysTurnNext[i] );
            badGuysRot[i]= chgRotation( badGuysRot[i], 1 );
            set_bitmap( sPtr, badGuysRot[i] );

            sPtr->dx= tankFwdX[ badGuysRot[i] ];
            sPtr->dy= tankFwdY[ badGuysRot[i] ];

            badGuysTurnNext[i] = fast_tick + random(MOVE_DELAY_VARIATION) + MOVE_DELAY_MIN;
            }

         // shoot if time to do so
         //--- fire shell ---
         if ( fast_tick > badGuysShootNext[ i ] )
            {
            shellDx= shellVelX[ badGuysRot[i] ];
            shellDy= shellVelY[ badGuysRot[i] ];
            start_next_in_spritebank( badShellsBank,
                     SPR_CHG_FORW_HOLD,   SPR_DIE_AT_XYLIMIT,
                     sPtr->x+sPtr->x_offs1-SHELL_CENTER,
                     sPtr->y+sPtr->y_offs1-SHELL_CENTER,
                     shellDx, shellDy );
            badGuysShootNext[i] = fast_tick + random( SHOOT_DELAY_VARIATION )+ SHOOT_DELAY_MIN;
            }
         }
      }
}

/*---------------------------------------------------------------------------

   startNewBadGuy()

---------------------------------------------------------------------------*/
static void startNewBadGuy( void )
{
int i;
sprite_t far *sPtr;

   for(i = 0; i < NUM_BADGUY_SPRITES; i++)
      {
      sPtr = badGuyBank->sprites+i;
      if( sPtr->change_mode == SPR_CHG_DEAD )
         {
         start_sprite( sPtr, SPR_CHG_STOP, SPR_HOLD_AT_XYLIMIT,
                       gguy->x+random(200), gguy->y+90, 0, 0 );
         sPtr->x_delay=2; sPtr->y_delay=2;   // delay between moves

         badGuysRot[ i ] = 8;

         set_bitmap( sPtr, badGuysRot[i] );
         sPtr->dx= tankFwdX[ badGuysRot[i] ];
         sPtr->dy= tankFwdY[ badGuysRot[i] ];

         badGuysAlive++;
         break;
         }
      }
}

/*---------------------------------------------------------------------------

   eraseAndDraw()

---------------------------------------------------------------------------*/
void eraseAndDraw( void )
{
   /* erase old stuff */
   draw_erase_rects(&(gb_h[page]));

   // update tiles
   update_map_except_high_dressings( map );

   // draw all sprites
   offset_draw_all_banks(map->x-map->win_left, map->y-map->win_top );

   // draw high dressings on map
   draw_high_dressings(map);
}

/*---------------------------------------------------------------------------

   checkCollisions()

---------------------------------------------------------------------------*/
void checkCollisions(void)
{
sprite_t far *sPtr;


   //--- check for missiles hitting bad guys
   if( (sPtr = detect_collisions(0, missileBank, badGuyBank) ) != NULL )
    {
    sPtr->change_mode = SPR_CHG_DEAD;  // kill bad guy
    start_next_in_spritebank(explodeBank,   SPR_CHG_FORW_DIE,
                               SPR_NO_LIMITS, sPtr->x+(sPtr->curr_width>>2),
                               sPtr->y+(sPtr->curr_height>>2), 0, 0);
    badGuysHit++;
    badGuysAlive--;
    }

   //--- check goodguy hitting bad guys
   if( (sPtr = detect_collisions(0, badGuyBank, gguyBank) ) != NULL )
    {
//    sPtr->change_mode = SPR_CHG_DEAD;  // don't kill good guy
    start_next_in_spritebank(explodeBank,   SPR_CHG_FORW_DIE,
                               SPR_NO_LIMITS, sPtr->x+(sPtr->curr_width>>2),
                               sPtr->y+(sPtr->curr_height>>2), 0, 0);
    badGuysHit++;
    badGuysAlive--;
    }

   //--- check for bad guys shells hitting good guy
   if( (sPtr = detect_collisions(0, badShellsBank, gguyBank) ) != NULL )
    {
//    sPtr->change_mode = SPR_CHG_DEAD;  // don't kill good guy!!!
    start_next_in_spritebank(explodeBank,   SPR_CHG_FORW_DIE,
                               SPR_NO_LIMITS, sPtr->x+(sPtr->curr_width>>2),
                               sPtr->y+(sPtr->curr_height>>2), 0, 0);
   goodGuyHit++;
    }

}

/*---------------------------------------------------------------------------

    move()

---------------------------------------------------------------------------*/
static void move(void)
{
short tileAttrib, oldX, oldY, xMapMove, yMapMove, screenX, screenY;

   oldX= gguy->x; oldY= gguy->y;

   // see if about to collide with impassable tile
   tileAttrib= will_collide_with_tile( map, gguy );

   stop_sprites_at_impassable_tiles( map, badGuyBank );
   kill_sprites_at_impassable_tiles( map, missileBank );
   kill_sprites_at_impassable_tiles( map, badShellsBank );

   move_limit_all_banks();               // move sprites
   move_limit_spritebank(map->spr_bank); // move map anims (sprites)

   if ( tileAttrib & TILE_IMPASSABLE )
      {
      gguy->x = oldX;
      gguy->y = oldY;
      }

   // turn, move goodguy
   switch(controlX)
      {
      case X_LEFT:
         // limit turn rate
         if ( nextTurn == -1 || fast_tick > nextTurn )
            {
            gguyRotation= chgRotation( gguyRotation, -1 );
            set_bitmap( gguy, gguyRotation );
            nextTurn = fast_tick+ TURN_DELAY;
            }
         break;

      case X_CENTER:
         nextTurn = -1;
         break;

      case X_RIGHT:
         // limit turn rate
         if ( nextTurn == -1 || fast_tick > nextTurn )
            {
            gguyRotation= chgRotation( gguyRotation, 1 );
            set_bitmap( gguy, gguyRotation );
            nextTurn = fast_tick+ TURN_DELAY;
            }
         break;
      }

   switch(controlY)
      {
      case Y_UP:
         gguy->dx= tankFwdX[ gguyRotation ];
         gguy->dy= tankFwdY[ gguyRotation ];
         break;

      case Y_DOWN:
         gguy->dx= tankRevX[ gguyRotation ];
         gguy->dy= tankRevY[ gguyRotation ];
         break;

      case Y_CENTER:
         gguy->dx= 0;
         gguy->dy= 0;
         break;
      }

   // move goodguy with respect to map
   if ( gguy->x < 0 )
      gguy->x = 0;
   else
      if ( gguy->x > map->max_map_x-TANK_APPROX_WIDTH )
         gguy->x = map->max_map_x-TANK_APPROX_WIDTH;


   if ( gguy->y < 0 )
      gguy->y = 0;
   else
      if ( gguy->y > map->max_map_y-TANK_APPROX_HEIGHT )
         gguy->y = map->max_map_y-TANK_APPROX_HEIGHT;

   // calculate the screenX, screenY for goodguy
   screenX= gguy->x-map->x+WIN_LEFT;
   screenY= gguy->y-map->y+WIN_TOP;

   xMapMove=0;
   yMapMove=0;

   // move map if goodguy screen x,y at any limits, or move goodguy on map
   if ( screenX < WIN_LEFT+MAPMOVE_MARGIN && gguy->dx< 0 )
      xMapMove= gguy->dx;
   else
      if ( screenX > WIN_RIGHT-MAPMOVE_MARGIN-TANK_APPROX_HEIGHT && gguy->dx> 0 )
         xMapMove= gguy->dx;

   if ( screenY < WIN_TOP+MAPMOVE_MARGIN && gguy->dy < 0)
      yMapMove= gguy->dy;
   else
      if ( screenY > WIN_BOTTOM-MAPMOVE_MARGIN-TANK_APPROX_WIDTH && gguy->dy > 0)
         yMapMove= gguy->dy;

   // *** IMPORTANT: update map uses a x_shift_rect, which must have
   //                dx= multiple of 4 (video ram byte boundaries).

   // move map if xMapMove or yMapMove (xMapMove now = 0 or mult of 4 )
   if ( xMapMove || yMapMove)
      move_map_xy(map, xMapMove, yMapMove );
}

/*---------------------------------------------------------------------------

    userControl()

---------------------------------------------------------------------------*/
static void userControl( void)
{
event_t event;
short gotEvent = 0, shellDx, shellDy;

   gotEvent = get_event(&event);

   if ( gotEvent )
      {
      switch ( event.type )
         {
         // latch joystick position
         case E_JOY_X_LEFT:   controlX= X_LEFT;   break;
         case E_JOY_X_CENTRE: controlX= X_CENTER; break;
         case E_JOY_X_RIGHT:  controlX= X_RIGHT;  break;

         case E_JOY_Y_UP:     controlY= Y_UP;     break;
         case E_JOY_Y_CENTRE: controlY= Y_CENTER; break;
         case E_JOY_Y_DOWN:   controlY= Y_DOWN;  break;


         case E_B0_DN:
         case E_B1_DN:                 //--- fire shell ---
            shellDx= shellVelX[ gguyRotation ];
            shellDy= shellVelY[ gguyRotation ];
            start_next_in_spritebank( missileBank,
                     SPR_CHG_FORW_HOLD,   SPR_DIE_AT_XYLIMIT,
                     gguy->x+gguy->x_offs1-SHELL_CENTER,
                     gguy->y+gguy->y_offs1-SHELL_CENTER,
                     shellDx, shellDy );
            break;

    case E_KEY:
       if ( event.sub_type == E_DOWN )
          {
          switch( event.d2 )
             {
             case ESC:  gameOver = 1;  break;

             // latch arrow keys for position control
             case LEFT:   controlX= X_LEFT;   break;
             case RIGHT:  controlX= X_RIGHT;  break;
             case UP:     controlY= Y_UP;     break;
             case DOWN:   controlY= Y_DOWN;  break;

             case SPACE:
                //--- fire shell ---
                shellDx= shellVelX[ gguyRotation ];
                shellDy= shellVelY[ gguyRotation ];
                start_next_in_spritebank( missileBank,
                         SPR_CHG_FORW_HOLD,   SPR_DIE_AT_XYLIMIT,
                         gguy->x+gguy->x_offs1-SHELL_CENTER,
                         gguy->y+gguy->y_offs1-SHELL_CENTER,
                         shellDx, shellDy );
                break;
             }
          }
          else
          {
          if ( event.sub_type == E_UP )
             {
             switch ( event.d2 )
                {
                case UP:    controlY= Y_CENTER; break;
                case DOWN:  controlY= Y_CENTER; break;
                case LEFT:  controlX= X_CENTER; break;
                case RIGHT: controlX= X_CENTER; break;
                }
             }
          }
          break;
       }
    }
}

/*---------------------------------------------------------------------------

   startLevel()

---------------------------------------------------------------------------*/
void startLevel( short level )
{
short i;
long timeToStartAction= fast_tick+LEVEL_START_DELAY;

   gguy->dx=0;
   gguy->dy=0;

   map->x= 0; map->y = 130;
   gguy->x = 150; gguy->y= 200;
   gguy->x_delay=0; gguy->y_delay=0;   // delay between moves


   // draw screen, flip pages, draw on other page too
   drawScreen();
   eraseAndDraw();
   x_page_flip(0,0);
   page= page ? 0 : 1;
   eraseAndDraw();

   fade_in( (char far *)pal, 0 );

   // drive tank around for a while without bad guys
   while( (fast_tick < timeToStartAction) && !gameOver )
      {
      userControl();
      move();
      checkCollisions();
      updateScreenInfo(level);

      /* page flip */
      x_page_flip(0,0);
      page= page ? 0 : 1;
      }

   // start right number of bad guys
   for( i = 0; i < level; i++ )
      startNewBadGuy();
}

/*---------------------------------------------------------------------------

   endLevel()

---------------------------------------------------------------------------*/
void endLevel( short level )
{
long timeToEndAction= fast_tick+LEVEL_END_DELAY;

   // drive tank around for a while without bad guys
   while( (fast_tick < timeToEndAction) && !gameOver )
      {
      eraseAndDraw();
      userControl();
      move();
      updateScreenInfo(level);

      x_page_flip(0,0);
      page= page ? 0 : 1;
      }

   fade_out(0);

   // clean up by erasing old stuff, both pages
   draw_erase_rects(&(gb_h[page]));
   x_page_flip(0,0);
   page= page ? 0 : 1;
   draw_erase_rects(&(gb_h[page]));

}

/*---------------------------------------------------------------------------

   playLevel()

---------------------------------------------------------------------------*/
void playLevel( short level )
{
   levelOver=0;
   while( !levelOver && !gameOver )
      {
      eraseAndDraw();
      userControl();
      move();
      badGuysTurnShoot();
      checkCollisions();
      updateScreenInfo(level);

      if ( badGuysAlive < 1 )
         {
         levelOver=1;
         break;
         }

      /* page flip */
      x_page_flip(0,0);
      page= page ? 0 : 1;
      }

   eraseAndDraw();
   updateScreenInfo(level);

   /* page flip */
   x_page_flip(0,0);
   page= page ? 0 : 1;

   drawScreen();
   eraseAndDraw();
   updateScreenInfo(level);
}



