;; File: XTRUD.LSP   olasov@cs.columbia.edu   ben@syska.com
;;
;; XTRUD.LSP extrudes a specified polyline section through a specified
;; polyline path.  The section is positioned along the polyline path
;; at each of the path's vertices.  The method for positioning the section
;; at each vertex is a user selectable parameter, as is the possibility
;; of alternating placement of a second section along the path.
;;
;; The XTRUD user is prompted successively for:
;;
;;       SHAPE to extrude:
;;
;;       Extrusion PATH:
;;
;;       Resize XY scaling at path intersecs [Y N]:
;;
;;       [optional] Alternation value for SHAPE iterations:
;;              Alternation type [R]everse [N]ormal:
;;
;;       Position shape at path vertices with reference to section's
;;       [C]entroid, [R]ightmost, [L]eftmost or [S]elected point on shape:
;;
;;       [optional] Divide path by = segs [Y N]:
;;              Number of path segs:
;;
;; XTRUD.LSP uses a number of global variables (ratherthan local variables)
;; to avoid repetitively passing large lists as arguments.
;;
;; Special requirements: None


;; set functions and variables to nil and garbage collect before
;; loading
(setq collect_vertices  nil     connect_plines    nil
      consecutive-pts   nil     resize_shp        nil
      _cnnct_shps       nil     delete_ss         nil
      extrude           nil     c:xtrud           nil
      translate_pts     nil     pos_in_list       nil
      l_left            nil     origin_pt         nil)

(gc) (vmon) (expand 3) 

(defun c:xtrud ()
       (setvar "cmdecho" 0)
       (setq shp* (_user_ent (list "POLYLINE") "\nSHAPE to extrude: "))
       (setq pth* (_user_ent (list "POLYLINE" "LINE") "\nExtrusion PATH: "))
       (setq resize?  ;; why resize?  because the diagonal section is longer
            (strcase (userstr (if resize? resize? "N") ;; than normal section.
                            "\nResize XY scaling at path intersecs [Y N]")))
       (setq alt1
             (userint (if alt1 alt1 1) 
                      "\nAlternation value for SHAPE iterations"))
       (if (and alt1 (> alt1 1))
           (setq alt_tp
                 (strcase (userstr (if alt_tp alt_tp "N") 
                          "\nAlternation type [R]everse [N]ormal"))))
       (setq instyp
            (strcase (userstr (if instyp instyp "R")
"\n[C]entroid, [R]ightmost, [L]eftmost or [S]elect vertex for shape insertion [C R L S]")))
       (setq dvd_pth?
             (strcase (userstr (if dvd_pth? dvd_pth? "N")
                      "\nDivide path by = segs [Y N]")))
       (if (equal dvd_pth? "Y")
           (progn (setq op (trans (origin_pt pth*) pth* 1)
                        res (userint (if res res 100) "\nPath segs")
                        plnds (divide_pline pth* res op))
                  (setq pth* nil pth* (pline plnds))))
       (extrude shp* pth*)
        'done)

(defun extrude ($shape $path)
       (cond ((not (and $shape $path)) nil)
             (T (setq vrts (collect_vertices $path)
                      clay (getvar "clayer")
                      shplayer (cdr (assoc 8 (entget $shape)))
                      &&p1 nil  &&p2 nil open_path? nil &&rot nil &&rotl nil
                      inc1 1    inc2 (if (equal alt_tp "N") alt1 1))
                (if (equal (l_left vrts) (last vrts))
                    (setq vrts (reverse vrts)))
                (setq $$basept (car vrts)
                      $shapes nil
                      $shapes (ssadd)
                      $shp_bp (cond ((equal instyp "C")
                                     (centroid (collect_vertices $shape)))
                                    ((equal instyp "L")
                                     (leftmost (collect_vertices $shape)))
                                    ((equal instyp "R")
                                     (rightmost (collect_vertices $shape)))
                                    ((equal instyp "S")
                                     (getpoint "\nSelect insertion point: "))
                                   (T (rightmost (collect_vertices $shape)))))
                (command "_layer" "_m" shplayer "")
                (temp_block $shape $shp_bp)
                (foreach vrt vrts
                         (setq &&p1 nil &&p2 nil &&rotl &&rot)
                         (if (and (not (equal vrt (car vrts)))
                                  (not (equal vrt (last vrts))))
                             ;; VRT is an intermediate point
                             (progn (princ "\ncase 1: intrmd pt")
                                    (setq case 1
                                          &&adj_pts (consecutive-pts vrt vrts)
                                          &&p1 (car &&adj_pts)
                                          &&p2 (last &&adj_pts)
                                          &&rot (rad2deg (opp_bisect_ang vrt &&p1 &&p2))))
                                 ;; VRT is the 1st or last point
                             (if (equal (car vrts) (last vrts))
                                 ;; and the 1st & last points are duplicates
                                 (if (eq vrt (car vrts)) 
                                     ;; and VRT is the first point
                                     (progn (princ "\ncase 2: 1st pt <dup>")
                                     (setq case 2
                                           &&p1 (cadr vrts) ;; a closed pgon
                                           &&p2 (cadr (reverse vrts))
                                           &&rot  ;;if VRT is the start pt 
                                           (rad2deg (opp_bisect_ang vrt ;L 1 2 
                                                                    &&p1
                                                                    &&p2))) )
                                     ;; and VRT is the last point
                                     (progn (princ "\ncase 3: last pt <dup>")
                                            (setq case 3 &&p1 nil &&p1 nil &&rot nil)))
                                 ;; 1st & last points are NOT duplicates
                                 (if (setq closed? (closedp $path))
                                     ;; but the pline is closed
                                     (if (eq vrt (car vrts)) 
                                         (progn (princ "\ncase 4: 1st pt <closed/non-dup>")
                                         (setq case 4
                                               &&p1 (cadr vrts)
                                               &&p2 (if (or (equal (last vrts) 
                                                                   (cadr (reverse vrts))) 
                                                            (equal (car vrts) 
                                                                   (cadr vrts))) 
                                                         (cadr (reverse vrts)) 
                                                         (cadr (reverse vrts)))
                                               &&rot  ;;if VRT is the start pt 
                                               (rad2deg (opp_bisect_ang vrt ;L 1 2 
                                                                        &&p1
                                                                        &&p2))))
                                         ;;if VRT is the last pt 
                                         (progn (princ "\ncase 5: last pt <closed/non-dup>")
                                         (setq case 5
                                               &&p1 (car vrts)
                                               &&p2 (cadr (reverse vrts))
                                               &&rot  ;;if VRT is the start pt 
                                               (rad2deg (opp_bisect_ang vrt ;L 1 2 
                                                                        &&p1
                                                                        &&p2)))))
                                     ;; the pline is open
                                     (progn (setq open_path? 'T)
                                            (if (/= (caddr (car vrts))
                                                    (caddr (last vrts)))
                                                (progn (princ "\ncase 6: 3dpoly endpt")
                                                       (setq case 6
                                                             &&p1 nil &&p2 nil
                                                             &&rot ;; VRT last of VRTS & != 1st of VRTS
                                                             (if (eq vrt (car vrts))      
                                                                 (rad2deg (opp_bisect_ang
                                                                                      vrt         ;; create projected startpt
                                                                                      (polar vrt  ;; for use by OPP_BISECT_ANG
                                                                                             (angle (cadr vrts) vrt)
                                                                                             (distance (cadr vrts) vrt))
                                                                                      (cadr vrts)))
                                                                (rad2deg (opp_bisect_ang vrt 
                                                                                         (cadr (reverse vrts))
                                                                              ;; create projected endpt for use by OPP_BISECT_ANG
                                                                                         (polar vrt 
                                                                                                (angle (cadr (reverse vrts))
                                                                                                vrt)
                                                                                                (distance (cadr vrts) vrt)))))))
                                                (progn (princ "\ncase 7: endpt <open contour>")
                                                       (setq case 7
                                                             opposite_pt (if (eq vrt (last vrts))
                                                                          (car vrts)
                                                                          (last vrts))
                                                             rightang? (right_angle vrt opposite_pt)
                                                             prior_pt (if (eq vrt (last vrts))
                                                                          (cadr (reverse vrts))
                                                                          (cadr vrts))
                                                             &&p1 prior_pt &&p2 opposite_pt)
                                                       (if rightang?
                                                           (setq &&rot (rad2deg (angle vrt opposite_pt)))
                                                           (if (and (nth 2 vrts) (nth 1 vrts) 
                                                                    (nth 2 (reverse vrts)) (nth 1 (reverse vrts)))
    
                                                               (if (eq vrt (car vrts))
                                                                       (setq &&rot ;; VRT last of VRTS & != 1st of VRTS
                                                                         (rad2deg (opp_bisect_ang vrt
                                                                                                  (next_pt_in_seq vrt
                                                                                                                  (nth 1 vrts)
                                                                                                                      (nth 2 vrts))
                                                                                                  (cadr vrts)))) 
                                                                   (setq &&rot ;; VRT last of VRTS & != 1st of VRTS
                                                                         (rad2deg (opp_bisect_ang vrt
                                                                                                      (next_pt_in_seq vrt
                                                                                                                  (nth 1 (reverse vrts))
                                                                                                                  (nth 2 (reverse vrts)))
                                                                                                  (cadr (reverse vrts))))))
                                                                   (if (eq vrt (car vrts))
                                                                   (setq &&rot ;; VRT last of VRTS & != 1st of VRTS
                                                                         (rad2deg (angle (cadr vrts) vrt)))
                                                                   (setq &&rot ;; VRT last of VRTS & != 1st of VRTS
                                                                         (rad2deg (angle (cadr (reverse vrts)) vrt))))))))))))
                             ;; to insert or not to insert
                             (if &&rot
                                (progn (setq go nil)
                                       (cond ((<= alt1 1)
                                             ; no alternation
                                              (setq go 'T))
                                             ((equal alt_tp "N")
                                             ; [N]ormal alternation
                                              (if (>= inc2 alt1)
                                                  (setq go 'T)))
                                             (T ; [R]everse alternation
                                                  (setq go 'T)))
                                           (if go
                                               (progn (xz_ucs &&rot)
                                                      (if (= (length vrts) 2)
                                                          (command "_insert" "*TEMP" (trans vrt pth* 1) ""
                                                                              (rad2deg (angle (if (equal vrt (car vrts))(last vrts) (car vrts)) )))
                                                          (command "_insert" "*TEMP" (trans vrt pth* 1) "" "0.0"))
                                                      (ssadd (entlast) $shapes)
                                                      (if (and (equal resize? "Y")
                                                               &&p1 &&p2
                                                               (/= (rem &&rot 90.0) 0.0))
                                                           (progn (resize_shp (entlast)
                                                                      (bisector vrt &&p1 &&p2) vrt)
                                                          (command "_ucs" "_w")))))))

                (princ (strcat "\nINC2 is: " (itoa inc2) " at node "))
                (princ inc1) (princ " of ")(princ (length vrts)) 
                (if (> inc2 alt1)
                    (if (equal alt_tp "N")
                        (setq inc2 2)
                        (setq inc2 1))
                    (setq inc2 (1+ inc2)))
                (setq inc1 (1+ inc1)))
                (if (and $shapes
                         (> (sslength $shapes) 1))
                    (_cnnct_shps $shapes))
                (command "_layer" "_m" clay ""))))
     
(defun next_pt_in_seq ($p1 $p2 $p3 / $ang3 $ang2 $dist21 $dltang $P0)
       (cond ((not (and $p1 $p2 $p3)) nil)
             (T (setq $ang3 (angle $p3 $p2)
                      $ang2 (angle $p2 $p1)
                      $dist21 (distance $p2 $p1)
                      $dltang (min (abs (- $ang3 $ang2))
                                   (abs (- $ang2 $ang3))))
                 (if (> $ang3 $ang2) 
                     (setq $ang1 (- $ang2 $dltang))
                     (setq $ang1 (+ $ang2 $dltang)))
                 (setq $p0 (polar $p1 $ang1 $dist21)))) $p0)

(defun delete_ss ($ss$)
       (if (and $ss$ (> (sslength $ss$) 0))
           (command "_erase" $ss$ "")))

(defun temp_block (_e_ bp)
       (if (blockp "TEMP")
           (command "_block" "TEMP" "_Y" bp _e_ "")
           (command "_block" "TEMP" bp _e_ ""))
       (command "_oops")
       (command "_ucs" "_w"))

(defun extract_value (key enm) 
       (if enm (cdr (assoc key (entget enm)))))

(defun _cnnct_shps ($sset$)
       (setq i      0     go      nil 
             inc1   1     inc2    (if (equal alt_tp "N") 2 alt1)
             alt2   (if (equal alt_tp "N") 2  alt1)
             $sset$l (sslength $sset$)
             clayer (getvar "clayer"))
       (setq first-shp (ssname $sset$ 0)
             last-shp (ssname $sset$ (1- $sset$l)))
       (command "_layer" "_n" "tmp1,tmp2" "_c" "6" "tmp1" "_c" "6" "tmp2" "_s" 
                clayer "_f" "tmp1" "_f" "tmp2" "")
       (command "_ucs" "_w")
       (repeat (1- $sset$l)
               (setq go nil
                     shap1 (ssname $sset$ i)
                     shap2 (ssname $sset$ (1+ i)))
               (cond ((<= alt2 1)          ; no alternation
                      (setq go 'T))
                     ((equal alt_tp "N")   ; [N]ormal alternation
                      (if (or (< alt1 2)
                              (= inc2 alt2))
                          (setq go 'T)))
                     (T (if (< inc2 alt2) ; [R]everse alternation
                            (setq go 'T))))
               (if (and shap1 shap2 go)
                   (connect_plines shap1 shap2))
               (setq i (1+ i))
               (if (equal alt_tp "N")
                   (if (>= inc2 alt2)
                       (setq inc2 1)
                       (setq inc2 (1+ inc2)))
                   (if (>= inc2 alt2)
                       (setq inc2 1)
                       (setq inc2 (1+ inc2))))
               (setq inc1 (1+ inc1)))
       (connect_plines first-shp last-shp)
;       (delete_ss $sset$) ;; uncomment this if you want to delete shapes
       (redraw))
                
(defun origin_pt (pln)
       (if (equal (cdr (assoc 0 (entget pln))) "POLYLINE")
           (cdr (assoc 10 (entget (entnext pln))))
           (cdr (assoc 10 (entget pln)))))

(defun opp_bisect_ang (vertex node1 node2 / @1 @2 @n1 @n2 @ct @ng)
       (setq @1 (angle vertex node1)
             @2 (angle vertex node2)
             @n1 (polar vertex @1 1.0)
             @n2 (polar vertex @2 1.0)
             @ct (centroid (list vertex @n1 @n2)))
       (setq @ng (angle vertex @ct)))

;; returns the 2D or 3d centroid of a list of vertices
(defun centroid (verts / _verts num_verts x_avg y_avg z_avg centrd z_coords)
       (cond ((or (null verts)
                  (null (listp verts))
                  (member nil (mapcar 'listp verts)))
               nil)
             (T (if (setq z_coords (mapcar 'caddr _verts))
                    (setq z_avg (apply '+ z_coords))
                    (setq z_avg nil))
                (setq _verts (unique_atoms verts)
                      num_verts (length _verts)
                      x_avg (/ (apply '+ (mapcar 'car _verts)) num_verts)
                      y_avg (/ (apply '+ (mapcar 'cadr _verts)) num_verts)
                      centrd (if z_avg
                                 (list x_avg y_avg z_avg)
                                 (list x_avg y_avg))))) centrd)


(defun 2d (&pnt)
       (list (car &pnt) (cadr &pnt)))

(defun resize_shp (plln $coeff translation_basept)  
       (setq _ent nil _ent_ nil ^^ent nil
             ^ent nil pnt nil new_pnt nil)
       (setq _ent plln
             _ent_ (entget _ent))
       (if (not (member translation_basept (collect_vertices plln)))
           (setq trans_basept (trans translation_basept 0 plln))
           (setq trans_basept translation_basept))
       (command "_ucs" "_e" plln)
       (while (and (setq _ent (entnext _ent))
                   (not (equal (setq etyp_ (extract_value 0 _ent)) "SEQUEND")))
              (if (not (equal etyp_ "SEQUEND"))
                  (progn (setq ^ent (entget _ent)
                                pnt (cdr (assoc 10 ^ent))
                                y_ (cadr pnt)) ;;preserve the Z value
                         (if pnt
                             (progn (setq ngl (angle trans_basept pnt)
                                          dstnc (distance (2d trans_basept) (2d pnt))
                                          new_pnt (polar trans_basept ngl (* dstnc $coeff))
                                          new_pnt (list (car new_pnt) y_ (last new_pnt))
                                          ^^ent (subst (cons 10 new_pnt)
                                                       (cons 10 pnt) ^ent))
                                    (entmod ^^ent))))))
;       (command "_ucs" "_w")
)

(defun c:rs ()
       (setq **p** (car (entsel "\nshape: ")))
       (if **p** (resize_shp **p** tmp21 (origin_pt **p**))))

;; find adjacent items to item in list
(defun consecutive-pts (|pt| lst)
        (cond ;; points on either side of |PT|
              ((and (cadr (member |pt| lst))
                    (cadr (member |pt| (reverse lst))))
               (list (cadr (member |pt| (reverse lst))) |pt|
                     (cadr (member |pt| lst))))

              ((and (null (cadr (member |pt| lst)))
                    (cadr (member |pt| (reverse lst))))
               (list (cadr (member |pt| (reverse lst))) |pt|))

              ((and (cadr (member |pt| lst))
                    (null (cadr (member |pt| (reverse lst)))))
               (list |pt| (cadr (member |pt| lst))))
              (T nil)))

;; midpoint of vector p1 p2
(defun midpt (p1 p2)
        (polar p1 (angle p1 p2) (/ (distance p1 p2) 2.000000)))

(defun RAD2DEG (ang)
       (* ang (/ 360 (* pi 2.000000))))

(untrace origin_pt copy_ent relocate opp_bisect_ang resize_shp)

;;; returns 'T if block exists, nil otherwise
(defun blockp (blockname)
       (if (null blockname) nil
           (if (member (strcase blockname) (listify_blocks))
               'T nil)))

;;; returns a list of all blocks in drawing
(defun listify_blocks (/ block blocks nam)
       (setq block (cdr (assoc 2 (tblnext "block" T))) ;;rewind block table
             blocks (list  block))
       (while (setq block (tblnext "block")) ;;construct block list
              (setq nam (cdr (assoc 2 block))
                    blocks (cons nam blocks))) blocks)

(defun 3d-face (pt-list flavor / p1 p2 p3 p4)
       (setq p1 (nth 0 pt-list)
             p2 (nth 1 pt-list) 
             p3 (nth 2 pt-list)
             p4 (nth 3 pt-list))
       (if (and p1 p2 p3)
           (if (equal flavor 'invisible_sides)
               (command "_3DFACE" "_i" p1 "_i" p2 "_i" p3
                        (if p4 "_i") (if p4 p4 "") "")
               (command "_3DFACE" p1 p2 p3 (if p4 p4 "") "")))
       (entlast))

(defun ss2ptlist (sset)
       (mapcar '(lambda (enm)
                        (cdr (assoc 10 (entget enm))))
               (ss2enamlist sset)))

;; convert a selection set to a list of entity lists
(defun ss2enamlist (ss / entlist ctr)
       (if ss (progn
           (setq ctr 0)
           (repeat (sslength ss)
                   (progn (setq entlist (cons (ssname ss ctr) entlist))
                          (setq ctr (1+ ctr)))))) (if entlist entlist))

(defun usrdist (dflt prmpt / var)
       (if (setq var
                 (getdist (if dflt (strcat prmpt " <" (rtos dflt 4 4) ">: ")
                          (strcat prmpt ": ")))) var dflt))

(defun pline (@pnts @w)
       (if (and @pnts (listp @pnts))
           (progn (command "_pline" (car @pnts) "_w" @w @w)
                  (foreach @pnt (cdr @pnts) (command @pnt))
                  (command ""))))

(defun leftmost (pointlist)
       (nth (pos_in_list (apply 'min
                                (mapcar 'car pointlist))
                         (mapcar 'car pointlist))
            pointlist))

(defun rightmost (pointlist)
       (nth (pos_in_list (apply 'max
                                (mapcar 'car pointlist))
                         (mapcar 'car pointlist))
            pointlist))

(defun lleft (verts)
       (list (apply 'min (mapcar 'car verts))
             (apply 'min (mapcar 'cadr verts))))

(defun lright (verts)
       (list (apply 'max (mapcar 'car verts))
             (apply 'min (mapcar 'cadr verts))))

(defun l_left (verts / _xmin _ymin tmplst ll_pt)
       (setq _ymin (apply 'min (mapcar 'cadr verts)))
       (foreach vert verts
                (if (equal (cadr vert) _ymin)
                    (setq tmplst
                          (if tmplst (cons vert tmplst)
                                     (list vert)))))
       (setq _xmin (apply 'min (mapcar 'car tmplst))
             ll_pt (nth (pos_in_list _xmin (mapcar 'car tmplst)) tmplst)))

(defun closedp (pl)
       (equal (cdr (assoc 70 (entget pl))) 1))

;; returns the proportion of the diagonal intersecting
;; the centroid of triangle VTX EP1 EP2 in relation to
;; the implied base
;; (vector EP2 (polar EP2 (ang normal to vector VTX EP2) baselen))
;;
;;                          EP2
;;                        +  -- -- -- --+
;;                        | \         / |
;;                        |   \     /    
;;                        |     \ /     |
;;                        |     / \      
;;                        |   *     \   |
;;                        | /  C      \  
;;                   VTX  +_____________+  EP1
;;

(defun bisector (vtx ep1 ep2 / 
                 @1 @2 @n1 @n2 @n3 @mp1 @@mp1 @@ep2 @@vtx @ct @ng1 @ng2 @ng3)
       (setq @1 (angle vtx ep1)
             @2 (angle vtx ep2)
             @n1 (polar vtx @1 1.0)
             @n2 (polar vtx @2 1.0)
             @ct (centroid (list vtx @n1 @n2))
             @ng1 (angle vtx @ct)
             @ng2 (angle vtx ep2))
       (if (>= @ng2 pi)
           (setq @ng3 (- @ng2 (/ pi 2.0)))
           (setq @ng3 (+ @ng2 (/ pi 2.0))))
       (setq @mp1 (midpt vtx ep2)
             @@mp1 (polar @mp1 @ng3 1.0)
             @@ep2 (polar ep2 @ng3 1.0)
             @@vtx (inters @ct vtx @@ep2 @@mp1 nil)
             diag_l (if @@vtx (distance vtx @@vtx) 0)))

(defun connect_plines (pl1 pl2 / i_ pl_pts1 pl_pts2 pl_pts_l)
;       (textscr)
       (setq i_ 0
             pl_pts1 (collect_vertices pl1)
             pl_pts2 (collect_vertices pl2)
;             pl_pts3 (append (list (last pl_pts1)) (cdr pl_pts1))
;             pl_pts4 (append (list (last pl_pts2)) (cdr pl_pts2))
)
       (if (and pl_pts1 pl_pts2)
           (progn (setq pl_pts_l (length pl_pts1))
                  (command "_3dmesh" pl_pts_l 5)
                  (repeat pl_pts_l
                          (command (trans (nth i_ pl_pts1) pl1 1)
                                   (trans (nth i_ pl_pts2) pl2 1)
                                   (trans (closest (nth i_ pl_pts2)
                                                   (aux_remove (nth i_ pl_pts2)
                                                               pl_pts2))
                                           pl2 1)
                                   (trans (closest (nth i_ pl_pts1)
                                                   (aux_remove (nth i_ pl_pts1)
                                                               pl_pts1))
                                           pl1 1)
                                   (trans (nth i_ pl_pts1) pl1 1))
                          (setq i_ (1+ i_))))))

(defun connect_plines (pl1 pl2 / i_ pl_pts1 pl_pts2 pl_pts_l)
       (setq i_ 0
             pl_pts1 (collect_vertices pl1)
             pl_pts2 (collect_vertices pl2))
       (if (and pl_pts1 pl_pts2)
           (progn (setq pl_pts_l (length pl_pts1))
                  (command "_3dmesh" pl_pts_l 2)
                  (repeat pl_pts_l
                          (command (trans (nth i_ pl_pts1) pl1 1)
                                   (trans (nth i_ pl_pts2) pl2 1))
                          (setq i_ (1+ i_))))))

;; returns sequence position of item in list
;; compatible with NTH
(defun pos_in_list (item lst) 
        (if (null (member item lst))
            nil
            (- (length lst) (length (member item lst)))))

(defun xz_ucs (z_rot)
       (command "_ucs" "_w")
       (command "_ucs" "_z" z_rot)
       (command "_ucs" "_x" 90))

(defun normal (bs_pt rf_pt)
       (/ (- (angle bs_pt rf_pt)
             (angle rf_pt bs_pt)) 2.0))

(defun collect_vertices (polyln / __ent *polyln* ##pt ##pts)
       (if (= (cdr (assoc 0 (setq *polyln* (entget polyln)))) "POLYLINE")
           (progn (setq __ent (entnext polyln))
                  (while (setq *ent* (entget __ent)
                               ##pt (cdr (assoc 10 *ent*)))
                         (setq ##pts (cons ##pt ##pts)
                               __ent (entnext __ent)))
                         (if (and (equal (cdr (assoc 70 (entget polyln))) 1)
                                  (not (equal (car ##pts) (last ##pts))))
                             (setq ##pts (reverse (cons (car ##pts) (reverse ##pts))))))
           (if (= (cdr (assoc 0 (setq *polyln* (entget polyln)))) "LINE")
               (setq ##pts (list (extract_value 10 polyln)
                                 (extract_value 11 polyln)))
               (princ "\ncollect_vertices: not a POLYLINE.")))
       (if ##pts (setq np (length ##pts)))
       ##pts)

(defun divide_pline (enm n_vrts_ _pt_  / node-1 node-n node1 noden pt_ss pt_l$$)
;       (setvar "cmdecho" 0)
       (setq enm-nodes (collect_vertices enm)
             node-1 (car enm-nodes)
             node-n (last enm-nodes))
       (command "_undo" "_c" "_one")
       (setq clayer (getvar "clayer"))
       (make_la "$PT$")
       (command "_divide" _pt_ (1+ n_vrts_))
       (setq pt_ss (ssget "x" (list (cons 0 "POINT")
                                    (cons 8 "$PT$")))
             pt_l$$ (ss2nodelist pt_ss)
             pt_ss nil)
       (gc)
       (setq node1 (car pt_l$$)
             noden (last pt_l$$)
             *node1 (closest node1 (list node-1 node-n))
             *noden (closest noden (list node-1 node-n))
             pt_l$$ (append (list *node1) pt_l$$ (list *noden)))
       (command "_undo" "1")
       (command "_undo" "1")
       (command "_undo" "1")
       (command "_layer" "_s" clayer "")
       pt_l$$)

;; SS2NODELIST returns a list of the origin points
;; of the members of SS
(defun ss2nodelist (ss / *ent* pt pts &i ssl)
       (if (or (null ss) (/= (type ss) 'PICKSET)) nil
           (progn (setq &i 0
                        ssl (sslength ss))
                  (repeat ssl
                         (setq ent (ssname ss &i)
                               *ent* (entget ent)
                               pt (cdr (assoc 10 *ent*))
                               pts (cons pt pts))
                         (setq &i (1+ &i))))) pts)

(defun make_la (%nam%)
       (if %nam%
           (progn (setq %clay (getvar "clayer"))
                  (if (layerp %nam%)
                      (if (not (equal %clay %nam%))
                          (command "_layer""_t"%nam%"_on"%nam%"_s"%nam%""))
                      (command "_layer" "_m" %nam% "")))))

(defun pline (vtxlist) 
       (entmake (list (quote (0 . "POLYLINE")))) ; entmake polyline header
       (foreach vtx vtxlist                      ; construct polyline vertices
                (entmake (list (quote (0 . "VERTEX"))
                (cond ((null vtx) nil)
                      ((< (length vtx) 3)
                       (list 10 (car vtx) (cadr vtx)))
                     (T (list 10 (car vtx) (cadr vtx) (caddr vtx)))))))
       (entmake '((0 . "SEQEND"))))


;;; returns 'T if layer exists, nil otherwise
(defun layerp (layername)
       (if (null layername) nil
           (if (member (strcase layername) (listify_layers))
               'T nil)))

(defun unique_atoms (lst / tmp unique_lst)
       (setq tmp lst)
       (repeat (length lst)
               (if (null unique_lst)
                   (setq unique_lst (list (car tmp)))
                   (if (not (member (car tmp) (cdr tmp)))
                       (if unique_lst
                           (if (not (member (car tmp) unique_lst))
                               (setq unique_lst (cons (car tmp) unique_lst))))))
                (setq tmp (cdr tmp)))
        (reverse unique_lst))

(defun _user_ent (ent_type_lst _prm / __ent)
      (while (not (member (extract_value 0 
                                         (setq __ent (car (entsel _prm))))
                ent_type_lst))) __ent)

;;; returns a list of all layers in drawing
(defun listify_layers (/ layer layers nam)
       (setq layer (cdr (assoc 2 (tblnext "layer" T))) ;;rewind layer table
             layers (list  layer))
       (while (setq layer (tblnext "layer")) ;;construct layer list
              (setq nam (cdr (assoc 2 layer))
                    layers (cons nam layers))) layers)

(defun userdist (refpt dflt prmpt / var)
       (if (setq var
                 (getdist (if refpt refpt)
                          (if dflt (strcat prmpt " <" (rtos dflt 4 2) ">: ")
                                   (strcat prmpt ": ")))) var dflt))
 
(defun userint (dflt prmpt / var)
       (if (setq var
                 (getint (if dflt (strcat prmpt " <" (itoa dflt) ">: ")
                                  (strcat prmpt ": ")))) var dflt))

(defun user_ent (_ent_type _prm / __ent)
      (while (not (equal (extract_value 0 
                                        (setq __ent (car (entsel _prm))))
                _ent_type))) __ent)

(defun userstr (dflt prmpt / var) ;;DFLT and PRMPT are strings
       (setq var (getstring t (if (and dflt (/= dflt ""))
                                (strcat prmpt " <" dflt ">: ")
                                (strcat prmpt ": "))))
       (cond ((/= var "") var)
             ((and dflt (= var "")) dflt)
             (T dflt)))

(defun right_angle (p1 p2 / hpi a1 a2)
       (setq hpi (limit_prec (/ pi 2.0) 4)
             a1 (limit_prec (angle p1 p2) 4)
             a2 (limit_prec (angle p2 p1) 4))
       (or (= (rem a1 hpi) 0.0)
           (= (rem a2 hpi) 0.0)))

(defun limit_prec (n p)
       (read (rtos n 2 p)))

(defun c:rdp ()
       (setq *shp1* (_user_ent (list "POLYLINE") "\ncontour to redraw: ")
             op (trans (origin_pt *shp1*) *shp1* 1)
             res (userint  (if res res 100) "\nResolution")
             plnds (divide_pline *shp1* res op))
       (pline plnds)
       (entdel *shp1*)
       (setq *shp1* nil plnds nil)
       (redraw)
       'done)
 
(defun XY (pt) ;; convert 3D point to 2D
       (list (car pt) (cadr pt)))

;; find closest point in node list "nodes" to point "pt"'
(defun closest (pt nodes)
       (nth
          (1- (pos-in-list
                 (apply 'min (mapcar '(lambda (node) (distance pt node)) nodes))
                             (mapcar '(lambda (node) (distance pt node)) nodes)))
        nodes))

(defun aux_remove (atm lst) ;; This only works on a list of unique atoms!!!
       (cond ((null lst) nil) 
             ((null (member atm lst)) lst)
             ((equal atm (car lst)) (cdr lst))
             (t (append (reverse (cdr (member atm (reverse lst))))
                        (cdr (member atm lst))))))

(defun explode (str / firstchr *str*)  ;; iterative text explosion
      (if (null str) nil
          (repeat (strlen str)
                  (progn
                      (setq *str* (cons (setq firstchr (substr str 1 1)) *str*))
                      (setq str (substr str 2))))) (reverse *str*))

(defun concat (lst / str)
       (if (or (null lst)
               (/= (type lst) 'LIST)) nil
           (apply 'strcat lst)))

(defun pos-in-list (item lst)
        (if (null (member item lst))
            nil
            (- (length lst) (length (cdr (member item lst))))))

;(c:xtrud)
(princ "\nC:XTRUD loaded - type XTRUD to use.")
(princ)

