//===========================================================================
//
// Quake BSP to Unreal brush converter, by Jeff Preshing, August 1998
//
// Thanks to Olivier Montanuy for his online decription of the BSP format.
// I cut and pasted a few data types from his page:
//
// http://www.gamers.org/dEngine/quake/spec/quake-spec34/qkspec_4.htm#CBSP0 
//
//===========================================================================

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>

#define MAX_PATH 256
#define MAX_MEMBLOCK 20000000 // Don't try to alloc more than this, ever
#define INVALID_SIZE(x) ((x) < 0 || (x) >= MAX_MEMBLOCK)
#define DEFAULT_SCALE_FACTOR 1.35f
#define VERSION 1.0f

enum DO_TYPE { DO_NOTHING, DO_LIST, DO_EXTRACT };

struct ConverterOptions {
  DO_TYPE doWhat;
  int modelNum;
  char *filename;
	float scaleFactor;
	int mirror;
  int insideOut;
  int textures;
  int keepWater;

  ConverterOptions() {
    doWhat = DO_NOTHING;
    modelNum = -1;
    filename = 0;
		scaleFactor = DEFAULT_SCALE_FACTOR;
		mirror = 1;
    insideOut = 0;
    textures = 0;
    keepWater = 0;
  }

  float thisScaleX;
  int thisFaceBackwards;
};

struct ModelProgress {
  float thisScaleX;
  int thisFaceBackwards;
};

ConverterOptions g_opt;

typedef unsigned long u_long;
typedef unsigned short u_short;
typedef unsigned char u_char;

void parseFilename(const char *from, char *pathname, char *basename) {
	const char *p = from + strlen(from);
	const char *e = p;

	for (; p > from; p--)	// strip pathname
		if (p[-1] == '\\')
			break;

	for (; e > p; e--) // strip extension
		if (*e == '.')
			break;

	// return it
	if (pathname) {
		memcpy(pathname, from, p - from);
		pathname[p - from] = 0;
	}
	if (basename) {
		memcpy(basename, p, e - p);
		basename[e - p] = 0;
	}
}

//====================================================================

typedef struct                 // A Directory entry
{ long  offset;                // Offset to entry, in bytes, from start of file
  long  size;                  // Size of entry in file, in bytes
} dentry_t;

typedef struct                 // The BSP file header
{ long  version;               // Model version, must be 0x17 (23).
  dentry_t entities;           // List of Entities.
  dentry_t planes;             // Map Planes.
                               // numplanes = size/sizeof(plane_t)
  dentry_t miptex;             // Wall Textures.
  dentry_t vertices;           // Map Vertices.
                               // numvertices = size/sizeof(vertex_t)
  dentry_t visilist;           // Leaves Visibility lists.
  dentry_t nodes;              // BSP Nodes.
                               // numnodes = size/sizeof(node_t)
  dentry_t texinfo;            // Texture Info for faces.
                               // numtexinfo = size/sizeof(texinfo_t)
  dentry_t faces;              // Faces of each surface.
                               // numfaces = size/sizeof(face_t)
  dentry_t lightmaps;          // Wall Light Maps.
  dentry_t clipnodes;          // clip nodes, for Models.
                               // numclips = size/sizeof(clipnode_t)
  dentry_t leaves;             // BSP Leaves.
                               // numlaves = size/sizeof(leaf_t)
  dentry_t lface;              // List of Faces.
  dentry_t edges;              // Edges of faces.
                               // numedges = Size/sizeof(edge_t)
  dentry_t ledges;             // List of Edges.
  dentry_t models;             // List of Models.
                               // nummodels = Size/sizeof(model_t)
} dheader_t;

typedef float scalar_t;        // Scalar value

struct vec3_t                  // Vector or Position
{ scalar_t x;                  // horizontal
  scalar_t y;                  // horizontal
  scalar_t z;                  // vertical

  vec3_t() {
  }
  inline vec3_t(scalar_t x, scalar_t y, scalar_t z) : x(x), y(y), z(z) {
  }
  inline float operator*(const vec3_t &v) const {
    return x * v.x + y * v.y + z * v.z;
  }
  inline vec3_t operator+(const vec3_t &v) const {
    return vec3_t(x + v.x, y + v.y, z + v.z);
  }
  inline friend vec3_t operator*(float f, const vec3_t &v) {
    return vec3_t(f * v.x, f * v.y, f * v.z);
  }
};

typedef struct                 // Bounding Box, Float values
{ vec3_t   min;                // minimum values of X,Y,Z
  vec3_t   max;                // maximum values of X,Y,Z
} boundbox_t;

typedef struct                 // Bounding Box, Short values
{ short   min;                 // minimum values of X,Y,Z
  short   max;                 // maximum values of X,Y,Z
} bboxshort_t;

struct edge_t
{ u_short vertex0;             // index of the start vertex
                               //  must be in [0,numvertices[
  u_short vertex1;             // index of the end vertex
                               //  must be in [0,numvertices[

	inline int verifyIntegrity();
};

struct face_t
{ u_short plane_id;            // The plane in which the face lies
                               //           must be in [0,numplanes[ 
  u_short side;                // 0 if in front of the plane, 1 if behind the plane
  long ledge_id;               // first edge in the List of edges
                               //           must be in [0,numledges[
  u_short ledge_num;           // number of edges in the List of edges
  u_short texinfo_id;          // index of the Texture info the face is part of
                               //           must be in [0,numtexinfos[ 
  u_char typelight;            // type of lighting, for the face
  u_char baselight;            // from 0xFF (dark) to 0 (bright)
  u_char light[2];             // two additional light models  
  long lightmap;               // Pointer inside the general light map, or -1
                               // this define the start of the face light map

	int getEdgeIndex(int i);
	vec3_t *getHeadVertOfEdge(int i);
	int verifyIntegrity();
	void exportToUnrealPolygon(FILE *file);
};

struct mipheader_t {               // Mip texture list header
  long numtex;                 // Number of textures in Mip Texture list
  long offset;                 // Offset to each of the individual texture

  struct miptex_t {                // Mip Texture
    char   name[16];             // Name of the texture.
    u_long width;                // width of picture, must be a multiple of 8
    u_long height;               // height of picture, must be a multiple of 8
    u_long offset1;              // offset to u_char Pix[width   * height]
    u_long offset2;              // offset to u_char Pix[width/2 * height/2]
    u_long offset4;              // offset to u_char Pix[width/4 * height/4]
    u_long offset8;              // offset to u_char Pix[width/8 * height/8]
  };

  inline miptex_t *getMipTex(long n) {
    return (miptex_t *) ((char *) this + (&offset)[n]);
  }
  inline int isSky(long n) {
    char *name = getMipTex(n)->name;
    return (name[0] == 's' && name[1] == 'k' && name[2] == 'y');
  }
};                 //  from the beginning of mipheader_t


struct texinfo_t
{ vec3_t   vectorS;            // S vector, horizontal in texture space)
  scalar_t distS;              // horizontal offset in texture space
  vec3_t   vectorT;            // T vector, vertical in texture space
  scalar_t distT;              // vertical offset in texture space
  u_long   texture_id;         // Index of Mip Texture
                               //           must be in [0,numtex[
  u_long   animated;           // 0 for ordinary textures, 1 for water 

  int verifyIntegrity();
};

struct model_t
{ boundbox_t bound;            // The bounding box of the Model
  vec3_t origin;               // origin of model, usually (0,0,0)
  long node_id0;               // index of first BSP node
  long node_id1;               // index of the first Clip node
  long node_id2;               // index of the second Clip node
  long node_id3;               // usually zero
  long numleafs;               // number of BSP leaves
  long face_id;                // index of Faces
  long face_num;               // number of Faces

	inline face_t *getFace(int f);
	inline void dumpInfo();
	int verifyIntegrity();
	void exportToUnrealBrush(FILE *file);
};

struct QuakeBSPFile {
  dheader_t dir;

  int num_verts;
  vec3_t *verts;
  int num_edges;
  edge_t *edges;
  int num_ledges;
  int *ledges;
	int num_faces;
	face_t *faces;
	int num_models;
	model_t *models;
  int num_texinfos;
  texinfo_t *texinfos;
  mipheader_t *mipinfo;

	int loadFile(FILE *file);
	int verifyIntegrity();
	void listBSPContents();
	int convertBSP();
	int exportToUnrealBrush(int m);
};

QuakeBSPFile g_bsp;


//========================================================

inline int edge_t::verifyIntegrity() {
	return (vertex0 < g_bsp.num_verts && vertex1 < g_bsp.num_verts);
}

inline int face_t::getEdgeIndex(int i) {
	return g_bsp.ledges[ledge_id + i];
}

inline u_short getHeadVertIndex(int e) {
  return (e > 0 ? g_bsp.edges[e].vertex0 : g_bsp.edges[-e].vertex1);
}

inline u_short getTailVertIndex(int e) {
  return (e > 0 ? g_bsp.edges[e].vertex1 : g_bsp.edges[-e].vertex0);
}

inline int verifyEdgeIndexIntegrity(int e) {
	return (e > 0 ? e < g_bsp.num_edges :
	        (e != 0 && -e < g_bsp.num_edges));
}

inline int texinfo_t::verifyIntegrity() {
  return (texture_id < (unsigned) g_bsp.mipinfo->numtex);
}

inline vec3_t *face_t::getHeadVertOfEdge(int i) {
	return g_bsp.verts + getHeadVertIndex(getEdgeIndex(i));
}

int face_t::verifyIntegrity() {
	if (ledge_id < 0)
		return 0;
	if (ledge_num < 3)
		return 0;
	if (ledge_id + ledge_num > g_bsp.num_ledges)
		return 0;
  if (texinfo_id >= g_bsp.num_texinfos)
    return 0;
  if (!g_bsp.texinfos[texinfo_id].verifyIntegrity())
    return 0;

  // Check that we've got a closed loop of connected edges for this face
	int e, t1, h2, t2;
	e = getEdgeIndex(0);
	verifyEdgeIndexIntegrity(e);
	t1 = getTailVertIndex(e);
	if (t1 > g_bsp.num_verts)
		return 0;
	for (int i = 1; i < ledge_num; i++) {
		e = getEdgeIndex(i);
		verifyEdgeIndexIntegrity(e);
		t2 = getTailVertIndex(e);
		if (t2 > g_bsp.num_verts)
			return 0;
		h2 = getHeadVertIndex(e);
		if (h2 != t1) // contiguity
			return 0;
		t1 = t2;
	}
	h2 = getHeadVertIndex(getEdgeIndex(0));
	if (h2 != t1) // contiguity
		return 0;

	return 1;
}

inline void model_t::dumpInfo() {
	printf("Faces: %d  \tApprox. size: %.2f x %.2f x %.2f\n",
				 face_num,
				 bound.max.x - bound.min.x,
				 bound.max.y - bound.min.y,
				 bound.max.z - bound.min.z);
}

inline void vecPrint(FILE *file, const vec3_t &v) {
	fprintf(file, "%+013.6f,%+013.6f,%+013.6f\n",
				  v.x * g_opt.scaleFactor * g_opt.thisScaleX,
				  v.y * g_opt.scaleFactor,
				  v.z * g_opt.scaleFactor);
}

void face_t::exportToUnrealPolygon(FILE *file) {
	vec3_t *v;
  texinfo_t *tex;
  int isSky = 0;

  tex = g_bsp.texinfos + texinfo_id;
  if (tex->animated) {
    if (g_bsp.mipinfo->isSky(tex->texture_id))
      isSky = 1;
    else
      if (!g_opt.keepWater)
        return; // don't want to export water surface.
  }

	fputs("  Begin Polygon\n", file);
  if (!isSky && g_opt.textures) {
    float SS, TT, ST, denom;
    SS = tex->vectorS * tex->vectorS;
    TT = tex->vectorT * tex->vectorT;
    ST = tex->vectorS * tex->vectorT;
    denom = SS * TT - ST * ST;

    if (denom != 0) {
      float sPart, tPart;
      sPart = tex->distT * ST - tex->distS * TT;
      tPart = tex->distT * SS - tex->distS * ST;
      fprintf(file, "    Origin   ");
      vecPrint(file, sPart * tex->vectorS + tPart * tex->vectorT);
      fprintf(file, "    TextureU ");
      vecPrint(file, tex->vectorS);
      fprintf(file, "    TextureV ");
      vecPrint(file, tex->vectorT);
    } // else bad texture space vectors.
  }
	if (g_opt.thisFaceBackwards) {
		for (int i = ledge_num - 1; i >= 0; i--) {	// verts in opposite order
			v = getHeadVertOfEdge(i);
			fprintf(file, "    Vertex   ");
      vecPrint(file, *v);
		}
	} else {
		for (int i = 0; i < ledge_num; i++) {
			v = getHeadVertOfEdge(i);
			fprintf(file, "    Vertex   ");
      vecPrint(file, *v);
		}
	}
	fputs("  End Polygon\n", file);
}

inline face_t *model_t::getFace(int f) {
	return g_bsp.faces + face_id + f;
}

int model_t::verifyIntegrity() {
	if (face_id < 0)
		return 0;
	if (face_num < 0)
		return 0;
	if (face_id + face_num > g_bsp.num_faces)
		return 0;
	for (int i = 0; i < face_num; i++) {
		if (!getFace(i)->verifyIntegrity())
			return 0;
	}
	return 1;
}

void model_t::exportToUnrealBrush(FILE *file) {
	fputs("Begin PolyList\n", file);
	for (int i = 0; i < face_num; i++)
		getFace(i)->exportToUnrealPolygon(file);
	fputs("End PolyList\n", file);
}

//========================================================

int QuakeBSPFile::loadFile(FILE *file) {
	fread(&dir, sizeof(dheader_t), 1, file);
	if (dir.version != 0x1D) {
		puts("I only work with BSP version 0x1D files.");
		return 0;
	}
	// Read the vertices
	if (INVALID_SIZE(dir.vertices.size))
		return 0;
	num_verts = dir.vertices.size / sizeof(vec3_t);
	verts = new vec3_t[num_verts];
	fseek(file, dir.vertices.offset, SEEK_SET);
	fread(verts, sizeof(vec3_t), num_verts, file);
	// Read the edges
	if (INVALID_SIZE(dir.edges.size))
		return 0;
	num_edges = dir.edges.size / sizeof(edge_t);
	edges = new edge_t[num_edges];
 	fseek(file, dir.edges.offset, SEEK_SET);
	fread(edges, sizeof(edge_t), num_edges, file);
	// Read the ledges
	if (INVALID_SIZE(dir.ledges.size))
		return 0;
	num_ledges = dir.ledges.size / sizeof(int);
	ledges = new int[num_ledges];
	fseek(file, dir.ledges.offset, SEEK_SET);
	fread(ledges, sizeof(int), num_ledges, file);
	// Read the faces
	if (INVALID_SIZE(dir.faces.size))
		return 0;
	num_faces = dir.faces.size / sizeof(int);
	faces = new face_t[num_faces];
	fseek(file, dir.faces.offset, SEEK_SET);
	fread(faces, sizeof(face_t), num_faces, file);
	// Read the models
	if (INVALID_SIZE(dir.models.size))
		return 0;
	num_models = dir.models.size / sizeof(model_t);
	models = new model_t[num_models];
	fseek(file, dir.models.offset, SEEK_SET);
	fread(models, sizeof(model_t), num_models, file);
  // Read the texinfos
  if (INVALID_SIZE(dir.texinfo.size))
    return 0;
  num_texinfos = dir.texinfo.size / sizeof(texinfo_t);
  texinfos = new texinfo_t[num_texinfos];
  fseek(file, dir.texinfo.offset, SEEK_SET);
  fread(texinfos, sizeof(texinfo_t), num_texinfos, file);
  // Read the mipinfos
  if (INVALID_SIZE(dir.miptex.size))
    return 0;
  mipinfo = (mipheader_t *) new char[dir.miptex.size];
  fseek(file, dir.miptex.offset, SEEK_SET);
  fread(mipinfo, 1, dir.miptex.size, file);
	return 1;
}

int QuakeBSPFile::verifyIntegrity() {
	for (int i = 0; i < num_models; i++) {
		models[i].verifyIntegrity();
	}
	return 1;
}

void QuakeBSPFile::listBSPContents() {
	puts("Dump of models in this BSP...");
	for (int i = 0; i < num_models; i++) {
		printf("# %d: \t", i);
		models[i].dumpInfo();
	}
	printf("-- %d models total.\n", i);
}

int QuakeBSPFile::exportToUnrealBrush(int m) {
	char basename[MAX_PATH], outname[MAX_PATH];
	FILE *file = NULL;

	parseFilename(g_opt.filename, NULL, basename);
	sprintf(outname, "%s%03d.t3d", basename, m);
	file = fopen(outname, "w");
	if (!file) {
		printf("Couldn't open output file '%s'.\n", outname);
		return 0;
	}

  g_opt.thisScaleX = g_opt.mirror ? -1.f : 1.f;
  g_opt.thisFaceBackwards = 
    ((g_opt.insideOut && m != 0) || (!g_opt.insideOut && m == 0));
	models[m].exportToUnrealBrush(file);
	fclose(file);
	return 1;
}

int QuakeBSPFile::convertBSP() {
	int rc = 0;
	if (g_opt.modelNum == -1) {
		for (int m = 0; m < g_bsp.num_models; m++)
			if (!exportToUnrealBrush(m))
				goto cleanup;
    printf("Converted %d model(s).\n", m);
	} else {
    if (g_opt.modelNum < 0 || g_opt.modelNum >= g_bsp.num_models) {
      puts("Model number out of range.");
      goto cleanup;
    }
		if (!exportToUnrealBrush(g_opt.modelNum))
			goto cleanup;
    printf("Converted model #%d.\n", g_opt.modelNum);
	}
	rc = 1;

cleanup:
	return rc;
}

void showHelp() {
	puts("Usage:  Bsp2t3d [options] <input file>\n"
		   "Options:  -l      List individual models only\n"
			 "          -a      Extract all models\n"
			 "          -e #    Extract model number only\n"
			 "          -f      Flip the model horizontally\n"
			 "          -s ###  Scale the model by the given factor\n"
			 "          -i      Turn faces inside-out\n"
			 "          -w      Include water surfaces\n"
       "          -t      Include texture co-ordinates");
}

int main(int argc, char *argv[]) {
	FILE *file = NULL;
	int rc = -1, helpme = 1;

	puts("");
  for (int arg = 1; arg < argc; arg++) {
    if (argv[arg][0] != '-')
      break;
    switch (argv[arg][1]) {
    case '?':
    case 'h':
			rc = 0;
      goto cleanup;
		case 'l':
			g_opt.doWhat = DO_LIST;
			break;
		case 'e':
			g_opt.doWhat = DO_EXTRACT;
			if (++arg >= argc) {
				puts("-e requires an argument.\n");
				goto cleanup;
			}
			g_opt.modelNum = atoi(argv[arg]);
			break;
		case 'a':
			g_opt.doWhat = DO_EXTRACT;
			g_opt.modelNum = -1;
			break;
		case 'f':
			g_opt.mirror = 0; // actually 'unflips' it
			break;
		case 's':
			if (++arg >= argc) {
				puts("-s requires an argument.\n");
				goto cleanup;
			}
			sscanf(argv[arg], "%f", &g_opt.scaleFactor);
			if (g_opt.scaleFactor <= 0) {
				puts("Scale factor should be a positive number, eg. '2.500'.\n");
				goto cleanup;
			}
			g_opt.scaleFactor *= DEFAULT_SCALE_FACTOR;	// So '1.00' corresponds to DEFAULT_SCALE_FACTOR.
			break;
    case 'i':
      g_opt.insideOut = 1;
      break;
    case 't':
      g_opt.textures = 1;
      break;
    case 'w':
      g_opt.keepWater = 1;
      break;
    default:
      puts("Huh?\n");
      goto cleanup;
    }
  }
  if (arg != argc - 1) {
    puts("You must specify exactly one input file.\n");
    goto cleanup;
  }
  g_opt.filename = argv[arg];

	if (g_opt.doWhat == DO_NOTHING) {
		puts("Specify -l, -e or -a for something to do.\n");
		goto cleanup;
	}

	helpme = 0;	
	file = fopen(g_opt.filename, "rb");
  if (!file) {
    printf("Couldn't open \"%s\" for read.\n", g_opt.filename);
    goto cleanup;
  }
  if (!g_bsp.loadFile(file)) {
    puts("Error while trying to read the BSP.\n");
		goto cleanup;
	}
	if (!g_bsp.verifyIntegrity()) {
		puts("This BSP is messed up.");
		goto cleanup;
	}
	switch (g_opt.doWhat) {
		case DO_LIST:
	    g_bsp.listBSPContents();
			break;
		case DO_EXTRACT:
			g_bsp.convertBSP();
			break;
  }

cleanup:
	if (helpme)
		showHelp();
	if (file)
		fclose(file);
  printf("\nQuake BSP to Unreal brush converter version %.1f, by Pointdexter\n", VERSION);
	return rc;
}
